/*
 * Copyright (C) 2006 Hanhua Feng (hanhua@users.sourceforge.net)
 */
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <sys/time.h>
#include <dockapp.h>
#include <ctype.h>
#include "panel.xpm"
#include "ascii.xpm"

#define CHAR_WIDTH 7
#define CHAR_HEIGHT 9

char *description = 
    "\nWindowMaker DockApp: network throughput monitor for Linux.\n"
    "  Information displayed:\n"
    "    First line:   running period\n"
    "    Second line:  CPU utilization rate during this period\n"
    "    Third line:   bytes received from the specified network device\n"
    "    Forth line:   bytes sent to the specified network device\n"
    "  Mouse click operations:\n"
    "    Left button:  Refresh the data immediately.\n"
    "    Right button: Reset all data to zero.\n";

char *version = "wmnethru version 0.1";
char *device = "eth0";

#define WMDA_SIZE 56

char *display = "";
int refresh_time = 1000;
char *color_string = NULL;

long long init_thru[2];
long long last_thru[2];
long long init_time;
long long thru_wrap[2];
double init_usage[2];

Pixmap panel_pixmap, panel_mask, ascii_pixmap, ascii_mask;

static DAProgramOption options[] = {
    { "-display", NULL, "Set X display", DOString, True, {&display} },
    { "-r", NULL, "Refresh time", DONatural, True, {&refresh_time} },
    { "-d", NULL, "Network device name (default: eth0)", DOString, True, {&device} },
    { "-c", NULL, "Font color", DOString, True, {&color_string} }
};


long long num_sent = 0, num_recv = 0;

static long long get_curr_time()
{
    struct timeval tv;
    gettimeofday( &tv, NULL );
    return (long long) tv.tv_sec * 1000000LL + tv.tv_usec;
}

static void draw_string( GC gc, int x, int y, const char *str )
{
    int i, offset;

    for ( i=0; i<7 && str[i]; i++ )
    {
        switch ( str[i] )
        {
        case 'K':
            offset = 10;
            break;
        case 'M':
            offset = 11;
            break;
        case 'G':
            offset = 12;
            break;
        case '.':
            offset = 13;
            break;
        case 'H':
            offset = 14;
            break;
        case '%':
            offset = 15;
            break;
        default:
            if ( str[i] >= '0' && str[i] <= '9' )
                offset = str[i] - '0';
            else
                offset = 16;
            break;
        } 

        XCopyArea( DADisplay, ascii_pixmap, panel_pixmap, gc, 
                   offset*CHAR_WIDTH, 0, CHAR_WIDTH-1, CHAR_HEIGHT, x+i*CHAR_WIDTH, y );
    }
}

static void get_curr_thru( long long thru[2], const char *dev )
{
    int i;
    char *p, buff[256], *p1;
    FILE *fp = fopen( "/proc/net/dev", "r" );

    thru[0] = thru[1] = -1LL;
    if ( fp )
    {
        while( fgets( buff, 256, fp ) )
        {
            p = buff;
            while ( *p != '\0' && isspace(*p) )
                p++;
            p1 = strchr( p, ':' );
            if ( p1 == NULL )
                continue;
            *p1 = '\0';
            if ( 0 == strcmp( p, dev ) )
            {
                p = p1+1;
                thru[0] = strtoll( p, &p, 10 );
                for ( i=0; i<7; i++ )
                {
                    if ( p == NULL )
                        continue;
                    strtoll( p, &p, 10 );
                }
                if ( p == NULL )
                    continue;
                thru[1] = strtoll( p, &p, 10 );
            }
        }

        fclose( fp );
    }
}

static void get_curr_usage( double usage[2] )
{
    char buff[256];
    FILE *fp = fopen( "/proc/uptime", "r" );
    usage[0] = usage[1] = 0.0;
    if ( fp != NULL )
    {
        if ( fgets( buff, sizeof(buff), fp ) )
        {
            char * p;
            usage[0] = strtod( buff, &p );
            usage[1] = strtod( p, &p );
        }
        fclose( fp );
    }
}

static void init()
{
    get_curr_thru( init_thru, device );
    memcpy( last_thru, init_thru, sizeof(init_thru) );
    thru_wrap[0] = thru_wrap[1] = 0;

    init_time = get_curr_time();
    get_curr_usage( init_usage );
}

static void cb_button_press( int button, int state, int x, int y )
{
    /* reset if the right mouse button is clicked */
    if ( button == 3 )
    {
        init();
    }
}

static void time_to_str( char *buff, unsigned int t )
{
    if ( t < 36000 )
        sprintf( buff, "%1dH%02dM%02d", t/3600, (t%3600)/60, t%60 );
    else if ( t < 3600 * 1000 )
        sprintf( buff, "%3dH%02dM", t/3600, (t%3600)/60 );
    else if ( t < 3600 * 1000000u )
        sprintf( buff, "%6dH" , t/3600 );
    else
        strcpy( buff, "%%%%%%%" );
}

static void thru_to_str( char *buff, long long n )
{
    if ( n < 0 )
        strcpy( buff, "......." );
    else if ( n < 1000LL )
        sprintf( buff, "%7d", (int)n );
    else if ( n < 1000000LL )
        sprintf( buff, "%6.2fK", (double)n/1000.0 );
    else if ( n < 1000000000LL )
        sprintf( buff, "%6.2fM", (double)n/1E6 );
    else if ( n < 1000000000000LL )
        sprintf( buff, "%6.2fG", (double)n/1E9 );
    else
        sprintf( buff, "%6dG", (int)( n / 1000000000LL ) );
}


static void usage_to_str( char *buff, double time, double idle )
{
    if ( time <= 0.0 )
        strcpy( buff, "%G%M%K%H" );
    else
        sprintf( buff, "%6.2f%%", (1.0-idle/time)*100.0 );
}

static long long count_wrap( long long curr, long long last )
{
    if ( curr >= last || curr >= (1LL<<31) || last < (1LL<<31) )
        return 0;
    return ( 1LL << 32 );
}

static void update( GC gc, const char * dev )
{
    char buff[16];
    long long curr_thru[2];
    double curr_usage[2];

    time_to_str( buff, (unsigned int)((get_curr_time()-init_time )/1000000LL) );
    draw_string( gc, 3, 2, buff );

    get_curr_usage( curr_usage );
    usage_to_str( buff, curr_usage[0]-init_usage[0], 
                  curr_usage[1]-init_usage[1] );
    draw_string( gc, 3, 2+14, buff );

    get_curr_thru( curr_thru, dev );

    /* Linux 32 bit version have wrap-around problem */
    thru_wrap[0] += count_wrap( curr_thru[0], last_thru[0] );
    thru_wrap[1] += count_wrap( curr_thru[1], last_thru[1] );

    thru_to_str( buff, curr_thru[0] - init_thru[0] + thru_wrap[0] );
    draw_string( gc, 3, 2+14*2, buff );
    thru_to_str( buff, curr_thru[1] - init_thru[1] + thru_wrap[1] );
    draw_string( gc, 3, 2+14*3, buff );

    memcpy( last_thru, curr_thru, sizeof(curr_thru) );

    DASetPixmap( panel_pixmap );
}

int main( int argc, char *argv[] )
{
    unsigned int height = 0, width = 0;
    GC gc;
    DACallbacks cbs = { NULL, cb_button_press, NULL, NULL, NULL, NULL, NULL };
    long long last_refresh_time = 0;

    DAParseArguments( argc, argv, options, sizeof(options)/sizeof(options[0]),
                      description, version );

    DAInitialize( display, "wmnethru", WMDA_SIZE, WMDA_SIZE, argc, argv );

    if ( color_string != NULL )
    {
        static char buff1[16], buff2[16];
        XColor color;
        XParseColor( DADisplay,
                     DefaultColormap(DADisplay, DefaultScreen(DADisplay)), 
                     color_string, &color );
        sprintf( buff1, "@\tc #%02X%02X%02X", 
                 color.red>>8, color.green>>8, color.blue>>8 );
        ascii_xpm[2] = buff1;
        sprintf( buff2, "*\tc #%02X%02X%02X", 
                 color.red>>9, color.green>>9, color.blue>>9 );
        ascii_xpm[3] = buff2;
    }

    DAMakePixmapFromData( ascii_xpm, &ascii_pixmap, &ascii_mask, &height, &width );
    DAMakePixmapFromData( panel_xpm, &panel_pixmap, &panel_mask, &height, &width );

    DASetPixmap( panel_pixmap );

    gc = DefaultGC( DADisplay, DefaultScreen( DADisplay ) );

    DASetCallbacks( &cbs );
    DAShow();
    
    init();
    for (;;)
    {
        XEvent e;
        long long now = get_curr_time();
        if ( now > last_refresh_time + refresh_time * 1000LL )
        {
            last_refresh_time = now;
            update( gc, device );
        }

        while ( XPending( DADisplay ) )
        {
            XNextEvent( DADisplay, &e );
            DAProcessEvent( &e );
            update( gc, device );
        }

        usleep( 100000L );
    }

    return 0;
}
