/*
 * Copyright (C) 2004 Sven Schaepe <schaepe@web.de>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <linux/version.h>

#include <X11/Xlib.h>
#include <X11/xpm.h>
#include <X11/extensions/shape.h>

#include "../wmgeneral/wmgeneral.h"
#include "../wmgeneral/misc.h"

#include "wmcpum-master.xpm"

#define v_VERSION "0.1.0"

/* digit */
#define digit_width		6
#define digit_height		7
/* green */
#define digit_1_x		5
#define digit_1_y		66
/* reset */
#define digit_reset_x		28
#define digit_reset_y		29

char wmcpum_mask_bits[64*64];
int wmcpum_mask_width = 64;
int wmcpum_mask_height = 64;

static void drawInteger(int dx,int dy,int value,int count,int digits)
{
    int i,j,length;
    if(value==0) {
	if(count) {
	    length=count;
	} else {
	    length=1;
	}
    } else {
	length=((int)log10((double)value))+1;
    }
    for(i=0;i<length;i++,value/=10) {
	copyXPMArea((value-(value/10)*10)*digit_width+digit_1_x,digit_1_y,digit_width,digit_height,dx-digit_width*i,dy);
    }
    for(j=0;j<digits-length;j++,i++) {
	copyXPMArea(digit_reset_x,digit_reset_y,digit_width,digit_height,dx-digit_width*i,dy);
    }
}

static int getCPUMHz()
{
    char line[256];
    FILE *file=fopen("/proc/cpuinfo","r");
    char MHZ[]="cpu MHz";
    double mhz;
    while(fgets(line,sizeof(line),file)!=NULL) {
	char *ptr=line;
	char *tag=strsep(&ptr,":");
	if(strncmp(tag,MHZ,sizeof(MHZ)-1)==0) {
	    sscanf(ptr,"%lf",&mhz);
	}
    }
    fclose(file);
    return((int)mhz);

}
static int getValue(size_t targetPathSize,char *path,char *file)
{
    char name[targetPathSize];
    snprintf(name,targetPathSize,"%s/%s",path,file);
    FILE *f;
    int value;
    if((f=fopen(name,"r"))==NULL) {
	fprintf(stderr, "Unable to read value from %s.\n",name);
    } else {
	fscanf(f,"%d",&value);
	fclose(f);
    }
    return value;
}

static void getValues(char *sensorPath,int sensorPathLength) 
{
    int value=getValue(sensorPathLength,sensorPath,"fan2_input");
    drawInteger(52,40,value,0,0);

    value=getValue(sensorPathLength,sensorPath,"temp2_input");
    drawInteger(34,7,value/1000,0,0);
    value=abs((value-(value/1000)*1000)/100);
    drawInteger(43,7,value,0,0);

    value=getValue(sensorPathLength,sensorPath,"temp1_input");
    drawInteger(34,18,value/1000,0,0);
    value=abs((value-(value/1000)*1000)/100);
    drawInteger(43,18,value,0,0);

    value=getValue(sensorPathLength,sensorPath,"in0_input");
    drawInteger(33,51,value/1000,0,0);
    value=abs((value-(value/1000)*1000)/10);
    drawInteger(48,51,value,2,0);

    value=getCPUMHz();
    drawInteger(52,29,value,0,4);

    RedrawWindowXY(0,0);
}  

static char* getPath(char *path,char *device,size_t deviceSize)
{
    char *tmp=strdup(device);
    snprintf(device,deviceSize,"%s/%s",path,tmp);
    return device;
}

static char* getSysSensorPath(char *deviceString,size_t deviceStringSize)
{
    return getPath("/sys/bus/i2c/devices",deviceString,deviceStringSize);
}

static void usage(char *progname)
{
    fprintf(stdout,"WindowMaker CPU Monitor "
		    "%s-%s -d <string> [-p <string>]\n"
		    "\t-d <string>	i2c device (e.g. 1-0290) (default path: /sys/bus/i2c/devices)\n"
		    "\t-p <string>	if default path not supported then use this option\n"
		    "\t-h		this\n"
		    ,progname,v_VERSION);
}

int main(int argc,char *argv[])
{
    FILE *file;
    char line[256];
    char xx[256];
    char deviceString[256]="\0";
    char path[256]="\0";
    char *sensorPath=NULL;
    int revision=0,release=0,patchlevel=0;
    XEvent event;
    int opt;

    if(argc==1) {
	usage(argv[0]);
	exit(0);
    }
    while((opt=getopt(argc,argv,"hp:d:"))!=-1) {
	switch(opt) {
	    case 'p':
		strncpy(path,optarg,sizeof(path));
		break;
	    case 'd':
		strncpy(deviceString,optarg,sizeof(deviceString));
		break;
	    case 'h':
	    default:
		usage(argv[0]);
		exit(0);
	}
    }

    file=fopen("/proc/version","r");
    fgets(line,sizeof(line),file);
    sscanf(line,"%s %s %d.%d.%d%s",xx,xx,&revision,&release,&patchlevel,xx);
    fclose(file);

    createXBMfromXPM(wmcpum_mask_bits,wmcpum_master_xpm,wmcpum_mask_width,wmcpum_mask_height);
    openXwindow(argc,argv,wmcpum_master_xpm,wmcpum_mask_bits,wmcpum_mask_width,wmcpum_mask_height);
    RedrawWindowXY(0,0);

    if(strlen(path)) {
	sensorPath=getPath(path,deviceString,sizeof(deviceString)-1);
    } else {
	sensorPath=getSysSensorPath(deviceString,sizeof(deviceString)-1);
    }
    getValues(sensorPath,sizeof(deviceString)-1);
    while(1) {
	while (XPending(display)) {
	    XNextEvent(display, &event);
	    switch (event.type) {
		case Expose:
		    RedrawWindowXY(0,0);
		    break;
		case DestroyNotify:
		    XCloseDisplay(display);
		    exit(0);
	    }
	}
	getValues(sensorPath,sizeof(deviceString)-1);
	usleep(1000000L);
    }
}
