/* wmchargemon - A battery charge monitor dockapp
 * Copyright (C) 2006, Sergio Di Mico <sedimico@inwind.it>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */ 

#include <string.h>
#include <dockapp.h>
#include <dirent.h>
#include <errno.h>
#include <math.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "img/battery_green.xpm"
#include "img/battery_yellow.xpm"
#include "img/battery_red.xpm"
#include "img/ac.xpm"
#include "img/digits_green.xpm"
#include "img/digits_yellow.xpm"
#include "img/digits_red.xpm"
#include "img/green.xpm"
#include "img/red.xpm"
#include "img/yellow.xpm"
#include "img/gray.xpm"

#ifdef RES_CLASSNAME
	#undef RES_CLASSNAME
#endif
#define RES_CLASSNAME "WMChargeMon"

typedef struct {
	Pixmap background;
	Pixmap digits;
	Pixmap symbol;
} Dress;

void destroy(void);
void parseopts(void);
void set_label(unsigned short, Dress*);
unsigned short read_battery_status(char *);

long optval_ptr[4];
unsigned short	height, width, level_curr,
				level_low, level_critical, update_interval;

char battery[5] = "BAT0";
char battery_acpi_dir_path[24] = "/proc/acpi/battery/", status[12];

Window btn_window;

Pixmap	icon_red, icon_green, icon_yellow,
		mask_red, mask_green, mask_yellow,
		icon_ac, mask_ac, icon_curr, mask_curr,
		icon_digits_red, icon_digits_green, icon_digits_yellow,
		mask_digits_red, mask_digits_green, mask_digits_yellow,
		icon_battery_red, icon_battery_green, icon_battery_yellow,
		mask_battery_red, mask_battery_green, mask_battery_yellow;
GC gc;
XEvent ev;

/*The various widget sets grouped by colors*/
Dress dress_red, dress_yellow, dress_green, dress_green_ac;

/*Configuration options*/
DAProgramOption program_opts[] = {
	{
		"-b", "--battery",
		"The battery to be monitored, defaults to BAT0",
		DOString, False, &optval_ptr[0]
	},
	{
		"-u", "--update",
		"The time interval, in seconds, between the\n\
				updates of the dockapp, defaults to 5",
		DONatural, False, &optval_ptr[1]
	},
	{
		"-l", "--low",
		"The battery percent level to be considered\n\
				low, defaults to 10",
		DONatural, False, &optval_ptr[2]
	},
	{
		"-c", "--critical",
		"The battery percent level to be considered\n\
				critical, defaults to 5",
		DONatural, False, &optval_ptr[3]
	}
};

/******************************************************************************/
int main(int argc, char **argv){

    DACallbacks eventCallbacks = {
		destroy, /* destroy */
		NULL, /* buttonPress */
		NULL, /* buttonRelease */
		NULL, /* motion (mouse) */
		NULL, /* mouse enters window */
		NULL, /* mouse leaves window */
		NULL /* timeout */
    };

	DAParseArguments(
		argc, argv, program_opts, 4,
		"wmchargemon - a battery charge monitor dockapp",
		"0.2 - Version based on libdockapp 0.6.1"
	);

	#ifdef _DEBUG_
		printf("DEBUG: Parsing options...\n");
	#endif
	parseopts();
	#ifdef _DEBUG_
		printf("DEBUG: Battery: %s, Low: %hd, Critical: %hd, "
				"Refresh intv.: %hd\n", battery, level_low,
				level_critical, update_interval);
	#endif

	DASetExpectedVersion(20050420);

	DAInitialize("", "WMChargeMon", 56, 56, argc, argv);

	DASetCallbacks(&eventCallbacks);
	
	gc = DefaultGC(DADisplay, DefaultScreen(DADisplay));

	#ifdef _DEBUG_
		printf("DEBUG: Building Pixmaps...\n");
	#endif
	DAMakePixmapFromData(
		red_xpm,
		&icon_red,
		&mask_red,
		&height,
		&width);
	DAMakePixmapFromData(
		green_xpm,
		&icon_green,
		&mask_green,
		&height,
		&width);
	DAMakePixmapFromData(
		gray_xpm,
		&icon_curr,
		&mask_curr,
		&height,
		&width);
	DAMakePixmapFromData(
		yellow_xpm,
		&icon_yellow,
		&mask_yellow,
		&height,
		&width);
	DAMakePixmapFromData(
		digits_red_xpm,
		&icon_digits_red,
		&mask_digits_red,
		&height,
		&width);
	DAMakePixmapFromData(
		digits_green_xpm,
		&icon_digits_green,
		&mask_digits_green,
		&height,
		&width);
	DAMakePixmapFromData(
		digits_yellow_xpm,
		&icon_digits_yellow,
		&mask_digits_yellow,
		&height,
		&width);
	DAMakePixmapFromData(
		battery_red_xpm,
		&icon_battery_red,
		&mask_battery_red,
		&height,
		&width);
	DAMakePixmapFromData(
		battery_green_xpm,
		&icon_battery_green,
		&mask_battery_green,
		&height,
		&width);
	DAMakePixmapFromData(
		battery_yellow_xpm,
		&icon_battery_yellow,
		&mask_battery_yellow,
		&height,
		&width);
	DAMakePixmapFromData(
		ac_xpm,
		&icon_ac,
		&mask_ac,
		&height,
		&width);

	/*Associating the dresses with the right components*/
	dress_red.background = icon_red;
	dress_red.digits = icon_digits_red;
	dress_red.symbol = icon_battery_red;
	dress_yellow.background = icon_yellow;
	dress_yellow.digits = icon_digits_yellow;
	dress_yellow.symbol = icon_battery_yellow;
	dress_green.background = icon_green;
	dress_green.digits = icon_digits_green;
	dress_green.symbol = icon_battery_green;
	dress_green_ac.background = icon_green;
	dress_green_ac.digits = icon_digits_green;
	dress_green_ac.symbol = icon_ac;
	
	DAShow();

	while (1) {
		#ifdef _DEBUG_
			printf("Starting update cycle.\n");
			printf("DEBUG: Reading battery level.\n");
		#endif
		level_curr = read_battery_status(battery_acpi_dir_path);
		
		if (!strcmp(status, "discharging")) {
			if (level_curr < level_low + 1) {
				set_label(
					level_curr,
					&dress_yellow);
				DASetPixmap(icon_curr);
			}
			if (level_curr < level_critical + 1) {
				set_label(
					level_curr,
					&dress_red);
				DASetPixmap(icon_curr);
			}
			if (level_curr > level_low) {
				set_label(
					level_curr,
					&dress_green);
				DASetPixmap(icon_curr);
			}
		} else {
			set_label(
				level_curr,
				&dress_green_ac);
			DASetPixmap(icon_curr);
		}
		sleep(update_interval);
		
		/* handle all pending X events */
		while (XPending(DADisplay)) {
				XNextEvent(DADisplay, &ev);
				DAProcessEvent(&ev);
		}
		usleep(10000);
	}

/*I prefer not to use it because it's CPU-hungry*/
//	DAEventLoop();
	return 0;
}

/******************************************************************************/
void parseopts(void) {
	DIR *battery_acpi_dir_handle;
	Bool found_flag;
	struct dirent *dir_index;

	/*Set default value for the options, and possibly change it*/
	sprintf(battery, "BAT0");
	if (program_opts[0].used == True) {
		if  ((strlen(program_opts[0].value.string[0]) == 4) &&
			(strncmp(program_opts[0].value.string[0], "BAT", 3) == 0))
			sprintf(battery, program_opts[0].value.string[0]);
	}
	/*Input validation*/
	battery_acpi_dir_handle = opendir(battery_acpi_dir_path);
	if (battery_acpi_dir_handle == NULL) {
		switch (errno) {
			case ENOENT:
				fprintf(stderr, "Sorry. No ACPI support for your battery.\n");
				break;
			case EACCES:
				fprintf(stderr, "Sorry. Permission denied while reading %s.\n",
					battery_acpi_dir_path);
				break;
			default: 
				fprintf(stderr, "Sorry. An error occurred while reading %s.\n",
					battery_acpi_dir_path);
				break;
		}
		exit(1);
	}

	found_flag = False;

	errno = 0;

	while ((dir_index = readdir(battery_acpi_dir_handle)) != NULL) {
		if (!strcmp(dir_index->d_name, battery)) {
			found_flag = True;
			break;
		}
	}

	/*Error or normal end of scan?*/
	switch (errno) {
		case 0: break;
		case EBADF:
			fprintf(stderr, "Sorry. Error while reading %s.\n",
				battery_acpi_dir_path);
			break;
			exit(1);
	}

	if (found_flag == False) {
		fprintf(stderr, "Sorry. Battery %s is not present. Try to specify\n"
		"another one with the option \"-b\"\n", battery);
		free(dir_index);
		free(battery_acpi_dir_handle);
		exit(1);
	}
	
	sprintf(battery_acpi_dir_path, "/proc/acpi/battery/%s", battery);
	free(dir_index);
	free(battery_acpi_dir_handle);
	
	update_interval = 5;
	if (program_opts[1].used == True) {
		if (*(program_opts[1].value.integer) > 0 &&
			*(program_opts[1].value.integer) < 65536)
			update_interval = (unsigned short) *(program_opts[1].value.integer);
		else {
			fprintf(stderr, "Sorry. The update interval must be a value\n"
					"between 0 and 65535.\n");
			destroy();
			exit(1);
		}
	}
	
	level_low = 10;
	if (program_opts[2].used == True) {
		if (*(program_opts[2].value.integer) > 0 &&
			*(program_opts[2].value.integer) < 100)
			level_low = (unsigned short) *(program_opts[2].value.integer);
		else {
			fprintf(stderr, "Sorry. The low battery level must be a value\n"
					"between 0 and 100.\n");
			destroy();
			exit(1);
		}
	}
	
	level_critical = 5;
	if (program_opts[3].used == True) {
		if (*(program_opts[3].value.integer) > 0 &&
			*(program_opts[3].value.integer) < level_low)
			level_critical = (unsigned short) *(program_opts[3].value.integer);
		else {
			fprintf(stderr,"Sorry. The critical battery level must be a value\n"
					"between 0 and %hd, unless a different value for the\n"
					"low battery level is specified with the option \"-l\"\n",
					level_low);
			destroy();
			exit(1);
		}
	}
	destroy();
	return;
}

/******************************************************************************/
unsigned short read_battery_status(char *battery_path){
	FILE *batt_file;
	unsigned batt_curr = 0, batt_max = 0;
	char batt_file_path[31], batt_file_line[40];
	Bool capacity_flag, state_flag;

	memset(status, 0, 12); 
	sprintf(batt_file_path, "%s/info", battery_path);

	#ifdef _DEBUG_
		printf("DEBUG: Opening file: %s\n", batt_file_path);
	#endif
	batt_file = fopen(batt_file_path, "r");

	if (batt_file == NULL) {
		switch (errno) {
			case EACCES:
				fprintf(stderr, "Sorry. Access denied to %s.\n",
					batt_file_path);
				break;
			default:
				fprintf(stderr, "Sorry. Error while reading %s.\n",
					batt_file_path);
				break;
		}
		exit(1);
	} 

	while (!feof(batt_file)) {
		memset(batt_file_line, 0, 40);
		fgets(batt_file_line, 40, batt_file);
		#ifdef _DEBUG_
			printf("DEBUG: Current Line: %s", batt_file_line);
		#endif
		if (strncmp(batt_file_line, "last full capacity:", 19) == 0) {
			sscanf(batt_file_line, "%*19c %u %*s", &batt_max);
			#ifdef _DEBUG_
				printf("DEBUG: Found maximum capacity: %hd mWh.\n", batt_max);
			#endif
			break;
		}
	}

	fclose(batt_file);

	sprintf(batt_file_path, "%s/state", battery_path);
	
	#ifdef _DEBUG_
		printf("DEBUG: Opening file: %s.\n", batt_file_path);
	#endif
	batt_file = fopen(batt_file_path, "r");

	if (batt_file == NULL) {
		switch (errno) {
			case EACCES:
				fprintf(stderr, "Sorry. Access denied to %s.\n",
					batt_file_path);
				break;
			default:
				fprintf(stderr, "Sorry. Error while reading %s.\n",
					batt_file_path);
				break;
		}
		exit(1);
	}

	capacity_flag = False;
	state_flag = False;

	while (!feof(batt_file)) {
		memset(batt_file_line, 0, 40);
		fgets(batt_file_line, 40, batt_file);
		#ifdef _DEBUG_
			printf("DEBUG: Current line: %s", batt_file_line);
		#endif
		if (strncmp(batt_file_line, "charging state:", 15) == 0) {
			sscanf(batt_file_line, "%*15c %s", status);
			#ifdef _DEBUG_
				printf("DEBUG: Found charging state: %s.\n", status);
			#endif
			state_flag = True;
		}
		if (strncmp(batt_file_line, "remaining capacity:", 19) == 0) {
			sscanf(batt_file_line, "%*19c %u %*s", &batt_curr);
			#ifdef _DEBUG_
				printf("DEBUG: Found current capacity: %hd mWh.\n", batt_curr);
			#endif
			capacity_flag = True;
		}
		if (capacity_flag == True && state_flag == True) break;
	}

	fclose(batt_file);

	if (batt_max == 0) {
		fprintf(stderr, "Sorry. Error while determining the maximum capacity "
			"of this battery.\n");
		exit(1);
	}

	#ifdef _DEBUG_
		printf("DEBUG: Current battery level: %hd%%.\n",
			(unsigned short) (batt_curr * 100 / batt_max));
	#endif

	return (unsigned short) (batt_curr * 100 / batt_max);
}

/******************************************************************************/
void destroy(void){
	return;
}

/******************************************************************************/
void set_label(unsigned short value, Dress *dress) {
	unsigned char positions, display_posx[4];
	unsigned short curr_digit;

	if (value < 100 && value > 9) {
		positions = 2;
		display_posx[2] = 7;
		display_posx[1] = 21;
		display_posx[0] = 35;
	} else
	if (value < 10) {
		positions = 1;
		display_posx[1] = 14;
		display_posx[0] = 28;
	}
	else {
		positions = 3;
		display_posx[3] = 0;
		display_posx[2] = 14;
		display_posx[1] = 28;
		display_posx[0] = 42;
	} 
	
	/*Erase the display*/
	XCopyArea(
		DAGetDisplay(NULL),
		dress->background,
		icon_curr,
		gc,
		0,
		0,
		56,
		56,
		0,
		0	
	);
	
	while (positions != 0) {
		curr_digit = value / ((unsigned short) pow(10, --positions));
		value -= ((unsigned short) pow(10, positions)) * curr_digit;
		XCopyArea(
			DAGetDisplay(NULL),
			dress->digits,
			icon_curr,
			gc,
			14 * curr_digit,
			0,
			14,
			24,
			display_posx[positions + 1],
			7	
		);
		XFlush(DADisplay);
	}

	/*Display the percent sign*/
	XCopyArea(
		DAGetDisplay(NULL),
		dress->digits,
		icon_curr,
		gc,
		140,
		0,
		14,
		24,
		display_posx[0],
		7
	);
	
	/*Display the symbol below the numbers*/
	XCopyArea(
		DAGetDisplay(NULL),
		dress->symbol,
		icon_curr,
		gc,
		0,
		0,
		52,
		20,
		2,
		32
	);
	
	XFlush(DADisplay);
	return;
}
