/*  ---------------------------------------------------------------
    wmALMS.  Another LM Sensors applet.
    Copyright (C) 2001,  Michael Glickman  <wmalms@yahoo.com>
    License: GPL
    --------------------------------------------------------------- */

#include "wmalms.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

#if HAVE_LIMITS_H
#include <limits.h>
#endif

#ifndef LONG_MAX
#define LONG_MAX 0x7fffffff
#endif

#ifndef LONG_MIN
#define LONG_MIN (-LONG_MAX - 1)
#endif

#if HAVE_SENSORS_SENSORS_H
# define USE_LM_SENSORS
# include <sensors/sensors.h>
# include <sensors/chips.h>
# include <sensors/error.h>
# define DEFAULT_CONFIG_FILE_NAME "sensors.conf"
const sensors_chip_name *chip_nm;
#endif

extern int CurPane;
extern int ValueCount;
extern int PaneCount;
extern int PaneValueCount;
extern int WndSize, WndNormSize;
extern Bool Fahrenheit;

extern const int FldSeparator;


/*
   Temperature value are 10-scaled.
   Fan speeds are always integral and just rounded 
*/

LMS_VALUE lmsval[VALUE_COUNT_MAX];
char *DevName = NULL;
int ProcessAlarm;

/* -------------------------------------------------------------------------- */
#ifdef USE_LM_SENSORS
static int LookupNumber(int number, int start_index)
{
  int i;
  int number1;
  char *delim;
  

  for (i=start_index; i<ValueCount; i++)  
  {
	number1 = strtol(lmsval[i].name, &delim, 0);
	if (*delim == '\0' && (number1 == number))
	  return i;
  }
  
  return -1;
  

}

static int LookupName(const char *name, int start_index)
{
  int i;
  const char *name1;

  for (i=start_index; i<ValueCount; i++)  
  {
	name1 = lmsval[i].name;
	if (name1 != NULL && strcasecmp(lmsval[i].name, name) == 0)
	  return i;
  }
  
  return -1;
  

}

static void ProcessEntry(int number, int entry_no, const char *name, const char *label, int type)
{
  double value;
  int decimals;

  if (type == 0)
  {
    memcpy(lmsval[entry_no].name, name,  7);
	*(lmsval[entry_no].name+7) = '\0';


    if (*(lmsval[entry_no].label) == '\0')
	{
  	  memcpy(lmsval[entry_no].label, label,  4);
	  *(lmsval[entry_no].label+4) = '\0';
    }	
  }	

  switch(type)
  {
	case 0:
      lmsval[entry_no].number = number;
	  if (lmsval[entry_no].decimals < 0)
	  {
  	    if (memcmp(name, "fan", 3) == 0)
		  decimals = 0;
		else
		  decimals = 1;		  
    	lmsval[entry_no].decimals = decimals;
	  }		
	  break;
	  
	  
	case -1:
      lmsval[entry_no].number_min = number;
	  if (lmsval[entry_no].min_value == LONG_MIN &&
		  sensors_get_feature(*chip_nm, number, &value) >= 0) 
	  {
		decimals = lmsval[entry_no].decimals;
		while (--decimals >= 0) value *= 10.;
		lmsval[entry_no].min_value = (long) value;
	  }		

	  break;
  
	default:
      lmsval[entry_no].number_max = number;
	  if (lmsval[entry_no].max_value == LONG_MAX &&
		  sensors_get_feature(*chip_nm, number, &value) >= 0) 
	  { decimals = lmsval[entry_no].decimals;
		while (--decimals >= 0) value *= 10.;
		lmsval[entry_no].max_value = (long) value;
	  }		
	  break;
  
  }
}

static Bool ProcessFeature(const sensors_feature_data *sfd, const char *name, int type)
{
  int entry_no;
  const char *label;
  int number;
  Bool Found;

  number = sfd->number;

  if (sensors_get_ignored(*chip_nm, number) <= 0) return False;

  if (sensors_get_label(*chip_nm, number, (char **) &label) < 0) label = name;

  Found = False;  

  entry_no = -1;  
  while ((entry_no = LookupName(name, entry_no+1)) >=  0)
  {
  	Found = True;
	ProcessEntry(number, entry_no, name, label, type);
  }	

  if (type == 0)
  {
    entry_no = -1;  
    while ((entry_no = LookupName(label, entry_no+1)) >=  0)
	{
	  Found = True;
	  ProcessEntry(number, entry_no, name, label, type);
    }	

    entry_no = -1;  
    while ((entry_no = LookupNumber(number, entry_no+1)) >=  0)
	{
	  Found = True;
	  ProcessEntry(number, entry_no, name, label, type);
    }	
	
	
  }
  
  return Found;
}

#endif


void ClearLMS(void)
{
  int i;
  for (i=0; i<ValueCount; i++)
  {
	memset(&(lmsval[i]), '\0', sizeof(LMS_VALUE));
	strcpy(lmsval[i].name ,"");
	strcpy(lmsval[i].label ,"");

	lmsval[i].number = -1;
	lmsval[i].number_max = -1;
	lmsval[i].number_min = -1;
	lmsval[i].min_value = LONG_MIN;
	lmsval[i].max_value = LONG_MAX;
	lmsval[i].old_value = LONG_MIN;
	lmsval[i].new_value = 0;
	lmsval[i].decimals = -1;
	lmsval[i].alarm = 1;
	lmsval[i].alarm_stat = 1;
  }

}

Bool ParseLMSPref(int pos, char *pref)
{ 
  char *ptr;
  int len;
  long val;
  char *eptr;
  double prefd;

  /* Getting name */  
  ptr = strchr(pref, FldSeparator);
  /* prtnext = strpbrk(pref, ";"); */
  if (ptr) *ptr++ = '\0';

  len = strlen(pref);	 
  if (len >0 && len <= 7)
	strcpy(lmsval[pos].name, pref);
  if (ptr==NULL) return True;    
  
  /* Getting title */  
  ptr = strchr(pref=ptr, FldSeparator);
  if (ptr) *ptr++ = '\0';

  len = strlen(pref);	 
  if (len > 0 && len <= 4)
    strcpy(lmsval[pos].label, pref);
  if (ptr==NULL) return True;    

  /* Getting decimals */  
  ptr = strchr(pref=ptr, FldSeparator);
  if (ptr) *ptr++ = '\0';

  len = strlen(pref);	 
  if (len == 1 && isdigit(*pref))
  {	len = *pref - '0';
	if (len >= 0 && len <= 3)
      lmsval[pos].decimals = len;
  }
  if (ptr==NULL) return True;    
  
  /* Getting min value */  
  ptr = strchr(pref=ptr, FldSeparator);
  if (ptr) *ptr++ = '\0';

  prefd = strtod(pref, &eptr);
  if (*eptr == '\0')
  {
	  val = lmsval[pos].decimals;
	  while (--val >= 0) prefd *= 10.;
      lmsval[pos].min_value = (long) prefd;
  }	  
  if (ptr==NULL) return True;    

  /* Getting max value */  
  ptr = strchr(pref=ptr, FldSeparator);
  if (ptr) *ptr++ = '\0';

  prefd = strtod(pref, &eptr);
  if (*eptr == '\0')
  {
	  val = lmsval[pos].decimals;
	  while (--val >= 0) prefd *= 10.;
      lmsval[pos].max_value = (long) prefd;
  }	  
  if (ptr==NULL) return True;    

  /* Getting alarm switch */  
  if (strchr("Nn0", *ptr) != NULL)
      lmsval[pos].alarm = 0;
  else	  
  if (strchr("Yy1", *ptr) != NULL)
      lmsval[pos].alarm = 1;
  
  return True;
	
}			

Bool InitLMS(void)
{
#ifdef USE_LM_SENSORS
  char cur_path[256];		/* This should suffice in our case */
  const char **cur_dir_ptr;	
  const char *cur_dir;
  FILE *f = NULL;
  int	chip_nr1, chip_nr2;
  int  res;
  const sensors_feature_data *sfd;
  char *name, *ptr;
  int  type;
  Bool found_device;

  static const char *config_file_dir[] = 
  { "/etc", "/usr/lib/sensors", "/usr/local/lib/sensors", "/usr/lib",
  	"/usr/local/lib", ".", NULL };
#endif

#ifdef USE_LM_SENSORS
  /* Process with config file */

  cur_dir_ptr = config_file_dir;

  while (f==NULL && (cur_dir=*cur_dir_ptr++) != NULL)
  {	
	sprintf (cur_path, "%s/%s", cur_dir, DEFAULT_CONFIG_FILE_NAME);
	f = fopen(cur_path,"r");
  }			

  if (f == NULL)
  {	fprintf (stderr, "Couldn't locate '%s'\n", DEFAULT_CONFIG_FILE_NAME);
		return False;
  }	

  res = sensors_init(f);
  fclose(f);

  if (res != 0)
  {
	fprintf(stderr,"%s\n",sensors_strerror(res));
  	if (res == -SENSORS_ERR_PROC)
	fprintf(stderr,
  	  "/proc/sys/dev/sensors/chips or /proc/bus/i2c unreadable;\n"
      "Make sure you have done 'modprobe i2c-proc'!\n");
		return False;	      
  }

  chip_nr1 = 0; 
  found_device = False;
  
   
  while(!found_device &&
         (chip_nm = sensors_get_detected_chips(&chip_nr1)) != NULL)
  {
    if (DevName == NULL || strcasecmp(DevName, "any") == 0)
	  found_device = (strcmp(chip_nm->prefix, "eeprom") != 0);
	else
    if (strchr(DevName, '-'))
    {
	  if (chip_nm->bus == SENSORS_CHIP_NAME_BUS_ISA)
    	sprintf(cur_path, "%s-isa-%04x",chip_nm->prefix,chip_nm->addr);
	  else
    	sprintf(cur_path, "%s-i2c-%d-%02x",chip_nm->prefix,chip_nm->bus,chip_nm->addr);
			
	  found_device = (strcmp(cur_path, DevName) == 0);
	}			
	else		
	  found_device = (strcmp(chip_nm->prefix, DevName) == 0);
  }	  

  if (chip_nm == NULL)
  {
	if (DevName != NULL)
	  fprintf(stderr, "Sensor chip '%s' is not found!\n", DevName);
	else	      
	  fprintf(stderr, "Sensor chip is not found!\n");
	return False;
  }	


  /* Found chip! Now process its devices */
  chip_nr1 = chip_nr2 = 0;         
  
  while((sfd=sensors_get_all_features(*chip_nm, &chip_nr1, &chip_nr2)) != NULL)
  {
	name = strdup(sfd->name);
	ptr = strstr(name, "_min");
	if (ptr) 
	{	*ptr = '\0';
	  	type = -1;
	}
	else
	{
	  ptr = strstr(name, "_max");
  	  if (ptr==NULL) ptr=strstr(name, "_over");
	  if (ptr) 
	  {	*ptr = '\0';
	  	type = 1;
	  }
	  else
		type = 0;
	}
	  
	ProcessFeature(sfd, name, type);
	free(name);
	
  }
#endif

    return True;
}


void ShowValue(int pos, long value, int decimals, Bool alarm)
{
   
  int x, y, y1, v, i, k;
  char	txt[11];
  char *txtptr;
  char dig;
  int nwid, nhei;

  sprintf(txt, "%4ld", value);
  txtptr = txt + (strlen(txt)-4);
  nhei = 7;

  if (WndSize == WndNormSize)
  {  y = 6 + 14*pos;
	 x = 35;
     nwid = 5;
  }
  else
  {  y = 2 + 12*pos;
	 x = 27;
     nwid = 4;
  }
  
  if (decimals > 0)
	x -= 3;

  k = 4-decimals;

  y1 = WndSize;
  if (alarm) y1 += nhei;
  	
    
  for (i=0; i<k; i++)
  {
	dig = txtptr[i];
	v= (dig == ' ') ? 10 : (dig == '-' ? 11 : txtptr[i]-'0');
    CopyPixmap(v*nwid,y1,nwid, nhei,x,y);
	x += nwid+1;
  }

  if (decimals > 0)
  {
	CopyPixmap(nwid*12, y1, 3, 7, x, y);
	x +=3;
    for (i= k; i<4; i++)
	{
	  dig = txtptr[i];
	  v= (dig == ' ') ? 10 : txtptr[i]-'0';
  	  CopyPixmap(v*nwid, y1,nwid,nhei,x,y);
	  x += nwid+1;
    }
  }

  
}

void UpdateLMS(void)
{
#ifdef USE_LM_SENSORS
  double	d;
  int	number;
  int  decimals;
#endif
  int i;
  long value;

  for (i=0; i<ValueCount; i++)
  {

#ifdef USE_LM_SENSORS
	number = lmsval[i].number;
	if (number >= 0) 
	{
	    sensors_get_feature(*chip_nm, number, &d);
		decimals = lmsval[i].decimals;
		while (--decimals >= 0) d *= 10.;
		value = (long)d;
	}
	else
#endif
		value = 0;

	lmsval[i].new_value = value;
  }
}

Bool ShowLMS(void)
{

  int i, i1;
  int istart;
  int  decimals;
  Bool lmsalarm[PANE_COUNT_MAX];
  long value, minv, maxv;
  
  Bool alarm_stat, global_alarm_stat;
  
  for (i=0; i<PaneCount; i++)
	lmsalarm[i] = False;
  
  global_alarm_stat = False; 

  for (i=0; i<ValueCount; i++)
  {
	value = lmsval[i].new_value;
	maxv = lmsval[i].max_value;		
	minv = lmsval[i].min_value;		

    if (maxv >= minv)
		alarm_stat = (value < minv) || (value > maxv);		
	else
		alarm_stat = (value < maxv) || (value > minv);		
		
	lmsval[i].alarm_stat = alarm_stat;
	
	if (alarm_stat && lmsval[i].alarm)
	{
	  lmsalarm[i / PaneValueCount] = True;
	  global_alarm_stat = True; 
	}  
  }
  	
  SetGlobalAlarmState(global_alarm_stat);
  
  if (ProcessAlarm && !lmsalarm[CurPane])
  {
	i = CurPane;
	do i = (i + 1) % PaneCount;
	while (i != CurPane && lmsalarm[i] == False); 

	SetCurrentPage(i);
  }
  
  istart = CurPane * PaneValueCount;

  for (i=0; i<PaneValueCount; i++)
  {
	i1 = istart + i;
    value = lmsval[i1].new_value;
    if (value != lmsval[i1].old_value)
    {
      if (Fahrenheit && memcmp(lmsval[i1].name, "temp", 4) == 0)
	  {
		minv = 32; 
		decimals = lmsval[i1].decimals;
		while (--decimals >= 0) minv *= 10.;
	    value = (value * 9)/5 + minv;
	  }		
	  ShowValue(i, value, lmsval[i1].decimals, lmsval[i1].alarm_stat);
	  lmsval[i1].old_value = value;
	}
  }      

  return True;
}
