.\" t
.\" @(#)asimage.3		08/01/2007
.TH asimage 1 "AfterStep v.2.2.6" "Aug  1 2007" "AfterStep X11 window manager"
.UC
.SH NAME
\fBasimage\fP\ - internal structures and methods used for image manipulation in libAfterImage
libAfterImage/asimage.h
.SH NAMEasimage
defines main structures and function for image manipulation.

.SH DESCRIPTION
.nf
libAfterImage provides powerful functionality to load, store
and transform images. It allows for smaller memory utilization by
utilizing run-length encoding of the image data. There could be
different levels of compression selected, allowing to choose best
speed/memory ratio.
.fi 
.SH SEE ALSO
.nf
Structures :
         ASImage
         ASImageManager
         ASImageBevel
         ASImageDecoder
         ASImageOutput
         ASImageLayer
         ASGradient

Functions :
         asimage_init(), asimage_start(), create_asimage(),
         clone_asimage(), destroy_asimage()

  ImageManager Reference counting and managing :
         create_image_manager(), destroy_image_manager(),
         store_asimage(), fetch_asimage(), query_asimage(),
         dup_asimage(), release_asimage(),
         release_asimage_by_name(), forget_asimage(),
         safe_asimage_destroy()

  Gradients helper functions :
         flip_gradient(), destroy_asgradient()

  Layers helper functions :
         init_image_layers(), create_image_layers(),
         destroy_image_layers()

  Encoding :
         asimage_add_line(),    asimage_add_line_mono(),
         asimage_print_line(), get_asimage_chanmask(),
         move_asimage_channel(), copy_asimage_channel(),
         copy_asimage_lines()

  Decoding
         start_image_decoding(), stop_image_decoding(),
         asimage_decode_line (), set_decoder_shift(),
         set_decoder_back_color()

  Output :
         start_image_output(), set_image_output_back_color(),
         toggle_image_output_direction(), stop_image_output()

Other libAfterImage modules :
         ascmap.h asfont.h asimage.h asvisual.h blender.h export.h
         import.h transform.h ximage.h
.fi 
.SH AUTHOR
.nf
Sasha Vasko <sasha at aftercode dot net>
.fi libAfterImage/ASAltImFormats
.SH NAMEASAltImFormats
identifies what output format should be used for storing the transformation result. Also identifies what data is currently stored in alt member of ASImage structure.

.SH SOURCE
Source : 
.nf
typedef enum {
    ASA_ASImage = 0,
    ASA_XImage,
    ASA_MaskXImage,
    /* temporary XImages to be allocated from static pool of memory :*/
    ASA_ScratchXImage,  
    ASA_ScratchMaskXImage,
    
    ASA_ScratchXImageAndAlpha,

    ASA_ARGB32,
    ASA_Vector,       /* This cannot be used for transformation's result
                       * format */
    ASA_Formats
}ASAltImFormats;
.fi libAfterImage/ASImage
.SH NAMEASImage
is the main structure to hold image data.

.SH DESCRIPTION
.nf
Images are stored internally split into ARGB channels, each split
into scanline. Actuall data is stored using ASStorage container. Inside
ASImage data structure we only store IDs pointing to data in ASStorage
ASStorage implements reference counting, data compression, 
automatic memory defragmentation and other nice things.
.fi 
.SH SEE ALSO
.nf
 asimage_init()
 asimage_start()
 create_asimage()
 destroy_asimage()
.fi 
.SH SOURCE
Source : 
.nf
struct ASImageAlternative;
struct ASImageManager;

/* magic number identifying ASFont data structure */
#define MAGIC_ASIMAGE            0xA3A314AE

typedef struct ASImage
{

  unsigned long magic ;

  unsigned int width, height;       /* size of the image in pixels */

  /* arrays of storage ids of stored scanlines of particular channel: */
  ASStorageID *alpha,
              *red,
              *green,
              *blue;
  
  ASStorageID *channels[IC_NUM_CHANNELS]; 
                                    /* merely a shortcut so we can
                                     * somewhat simplify code in loops */

  ARGB32 back_color ;               /* background color of the image, so
                                     * we could discard everything that
                                     * matches it, and then restore it
                                     * back. */

  struct ASImageAlternative
  {  /* alternative forms of ASImage storage : */
    XImage *ximage ;                /* pointer to XImage created as the
                                     * result of transformations whenever
                                     * we request it to output into
                                     * XImage ( see to_xim parameter ) */
    XImage *mask_ximage ;           /* XImage of depth 1 that could be
                                     * used to store mask of the image */
    ARGB32 *argb32 ;                /* array of widthxheight ARGB32
                                     * values */
    double *vector ;                /* scientific data that should be used
                                     * in conjunction with
                                     * ASScientificPalette to produce
                                     * actuall ARGB data */
  }alt;

  struct ASImageManager *imageman;  /* if not null - then image could be
                                     * referenced by some other code */
  int                    ref_count ;/* this will tell us what us how many
                                     * times */

    
  char                  *name ;     /* readonly copy of image name 
                                     * this name is a hash value used to 
                                     * store image in the image-man's hash,
                                     * and gets freed automagically on image 
                                     * removal from hash */

#define ASIM_DATA_NOT_USEFUL    (0x01<<0)
#define ASIM_VECTOR_TOP2BOTTOM  (0x01<<1)
#define ASIM_XIMAGE_8BIT_MASK   (0x01<<2)
#define ASIM_NO_COMPRESSION     (0x01<<3) /* Do not use compression to 
                                           * save some computation time
                                           */
#define ASIM_ALPHA_IS_BITMAP    (0x01<<4) 
#define ASIM_RGB_IS_BITMAP      (0x01<<5) 
#define ASIM_XIMAGE_NOT_USEFUL  (0x01<<6)
#define ASIM_NAME_IS_FILENAME   (0x01<<7)

  ASFlagType             flags ;    /* combination of the above flags */
  
} ASImage;
.fi libAfterImage/ASImageManager
.SH NAMEASImageManager
structure to be used to maintain list of loaded images for given set of search paths and gamma. Images are named and reference counted.

.SH SOURCE
Source : 
.nf
typedef struct ASImageManager
{
    ASHashTable  *image_hash ;
    /* misc stuff that may come handy : */
    char         *search_path[MAX_SEARCH_PATHS+1];
    double        gamma ;
}ASImageManager;
.fi libAfterImage/LIMITS
.SH NAMEMAX_IMPORT_IMAGE_SIZE
effectively limits size of the allowed images to be loaded from files. That is needed to be able to filter out corrupt files.
MAX_BEVEL_OUTLINE
Limit on bevel outline to be drawn around the image.
MAX_SEARCH_PATHS
Number of search paths to be used while loading images from files.

Source : 
.nf
#define MAX_IMPORT_IMAGE_SIZE   8000
#define MAX_BEVEL_OUTLINE       100
#define MAX_SEARCH_PATHS        8      /* prudently limiting ourselfs */
.fi libAfterImage/ASVectorPalette
.SH NAMEASVectorPalette
contains pallette allowing us to map double values in vector image data into actuall ARGB values.

.SH SOURCE
Source : 
.nf
typedef struct ASVectorPalette
{
    unsigned int npoints ;
    double *points ;
    CARD16 *channels[IC_NUM_CHANNELS] ;   /* ARGB data for key points. */
    ARGB32  default_color;
}ASVectorPalette;
.fi libAfterImage/asimage/ASImageLayer
.SH NAMEASImageLayer
specifies parameters of the image superimposition.

.SH DESCRIPTION
.nf
libAfterImage allows for simultaneous superimposition (overlaying) of
arbitrary number of images. To facilitate this ASImageLayer structure
has been created in order to specify parameters of each image
participating in overlaying operation. Images need not to be exact
same size. For each image its position on destination is specified
via dst_x and dst_y data members. Each image maybe tiled and clipped
to fit into rectangle specified by clip_x, clip_y, clip_width,
clip_height ( in image coordinates - not destination ). If image is
missing, then area specified by dst_x, dst_y, clip_width, clip_height
will be filled with solid_color.
Entire image will be tinted using tint parameter prior to overlaying.
Bevel specified by bevel member will be drawn over image prior to
overlaying. Specific overlay method has to be specified.
merge_scanlines method is pointer to a function,
that accepts 2 ASScanlines as arguments and performs overlaying of
first one with the second one.
There are 15 different merge_scanline methods implemented in
libAfterImage, including alpha-blending, tinting, averaging,
HSV and HSL colorspace operations, etc.
.fi 
.SH NOTES
.nf
ASImageLayer s could be organized into chains using next pointers.
Since there could be a need to rearrange layers and maybe bypass some
layers - we need to provide for flexibility, while at the same time
allowing for simplicity of arrays. As the result next pointers could
be used to link together continuous arrays of layer, like so :
array1: [layer1(next==NULL)][layer2(next!=NULL)]
         ____________________________|
         V
array2: [layer3(next==NULL)][layer4(next==NULL)][layer5(next!=NULL)]
         ________________________________________________|
         V
array3: [layer6(next==NULL)][layer7(next==layer7)]
                               ^______|

While iterating throught such a list we check for two conditions -
exceeding count of layers and layer pointing to self. When any of
that is met - we stopping iteration.
.fi 
.SH SEE ALSO
.nf
merge_layers()
blender.h
.fi 
.SH SOURCE
Source : 
.nf
typedef struct ASImageLayer
{
    ASImage *im;
    ARGB32   solid_color ;              /* If im == NULL, then fill
                                         * the area with this color. */

    int dst_x, dst_y;                   /* placement in overall
                                         * composition */

    /* clip area could be partially outside of the image -
     * image gets tiled in it */
    int clip_x, clip_y;
    unsigned int clip_width, clip_height;

    ARGB32 tint ;                       /* if 0 - no tint */
    struct ASImageBevel *bevel ;        /* border to wrap layer with
                                         * (for buttons, etc.)*/

    /* if image is clipped then we need to specify offsets of bevel as
     * related to clipped rectangle. Normally it should be :
     * 0, 0, im->width, im->height. And if width/height left 0 - it will
     * default to this values. Note that clipped image MUST be entirely
     * inside the bevel rectangle. !!!*/
    int bevel_x, bevel_y;
    unsigned int bevel_width, bevel_height;

    int merge_mode ;                        /* reserved for future use */
    merge_scanlines_func merge_scanlines ;  /* overlay method */
    struct ASImageLayer *next;              /* optional pointer to next
                                             * layer. If it points to
                                             * itself - then end of the
                                             * chain.*/
    void *data;                             /* hook to hung data on */
}ASImageLayer;
.fi libAfterImage/asimage/GRADIENT_TYPE_flags
.SH FUNCTION
.nf
Combination of this flags defines the way gradient is rendered.
.fi 
.SH NAMEGRADIENT_TYPE_DIAG
when set it will cause gradient's direction to be rotated by 45 degrees
GRADIENT_TYPE_ORIENTATION
will cause gradient direction to be rotated by 90 degrees. When combined with GRADIENT_TYPE_DIAG - rotates gradient direction by 135 degrees.

.SH SOURCE
Source : 
.nf
#define GRADIENT_TYPE_DIAG          (0x01<<0)
#define GRADIENT_TYPE_ORIENTATION   (0x01<<1)
#define GRADIENT_TYPE_MASK          (GRADIENT_TYPE_ORIENTATION| \
                                     GRADIENT_TYPE_DIAG)
.fi libAfterImage/asimage/GRADIENT_TYPE
.SH FUNCTION
.nf
This are named combinations of above flags to define type of gradient.
.fi 
.SH NAMEGRADIENT_Left2Right
normal left-to-right gradient.
GRADIENT_TopLeft2BottomRight
diagonal top-left to bottom-right.
GRADIENT_Top2Bottom
vertical top to bottom gradient.
GRADIENT_BottomLeft2TopRight
diagonal bottom-left to top-right.

.SH SOURCE
Source : 
.nf
#define GRADIENT_Left2Right             0
#define GRADIENT_TopLeft2BottomRight    GRADIENT_TYPE_DIAG
#define GRADIENT_Top2Bottom             GRADIENT_TYPE_ORIENTATION
#define GRADIENT_BottomLeft2TopRight    (GRADIENT_TYPE_DIAG| \
                                         GRADIENT_TYPE_ORIENTATION)
.fi libAfterImage/ASGradient
.SH NAMEASGradient
describes how gradient is to be drawn.

.SH DESCRIPTION
.nf
libAfterImage includes functionality to draw multipoint gradients in
4 different directions left->right, top->bottom and diagonal
lefttop->rightbottom and bottomleft->topright. Each gradient described
by type, number of colors (or anchor points), ARGB values for each
color and offsets of each point from the beginning of gradient in
fractions of entire length. There should be at least 2 anchor points.
very first point should have offset of 0. and last point should have
offset of 1. Gradients are drawn in ARGB colorspace, so it is possible
to have semitransparent gradients.
.fi 
.SH SEE ALSO
.nf
make_gradient()
.fi 
.SH SOURCE
Source : 
.nf
typedef struct ASGradient
{
    int         type;     /* see GRADIENT_TYPE above */
    
    int         npoints;  /* number of anchor points */
    ARGB32     *color;    /* ARGB color values for each anchor point*/
    double     *offset;   /* offset of each point from the beginning in
                           * fractions of entire length */
}ASGradient;
.fi libAfterImage/asimage/flip
.SH FUNCTION
.nf
This are flags that define rotation angle.
.fi 
.SH NAMEFLIP_VERTICAL
defines rotation of 90 degrees counterclockwise.
FLIP_UPSIDEDOWN
defines rotation of 180 degrees counterclockwise. combined they define rotation of 270 degrees counterclockwise.

.SH SOURCE
Source : 
.nf
#define FLIP_VERTICAL       (0x01<<0)
#define FLIP_UPSIDEDOWN     (0x01<<1)
#define FLIP_MASK           (FLIP_UPSIDEDOWN|FLIP_VERTICAL)
.fi libAfterImage/asimage/flip_gradient()
.SH NAMEflip_gradient()
- rotates gradient in 90 degree increments.

.SH SYNOPSIS
.nf
ASGradient *flip_gradient( ASGradient *orig, int flip );
.fi 
.SH INPUTS
.IP "orig"
- pointer to original ASGradient structure to be rotated.
.IP "flip"
- value defining desired rotation.

.SH RETURN VALUE
.nf
Same as original gradient if flip is 0. New gradient structure in any
other case.
.fi 
.SH DESCRIPTION
.nf
Rotates ( flips ) gradient data in 90 degree increments. When needed
order of points is reversed.
.fi libAfterImage/asimage/tint
.SH FUNCTION
.nf
We use 32 bit ARGB values to define how tinting should be done.
The formula for tinting particular channel data goes like that:
tinted_data = (image_data * tint)/128
So if tint channel value is greater then 127 - same channel will be
brighter in destination image; if it is lower then 127 - same channel
will be darker in destination image. Tint channel value of 127
( or 0x7F hex ) does not change anything.
Alpha channel is tinted as well, allowing for creation of
semitransparent images. Calculations are performed in 24.8 format -
with 8 bit precision. Result is saturated to avoid overflow, and
precision is carried over to next pixel ( error diffusion ), when con
verting 24.8 to 8 bit format.
.fi 
.SH NAMETINT_NONE
special value that disables tinting
TINT_LEAVE_SAME
also disables tinting.

.SH SOURCE
Source : 
.nf
#define TINT_NONE           0
#define TINT_LEAVE_SAME     (0x7F7F7F7F)
#define TINT_HALF_DARKER    (0x3F3F3F3F)
#define TINT_HALF_BRIGHTER  (0xCFCFCFCF)
#define TINT_RED            (0x7F7F0000)
#define TINT_GREEN          (0x7F007F00)
#define TINT_BLUE           (0x7F00007F)
.fi libAfterImage/asimage/compression
.SH FUNCTION
.nf
Defines the level of compression to attempt on ASImage scanlines.
.fi 
.SH NAMEASIM_COMPRESSION_NONE
defined as 0 - disables compression.
ASIM_COMPRESSION_FULL
defined as 100 - highest compression level. Anything in between 0 and 100 will cause only part of the scanline to be compressed. This is obsolete. Now all images are compressed if possible.
libAfterImage/asimage/asimage_init()
.SH NAMEasimage_init()
frees datamembers of the supplied ASImage structure, and initializes it to all 0.

.SH SYNOPSIS
.nf
void asimage_init (ASImage * im, Bool free_resources);
.fi 
.SH INPUTS
.IP "im"
- pointer to valid ASImage structure
.IP "free_resources"
- if True will make function attempt to free all non-NULL pointers.
libAfterImage/asimage/flush_asimage_cache()
.SH NAMEflush_asimage_cache()
destroys XImage and mask XImage kept from previous conversions to/from X Pixmap.

.SH SYNOPSIS
.nf
void flush_asimage_cache (ASImage * im );
.fi 
.SH INPUTS
.IP "im"
- pointer to valid ASImage structure
libAfterImage/asimage/asimage_start()
.SH NAMEasimage_start()
Allocates memory needed to store scanline of the image of supplied size. Assigns all the data members valid values. Makes sure that ASImage structure is ready to store image data.

.SH SYNOPSIS
.nf
void asimage_start (ASImage * im, unsigned int width,
                                  unsigned int height,
                                  unsigned int compression);
.fi 
.SH INPUTS
.IP "im"
- pointer to valid ASImage structure
.IP "width"
- width of the image
.IP "height"
- height of the image
.IP "compression"
- level of compression to perform on image data. compression has to be in range of 0-100 with 100 signifying highest level of compression.

.SH NOTES
.nf
In order to resize ASImage structure after asimage_start() has been
called, asimage_init() must be invoked to free all the memory, and
then asimage_start() has to be called with new dimensions.
.fi libAfterImage/asimage/create_asimage()
.SH NAMEcreate_asimage()
Performs memory allocation for the new ASImage structure, as well as initialization of allocated structure based on supplied parameters.

.SH SYNOPSIS
.nf
ASImage *create_asimage( unsigned int width,
                         unsigned int height,
                         unsigned int compression);
.fi 
.SH INPUTS
.IP "width"
- desired image width
.IP "height"
- desired image height
.IP "compression"
- compression level in new ASImage( see asimage_start() for more ).

.SH RETURN VALUE
.nf
Pointer to newly allocated and initialized ASImage structure on
Success. NULL in case of any kind of error - that should never happen.
.fi libAfterImage/asimage/clone_asimage()
.SH NAMEclone_asimage()


.SH SYNOPSIS
.nf
ASImage *clone_asimage(ASImage *src, ASFlagType filter );
.fi 
.SH INPUTS
.IP "src"
- original ASImage.
.IP "filter"
- bitmask of channels to be copied from one image to another.

.SH RETURN VALUE
.nf
New ASImage, as a copy of original image.
.fi 
.SH DESCRIPTION
.nf
Creates exact clone of the original ASImage, with same compression,
back_color and rest of the attributes. Only ASImage data will be
carried over. Any attached alternative forms of images (XImages, etc.)
will not be copied. Any channel with unset bit in filter will not be
copied. Image name, ASImageManager and ref_count will not be copied -
use store_asimage() afterwards and make sure you use different name,
to avoid clashes with original image.
.fi libAfterImage/asimage/destroy_asimage()
.SH NAMEdestroy_asimage()
frees all the memory allocated for specified ASImage.

.SH SYNOPSIS
.nf
void destroy_asimage( ASImage **im );
.fi 
.SH INPUTS
.IP "im"
- pointer to valid ASImage structure.

.SH NOTES
.nf
If there was XImage attached to it - it will be deallocated as well.
.fi 
.SH EXAMPLE
.nf
asview.c: ASView.5
.fi libAfterImage/asimage/asimage_replace()
.SH NAMEasimage_replace()
will replace ASImage's data using data from another ASImage

.SH SYNOPSIS
.nf
Bool asimage_replace (ASImage *im, ASImage *from);
.fi 
.SH INPUTS
.IP "im"
- pointer to valid ASImage structure.
.IP "from"
- pointer to ASImage from which to take the data.

.SH NOTES
.nf
this function updates image without reallocating structure itself, which 
means that all pointers to it will still be valid. If that function 
succeeds - [from] ASImage will become unusable and should be deallocated 
using free() call.
.fi libAfterImage/asimage/set_asimage_vector()
.SH NAMEset_asimage_vector()
This function replaces contents of the vector member of ASImage structure with new double precision data.

.SH SYNOPSIS
.nf
set_asimage_vector( ASImage *im, register double *vector );
.fi 
.SH INPUTS
.IP "im"
- pointer to valid ASImage structure.
.IP "vector"
- scientific data to attach to the image.

.SH NOTES
.nf
Data must have size of width*height ahere width and height are size of 
the ASImage.
.fi libAfterImage/asimage/vectorize_asimage()
.SH NAMEvectorize_asimage()
This function replaces contents of the vector member of ASImage structure with new double precision data, generated from native ARGB32 image contents. Color palette is generated by indexing color values using max_colors, dither and opaque_threshold parameters.

.SH SYNOPSIS
.nf
ASVectorPalette* vectorize_asimage( ASImage *im, 
                                    unsigned int max_colors, 
                                    unsigned int dither,  
                                    int opaque_threshold );
.fi 
.SH INPUTS
.IP "im"
- pointer to valid ASImage structure.
.IP "max_colors"
- maximum size of the colormap.
.IP "dither"
- number of bits to strip off the color data ( 0...7 )
.IP "opaque_threshold"
- alpha channel threshold at which pixel should be treated as opaque

.SH RETURN VALUE
.nf
pointer to the ASVectorPalette structure that could be used for 
reverse conversion from double values to ARGB32. 
.fi 
.SH NOTES
.nf
alt.vector member of the supplied ASImage will be replaced and will 
contain WIDTHxHEIGHT double values representing generated scientific 
data.
.fi libAfterImage/asimage/create_image_manager()
.SH NAMEcreate_image_manager()
create ASImage management and reference counting object.

.SH SYNOPSIS
.nf
ASImageManager *create_image_manager( ASImageManager *reusable_memory,
                                      double gamma, ... );
.fi 
.SH INPUTS
.IP "reusable_memory"
- optional pointer to a block of memory to be used to store ASImageManager object.
.IP "double"
gamma - value of gamma correction to be used while loading images from files.
.IP "..."
- NULL terminated list of up to 8 PATH strings to list locations at which images could be found.

.SH DESCRIPTION
.nf
Creates ASImageManager object in memory and initializes it with
requested gamma value and PATH list. This Object will contain a hash
table referencing all the loaded images. When such object is used while
loading images from the file - gamma and PATH values will be used, so
that all the loaded and referenced images will have same parameters.
File name will be used as the image name, and if same file is attempted
to be loaded again - instead reference will be incremented, and
previously loaded image will be retyrned. All the images stored in
ASImageManager's table will contain a back pointer to it, and they must
be deallocated only by calling release_asimage(). destroy_asimage() will
refuse to deallocate such an image.
.fi libAfterImage/asimage/destroy_image_manager()
.SH NAMEdestroy_image_manager()
destroy management obejct.

.SH SYNOPSIS
.nf
void destroy_image_manager( struct ASImageManager *imman, 
                               Bool reusable );
.fi 
.SH INPUTS
.IP "imman"
- pointer to ASImageManager object to be deallocated
.IP "reusable"
- if True, then memory that holds object itself will not be deallocated. Usefull when object is created on stack.

.SH DESCRIPTION
.nf
Destroys all the referenced images, PATH values and if reusable is False,
also deallocates object's memory.
.fi libAfterImage/asimage/store_asimage()
.SH NAMEstore_asimage()
add ASImage to the reference.

.SH SYNOPSIS
.nf
Bool store_asimage( ASImageManager* imageman, ASImage *im, 
                       const char *name );
.fi 
.SH INPUTS
.IP "imageman"
- pointer to valid ASImageManager object.
.IP "im"
- pointer to the image to be stored.
.IP "name"
- unique name of the image.

.SH DESCRIPTION
.nf
Adds specifyed image to the ASImageManager's list of referenced images.
Stored ASImage could be deallocated only by release_asimage(), or when
ASImageManager object itself is destroyed.
.fi libAfterImage/asimage/relocate_asimage()
.SH NAMErelocate_asimage()
relocate ASImage into a different image manager.

.SH SYNOPSIS
.nf
void     relocate_asimage( ASImageManager* to_imageman, ASImage *im );
.fi 
.SH INPUTS
.IP "to_imageman"
- pointer to valid ASImageManager object.
.IP "im"
- pointer to the image to be stored.

.SH DESCRIPTION
.nf
Moves image from one ASImageManager's list of referenced images into 
another ASImageManager. Reference count will be kept the same.
.fi libAfterImage/asimage/fetch_asimage()
.SH NAMEfetch_asimage()

query_asimage()


.SH SYNOPSIS
.nf
ASImage *fetch_asimage( ASImageManager* imageman, const char *name );
ASImage *query_asimage( ASImageManager* imageman, const char *name );
.fi 
.SH INPUTS
.IP "imageman"
- pointer to valid ASImageManager object.
.IP "name"
- unique name of the image.

.SH DESCRIPTION
.nf
Looks for image with the name in ASImageManager's list and if found,
returns pointer to it. Note that query_asimage() does not increment 
reference count, while fetch_asimage() does. Therefore if fetch_asimage()
is used - release_asimage() should be called , when image is no longer 
in use.
.fi libAfterImage/asimage/dup_asimage()
.SH NAMEdup_asimage()
increment reference count of stored ASImage.

.SH SYNOPSIS
.nf
ASImage *dup_asimage( ASImage* im );
.fi 
.SH INPUTS
.IP "im"
- pointer to already referenced image.
libAfterImage/asimage/release_asimage()
.SH NAMErelease_asimage()
decrement reference count for given ASImage.
release_asimage_by_name()
decrement reference count for ASImage identifyed by its name.

.SH SYNOPSIS
.nf
int release_asimage( ASImage *im );
int release_asimage_by_name( ASImageManager *imman, char *name );
.fi 
.SH INPUTS
.IP "im"
- pointer to already referenced image.
.IP "imageman"
- pointer to valid ASImageManager object.
.IP "name"
- unique name of the image.

.SH DESCRIPTION
.nf
Decrements reference count on the ASImage object and destroys it if
reference count is below zero.
.fi libAfterImage/asimage/forget_asimage()
.SH NAMEforget_asimage()
remove ASImage from ASImageManager's hash by pointer.
forget_asimage_name()
remove ASImage from ASImageManager's hash by its name.

.SH SYNOPSIS
.nf
void     forget_asimage( ASImage *im );
void  forget_asimage_name( ASImageManager *imman, const char *name );
.fi 
.SH INPUTS
.IP "im"
pointer to already referenced image.
.IP "imageman"
pointer to valid ASImageManager object.
.IP "name"
unique name of the image.
libAfterImage/safe_asimage_destroy()
.SH NAMEsafe_asimage_destroy()
either release or destroy asimage, checking if it is attached to ASImageManager.

.SH SYNOPSIS
.nf
int      safe_asimage_destroy( ASImage *im );
.fi 
.SH INPUTS
.IP "im"
pointer to and ASImage structure.
libAfterImage/print_asimage_manager()
.SH NAMEprint_asimage_manager()
prints list of images referenced in given ASImageManager structure.
libAfterImage/asimage/destroy_asgradient()
.SH NAMEdestroy_asgradient()
- destroy ASGradient structure, deallocating all associated memory
libAfterImage/asimage/init_image_layers()
.SH NAMEinit_image_layers()
- initialize set of ASImageLayer structures.

.SH SYNOPSIS
.nf
void init_image_layers( register ASImageLayer *l, int count );
.fi 
.SH INPUTS
.IP "l"
- pointer to valid ASImageLayer structure.
.IP "count"
- number of elements to initialize.

.SH DESCRIPTION
.nf
Initializes array on ASImageLayer structures to sensible defaults.
Basically - all zeros and merge_scanlines == alphablend_scanlines.
.fi libAfterImage/asimage/create_image_layers()
.SH NAMEcreate_image_layers()
- allocate and initialize set of ASImageLayer's.

.SH SYNOPSIS
.nf
ASImageLayer *create_image_layers( int count );
.fi 
.SH INPUTS
.IP "count"
- number of ASImageLayer structures in allocated array.

.SH RETURN VALUE
.nf
Pointer to newly allocated and initialized array of ASImageLayer
structures on Success. NULL in case of any kind of error - that
should never happen.
.fi 
.SH DESCRIPTION
.nf
Performs memory allocation for the new array of ASImageLayer
structures, as well as initialization of allocated structure to
sensible defaults - merge_func will be set to alphablend_scanlines.
.fi libAfterImage/asimage/destroy_image_layers()
.SH NAMEdestroy_image_layers()
- destroy set of ASImageLayer structures.

.SH SYNOPSIS
.nf
void destroy_image_layers( register ASImageLayer *l,
                           int count,
                           Bool reusable );
.fi 
.SH INPUTS
.IP "l"
- pointer to pointer to valid array of ASImageLayer structures.
.IP "count"
- number of structures in array.
.IP "reusable"
- if True - then array itself will not be deallocates - which is usable when it was allocated on stack.

.SH DESCRIPTION
.nf
frees all the memory allocated for specified array of ASImageLayer s.
If there was ASImage and/or ASImageBevel attached to it - it will be
deallocated as well.
.fi libAfterImage/asimage/asimage_add_line()
.SH NAMEasimage_add_line()


.SH SYNOPSIS
.nf
size_t asimage_add_line ( ASImage * im, ColorPart color,
                          CARD32 * data, unsigned int y);
.fi 
.SH INPUTS
.IP "im"
- pointer to valid ASImage structure
.IP "color"
- color channel's number
.IP "data"
- raw channel data of 32 bits per pixel - only lowest 8 bits gets encoded.
.IP "y"
- image row starting with 0

.SH RETURN VALUE
.nf
asimage_add_line() return size of the encoded channel scanline in
bytes. On failure it will return 0.
.fi 
.SH DESCRIPTION
.nf
Encodes raw data of the single channel into ASImage channel scanline.
based on compression level selected for this ASImage all or part of
the scanline will be RLE encoded.
.fi libAfterImage/asimage/asimage_add_line_mono()
.SH NAMEasimage_add_line_mono()


.SH SYNOPSIS
.nf
size_t asimage_add_line_mono ( ASImage * im, ColorPart color,
                               CARD8 value, unsigned int y);
.fi 
.SH INPUTS
.IP "im"
- pointer to valid ASImage structure
.IP "color"
- color channel's number
.IP "value"
- value for the channel
.IP "y"
- image row starting with 0

.SH RETURN VALUE
.nf
asimage_add_line_mono() return size of the encoded channel scanline
in bytes. On failure it will return 0.
.fi 
.SH DESCRIPTION
.nf
encodes ASImage channel scanline to have same color components
value in every pixel. Useful for vertical gradients for example.
.fi libAfterImage/asimage/get_asimage_chanmask()
.SH NAMEget_asimage_chanmask()


.SH SYNOPSIS
.nf
ASFlagType get_asimage_chanmask( ASImage *im);
.fi 
.SH INPUTS
.IP "im"
- valid ASImage object.

.SH DESCRIPTION
.nf
goes throu all the scanlines of the ASImage and toggles bits 
representing those components that have at least some data.
.fi libAfterImage/asimage/move_asimage_channel()
.SH NAMEmove_asimage_channel()


.SH SYNOPSIS
.nf
void move_asimage_channel( ASImage *dst, int channel_dst,
                           ASImage *src, int channel_src );
.fi 
.SH INPUTS
.IP "dst"
- ASImage which will have its channel substituted;
.IP "channel_dst"
- what channel to move data to;
.IP "src"
- ASImage which will donate its channel to dst;
.IP "channel_src"
- what source image channel to move data from.

.SH DESCRIPTION
.nf
MOves channel data from one ASImage to another, while discarding
what was already in destination's channel.
.fi 
.SH NOTES
.nf
Source image (donor) will loose its channel data, as it will be
moved to destination ASImage. Also there is a condition that both
images must be of the same width - otherwise function returns
without doing anything. If height is different - the minimum of
two will be used.
.fi libAfterImage/asimage/copy_asimage_channel()
.SH NAMEcopy_asimage_channel()


.SH SYNOPSIS
.nf
void copy_asimage_channel( ASImage *dst, int channel_dst,
                           ASImage *src, int channel_src );
.fi 
.SH INPUTS
.IP "dst"
- ASImage which will have its channel substituted;
.IP "channel_dst"
- what channel to copy data to;
.IP "src"
- ASImage which will donate its channel to dst;
.IP "channel_src"
- what source image channel to copy data from.

.SH DESCRIPTION
.nf
Same as move_asimage_channel() but makes copy of channel's data
instead of simply moving it from one image to another.
.fi libAfterImage/asimage/copy_asimage_lines()
.SH NAMEcopy_asimage_lines()


.SH SYNOPSIS
.nf
void copy_asimage_lines( ASImage *dst, unsigned int offset_dst,
                         ASImage *src, unsigned int offset_src,
                         unsigned int nlines, ASFlagType filter );
.fi 
.SH INPUTS
.IP "dst"
- ASImage which will have its channel substituted;
.IP "offset_dst"
- scanline in destination image to copy to;
.IP "src"
- ASImage which will donate its channel to dst;
.IP "offset_src"
- scanline in source image to copy data from;
.IP "nlines"
- number of scanlines to be copied;
.IP "filter"
- specifies what channels should be copied.

.SH DESCRIPTION
.nf
Makes copy of scanline data for continuos set of scanlines, affecting
only those channels marked in filter.
NOTE
Images must be of the same width.
.fi libAfterImage/asimage/verbosity
.SH FUNCTION
.nf
This are flags that define what should be printed by
asimage_print_line():
    VRB_LINE_SUMMARY    - print only summary for each scanline
    VRB_LINE_CONTENT    - print summary and data for each scanline
    VRB_CTRL_EXPLAIN    - print summary, data and control codes for each
                          scanline
.fi 
.SH SOURCE
Source : 
.nf
#define VRB_LINE_SUMMARY    (0x01<<0)
#define VRB_LINE_CONTENT    (0x01<<1)
#define VRB_CTRL_EXPLAIN    (0x01<<2)
#define VRB_EVERYTHING      (VRB_LINE_SUMMARY|VRB_CTRL_EXPLAIN| \
                             VRB_LINE_CONTENT)
.fi libAfterImage/asimage/asimage_print_line()
.SH NAMEasimage_print_line()


.SH SYNOPSIS
.nf
    unsigned int asimage_print_line ( ASImage * im, ColorPart color,
                                      unsigned int y,
                                      unsigned long verbosity);
.fi 
.SH INPUTS
.IP "im"
- pointer to valid ASImage structure
.IP "color"
- color channel's number
.IP "y"
- image row starting with 0
.IP "verbosity"
- verbosity level - any combination of flags is allowed

.SH RETURN VALUE
.nf
amount of memory used by this particular channel of specified
scanline.
.fi 
.SH DESCRIPTION
.nf
asimage_print_line() prints data stored in specified image scanline
channel. That may include simple summary of how much memory is used,
actual visible data, and/or RLE control codes. That helps to see
how effectively data is encoded.

Useful mostly for debugging purposes.
.fi libAfterImage/asimage/get_asimage_channel_rects()
.SH NAMEget_asimage_channel_rects()
- translate image into a list of rectangles.

.SH SYNOPSIS
.nf
XRectangle* 
    get_asimage_channel_rects( ASImage *src, int channel, 
                               unsigned int threshold, 
                               unsigned int *rects_count_ret ); 
.fi 
.SH INPUTS
.IP "src"
- ASImage which will donate its channel to dst;
.IP "channel"
- what source image channel to copy data from;
.IP "threshold"
- threshold to compare channel values against;
.IP "rects_count_ret"
- returns count of generated rectangles.

.SH DESCRIPTION
.nf
This function will translate contents of selected channel 
(usualy alpha) into a list of rectangles, ecompasing regions 
with values above the threshold. This is usefull to generate shape
of the window to be used with X Shape extention.
.fi 
