/* ===========================================================================
 * xfmain.c: CD player using XForm library
 * ===========================================================================
 * Copyright (c) 1997-99 Denis Bourez. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *    This product includes software developed by Denis Bourez
 * 4. Neither the name of the author nor the names of any co-contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY DENIS BOUREZ AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL DENIS BOUREZ OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ===========================================================================
 */

/* ============================================================================
 * Includes:
 * ============================================================================ */ 

#include <stdio.h>
#include <sys/param.h>
#include <sys/types.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <X11/Xlib.h>
#include <X11/xpm.h>
#include <X11/extensions/shape.h>
#include <errno.h>
#include <unistd.h>
#include <grp.h>
#include <pwd.h>

#include <forms.h>

#include <sys/mount.h>

#include "../lib/struct.h"
#include "../config.h"
#include "ascd.h"
#include "xfascd.h"

#include "pixmaps/xpm_st_off.xpm"
#include "pixmaps/xpm_st_on.xpm"
#include "pixmaps/xpm_pa_off.xpm"
#include "pixmaps/xpm_pa_on.xpm"
#include "pixmaps/xpm_pl_off.xpm"
#include "pixmaps/xpm_pl_on.xpm"
#include "pixmaps/xpm_cue.xpm"
#include "pixmaps/xpm_rew.xpm"
#include "pixmaps/xpm_prev.xpm"
#include "pixmaps/xpm_next.xpm"
#include "pixmaps/loop_on.xpm"
#include "pixmaps/loop_off.xpm"
#include "pixmaps/intro_on.xpm"
#include "pixmaps/intro_off.xpm"
#include "pixmaps/xpm_lleft.xpm"
#include "pixmaps/xpm_lright.xpm"

#ifdef BLUE_BACK
#  include "pixmaps/gradient.xpm"
#  include "pixmaps/gradient_s.xpm"
#endif

#ifdef FRENCH
#  include "messages/francais.h"
#else
#  include "messages/english.h"
#endif

/* ============================================================================
 * Globals & #defines:
 * ============================================================================ */ 

#define PROGRAM "xfascd"

/* xforms border widths */

#define BBW -3
#define XBBW -1
#define PBW 3
#define SBW 3

/* window placement */

#define CENTER 0
#define LEFT 1
#define RIGHT 2
#define TOP 3
#define BOTTOM 4

char device[128] = DEFAULTDEVICE;

extern cd_status();
extern play_cd();
extern stop_cd();
extern eject_cd();
extern pause_cd();

extern load();
extern save();
extern split_workmandb();
extern WMstrdup(char *);

extern open();

extern cd_control(int);

extern int cur_track, cur_index, cur_lasttrack, cur_firsttrack, cur_pos_abs ;
extern int cur_frame, cur_pos_rel, cur_tracklen, cur_cdlen, cur_ntracks ;
extern int cur_nsections, cur_listno, cur_stopmode, exit_on_eject, cur_balance ;
extern enum cd_modes cur_cdmode;
extern char *cur_artist, *cur_cdname, *cur_trackname;
extern char cur_contd, cur_avoid;
extern char *cd_device;
extern struct cdinfo_wm thiscd, *cd;
extern int found_in_db;

int cur_balance = 10;
int info_modified = 0;

int big_spaces = 0;

#define MAX_VOL 256

int update_xpm = 1; /* do we need to redraw buttons XPMs? */

unsigned long aide_timeout;
unsigned long aide_mtimeout;

int main_x;
int main_y; /* geometry of main window */

int db_place;
int o_place;

unsigned int debug = 0;          /* if 1, verbose output on stderr */
unsigned int time_mode = 0;      /* display mode for the counter. see function RedrawWindow() */
unsigned int cue_time = 10;      /* nbr of seconds for cue -> see -c command line option */
unsigned int autoplay = 0;       /* if set, play a disk when it is inserted -> see -a c.l. option  */
unsigned int autorepeat = FALSE; /* zzzzzzzzzzzzz */
unsigned int datatrack = 0;      /* is the current track a data track ? */
unsigned int direct_access = 0;  /* pos. to reach with the ruler */
unsigned int direct_track = 0;

unsigned int volume = MAX_VOL ;  /* CD volume */

unsigned int loop_mode = 0;      /* loop from loop_1 to loop_2 */
unsigned int loop_start_track = 0, /* tracks to loop */
    loop_end_track =0;
unsigned int loop_1 = 0;         /* pos. 1 */
unsigned int loop_2 = 0;         /* pos. 2 */

unsigned int intro_mode = 0; /* play only the beginning of the tracks */

unsigned int wanna_play = FALSE;
unsigned int do_autorepeat = FALSE;
int wanted_track = 0;

unsigned int anomalie = 0; /* cd_control return value */

/* from/for WorkMan database code: */

int mark_a = 0;
int mark_b = 0;
int cur_stopmode = 0;
int cur_playnew = 0;

int voldev[6];
int volold[6];
int vol[6];

#define MAX_LINE_NAME 16

int db_play = 0; /* autoplay of tracks when selected in db browser */
int db_edit = 0; /* current edited track nbr */

/* ============================================================================
 * 'Dumb' functions needed if we don't want to change Workman's code
 * ============================================================================ */ 

void disable_save() { }
int get_playnew() { return autoplay; }
void set_abtimer(int a, int b) { }
void about_set_drivetype(char *vendor, char *model, char *rev) { }

/* ============================================================================
 * For debug mode:
 * ============================================================================ */ 

deb(char *what) {
    if (debug) fprintf(stderr, "%s\n", what);
}

/* ============================================================================
 * Misc functions:
 * ============================================================================ */ 

void help_commandline() 
{
    fprintf(stderr, "%s %s, %s\n\n", PROGRAM, VERSION, L_HELP1);
    
    fprintf(stderr,"%s xfascd [-options ...] \n\n", L_HELP2);
    
    fprintf(stderr,"%s\n\n", L_HELP3);
    
    fprintf(stderr,"%s\n", L_HELP4);
    fprintf(stderr,"a                    No          Auto play inserted disks (buggy!)\n");
    fprintf(stderr,"t                    No          Auto repeat\n");
    fprintf(stderr,"c Cue Time           10          Cue Time in seconds\n");
#ifdef NO_D_DEVICE
    fprintf(stderr,"d Device             No          CDROM drive device, autoprobe mode\n");
#else
    fprintf(stderr,"d Device             /dev/cdrom  CDROM drive device\n");
#endif	
    fprintf(stderr,"v                    No          Verbose: debug infos on stderr\n");
    
    fprintf(stderr,"\nSample: ascd -a -c 5 -d /dev/wcd0a\n\n");
      
    exit(1);
} 

void parse_commandline(int argc, char *argv[]) 
{
    /*char WhichLine[MAX_LINE_NAME];*/
    int i, j;
    char *Argument;
    
    /*WhichLine[MAX_LINE_NAME] = 0;*/

    for(i = 1; i < argc; i++)
	{
	    Argument = argv[i];

	    if (Argument[0] == '-') {
		switch(Argument[1]) {
		case 'd': /* Device */
		    if(++i >= argc) help_commandline();
		    cd_device = malloc(strlen(argv[i]) + 1);
		    strcpy(cd_device, argv[i]);
		    continue;
		case 'c': /* Cue Time */
		    if(++i >= argc) help_commandline();
		    cue_time = atoi(argv[i]);
		    continue;
		case 'a': /* AutoPlay */
		    autoplay = TRUE;
		    continue;
		case 'v': /* Verbose : debug mode */
		    debug = 1;
		    continue;
		case 't' : /* AutoRepeat */
		    autorepeat = TRUE;
		    continue;
		case 'h':  /* help_commandline */
		    if(++i >= argc) help_commandline();
		    continue;
		default:
		    help_commandline();
		}
	    }
	    else
		help_commandline();
	}
    
} 

void load_rc_file() 
{
    FILE *in;
    struct passwd *pw;
    char st[256];

    if (NULL == (pw = getpwuid(getuid()))) {
        return;
    } else {
        sprintf(st, "%s/.xfascdrc", pw->pw_dir);
        if ((in = fopen(st, "r"))) {
            while ( fgets(st, 255, in) ) {
                if (strlen(st) > 0) st[strlen(st) - 1] = 0;
                if (strncmp(st, "CD_DEVICE=", 10) == 0) strcpy(cd_device, st + 10);
                else if (strncmp(st, "AUTOREPEAT=", 11) == 0) autorepeat = atoi(st + 11);
                else if (strncmp(st, "AUTOPLAY=", 9) == 0) autoplay = atoi(st + 9);
                else if (strncmp(st, "AUTODBPLAY=", 11) == 0) db_play = atoi(st + 11);
                else if (strncmp(st, "CUE_TIME=", 9) == 0) cue_time = atoi(st + 9);
                else if (strncmp(st, "MAIN_X=", 7) == 0) main_x = atoi(st + 7);
                else if (strncmp(st, "MAIN_Y=", 7) == 0) main_y = atoi(st + 7);
                else if (strncmp(st, "OPLACE=", 7) == 0) o_place = atoi(st + 7);
                else if (strncmp(st, "DBPLACE=", 8) == 0) db_place = atoi(st + 8);
            }
            fclose(in);
            return;
        } else {
            return;
        }
    }
} 

void aide(char *msg, unsigned int helpline_mode) 
{
    if (! f_liste->visible) {
	fl_set_object_label(f_looplabel, msg);
	if (helpline_mode == 0) {
	    fl_gettime(&aide_timeout, &aide_mtimeout);
	}
    }
} 

void newtext(char *chaine) { 
    /*(void)aide(chaine, 0);*/
}


void database_window() 
{
    int i;
    char txt[100];
    
    if (! fl_form_is_visible(f_database)) {
	fl_set_input(f_db_title, cd->cdname);
	fl_set_input(f_db_artist, cd->artist);
	fl_clear_browser(f_db_browser); 

	for (i = 0; i < cur_ntracks; i++) {
	    sprintf(txt, "%2d ", i + 1);
	    if (cd->trk[i].songname != NULL) strcat(txt,  cd->trk[i].songname);
	    fl_add_browser_line(f_db_browser, txt);
	}
	
	if (cd->trk[cur_track - 1].songname != NULL) fl_set_input(f_db_track, cd->trk[cur_track - 1].songname);
	sprintf(txt, L_TRACKNAME, cur_track);
	fl_set_object_label(f_db_tracklabel, txt);
	fl_set_button(f_db_cont, cd->trk[cur_track - 1].contd);

        if (db_play) fl_set_button(f_db_autoplay, 1);
        else fl_set_button(f_db_autoplay, 0);

        db_edit = cur_track;

        switch(db_place) {
	case RIGHT:
	    fl_set_form_position(f_database, f_main->x + f_main->w + 12, f_main->y);
	    fl_show_form(f_database, FL_PLACE_GEOMETRY, FL_FULLBORDER, L_DB_WTITLE);
	    break;
	case BOTTOM:
	    fl_set_form_position(f_database, f_main->x, f_main->y + f_main->h + 30);
	    fl_show_form(f_database, FL_PLACE_GEOMETRY, FL_FULLBORDER, L_DB_WTITLE);
	    break;
	case LEFT:
	    fl_set_form_position(f_database, f_main->x - f_database->w - 12, f_main->y);
	    fl_show_form(f_database, FL_PLACE_GEOMETRY, FL_FULLBORDER, L_DB_WTITLE);
	    break;
	case TOP:
	    fl_set_form_position(f_database, f_main->x, f_main->y - f_database->h - 30);
	    fl_show_form(f_database, FL_PLACE_GEOMETRY, FL_FULLBORDER, L_DB_WTITLE);
	    break;
	default:
	    fl_show_form(f_database, FL_PLACE_CENTER, FL_FULLBORDER, L_DB_WTITLE);
	    break;
        }
    }
    else fl_hide_form(f_database);

} 

void options_window() 
{
    char txt[100];
    
    if (! fl_form_is_visible(f_options)) {
        sprintf(txt, "%s %s \n%s\n%s", PROGRAM, VERSION, L_HELP1, L_WEB);
        fl_set_object_label(o_s1, txt);
        sprintf(txt, "%d", cue_time);
	fl_set_input(o_cue, txt);
        fl_set_button(o_autoplay, autoplay);
        fl_set_button(o_autorepeat, autorepeat);
        fl_set_button(o_db_autoplay, db_play);

        fl_set_button(o_ocenter, 0);
        fl_set_button(o_oleft, 0);
        fl_set_button(o_oright, 0);
        fl_set_button(o_otop, 0);
        fl_set_button(o_obottom, 0);

        fl_set_button(o_dcenter, 0);
        fl_set_button(o_dleft, 0);
        fl_set_button(o_dright, 0);
        fl_set_button(o_dtop, 0);
        fl_set_button(o_dbottom, 0);

        switch(db_place) {
	case RIGHT:
	    fl_set_button(o_dright, 1);
	    break;
	case BOTTOM:
	    fl_set_button(o_dbottom, 1);
	    break;
	case LEFT:
	    fl_set_button(o_dleft, 1);
	    break;
	case TOP:
	    fl_set_button(o_dtop, 1);
	    break;
	default:
	    fl_set_button(o_dcenter, 1);
	    break;
        }
        
        switch(o_place) {
	case RIGHT:
	    fl_set_button(o_oright, 1);
	    fl_set_form_position(f_options, f_main->x + f_main->w + 12, f_main->y);
	    fl_show_form(f_options, FL_PLACE_GEOMETRY, FL_FULLBORDER, L_PREF_WTITLE);
	    break;
	case BOTTOM:
	    fl_set_button(o_obottom, 1);
	    fl_set_form_position(f_options, f_main->x, f_main->y + f_main->h + 30);
	    fl_show_form(f_options, FL_PLACE_GEOMETRY, FL_FULLBORDER, L_PREF_WTITLE);
	    break;
	case LEFT:
	    fl_set_button(o_oleft, 1);
	    fl_set_form_position(f_options, f_main->x - f_options->w - 12, f_main->y);
	    fl_show_form(f_options, FL_PLACE_GEOMETRY, FL_FULLBORDER, L_PREF_WTITLE);
	    break;
	case TOP:
	    fl_set_button(o_otop, 1);
	    fl_set_form_position(f_options, f_main->x, f_main->y - f_options->h - 30);
	    fl_show_form(f_options, FL_PLACE_GEOMETRY, FL_FULLBORDER, L_PREF_WTITLE);
	    break;
	default:
	    fl_set_button(o_ocenter, 1);
	    fl_show_form(f_options, FL_PLACE_CENTER, FL_FULLBORDER, L_PREF_WTITLE);
	    break;
        }

    }
    else {
        cue_time = atoi(fl_get_input(o_cue));
        if (cue_time < 1) cue_time = 1;
        fl_hide_form(f_options);
    }
} 

/* ============================================================================
 * Call Backs:
 * ============================================================================ */ 

void play_cb(FL_OBJECT * ob, long data) 
{
    if ((cur_cdmode == PLAYING) || (cur_cdmode == PAUSED)) {
	if (fl_get_button_numb(ob) == FL_RIGHT_MOUSE) {
	    aide(L_H_PAUSE, 0);
	} else {
	    cd_control(PAUSE);
	}
    } else {
	if (fl_get_button_numb(ob) == FL_RIGHT_MOUSE) {
	    aide(L_H_PLAY, 0);
	} else {
	    cd_control(PLAY);
	}
    }
} 

void pause_cb(FL_OBJECT * ob, long data) 
{
    if ((cur_cdmode == PLAYING) || (cur_cdmode == PAUSED)) {
	if (fl_get_button_numb(ob) == FL_RIGHT_MOUSE) {
	    aide(L_H_PAUSE, 0);
	} else {
	    cd_control(PAUSE);
	}
    } else {
	if (fl_get_button_numb(ob) == FL_RIGHT_MOUSE) {
	    aide(L_H_PLAY, 0);
	} else {
	    cd_control(PLAY);
	}
    }
} 

void stop_cb(FL_OBJECT * ob, long data) 
{
    if (fl_get_button_numb(ob) == FL_RIGHT_MOUSE) {
	aide(L_H_STOP, 0);
    } else {
	cd_control(STOP);
    }
} 

void prevtrack_cb(FL_OBJECT * ob, long data) 
{
    if (fl_get_button_numb(ob) == FL_RIGHT_MOUSE) {
	aide(L_H_PREV, 0);
    } else if (fl_get_button_numb(ob) == FL_MIDDLE_MOUSE) {
	cd_control(FIRST);
    } else {
	cd_control(DNTRACK);
    }
} 

void nexttrack_cb(FL_OBJECT * ob, long data) 
{
    if (fl_get_button_numb(ob) == FL_RIGHT_MOUSE) {
	aide(L_H_NEXT, 0);
    } else if (fl_get_button_numb(ob) == FL_MIDDLE_MOUSE) {
	cd_control(LAST);
    } else {
	cd_control(UPTRACK);
    }
} 

void prev_cb(FL_OBJECT * ob, long data) 
{
   if (fl_get_button_numb(ob) == FL_RIGHT_MOUSE) {
	aide(L_H_REV, 0);
    } else {
	    cd_control(REV);
    }
} 

void next_cb(FL_OBJECT * ob, long data) 
{
   if (fl_get_button_numb(ob) == FL_RIGHT_MOUSE) {
	aide(L_H_CUE, 0);
    } else {
	cd_control(CUE);
    }
} 

void intro_cb(FL_OBJECT * ob, long data) 
{
   if (fl_get_button_numb(ob) == FL_RIGHT_MOUSE) {
	aide(L_H_INTRO, 0);
    } else {
	update_xpm = 1;
	cd_control(INTROSCAN);
	if (intro_mode)
	    aide(L_H_INTROMODE1, 0);
	else
	    aide(L_H_INTROMODE0, 0);
    }
}

void lstart_cb(FL_OBJECT * ob, long data) 
{
    if (fl_get_button_numb(ob) == FL_RIGHT_MOUSE) {
	aide(L_H_LSTART, 0);
    } else {
	aide(L_H_LOOPL, 0);
	loop_1 = cur_pos_rel;
	loop_start_track = cur_track;
    }
} 

void lend_cb(FL_OBJECT * ob, long data) 
{
    if (fl_get_button_numb(ob) == FL_RIGHT_MOUSE) {
	aide(L_H_LEND, 0);
    } else {
	aide(L_H_LOOPR, 0);
	loop_2 = cur_pos_rel;
	loop_end_track = cur_track;
    }
} 

void lgo_cb(FL_OBJECT * ob, long data) 
{
    if (fl_get_button_numb(ob) == FL_RIGHT_MOUSE) {
	aide(L_H_LGO, 0);
    } else {
	update_xpm = 1;
	if (loop_mode) {
	    loop_mode = 0;
	    aide(L_H_LOOPMODE0, 0);
	} else {
	    cd_control(LOOP);
	    aide(L_H_LOOPMODE1, 0);
	}
	if (anomalie) aide(L_H_LOOPERROR, 0);

    }
} 

void label_cb(FL_OBJECT * ob, long data) 
{
    if (fl_get_button_numb(ob) == FL_RIGHT_MOUSE) {
	aide(L_H_LABEL, 0);
    } else {
	database_window();
    }
}

void title_cb(FL_OBJECT * ob, long data) 
{
    int i;
    char txt[100];
    
    if (fl_get_button_numb(ob) == FL_RIGHT_MOUSE) {
	aide(L_H_TITLE, 0);
    } else {
	if (! f_liste->visible) {
	    fl_clear_browser(f_liste); 
	    
	    for (i = 0; i < cur_ntracks; i++) {
		if (cd->trk[i].songname != NULL) {
		    strcpy(txt,  cd->trk[i].songname);
		    if (strcmp(txt, "") == 0) sprintf(txt, "%s %d", L_TRACK, i + 1);
		} else {
		    sprintf(txt, "%s %d", L_TRACK, i + 1);
		}
		fl_add_browser_line(f_liste, txt);
	    }
	    fl_show_object(f_liste);
	} else {
	    fl_freeze_form(f_main);
	    fl_hide_object(f_liste);
	    fl_unfreeze_form(f_main);
	}
    }
} 

void liste_cb(FL_OBJECT * ob, long data) 
{
    fl_freeze_form(f_main);
    fl_hide_object(f_liste);
    fl_unfreeze_form(f_main);
    direct_track = fl_get_browser(f_liste);
    cd_control(DIRECTTRACK);
} 

void trackslider_cb(FL_OBJECT * ob, long data) 
{
    direct_access = fl_get_slider_value(f_trackslider);
    cd_control(DIRECTACCESS);
} 

void diskslider_cb(FL_OBJECT * ob, long data) 
{
    direct_access = fl_get_slider_value(f_diskslider);
    cd_control(GLOBALACCESS);
} 

void vol_cb(FL_OBJECT * ob, long data) 
{
    volume = fl_get_slider_value(f_vol);
    cd_volume(volume, 10, MAX_VOL);
} 

void more_cb(FL_OBJECT * ob, long data) 
{
    switch (fl_get_menu(ob)) {
    case 1:
	cd_control(FIRST);
	break;
    case 2:
	cd_control(LAST);
	break;
    case 3:
	database_window();
	break;
    case 4:
	options_window();
	break;
    case 5:
	cd_control(INTROSCAN);
	break;
    case 6:
	cd_control(LOOP);
	if (anomalie) {
	    aide(L_H_LOOPERROR, 0);
	} else {
	    if (loop_mode) {
		aide(L_H_LOOPMODE1, 0);
	    } else {
		aide(L_H_LOOPMODE0, 0);
	    }
	}
	break;
    case 7: /* erase loop flags */
	loop_1 = 0;
	loop_2 = 0;
	loop_start_track = 0;
	loop_end_track = 0;
	loop_mode = 0;
	aide(L_H_LOOPERASE, 0);
	break;
    case 8 :
        loop_1 = cur_pos_rel;
        loop_start_track = cur_track;
	aide(L_H_LOOPL, 0);
        break;
    case 9 :
        loop_2 = cur_pos_rel;
        loop_end_track = cur_track;
	aide(L_H_LOOPR, 0);
        break;
    case 10: /* loop whole CD */
	if (cur_ntracks > 0) {
	    loop_1 = 0;
	    loop_start_track = 1;
	    loop_end_track = cur_ntracks;
	    loop_2 = cd->trk[cur_ntracks - 1].length;
	}
	break;
    case 11: /* loop current track */
	loop_1 = 0;
	loop_start_track = cur_track;
	loop_end_track = cur_track;
	loop_2 = cur_tracklen;
	break;
    case 12: /* loop from the begining of this track */
	loop_1 = 0;
	loop_start_track = cur_track;
	loop_end_track = cur_track;
	loop_2 = cur_pos_rel;
	break;
    case 13: /* loop from now to the end of this track */
	loop_1 = cur_pos_rel;
	loop_start_track = cur_track;
	loop_end_track = cur_track;
	loop_2 = cur_tracklen;
	break;
    case 14: /* loop all tracks <= this one */
	loop_1 = 0;
	loop_start_track = 1;
	loop_end_track = cur_track;
	loop_2 = cur_tracklen;
	break;
    case 15: /* loop all tracks >= this one */
	if (cur_ntracks > 0) {
	    loop_1 = 0;
	    loop_start_track = cur_track;
	    loop_end_track = cur_ntracks;
	    loop_2 = cd->trk[cur_ntracks - 1].length;
	}
	break;
    case 16: /* quit */
	exit(0);
	break;
    default:
	break;
    }
} 

/* ------------------------------------------------------------------------- */

int cd_events(XEvent *ev, void *data) 
{ 
    unsigned int no_disk = 0;
    char dummy[81];
    char track[3];
    char cdtime[5];
    unsigned long tm,mtm;
    int mn, sc;

    if (cur_track < 1) cur_track = 1;
    
    if (cur_cdmode == EJECTED) no_disk = 1;

    /* taken from ascd 0.9.2: */
    if (no_disk == 1) {
	if (autoplay) {
	    /*dodo = RDTIME2;*/
	    cd_status();
	}
	if (cur_cdmode != EJECTED) no_disk = 0;
	if ( (cur_cdmode == STOPPED) && (autoplay) )  {
	    cd_control(PLAY);
	    cd_status();
	    /*dodo = RDTIME;*/
	    /*RedrawWindow(&backXPM);*/
	}
    }
    
    /* the old code: */

    /*
    if (no_disk == 1) {
	if (autoplay) {
	    cd_status();
	}
	if (cur_cdmode != EJECTED) no_disk = 0;
	if ( (cur_cdmode == STOPPED) && (autoplay) ) {
	    cd_control(PLAY);
	}
    }
    */


    /* this was taken from ascd 0.9.2 too! */
    if ((cur_cdmode == PLAYING) || (wanna_play)) cd_status();

    if ((cur_track == cur_ntracks) &&
	(wanna_play) && 
	(cur_cdmode != PAUSED) && 
	(cur_cdmode != PLAYING)
	) {

	if (autorepeat) {
	    do_autorepeat = TRUE;
	    cd_control(PLAY);
	} else {
	    cur_cdmode = STOPPED;
	    cd_status();
	    cur_track = 0;
	}
    }

    
    /* The Loop Mode : */

    if ( (cur_track == loop_end_track )
	 && (cur_pos_rel >= loop_2) 
         && (loop_mode) ) {
        cd_control(LOOP);
    }
    
    /* The Intro Scan Mode : */
    
    if ( (cur_pos_rel > cue_time) && (intro_mode) ) cd_control(INTRONEXT);

    /* Screen updates: we freeze the window */
    
    fl_freeze_form(f_main);

    /* CD Informations */

    if (found_in_db != 0) {

	if (cur_cdmode != EJECTED) {
	    if (cd->cdname != NULL) sprintf(dummy,"%s\n", cd->cdname);
	    else strcpy(dummy, L_NEWCD);
	    
	    if (cd->artist != NULL) strcat(dummy, cd->artist);
	    else strcat(dummy, L_NEWARTIST);
	    
	    fl_set_object_label(f_label, dummy);
	    
	    if (cur_track < 1) cur_track = 1;
            
	    if (cd->trk[cur_track - 1].songname != NULL) {
		sprintf(dummy, "%s", cd->trk[cur_track - 1].songname);
		if (strcmp(dummy, "") == 0) sprintf(dummy, "%s %d", L_TRACK, cur_track);
		fl_set_object_label(f_title, dummy);
	    } else {
		sprintf(dummy, "%s %d", L_TRACK, cur_track);
		fl_set_object_label(f_title, dummy);
	    }
	} else {
	    fl_set_object_label(f_title, "");
	    fl_set_object_label(f_label, "");
	}
        
    } else {
        fl_set_object_label(f_label, L_NOTFOUND);
        fl_set_object_label(f_title, "");
    }

    /* update the XPM buttons: */

    if (update_xpm) {

	deb("Have to update the buttons.");

        fl_free_pixmap_pixmap(f_play);
        fl_free_pixmap_pixmap(f_stop);
        fl_free_pixmap_pixmap(f_pause);
        fl_free_pixmap_pixmap(f_lgo);
        
        if (cur_cdmode == PLAYING) {
            fl_set_pixmap_data(f_play, xpm_pl_on);
        } else {
            fl_set_pixmap_data(f_play, xpm_pl_off);
        } 

	if (cur_cdmode == PAUSED) {
            fl_set_pixmap_data(f_pause, xpm_pa_on);
        } else {
            fl_set_pixmap_data(f_pause, xpm_pa_off);
        }
        
        if ((cur_cdmode != PLAYING) && (cur_cdmode != PAUSED)) {
            fl_set_pixmap_data(f_stop, xpm_st_on);
        } else {
            fl_set_pixmap_data(f_stop, xpm_st_off);
        }
        
        if (loop_mode) {
            fl_set_pixmap_data(f_lgo, loop_on);
        } else {
            fl_set_pixmap_data(f_lgo, loop_off);
        }

        if (intro_mode) {
            fl_set_pixmap_data(f_intro, xpm_intro_on);
        } else {
            fl_set_pixmap_data(f_intro, xpm_intro_off);
        }
       
        update_xpm = 0;
    }

    /* no screen update if there's the track list. */
    
    if (! f_liste->visible) {

	/* help lines stay on screen only for 3 seconds... */
	if (aide_timeout > 0) {
	    fl_gettime(&tm, &mtm);
	    if ((tm - aide_timeout) > 3) {
		fl_set_object_label(f_looplabel, "");
		aide_timeout = 0;
	    }
	}

        /* update the track display */
    
        sprintf(dummy, "%d", cur_track);
        
        if (strlen(dummy) == 1) 
            sprintf(track, "0%s", dummy);
        else
            strcpy(track, dummy);

        fl_set_object_label(f_tracknbr, track);
        
        sprintf(dummy, "%d", cur_ntracks);
        if (strlen(dummy) == 1) 
            sprintf(track, "0%s", dummy);
        else
            strcpy(track, dummy);
        fl_set_object_label(f_trackmax, track);
        
        /* update the counters */

        if ((cur_cdmode == PLAYING) || (cur_cdmode == PAUSED)) {

            sc = cur_cdlen % 60;
            mn = cur_cdlen / 60;
            sprintf(cdtime, "%02d.%02d", mn, sc);
            fl_set_object_label(f_totdisk, cdtime);
            
            sc = cur_tracklen % 60;
            mn = cur_tracklen / 60;
            sprintf(cdtime, "%02d.%02d", mn, sc);
            fl_set_object_label(f_tottrack, cdtime);
            
            sc = cur_pos_rel % 60;
            mn = cur_pos_rel / 60;
            sprintf(cdtime, "%02d.%02d", mn, sc);
            fl_set_object_label(f_eltrack, cdtime);
            
            sc = cur_pos_abs % 60;
            mn = cur_pos_abs / 60;
            sprintf(cdtime, "%02d.%02d", mn, sc);
            fl_set_object_label(f_eldisk, cdtime);
            
            sc = (cur_cdlen - cur_pos_abs) % 60;
            mn = (cur_cdlen - cur_pos_abs) / 60;
            sprintf(cdtime, "%02d.%02d", mn, sc);
            fl_set_object_label(f_redisk, cdtime);
            
            sc = (cur_tracklen - cur_pos_rel) % 60;
            mn = (cur_tracklen - cur_pos_rel) / 60;
            sprintf(cdtime, "%02d.%02d", mn, sc);
            fl_set_object_label(f_retrack, cdtime);
        } else {
            fl_set_object_label(f_eltrack, "");
            fl_set_object_label(f_eldisk, "");
            fl_set_object_label(f_retrack, "");
            fl_set_object_label(f_redisk, "");
        }
        
	/* update the loop display: */
        
        if ( ((loop_start_track > 0) || (loop_end_track > 0)) && (! intro_mode) ) {
	    /*
          if (loop_mode) fl_set_object_label(f_looplabel, L_LOOPING);
            else fl_set_object_label(f_looplabel, L_LOOP);
	    */
            
            sprintf(track, "%02d", loop_start_track);
            fl_set_object_label(f_lstarttrack, track);
            
            sprintf(track, "%02d", loop_end_track);
            fl_set_object_label(f_lendtrack, track);
            
            sc = (loop_1) % 60;
            mn = (loop_1) / 60;
            sprintf(cdtime, "%02d.%02d", mn, sc);
            fl_set_object_label(f_lstarttime, cdtime);
            
            sc = (loop_2) % 60;
            mn = (loop_2) / 60;
            sprintf(cdtime, "%02d.%02d", mn, sc);
            fl_set_object_label(f_lendtime, cdtime);
        } else {
            fl_set_object_label(f_lstarttrack, "");
            fl_set_object_label(f_lendtrack, "");
            fl_set_object_label(f_lstarttime, "");
            fl_set_object_label(f_lendtime, "");
            /*fl_set_object_label(f_looplabel, "");*/
        }

        /*if (intro_mode) fl_set_object_label(f_looplabel, L_SCAN);*/
    
        /* update the 2 sliders */

        fl_set_slider_bounds(f_vol, 0, MAX_VOL);
        fl_set_slider_bounds(f_trackslider, 0, (double)cur_tracklen);
        fl_set_slider_bounds(f_diskslider, 0, (double)cur_cdlen);
        
        if ((cur_cdmode == PLAYING) || (cur_cdmode == PAUSED)) {
            fl_set_slider_value(f_trackslider, (double)cur_pos_rel);
            fl_set_slider_value(f_diskslider, (double)cur_pos_abs);
        } else {
            fl_set_slider_value(f_trackslider, 0);
            fl_set_slider_value(f_diskslider, 0);
        }
    
    }
    /* that's all, unfreeze window */
    
    fl_unfreeze_form(f_main);
    
    return 0;
} 
	  

	     
void db_close_cb(FL_OBJECT * ob, long data) 
{
    fl_hide_form(f_database);
} 

void db_autoplay_cb(FL_OBJECT * ob, long data) 
{
    if (fl_get_button(f_db_autoplay) == 0) db_play = 0;
    else db_play = 1;
    
} 

void db_browser_cb(FL_OBJECT * ob, long data) 
{
    char dummy[40];
    
    db_edit = fl_get_browser(f_db_browser);
    
    fl_set_input(f_db_track, fl_get_browser_line(f_db_browser, db_edit) + 3);
    sprintf(dummy, L_TRACKNAME, db_edit);
    fl_set_object_label(f_db_tracklabel, dummy);
    fl_set_button(f_db_cont, cd->trk[fl_get_browser(f_db_browser) - 1].contd);

    if (db_play) {
	direct_track = db_edit;
	cd_control(DIRECTTRACK);
    }
    
} 

void db_artist_cb(FL_OBJECT * ob, long data) 
{
    strcpy(cd->artist, fl_get_input(f_db_artist));
    save();
    load();
} 

void db_title_cb(FL_OBJECT * ob, long data) 
{
    strcpy(cd->cdname, fl_get_input(f_db_title));
    save();
    load();
} 

void db_track_cb(FL_OBJECT * ob, long data) 
{
    int i;
    char txt[100];
    
    cd->trk[db_edit - 1].songname = malloc(strlen(fl_get_input(f_db_track)) + 1);
    strcpy(cd->trk[db_edit - 1].songname, fl_get_input(f_db_track));
    save();
    load();

    i = db_edit++ ;
    
    if (i < thiscd.ntracks) {
	db_edit = i + 1;
    } else {
        db_edit = 1;
    }
    
    fl_clear_browser(f_db_browser); 
    for (i = 0; i < cur_ntracks; i++) {
        sprintf(txt, "%2d ", i + 1);
        if (cd->trk[i].songname != NULL) strcat(txt, cd->trk[i].songname);
        fl_add_browser_line(f_db_browser, txt);
    }
    
    if (cd->trk[db_edit - 1].songname != NULL) {
        fl_set_input(f_db_track, cd->trk[db_edit - 1].songname);
    } else {
        fl_set_input(f_db_track, "");
    }
    
    sprintf(txt, L_TRACKNAME, db_edit);
    fl_set_object_label(f_db_tracklabel, txt);
    
    fl_set_focus_object(f_database, f_db_track);
} 

void o_close_cb(FL_OBJECT * ob, long data) 
{
    options_window();
} 

void o_save_cb(FL_OBJECT * ob, long data) 
{
    FILE *out;
    struct passwd *pw;
    char st[128];

    if (NULL == (pw = getpwuid(getuid()))) {
	/* hum... big problem... */
        return;
    } else {
        sprintf(st, "%s/.xfascdrc", pw->pw_dir);
        if ((out = fopen(st, "w"))) {
            fprintf(out, "#### Generated by %s %s ####\n", PROGRAM, VERSION);
            fprintf(out, "MAIN_X=%d\n", f_main->x);
            fprintf(out, "MAIN_Y=%d\n", f_main->y);
            fprintf(out, "OPLACE=%d\n", o_place);
            fprintf(out, "DBPLACE=%d\n", db_place);
#ifndef NO_D_DEVICE
	    fprintf(out, "CD_DEVICE=%s\n", cd_device);
#endif	   
            fprintf(out, "AUTOREPEAT=%d\n", autorepeat);
            fprintf(out, "AUTOPLAY=%d\n", autoplay);
            fprintf(out, "AUTODBPLAY=%d\n", db_play);
            fprintf(out, "CUE_TIME=%d\n", atoi(fl_get_input(o_cue)));
            fclose(out);
            options_window();
            return;
        } else {
            return;
        }
    }
} 

void o_autoplay_cb(FL_OBJECT * ob, long data) 
{
    if (fl_get_button(o_autoplay) == 0) autoplay = 0;
    else autoplay = 1;
} 

void o_autorepeat_cb(FL_OBJECT * ob, long data) 
{
    if (fl_get_button(o_autorepeat) == 0) autorepeat = 0;
    else autorepeat = 1;
} 

void o_db_autoplay_cb(FL_OBJECT * ob, long data) 
{
    if (fl_get_button(o_db_autoplay) == 0) db_play = 0;
    else db_play = 1;
} 


void o_dcenter_cb(FL_OBJECT * ob, long data) 
{
    db_place = CENTER;
} 
void o_dleft_cb(FL_OBJECT * ob, long data) 
{
    db_place = LEFT;
} 
void o_dright_cb(FL_OBJECT * ob, long data) 
{
    db_place = RIGHT;
} 
void o_dtop_cb(FL_OBJECT * ob, long data) 
{
    db_place = TOP;
} 
void o_dbottom_cb(FL_OBJECT * ob, long data) 
{
    db_place = BOTTOM;
} 


void o_ocenter_cb(FL_OBJECT * ob, long data) 
{
    o_place = CENTER;
} 
void o_oleft_cb(FL_OBJECT * ob, long data) 
{
    o_place = LEFT;
} 
void o_oright_cb(FL_OBJECT * ob, long data) 
{
    o_place = RIGHT;
} 
void o_otop_cb(FL_OBJECT * ob, long data) 
{
    o_place = TOP;
} 
void o_obottom_cb(FL_OBJECT * ob, long data) 
{
    o_place = BOTTOM;
} 


/* ============================================================================
 * Main:
 * ============================================================================ */ 

int main(int argc, char *argv[])
{
    extern char *rcfile, *dbfiles;  
    char dummy[256];
    int popup, more_popup;
   
    /* default values: */

#ifndef NO_D_DEVICE
    cd_device = malloc(strlen(DEFAULTDEVICE) + 1);
    strcpy(cd_device, DEFAULTDEVICE);
#endif
     
    main_x = main_y = 10;
    o_place = BOTTOM;
    db_place = RIGHT;

    /* init the WorkMan CD database: */

    rcfile = getenv("WORKMANRC");
    if (rcfile)
	rcfile = (char *)WMstrdup(rcfile);
    dbfiles = getenv("WORKMANDB");
    if (dbfiles)
	dbfiles = (char *)WMstrdup(dbfiles);

    split_workmandb();
    
    /* load settings in ~/.xfascrc file: */

    load_rc_file();
    
    /* the command-line options have priority: */

    parse_commandline(argc, argv);
   
    deb("Initialising Forms...");
    fl_initialize(&argc, argv, 0, 0, 0);
    deb("Create the Forms...");
    create_the_forms();

    /* make the popups: */

    deb("Popups...");
    more_popup = fl_newpup(fl_default_win());
    fl_addtopup(more_popup, L_OPTPOPUP);
    fl_set_menu_popup(f_more, more_popup);
    
    /* change a few things in XForms default properties: */

    deb("Adjusting Forms objects properties...");
    fl_setpup_fontsize(10);
    fl_setpup_fontstyle(0);

    fl_set_object_bw(f_play, XBBW);
    fl_set_object_bw(f_pause, XBBW);
    fl_set_object_bw(f_stop, XBBW);
    fl_set_object_bw(f_prev, XBBW);
    fl_set_object_bw(f_prevtrack, XBBW);
    fl_set_object_bw(f_next, XBBW);
    fl_set_object_bw(f_nexttrack, XBBW);
    fl_set_object_bw(f_lgo, XBBW);
    
    fl_set_object_bw(f_more, BBW);
    fl_set_object_bw(f_lstart, XBBW);
    fl_set_object_bw(f_lend, XBBW);
    fl_set_object_bw(f_intro, XBBW);

    fl_set_object_bw(f_label, PBW);
    fl_set_object_bw(f_title, PBW);
    fl_set_object_bw(f_display, PBW);
    fl_set_object_bw(f_display2, PBW);

    /*    fl_set_object_bw(f_rel, PBW);
	  fl_set_object_bw(f_abs, PBW);*/
    fl_set_object_bw(f_lab1, PBW);
    fl_set_object_bw(f_lab2, PBW);

    fl_set_object_bw(f_trackslider, SBW);
    fl_set_object_bw(f_diskslider, SBW);

    fl_set_object_label(f_label, "");

    fl_set_object_label(f_lbl1, L_LBL1);
    fl_set_object_label(f_lbl2, L_LBL2);

    fl_set_object_label(f_lbl4, L_LBL4);
    fl_set_object_label(f_lbl5, L_LBL5);
    fl_set_object_label(f_lab1, L_LBL1);
    fl_set_object_label(f_lab2, L_LBL2);
    
    fl_set_object_label(f_db_autoplay, L_DBAUTOPLAY);
    fl_set_object_label(f_db_cont, L_DBCONT);
    fl_set_object_label(f_db_close, L_DBCLOSE);
    fl_set_object_label(f_db_artist, L_DBARTIST);
    fl_set_object_label(f_db_title, L_DBTITLE);

    fl_set_object_label(o_close, L_DBCLOSE);
    fl_set_object_label(o_save, L_OSAVE);
    fl_set_object_label(o_autorepeat, L_OAUTOREPEAT);
    fl_set_object_label(o_autoplay, L_OAUTOPLAY);
    fl_set_object_label(f_seconds, L_SECONDS);

    fl_set_object_label(o_dcenter, L_CENTER);
    fl_set_object_label(o_dleft, L_LEFT);
    fl_set_object_label(o_dright, L_RIGHT);
    fl_set_object_label(o_dtop, L_TOP);
    fl_set_object_label(o_dbottom, L_BOTTOM);

    fl_set_object_label(o_ocenter, L_CENTER);
    fl_set_object_label(o_oleft, L_LEFT);
    fl_set_object_label(o_oright, L_RIGHT);
    fl_set_object_label(o_otop, L_TOP);
    fl_set_object_label(o_obottom, L_BOTTOM);


    /* Init the call back functions: */

    deb("Setting Forms callbacks...");
    fl_set_object_callback(f_title, title_cb, 0);
    fl_set_object_callback(f_label, label_cb, 0);
    fl_set_object_callback(f_liste, liste_cb, 0);

    fl_set_object_callback(f_play, play_cb, 0);
    fl_set_object_callback(f_pause, pause_cb, 0);
    fl_set_object_callback(f_stop, stop_cb, 0);
    fl_set_object_callback(f_prevtrack, prevtrack_cb, 0);
    fl_set_object_callback(f_nexttrack, nexttrack_cb, 0);
    fl_set_object_callback(f_prev, prev_cb, 0);
    fl_set_object_callback(f_next, next_cb, 0);

    fl_set_object_callback(f_more, more_cb, 0);

    fl_set_object_callback(f_lstart, lstart_cb, 0);
    fl_set_object_callback(f_lgo, lgo_cb, 0);
    fl_set_object_callback(f_lend, lend_cb, 0);

    fl_set_object_callback(f_intro, intro_cb, 0);
 
    fl_set_object_callback(f_trackslider, trackslider_cb, 0);
    fl_set_object_callback(f_diskslider, diskslider_cb, 0);
    fl_set_object_callback(f_vol, vol_cb, 0);

    fl_set_object_callback(f_db_close, db_close_cb, 0);
    fl_set_object_callback(f_db_autoplay, db_autoplay_cb, 0);
    fl_set_object_callback(f_db_browser, db_browser_cb, 0);
    fl_set_object_callback(f_db_artist, db_artist_cb, 0);
    fl_set_object_callback(f_db_title, db_title_cb, 0);
    fl_set_object_callback(f_db_track, db_track_cb, 0);
    
    fl_set_object_callback(o_close, o_close_cb, 0);
    fl_set_object_callback(o_save, o_save_cb, 0);
    fl_set_object_callback(o_autoplay, o_autoplay_cb, 0);
    fl_set_object_callback(o_autorepeat, o_autorepeat_cb, 0);
    fl_set_object_callback(o_db_autoplay, o_db_autoplay_cb, 0);

    fl_set_object_callback(o_dcenter, o_dcenter_cb, 0);
    fl_set_object_callback(o_dleft, o_dleft_cb, 0);
    fl_set_object_callback(o_dright, o_dright_cb, 0);
    fl_set_object_callback(o_dtop, o_dtop_cb, 0);
    fl_set_object_callback(o_dbottom, o_dbottom_cb, 0);
    
    fl_set_object_callback(o_ocenter, o_ocenter_cb, 0);
    fl_set_object_callback(o_oleft, o_oleft_cb, 0);
    fl_set_object_callback(o_oright, o_oright_cb, 0);
    fl_set_object_callback(o_otop, o_otop_cb, 0);
    fl_set_object_callback(o_obottom, o_obottom_cb, 0);

    /* set the pixmaps: */

    deb("Setting buttons pixmaps...");
    fl_set_pixmap_data(f_play, xpm_pl_off);
    fl_set_pixmap_data(f_stop, xpm_st_off);
    fl_set_pixmap_data(f_pause, xpm_pa_off);
    fl_set_pixmap_data(f_prevtrack, xpm_prev);
    fl_set_pixmap_data(f_nexttrack, xpm_next);
    fl_set_pixmap_data(f_prev, xpm_rew);
    fl_set_pixmap_data(f_next, xpm_cue);
    fl_set_pixmap_data(f_lgo, loop_off);
    fl_set_pixmap_data(f_lstart, xpm_lleft);
    fl_set_pixmap_data(f_lend, xpm_lright);
    fl_set_pixmap_data(f_intro, xpm_intro_off);

#ifdef BLUE_BACK
    deb("Setting background pixmaps...");
    fl_set_pixmap_data(f_background, gradient);
    fl_set_pixmap_data(o_background, gradient);
    fl_set_pixmap_data(db_background, gradient);
    
    fl_set_pixmap_data(f_title, gradient_s);
#endif
    
    /* this function handle CD events and redraw the screen: */

    deb("Initialising idle callback...");
    fl_set_idle_callback(cd_events, 0);

    deb("Checking CD status...");
    cd_status();
    if (debug) sprintf(dummy, "  -> current status is [%d]", cur_cdmode);
    deb(dummy);
    
    deb("CD Volume...");
    volume = MAX_VOL;
    deb("  -> setting hardware CD volume");
    if (cur_cdmode != EJECTED) cd_volume(volume, 10, MAX_VOL);
    deb("  -> setting slider boundaries");
    fl_set_slider_bounds(f_vol, 0, MAX_VOL);
    deb("  -> setting slider default position");
    fl_set_slider_value(f_vol, volume);
    deb("... done.");

    deb("Dealing with CD Mode...");

    if ((autoplay) && (cur_cdmode == STOPPED)) {
	deb("  -> Autoplay enabled. Switching to PLAY mode...");
	cur_track = 1;
	cd_control(PLAY);
	cd_status();
	deb("     ... done.");
    }

    deb("... done.");

    deb("Display main window...");
    fl_hide_object(f_liste);
    fl_set_form_position(f_main, main_x, main_y);
    fl_show_form(f_main, FL_PLACE_SIZE|FL_PLACE_GEOMETRY, FL_FULLBORDER, PROGRAM);

    aide(L_H_STARTUP, 0);

    deb("Entering main loop...");
    
    /* The Main Loop: */
    
    while (1) {
	fl_do_forms();
    }
    
    return 0;
}
