/*
 * Copyright (c) 1999  Albert Dorofeev <Albert@mail.dma.be>
 * For the updates see http://bewoner.dma.be/Albert/
 *
 * This software is distributed under GPL. For details see LICENSE file.
 */

/* kvm/uvm use (BSD port) code:
 * Copyright (c) 2000  Scott Aaron Bamford <sab@zeekuschris.com>
 * BSD additions for for this code are licensed BSD style.
 * All other code and the project as a whole is under the GPL.
 * For details see LICENSE.
 * BSD systems dont have /proc/meminfo. it is still posible to get the disired
 * information from the uvm/kvm functions. Linux machines shouldn't have
 * <uvm/vum_extern.h> so should use the /proc/meminfo way. BSD machines (NetBSD
 * i use, but maybe others?) dont have /proc/meminfo so we instead get our info
 * using kvm/uvm.
 */

#include <stdio.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>

#include "state.h"

#include "config.h"

#ifdef HAVE_UVM_UVM_EXTERN_H
/* sab - 2000/01/21
 * this should only happen on *BSD and will use the BSD kvm/uvm interface
 * instead of /proc/meminfo
 */
#include <sys/types.h>
#include <sys/param.h>
#include <sys/sysctl.h>
  
#include <uvm/uvm_extern.h>
#endif /* HAVE_UVM_UVM_EXTERN_H */

extern struct asmem_state state;

#ifndef HAVE_UVM_UVM_EXTERN_H
#define BUFFER_LENGTH 400
int fd;
char buf[BUFFER_LENGTH];
#endif /* !HAVE_UVM_UVM_EXTERN */

void error_handle( int place, const char * message )
{
	int error_num;
	error_num = errno;
	/* if that was an interrupt - quit quietly */
	if (error_num == EINTR) {
		printf("asmem: Interrupted.\n");
		return;
	}
	switch ( place )
	{
	case 1: /* opening the /proc/meminfo file */
		switch (error_num)
		{
		case ENOENT :
			printf("asmem: The file %s does not exist. "
			"Weird system it is.\n", state.proc_mem_filename);
			break;
		case EACCES :
			printf("asmem: You do not have permissions "
			"to read %s\n", state.proc_mem_filename);
			break;
		default :
			printf("asmem: cannot open %s. Error %d: %s\n",
				state.proc_mem_filename, errno,
				sys_errlist[errno]);
			break;
		}
		break;
	default: /* catchall for the rest */
		printf("asmem: %s: Error %d: %s\n",
			message, errno, sys_errlist[errno]);
	}
}

#ifdef DEBUG
/* sab - 2000/01/21
 * Moved there here so it can be used in both BSD style and /proc/meminfo style
 * without repeating code and alowing us to keep the two main functions seperate
 */
#define verb_debug() { \
       printf("+- Total : %ld, used : %ld, free : %ld \n", \
                       state.fresh.total, \
                       state.fresh.used,\
                       state.fresh.free);\
       printf("|  Shared : %ld, buffers : %ld, cached : %ld \n",\
                       state.fresh.shared,\
                       state.fresh.buffers,\
                       state.fresh.cached);\
       printf("+- Swap total : %ld, used : %ld, free : %ld \n",\
                       state.fresh.swap_total,\
                       state.fresh.swap_used,\
                       state.fresh.swap_free);\
       }
#else
#define verb_debug()
#endif /* DEBUG */

#ifdef HAVE_UVM_UVM_EXTERN_H
/* using kvm/uvm (BSD systems) ... */

#define pagetok(size) ((size) << pageshift)

int read_meminfo()
{
      int pagesize, pageshift;
      int mib[2];
      size_t usize;
      struct uvmexp uvm_exp;

      /* get the info */
      mib[0] = CTL_VM;
      mib[1] = VM_UVMEXP;
      usize = sizeof(uvm_exp);
      if (sysctl(mib, 2, &uvm_exp, &usize, NULL, 0) < 0) {
        fprintf(stderr, "asmem: sysctl uvm_exp failed: %s\n",
            strerror(errno));
          return -1;
      }

      /* setup pageshift */
      pagesize = uvm_exp.pagesize;
      pageshift = 0;
      while (pagesize > 1)
      {
              pageshift++;
              pagesize >>= 1;
      }

      /* update state */
      state.fresh.total =  pagetok(uvm_exp.npages);
      state.fresh.used = pagetok(uvm_exp.active);
      state.fresh.free = pagetok(uvm_exp.free);
      state.fresh.shared = 0;  /* dont know how to get these */
      state.fresh.buffers = 0;
      state.fresh.cached = 0;
      state.fresh.swap_total =  pagetok(uvm_exp.swpages);
      state.fresh.swap_used = pagetok(uvm_exp.swpginuse);
      state.fresh.swap_free = pagetok(uvm_exp.swpages-uvm_exp.swpginuse);
      verb_debug();
      return 0;
}

#else
/* default /proc/meminfo (Linux) method ... */

int read_meminfo()
{
	int result;
	result = lseek(fd, 0, SEEK_SET);
	if ( result < 0 ) {
		error_handle(2, "seek");
		return -1;
	}
	result = read(fd, buf, sizeof buf);
	switch(result)
	{
	case 0 : /* Huh? End of file? Pretend this did not happen... */
		break;
	case -1 :
		error_handle(2, "read");
		return -1;
	default :
	}
	buf[result-1] = 0;
	result = sscanf(buf, "%*[^\n]%*s %ld %ld %ld %ld %ld %ld\n%*s %ld %ld %ld",
		&state.fresh.total,
		&state.fresh.used,
		&state.fresh.free,
		&state.fresh.shared,
		&state.fresh.buffers,
		&state.fresh.cached,
		&state.fresh.swap_total,
		&state.fresh.swap_used,
		&state.fresh.swap_free
		);
	switch(result)
	{
	case 0 :
	case -1 :
		printf("asmem: invalid input character while "
			"reading %s\n", state.proc_mem_filename);
		return -1;
	}
	verb_debug();
	return 0;
}

#endif /* (else) HAVE_UVM_UVM_EXTERN_H */

int open_meminfo()
{
#ifndef HAVE_UVM_UVM_EXTERN_H
	int result;
	if ((fd = open(state.proc_mem_filename, O_RDONLY)) == -1) {
		error_handle(1, "");
		return -1;
	}
#endif /* !HAVE_UVM_UVM_EXTERN_H */
	return 0;
}

int close_meminfo()
{
#ifndef HAVE_UVM_UVM_EXTERN_H
	close(fd);
#endif /* !HAVE_UVM_UVM_EXTERN_H */
	return 0;
}

