/*
 * Copyright (C) 1998 Ethan Fischer <allanon@crystaltokyo.com>
 */

#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <time.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/types.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xresource.h>
#include <X11/Intrinsic.h>
#include <X11/Xatom.h>

#include <X11/xpm.h>
#include <X11/extensions/shape.h>

#include "asexec.h"

Display* dpy;
int screen;
Window Root;
Colormap MyColormap;
GC DrawGC;
Window win;
unsigned int MyEvents = ExposureMask | ButtonPressMask | KeyPressMask | EnterWindowMask | LeaveWindowMask;
int x_fd;

Atom Protocols[1];
Atom WM_STATE_atom;
Atom WM_PROTOCOLS_atom;

int Width = 1;
int Height = 1;
int is_shaped = 0;

void follow_binding(binding_t* binding);
int My_XNextEvent(Display* dpy, XEvent* event);

void change_window_name (Window win, const char *str)
{
	XTextProperty name;

	XStringListToTextProperty ((char**)&str, 1, &name);
	XSetWMName (dpy, win, &name);
	XSetWMIconName (dpy, win, &name);
	XFree (name.value);
}

void draw_icon(void)
{
	if ((*current_state).icon.pixmap == None)
		XClearWindow(dpy, win);
	else
		XCopyArea(dpy, (*current_state).icon.pixmap, win, DrawGC, 
				  0, 0, (*current_state).icon.width, (*current_state).icon.height,
				  (Width - (*current_state).icon.width) / 2, (Height - (*current_state).icon.height) / 2);
}

void my_timer_handler(void* data)
{
	state_t* state = (state_t*)data;
	binding_t* binding;
	for (binding = (*state).binding ; binding != NULL ; binding = (*binding).next)
		if ((*binding).type == Type_Timeout)
			break;
	if (binding != NULL)
		follow_binding(binding);
}

void change_state(state_t* state)
{
	binding_t* binding;
	swallow_rec_t* rec;
	swallow_t* swallow;
	/* remove stale timeouts */
	while (timer_remove_by_data(current_state));
	/* find and add a timeout */
	for (binding = (*state).binding ; binding != NULL ; binding = (*binding).next)
		if ((*binding).type == Type_Timeout)
			break;
	if (binding != NULL)
		timer_new((*binding).timeout, my_timer_handler, state);
	/* make sure any swallowed windows are in the right place, and grab 
	 * any necessary buttons */
	for (swallow = first_swallow ; swallow != NULL ; swallow = (*swallow).next)
		if ((*swallow).window != None)
		{
			for (rec = (*state).swallow_rec ; rec != NULL ; rec = (*rec).next)
				if (!strcmp((*rec).swallow_label, (*swallow).name))
					break;
			if (rec != NULL)
			{
				if ((*rec).x == -1 && (*rec).y == -1)
					XMoveWindow(dpy, (*swallow).window, (Width - (*swallow).width) / 2, (Height - (*swallow).height) / 2);
				else
					XMoveWindow(dpy, (*swallow).window, (*rec).x, (*rec).y);
				for (binding = (*state).binding ; binding != NULL ; binding = (*binding).next)
					if ((*binding).type == Type_Mouse)
						XGrabButton(dpy, (*binding).button, AnyModifier, (*swallow).window, False, ButtonPressMask, GrabModeAsync, GrabModeAsync, None, None);
			}
			else
			{
				XMoveWindow(dpy, (*swallow).window, -999 - Width, -999 - Height);
				XUngrabButton(dpy, AnyButton, AnyModifier, (*swallow).window);
			}
		}
	/* that's all, if we're entering the same state */
	if (state == current_state)
		return;
	/* shape or unshape the window as necessary */
	if ((*state).icon.mask != None)
	{
		/* shape the window */
		XShapeCombineMask(dpy, win, ShapeBounding, (Width - (*state).icon.width) / 2, (Height - (*state).icon.height) / 2, (*state).icon.mask, ShapeSet);
		is_shaped = 1;
	}
	else if (is_shaped)
	{
		/* unshape the window */
		XShapeCombineMask(dpy, win, ShapeBounding, 0, 0, None, ShapeSet);
#if 0
		XRectangle rect;
		rect.x = (Width - (*state).icon.width) / 2;
		rect.y = (Height - (*state).icon.height) / 2;
		rect.width = Width;
		rect.height = Height;
		XShapeCombineRectangles(dpy, win, ShapeBounding, 0, 0, &rect, 1, ShapeSet, Unsorted);
#endif
		is_shaped = 0;
	}
	/* change state */
	current_state = state;
	/* update display */
	draw_icon();
}

void follow_binding(binding_t* binding)
{
	state_t* state = find_state_by_name((*binding).child);
	if (state != NULL)
	{
		if ((*binding).exec != NULL && !fork())
			if (execl ("/bin/sh", "/bin/sh", "-c", (*binding).exec, NULL) == -1)
				exit (100);
		change_state(state);
	}
}

void asexec_init1(int argc, char** argv)
{
	dpy = XOpenDisplay(NULL);
	x_fd = ConnectionNumber(dpy);
	screen = DefaultScreen(dpy);
	Root = DefaultRootWindow(dpy);
{
	XGCValues gcv;
	gcv.graphics_exposures = False;
	DrawGC = XCreateGC(dpy, Root, GCGraphicsExposures, &gcv);
}
	Protocols[0] = XInternAtom (dpy, "WM_DELETE_WINDOW", False);
	WM_PROTOCOLS_atom = XInternAtom(dpy, "WM_PROTOCOLS", False);
{
	win = XCreateSimpleWindow(dpy, Root, 64, 64, 64, 64, 0, 0, 0);
	XSetWMProtocols(dpy, win, Protocols, 1);
	change_window_name(win, NAME);
	XSelectInput(dpy, win, MyEvents);
}
{
	XWindowAttributes attr;
	XGetWindowAttributes(dpy, win, &attr);
	MyColormap = attr.colormap;
}
{
	WM_STATE_atom = XInternAtom(dpy, "WM_STATE", True);
	/* watch for mapping notifies, so we can swallow other apps */
	XSelectInput(dpy, Root, SubstructureNotifyMask);
}
}

void asexec_init2(int argc, char** argv)
{
	state_t* state;
	swallow_t* swallow;
	/* if the user hasn't set the size, set it to the maximum of our pixmaps */
	if (Width == 1 && Height == 1)
		for (state = first_state ; state != NULL ; state = (*state).next)
		{
			if (Width < (*state).icon.width)
				Width = (*state).icon.width;
			if (Height < (*state).icon.height)
				Height = (*state).icon.height;
		}
	XResizeWindow(dpy, win, Width, Height);
	change_state(current_state);
	XMapRaised(dpy, win);
	/* start our children */
	for (swallow = first_swallow ; swallow != NULL ; swallow = (*swallow).next)
		if ((*swallow).exec != NULL && !fork())
			if (execl ("/bin/sh", "/bin/sh", "-c", (*swallow).exec, NULL) == -1)
				exit (100);
}

Window find_client_window_recurse(Window w)
{
	Atom type = None;
	Window result = None;
	int format;
	unsigned long nitems, after;
	unsigned char *data;
	if (WM_STATE_atom == None)
		return None;
	XGetWindowProperty(dpy, w, WM_STATE_atom, 0, 0, False, AnyPropertyType, &type, &format, &nitems, &after, &data);
	if (type == None)
	{
		Window root, parent, *children;
		unsigned int nchildren;
		if (XQueryTree(dpy, w, &root, &parent, &children, &nchildren))
		{
			int i;
			for (i = 0 ; i < nchildren ; i++)
			{
				result = find_client_window_recurse(children[i]);
				if (result != None)
					break;
			}
			if (children != NULL)
				XFree(children);
		}
	}
	else
		result = w;
	return result;
}

Window find_client_window(Window w)
{
	Window result = find_client_window_recurse(w);
	if (result == None)
		result = w;
	return result;
}

/***************************************************************************
 *
 * ICCCM Client Messages - Section 4.2.8 of the ICCCM dictates that all
 * client messages will have the following form:
 *
 *     event type	ClientMessage
 *     message type	WM_PROTOCOLS
 *     window		window
 *     format		32
 *     data[0]		message atom
 *     data[1]		time stamp
 *
 ****************************************************************************/
void send_clientmessage(Window window, Atom atom, Time timestamp)
{
  XEvent event;

  event.xclient.type = ClientMessage;
  event.xclient.window = window;
  event.xclient.message_type = WM_PROTOCOLS_atom;
  event.xclient.format = 32;
  event.xclient.data.l[0] = atom;
  event.xclient.data.l[1] = timestamp;
  XSendEvent(dpy, window, False, 0, &event);
}

void asexec_done(void)
{
	swallow_t* swallow;
	for (swallow = first_swallow ; swallow != NULL ; swallow = (*swallow).next)
		if ((*swallow).window != None)
		{
			send_clientmessage((*swallow).window, Protocols[0], CurrentTime);
			(*swallow).window = None;
			XSync(dpy, 0);
		}
	exit(0);
}

void asexec_handle_events(void)
{
	XEvent event;
	if (My_XNextEvent(dpy, &event))
	{
		switch (event.type)
		{
		case Expose:
			draw_icon();
			break;
		case ClientMessage:
			if (event.xclient.format == 32 && event.xclient.data.l[0] == Protocols[0])
				asexec_done();
			break;
		case KeyPress:
		{
#if 0
			KeySym keysym = XKeycodeToKeysym(dpy, event.xkey.keycode, 0);
printf("keypress %04x == %04x == '%c'\n", event.xkey.keycode, keysym, keysym);
#endif
		}
			break;
		case ButtonPress:
		{
			binding_t* binding;
			for (binding = (*current_state).binding ; binding != NULL ; binding = (*binding).next)
				if ((*binding).type == Type_Mouse && (*binding).button == event.xbutton.button)
					break;
			if (binding != NULL)
				follow_binding(binding);
		}
			break;
		case EnterNotify:
		if (event.xcrossing.detail != NotifyInferior)
		{
			binding_t* binding;
			for (binding = (*current_state).binding ; binding != NULL ; binding = (*binding).next)
				if ((*binding).type == Type_Enter)
					break;
			if (binding != NULL)
				follow_binding(binding);
		}
			break;
		case LeaveNotify:
		if (event.xcrossing.detail != NotifyInferior)
		{
			binding_t* binding;
			for (binding = (*current_state).binding ; binding != NULL ; binding = (*binding).next)
				if ((*binding).type == Type_Leave)
					break;
			if (binding != NULL)
				follow_binding(binding);
		}
			break;
		case MapNotify:
		{
			swallow_t* swallow;
			XTextProperty prop;
			Window w;
			for (swallow = first_swallow ; swallow != NULL ; swallow = (*swallow).next)
				if ((*swallow).window == event.xmap.window)
				{
					XSetWindowBorderWidth(dpy, (*swallow).window, 0);
					break;
				}
			if (swallow != NULL)
				break;
			w = find_client_window(event.xmap.window);
			if (XGetWMName(dpy, w, &prop))
			{
				for (swallow = first_swallow ; swallow != NULL ; swallow = (*swallow).next)
					if (!strcmp((*swallow).window_name, prop.value))
						break;
				if (swallow != NULL && (*swallow).window == None)
				{
					Window root;
					int junk;
					(*swallow).window = w;
					XSelectInput(dpy, w, StructureNotifyMask);
					XGetGeometry(dpy, w, &root, &junk, &junk, &(*swallow).width, &(*swallow).height, &junk, &junk);
					if (Width < (*swallow).width || Height < (*swallow).height)
					{
						if (Width < (*swallow).width)
							Width = (*swallow).width;
						if (Height < (*swallow).height)
							Height = (*swallow).height;
						XResizeWindow(dpy, win, Width, Height);
					}
					/* place the window out of sight, and let change_state() fix it */
					XReparentWindow(dpy, w, win, -999 - Width, -999 - Height);
					change_state(current_state);
				}
			}
		}
			break;
		default:
			break;
		}
	}
}

int My_XNextEvent(Display* dpy, XEvent* event)
{
  fd_set in_fdset;
  struct timeval tv;
  struct timeval *t = NULL;

  if (XPending (dpy))
    {
      XNextEvent (dpy, event);
      return 1;
    }

  FD_ZERO (&in_fdset);
  FD_SET (x_fd, &in_fdset);
  if (timer_delay_till_next_alarm ((time_t *) & tv.tv_sec, (time_t *) & tv.tv_usec))
    t = &tv;

#ifdef __hpux
  select (x_fd + 1, (int *) &in_fdset, NULL, NULL, t);
#else
  select (x_fd + 1, &in_fdset, NULL, NULL, t);
#endif

  /* get X events */
  if (FD_ISSET (x_fd, &in_fdset))
    {
      XNextEvent (dpy, event);
      return 1;
    }

  /* handle timeout events */
  timer_handle ();

  return 0;
}

char* find_icon(const char* name)
{
	const char* ppath = PixmapPath;
	char* path = malloc(strlen(PixmapPath) + 1 + strlen(name) + 1);
	while (*ppath != '\0')
	{
		FILE* fp;
		const char* ptr;
		for (ptr = ppath ; *ptr != '\0' && *ptr != ':' ; ptr++);
		sprintf(path, "%.*s/%s", ptr - ppath, ppath, name);
		if ((fp = fopen(path, "r")) != NULL)
		{
			fclose(fp);
			return path;
		}
		ppath = ptr;
		if (*ppath == ':')
			ppath++;
	}
	free(path);
	return NULL;
}

int load_icon(const char* name, icon_t* icon)
{
	XpmAttributes xpm_attributes;
	char* path = find_icon(name);
	
	if (path == NULL)
		return 1;

	xpm_attributes.valuemask = XpmSize | XpmReturnPixels | XpmColormap;
	xpm_attributes.colormap = MyColormap;

	if (XpmReadFileToPixmap(dpy, Root, path, &(*icon).pixmap, &(*icon).mask, &xpm_attributes) != XpmSuccess)
	{
		free(path);
		return 1;
	}

	(*icon).width = xpm_attributes.width;
	(*icon).height = xpm_attributes.height;
	free(path);

	return 0;
}


