/*
 * Copyright (C) 1998 Ethan Fischer <allanon@crystaltokyo.com>
 */

#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xresource.h>
#include <X11/Intrinsic.h>
#include <X11/Xatom.h>

#include <X11/xpm.h>

#include "asexec.h"

swallow_t* first_swallow = NULL;
state_t* first_state = NULL;
state_t* current_state = NULL;
const char* MyName;
char* PixmapPath = ".";
void change_window_name(Window win, const char *str);

char* parse_quoted_token(char* str, int* len_parsed, int action)
{
	char* ptr1;
	char* ptr2;
	char* result = NULL;
	int len = 0;
	for (ptr1 = str ; *ptr1 != '\0' && *ptr1 != '"' ; ptr1++);
	if (*ptr1 != '\0')
		ptr1++;
	for (ptr2 = ptr1 ; *ptr2 != '\0' && *ptr2 != '"' ; ptr2++);
	if (*ptr2 != '\0')
	{
		len = ptr2 - str + 1;
		if (action == 1) /* allocate */
			result = mystrndup(ptr1, ptr2 - ptr1);
		else if (action == 2) /* modify */
		{
			result = ptr1;
			*ptr2 = '\0';
		}
	}
	if (len_parsed != NULL)
		*len_parsed = len;
	return result;
}

char* strip_whitespace(char* str, int* len_parsed, int action)
{
	char* ptr1;
	char* ptr2;
	char* result = NULL;
	int len = 0;
	/* strip leading and trailing whitespace */
	for (ptr1 = str ; isspace(*ptr1) ; ptr1++);
	ptr2 = ptr1 + strlen(ptr1);
	if (*ptr1 != '\0')
		for ( ; isspace(*(ptr2 - 1)) && ptr2 > ptr1 ; ptr2--);
	len = ptr2 - str;
	if (*ptr2 != '\0')
		len++;
	if (action == 1) /* allocate */
		result = mystrndup(ptr1, ptr2 - ptr1);
	else if (action == 2) /* modify */
	{
		result = ptr1;
		*ptr2 = '\0';
	}
	if (len_parsed != NULL)
		*len_parsed = len;
	return result;
}

swallow_t* new_swallow(void)
{
	swallow_t* swallow = malloc(sizeof(swallow_t));
	(*swallow).next = first_swallow;
	(*swallow).name = NULL;
	(*swallow).window_name = NULL;
	(*swallow).window = None;
	first_swallow = swallow;
	return swallow;
}

void delete_swallow(swallow_t* swallow, int unparent_child)
{
	/* unlink from the list of swallows */
	if (first_swallow == swallow)
		first_swallow = (*swallow).next;
	else if (first_swallow != NULL)
	{
		swallow_t* ptr;
		for (ptr = first_swallow ; (*ptr).next != NULL ; ptr = (*ptr).next)
			if ((*ptr).next == swallow)
				break;
		if (ptr != NULL)
			(*ptr).next = (*swallow).next;
	}
	/* free our members */
	if ((*swallow).name != NULL)
		free((*swallow).name);
	if ((*swallow).window_name != NULL)
		free((*swallow).window_name);
	if (unparent_child && (*swallow).window != None)
		XReparentWindow(dpy, (*swallow).window, Root, 0, 0);
	/* free our own mem */
	free(swallow);
}

swallow_rec_t* new_swallow_rec(state_t* state)
{
	swallow_rec_t* swallow_rec = malloc(sizeof(swallow_rec_t));
	(*swallow_rec).next = (*state).swallow_rec;
	(*swallow_rec).swallow_label = NULL;
	(*swallow_rec).parent = state;
	(*swallow_rec).x = -1;
	(*swallow_rec).y = -1;
	(*state).swallow_rec = swallow_rec;
	return swallow_rec;
}

void delete_swallow_rec(swallow_rec_t* swallow_rec)
{
	state_t* state = (*swallow_rec).parent;
	/* unlink swallow_rec from the list of swallow_recs */
	if (state == NULL)
		;
	else if ((*state).swallow_rec == swallow_rec)
		(*state).swallow_rec = (*swallow_rec).next;
	else if ((*state).swallow_rec != NULL)
	{
		swallow_rec_t* ptr;
		for (ptr = (*state).swallow_rec ; (*ptr).next != NULL ; ptr = (*ptr).next)
			if ((*ptr).next == swallow_rec)
				break;
		if (ptr != NULL)
			(*ptr).next = (*swallow_rec).next;
	}
	/* free our members */
	if ((*swallow_rec).swallow_label != NULL)
		free((*swallow_rec).swallow_label);
	/* free our own mem */
	free(swallow_rec);
}

binding_t* new_binding(state_t* state)
{
	binding_t* binding = malloc(sizeof(binding_t));
	(*binding).next = (*state).binding;
	(*binding).name = NULL;
	(*binding).parent = state;
	(*binding).child = NULL;
	(*binding).exec = NULL;
	(*binding).type = Type_None;
	(*state).binding = binding;
	return binding;
}

void delete_binding(binding_t* binding)
{
	state_t* state = (*binding).parent;
	/* unlink binding from the list of bindings */
	if (state == NULL)
		;
	else if ((*state).binding == binding)
		(*state).binding = (*binding).next;
	else if ((*state).binding != NULL)
	{
		binding_t* ptr;
		for (ptr = (*state).binding ; (*ptr).next != NULL ; ptr = (*ptr).next)
			if ((*ptr).next == binding)
				break;
		if (ptr != NULL)
			(*ptr).next = (*binding).next;
	}
	/* free our members */
	if ((*binding).name != NULL)
		free((*binding).name);
	if ((*binding).child != NULL)
		free((*binding).child);
	if ((*binding).exec != NULL)
		free((*binding).exec);
	/* free our own mem */
	free(binding);
}

binding_t* find_binding_by_name(const state_t* state, const char* name)
{
	binding_t* binding;
	for (binding = (*state).binding ; binding != NULL ; binding = (*binding).next)
		if (!strcmp(name, (*binding).name))
			break;
	return binding;
}

/* comment out debugging functions */
#if 0
void print_binding(binding_t* binding)
{
	fprintf(stderr, "'%s'", (*binding).name);
	fprintf(stderr, ": => '%s'", (*binding).child);
	switch ((*binding).type)
	{
	case Type_Mouse:
		fprintf(stderr, ": Mouse %d", (*binding).button);
	default:
		fprintf(stderr, ": unknown type");
	}
	fprintf(stderr, "\n");
}

void print_state(state_t* state)
{
	binding_t* binding;
	fprintf(stderr, "state '%s'\n", (*state).name);
	for (binding = (*state).binding ; binding != NULL ; binding = (*binding).next)
	{
		fprintf(stderr, "  ");
		print_binding(binding);
	}
}

void print_all_states(void)
{
	state_t* state;

	for (state = first_state ; state != NULL ; state = (*state).next)
		print_state(state);
}
#endif

state_t* new_state(void)
{
	state_t* state = malloc(sizeof(state_t));
	(*state).next = first_state;
	(*state).name = NULL;
	(*state).icon.pixmap = None;
	(*state).icon.mask = None;
	(*state).binding = NULL;
	(*state).swallow_rec = NULL;
	first_state = state;
	return state;
}

void delete_state(state_t* state)
{
	/* unlink state from the list of states */
	if (first_state == state)
		first_state = (*state).next;
	else if (first_state != NULL)
	{
		state_t* ptr;
		for (ptr = first_state ; (*ptr).next != NULL ; ptr = (*ptr).next)
			if ((*ptr).next == state)
				break;
		if (ptr != NULL)
			(*ptr).next = (*state).next;
	}
	/* free our members */
	if ((*state).name != NULL)
		free((*state).name);
	if ((*state).icon.pixmap != None)
		XFreePixmap(dpy, (*state).icon.pixmap);
	if ((*state).icon.mask != None)
		XFreePixmap(dpy, (*state).icon.mask);
	while ((*state).binding != NULL)
		delete_binding((*state).binding);
	/* free our own mem */
	free(state);
}

state_t* find_state_by_name(const char* name)
{
	state_t* state;
	for (state = first_state ; state != NULL ; state = (*state).next)
		if (!strcmp(name, (*state).name))
			break;
	return state;
}

int parse_state(FILE* fp, state_t* state)
{
	char* buf = malloc(8192);
	while (fgets(buf, 8192, fp) != NULL)
	{
		char* ptr;
		int error = 0;
		char* name;
		int n;
		/* strip leading and trailing whitespace */
		ptr = strip_whitespace(buf, NULL, 2);
		/* ignore comments and blank lines */
		if (*ptr == '#' || *ptr == '\0')
			continue;
		/* ignore lines that aren't ours */
		if (*ptr != '*' || mystrncasecmp(ptr + 1, MyName, strlen(MyName)))
			continue;
		/* skip whitespace after our name */
		for (ptr += 7 ; isspace(*ptr) ; ptr++);
		/* check for done keyword */
		if (!mystrncasecmp(ptr, "~state", 6))
			break;
		/* check for swallow keyword */
		if (!mystrncasecmp(ptr, "swallow", 7))
		{
			swallow_rec_t* swallow_rec = NULL;
			name = parse_quoted_token(ptr + 7, &n, 1);
			if (name != NULL)
			{
				char* tmp;
				ptr += 7 + n;
				swallow_rec = new_swallow_rec(state);
				(*swallow_rec).swallow_label = name;
				n = strtol(ptr, &tmp, 10);
				if (ptr != tmp)
					(*swallow_rec).x = n;
				n = strtol(ptr = tmp, &tmp, 10);
				if (ptr != tmp)
					(*swallow_rec).y = n;
				continue;
			}
		}
		/* get name of binding */
		name = parse_quoted_token(ptr, &n, 2);
		if (name == NULL)
			error = 1;
		if (!error)
		{
			/* find or create the binding */
			binding_t* binding = find_binding_by_name(state, name);
			if (binding == NULL)
			{
				binding = new_binding(state);
				(*binding).name = mystrdup(name);
			}
			/* skip whitespace after binding label */
			for (ptr += n ; isspace(*ptr) ; ptr++);
			/* handle keywords */
			if (!mystrncasecmp(ptr, "exec", 4))
				(*binding).exec = strip_whitespace(ptr + 4, NULL, 1);
			else if (!mystrncasecmp(ptr, "bind", 4))
			{
				char* to_name = parse_quoted_token(ptr + 4, &n, 2);
				if (to_name == NULL)
					error = 2;
				else
				{
					/* skip whitespace after "to" label */
					for (ptr += 4 + n ; isspace(*ptr) ; ptr++);
					if (!mystrncasecmp(ptr, "mouse", 5))
					{
						(*binding).type = Type_Mouse;
						(*binding).button = strtol(ptr + 5, NULL, 10);
					}
					else if (!mystrncasecmp(ptr, "enter", 5))
						(*binding).type = Type_Enter;
					else if (!mystrncasecmp(ptr, "leave", 6))
						(*binding).type = Type_Leave;
					else if (!mystrncasecmp(ptr, "timeout", 7))
					{
						(*binding).type = Type_Timeout;
						(*binding).timeout = strtol(ptr + 7, NULL, 10);
					}
					else
						error = 3;
					if (!error)
					{
						if ((*binding).child != NULL)
							free((*binding).child);
						(*binding).child = mystrdup(to_name);
					}
				}
			}
			else
				error = 4;
		}
		if (error)
			fprintf(stderr, "%s: error %d in config: '%s'\n", MyName, error, ptr);
	}
	free(buf);
	return 0;
}

void parse_config(const char* filename)
{
	FILE* fp = fopen("asexecrc", "r");
	char* buf;
	if (fp == NULL)
	{
		fprintf(stderr, "%s: unable to open config file %s\n", MyName, filename);
		exit(1);
	}
	buf = malloc(8192);
	while (fgets(buf, 8192, fp) != NULL)
	{
		char* ptr;
		char* tmp;
		int error = 1;
		/* strip leading and trailing whitespace */
		ptr = strip_whitespace(buf, NULL, 2);
		/* ignore comments and blank lines */
		if (*ptr == '#' || *ptr == '\0')
			continue;
		/* ignore lines that aren't ours */
		if (*ptr != '*' || mystrncasecmp(ptr + 1, MyName, strlen(MyName)))
			continue;
		/* skip whitespace after our name */
		for (ptr += 7 ; isspace(*ptr) ; ptr++);
		/* look for a keyword */
		if (!mystrncasecmp(ptr, "pixmappath", 10))
		{
			error = 0;
			PixmapPath = parse_quoted_token(ptr + 10, NULL, 1);
		}
		if (!mystrncasecmp(ptr, "swallow", 7))
		{
			int n;
			char* name = parse_quoted_token(ptr + 7, &n, 1);
			char* command = strip_whitespace(ptr + 7 + n, NULL, 1);
			if (name != NULL && command != NULL)
			{
				swallow_t* swallow = new_swallow();
				error = 0;
				(*swallow).name = mystrdup(name);
				(*swallow).window_name = mystrdup(name);
				(*swallow).exec = mystrdup(command);
			}
		}
		if (!mystrncasecmp(ptr, "state", 5))
		{
			int n;
			char* name = parse_quoted_token(tmp = ptr + 5, &n, 1);
			char* icon_name = parse_quoted_token(tmp += n, &n, 1);
			state_t* state = NULL;
			error = 0;
			if (name == NULL || icon_name == NULL)
				error = 1;
			if (!error)
			{
				state = new_state();
				(*state).name = name;
				error = load_icon(icon_name, &(*state).icon) ? 2 : 0;
				if (error)
					fprintf(stderr, "%s: unable to load pixmap '%s'\n", MyName, icon_name);
			}
			if (!error)
				error = parse_state(fp, state) ? 3 : 0;
			if (!error)
				if (current_state == NULL)
					current_state = state;
			if (error)
			{
				if (state != NULL)
					delete_state(state);
				else if (name != NULL)
					free(name);
			}
			if (icon_name != NULL)
				free(icon_name);
		}
		if (error == 1)
			fprintf(stderr, "%s: error in config: '%s'\n", MyName, ptr);
	}
	free(buf);
	fclose(fp);
}

char* find_rcfile(void)
{
	FILE* fp;
	char* home = getenv("HOME");
	char* rcfile;
	char* afterdir = AFTERDIR;
	
	if (home == NULL)
		home = "";
	rcfile = malloc(strlen(home) + strlen(afterdir) + strlen(NAME) + 4 + 1);
	if (*afterdir == '~')
		sprintf(rcfile, "%s/%s/%s", home, NAME, afterdir + 1);
	else
		sprintf(rcfile, "%s/%s", afterdir, NAME);
	if ((fp = fopen(rcfile, "r")) != NULL)
	{
		fclose(fp);
		return rcfile;
	}
	sprintf(rcfile, "%s/.%src", home, NAME);
	if ((fp = fopen(rcfile, "r")) != NULL)
	{
		fclose(fp);
		return rcfile;
	}
	sprintf(rcfile, "%src", NAME);
	if ((fp = fopen(rcfile, "r")) != NULL)
	{
		fclose(fp);
		return rcfile;
	}
	
	free(rcfile);
	fprintf(stderr, "%s: unable to locate your %s config file\n", MyName, NAME);
	exit(1);
	return NULL;
}

void usage(void)
{
	fprintf(stderr, "%s version %s\n", NAME, VERSION);
	fprintf(stderr, "usage: %s [--help] [--version]\n", MyName);
	exit(1);
}

int main(int argc, char** argv)
{
	int i;
	char* rcfile;
	MyName = argv[0];
	for (i = 1 ; i < argc ; i++)
	{
		if (!strcmp(argv[i], "-v") || !strcmp(argv[i], "--version"))
		{
			printf("%s version %s\n", NAME, VERSION);
			exit(0);
		}
		else if (!strcmp(argv[i], "-h") || !strcmp(argv[i], "--help"))
			usage();
		else
		{
			fprintf(stderr, "%s: unknown option '%s'\n", MyName, argv[i]);
			usage();
		}
	}
	asexec_init1(argc, argv);
	rcfile = find_rcfile();
	parse_config(rcfile);
	free(rcfile);
	if (first_state == NULL)
	{
		fprintf(stderr, "%s: no valid states\n", MyName);
		exit(1);
	}
	asexec_init2(argc, argv);
	while (1)
		asexec_handle_events();
	return 0;
}



