/*
 *  ascp: a control panel for the AfterStep window manager
 *  Copyright (C) 1998  Nwanua Elumeze
 *
 *  License as specified in the file, COPYING.
 *
 */


/*
 * Naming Convention:
 *
 * global variables:  GlobalVariables;
 * local variables:  localvariables;
 * application wide global variables:  APPWIDEGLOBVAR;
 *
 * local functions : regular_functions; 
 * application wide functions:   ascp_foobar;
 *
 */

#include "ascp.h"

/* 
 * Global Variables used within within this file only.
 * 
 * The philosophy is: "If a widget can keep a value,
 * there's no need to also have an element to also keep 
 * that value. The widget can be queried or assigned 
 * without writing additional callbacks.".
 */

/*----------------------------------*/

FILE *IoFile;
gchar *CurrentIcon;

GtkWidget *PropertyButtons[14];
GtkWidget *StyleEntryName;
GtkWidget *StyleList;

GtkWidget *StyleEntryIcon;

GtkSpinButton *StartDeskSpinner;
GtkSpinButton *BorderWidthSpinner;
GtkSpinButton *HandleWidthSpinner;


#if 0
enum Fields	
{
	HANDLES		= (1 << 0),	/* window has resize handles (lowbar) */
	BACK		= (1 << 1 ), 	/* window stays on back */
	TITLE		= (1 << 2 ),	/* window has a titlebar */
	WINLIST		= (1 << 3 ),	/* window shows up in the winlist */
	ICONIC		= (1 << 4 ),	/* window starts iconified */
	CIRCULATE	= (1 << 5 ),	/* window can be circulated to (with alt-tab or similar) */
	ICONTITLE	= (1 << 6 ),	/* icon has a title */
	STICKY		= (1 << 7 ),	/* window starts sticky */
	TOP			= (1 << 8 ), 	/* window stays on top */
	NOFOCUS		= (1 << 9 ), 	/* window won't take focus */
	ICON		= (1 << 10),	/* window has a special icon */
	DESK		= (1 << 11),	/* window starts on a particular desk */
	LAST_field
};
#endif

enum fields  
{
HANDLES     = (1 <<  0),    /* window has resize handles (lowbar) */
TITLE       = (1 <<  1),    /* window has a titlebar */
ICONIC      = (1 <<  2),    /* window starts iconified */
WINLIST     = (1 <<  3),    /* window shows up in the winlist */
STICKY      = (1 <<  4),    /* window starts sticky */
TOP         = (1 <<  5),    /* window stays on top */
BACK        = (1 <<  6),    /* window stays on back */
NOFOCUS     = (1 <<  7),    /* window won't take focus */
DESK        = (1 <<  8),    /* window starts on a particular desk */
ICON        = (1 <<  9),    /* window has a special icon */
ICONTITLE   = (1 << 11),    /* icon has a title */
CIRCULATE   = (1 << 12),    /* window can be circulated to (with alt-tab or similar) */
LAST_field
};


typedef struct style
{
	struct style* next;	/* next style in list */
	gchar* name;		/* window name this style is associated with */
	gint on_flags;		/* features which have been explicitly turned on */
	gint off_flags;		/* features which have been explicitly turned off */
	gchar* icon;	    /* icon name */
	gint on_buttons;	/* titlebar buttons that are specifically allowed */
	gint off_buttons;	/* titlebar buttons that are specifically disallowed */
	gchar* forecolor;	/* foreground color */
	gchar* backcolor;	/* background color */
	gint desk;			/* window starts on this desk */
	gint border_width;	/* border width of client window, titlebar, lowbar, etc */
	gint handle_width;	/* width of resize handles */
} style_t;

typedef struct {
	gchar* keyword;
	gint (*func)(style_t*, gchar*, gint, gint);
	gint arg1;
	gint arg2;
} config_t;

//style_t *CurrentStyle = NULL;

static void parse_database(gfloat, gfloat);
static void add_list_entry(GtkWidget *, gpointer *);
static void affect_style_entry_button(GtkWidget *, gint);
static gint style_setint(style_t* style, gchar* line, gint arg1, gint arg2);
static gint style_setflag(style_t* style, gchar* line, gint arg1, gint arg2);
static gint style_setstring(style_t* style, gchar* line, gint arg1, gint arg2);
static gint style_setcolors(style_t* style, gchar* line, gint arg1, gint arg2);
static gint style_setbutton(style_t* style, gchar* line, gint arg1, gint arg2);
static gint style_item_selected(GtkWidget *, gint, gint, GdkEventButton *, gpointer data);

/* in general, these options have both an "on" and an "off" keyword (like 
 * "Title" and "NoTitle", for instance); it's important that we know if 
 * the flag was set on or off, since if a feature is turned off in a 
 * general style (like the "*" style), it may be turned back on in a later 
 * style; for example:
 *
 * Style "*" NoButton 1, BorderWidth 2
 * Style "xterm" Button 1, NoHandles
 *
 * this will hide the leftmost button on the titlebar for any window 
 * except xterm windows; it will also turn off resize handles on all 
 * windows
 *
 * it will also give a 2-pixel border to xterm windows - note that 
 * BorderWidth only affects windows with NoHandles (this is in the 
 * manpage), so all other windows will receive the normal 1-pixel border
 *
 * "Icon", "NoIcon"
 * "Color", "ForeColor", "BackColor"
 * "NoFocus"
 * "Title", "NoTitle"
 * "IconTitle", "NoIconTitle"
 * "Handles", "NoHandles"
 * "Button", "NoButton"
 * "WindowListHit", "WindowListSkip"
 * "CirculateHit", "CirculateSkip"
 * "StartNormal", "StartIconic"
 * "StaysOnTop", "StaysPut", "StaysOnBack"
 * "Slippery", "Sticky"
 * "BorderWidth"
 * "HandleWidth"
 * "StartsAnywhere", "StartsOnDesk"
 */

config_t StyleConfig[] = {
{ "Icon", style_setstring, offsetof(style_t, icon), ICON },
{ "NoIcon", style_setflag, 0, ICON },
{ "Color", style_setcolors, 0, 0 },
{ "ForeColor", style_setstring, offsetof(style_t, forecolor), 0 },
{ "BackColor", style_setstring, offsetof(style_t, backcolor), 0 },
{ "NoFocus", style_setflag, NOFOCUS, 0 },
{ "Title", style_setflag, TITLE, 0 },
{ "NoTitle", style_setflag, 0, TITLE },
{ "IconTitle", style_setflag, ICONTITLE, 0 }, 
{ "NoIconTitle", style_setflag, 0, ICONTITLE }, 
{ "Handles", style_setflag, HANDLES, 0 },
{ "NoHandles", style_setflag, 0, HANDLES },
{ "Button", style_setbutton, 1, 0 },
{ "NoButton", style_setbutton, 0, 0 },
{ "WindowListHit", style_setflag, WINLIST, 0 },
{ "WindowListSkip", style_setflag, 0, WINLIST },
{ "CirculateHit", style_setflag, CIRCULATE, 0 },
{ "CirculateSkip", style_setflag, 0, CIRCULATE },
{ "StartNormal", style_setflag, 0, ICONIC },
{ "StartIconic", style_setflag, ICONIC, 0 },
{ "StaysOnTop", style_setflag, TOP, BACK },
{ "StaysPut", style_setflag, 0, TOP | BACK },
{ "StaysOnBack", style_setflag, BACK, TOP },
{ "Slippery", style_setflag, 0, STICKY },
{ "Sticky", style_setflag, STICKY, 0 },
{ "BorderWidth", style_setint, offsetof(style_t, border_width), 0 },
{ "HandleWidth", style_setint, offsetof(style_t, handle_width), 0 },
{ "StartsAnywhere", style_setflag, 0, DESK },
{ "StartsOnDesk", style_setint, offsetof(style_t, desk), DESK },
{ NULL }
};


style_t* new_style(void)
{
	style_t* style = g_new(style_t, 1);
	
	if (style == NULL)
		return NULL;

	/* initialize the rest of the fields */
	style->name = NULL;
	style->on_flags = 0;
	style->off_flags = 0;
	style->icon = NULL;
	style->on_buttons = 0;
	style->off_buttons = 0;
	style->forecolor = NULL;
	style->backcolor = NULL;
	style->desk = 0; 
	style->border_width = -1;
	style->handle_width = -1;

	return style;
}

void print_style(style_t* style)
{

	printf("Style '%s':\n", style->name);
	printf( 
			"  %08x\n"
			"  %08x\n"
			"  '%s'\n"
			"  %08x\n"
			"  %08x\n"
			"  %s\n"
			"  %s\n"
			"  %d\n"
			"  %d\n"
			"  %d\n"
			, 
style->on_flags,
style->off_flags,
style->icon,
style->on_buttons,
style->off_buttons,
style->forecolor,
style->backcolor,
style->desk,
style->border_width,
style->handle_width

);	
			g_print("%01x %01x %01x %01x %01x"
					"%01x %01x %01x %01x %01x\n", 
					style->on_flags & (1 << 0),
					style->on_flags & (1 << 1),
					style->on_flags & (1 << 2),
					style->on_flags & (1 << 3),
					style->on_flags & (1 << 4),
					style->on_flags & (1 << 5),
					style->on_flags & (1 << 6),
					style->on_flags & (1 << 7),
					style->on_flags & (1 << 8),
					style->on_flags & (1 << 9));

	/*
		g_print("\n\nyou entered buttons %01x %01x %01x %01x\n",  
				style->off_buttons   & (1 << 1),
				style->off_buttons   & (1 << 2),
				style->off_buttons   & (1 << 3),
				style->off_buttons   & (1 << 4));
*/
			
}

/* sets all flags in arg1, and clears all flags in arg2 */
static gint style_setflag(style_t* style, gchar* line, gint arg1, gint arg2)
{
	//g_print("the args are %d %d\n", arg1, arg2);
	style->on_flags |= arg1;
	style->off_flags |= arg2;
	return 0;
}

/* reads an int from line, stores it in the style structure at offset arg1, 
 * and sets all flags in arg2 */
static gint style_setint(style_t* style, gchar* line, gint arg1, gint arg2)
{
	gchar* t;
	gint* val = (gint*)((void*)style + arg1);
	style->on_flags |= arg2;
	*val = strtol(line, &t, 10);
	return (t - line > 0) ? (t - line) : -4;
}

/* reads a (non-quoted) string from line, stores it in the style structure 
 * at offset arg1, and sets all flags in arg2 */
static gint style_setstring(style_t* style, gchar* line, gint arg1, gint arg2)
{
	gchar* t1, *t2;
	gchar** val = (gchar**)((void*)style + arg1);
	if (*val != NULL)
		g_free(*val);
	style->on_flags |= arg2;
	for (t1 = line ; *t1 != '\0' && isspace(*t1) ; t1++);
	for (t2 = t1 ; *t2 != '\0' && *t2 != ',' && !isspace(*t2) ; t2++);
	*val = ascp_strndup(t1, t2 - t1);
	return (t2 - line > 0) ? (t2 - line) : -4;
}

/* reads forecolor and backcolor from line */
static gint style_setcolors(style_t* style, gchar* line, gint arg1, gint arg2)
{
	gchar* t;
	if (style->backcolor != NULL)
		g_free(style->backcolor);
	if (style->forecolor != NULL)
		g_free(style->forecolor);
	for (t = line ; *t != '\0' && *t != ',' && *t != '/' ; t++);
	style->forecolor = ascp_strndup(line, t - line);
	if (*t != '\0') {
		gchar* t2 = ++t;
		for ( ; *t != '\0' &&  *t != ',' && *t != '/' ; t++);
		style->backcolor = ascp_strndup(line, t - t2);
	}
	return (t - line > 0) ? (t - line) : -4;
}

/* sets the on/off titlebar button bitflags 
 * if a button has its on_button flag set, that button is explicitly allowed
 * if a button has its off_button flag set, that button is explicitly disallowed
 */
static gint style_setbutton(style_t* style, gchar* line, gint arg1, gint arg2)
{
	gchar* t;
	gint button = strtol(line, &t, 10);
	if (arg1)
		style->on_buttons |= 1 << button;
	else
		style->off_buttons |= 1 << button;
	return (t - line > 0) ? (t - line) : -4;
}

static gint init_style(style_t* style, gchar* line)
{
	int error = 0;
	gchar* ptr = NULL;

	for ( ; isspace(*line) ; line++);

	/* find the style name */ 
	if (*line != '"')
		error = 1;

	if (!error)
	{
		ptr = ++line;

		while (*line != '\0' && *line != '"')
			line++;

		if (*line != '"')
			error = 2;
	}

	/* style name found - save it and start parsing the options */
	if (!error) 
	{
		style->name = ascp_strndup(ptr, line - ptr);
		line++;

		while (*line != '\0' && !error)
		{
			config_t* c;

			/* isolate the keyword */
			for ( ; isspace(*line) ; line++)
				;

			ptr = line;

			for ( ; *line != '\0' && !isspace(*line) ; line++)
				;

			/* identify the keyword */
			for (c = StyleConfig ; c->keyword != NULL ; c++)
			{
				if (!ascp_strncasecmp(ptr, c->keyword, strlen(c->keyword))) 
				{
					gint n;
					//g_print("for %s  ",  style->name);
					n = c->func(style, line, c->arg1, c->arg2);

					if (n < 0)
						error = -n;

					line += n;
					break;
				}
			}

			if (c->keyword == NULL)
				error = 5;

			/* scan forward to the next ',' */
			if (!error && *line != '\0')
			{
				for ( ; *line != '\0' && *line != ',' ; line++);

				if (*line == ',')
					line++;
			}

		}
	}
	//print_style(style);
	return error;
}



					 

gint style_item_selected(GtkWidget   *clist, gint row, gint column,
			GdkEventButton *event, gpointer data)
{
	int i;
	style_t*  entry = (style_t *) g_malloc(sizeof(style_t *));


	entry = (style_t *) gtk_clist_get_row_data (GTK_CLIST (clist), row);

	if(entry)
	{
		gint index[] = { 
				BACK,
				HANDLES,
				TITLE,
				WINLIST,
				ICONIC,
				CIRCULATE,
				ICONTITLE,
				STICKY,
				TOP,
				NOFOCUS};



		//g_print("looks like %s on row %d\n", entry->name, row);
		//print_style(entry);
		//CurrentStyle = entry;

		gtk_entry_set_text (GTK_ENTRY (StyleEntryName), entry->name);

		for(i=0; i < 10; i++)
		{

			gtk_toggle_button_set_state( GTK_TOGGLE_BUTTON(PropertyButtons[i]), 
				//entry->on_flags & (1 << (i+1)));
				entry->on_flags & index[i]);
	//		g_print("%01x\n", entry->on_flags & (1 << (i+1)));
		}
		gtk_spin_button_set_value (StartDeskSpinner, entry->desk);
		gtk_spin_button_set_value (BorderWidthSpinner, entry->border_width);
		gtk_spin_button_set_value (HandleWidthSpinner, entry->handle_width);



		for(i=1; i < 5; i++)
		{
			gtk_toggle_button_set_state( GTK_TOGGLE_BUTTON(PropertyButtons[i+9]), 
				(gboolean) entry->off_buttons & (1 << i));
		}

		if(GTK_BIN(StyleEntryIcon)->child)
			gtk_container_remove(GTK_CONTAINER(StyleEntryIcon),
			     GTK_BIN(StyleEntryIcon)->child);
		
		if(entry->icon)
		{
			gtk_container_add(GTK_CONTAINER(StyleEntryIcon), 
				ascp_make_pixmap(entry->icon, SINGLE));
		}
		else
		{
			gtk_container_add(GTK_CONTAINER(StyleEntryIcon), gtk_label_new("\nNone\n"));
		}

		gtk_widget_draw_default(StyleEntryIcon);
		gtk_widget_show_all(StyleEntryIcon);



		 return TRUE;
	}
	else
	{
		g_print("Can't find such entry %p\n", entry);
		 return FALSE;
	}


#if 0
	for (i = 0, ptr = FirstStyle ; ptr != NULL && i != row ; ptr = ptr->next, i++);

	if (ptr != NULL)
	{
		//print_style(ptr);
		CurrentStyle = ptr;
		gtk_entry_set_text (GTK_ENTRY (StyleEntryName), ptr->name);
		for(i=0; i < 11; i++)
		{
			gtk_toggle_button_set_state( GTK_TOGGLE_BUTTON(PropertyButtons[i]), 
				ptr->on_flags & (1 << i));
		}
		gtk_spin_button_set_value (Spin, ptr->desk);
		
		/* catch zero length icon names and reset to "none" */
		/* (
		if( ptr->icon != NULL && strlen(ptr->icon) >  0) 
			gtk_entry_set_text (GTK_ENTRY (Icon_entry), ptr->icon);
		else 
			gtk_entry_set_text (GTK_ENTRY (Icon_entry), "none");
			*/
		
		//set_ListPixmap(StyleEntryIcon, NULL);
	}
#endif
}


/* 
 * One of the property buttons was pressed. 
 * Find out which one it was and toggle its
 * current state.
 */
/*
void property_button_callback (GtkWidget *widget, gpointer data)
{
	if (GTK_TOGGLE_BUTTON (widget)->active)
		CurrentStyle->on_flags |= (gint)data;
	else
		CurrentStyle->off_flags |= (gint)data;
}
*/




static void parse_database(gfloat min, gfloat max)
{
	gchar *buffer = (gchar*) g_malloc0(BUFFERSIZE);
	gfloat incr = (max - min)/FILESIZE;
	gint number=0;
	FILE *backupfile;


	ascp_update_status(NULL, "database", min);

	backupfile = ascp_open_file("database.ascp", "w");
	IoFile = ascp_open_file("database", "r");


	if(1 || FILESIZE > 1)
	{
		while (fgets(buffer, BUFFERSIZE, IoFile) != NULL)
		{
			char* line = buffer;

			ascp_update_status(NULL, NULL, min);
			min = ( min >= max) ? max : min + incr*strlen(buffer);

			fputs(line, backupfile);

			for ( ; isspace(*line) ; line++);
			if( !ascp_strncasecmp(line, "Style", 5) )
			{
				style_t *entry = (style_t *) g_malloc0(sizeof(style_t *));

				entry = new_style();
				init_style(entry, line + 5);

				if(entry)
				{
					gchar *label[] = {entry->name};
					gtk_clist_append (GTK_CLIST(StyleList), label); 
					gtk_clist_set_row_data(GTK_CLIST(StyleList), number++, (style_t *) entry);
					

				}
			}
		}
	}
	else 
	{
		style_t *style;
		style = new_style();
		init_style(style, "\"ascp\"   Icon interface.xpm");
	}
	
	fclose(IoFile);
	fclose(backupfile);

}



static void add_list_entry(GtkWidget *widget, gpointer *data)
{
	gchar *buffer;
	style_t* ptr;

	buffer = g_malloc(256);
	ptr = (style_t *) g_malloc(sizeof(style_t *));

	buffer = gtk_entry_get_text(GTK_ENTRY(StyleEntryName));
	ptr->name = g_malloc(strlen(buffer));
	printf("%d   ", strlen(buffer));
	strcpy(ptr->name, buffer);

	/*
	ptr->on_flags = CurrentStyle->on_flags;
	ptr->off_flags = CurrentStyle->off_flags;

	CurrentStyle = ptr;

*/
	free(buffer);
}

/*
void get_spin_value(GtkWidget *widget, gpointer blank)
{
	CurrentStyle->desk = gtk_spin_button_get_value_as_int (Spin);
	
	return;
}
*/



GtkWidget * ascp_draw_database(gfloat min, gfloat max)
{

	GtkWidget *tabletop =  gtk_table_new(12, 23, TRUE);
	GtkWidget	*frame;
	
	if(ASCPERROR)
		return gtk_label_new("ascp Error");



	
	{
		GtkWidget *scrolledwin = gtk_scrolled_window_new (NULL, NULL);

		 gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwin),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

		StyleList = gtk_clist_new (1);
		gtk_signal_connect(GTK_OBJECT(StyleList), "select_row",
			GTK_SIGNAL_FUNC(style_item_selected), NULL);

	
#ifdef GTK_STABLE_VERSION
		gtk_container_add(GTK_CONTAINER(scrolledwin), StyleList);
#else
		gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolledwin), StyleList);
#endif
		gtk_table_attach_defaults( GTK_TABLE(tabletop), scrolledwin, 0, 8, 0, 8);

	}
	

 	/* 
     * Create the entry box that will contain the name
     * of the currently selected item in the list.
     */
    {	
        frame = gtk_frame_new (NULL);
        StyleEntryName = gtk_entry_new_with_max_length (30);
        gtk_entry_set_text (GTK_ENTRY (StyleEntryName), " ");
        gtk_container_add(GTK_CONTAINER(frame), StyleEntryName);
        gtk_table_attach_defaults( GTK_TABLE(tabletop), frame, 0, 8, 8, 10);
    }

					
	/* 
	 * Create the style buttons; add callbacks.
	 */
    {
        gint i=0;
        GtkWidget *vbox;
        gchar   *title[] = {
            "No Handles",
            "Stays Back",
            "Has Titlebar",
            "Is in WinList",
            "Starts Iconic",
            "Can Circulate",
            "Has Icon Title",
            "Sticks on Page",
            "Always on Top",
			"Never Focused",
		};

		vbox = gtk_vbox_new(TRUE, 0);
			
		for(i=0; i < 10; i++)
		{
			PropertyButtons[i] = gtk_check_button_new_with_label( *(title+i) );

			gtk_box_pack_start(GTK_BOX(vbox), PropertyButtons[i], TRUE, TRUE, 0);
			
		}
		gtk_table_attach_defaults( GTK_TABLE(tabletop), vbox, 8, 15, 0, 12);

    }

	
	/* 
     * Create the 'special' starts-on-desk label and spin button; 
     * attach callbacks.
     */
	
    {

        GtkWidget *hbox;
        GtkWidget *vbox;
        GtkWidget *vboxl;
        GtkWidget *label;
		GtkWidget *spinner;
		GtkAdjustment *adj;
		gchar *title[] = {"1", "2", "3", "4"};
		gint i;

		label = gtk_label_new("Hide Buttons");
		gtk_table_attach_defaults(GTK_TABLE(tabletop), label, 15, 23, 0, 1);

		hbox = gtk_hbox_new(TRUE, 0);

		for(i=10; i<14; i++)
		{
			PropertyButtons[i] = gtk_check_button_new_with_label( *(title+ (i-10)) );

			gtk_box_pack_start(GTK_BOX(hbox), PropertyButtons[i], TRUE, TRUE, 1);
			
		}
		gtk_table_attach_defaults( GTK_TABLE(tabletop), hbox, 15, 23, 1, 2);


		vbox = gtk_vbox_new(TRUE, 0);
		vboxl = gtk_vbox_new(TRUE, 0);

		label = gtk_label_new("Border Width  ");
		gtk_box_pack_start(GTK_BOX(vboxl), label, TRUE, TRUE, 0);
		//gtk_table_attach_defaults(GTK_TABLE(tabletop), label, 15, 20, 2, 3);
		adj =  (GtkAdjustment * ) gtk_adjustment_new (0, 0, 4, 1, 5, 0);
		spinner = gtk_spin_button_new (adj, 1, 0);
		BorderWidthSpinner = GTK_SPIN_BUTTON (spinner);
		gtk_spin_button_set_update_policy (BorderWidthSpinner, GTK_UPDATE_ALWAYS);
		//gtk_table_attach_defaults(GTK_TABLE(tabletop), spinner, 20, 22, 2, 3);
		gtk_box_pack_start(GTK_BOX(vbox), spinner, TRUE, TRUE, 0);


		label = gtk_label_new("Handle Width ");
		//gtk_table_attach_defaults(GTK_TABLE(tabletop), label, 15, 20, 3, 4);
		gtk_box_pack_start(GTK_BOX(vboxl), label, TRUE, TRUE, 0);
		adj =  (GtkAdjustment * ) gtk_adjustment_new (0, 0, 9, 1, 5, 0);
		spinner = gtk_spin_button_new (adj, 1, 0);
		HandleWidthSpinner = GTK_SPIN_BUTTON (spinner);
		gtk_spin_button_set_update_policy (HandleWidthSpinner, GTK_UPDATE_ALWAYS);
		//gtk_table_attach_defaults(GTK_TABLE(tabletop), spinner, 20, 22, 4, 5);
		gtk_box_pack_start(GTK_BOX(vbox), spinner, TRUE, TRUE, 1);



		label = gtk_label_new( "Starts on Desk");
		//gtk_table_attach_defaults(GTK_TABLE(tabletop), label, 15, 20, 6, 8);
		gtk_box_pack_start(GTK_BOX(vboxl), label, TRUE, TRUE, 1);

		adj =  (GtkAdjustment * ) gtk_adjustment_new (0, 0, 4, 1, 5, 0);
		spinner = gtk_spin_button_new (adj, 1, 0);
		StartDeskSpinner = GTK_SPIN_BUTTON (spinner);
		gtk_spin_button_set_update_policy (StartDeskSpinner, GTK_UPDATE_ALWAYS);
		//gtk_table_attach_defaults(GTK_TABLE(tabletop), spinner, 20, 22 , 6, 7);
		gtk_box_pack_start(GTK_BOX(vbox), spinner, TRUE, TRUE, 1);

		gtk_table_attach_defaults(GTK_TABLE(tabletop), vboxl, 15, 20,  2, 7);
		gtk_table_attach_defaults(GTK_TABLE(tabletop), vbox, 20, 22 , 2, 7);

	}

	/*
     * Create the button that will contain the icon for the list 
     * item selected and put it in a nice frame.
     * Set it up to call up a filesel widget.
     * In addition, create a menu that will popup in response
     * to second or third button mouse clicks.
     */
    {
		GtkWidget *localtable;
		GtkWidget *menu;
        GtkWidget *menuitem;
        gchar *menumsg[] = {
                "None"
                };

        gint i=0;
        /*
         * A compromise between adequate spacing between the
         * check buttons, and a still nice looking pixmap button :-)
         */
		localtable = gtk_table_new(18, 18, TRUE);


		frame = gtk_frame_new (NULL);


		 menu = gtk_menu_new();

        for(i=0; i < (sizeof(menumsg)/sizeof(*menumsg)); i++)
        {
             menuitem = gtk_menu_item_new_with_label(menumsg[i]);
             gtk_menu_append(GTK_MENU (menu), menuitem);
             gtk_widget_show(menuitem);
			 gtk_signal_connect(GTK_OBJECT(menuitem), "activate",
				GTK_SIGNAL_FUNC(affect_style_entry_button), (gpointer) i);
        }


		StyleEntryIcon = gtk_button_new();

		gtk_signal_connect_object(GTK_OBJECT(StyleEntryIcon), "button_press_event",
			GTK_SIGNAL_FUNC (ascp_alt_menu), GTK_OBJECT(menu));
       	gtk_signal_connect(GTK_OBJECT(StyleEntryIcon), "button_press_event",
                GTK_SIGNAL_FUNC (ascp_xpm_browser), NULL);
		
		gtk_widget_set_sensitive(StyleEntryIcon, FALSE);
		gtk_container_add (GTK_CONTAINER (frame), StyleEntryIcon );
		gtk_container_border_width (GTK_CONTAINER (StyleEntryIcon), 2);
        gtk_table_attach_defaults(GTK_TABLE(localtable), frame,  2, 16, 4, 17);
        gtk_table_attach_defaults(GTK_TABLE(tabletop), localtable, 17, 23, 6, 12);
	
	}

 
	/*
	 * Create the two command buttons: "New", "Change" & "Delete". 
     * Also create the alternate menu options.
     */
    {

        GtkWidget *button;

        button = gtk_button_new_with_label("New");
        /*
        gtk_signal_connect(GTK_OBJECT(button), "clicked",
            GTK_SIGNAL_FUNC(add_ListEntry), NULL);
            */
        gtk_table_attach( GTK_TABLE(tabletop), button, 0, 2, 10, 12,
            GTK_SHRINK, GTK_SHRINK, 0, 0);
		gtk_widget_set_sensitive(button, FALSE);


		button = gtk_button_new_with_label("Change");
        /*
        gtk_signal_connect(GTK_OBJECT(button), "clicked",
        GTK_SIGNAL_FUNC(delete_ListEntry), (gpointer) button);
        */
        gtk_table_attach( GTK_TABLE(tabletop), button, 2, 5, 10, 12,
            GTK_SHRINK, GTK_SHRINK, 0, 0);
		gtk_widget_set_sensitive(button, FALSE);


		button = gtk_button_new_with_label("Delete");
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(ascp_clist_remove_selection), StyleList);
        gtk_table_attach( GTK_TABLE(tabletop), button, 5, 8, 10, 12,
            GTK_SHRINK, GTK_SHRINK, 0, 0);
	}

	parse_database(min, max);
			
	gtk_widget_show_all(tabletop);
	return tabletop;
}


void ascp_write_database()
{ 
	//struct style *entry;
	/*
	gchar *properties[8] = {
			"NoHandles",
			"NoTitle",
			"Iconic",
			"WindowListSkip",
			"Sticky", 
			"StaysOnTop",
			"NoFocus",
			"StartsOnDesk"};
			*/
	
	
	//IoFile = ascp_open_file("database", "w");
	// changed write to read:  if(( IoFile = fopen( buffer ,"w")) == NULL)
	//if(( IoFile = fopen( "DATABASE","w")) == NULL)


/*	
	fprintf(IoFile, "\n\n#----------------------------------------------------------------\n");
	fprintf(IoFile, "# This file was written by %s\n", ASCPVERSION);
	fprintf(IoFile, "#----------------------------------------------------------------\n\n");
				  
	entry = List;
	while( entry != NULL )
	{
		int i;


		fprintf(IoFile, "Style   \"%s\"        ", entry->name);
		fprintf(IoFile, "
		for(i=0; i<7; i++)
		{
		*/
			/* I would like an easy way to get rid of trailing ',' at 
			 * the end of a line.
			 * Right now, they don't seem to cause problems for
			 * AfterStep, but at the very least: "it's a challenge to be 
			 * removed!"
			 */
	/*
			if(entry->properties[i]) 
				fprintf(IoFile, "%s,", properties[i]);
		}
			
			
		if( ascp_strncasecmp(entry->icon, "none", 4)) 
				fprintf(IoFile, " %s ", entry->icon);

		if( entry->properties[DESK]) 
				fprintf(IoFile, "StartsOnDesk %d", entry->properties[DESK]);
		

		fprintf(IoFile, "\n");

		entry = entry->next;
	}

	fprintf(IoFile, "\n#----------------------------------------------------------------\n");
	fprintf(IoFile, "# And that's all ascp wrote");
	fprintf(IoFile, "\n#----------------------------------------------------------------\n");
	fprintf(IoFile, "\n\n");
*/
	//fclose(IoFile);
					
}

static void affect_style_entry_button(GtkWidget *widget, gint dowhat)
{
	if(dowhat == 0)
	{
		strcpy(CurrentIcon, "-");
		
		if(GTK_BIN(StyleEntryIcon)->child)
			gtk_container_remove(GTK_CONTAINER(StyleEntryIcon),
					GTK_BIN(StyleEntryIcon)->child);
		
		gtk_container_add(GTK_CONTAINER(StyleEntryIcon), gtk_label_new("\nNone\n"));
		gtk_widget_draw_default(StyleEntryIcon);
		gtk_widget_show_all(StyleEntryIcon);
	}
}

