
/*
 *  ascp: a control panel for the AfterStep window manager
 *  Copyright (C) 1998  Nwanua Elumeze
 *
 *  You may distribute under the terms of the GNU General Public
 *  License as specified in the file, COPYING.
 *
 */


/* 
 * Naming Convention:
 *  
 * global variables:  GlobalVariables;
 * local variables:  localvariables;
 * application wide global variables:  APPWIDEGLOBVAR;
 *
 * local functions : regular_functions; 
 * application wide functions:   ascp_foobar;
 *
 */
				  
#include "xpm.h"


#include "ascp.h"


/* 
 * Global Variables used within within this file only.
 *
 * The philosophy is: "If a widget can keep a value,
 * there's no need to also have an element to also keep 
 * that value. The widget can be queried or assigned 
 * without writing additional callbacks.".
 */

/*----------------------------------*/
static FILE *IoFile;
/*----------------------------------*/

GtkWidget *EventList;
GtkWidget *SoundList;
GtkWidget *PlayButton;
GtkWidget *CurrentSound;
GtkWidget *CurrentEvent;
GtkWidget *AudioPlayCmd;
GtkWidget *AudioPath;
GtkSpinButton *AudioDelay;
gint CurrentRow=0;


static void parse_audio(gfloat, gfloat);
static void init_and_append_entry(gchar *, gint);
static gint add_sound_entry(GtkWidget *, gint);
static gint remove_sound_entry(GtkWidget *, GtkWidget *);
static gint change_event_sound(GtkWidget *, gpointer);
static gint play_sound(GtkWidget *, GdkEvent *, gpointer);
static gint event_entry_selected(GtkWidget  *, gint, gint, GdkEventButton *, gpointer *);
static gint sound_entry_selected(GtkWidget  *, gint, gint, GdkEventButton *, gpointer *);





GtkWidget * ascp_draw_audio(gfloat min, gfloat max)
{
	/*      
	 * The toplevel table that shall be returned to
	 * populate_frame().
	 */ 
	GtkWidget *tabletop =  gtk_table_new(14, 14, TRUE);

	if(ASCPERROR)
		return gtk_label_new("ascp Error");

	/* 
	 * Entries for selected event and sound.
	 */
	{
		GtkWidget *frame;

		frame =  gtk_frame_new(NULL);
		CurrentEvent = gtk_entry_new_with_max_length(30);
		gtk_entry_set_text (GTK_ENTRY (CurrentEvent), "");
		gtk_container_add(GTK_CONTAINER(frame), CurrentEvent);
		gtk_table_attach_defaults( GTK_TABLE(tabletop), frame, 0, 5, 8, 10);


		frame =  gtk_frame_new(NULL);
		CurrentSound = gtk_entry_new_with_max_length(30);
		gtk_entry_set_text (GTK_ENTRY (CurrentSound), "");
		gtk_container_add(GTK_CONTAINER(frame), CurrentSound);
		gtk_table_attach_defaults( GTK_TABLE(tabletop), frame, 9, 14, 8, 10);
	}



	/* 
	 * The two lists: "event" and "sound"; 
	 */
	{
		GtkWidget *frame;
		GtkWidget *scrolledwin;

		frame = gtk_frame_new("Event");
		scrolledwin = gtk_scrolled_window_new (NULL, NULL);
		gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwin),
			GTK_POLICY_AUTOMATIC,  GTK_POLICY_AUTOMATIC);

		EventList = gtk_clist_new(1);
		gtk_signal_connect(GTK_OBJECT(EventList), "select_row",
			GTK_SIGNAL_FUNC(event_entry_selected), CurrentEvent);
#ifdef GTK_STABLE_VERSION
		gtk_container_add(GTK_CONTAINER(scrolledwin), EventList);
#else
		gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolledwin), EventList);
#endif
		gtk_container_add(GTK_CONTAINER(frame), scrolledwin);
		gtk_table_attach_defaults( GTK_TABLE(tabletop), frame, 0, 5, 0, 8);



		frame = gtk_frame_new("Sound");
		scrolledwin = gtk_scrolled_window_new (NULL, NULL);
		gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwin),
			GTK_POLICY_AUTOMATIC,  GTK_POLICY_AUTOMATIC);

		SoundList = gtk_clist_new(1);
		gtk_signal_connect(GTK_OBJECT(SoundList), "select_row",
			GTK_SIGNAL_FUNC(sound_entry_selected), (gpointer *) CurrentSound);
#ifdef GTK_STABLE_VERSION
		gtk_container_add(GTK_CONTAINER(scrolledwin), SoundList);
#else
		gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolledwin), SoundList);
#endif
		gtk_container_add(GTK_CONTAINER(frame), scrolledwin);
		gtk_table_attach_defaults( GTK_TABLE(tabletop), frame, 9, 14, 0, 8);
	}



	/* 
	 * The two buttons to manipulate new sound entries.
	 */
	{
		GtkWidget *button;

		button = gtk_button_new_with_label("Remove");
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(remove_sound_entry), (gpointer *) CurrentSound);
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(ascp_clist_remove_selection), SoundList);
		gtk_table_attach( GTK_TABLE(tabletop), button, 9, 11, 10, 12, 
			GTK_SHRINK, GTK_SHRINK, 0, 0);


		button = gtk_button_new_with_label("  Add  ");
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(add_sound_entry), NULL);

		gtk_table_attach( GTK_TABLE(tabletop), button, 12, 14, 10, 12,
			GTK_SHRINK, GTK_SHRINK, 0, 0);


	}


	/* 
	 * Delay.
	 */
	{
		GtkWidget *label;
		GtkAdjustment *adj;
		GtkWidget *spinner;
		GtkWidget *subtable;


		subtable =  gtk_table_new(14, 14, TRUE);

		label = gtk_label_new("Timeout");
		gtk_table_attach_defaults(GTK_TABLE(tabletop), label, 6, 8, 0, 3); 


		adj =  (GtkAdjustment * ) gtk_adjustment_new (0, 1, 4, 1, 5, 0);
		spinner = gtk_spin_button_new (adj, 1, 0);
		AudioDelay = GTK_SPIN_BUTTON (spinner);
		gtk_spin_button_set_value(GTK_SPIN_BUTTON(AudioDelay), 1);
		gtk_spin_button_set_update_policy (AudioDelay, GTK_UPDATE_ALWAYS);
		gtk_table_attach_defaults(GTK_TABLE(subtable), spinner, 3, 14, 0, 2); 


		gtk_table_attach_defaults(GTK_TABLE(tabletop), subtable, 6, 8, 3, 5); 
	}
 
	/*
	 * The path.
	 */
	{
		GtkWidget *frame;
		
		frame =  gtk_frame_new("Sound Search Path");

		AudioPath = gtk_entry_new_with_max_length(100);
		gtk_entry_set_text (GTK_ENTRY (AudioPath), "");
		gtk_container_add(GTK_CONTAINER(frame), AudioPath);
		gtk_table_attach_defaults( GTK_TABLE(tabletop), frame, 0, 8, 11, 14);

	}


	/*
	 * The play command.
	 */
	{
		GtkWidget *frame;
		
		//frame =  gtk_frame_new("Play Command");
		frame =  gtk_frame_new(NULL);

		AudioPlayCmd = gtk_entry_new_with_max_length(30);
		gtk_entry_set_text (GTK_ENTRY (AudioPlayCmd), "");
		gtk_container_add(GTK_CONTAINER(frame), AudioPlayCmd);
		gtk_table_attach_defaults( GTK_TABLE(tabletop), frame, 9, 14, 12, 14);

	}


	/*
	 * What _does_ the sound sound like?
	 */
	{
		GtkWidget *frame;

		frame =  gtk_frame_new(NULL);

		PlayButton = gtk_button_new_with_label("Play");
		gtk_signal_connect(GTK_OBJECT(PlayButton), "clicked",
				GTK_SIGNAL_FUNC(play_sound), NULL);
		gtk_container_border_width (GTK_CONTAINER (PlayButton), 2);
		gtk_container_add(GTK_CONTAINER(frame), PlayButton);
		gtk_table_attach_defaults(GTK_TABLE(tabletop), frame, 6, 8, 5, 7); 

		/* check if path is correct; if not error_popup */
		
	}

	/*
	 * Change the sound for the event selected.
	 */
	{
		GtkWidget *frame;
		GtkWidget *button;

		frame =  gtk_frame_new(NULL);

		button = gtk_button_new_with_label("<-Change-");
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
				GTK_SIGNAL_FUNC(change_event_sound), NULL);
		gtk_container_border_width (GTK_CONTAINER (button), 2);
		gtk_container_add(GTK_CONTAINER(frame), button);
		gtk_table_attach_defaults(GTK_TABLE(tabletop), frame, 5, 9, 8, 10); 

		
	}

	parse_audio(min, max);


	gtk_widget_show_all(tabletop);
	return tabletop;
}

static gint play_sound(GtkWidget *blank, GdkEvent *event, gpointer alsoblank)
{
	gint error=FALSE;

	/*
	 * First, is the play command valid? 
	 */
	struct stat filestat;

	gchar *noise = g_malloc0(128);



	if( stat((gchar *)gtk_entry_get_text(GTK_ENTRY(AudioPlayCmd)), &filestat)  == -1)
	{
		GtkWidget *button = gtk_button_new_with_label("Close");
		
		gchar *msg = "Either the play command \"%s\"\n"
					 "was not found, or it is not executable by you.\n\n"
					 "Please correct the path and try again.\n\n"
					 "Alternatively, you can check to make sure you have execute\n"
					 "and read permissions for this command.";

		gchar *buffer = g_malloc0(strlen(msg) + 30);

		sprintf(buffer, msg, (gchar *)gtk_entry_get_text(GTK_ENTRY(AudioPlayCmd)) );

		ascp_error_popup("Cannot play sound\n", buffer, button);

		error = TRUE;
	}

	sprintf(noise, "%s/%s", gtk_entry_get_text (GTK_ENTRY (AudioPath)),
		gtk_entry_get_text (GTK_ENTRY (CurrentEvent)));

	if( stat(noise, &filestat)  == -1 )
	{
		GtkWidget *button = gtk_button_new_with_label("Close");
		
		gchar *msg = "The sound file:\n\n"
				 	 "  %s  \n"
					 "was not found, or it is not readable by you.\n\n"
					 "Please correct the path and try again.\n\n"
					 "Alternatively, you can check to make sure you have \n"
					 "read permissions for this sound file.";

		gchar *buffer = g_malloc0(strlen(msg) + strlen(noise));

		sprintf(buffer, msg, noise);

		ascp_error_popup("Cannot play sound\n", buffer, button);

		error = TRUE;
	}

	//g_print("noise is %s and file is %d\n", noise, (gint) filestat.st_blocks );

	if(error ||  strlen((gchar *)gtk_entry_get_text(GTK_ENTRY(CurrentEvent)) ) < 1)
		return TRUE;


	if ( !error && fork() == 0 )
	{
				
		if ( -1 == execlp((gchar *)gtk_entry_get_text (GTK_ENTRY (AudioPlayCmd)), "sound", noise, 0))
		{
			perror("ERROR> playing sound");
		}
		
		//g_free(noise);
		exit(EXIT_SUCCESS);

	}


	return TRUE;
}





static void init_and_append_entry(gchar *line, gint row)
{
	gchar *event = g_malloc0(16);
	gchar *sound = g_malloc0(16);


	for( ; isspace(*line) ; line++)
		;


	/* 
	 * We've got a line with two entries.
	 * Let's break 'em up into tokens.
	 */

	sscanf(line, "%s %s", event, sound);
	{
		gchar *label[1] = {event};
		gtk_clist_append (GTK_CLIST(EventList), label);
	}
	{
		gchar *label[1] = {sound};
		gtk_clist_append (GTK_CLIST(SoundList), label);
	}

	/* 
	 * Even though the SoundList has a list of sounds, they are not
	 * necessarily going to be the same for the event. For now, associate a
	 * sound with an event.
	 */
	gtk_clist_set_row_data(GTK_CLIST(EventList), row, (gchar  *) sound);  

	g_free(event);
}

static gint change_event_sound(GtkWidget *button, gpointer null)
{

	gtk_clist_set_row_data(GTK_CLIST(EventList), CurrentRow, 
		(gchar  *) gtk_entry_get_text(GTK_ENTRY(CurrentSound))); 

	gtk_entry_set_text (GTK_ENTRY (CurrentEvent), 
		(gchar  *) gtk_entry_get_text(GTK_ENTRY(CurrentSound)));

	return TRUE;
}

static gint event_entry_selected(GtkWidget   *clist, gint row, gint column,
	GdkEventButton *event, gpointer *entry)
{
	gtk_entry_set_text(GTK_ENTRY (entry), 
		(gchar *) gtk_clist_get_row_data (GTK_CLIST (clist), row));

	CurrentRow = row;

	return TRUE;
}


static gint sound_entry_selected(GtkWidget   *clist, gint row, gint column,
	GdkEventButton *event, gpointer *entry)
{
	gchar *text[1] = {g_malloc0(16)};

	gtk_clist_get_text(GTK_CLIST(clist), row, column, text);
	gtk_entry_set_text(GTK_ENTRY (entry), text[0]);

	return TRUE;
}


static gint add_sound_entry(GtkWidget *button, gint which)
{
	gchar *text[1] = {gtk_entry_get_text(GTK_ENTRY(CurrentSound))};

	gtk_clist_append(GTK_CLIST(SoundList), text);

	return TRUE;
}


static gint remove_sound_entry(GtkWidget *button, GtkWidget *entry)
{
	gtk_entry_set_text (GTK_ENTRY (entry), "");

	return TRUE;
}

static void parse_audio(gfloat min, gfloat max)
{
	gfloat incr;
	gint row=0;
	gchar *buffer = (gchar*) g_malloc0(BUFFERSIZE);
	FILE *backupfile;
	

	backupfile = ascp_open_file("audio.ascp", "w");
	IoFile = ascp_open_file("audio", "r");

	incr = (max - min)/FILESIZE;

	ascp_update_status(NULL, "audio", min);

	while (fgets(buffer, BUFFERSIZE, IoFile) != NULL)
	{
		char* line = buffer;

		/*      
	 	 * Show our progress per total bytes per line read.
	 	 */
		ascp_update_status(NULL, NULL, min); 
		min = ( min >= max) ? max : min + incr*strlen(buffer); 

		fputs(line, backupfile);

		for ( ; isspace(*line) ; line++);

		if( !ascp_strncasecmp(line, "*Audio", 6) )
		{
			line += 6;

			if( !ascp_strncasecmp(line, "Path", 4))
			{
				gchar *cmd = g_malloc0(256);

				sscanf(line + 4, "%s", cmd);
				gtk_entry_set_text (GTK_ENTRY (AudioPath), cmd);
				
				g_free(cmd);
			}
			else if( !ascp_strncasecmp(line, "PlayCmd", 7))
			{
				gchar *cmd = g_malloc0(64);

				sscanf(line + 7, "%s", cmd);
				gtk_entry_set_text (GTK_ENTRY (AudioPlayCmd), cmd);
				
				g_free(cmd);
			}
			else if( !ascp_strncasecmp(line, "Delay", 5))
			{
				gint value;

				sscanf(line + 5, "%d", &value);
				gtk_spin_button_set_value(GTK_SPIN_BUTTON(AudioDelay), value);
				
			}
		    else if( isspace(*line))
			{
				init_and_append_entry(line, row++);
			}
		}
		
		

	}

	fclose(IoFile);
	fclose(backupfile);
}


void ascp_write_audio()
{
	gint row;
	gchar *text[1] = {g_malloc0(16)};

	IoFile = ascp_open_file("audio", "w");

	fprintf(IoFile, "\n\n#----------------------------------------------------------------\n");
	fprintf(IoFile, "# This file was written by %s\n", ASCPVERSION);
	fprintf(IoFile, "#----------------------------------------------------------------\n\n");

	fprintf(IoFile, "#----------------------------------------------------------------\n");
	fprintf(IoFile, "#\n");
	fprintf(IoFile, "*AudioPath %s\n", (gchar *) gtk_entry_get_text (GTK_ENTRY(AudioPath)));
	fprintf(IoFile, "*AudioPlayCmd  %s\n", (gchar *) gtk_entry_get_text (GTK_ENTRY(AudioPlayCmd)));
	fprintf(IoFile, "*AudioDelay  %d\n", 
			gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON (AudioDelay)) );
	fprintf(IoFile, "\n\n");


	
	fprintf(IoFile, "#----------------------------------------------------------------\n");
	fprintf(IoFile, "# The events:\n\n");
	row = 0;
	while(gtk_clist_get_text(GTK_CLIST(EventList), row++, 0, text))
	{
		fprintf(IoFile, "*Audio %s %-20s\n", text[0], 
			(gchar *) gtk_clist_get_row_data (GTK_CLIST (EventList), row-1));
	}

	fprintf(IoFile, "\n\n#----------------------------------------------------------------\n");
	fprintf(IoFile, "# And that's all ascp wrote");
	fprintf(IoFile, "\n#----------------------------------------------------------------\n");
	fprintf(IoFile, "\n\n");


	fclose(IoFile);

}



