/*  
 *  ascp: a control panel for the AfterStep window manager
 *  Copyright (C) 1998  Nwanua Elumeze
 *
 *  You may distribute under the terms of the GNU General Public
 *  License as specified in the file, COPYING.
 *
 */


/* 
 * Naming Convention:
 *	
 * global variables:  GlobalVariables;
 * local variables:  localvariables;
 * application wide global variables:  APPWIDEGLOBVAR;
 *
 * local functions : regular_functions; 
 * application wide functions:   ascp_foobar;
 *
 */


#include "ascp.h"
#include "xpm.h"

#define min(a, b)       ((a) < (b) ? (a) : (b))

/*-----------------------------------------*/
static void draw_toplevel(void);
static void destroy_splash_screen (void);
static void check_for_config_files (void);
static void create_splash_screen (gchar *);
static GtkWidget * make_icon(gchar *name, gint tab); 
static GtkWidget * populate_frame(gint, gfloat, gfloat);
static gint change_xpm_name(GtkWidget *, GdkEvent *, gchar *);
/*-----------------------------------------*/


/*-----------------------------------------*/
static GtkWidget *SplashScreen = NULL;
static GtkWidget *MainNotebook = NULL;
static GtkWidget *Label1 = NULL;
static GtkWidget *Label2 = NULL;
static GtkWidget *ProgressBar = NULL;
static gint IdleTag = -1;

static gchar *ConfigFiles[] = {
/* fix this */
			"base.8bpp",
			"wharf",
			"wharf",
			"database",
			"pager",
			"animate",
			"audio",
			"autoexec",
			"winlist",
			NULL};

static gint NumOfConfigFiles  = 
			( (sizeof(ConfigFiles)-sizeof(*ConfigFiles)) / sizeof(*ConfigFiles) );



static gchar *TopMenuIcons[] = {
			"path.xpm",
			"wharf.xpm",
			"wharf2.xpm",
			"whandling.xpm",
			"desktop.xpm",
			"MaelstromInFrame.xpm",
			"sound.xpm",
			"init.xpm",
			"woptions.xpm",
			NULL};


static gchar *TopMenuNames[] = {
			"Paths\n",
        	"Wharf\nEntries",
        	"Wharf\nStyle",
        	"Window\nHandling",
        	"Pager\n",
			"Animate\n",
			"Audio\n",
			"AutoExec\n",
			"Window\nList",
			NULL};

 static gint TopMenuNumber = 
			( (sizeof(TopMenuIcons)-sizeof(*TopMenuIcons)) / sizeof(*TopMenuIcons) );

			
static GtkWidget *GlobalWidget=NULL;
static GtkWidget *GlobalIcon=NULL;
static gchar *GlobalData=NULL;
static GtkWidget *ColorselDlg;
static gdouble Color[3];
// static GtkWidget *LogoArea = NULL;

/*-----------------------------------------*/


/* First check to make sure all configuration files exist.
 * On pass, setup the TOPLEVELWINDOW, MainNotebook and the 
 * four toplevel buttons.
 * Then call the various functions to draw and read
 * corresponding config files, all the while updating
 * ProgressBar.
 */
void ascp_init(void)
{
	
	GlobalData = (gchar *) g_malloc0(64);
	create_splash_screen("ascp Startup");
			

	/*
	 * Check to see if all the needed configuration files exist.
	 * If so, continue; otherwise exit gracefully.
	 * 
	 * If an error occurs, ascp will not return to this point: 	
	 * check_for_config_files() will call ascp_error_popup() 
	 * with a ascp_quit() action.
	 */
	check_for_config_files();

	if(!ASCPERROR)
	{
		/* 
		 * Looks like all's well at this point.
		 * Setup application wide variables like pixmaps, and
		 * ascp specific options.
		 * Go ahead and create the main window and notebooks.
		 */

		draw_toplevel();
	
		/* 
		 * Ensure that the window has NOT been created.
		 */
		BROWSERWINDOW = NULL;
		
	}
	
	 
}
			
	
/* 
 * If a configuration file cannot be found, pop up an error dialog 
 * and prevent the rest of ascp from loading.
 * Otherwise return succesfully.
 */

static void check_for_config_files (void)
{

	gchar *filename;
	struct stat filestat;
	gfloat nth=0.0;
	gint i=0;
	
	ascp_update_status("Checking for Configuration Files", NULL, -1);

	i=0;
	
	while(*(ConfigFiles + i))
	{	
		filename = ascp_make_path(HOMEDIR, ASHOMEPATH, *(ConfigFiles + i));

		ascp_update_status(NULL, filename, nth++/NumOfConfigFiles);

		if( stat(filename, &filestat)  == -1)
		{
			g_free(filename);
			filename = ascp_make_path(ASSHAREPATH, "", *(ConfigFiles+i));
			ascp_update_status(NULL, filename, nth++/NumOfConfigFiles);
		}
		
		if( stat(filename, &filestat)  == -1)
		{
			GtkWidget *button = gtk_button_new_with_label("Exit!");

			gchar *msg = "The configuration file \"%s\" was not found.\n"
						 "I am looking in\n"
						 " %s/%s/%s\n"
						 "and %s/%s.\n\n"
                         "Please exit and make sure the file exists,\n"
						 "and that you have permissions to read it.";

			gchar *buffer = g_malloc0( strlen(msg) + 3*strlen(*(ConfigFiles+i)) 
				+ strlen(HOMEDIR) + strlen(ASHOMEPATH) + strlen(ASSHAREPATH) + 3);

			sprintf(buffer, msg, *(ConfigFiles+i), HOMEDIR, ASHOMEPATH, *(ConfigFiles+i),
					ASSHAREPATH, *(ConfigFiles+i) );

			gtk_signal_connect_object (GTK_OBJECT (button), "clicked", 
    			GTK_SIGNAL_FUNC (ascp_exit), NULL);

			ascp_error_popup("file not found", buffer, button);
			
			gtk_widget_destroy(SplashScreen);
			g_free(buffer);
			ASCPERROR = TRUE;
			break;
		}
		else
		{		
			i++;
		}
		/* 
		 *	No errors, show a filled ProgressBar.
		 */ 
		ascp_update_status(NULL, filename, nth/NumOfConfigFiles);

	}

	return;
}



void ascp_error_popup (gchar *message, gchar *text, GtkWidget *actionbox)
{   
    GtkWidget *window; 
    GtkWidget *label; 
	gchar *title = g_malloc0(15 + strlen(message));
    
    window = gtk_dialog_new ();
    
	sprintf(title, "ascp Error: %s", message);
    gtk_window_set_title(GTK_WINDOW(window), title);
    
    gtk_signal_connect (GTK_OBJECT (window), "delete_event",
        GTK_SIGNAL_FUNC (gtk_object_destroy), GTK_OBJECT (window));

    gtk_signal_connect (GTK_OBJECT (window), "destroy",
        GTK_SIGNAL_FUNC (gtk_false), GTK_OBJECT (window));


    gtk_signal_connect_object (GTK_OBJECT (actionbox), "clicked",
                 GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT (window));




    label = gtk_label_new (text);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->vbox),

	/* 
	 * Give a nice overall size to the error dialog 
	 */
    gtk_label_new(
			"______________________________________________________________\n"),
   			TRUE, TRUE, 0);

    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->vbox), label, TRUE, TRUE, 0);


    gtk_box_pack_start(GTK_BOX (GTK_DIALOG (window)->action_area), actionbox,
                FALSE, FALSE, 0);

	  GTK_WIDGET_SET_FLAGS (actionbox, GTK_CAN_DEFAULT);
	  gtk_widget_grab_default (actionbox);

	if(SplashScreen)
		destroy_splash_screen();

    if (!GTK_WIDGET_VISIBLE (window))
        gtk_widget_show_all (window);
    else
        gtk_widget_destroy (window);

}


static void create_splash_screen (gchar *title)
{
      GtkWidget *vbox;

      SplashScreen = gtk_window_new(GTK_WINDOW_DIALOG);
      gtk_window_set_wmclass (GTK_WINDOW(SplashScreen), "ascp_startup", "ascp");
      gtk_window_set_title(GTK_WINDOW(SplashScreen), title);
	  gtk_widget_set_usize(SplashScreen, 400, 60);

      vbox = gtk_vbox_new(FALSE, 4);
      gtk_container_add(GTK_CONTAINER(SplashScreen), vbox);

	//gtk_widget_push_visual (gtk_preview_get_visual ());
	//gtk_widget_push_colormap  (gtk_preview_get_cmap ());

	/* the logo here */

      //gtk_widget_pop_colormap ();
      //gtk_widget_pop_visual ();

      Label1 = gtk_label_new("");
      gtk_box_pack_start_defaults(GTK_BOX(vbox), Label1);
      Label2 = gtk_label_new("");
      gtk_box_pack_start_defaults(GTK_BOX(vbox), Label2);

      ProgressBar = gtk_progress_bar_new();
      gtk_box_pack_start_defaults(GTK_BOX(vbox), ProgressBar);

      gtk_widget_show(vbox);
      gtk_widget_show(Label1);
      gtk_widget_show(Label2);
      gtk_widget_show(ProgressBar);
	   gtk_window_position(GTK_WINDOW(SplashScreen),
                  GTK_WIN_POS_CENTER);

      gtk_widget_show(SplashScreen);

      gtk_window_set_policy (GTK_WINDOW (SplashScreen), FALSE, TRUE, FALSE);
}



static void destroy_splash_screen (void)
{
  if(SplashScreen)
    {
      gtk_widget_destroy(SplashScreen);
      SplashScreen = Label1 = Label2 = ProgressBar =  NULL;
      //logo_pixmap = NULL;
      gtk_idle_remove(IdleTag);
    }
}


static void my_idle_proc(void)
{
  /* Do nothing. This is needed to stop ascp
     from blocking in gtk_main_iteration() */
}



void ascp_update_status(gchar *label1val, gchar *label2val, gfloat pctprogress)
{
  if(SplashScreen)
	{
			
		//GdkRectangle area = {0, 0, -1, -1};

		if(label1val && strcmp(label1val, GTK_LABEL(Label1)->label))
		{
			gtk_label_set(GTK_LABEL(Label1), label1val);
		}
		if(label2val && strcmp(label2val, GTK_LABEL(Label2)->label))
		{
			gtk_label_set(GTK_LABEL(Label2), label2val);
		}
		if(pctprogress >= 0 )
		{
			gtk_progress_bar_update(GTK_PROGRESS_BAR(ProgressBar), pctprogress);
		}
		
		//gtk_widget_draw(SplashScreen, &area);
		IdleTag = gtk_idle_add((GtkFunction) my_idle_proc, NULL);
		gtk_main_iteration();
		gtk_idle_remove(IdleTag);
	}
}



/* 
 * Create the TOPLEVELWINDOW and MainNotebook; 
 * filling them with children as specified
 * in the various draw_...() function
 */

void draw_toplevel()
{

	GtkWidget *button;
	GtkWidget *box;
	GtkWidget *table;
	GtkWidget *frame;
	gint i;
	gchar *menutitles[] = {
			"Paths",
			"WharfEntry",
			"WharfStyle",
			"Database",
			"Pager",
			"Animate",
			"Audio",
			"Autoexec",
			"WinList"
			};
			


	if(ASCPERROR)
		return;
	
	ascp_update_status("Parsing files and building gui", NULL, -1);

	/* 
	 * Set up TOPLEVELWINDOW and connect signals to it. 
     */
	TOPLEVELWINDOW = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(TOPLEVELWINDOW),"AfterStep Control Panel");
	gtk_widget_set_usize (TOPLEVELWINDOW, 408, 348);
	gtk_signal_connect (GTK_OBJECT (TOPLEVELWINDOW), "delete_event",
		GTK_SIGNAL_FUNC (ascp_exit), NULL);
	gtk_container_border_width (GTK_CONTAINER (TOPLEVELWINDOW), 0);
	gtk_widget_realize(TOPLEVELWINDOW);


	/* 
	 * Setup hints for the window manager. 
	 */
	ascp_set_wm_hints();
	

	/*
	 * Create the application wide pixmaps.
	 */
	PIXMAP1 = gdk_pixmap_create_from_xpm_d (TOPLEVELWINDOW->window, 
		&MASK1,  &TRANSPARENT, book_closed_xpm);

	PIXMAP2 = gdk_pixmap_create_from_xpm_d (TOPLEVELWINDOW->window, 
		&MASK2,  &TRANSPARENT, book_open_xpm);
			
	PIXMAP3 = gdk_pixmap_create_from_xpm_d (TOPLEVELWINDOW->window, 
		&MASK3,  &TRANSPARENT, mini_page_xpm);
	

	/* 
	 * This table is the parent of the MainNotebook and 
	 * the four toplevel buttons: "Done", "Apply", "Exit", "Help".
	 */
	table = gtk_table_new(2,6, FALSE);
	gtk_container_add (GTK_CONTAINER (TOPLEVELWINDOW), table);


	/* 
	 * Set up MainNotebook, connecting signals and adding config
	 * pages, as its children.
	 */
	MainNotebook = gtk_notebook_new ();
	gtk_notebook_set_tab_pos (GTK_NOTEBOOK (MainNotebook), GTK_POS_TOP);
	gtk_table_attach(GTK_TABLE(table), MainNotebook, 0,6,0,1,
		GTK_FILL, GTK_FILL, 0, 0);
	gtk_notebook_set_scrollable (GTK_NOTEBOOK (MainNotebook), TRUE);
	gtk_notebook_popup_enable (GTK_NOTEBOOK (MainNotebook));
	gtk_container_border_width (GTK_CONTAINER (MainNotebook), 1);
	gtk_notebook_set_show_border( GTK_NOTEBOOK (MainNotebook), FALSE);
	gtk_widget_realize(MainNotebook);


	for (i=0; i < TopMenuNumber; i++)
	{
		frame = gtk_frame_new ( *(TopMenuNames+i) );
		gtk_container_border_width (GTK_CONTAINER (frame), 10);
		gtk_widget_set_usize (frame, 400, 240);

		box = populate_frame(i, (1.0*i)/TopMenuNumber, (1.0*i + 1)/TopMenuNumber);
		gtk_container_add (GTK_CONTAINER (frame), box);
		gtk_widget_show(box);
		gtk_widget_show(frame);

		if(ASCPERROR)
			return;

		/* 
		 * Here's a big delay if you are using gtk+1.0.x.
		 * I believe that's due to the spectacular colors of 
		 * the top tab icons, thanx Marco :-)
		 */
		/*
		gtk_notebook_append_page (GTK_NOTEBOOK (MainNotebook), 
			frame, make_icon (*(TopMenuIcons+i), i));
			*/
		
		gtk_notebook_append_page_menu(GTK_NOTEBOOK (MainNotebook), frame, 
			//make_icon (*(TopMenuIcons+i), i), make_icon (*(TopMenuIcons+i), i) );
			make_icon (*(TopMenuIcons+i), i), gtk_label_new(*(menutitles+i)) );
			//gtk_label_new(*(TopMenuNames+i)), gtk_label_new(*(TopMenuNames+i)) );
			
	}
	
	destroy_splash_screen ();
	gtk_widget_show(MainNotebook);

	
	/* 
	 * Setup the four toplevel buttons; adding callbacks.
	 */
	button = gtk_button_new_with_label ("Done");
	frame = gtk_frame_new (NULL);
	gtk_container_border_width (GTK_CONTAINER (frame), 0);
	gtk_signal_connect (GTK_OBJECT (button), "clicked",
		GTK_SIGNAL_FUNC (ascp_save_all), NULL );
	gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
		GTK_SIGNAL_FUNC (ascp_exit), NULL);
	gtk_container_add (GTK_CONTAINER (frame), button);
	gtk_table_attach(GTK_TABLE(table), frame, 0,1,1,2,
		GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show_all(frame);

	button = gtk_button_new_with_label ("Apply");
	frame = gtk_frame_new (NULL);
	gtk_container_border_width (GTK_CONTAINER (frame), 0);

	/* 
	 * Also add alternate menus to save just this page
	 * or all pages ("Apply" button).
	 */
	
	{
		GtkWidget *menu;
		GtkWidget *menuitem;
		gchar *menumsg[] = {
				"This page",
				"All pages"
				};
		
		menu = gtk_menu_new();
		
		menuitem = gtk_menu_item_new_with_label(menumsg[0]);
		gtk_menu_append(GTK_MENU (menu), menuitem);
		gtk_widget_show(menuitem);
		gtk_signal_connect(GTK_OBJECT(menuitem), "activate",
			GTK_SIGNAL_FUNC (ascp_save_page), (GtkWidget *) MainNotebook );
		gtk_widget_show(menuitem);
		

		menuitem = gtk_menu_item_new_with_label(menumsg[1]);
		gtk_menu_append(GTK_MENU (menu), menuitem);
		gtk_widget_show(menuitem);
		gtk_signal_connect(GTK_OBJECT(menuitem), "activate",
			GTK_SIGNAL_FUNC (ascp_save_all), NULL );
		gtk_widget_show(menuitem);
		

		gtk_signal_connect_object(GTK_OBJECT(button), "button_press_event",
			GTK_SIGNAL_FUNC (ascp_alt_menu), GTK_OBJECT(menu));
	}

	gtk_signal_connect (GTK_OBJECT (button), "clicked",
		GTK_SIGNAL_FUNC (ascp_save_page), (GtkWidget *) MainNotebook );
	gtk_container_add (GTK_CONTAINER (frame), button);
	gtk_table_attach(GTK_TABLE(table), frame, 1,2,1,2,
		GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show_all(frame);

	button = gtk_button_new_with_label ("Exit");
	frame = gtk_frame_new (NULL);
	gtk_container_border_width (GTK_CONTAINER (frame), 0);
	gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
		GTK_SIGNAL_FUNC (ascp_exit), NULL);
	gtk_container_add (GTK_CONTAINER (frame), button);
	gtk_table_attach(GTK_TABLE(table), frame, 2,3,1,2,
		GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show_all(frame);

	button = gtk_button_new_with_label ("Help");
	frame = gtk_frame_new (NULL);
	gtk_container_border_width (GTK_CONTAINER (frame), 0);
	gtk_signal_connect(GTK_OBJECT (button), "clicked",
		GTK_SIGNAL_FUNC (ascp_help), (GtkWidget *) MainNotebook );
	gtk_container_add (GTK_CONTAINER (frame), button);
	gtk_table_attach(GTK_TABLE(table), frame, 5,6,1,2,
		GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show_all(frame);


	/* 
	 * Useful during development only.
	 * May use in the future to jump to user defined pages...
	 * Read: used to jump to last page visited before quit.
	 * Allow user to turn on/off this "feature".
	 */
	gtk_notebook_set_page (GTK_NOTEBOOK(MainNotebook), 
		ASCPCONFIG.startpage);


	gtk_widget_show(table);
	gtk_widget_show(TOPLEVELWINDOW);

	return;
}

	
static GtkWidget *populate_frame(gint number, gfloat min, gfloat max)
{

	switch( number)
	{
		case  0: return( ascp_draw_base(min, max) );
		case  1: return( ascp_draw_wharf(min, max) );
		case  2: return( ascp_draw_wharfstyle(min, max) );
		case  3: return( ascp_draw_database(min, max) );
		case  4: return( ascp_draw_pager(min, max) );
		case  5: return( ascp_draw_animate(min, max) );
		case  6: return( ascp_draw_audio(min, max) );
		case  7: return( ascp_draw_autoexec(min, max) );
		case  8: return( ascp_draw_winlist(min, max) );
		default: return gtk_label_new("empty page");
	}

}


GtkWidget  *ascp_make_pixmap_from_data(gchar **xpm_data)
{
	GtkStyle    *style;
	GdkPixmap   *pixmap;
	GdkBitmap   *mask;
	style = gtk_widget_get_style(TOPLEVELWINDOW );
	pixmap = gdk_pixmap_create_from_xpm_d( TOPLEVELWINDOW->window,
		&mask, &style->bg[GTK_STATE_NORMAL], xpm_data); 

	return gtk_pixmap_new(pixmap, mask );
}



static GtkWidget * make_icon(gchar *name, int tab)
{
	GtkStyle    *style;
	GdkPixmap   *pixmap;
	GdkBitmap   *mask;
	FILE *infile = NULL;
	gchar* home_path = ascp_make_path(HOMEDIR, ASHOMEPATH, "");
	gchar* filename = NULL;

	filename = ascp_make_path(home_path, "ascp/icons/", name);
	infile = fopen (filename, "rb");

	if (infile == NULL)
	{
		g_free(filename);
		filename = ascp_make_path(ASSHAREPATH, "ascp/icons/", name);
		infile = fopen (filename, "rb");
	}

	g_free(home_path);
	

	if (infile != NULL)
	{
		fclose(infile);
		style = gtk_widget_get_style( TOPLEVELWINDOW );
		pixmap = gdk_pixmap_create_from_xpm(TOPLEVELWINDOW->window,
			&mask, &style->bg[GTK_STATE_NORMAL], filename);
		g_free(filename);
		return gtk_pixmap_new(pixmap, mask );
	}
	else
	{
		gchar *buffer = g_malloc0(strlen(*(TopMenuNames+tab))+20);
		GtkWidget *label;

		g_free(filename); 

		/* 
		 * The spacing exists to make the tabs still look 
         * nice in the event of a missing icon.
		 */
		sprintf(buffer, "\n%s\n                ", *(TopMenuNames+tab) );
		label = gtk_label_new(buffer);
		g_free(buffer);
		return label;
	}
    
}


/* 
 * Set up the for the window manager:
 * the default icon and name when it's iconized;
 * 
 */
void ascp_set_wm_hints(void)
{
	GtkStyle    *style;
    GdkPixmap   *pixmap;
    GdkBitmap   *mask;
    FILE *infile = NULL;
    gchar* home_path = ascp_make_path(HOMEDIR, ASHOMEPATH, "");
    gchar* filename = NULL;

	filename = ascp_make_path(home_path, "ascp/icons", HINTICON);
    infile = fopen (filename, "rb");
    
    if (infile == NULL)
    {
        g_free(filename);
        filename = ascp_make_path(ASSHAREPATH, "ascp/icons", HINTICON);
        infile = fopen (filename, "rb");
    }

    g_free(home_path);

    if (infile != NULL)
    {
        fclose(infile);
        style = gtk_widget_get_style( TOPLEVELWINDOW );
        pixmap = gdk_pixmap_create_from_xpm(TOPLEVELWINDOW->window,
            &mask, &style->bg[GTK_STATE_NORMAL], filename);
        g_free(filename);
		gdk_window_set_icon (TOPLEVELWINDOW->window, NULL, pixmap, mask);
    }
	else
	{
		g_print("Can't find hint icon... getting on with life.\n");
	}

	gdk_window_set_icon_name (TOPLEVELWINDOW->window, "ascp");
	gdk_window_set_functions (TOPLEVELWINDOW->window, GDK_FUNC_ALL | GDK_FUNC_RESIZE);
	gdk_window_set_decorations (TOPLEVELWINDOW->window, GDK_DECOR_ALL | GDK_DECOR_MENU);
	gdk_window_set_hints(TOPLEVELWINDOW->window,  200, 300, 408, 348, 408, 348, 1);

	return;
}


/* 
 * Catch events related to alt option menus. 
 * Of particular interest are the second and third buttons:
 */
gint ascp_alt_menu (GtkWidget *widget, GdkEvent *event)
{
	/* A button has been pressed. */
	if( event->type == GDK_BUTTON_PRESS)
	{
		 /* Find out which one it was. */
		GdkEventButton *bevent = (GdkEventButton *) event;

		if(bevent->button != 2 &&  bevent->button != 3)
			/* 
			 * False alarm.
			 * Tell calling code we've not handled this event;
			 */
			return FALSE;
	 

		/*
		 * Popup the menu that was attached to the calling button.
		 */
		gtk_menu_popup (GTK_MENU(widget), NULL, NULL, NULL, NULL,
			bevent->button, bevent->time);
		
		/* 
		 * Tell calling code we've handled this event. 
		 * Don't call any other possible handlers for this
		 * particular event.
		 */
		return TRUE;
	}
	
	/* 
	 * As  it did not find what it was looking for, this callback 
	 * did not handle the event. Tell calling code so.
	 */
	return FALSE;
			
}



/*
 * Hide or show the "child" depending on the state of the "widget"
 */
gint  ascp_checkbutton_hide_show(GtkWidget *widget, GtkWidget *child)
{
	if( GTK_TOGGLE_BUTTON (widget)->active && child) 
	{
		//gtk_widget_show((child));

		if(ASCPCONFIG.hide)
		{
			gtk_widget_show_all((child));
		}
		else
		{
			gtk_widget_set_sensitive (child, TRUE);
		}
	}
	else
	{

		if(ASCPCONFIG.hide)
		{
			gtk_widget_hide(GTK_WIDGET(child));
		}
		else
		{
			gtk_widget_set_sensitive (child, FALSE);
		}
	}

	/* 
	 * Tell calling code we've handled this event. 
	 */
	return TRUE;
}



/* suggestion to me: mebbe set a global pointer to point to the text at hand, hmm?*/


/* 
 * Create a browser window to show all the pixmaps in the
 * current/last directory.
 * Set up each one so a click on a preview will copy the image
 * to the calling button.
 */

gint ascp_xpm_browser (GtkWidget *hostbutton, GdkEvent *event, gchar *hostdata)
{
	GtkWidget *toptable;
	GtkWidget *topframe;
	GtkWidget *table;
	GtkWidget *frame;
	GtkWidget *button;
	GtkWidget *done;
	GtkWidget *scrolledwin;
	DIR *dir;
	struct dirent  *file;
	gchar *currdir = (gchar *) g_malloc0(64); 
	//gchar *iconname = (gchar *) g_malloc0(64);
	gint row=0, column=0;

	GdkEventButton *bevent = (GdkEventButton *) event;

	/* 
	 * A button has been pressed. 
	 * Find out which one it was. 
	 */
	if( event->type == GDK_BUTTON_PRESS &&  bevent->button != 1 )
	{
		/* 
		 * False alarm.
		 * Tell calling code we've not handled this event;
		 */
		return FALSE;
	}

	GlobalIcon = hostbutton;
	/* 
	 * Setup the window and pop it up if it doesn't 
	 * already exist. 
	 */
	if(!BROWSERWINDOW)
	{
		BROWSERWINDOW = gtk_window_new(GTK_WINDOW_TOPLEVEL);
		gtk_window_set_title(GTK_WINDOW(BROWSERWINDOW), "ascp Pixmap Browser");
		gtk_widget_set_usize (BROWSERWINDOW, 390, 300);
		gtk_signal_connect (GTK_OBJECT (BROWSERWINDOW), "delete_event",
			GTK_SIGNAL_FUNC (gtk_widget_hide), GTK_OBJECT (BROWSERWINDOW));
		gtk_widget_realize(BROWSERWINDOW);

		scrolledwin = gtk_scrolled_window_new (NULL, NULL);
		gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwin),
			 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

		toptable = gtk_table_new (6, 6, FALSE);;

		table = gtk_table_new (1, 1, TRUE);
		gtk_container_border_width (GTK_CONTAINER (table), 5);
		gtk_table_set_row_spacings (GTK_TABLE (table), 2);
		gtk_table_set_col_spacings (GTK_TABLE (table), 2);


		/* 
	 	 * Setup the entry box and the button at the bottom
	 	 * of the browser.
	 	 */
		{
			GtkWidget *entry;
			GtkWidget *frame;
		
			frame = gtk_frame_new (NULL);
			entry = gtk_entry_new_with_max_length (100);
			gtk_entry_set_text (GTK_ENTRY (entry), "Not implemented yet");
			gtk_widget_set_sensitive(entry, FALSE);
			gtk_container_add(GTK_CONTAINER(frame), entry);
			gtk_table_attach( GTK_TABLE(toptable), frame,
				0, 4, 5, 6, GTK_FILL, GTK_FILL, 0, 0);

			done = gtk_button_new_with_label("Done");
			gtk_signal_connect_object (GTK_OBJECT (done), "clicked",
				GTK_SIGNAL_FUNC (gtk_widget_hide), GTK_OBJECT (BROWSERWINDOW));
			gtk_table_attach(GTK_TABLE (toptable), done, 
				5, 6, 5, 6, GTK_FILL, GTK_FILL, 0, 0);
			GTK_WIDGET_SET_FLAGS (done, GTK_CAN_DEFAULT);
		}

		create_splash_screen("ascp Pixmap Browser");

		topframe = gtk_frame_new(NULL);
		gtk_widget_set_usize (topframe, 380, 260);
	
		//g_print("total %d", (gint) dir->dd_size);

		ascp_update_status("Building Pixmaps", NULL, -1);

		if(ASCPCONFIG.bpp == 8)
			currdir = ascp_make_path(HOMEDIR, ASHOMEPATH, "desktop/icons/8bpp");
		if(ASCPCONFIG.bpp == 16 || ASCPCONFIG.bpp == 24 || ASCPCONFIG.bpp == 32)
			currdir = ascp_make_path(HOMEDIR, ASHOMEPATH, "desktop/icons/16bpp");

		dir=opendir(currdir);
		if(!dir) 
		{
			GtkWidget *button = gtk_button_new_with_label("Close");

			ascp_error_popup("Directory not found\n", currdir, button);
			closedir(dir);
			return FALSE;
		}

				//g_print("hostname is %s\n", hostdata);


		//gchar *iconname = (gchar *) g_malloc0(strlen(file->d_name));


		while ((file = readdir (dir)))
		{

			//strcpy(iconname, file->d_name);
			
			//if (streq (iconname, ".") || streq (iconname, ".."))
			if (streq (file->d_name, ".") || streq (file->d_name, ".."))
				continue;
			
			if(strstr (file->d_name, ".xpm"))
			//if(strstr (iconname, ".xpm"))
			{	
				
				//ascp_update_status(NULL, iconname, 1.0 - (1.0/(gint) telldir(dir)) );
				ascp_update_status(NULL, file->d_name, (gfloat) telldir(dir)/2048 );
				gtk_main_iteration();
		
				button = gtk_button_new();
				frame = gtk_frame_new(NULL);
				gtk_container_border_width (GTK_CONTAINER (button), 4);
				gtk_widget_set_usize (frame, 84, 84);
				gtk_container_border_width (GTK_CONTAINER (frame), 1);

				gtk_container_add(GTK_CONTAINER(button),
					ascp_make_pixmap(ascp_make_path(currdir, "", file->d_name), BROWSE));
			
				{
					gchar *iconname = strdup(file->d_name);

					gtk_object_set_data(GTK_OBJECT(button), "icon", iconname);
				}
				gtk_signal_connect(GTK_OBJECT (button), "button_press_event", 
    				GTK_SIGNAL_FUNC (ascp_change_xpm), NULL /*file->d_name*/);


				gtk_signal_connect(GTK_OBJECT (button), "button_press_event", 
    				GTK_SIGNAL_FUNC (change_xpm_name), hostdata);
					

				gtk_container_add(GTK_CONTAINER(frame), button);
				gtk_table_attach(GTK_TABLE (table), frame, 
					column, column+1, row, row+1, GTK_FILL, GTK_FILL, 0, 0);

				column++;
				if (column == 4)
				{
					column = 0;
					row++;
				}
			}
		}
	
		//g_free(iconname);
		destroy_splash_screen();


#ifndef GTK_STABLE_VERSION
		gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolledwin), table); 
#else
		gtk_container_add (GTK_CONTAINER (scrolledwin), table);
#endif
		gtk_container_add(GTK_CONTAINER(topframe), scrolledwin);
		gtk_table_attach(GTK_TABLE (toptable), topframe, 
			0, 6, 0, 5, GTK_FILL, GTK_FILL, 0, 0);
		gtk_container_add (GTK_CONTAINER (BROWSERWINDOW), toptable);

		gtk_widget_show_all(BROWSERWINDOW);


		gtk_widget_grab_default (done);


		g_free(currdir);

		/* 
	 	 * Tell calling code we've handled this event. 
	 	 * Don't call any other possible handlers for this
	 	 * particular event.
	 	 */
		return TRUE;
	}
	else 
	{
		gtk_widget_show_all(BROWSERWINDOW);
		return TRUE;
	}

}
	
static gint change_xpm_name(GtkWidget *widget, GdkEvent *event, gchar *hostname)
{

	/* A button has been pressed. */
	if( event->type == GDK_BUTTON_PRESS)
	{
		 /* Find out which one it was. */
		GdkEventButton *bevent = (GdkEventButton *) event;

		if(bevent->button != 1 )
			/* 
			 * False alarm.
			 * Tell calling code we've not handled this event;
			 */
			return FALSE;
	
		if(hostname)
		{
			strcpy(hostname, GlobalData);
		}
		else
		{
			g_print("there's no name supplied for calling pixmap button!\n");
		}
		return TRUE;
	}
	return FALSE;
}


gint ascp_change_xpm(GtkWidget *from, GdkEvent *event, gchar *data)
{
	/* A button has been pressed. */
	if( event->type == GDK_BUTTON_PRESS)
	{
		 /* Find out which one it was. */
		GdkEventButton *bevent = (GdkEventButton *) event;

		if(bevent->button != 1)
			/* 
			 * False alarm.
			 * Tell calling code we've not handled this event;
			 */
			return FALSE;
	
#if 0
	GdkEventButton *bevent = (GdkEventButton *) event;
	GdkEventButton *bevent = (GdkEventButton *) event;

	if( /*event->type == GDK_BUTTON_PRESS &&  */ bevent->button == 1)
	{
#endif	
		if(GlobalData && GlobalData)
				strcpy(GlobalData, (gchar *) gtk_object_get_data(GTK_OBJECT(from), "icon"));
		else 
		{	
			g_print("there seems to be an error: ascp_change_xpm\n");
			exit(1);
		}

		if(GTK_BIN(GlobalIcon)->child)
		{
			gtk_container_remove(GTK_CONTAINER(GlobalIcon),  
				GTK_BIN(GlobalIcon)->child);
		}
		//gdk_window_clear( (GTK_BIN(GlobalIcon)->child)->window);

		//g_print("the icon is %s\n", data);
		gtk_container_add (GTK_CONTAINER(GlobalIcon), ascp_make_pixmap(GlobalData, SINGLE));

		gtk_widget_show_all(GlobalIcon);

		return TRUE;
	}

	return FALSE;
}

#ifndef GTK_STABLE_VERSION
void ascp_ctree_after_move (GtkCTree *ctree, GtkCTreeNode *child, GtkCTreeNode
*parent,
         GtkCTreeNode *sibling, gpointer data)
{
  char *source;
  char *target1;
  char *target2;

  gtk_ctree_get_node_info (ctree, child, &source,
               NULL, NULL, NULL, NULL, NULL, NULL, NULL);
  if (parent)
    gtk_ctree_get_node_info (ctree, parent, &target1,
                 NULL, NULL, NULL, NULL, NULL, NULL, NULL);
  if (sibling)
    gtk_ctree_get_node_info (ctree, sibling, &target2,
                 NULL, NULL, NULL, NULL, NULL, NULL, NULL);

  /*
  g_print ("Moving \"%s\" to \"%s\" with sibling \"%s\".\n", source,
       (parent) ? target1 : "nil", (sibling) ? target2 : "nil");
	   */


}

void ascp_ctree_after_press (GtkCTree *ctree, gpointer data)
{
  	//g_print ("selections %d\n", g_list_length (GTK_CLIST (ctree)->selection));
  	//g_print ("visible %d\n", g_list_length (GTK_CLIST (ctree)->row_list));

	return;
}

		

gint ascp_ctree_button_press (GtkCTree *ctree, GdkEventButton *event, gpointer data)
{
  gint row;
  gint column;
  GtkCTreeNode *work;
  gint res;
 
  res = gtk_clist_get_selection_info (GTK_CLIST (ctree), event->x,
event->y, &row, &column);
  if (!res && event->button != 3)
    return FALSE;

  work = GTK_CTREE_NODE (g_list_nth (GTK_CLIST (ctree)->row_list, row));

  switch (event->button)
    {
    case 1:
      if (GTK_CLIST (ctree)->selection_mode == GTK_SELECTION_MULTIPLE &&
      event->state & GDK_SHIFT_MASK)
    gtk_signal_emit_stop_by_name (GTK_OBJECT
(ctree),"button_press_event");
      break;
    case  2:
      if (GTK_CTREE_ROW (work)->children &&
 gtk_ctree_is_hot_spot (ctree, event->x, event->y))
    {
      if (GTK_CTREE_ROW (work)->expanded)
        gtk_ctree_collapse_recursive (ctree, work);
      else
        gtk_ctree_expand_recursive (ctree, work);
      ascp_ctree_after_press (ctree, NULL);
      gtk_signal_emit_stop_by_name (GTK_OBJECT (ctree),
                    "button_press_event");
    }
      break;
    default:
      break;
    }
  return FALSE;
}




gint ascp_ctree_button_release (GtkCTree *ctree, GdkEventButton *event, gpointer
data)
{
  gint row;
  gint column;
  GtkCTreeNode *work;
  gint res;
 
  res = gtk_clist_get_selection_info (GTK_CLIST (ctree), event->x,
event->y,
                      &row, &column);
  if (!res || event->button != 1)
    return FALSE;

  work = GTK_CTREE_NODE (g_list_nth (GTK_CLIST (ctree)->row_list, row));

  if (GTK_CLIST (ctree)->selection_mode == GTK_SELECTION_MULTIPLE &&
      event->state & GDK_SHIFT_MASK)
    {
      if (GTK_CTREE_ROW (work)->row.state == GTK_STATE_SELECTED)
        gtk_ctree_unselect_recursive (ctree, work);
      else
    gtk_ctree_select_recursive (ctree, work);
      ascp_ctree_after_press (ctree, NULL);
      gtk_signal_emit_stop_by_name (GTK_OBJECT (ctree),
                    "button_release_event");
    }


  return FALSE;
}



void ascp_ctree_remove_selection (GtkWidget *widget, GtkCTree *ctree)
{
  GtkCTreeNode *work;
  GtkCTreeNode *new_sel;
  GList *selection;

  selection = GTK_CLIST (ctree)->selection;
  new_sel = NULL;

  gtk_clist_freeze (GTK_CLIST (ctree));

  while (selection)
    {
      work = selection->data;
      if (GTK_CTREE_ROW (work)->is_leaf)
		;
   // pages--;
      else ;
    /*gtk_ctree_post_recursive (ctree, work,
                  (GtkCTreeFunc) count_items, NULL);
*/
      if (GTK_CLIST (ctree)->selection_mode == GTK_SELECTION_BROWSE)
    {
      if (GTK_CTREE_ROW (work)->children)
        {
          new_sel = GTK_CTREE_ROW (work)->sibling;
          if (!new_sel)
   new_sel = GTK_CTREE_NODE_NEXT (work);
        }
      else
        {
          if (GTK_CTREE_NODE_NEXT (work))
        new_sel = GTK_CTREE_NODE_NEXT (work);
          else
        new_sel = GTK_CTREE_NODE_PREV (work);
        }
    }

      gtk_ctree_remove (ctree, work);
      selection = GTK_CLIST (ctree)->selection;
    }

  if (new_sel)
    gtk_ctree_select (ctree, new_sel);

  gtk_clist_thaw (GTK_CLIST (ctree));
  ascp_ctree_after_press (ctree, NULL);
}

#endif //GTK_STABLE_VERSION




void ascp_clist_remove_selection (GtkWidget *widget, GtkCList *clist)
{
#ifndef GTK_STABLE_VERSION
	gtk_clist_remove (GTK_CLIST (clist), GTK_CLIST (clist)->focus_row);
#else
	g_print("no focus_row member in STABLE yet\n");
#endif //  GTK_LIB_DEP_CODE
}


GtkWidget  *ascp_make_pixmap(gchar *name, gint type)
{
	GtkStyle    *style; 
	GdkPixmap   *pixmap;
	GdkBitmap   *mask;  
	FILE *infile = NULL; 
	gchar *path[1] = {g_malloc0(16)};
	gchar* filename = NULL; 
	gint row=0;


	if( name == NULL  ||  !strcmp(name, "nil")  || !strcmp(name, "-"))
		return gtk_label_new( "\nNone\n");

	/* 
	 * Note: for this to work, the entire "ascp_draw_base" page must have been
	 * succesful: PIXMAPPATHLIST is defined there.
	 */
	if(type == SINGLE)
	{
		while( gtk_clist_get_text(GTK_CLIST(PIXMAPPATHLIST), row++, 0, path) && (infile == NULL) )
		{
			/* 
			 * Crude path expansion: I'm sure there's something better
			 * out there but here you are :-)
			 */

			/* First remove trailing garbage and tag on a '/' for good luck. */
			path[0][strlen(path[0])-1] = '/';
			path[0][strlen(path[0])] = '\0';
			
			for ( ; isspace(*path[0]) ; path[0]++);


			if(path[0][0] == '~') 
			{
				//g_print("home was %c", path[0][0]);
				path[0] += 1;
				filename = ascp_make_path(HOMEDIR, path[0], name);
				//g_print(" is now %s \n ", HOMEDIR);
			}
			else
				filename = ascp_make_path(path[0], "", name);

			//g*_print("trying path %s ", path[0]);
			

			infile = fopen (filename, "rb");

		}

		 
	}
	else if (type == BROWSE)
	{
		filename = ascp_make_path("", "", name);
		infile = fopen (filename, "rb");
	}


	if (infile != NULL)
	{
#if 0
  		GdkImlibImage *img;
		GtkWidget *pw;
		gint max_height, max_width;
		gboolean scale=FALSE;

     	//img = gdk_imlib_load_image (filename); 
     	img = gdk_imlib_load_image ("/tmp/wmakerconf-1.5/wmakerconf.xpm"); 
		//g_print("file: %s\n\n", filename);

		
    	if (img)
     	{
 	       	if (!max_height)
 	       	   max_height = img->rgb_height;
 	       	if (!max_width)
 	       	   max_width = img->rgb_width;
 	       	if (scale) 
 	       	   gdk_imlib_render (img, max_width, max_height);
 	      	else    
 	       	   gdk_imlib_render (img,
           	      min ((unsigned) img->rgb_width, max_width),
           	      min ((unsigned) img->rgb_height, max_height));

			pixmap = gdk_imlib_move_image (img);
        	mask   = gdk_imlib_move_mask (img);
        	gdk_imlib_kill_image (img);
			pw = gtk_pixmap_new (pixmap, mask);
			gtk_widget_show (pw);
 			return pw;
 	    }
	 	else 
#endif /*IMLIB*/
		{

			gtk_widget_push_visual (gtk_preview_get_visual ());
			gtk_widget_push_colormap  (gtk_preview_get_cmap ());

			fclose(infile);
			style = gtk_widget_get_style( TOPLEVELWINDOW);
			pixmap = gdk_pixmap_create_from_xpm(TOPLEVELWINDOW->window,
				&mask, &style->bg[GTK_STATE_NORMAL], filename);
			free(filename);

			gtk_widget_pop_colormap ();
			gtk_widget_pop_visual ();

			return gtk_pixmap_new(pixmap, mask );
	 	}
	}
	else
	{
		free(filename);
		return gtk_label_new( "Cannot\nFind\nPixmap");
	}

	g_free(path);
}
	


/* OK. Someone's frobbing the color wheel and interrupting
 * gtk_main(). Set the color on the drawingrea in question
 * to that which the troublemaker wants. Also make sure to 
 * tell the global Color[3].
 * Go back to sleep
 */

void ascp_color_changed_cb (GtkWidget *widget, colorwell *well)
{
	GdkColor gdk_color;
	GdkColormap *colormap;

	colormap = gdk_window_get_colormap (well->drawingarea->window);

	gtk_color_selection_get_color (well->colorsel, Color);

	/* Fit to a unsigned 16 bit integer (0..65535) and insert into the GdkColor structure */
	gdk_color.red = (guint16)(Color[0]*65535.0);
	gdk_color.green = (guint16)(Color[1]*65535.0);
	gdk_color.blue = (guint16)(Color[2]*65535.0);

	gdk_color_alloc (colormap, &gdk_color);

	gdk_window_set_background (well->drawingarea->window, &gdk_color);

	gdk_window_clear (well->drawingarea->window);
}



/* Sets the background color for a widget.  The widget must 
 * have an associated window.  Used as a callback for the 
 * "realize" event on colorwell frames.
 */
void ascp_set_background_color (GtkWidget *widget, GdkColor *color)
{
	GdkColormap *colormap;

	colormap = gdk_window_get_colormap (widget->window);

	gdk_color_alloc (colormap, color);

	gdk_window_set_background (widget->window, color);

	gdk_window_clear (widget->window);
}



/* Use gdk_parse_color() to convert a string to a 
 * color.  If the color is invalid, use #168aa1.
 */
void ascp_char_to_color(const gchar *string, GdkColor *color)
{
	if (!gdk_color_parse(string, color))
	{
		g_print("got unknown color '%s':  ", string);
	 	g_print("color should probably be in this format: #RRGGBB; "
				"defaulting to #168AA1\n");
		color->red   = 0x1600;
		color->green = 0x8a00;
		color->blue  = 0xa100;
	}
}


	

/*
 * Fit from a double to an unsigned 8 bit integer (0..255) 
 * for printing into wharf.
 * eg:      R        G        B 
 *      "0.501132 0.060147 1.000000"  becomes  
 *          7F       0F       FF
 * which gets printed as #7F0FFF
 * Without the .2 in %.2X * 0AD30B  will come out as AD3B
 * BTW, %X prints uppercase hex.  
 */

void  ascp_get_color(GtkWidget *widget,  gchar *int2text)
{

	if(int2text)
	{
		GdkColor *color;

		sprintf(int2text, "#%.2X%.2X%.2X", 
					(guint)(Color[0]*255.0), (guint)(Color[1]*255), (guint)(Color[2]*255));
		/*
		g_print("#%.2X%.2X%.2X\n", 
					(guint)(Color[0]*255.0), (guint)(Color[1]*255), (guint)(Color[2]*255));
					*/
		color = gtk_object_get_data(GTK_OBJECT(GlobalWidget), "color");
		if (color == NULL)
			return;
		ascp_char_to_color(int2text, color);

	}
	//else g_print("it was rather NULL\n");

	/* 
	 * We've popped down the ColorselDlg, so it's okay to 
	 * make a new one. More than one popped up ColorselDlg 
	 * would confuse everyone here at * ascp (r) inc. 
	 * Besides your desktop could get cluttered.
	 * See ascp_area_event(...) in gui.c
	 */
	ColorselDlg = NULL;
}
	 

		
/* Drawingarea event handler */
		
gint ascp_area_event (GtkWidget *widget, GdkEvent *event, gpointer *client_data)
{
	GdkColor *color;
	gdouble colors[3];


	/* create a colorwell object */
	colorwell *well = (colorwell *) g_malloc0(sizeof(colorwell));

	/* Check if we've received a button pressed event */
	if (event->type == GDK_BUTTON_PRESS  && ColorselDlg == NULL)
	{
		/*
		 * Yes, we have an event and there's no ColorselDlg yet! 
		 * Create color selection dialog
		 */
		ColorselDlg = gtk_color_selection_dialog_new("Select well colour");


		/*
		 * Get the GtkColorSelection widget.
	 	 */
		well->colorsel = (GtkColorSelection *) 
			GTK_COLOR_SELECTION_DIALOG(ColorselDlg)->colorsel;
		GlobalWidget = well->drawingarea = widget;

		/* 
		 * Connect to the "color_changed" signal, 
	 	 * set the client-data to the well structure which contains
	 	 * the drawingarea we're interested in, as well as the 
	 	 * related  colorsel widget.
	 	 */
		gtk_signal_connect(GTK_OBJECT(well->colorsel), "color_changed",
			(GtkSignalFunc)ascp_color_changed_cb, (gpointer) well);

		/* Connect its button labeled "OK"  to  get the color 
	 	 * for the widget and to pop down the color selection dialog.
	 	 * Note that the client_data is actually the string into which 
	 	 * we'll print the hex format of the color
	 	 */
		gtk_signal_connect(GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(ColorselDlg)->ok_button),
			"clicked", GTK_SIGNAL_FUNC(ascp_get_color), (gpointer *) client_data);
		gtk_signal_connect_object(
			GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(ColorselDlg)->ok_button),
			"clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(ColorselDlg));


		/*
		 *  Its cancel button simply pops down the dialog
	 	 */
		gtk_signal_connect(
			GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(ColorselDlg)->cancel_button),
			"clicked", GTK_SIGNAL_FUNC(ascp_get_color), NULL);
		gtk_signal_connect_object(
			GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(ColorselDlg)->cancel_button),
			"clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(ColorselDlg));
	

		gtk_widget_hide_all(GTK_WIDGET(GTK_COLOR_SELECTION_DIALOG(ColorselDlg)->help_button));
	


		color = gtk_object_get_data(GTK_OBJECT(widget), "color");
		
		/*
		g_print("in here the colours are: %f, %f, %f\n",
		color->red/65535.0,
		color->green/65535.0,
		color->blue/65535.0);
		*/

		colors[0] = color->red/65535.0;
		colors[1] = color->green/65535.0;
		colors[2] = color->blue/65535.0;
		gtk_color_selection_set_color(GTK_COLOR_SELECTION(
			GTK_COLOR_SELECTION_DIALOG(ColorselDlg)->colorsel), colors);
			
		gtk_widget_show(ColorselDlg);
		
		/* 
		 * The dialog has been handled, all's well
		 */
		return TRUE;
	}	
	
	/* 
	 * The dialog has not been handled; we nicely refuse 
	 * subsequent drawingarea demands to popup a dialog. 
	 * Here's a typical exchange:
	 * 
	 * drawingarea: "Can I have a dialog please?"
	 * dialog_spokesperson: "Nah!  'cause there's already one up, see?"
	 * drawingarea: "Ahh, I see. Well, thanks for nothing"
	 * 
	 * The drawingarea really doesn't see, but what is it to do?
	 */
	return FALSE;
}



