/*
 * Decompiled with CFR 0.152.
 */
package org.graalvm.visualvm.lib.jfluid.heap;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import org.graalvm.visualvm.lib.jfluid.heap.CacheDirectory;

abstract class AbstractLongMap {
    private final int VALUE_SIZE;
    final int ENTRY_SIZE;
    long fileSize;
    private long keys;
    final int KEY_SIZE;
    final int ID_SIZE;
    final int FOFFSET_SIZE;
    Data dumpBuffer;
    CacheDirectory cacheDirectory;

    AbstractLongMap(int size, int idSize, int foffsetSize, int valueSize, CacheDirectory cacheDir) throws FileNotFoundException, IOException {
        assert (idSize == 4 || idSize == 8);
        assert (foffsetSize == 4 || foffsetSize == 8);
        this.keys = (long)size * 4L / 3L;
        this.ID_SIZE = idSize;
        this.FOFFSET_SIZE = foffsetSize;
        this.KEY_SIZE = this.ID_SIZE;
        this.VALUE_SIZE = valueSize;
        this.ENTRY_SIZE = this.KEY_SIZE + this.VALUE_SIZE;
        this.fileSize = this.keys * (long)this.ENTRY_SIZE;
        this.cacheDirectory = cacheDir;
        this.dumpBuffer = cacheDir.createDumpBuffer(this.fileSize, this.ENTRY_SIZE);
    }

    protected void finalize() throws Throwable {
        if (this.cacheDirectory.isTemporary()) {
            this.dumpBuffer.deleteFile();
        }
        super.finalize();
    }

    Entry get(long key) {
        long index = this.getIndex(key);
        long mapKey;
        while ((mapKey = this.getID(index)) != key) {
            if (mapKey == 0L) {
                return null;
            }
            index = this.getNextIndex(index);
        }
        return this.createEntry(index);
    }

    Entry put(long key, long value) {
        long index = this.getIndex(key);
        while (true) {
            long mapKey;
            if ((mapKey = this.getID(index)) == 0L) {
                this.putID(index, key);
                return this.createEntry(index, value);
            }
            if (mapKey == key) {
                return this.createEntry(index);
            }
            index = this.getNextIndex(index);
        }
    }

    static Data getDumpBuffer(File f, RandomAccessFile file, int entrySize) throws IOException {
        long length = file.length();
        try {
            if (length > Integer.MAX_VALUE) {
                return new LongMemoryMappedData(f, file, length, entrySize);
            }
            return new MemoryMappedData(f, file, length);
        }
        catch (IOException ex) {
            if (ex.getCause() instanceof OutOfMemoryError) {
                return new FileData(f, file, length, entrySize);
            }
            throw ex;
        }
    }

    long getID(long index) {
        if (this.ID_SIZE == 4) {
            return (long)this.dumpBuffer.getInt(index) & 0xFFFFFFFFL;
        }
        return this.dumpBuffer.getLong(index);
    }

    void putID(long index, long key) {
        if (this.ID_SIZE == 4) {
            this.dumpBuffer.putInt(index, (int)key);
        } else {
            this.dumpBuffer.putLong(index, key);
        }
    }

    long getFoffset(long index) {
        if (this.FOFFSET_SIZE == 4) {
            return this.dumpBuffer.getInt(index);
        }
        return this.dumpBuffer.getLong(index);
    }

    void putFoffset(long index, long key) {
        if (this.FOFFSET_SIZE == 4) {
            this.dumpBuffer.putInt(index, (int)key);
        } else {
            this.dumpBuffer.putLong(index, key);
        }
    }

    void writeToStream(DataOutputStream out) throws IOException {
        out.writeLong(this.keys);
        out.writeInt(this.ID_SIZE);
        out.writeInt(this.FOFFSET_SIZE);
        out.writeInt(this.VALUE_SIZE);
        this.dumpBuffer.writeToStream(out);
    }

    AbstractLongMap(DataInputStream dis, CacheDirectory cacheDir) throws IOException {
        this.keys = dis.readLong();
        this.ID_SIZE = dis.readInt();
        this.FOFFSET_SIZE = dis.readInt();
        this.VALUE_SIZE = dis.readInt();
        this.KEY_SIZE = this.ID_SIZE;
        this.ENTRY_SIZE = this.KEY_SIZE + this.VALUE_SIZE;
        this.fileSize = this.keys * (long)this.ENTRY_SIZE;
        this.dumpBuffer = Data.readFromStream(dis, cacheDir, this.ENTRY_SIZE);
        this.cacheDirectory = cacheDir;
    }

    private long getIndex(long key) {
        long hash = key & Long.MAX_VALUE;
        return hash % this.keys * (long)this.ENTRY_SIZE;
    }

    private long getNextIndex(long index) {
        if ((index += (long)this.ENTRY_SIZE) >= this.fileSize) {
            index = 0L;
        }
        return index;
    }

    private static boolean isLinux() {
        String osName = System.getProperty("os.name");
        return osName.endsWith("Linux");
    }

    private static boolean isMacOS() {
        String osName = System.getProperty("os.name");
        return "Mac OS X".equals(osName);
    }

    private static boolean isAarch64() {
        String osArch = System.getProperty("os.arch");
        return "aarch64".equals(osArch);
    }

    abstract Entry createEntry(long var1);

    abstract Entry createEntry(long var1, long var3);

    private static class LongMemoryMappedData
    extends AbstractData {
        private static int BUFFER_SIZE_BITS = 30;
        private static long BUFFER_SIZE = 1L << BUFFER_SIZE_BITS;
        private static int BUFFER_SIZE_MASK = (int)(BUFFER_SIZE - 1L);
        private static int BUFFER_EXT = 32768;
        private MappedByteBuffer[] dumpBuffer;
        private final int entrySize;

        LongMemoryMappedData(File f, RandomAccessFile file, long length, int entry) throws IOException {
            super(f);
            this.dumpBuffer = LongMemoryMappedData.createBuffers(file, length);
            this.entrySize = entry;
        }

        @Override
        public byte getByte(long index) {
            return this.dumpBuffer[this.getBufferIndex(index)].get(this.getBufferOffset(index));
        }

        @Override
        public int getInt(long index) {
            return this.dumpBuffer[this.getBufferIndex(index)].getInt(this.getBufferOffset(index));
        }

        @Override
        public long getLong(long index) {
            return this.dumpBuffer[this.getBufferIndex(index)].getLong(this.getBufferOffset(index));
        }

        @Override
        public void putByte(long index, byte data) {
            this.dumpBuffer[this.getBufferIndex(index)].put(this.getBufferOffset(index), data);
        }

        @Override
        public void putInt(long index, int data) {
            this.dumpBuffer[this.getBufferIndex(index)].putInt(this.getBufferOffset(index), data);
        }

        @Override
        public void putLong(long index, long data) {
            this.dumpBuffer[this.getBufferIndex(index)].putLong(this.getBufferOffset(index), data);
        }

        private int getBufferIndex(long index) {
            return (int)(index >> BUFFER_SIZE_BITS);
        }

        private int getBufferOffset(long index) {
            return (int)(index & (long)BUFFER_SIZE_MASK);
        }

        @Override
        public void force() throws IOException {
            if (MemoryMappedData.MAP_MODE == FileChannel.MapMode.PRIVATE) {
                File newBufferFile = new File(this.bufferFile.getAbsolutePath() + ".new");
                long length = this.bufferFile.length();
                try (FileChannel channel = new FileOutputStream(newBufferFile).getChannel();){
                    int offset_start = 0;
                    for (int i = 0; i < this.dumpBuffer.length; ++i) {
                        MappedByteBuffer buf = this.dumpBuffer[i];
                        long offset_end = (long)(i + 1) * BUFFER_SIZE / (long)this.entrySize * (long)this.entrySize + (long)this.entrySize;
                        if (offset_end > length) {
                            offset_end = length;
                        }
                        buf.limit((int)(offset_end - (long)i * BUFFER_SIZE));
                        buf.position(offset_start);
                        channel.write(buf);
                        offset_start = (int)(offset_end - (long)(i + 1) * BUFFER_SIZE);
                    }
                }
                this.dumpBuffer = null;
                this.bufferFile.delete();
                newBufferFile.renameTo(this.bufferFile);
                this.dumpBuffer = LongMemoryMappedData.createBuffers(new RandomAccessFile(this.bufferFile, "rw"), length);
            } else {
                for (MappedByteBuffer buf : this.dumpBuffer) {
                    buf.force();
                }
            }
        }

        private static MappedByteBuffer[] createBuffers(RandomAccessFile file, long length) throws IOException {
            MappedByteBuffer[] dumpBuffer;
            try (FileChannel channel = file.getChannel();){
                dumpBuffer = new MappedByteBuffer[(int)((length + BUFFER_SIZE - 1L) / BUFFER_SIZE)];
                for (int i = 0; i < dumpBuffer.length; ++i) {
                    long position = (long)i * BUFFER_SIZE;
                    long size = Math.min(BUFFER_SIZE + (long)BUFFER_EXT, length - position);
                    dumpBuffer[i] = channel.map(MemoryMappedData.MAP_MODE, position, size);
                }
            }
            file.close();
            return dumpBuffer;
        }
    }

    private static class MemoryMappedData
    extends AbstractData {
        private static final FileChannel.MapMode MAP_MODE = MemoryMappedData.computeMapMode();
        MappedByteBuffer buf;

        MemoryMappedData(File f, RandomAccessFile file, long length) throws IOException {
            super(f);
            this.buf = MemoryMappedData.createBuffer(file, length);
        }

        @Override
        public byte getByte(long index) {
            return this.buf.get((int)index);
        }

        @Override
        public int getInt(long index) {
            return this.buf.getInt((int)index);
        }

        @Override
        public long getLong(long index) {
            return this.buf.getLong((int)index);
        }

        @Override
        public void putByte(long index, byte data) {
            this.buf.put((int)index, data);
        }

        @Override
        public void putInt(long index, int data) {
            this.buf.putInt((int)index, data);
        }

        @Override
        public void putLong(long index, long data) {
            this.buf.putLong((int)index, data);
        }

        @Override
        public void force() throws IOException {
            if (MAP_MODE == FileChannel.MapMode.PRIVATE) {
                File newBufferFile = new File(this.bufferFile.getAbsolutePath() + ".new");
                int length = this.buf.capacity();
                new FileOutputStream(newBufferFile).getChannel().write(this.buf);
                this.buf = null;
                this.bufferFile.delete();
                newBufferFile.renameTo(this.bufferFile);
                this.buf = MemoryMappedData.createBuffer(new RandomAccessFile(this.bufferFile, "rw"), length);
            } else {
                this.buf.force();
            }
        }

        private static MappedByteBuffer createBuffer(RandomAccessFile file, long length) throws IOException {
            try (FileChannel channel = file.getChannel();){
                MappedByteBuffer mappedByteBuffer = channel.map(MAP_MODE, 0L, length);
                return mappedByteBuffer;
            }
        }

        private static FileChannel.MapMode computeMapMode() {
            String mode = System.getProperty("org.graalvm.visualvm.lib.jfluid.heap.mapmode");
            if (mode != null) {
                if ("private".equals(mode)) {
                    return FileChannel.MapMode.PRIVATE;
                }
                return FileChannel.MapMode.READ_WRITE;
            }
            if (AbstractLongMap.isLinux()) {
                return FileChannel.MapMode.PRIVATE;
            }
            if (AbstractLongMap.isMacOS() && AbstractLongMap.isAarch64()) {
                return FileChannel.MapMode.PRIVATE;
            }
            return FileChannel.MapMode.READ_WRITE;
        }
    }

    private static class FileData
    extends AbstractData {
        RandomAccessFile file;
        byte[] buf;
        boolean bufferModified;
        long offset;
        int entrySize;
        long fileSize;
        static final int BUFFER_SIZE = 128;

        FileData(File fl, RandomAccessFile f, long length, int entry) throws IOException {
            super(fl);
            this.file = f;
            this.fileSize = length;
            this.entrySize = entry;
            this.buf = new byte[this.entrySize * 128];
        }

        @Override
        public synchronized byte getByte(long index) {
            int i = this.loadBufferIfNeeded(index);
            return this.buf[i];
        }

        @Override
        public synchronized int getInt(long index) {
            int i = this.loadBufferIfNeeded(index);
            int ch1 = this.buf[i++] & 0xFF;
            int ch2 = this.buf[i++] & 0xFF;
            int ch3 = this.buf[i++] & 0xFF;
            int ch4 = this.buf[i] & 0xFF;
            return (ch1 << 24) + (ch2 << 16) + (ch3 << 8) + (ch4 << 0);
        }

        @Override
        public synchronized long getLong(long index) {
            int i = this.loadBufferIfNeeded(index);
            return ((long)this.buf[i++] << 56) + ((long)(this.buf[i++] & 0xFF) << 48) + ((long)(this.buf[i++] & 0xFF) << 40) + ((long)(this.buf[i++] & 0xFF) << 32) + ((long)(this.buf[i++] & 0xFF) << 24) + (long)((this.buf[i++] & 0xFF) << 16) + (long)((this.buf[i++] & 0xFF) << 8) + (long)((this.buf[i++] & 0xFF) << 0);
        }

        @Override
        public synchronized void putByte(long index, byte data) {
            int i = this.loadBufferIfNeeded(index);
            this.buf[i] = data;
            this.bufferModified = true;
        }

        @Override
        public synchronized void putInt(long index, int data) {
            int i = this.loadBufferIfNeeded(index);
            this.buf[i++] = (byte)(data >>> 24);
            this.buf[i++] = (byte)(data >>> 16);
            this.buf[i++] = (byte)(data >>> 8);
            this.buf[i++] = (byte)(data >>> 0);
            this.bufferModified = true;
        }

        @Override
        public synchronized void putLong(long index, long data) {
            int i = this.loadBufferIfNeeded(index);
            this.buf[i++] = (byte)(data >>> 56);
            this.buf[i++] = (byte)(data >>> 48);
            this.buf[i++] = (byte)(data >>> 40);
            this.buf[i++] = (byte)(data >>> 32);
            this.buf[i++] = (byte)(data >>> 24);
            this.buf[i++] = (byte)(data >>> 16);
            this.buf[i++] = (byte)(data >>> 8);
            this.buf[i++] = (byte)(data >>> 0);
            this.bufferModified = true;
        }

        private int loadBufferIfNeeded(long index) {
            int i = (int)(index % (long)(this.entrySize * 128));
            long newOffset = index - (long)i;
            if (this.offset != newOffset) {
                try {
                    this.flush();
                    this.file.seek(newOffset);
                    this.file.readFully(this.buf, 0, this.getBufferSize(newOffset));
                }
                catch (IOException ex) {
                    ex.printStackTrace();
                }
                this.offset = newOffset;
            }
            return i;
        }

        private int getBufferSize(long off) {
            int size = this.buf.length;
            if (this.fileSize - off < (long)this.buf.length) {
                size = (int)(this.fileSize - off);
            }
            return size;
        }

        private void flush() throws IOException {
            if (this.bufferModified) {
                this.file.seek(this.offset);
                this.file.write(this.buf, 0, this.getBufferSize(this.offset));
                this.bufferModified = false;
            }
        }

        @Override
        public void force() throws IOException {
            this.flush();
        }
    }

    private static abstract class AbstractData
    implements Data {
        File bufferFile;

        private AbstractData(File file) {
            this.bufferFile = file;
        }

        @Override
        public void writeToStream(DataOutputStream out) throws IOException {
            out.writeUTF(this.bufferFile.getAbsolutePath());
            this.force();
        }

        @Override
        public void deleteFile() {
            this.bufferFile.delete();
        }
    }

    static interface Data {
        public static Data readFromStream(DataInputStream dis, CacheDirectory cacheDir, int entrySize) throws IOException {
            File tempFile = cacheDir.getCacheFile(dis.readUTF());
            try (RandomAccessFile file = new RandomAccessFile(tempFile, "rw");){
                Data data = AbstractLongMap.getDumpBuffer(tempFile, file, entrySize);
                return data;
            }
        }

        public byte getByte(long var1);

        public int getInt(long var1);

        public long getLong(long var1);

        public void putByte(long var1, byte var3);

        public void putInt(long var1, int var3);

        public void putLong(long var1, long var3);

        public void force() throws IOException;

        public void writeToStream(DataOutputStream var1) throws IOException;

        public void deleteFile();
    }

    abstract class Entry {
        Entry() {
        }
    }
}

