"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerParaRoute = registerParaRoute;
var _configSchema = require("@osd/config-schema");
var _notebooks = require("../../../common/constants/notebooks");
var _observability_saved_object_attributes = require("../../../common/types/observability_saved_object_attributes");
var _notebooks2 = require("../../adaptors/notebooks");
var _saved_objects_paragraphs_router = require("../../adaptors/notebooks/saved_objects_paragraphs_router");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function registerParaRoute(router) {
  /* --> Updates the input content in a paragraph
   * --> Runs the paragraph
   * --> Fetches the updated Paragraph (with new input content)
   */
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/paragraph/update/run/`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphId: _configSchema.schema.string(),
        paragraphInput: _configSchema.schema.string(),
        paragraphType: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const runResponse = await _notebooks2.BACKEND.updateRunFetchParagraph(context.observability_plugin.observabilityClient, request, _notebooks.wreckOptions);
      return response.ok({
        body: runResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  /* --> Updates the input content in a paragraph
   * --> Fetches the updated Paragraph (with new input content)
   */
  router.put({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/paragraph/`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphId: _configSchema.schema.string(),
        paragraphInput: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const saveResponse = await _notebooks2.BACKEND.updateFetchParagraph(opensearchNotebooksClient, request.body, _notebooks.wreckOptions);
      return response.ok({
        body: saveResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  /* --> Adds a new paragraph
   * --> Fetches the added Paragraph
   */
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/paragraph/`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphIndex: _configSchema.schema.number(),
        paragraphInput: _configSchema.schema.string(),
        inputType: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const addResponse = await _notebooks2.BACKEND.addFetchNewParagraph(opensearchNotebooksClient, request.body, _notebooks.wreckOptions);
      return response.ok({
        body: addResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  /* --> Update paragraphs in backend with paragraphs passed in
   * --> Fetches the added Paragraph
   */
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/set_paragraphs/`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphs: _configSchema.schema.arrayOf(_configSchema.schema.object({
          output: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.object({}, {
            unknowns: 'allow'
          }))),
          input: _configSchema.schema.object({
            inputText: _configSchema.schema.string(),
            inputType: _configSchema.schema.string()
          }),
          dateCreated: _configSchema.schema.string(),
          dateModified: _configSchema.schema.string(),
          id: _configSchema.schema.string()
        }))
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const updateNotebook = {
        paragraphs: request.body.paragraphs,
        dateModified: new Date().toISOString()
      };
      const updateResponse = await _notebooks2.BACKEND.updateNote(opensearchNotebooksClient, request.body.noteId, updateNotebook);
      return response.ok({
        body: updateResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  /* --> Deletes a paragraph
   * --> Fetches the all other Paragraphs as a list
   * --> Deletes all paragraphs if paragraphId is not provided
   */
  router.delete({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/paragraph`,
    validate: {
      query: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    const params = {
      noteId: request.query.noteId,
      paragraphId: request.query.paragraphId
    };
    try {
      const deleteResponse = await _notebooks2.BACKEND.deleteFetchParagraphs(opensearchNotebooksClient, params, _notebooks.wreckOptions);
      return response.ok({
        body: deleteResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  /* --> Clears output for all the paragraphs
   * --> Fetches the all Paragraphs as a list (with cleared outputs)
   */
  router.put({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/paragraph/clearall/`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const clearParaResponse = await _notebooks2.BACKEND.clearAllFetchParagraphs(opensearchNotebooksClient, request.body, _notebooks.wreckOptions);
      return response.ok({
        body: clearParaResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/paragraph/`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphIndex: _configSchema.schema.number(),
        paragraphInput: _configSchema.schema.string(),
        inputType: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const saveResponse = await (0, _saved_objects_paragraphs_router.createParagraphs)(request.body, context.core.savedObjects.client);
      return response.ok({
        body: saveResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.put({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/paragraph/clearall/`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const clearParaResponse = await (0, _saved_objects_paragraphs_router.clearParagraphs)(request.body, context.core.savedObjects.client);
      return response.ok({
        body: clearParaResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.delete({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/paragraph`,
    validate: {
      query: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const params = {
      noteId: request.query.noteId,
      paragraphId: request.query.paragraphId
    };
    try {
      const deleteResponse = await (0, _saved_objects_paragraphs_router.deleteParagraphs)(params, context.core.savedObjects.client);
      return response.ok({
        body: deleteResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/paragraph/update/run/`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphId: _configSchema.schema.string(),
        paragraphInput: _configSchema.schema.string(),
        paragraphType: _configSchema.schema.string(),
        dataSourceMDSId: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        })),
        dataSourceMDSLabel: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        }))
      })
    }
  }, async (context, request, response) => {
    try {
      const runResponse = await (0, _saved_objects_paragraphs_router.updateRunFetchParagraph)(request.body, context.core.savedObjects.client);
      return response.ok({
        body: runResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.put({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/paragraph/`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphId: _configSchema.schema.string(),
        paragraphInput: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const saveResponse = await (0, _saved_objects_paragraphs_router.updateFetchParagraph)(request.body, context.core.savedObjects.client);
      return response.ok({
        body: saveResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/set_paragraphs/`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphs: _configSchema.schema.arrayOf(_configSchema.schema.object({
          output: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.object({}, {
            unknowns: 'allow'
          }))),
          input: _configSchema.schema.object({
            inputText: _configSchema.schema.string(),
            inputType: _configSchema.schema.string()
          }),
          dateCreated: _configSchema.schema.string(),
          dateModified: _configSchema.schema.string(),
          id: _configSchema.schema.string(),
          dataSourceMDSId: _configSchema.schema.maybe(_configSchema.schema.string({
            defaultValue: ''
          })),
          dataSourceMDSLabel: _configSchema.schema.maybe(_configSchema.schema.string({
            defaultValue: ''
          }))
        }))
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.core.savedObjects.client;
    try {
      const updateNotebook = {
        paragraphs: request.body.paragraphs,
        dateModified: new Date().toISOString()
      };
      const updateResponse = await opensearchNotebooksClient.update(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, request.body.noteId, updateNotebook);
      return response.ok({
        body: updateResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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