#!/usr/bin/env python3
#
# Copyright (c) 2025 The NetBSD Foundation, Inc.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
# ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
# TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
# PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
# BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
#

"""commitlint is a hook to check commit messages for patterns.

Usage:
  [netbsd_hooks.commitlint.reject]
  desc.wip = re:^WIP(: .*)?$
  desc.fixup = re:^fixup!
  desc.squash = re:^squash!

  [netbsd_hooks.commitlint.require]
  desc.issue = re:\bissue([0-9]+)\b

"""

import re

from typing import Optional

from mercurial.i18n import _
from mercurial import (
    error,
    localrepo,
    pycompat,
    registrar,
    ui,
)

configtable = {}
configitem = registrar.configitem(configtable)

# [commitlint.require]
# desc.foo = ...
configitem(
    b'netbsd_hooks.commitlint.require',
    b'desc\\.[a-zA-Z][a-zA-Z0-9_-]*',
    default=None,
    generic=True,
)
# [commitlint.require]
# desc.bar = ...
configitem(
    b'netbsd_hooks.commitlint.reject',
    b'desc\\.[a-zA-Z][a-zA-Z0-9_-]*',
    default=None,
    generic=True,
)


def _lintpatterns(ui, key):
    if not ui.has_section(key):
        return []
    assert all(
        field.startswith(b'desc.')
        for field, pat in ui.configitems(key)
    )
    return [(pat, re.compile(pat)) for field, pat in ui.configitems(key)]


def _lint(ui, ctx, require, reject):
    desc = ctx.description()
    bad = False
    for pat, cpat in require:
        if not cpat.search(desc):
            ui.error(
                _(b'changeset %s desc fails to match required pattern: %s\n') %
                (ctx, pat),
            )
            bad = True
    for pat, cpat in reject:
        if cpat.search(desc):
            ui.error(
                _(b'changeset %s desc matches forbidden pattern: %s\n') %
                (ctx, pat),
            )
            bad = True
    return bad


def hook(
    ui: ui.ui,
    repo: localrepo.localrepository,
    hooktype: bytes,
    node: Optional[bytes] = None,
    **kwargs,
):
    # XXX Why pretxnchangegroup and not pretxnclose?
    if hooktype not in (b'pretxnchangegroup', b'pretxnclose-phase'):
        raise error.Abort(
            _(b'Unsupported hook type %r') % pycompat.bytestr(hooktype)
        )

    require = _lintpatterns(ui, b'netbsd_hooks.commitlint.require')
    reject = _lintpatterns(ui, b'netbsd_hooks.commitlint.reject')

    if hooktype == b'pretxnclose-phase':
        bad = _lint(ui, repo[node], require, reject)
    else:
        bad = False
        for rev in range(repo[node].rev(), len(repo)):
            if _lint(ui, repo[rev], require, reject):
                bad = True
    if bad:
        raise error.Abort(_(b'changeset description must be amended'))
