#!/usr/bin/env python3
#
# Copyright (c) 2025 The NetBSD Foundation, Inc.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
# ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
# TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
# PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
# BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
#

"""reject_hgmeta is a hook to forbid changes to .hg or .hg/* files.

This is a stop-gap measure to work around:

Mercurial issue10039: hg allows changesets affecting files called .hg
https://foss.heptapod.net/mercurial/mercurial-devel/-/issues/10039

Usage:
  [hooks]
  pretxnchangegroup.reject_hgmeta = python:....reject_hgmeta.hook
"""

from typing import Optional

from mercurial.i18n import _
from mercurial import (
    error,
    localrepo,
    match,
    pycompat,
    ui,
)


def hook(
    ui: ui.ui,
    repo: localrepo.localrepository,
    hooktype: bytes,
    node: Optional[bytes] = None,
    **kwargs,
) -> None:
    if hooktype != b'pretxnchangegroup':
        raise error.Abort(
            _(b'Unsupported hook type %r') % pycompat.bytestr(hooktype)
        )

    ctx = repo.unfiltered()[node]
    bad = set()
    for revno in repo.changelog.revs(start=ctx.rev()):
        rev = repo[revno]
        matcher = match.match(repo.root, b'', [
            # Note: The (?i) pattern, to enable case-insensitive
            # matching, must come first, or else it will cryptically
            # fail with `remote: invalid pattern'.
            b're:(?i).*/$',     # (?i) pattern must come first
            b're:.*//.*',
            b're:^(.*/)*.hg(\\.)?(/.*)*$',
            b're:^(.*/)*.hgignore/.*',
            b're:^(.*/)*.hgtags/.*',
            b're:^(.*/)HG(8B6C)?~[0-9](/.*)*$',  # Windows shortname
        ])
        for path in rev.manifest().walk(matcher):
            if path in bad:
                continue
            bad.add(path)
            ui.error(_(b'changeset %s has forbidden file %r\n') % (rev, path))
    if bad:
        raise error.Abort(_(b'changesets have forbidden files'))
