/* $NetBSD$ */

/*
 * Copyright (c) 2003 Dennis I. Chernoivanov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#if defined(_USE_HD44780)

#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>

#include <dev/ic/hd44780reg.h>
#include <dev/ic/hd44780_subr.h>

#include "paneld.h"

static int fd;
static int xpos, ypos;
static struct hd44780_info hi;

static int
hd44780_open(const char *node)
{
	if ( (fd = open(node, O_RDWR, 0)) == -1)
		return E_IO;
	if (ioctl(fd, HLCD_GET_INFO, &hi) == -1)
		return E_IO;
	return E_OK;
}

static int
hd44780_close(void)
{
	close(fd);
	fd = 0;
	return E_OK;
}

static void
hd44780_devcap(struct lcd_devcap *cap)
{
	cap->capabilities = DEV_CAP_AUTOREPEAT;
	if (cap->resist <= 0)
		cap->resist = 2000000; /* 2 sec */
	if (cap->repeat <= 0)
		cap->repeat = 400000;  /* .4 sec */
	if (cap->keyrate <= 0)
		cap->keyrate = 200000; /* .2 sec */
}

static int
hd44780_getsize(int *rows, int *lines)
{
	util_trace(LOG_DEBUG, "getsize: %d, %d", hi.phys_rows, hi.lines);

	*lines = hi.lines;
	*rows  = hi.phys_rows;
	return E_OK;
}

static int
hd44780_clear(void)
{
	if (ioctl(fd, HLCD_CLEAR, 0) == -1)
		return E_IO;
	return E_OK;
}

static int
hd44780_putchr(int dat)
{
	struct hd44780_io io;

	io.dat = dat;
	io.len = 1;

	if (ioctl(fd, HLCD_PUTC, &io) == -1)
		return E_IO;
	if (ioctl(fd, HLCD_CURSOR_LEFT, 0) == -1)
		return E_IO;
	return E_OK;
}

static int
hd44780_putstr(const char *buf, int len)
{
	int width;
	struct hd44780_io io;

	width = (hi.virt_rows - xpos);
	memset(io.buf, ' ', sizeof(io.buf));
	io.len = (len > width) ? width : len;
	memcpy(io.buf, buf, io.len);

	util_trace(LOG_DEBUG,
			"hd44780_putstr: '%s' [%d - %d]", buf, io.len, len);

	io.dat = ypos * hi.virt_rows + xpos;
	if (ioctl(fd, HLCD_WRITE, &io) == -1)
		return E_IO;
	return E_OK;
}

static int
hd44780_setpos(int x, int y)
{
	struct hd44780_io io;

	util_trace(LOG_DEBUG, "hd44780_setpos: %d, %d", x, y);

	switch(y) {
	case 0:
		io.dat = HD_ROW1_ADDR;
		break;
	case 1:
		io.dat = HD_ROW2_ADDR;
		break;
	default:
		return E_IO;
	}

	if (x >= hi.phys_rows)
		return E_IO;
	io.dat += x;
	util_trace(LOG_DEBUG, "real move to: %d", io.dat);
	if (ioctl(fd, HLCD_SET_CURSOR_POS, &io) == -1)
		return E_IO;

	xpos = x;
	ypos = y;

	return E_OK;
}

static int
hd44780_cursor(int on, int blink)
{
	struct hd44780_dispctl ctl;

	ctl.display_on = 1;
	ctl.blink_on = blink;
	ctl.cursor_on = on;

	if (ioctl(fd, HLCD_DISPCTL, &ctl) == -1)
		return E_IO;
	return E_OK;
}

static int
hd44780_getchr(int *c)
{
	u_int8_t buf;
	if (read(fd, &buf, sizeof(buf)) != sizeof(buf))
		return E_IO;
	*c = buf;
	return E_OK;
}

struct lcd_dev hd44780_ops = {
	hd44780_open,
	hd44780_close,
	hd44780_devcap,
	hd44780_getsize,
	hd44780_clear,
	hd44780_putchr,
	hd44780_putstr,
	hd44780_setpos,
	hd44780_cursor,
	hd44780_getchr
};

struct lcd_dev*
get_hd44780(void)
{
	return &hd44780_ops;
}

#endif
