/* $NetBSD$ */

/*
 * Copyright (c) 2003 Dennis I. Chernoivanov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <string.h>

#include "paneld.h"

struct lcd_dev *dev;
struct lcd_window *window;

static struct lcd_edit*
get_editor(void);

struct edit_line {
	char *buf;

	char *text;
	char *visible;		/* pointer to visible area of text */

	int cursor;		/* cursor position */
	int length;		/* line length */

	int x, y;		/* relative position */
};

struct window_rect {
	int tlen;		/* title length */
	char *title;		/* title */

	int width;
	int height;

	int edit_offset;
	int focus_offset;

	int vsize;		/* no of lines for menu items */
	int vcursor;		/* highlighted item */

	struct menu *mlist;
	struct menu *mtopmost;

	struct edit_line *line;
};

static struct window_rect *rect;

static int
wnd_clear(void)
{
	return dev->clear();
}

static int
wnd_set_caption(const char *s)
{
	int len = strlen(s);

	if (len > rect->tlen)
		len = rect->tlen;

	memset(rect->line->buf, ' ', rect->tlen);
	memcpy(rect->line->buf, s, len);

	dev->cursor(0, 0);
	if (dev->setpos(0, 0) == E_OK)
		return dev->putstr(rect->line->buf, rect->tlen);

	return E_IO;
}

static int
wnd_set_text(char *lines[])
{
	int i;

	if (dev->clear() != E_OK)
		return E_IO;

	dev->cursor(0, 0);
	for (i = 0; (i < rect->height) && (lines[i] != NULL); i++) {
		int len;
		if (dev->setpos(0, i) != E_OK)
			return E_IO;

		len = strlen(lines[i]);
		if (len > rect->width)
			len = rect->width;
		if (dev->putstr(lines[i], len) != E_OK)
			return E_IO;
	}

	return E_OK;
}

static int
wnd_1x1_open(struct lcd_dev *dev)
{
	rect->line = (struct edit_line*)cf_malloc(sizeof(struct edit_line));
	memset(rect->line, 0, sizeof(struct edit_line));

	if (dev->getsize(&rect->width, &rect->height) != E_OK)
		return E_IO;

	rect->line->buf = (char*)cf_malloc(rect->width);
	rect->tlen = rect->width / 3;

	rect->edit_offset = rect->tlen + 2;
	rect->focus_offset = rect->edit_offset;
	rect->line->x = rect->edit_offset;
	rect->line->y = 0;
	rect->line->length = rect->width - rect->line->x;

	return dev->clear();
}

static int
wnd_1x1_set_list(struct menu *m)
{
	dev->cursor(0, 0);
	return E_OK;
}

static int
wnd_1x1_set_focus(struct menu *m)
{
	int len;
	if (dev->setpos(rect->focus_offset, rect->line->y) != E_OK)
		return E_IO;

	len = strlen(m->nm);
	if (len > rect->line->length)
		len = rect->line->length;

	memset(rect->line->buf, ' ', rect->line->length);
	memcpy(rect->line->buf, m->nm, len);

	dev->cursor(0, 0);
	return dev->putstr(rect->line->buf, rect->line->length);
}

static int
wnd_1x2_set_focus(struct menu *m)
{
	int len;
	if (dev->setpos(0, rect->line->y) != E_OK)
		return E_IO;

	len = strlen(m->nm);
	if (len > (rect->width - rect->focus_offset))
		len = rect->width - rect->focus_offset;

	memset(rect->line->buf, ' ', rect->width);
	memcpy(rect->line->buf + rect->focus_offset, m->nm, len);

	dev->cursor(0, 0);
	return dev->putstr(rect->line->buf, rect->width);
}

static int
wnd_MxN_open(struct lcd_dev *dev)
{
	rect->line = (struct edit_line*)cf_malloc(sizeof(struct edit_line));
	memset(rect->line, 0, sizeof(struct edit_line));

	if (dev->getsize(&rect->width, &rect->height) != E_OK)
		return E_IO;

	rect->line->buf = (char*)cf_malloc(rect->width);
	rect->tlen = rect->width;

	rect->edit_offset = 0;
	rect->focus_offset = 2;

	rect->vsize = rect->height - 1;
	rect->vcursor = 0;
	rect->mlist = NULL;

	rect->line->x = rect->edit_offset;
	rect->line->y = (rect->vsize >> 1) + 1;
	rect->line->length = rect->width - rect->line->x;

	return dev->clear();
}

static int
wnd_MxN_draw_menu(struct menu *m, struct menu *focused)
{
	int i;
	struct menu *cur;
	int width = rect->width - rect->focus_offset - 2;

	for (i = 1, cur = m;
			(cur != NULL) && (i <= rect->vsize);
			cur = cur->next, i++) {
		int len;
		if (dev->setpos(rect->focus_offset, i) != E_OK)
			return E_IO;
		if ( (len = strlen(cur->nm)) > width)
			len = width;

		memset(rect->line->buf, ' ', rect->width);
		if ((rect->focus_offset > 0) && (cur == focused)) {
			rect->vcursor = i - 1;
			rect->line->buf[rect->focus_offset - 1] = '>';
		}
		memcpy(rect->line->buf + rect->focus_offset, cur->nm, len);
		if ((i == rect->vsize) && (cur->next != NULL))
			rect->line->buf[rect->width - 1] = 'v';
		if ((i == 1) && (rect->mlist != rect->mtopmost))
			rect->line->buf[rect->width - 1] = '^';
		if (dev->setpos(0, i) != E_OK)
			return E_IO;
		if (dev->putstr(rect->line->buf, rect->width) != E_OK)
			return E_IO;
	}

	return dev->setpos(0, rect->vcursor + 1);
}

static int
wnd_MxN_set_list(struct menu *m)
{
	rect->mlist = m;
	rect->mtopmost = m;
	dev->cursor(0, 0);
	return wnd_MxN_draw_menu(m, m);
}

static int
wnd_MxN_set_focus(struct menu *m)
{
	int i;
	struct menu *cur;
	int under_mtop = 0;

	if (rect->focus_offset > 0) {
		if (dev->setpos(0, rect->vcursor + 1) != E_OK)
			return E_IO;
		memset(rect->line->buf, ' ', rect->focus_offset);
		if (dev->putstr(rect->line->buf, rect->focus_offset) != E_OK)
			return E_IO;
	}

	for (i = 0, cur = rect->mlist;
			cur != NULL;
			cur = cur->next, i++) {
		if (cur >= rect->mtopmost)
			under_mtop++;
		if (cur == m)
			break;
	}

	if (cur == NULL)
		return E_IO;	/* no such menu??? */

	if (!under_mtop) {
		rect->mtopmost = cur;
		dev->cursor(0, 0);
		return wnd_MxN_draw_menu(cur, cur);
	}

	if (i < rect->vsize) {
		rect->vcursor = under_mtop - 1;
		if (rect->focus_offset > 0) {
			if (dev->setpos(rect->focus_offset - 1,
						rect->vcursor + 1) != E_OK)
				return E_IO;
			if (dev->putchr('>') != E_OK)
				return E_IO;
		}
		return E_OK;
	}

	for (i = under_mtop; i > rect->vsize; i--)
		rect->mtopmost = rect->mtopmost->next;

	dev->cursor(0, 0);
	return wnd_MxN_draw_menu(rect->mtopmost, m);
}

struct lcd_window*
window_create(void)
{
	static struct lcd_window wnd_1x1_ops = {
		wnd_1x1_open, wnd_set_caption, wnd_1x1_set_list,
		wnd_1x1_set_focus, wnd_clear, wnd_set_text, get_editor
	};

	static struct lcd_window wnd_1x2_ops = {
		wnd_MxN_open, wnd_set_caption, wnd_1x1_set_list,
		wnd_1x2_set_focus, wnd_clear, wnd_set_text, get_editor
	};

	static struct lcd_window wnd_MxN_ops = {
		wnd_MxN_open, wnd_set_caption, wnd_MxN_set_list,
		wnd_MxN_set_focus, wnd_clear, wnd_set_text, get_editor
	};

	int width, height;

	rect = (struct window_rect*)cf_malloc(sizeof(struct window_rect));
	memset(rect, 0, sizeof(struct window_rect));

	if (dev->getsize(&width, &height) == E_OK) {
		if (height == 1)
			return &wnd_1x1_ops;
		else if (height == 2)
			return &wnd_1x2_ops;
		else
			return &wnd_MxN_ops;
	}
	return NULL;
}

void
window_destroy(struct lcd_window *w)
{
	if (w != NULL)
		w->clear();
	dev->cursor(0, 0);
}

static int
edit_draw_line(struct edit_line *line)
{
	int len = strlen(line->visible);

	if (len > line->length)
		len = line->length;

	memset(line->buf, ' ', line->length);
	memcpy(line->buf, line->visible, len);

	if (line->text != line->visible)
		line->buf[0] = '<';
	if (line->length < strlen(line->visible))
		line->buf[line->length - 1] = '>';

	if (dev->setpos(line->x, line->y) == E_OK) {
		if (dev->putstr(line->buf, line->length) == E_OK)
			return dev->setpos(line->x + line->cursor, line->y);
	}
	return E_IO;
}

static int
edit_recalc_line(struct edit_line *line, int pos)
{
	int left_border;
	int right_border;

	char *ptr = line->text + pos;

	if ((pos < 0) || (pos > strlen(line->text)))
		return E_IO;
       
	left_border = (line->visible != line->text);
	right_border = (strlen(line->visible) > line->length);

	if ((ptr >= (line->visible + left_border)) &&
			(ptr <= (line->visible +
				 (line->length - right_border - 1)))) {
		/* setpos inside visible area */
		line->cursor = ptr - line->visible;
		return dev->setpos(line->x + line->cursor, line->y);
	} else if (ptr < (line->visible + left_border)) {
		/* too far to the left */
		if (ptr == line->text) {
			line->visible = ptr;
			line->cursor = 0;
		} else {
			line->visible = ptr - left_border;
			line->cursor = 1;
		}
	} else {
		/* too far to the right */
		if ((line->visible + line->length) == (ptr + right_border)) {
			line->visible = ptr - line->length + 2;
			line->cursor = line->length - 2;
		} else {
			char *lptr = ptr - line->length;
			if (!left_border) {
				line->visible = lptr + 1;
				line->cursor = line->length - 1;
			} else {
				line->visible = lptr; /*ptr - left_border;*/
				line->cursor = line->length - 1;
			}
			if (strlen(line->visible) > line->length)
				line->cursor--;
		}
	}

	return edit_draw_line(line);
}

static int
edit_setbuf(char *s, int align_hint)
{
	int i;
	struct edit_line *line = rect->line;

	line->cursor = 0;
	line->text = s;
	line->visible = s;
	line->x = rect->edit_offset;
	line->length = rect->width - line->x;

	if ((line->y != 0) && (align_hint == ALIGN_CENTER)) {
		int shift = (line->length - strlen(s)) >> 1;

		if (shift > 0) {
			line->x += shift;
			line->length -= shift;

			if (dev->setpos(0, line->y) != E_OK)
				return E_IO;
			memset(line->buf, ' ', shift);
			if (dev->putstr(line->buf, shift) != E_OK)
				return E_IO;
		}
	}

	/* clear edit area */
	memset(rect->line->buf, ' ', rect->width);
	for (i = 1; i <= rect->vsize; i++) {
		dev->setpos(0, i);
		dev->putstr(rect->line->buf, rect->width);
	}

	return edit_draw_line(line);
}

static int
edit_putchr(int c)
{
	rect->line->visible[rect->line->cursor] = c;
	return dev->putchr(c);
}

static int
edit_getchr(int *c)
{
	if ((c == NULL) || (rect->line->visible == NULL))
		return E_IO;
	*c = rect->line->visible[rect->line->cursor];
	return E_OK;
}

static int
edit_getpos(int *x)
{
	if ((x == NULL) || (rect->line->visible == NULL))
		return E_IO;
	*x = (rect->line->visible - rect->line->text) + rect->line->cursor;
	return E_OK;
}

static int
edit_setpos(int x)
{
	if (edit_recalc_line(rect->line, x) == E_OK)
		return edit_draw_line(rect->line);
	return E_IO;
}

static int
edit_left(void)
{
	int off = rect->line->visible - rect->line->text + rect->line->cursor;
	if (off > 0)
		return edit_setpos(--off);
	return E_OK;
}

static int
edit_right(void)
{
	int off = rect->line->visible - rect->line->text + rect->line->cursor;
	if (off < (strlen(rect->line->text) - 1))
		return edit_setpos(++off);
	return E_OK;
}

static struct lcd_edit*
get_editor(void)
{
	static struct lcd_edit edit_ops = {
		edit_setbuf,
		edit_putchr,
		edit_getchr,
		edit_getpos,
		edit_setpos,
		edit_left,
		edit_right
	};

	dev->cursor(1, 0);
	return &edit_ops;
}
