/* $NetBSD$ */

/*
 * Copyright (c) 2003 Dennis I. Chernoivanov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/utsname.h>

#include <unistd.h>
#include <malloc.h>
#include <string.h>

#include "paneld.h"

struct menu *root;
struct lcd_xlate *xlate;
extern struct lcd_window *window;

static int menu_xlate(struct menu*, int);

/*
 * Menu main loop: call menu operations in response to user
 * input
 */
int
enter_menu(struct menu *m)
{
	int err;

	util_trace(LOG_DEBUG, "enter menu %p:%s", m, m->nm);

	do {
		int xk;
		int key;

		if (scankey(&globals->devcap, 0, &key) != E_OK)
			return E_IO;

		switch( (xk = menu_xlate(m, key))) {
		case E_OK:
			err = E_OK;
			break;

		case E_INC:
			err = m->ops->inc(m);
			break;

		case E_DEC:
			err = m->ops->dec(m);
			break;

		case E_PREV:
			err = m->ops->prev(m);
			break;

		case E_NEXT:
			err = m->ops->next(m);
			break;

		default:
			return xk;
		}
	} while (err == E_OK);

	return err;
}

/*
 * Translate device input into internal events
 */
int
menu_xlate(struct menu *m, int c)
{
	if (m->xlate != NULL) {
		if (m->xlate->x_inc == c)
			return E_INC;
		if (m->xlate->x_dec == c)
			return E_DEC;
		if (m->xlate->x_next == c)
			return E_NEXT;
		if (m->xlate->x_prev == c)
			return E_PREV;
		if (m->xlate->x_enter == c)
			return E_SELECT;
	}
	return E_OK;
}

#define MAX_HEIGHT	255

/*
 * Display welcome message
 */
static void
display_banner(void)
{
	int err;
	int size;
	char *lines[MAX_HEIGHT];
	
	size = globals->buf_size;
	err = readconf(
			globals->banner_args,
			globals->banner_buf,
			&size);

	if ((err == E_OK) && (strlen(globals->banner_buf) > 0)) {
		int i, j;
		char *ptr = globals->banner_buf;
		for (i = 0, j = 0; i < size; i++) {
			if (globals->banner_buf[i] == 0) {
				lines[j++] = ptr;
				ptr = globals->banner_buf + i + 1;
				if (j == (MAX_HEIGHT - 1))
					break;
			}
		}

		if (j < (MAX_HEIGHT - 1)) {
			char *nl = strchr(ptr, '\n');
			if (nl != NULL)
				*nl = 0;
			lines[j++] = ptr;
		}
		lines[j] = 0;
	} else {
		struct utsname uts;
		if (uname(&uts) == -1)
			lines[0] = "NetBSD/Cobalt";
		else {
			int len = strlen(uts.sysname) + strlen(uts.machine);
			lines[0] = (char*)cf_malloc(len + 2);
			sprintf(lines[0], "%s/%s", uts.sysname, uts.machine);
		}
		lines[1] = NULL;
	}

	window->set_text(lines);
}

/*
 * Topmost menu loop:
 *  - Enter main menu loop on user input
 *  - Display welcome message otherwise
 */
int
run_menu(void)
{
	int err = E_OK;
	char *ioerr[2];
	char *badverr[2];

	ioerr[0] = "I/O Error";
	ioerr[1] = NULL;

	badverr[0] = "Bad input";
	badverr[1] = NULL;

	for (; (err == E_OK) || (err == E_EXIT);) {
		display_banner();
		for (;;) {
			int c;
			if (scankey(&globals->devcap, 1, &c) == E_OK) {
				if (c == globals->x_enter)
					break;
			}
		}
		window->clear();
		for (; (err = root->ops->select(root)) == E_OK;)
			;
		if (err == E_IO) {
			window->set_text(ioerr);
			sleep(1);
			err = E_OK;
		} else if (err == E_BADV) {
			window->set_text(badverr);
			sleep(1);
			err = E_OK;
		}
	}
	return err;
}
