/* $NetBSD$ */

/*
 * Copyright (c) 2003 Dennis I. Chernoivanov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>

#include "paneld.h"

#if defined(_USE_1X32)
#define _WIDTH	32
#define _HEIGHT	1
#elif defined(_USE_MXN)
#define _WIDTH	20
#define _HEIGHT	3
#else
#define _WIDTH	16
#define _HEIGHT	2
#endif

#define _CHECK_BUF(_x, _y) \
 do { \
	 int *ptr = NULL; \
	 if ((_x) >= _WIDTH) { \
		 fprintf(stderr, "%d is out of bounds [width]", _x); \
		 *ptr = 0; \
	 } \
	 if ((_y) >= _HEIGHT) { \
		 fprintf(stderr, "%d is out of bounds [height]", _y); \
		 *ptr = 0; \
	 } \
 } while(0)


static char screen[_HEIGHT][_WIDTH];

static int x;
static int y;
static int cursor_on;

static int
cons_open(const char *node)
{
	return E_OK;
}

static int
cons_close(void)
{
	return E_OK;
}

static void
cons_devcap(struct lcd_devcap *cap)
{
	memset(cap, 0, sizeof(struct lcd_devcap));
}

static int
cons_getsize(int *x, int *y)
{
	*x = _WIDTH;
	*y = _HEIGHT;
	return E_OK;
}

static int
cons_redraw(void)
{
	int i, j;
	for (i = 0; i < _WIDTH; i++)
		fprintf(stderr, "-");
	fprintf(stderr, "\n");
	for (i = 0; i < _HEIGHT; i++) {
		for (j = 0; j < _WIDTH; j++)
			fprintf(stderr, "%c", screen[i][j]);
		fprintf(stderr, "\n");
	}

	if (cursor_on) {
		for (i = 0; i < x; i++)
			fprintf(stderr, " ");
		fprintf(stderr, "^");
	}
	fprintf(stderr, "\n");
	for (i = 0; i < _WIDTH; i++)
		fprintf(stderr, "-");
	fprintf(stderr, "\n");
	return E_OK;
}

static int
cons_clear(void)
{
	memset(screen, ' ', sizeof(screen));
	return cons_redraw();
}

static int
cons_putchr(int c)
{
	util_trace(LOG_DEBUG, "cons_putchr at [%d][%d] = %c", y, x, c);

	screen[y][x] = c;
	return cons_redraw();
}

static int
cons_putstr(const char *buf, int len)
{
	int i;
	_CHECK_BUF(x + len -1, y);

	util_trace(LOG_DEBUG,
			"cons_putstr at [%d][%d] = %s[len = %d]",
			y, x, buf, len);

	for (i = 0; i < len; i++)
		screen[y][x + i] = buf[i];
	return cons_redraw();
}

static int
cons_setpos(int _x, int _y)
{
	_CHECK_BUF(_x, _y);
	x = _x;
	y = _y;
	return cons_redraw();
}

static int
cons_cursor(int on, int blink)
{
	cursor_on = on;
	return E_OK;
}

static int
cons_getchr(int *c)
{
	int _c = getchar();
	getchar(); /* skip eol */
	*c = _c;
	return E_OK;
}

struct lcd_dev cons_ops = {
	cons_open,
	cons_close,
	cons_devcap,
	cons_getsize,
	cons_clear,
	cons_putchr,
	cons_putstr,
	cons_setpos,
	cons_cursor,
	cons_getchr
};

struct lcd_dev*
get_console(void)
{
	return &cons_ops;
}
