/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     traverse_r_1d.c                                                */
/*                                                                          */
/* description:                                                             */
/*           recursive mesh traversal - 1d routines:                        */
/*           fill_macro_info_1d(), fill_elinfo_1d()                         */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*   fill_macro_info_1d:                              		            */
/*   ----------------                                   		    */
/*   fill EL_INFO structure for a macro element         		    */
/*--------------------------------------------------------------------------*/

static void fill_macro_info_1d(MESH *mesh, const MACRO_EL *mel,
			       EL_INFO *el_info)
{
  FUNCNAME("fill_macro_info_1d");
  EL       *nb;
  MACRO_EL *mnb;
  int      i, j;

  el_info->mesh     = mesh;
  el_info->macro_el = mel;
  el_info->el       = mel->el;
  el_info->parent   = nil;
  el_info->level    = 0;

  if (el_info->fill_flag & FILL_COORDS) {
    for (i = 0; i < N_VERTICES_1D; i++) {
      DEBUG_TEST_EXIT(mel->coord[i], "no mel->coord[%d]\n",i);

      for (j = 0; j < DIM_OF_WORLD; j++)
	el_info->coord[i][j] = mel->coord[i][j];
    }
  }

  if (el_info->fill_flag & (FILL_NEIGH|FILL_OPP_COORDS)) {
    REAL_D opp_coord;

    for (i = 0; i < N_NEIGH_1D; i++) 
    {
      nb = nil;
      if ((mnb = mel->neigh[i])) 
      {
	if (el_info->fill_flag & FILL_OPP_COORDS)
	{ 
	  for (j = 0; j < DIM_OF_WORLD; j++)
	    opp_coord[j] = mnb->coord[1-i][j];
	}

	nb = mnb->el;

	while (nb->child[0])  /*--- make nb nearest element             ---*/
	{
	  if (el_info->fill_flag & FILL_OPP_COORDS)
	  { 
	    for (j = 0; j < DIM_OF_WORLD; j++)
	      opp_coord[j] = 0.5*(mel->coord[1-i][j] + opp_coord[j]);
	  }
	  nb = nb->child[i];
	}

	if (el_info->fill_flag & FILL_OPP_COORDS)
	{ 
	  for (j = 0; j < DIM_OF_WORLD; j++)
	    el_info->opp_coord[i][j] = opp_coord[j];
	}
      }
      el_info->neigh[i] = nb;
      el_info->opp_vertex[i] = nb ? 1-i : -1;
    }
  }

  if (el_info->fill_flag & FILL_BOUND)
    for (i = 0; i < N_VERTICES_1D; i++)
      el_info->vertex_bound[i] = mel->vertex_bound[i];

  if(el_info->fill_flag & FILL_PROJECTION)
    el_info->active_projection = mel->projection[0];

  return;
}

/*--------------------------------------------------------------------------*/
/*   fill_el_info_1d:                                 		            */
/*   ------------                                       		    */
/*   fill EL_INFO structure for one child of an element   		    */
/*--------------------------------------------------------------------------*/

static void fill_elinfo_1d(int ichild, const EL_INFO *el_info_old,
			   EL_INFO *el_info)
{
  FUNCNAME("fill_elinfo_1d");
  int     i, j;
  EL      *nb;
  EL      *el = el_info_old->el;
  FLAGS   fill_flag = el_info_old->fill_flag;

  DEBUG_TEST_EXIT(el->child[0], "no children?\n");

  el_info->el = el->child[ichild];
  DEBUG_TEST_EXIT(el_info->el, "missing child %d?\n", ichild);

  el_info->macro_el  = el_info_old->macro_el;
  el_info->fill_flag = fill_flag;
  el_info->mesh      = el_info_old->mesh;
  el_info->parent    = el;
  el_info->level     = el_info_old->level + 1;

  if (fill_flag & FILL_COORDS) {
    const REAL_D *coord = el_info_old->coord;

    for (j = 0; j < DIM_OF_WORLD; j++) {
      el_info->coord[ichild][j] = coord[ichild][j];
      if (el->new_coord)
	el_info->coord[1-ichild][j] = el->new_coord[j];
      else
	el_info->coord[1-ichild][j] = 0.5*(coord[0][j] + coord[1][j]);
    }
  }

  if (fill_flag & (FILL_NEIGH|FILL_OPP_COORDS)) {
    REAL_D opp_coord;

    if (fill_flag & FILL_OPP_COORDS) {
      DEBUG_TEST_EXIT(fill_flag & FILL_COORDS,
		  "FILL_OPP_COORDS only with FILL_COORDS\n");
    }

    for (i = 0; i < N_NEIGH_1D; i++) {
      if (i == ichild) {
	nb = el->child[1-ichild];
	if (nb && el_info->fill_flag & FILL_OPP_COORDS) {
	  for (j = 0; j < DIM_OF_WORLD; j++)
	    opp_coord[j] = el_info_old->coord[1-i][j];
	}
      }
      else {
	nb = el_info_old->neigh[i];

	if (nb && el_info->fill_flag & FILL_OPP_COORDS) {
	  for (j = 0; j < DIM_OF_WORLD; j++)
	    opp_coord[j] = el_info_old->opp_coord[i][j];
	}
      }

      if (nb) {
	while (nb->child[0])  /*--- make nb nearest element              ---*/
	{
	  if (el_info->fill_flag & FILL_OPP_COORDS)
	  { 
	    for (j = 0; j < DIM_OF_WORLD; j++)
	      opp_coord[j] = 0.5*(el_info->coord[1-i][j] + opp_coord[j]);
	  }
	  nb = nb->child[i];
	}

	if (el_info->fill_flag & FILL_OPP_COORDS)
	{ 
	  for (j = 0; j < DIM_OF_WORLD; j++)
	    el_info->opp_coord[i][j] = opp_coord[j];
	}
      }
      el_info->neigh[i] = nb;
      el_info->opp_vertex[i] = nb ? 1-i : -1;
    }
  }

  if (fill_flag & FILL_BOUND) {
    el_info->vertex_bound[ichild]   = el_info_old->vertex_bound[ichild];
    el_info->vertex_bound[1-ichild] = INTERIOR;
  }

  if(fill_flag & FILL_PROJECTION)
    el_info->active_projection = el_info_old->active_projection;

  return;
}
