/******************************************************************************
 * This file is part of MPlayer Audio Visualization.                          *
 *                                                                            *
 *                                                                            *
 *  MPlayer Audio Visualization is free software; you can redistribute it     *
 *  and/or modify it under the terms of the GNU General Public License as     *
 *  published by the Free Software Foundation; either version 2 of the        *
 *  License, or (at your option) any later version.                           *
 *                                                                            *
 *  MPlayer Audio Visualization is distributed in the hope that it will be    *
 *  useful, but WITHOUT ANY WARRANTY; without even the implied warranty of    *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General *
 *  Public License for more details.                                          *
 *                                                                            * 
 *  You should have received a copy of the GNU General Public License         *
 *  along with MPlayer Audio Visualization; if not, write to the Free         *
 *  Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA         *
 *  02111-1307  USA                                                           *
 ******************************************************************************/
#define _C_SCREEN_
#include <screen.h>

int screen_init()
{
  int r;
  screen_mutex   = SDL_CreateMutex();
  messages_mutex = SDL_CreateMutex();

  /* Setup render */
  if ( ( r = render_init() ) != 0 )
    return( r );

  if ( __screen_change_resolution( 0 ) )
    {
      mpav_fmsg( stderr, "SDL ERROR: %s\n", SDL_GetError() );
      return( -1 );
    }  

  SDL_ShowCursor( 0 );
  SDL_EnableKeyRepeat( 0, 0 );

  if ( is_fs ) 
    screen_toggle_fullscreen();

  if ( ( screen_thread = SDL_CreateThread( (void*)__screen_thread_func, NULL ) ) == NULL )
    return( -1 );

  return( 0 );
}

void screen_quit()
{
  SDL_WaitThread( screen_thread, NULL );
  if ( is_fs ) 
    screen_toggle_fullscreen();

  /* End Render */
  render_quit();

  SDL_DestroyMutex( screen_mutex );
  screen_mutex = NULL;

  /* Clear messages */
  SDL_mutexP( messages_mutex );
  if ( message_list )
    {
      msglist_t *tmp = NULL;
      while ( message_list )
	{
	  tmp = msglist_delnode( &message_list, message_list );
	  msglist_destroynode( &tmp );
	}
    }
  SDL_mutexV( messages_mutex );

  SDL_DestroyMutex( messages_mutex );
  messages_mutex = NULL;
  
  mpav_debug( "Screen thread finished!\n" );
}

void screen_toggle_fullscreen()
{
  SDL_mutexP( screen_mutex );
  is_fs = ! is_fs;  
  SDL_WM_ToggleFullScreen( screen );
  SDL_mutexV( screen_mutex );
}

int screen_increase_resolution()
{
  return __screen_change_resolution( 1 );
}

int screen_decrease_resolution()
{
  return __screen_change_resolution( -1 );
}

void screen_toggle_messages()
{
  show_messages = ! show_messages;
}

int  screen_set_resolution( int width, int height )
{
  int r;
  char *newtitle = NULL;

  if ( ( width < 1 ) || ( height < 1 ) )
    return( -1 );

  SDL_mutexP( screen_mutex );  

  if ( ( screen = SDL_SetVideoMode( width, height, 32, SURFACE_FLAGS ) ) == NULL )
    {
      the_end = 1;
      mpav_debug( "ERROR: Failed to change resolution to %d x %d: %s\n", 
		  width, height, SDL_GetError() );      
      SDL_mutexV( screen_mutex );
      return( -2 );
    }

  newtitle = (char *)calloc( strlen( TITLE ) + 255, sizeof( char ) );
  sprintf( newtitle, "%s [%dx%d]", TITLE, width, height );
  SDL_WM_SetCaption( newtitle, NULL );
  free( newtitle );

  /* Change Render resolution */
  r = render_change_resolution();
  SDL_mutexV( screen_mutex );

  return( r );
}


/*****************************************************************************
 * Local Functions                                                           *
 *****************************************************************************/

int __screen_change_resolution( int d )
{
  static int i = DEF_RES;
  static int max = 12;
  static int resolx[] = { 320, 320, 400, 400, 512, 512, 640, 640, 640, 800, 800, 1024 };
  static int resoly[] = { 180, 240, 200, 300, 280, 384, 320, 400, 480, 400, 600, 768 };

  int ret = 0, j = 0;

  if ( SDL_mutexP( screen_mutex ) )
    { the_end = 1; return( -2 ); }

  if ( d < 0 )
    {
      if ( ( i - 1 ) >= 0 )
	i--;
      else
	ret = -1;
    }
  
  if ( (  d > 0 ) && ( ! ret ) )
    { 
      if ( ( i + 1 ) < max )
	i++;
      else
	ret = -1;
    }
  j = i;
  SDL_mutexV( screen_mutex );

  if ( ret )
    return( ret );
  else
    return screen_set_resolution( resolx[ j ], resoly[ j ] );
}


void __screen_thread_func()
{
  struct timeval t1, t2, t3, t4;
  long delay = 0;
  int frames = 0;
  char drop = 0;
  
  gettimeofday( &t3, NULL );
  while ( ! the_end )
    {      
      mpav_debug( "Screen loop...\n" );

      gettimeofday( &t1, NULL );

      SDL_mutexP( screen_mutex );      

      /* show render screen */
      if ( ! render_func() )
	{
	  SDL_mutexV( screen_mutex );      
	  usleep( 100 ); // don't suck CPU
	  continue;      
	}
      /* Show messages */
      __show_messages();
      /* Update screen */
      SDL_Flip( screen );

      SDL_mutexV( screen_mutex );	  
      
      gettimeofday( &t2, NULL );
      
      timersub( &t2, &t1, &t4 );

      delay = INTERVAL - ( t4.tv_sec * 1000000 + t4.tv_usec );
      
      mpav_debug( "Time to render frame: %ldus, "
		  "delay to next frame: %ldus "
		  "(interval is %dus)\n", 
		  ( t4.tv_sec * 1000000 + t4.tv_usec ),
		  delay,
		  INTERVAL );
      
      if ( delay > 0 )
	{
	  usleep( delay );
	  drop --;
	  if ( drop < 0 ) drop = 0;
	}
      else
	{
	  drop ++;
	  if ( drop > 5 )
	    {
	      mpav_fmsg( stderr, "CPU is not powerful enough!\n" );
	      drop = 5;
	    }
	}      

      /* FPS */
      frames ++;
      timersub( &t2, &t3, &t4 );
      delay = ( t4.tv_sec * 1000000 + t4.tv_usec );

      if ( delay >= 1000000 )
	{
	  fps = ((float) frames) * 1000000 / delay;
	  mpav_debug( "FPS: %.3f (frames: %d, time: %ld)\n", fps, frames, delay );
	  frames = 0;
	  gettimeofday( &t3, NULL );
	}
    }
}

void __show_messages() 
{
  SDL_mutexP( messages_mutex );
  if ( show_messages && message_list )
    {
      msglist_t *l = message_list;

      while ( l != NULL )
	{
	  if ( l->msg )
	    {
	      if ( l->shadow )
		draw_text_shadow( l->msg,
				  screen,
				  l->font,
				  l->fgcolor,
				  l->shcolor,
				  l->x,
				  l->y,
				  l->shx,
				  l->shy );
	      else
		draw_text( l->msg,
			   screen,
			   l->font,
			   l->fgcolor,
			   l->x,
			   l->y );
	    }
	  l = l->next;
	}
    }
  SDL_mutexV( messages_mutex );  
}
