# FIAIF is an Intelligent firewall
#
# description: Automates a packet filtering firewall with iptables.
#
# Script Author:	Anders Fugmann <afu at fugmann dot net>
# 
# FIAIF is an Intelligent firewall
# Copyright (C) 2002-2011 Anders Peter Fugmann
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

##############################################################################
# Check /proc/sys/net settings, and print out any strange settings
##############################################################################

function check_network_settings ()
{
    RES_ERRORS=""
    RES_WARNINGS=""
    echo "Examining system Configuration:"
    local PROC_PATH=/proc/sys/net/ipv4
    if [ \! -d ${PROC_PATH} ];then
	echo "Could not find $PROC_PATH."
	echo "Please make sure that you have"
	echo "compiled the kernel with CONFIG_PROC_FS enabled"
	echo "and mounted /proc."
	echo "Aborting system examination."
	return
    fi
    local ERRORS=0
    local WARNINGS=0
    
    # /proc/sys/net/ipv4/conf/all/accept_source_route
    local SOURCE_ROUTE=$(cat /proc/sys/net/ipv4/conf/all/accept_source_route)
    if (( SOURCE_ROUTE == 1 )); then
	echo "Problem: accept_source_route=1"
	echo "  Source routing should be disabled, to avoid IP-spoofing."
	echo "  Please note that SRR is not a safeguard against IP-spoofing"
	echo "Solution:"
	echo "  Make sure the line 'net/ipv4/conf/all/accept_source_route=0'"
	echo "  is in /etc/sysctl.conf"
	echo
	RES_ERRORS="${RES_ERRORS} net/ipv4/conf/all/accept_source_route=0"
	let ERRORS++
    fi
    
    # /proc/sys/net/ipv4/conf/all/accept_redirects
    local ACCEPT_REDIRECTS=$(cat /proc/sys/net/ipv4/conf/all/accept_redirects)
    if (( SOURCE_ROUTE == 1 )); then
	echo "Problem: accept_redirects=1"
	echo "  Do not accept redirects to minimize possibility of IP-spoofing."
	echo "Solution:"
	echo "  Make sure the line 'net/ipv4/conf/all/accept_redirects=0'"
	echo "  is in /etc/sysctl.conf"
	echo
	RES_ERRORS="${RES_ERRORS} net/ipv4/conf/all/accept_redirects=0" 
	let ERRORS++
    fi
    
    # /net/ipv4/ip_forward
    local NR_ZONES=0
    local DUMMY
    for DUMMY in $ZONES; do
	let NR_ZONES++
    done
    local IP_FORWARD=$(cat /proc/sys/net/ipv4/ip_forward)
    if (( NR_ZONES <= 1 && IP_FORWARD == 1 )); then
	# Only one zone, but ip_forward enabled
	echo "Problem: ip_forward=1"
	echo "  You have not specified multiple zones, yet 'ip_forward' is enabled."
	echo "  This poses a security risk."
	echo "Solution:"
	echo "  Make sure the line 'net/ipv4/ip_forward=0' is in /etc/sysctl.conf"
	echo
	RES_ERRORS="${RES_ERRORS} net/ipv4/ip_forward=0"
	let ERRORS++
    elif (( NR_ZONES >= 2 && IP_FORWARD == 0 )); then
	# Multiple zones, but no forwarding.
	echo "Problem: ip_forward=0"
	echo "  You have specified multiple zones, yet 'ip_forward' is not enabled."
        echo "  This means that your firewall cannot route packets between zones."
	echo "Solution:"
	echo "  Make sure the line 'net/ipv4/ip_forward=1' is in /etc/sysctl.conf"
	echo
	RES_ERRORS="${RES_ERRORS} net/ipv4/ip_forward=1"
	let ERRORS++
    fi

    # net/ipv4/conf/all/rp_filter=1
    local ZONE
    local GLOBAL_ZONES=0
    for ZONE in ${ZONES}; do
	read_zone ${ZONE}
	if (( $? == 0  && GLOBAL == 1 )); then
	    let GLOBAL_ZONES++
	fi
    done

    local RP_FILTER=$(cat /proc/sys/net/ipv4/conf/all/rp_filter)
    if (( GLOBAL_ZONES <= 1 )); then
	if (( RP_FILTER == 0 )); then 
	    echo "Warning: rp_filter=0"
	    echo "  Source validation by reversed path, as specified in RFC1812 is disabled."
	    echo "  If you only have one internet connection, this should be enabled."
	    echo "  It should be disabled if you have multible routes to the same network, e.g. two "
	    echo "  or more internet connections."
	    echo "Solution:"
	    echo "  Make sure the line 'net/ipv4/conf/all/rp_filter=1' is in /etc/sysctl.conf"
	    echo
	    let WARNINGS++
	    RES_WARNINGS="${RES_WARNINGS} net/ipv4/conf/all/rp_filter=1"
	fi
    else
	if (( RP_FILTER > 0 )); then
	    echo "Warning: rp_filter != 0"
	    echo "  Source validation by reversed path, as specified in RFC1812 is enabled."
	    echo "  Since you have specified more that two global zones, FIAIF assumes that you have more "
	    echo "  than one internet connection. If this is the case, rp_filter should be disabled."
	    echo "Solution:"
	    echo "  Make sure the line 'net/ipv4/conf/all/rp_filter=0' is in /etc/sysctl.conf"
            echo "  or use the 'rp_filter_mask' patch, which can be found at"
            echo "  http://www.linuxvirtualserver.org/~julian/. When this is applied" 
            echo "  you can saftly disregard this message."
	    echo
	    let ERRORS++
	    RES_ERRORS="${RES_ERRORS} net/ipv4/conf/all/rp_filter=0"
	fi
    fi
	    
    # net/ipv4/icmp_echo_ignore_broadcasts=1
    if (( $(cat /proc/sys/net/ipv4/icmp_echo_ignore_broadcasts) == 0 )); then
	echo "Warning: icmp_echo_ignore_broadcasts=0"
	echo "  System responds to broadcast ping."
	echo "  This is often used by hackers, who makes broadcast pings to see which hosts are up."
	echo "Solution:"
	echo "  Make sure the line 'net/ipv4/icmp_echo_ignore_broadcasts=1' is in /etc/sysctl.conf"
	echo
	RES_WARNINGS="${RES_WARNINGS} net/ipv4/icmp_echo_ignore_broadcasts=1"
	let WARNINGS++
    fi

    # net/ipv4/tcp_syncookies
    if [ -f /proc/sys/net/ipv4/tcp_syncookies ]; then
	if (( $(cat /proc/sys/net/ipv4/tcp_syncookies) == 0 )); then
	    echo "Warning: tcp_syncookies=0"
	    echo "  Send out syncookies when the syn backlog queue of a socket"
	    echo "  overflows. This is done to prevent against the common 'syn flood attack'"
	    echo "  This might break things on high load servers."
	    echo "Solution:"
	    echo "  Make sure the line 'net/ipv4/tcp_syncookies=1' is in /etc/sysctl.conf"
	    echo
	    RES_WARNINGS="${RES_WARNINGS} net/ipv4/tcp_syncookies=1"
	    let WARNINGS++
	fi
    else
	echo "Warning: Kernel compiled without 'CONFIG_SYNCOOKIES'"
	echo "  Send out syncookies when the syn backlog queue of a socket"
	echo "  overflows. This is to prevent against the common 'syn flood attack'"
	echo "  Do not enable this on high load servers."
	echo "Solution:"
	echo "  Recompile the kernel with 'CONFIG_SYNCOOKIES' option"
	echo
	let WARNINGS++
    fi

    # net/ipv4/tcp_ecn=1
    if [ -f /proc/sys/net/ipv4/tcp_ecn ]; then
        if (( $(cat /proc/sys/net/ipv4/tcp_ecn) == 0 )); then
	    echo "Warning: tcp_ecn=0"
	    echo "  ECN (Explicit Congestion Notification) is defined in RFC3168."
	    echo "  It allows routers on the internet to notify when congestion occurs,"
	    echo "  and can avoid packet being dropped due to queue overflow. Overall this is"
	    echo "  good for the internet."
	    echo "  The backside is that some really brain-damaged firewalls and routers do"
	    echo "  not allow packets with the ECN bit set (and thus not conforming to RFC3168). "
	    echo "Solution:"
	    echo "  Make sure the line 'net/ipv4/tcp_ecn=1' is in /etc/sysctl.conf"
	    echo
	    RES_WARNINGS="${RES_WARNINGS} net/ipv4/tcp_ecn=1" 
	    let WARNINGS++	
	fi
    else
	echo "Warning: Kernel not compiled with ECN support."
	echo "  ECN (Explicit Congestion Notification) is defined in RFC3168."
	echo "  It allows routers on the internet to notify when congestion occurs,"
	echo "  and can avoid packet being dropped due to queue overflow. Overall this is"
        echo "  good for the internet."
        echo "  The backside is that some really brain-damaged firewalls and routers do"
	echo "  not allow packets with the ECN bit set (and thus not conforming to RFC3168). "
    	echo "Solution:"
	echo "  Recompile the kernel with 'CONFIG_INET_ECN' option."
	echo
	let WARNINGS++
    fi

    # net/ipv4/tcp_fin_timeout > ${TCP_FIN_TIMEOUT}
    if (( $(cat /proc/sys/net/ipv4/tcp_fin_timeout) > ${TCP_FIN_TIMEOUT} )); then
	echo "Warning: tcp_fin_timeout > ${TCP_FIN_TIMEOUT}"
	echo "  System holds broken TCP connections for more than ${TCP_FIN_TIMEOUT} seconds"
	echo "  Holding these connectinos open can quickly overflow memory."
	echo "Solution:"
	echo "  Make sure the line 'net/ipv4/tcp_fin_timeout=${TCP_FIN_TIMEOUT}' is in /etc/sysctl.conf"
	echo
	RES_WARNINGS="${RES_WARNINGS} net/ipv4/tcp_fin_timeout=${TCP_FIN_TIMEOUT}"
	let WARNINGS++
    fi

    # /net/ipv4/ip_dynaddr
    local ZONE
    local DYNAMIC_ZONES=0
    for ZONE in ${ZONES}; do
	read_zone ${ZONE}
	if (( $? == 0  && DYNAMIC == 1 )); then
	    let DYNAMIC_ZONES++
	fi
    done
    local IP_DYNADDR=$(cat /proc/sys/net/ipv4/ip_dynaddr)
    if (( IP_DYNADDR == 1 && DYNAMIC_ZONES == 0 )); then
	echo "Warning: Dynamic addresses enabled."
	echo "  Dynamic addresses should only be enabled if the ip address of any of your interfaces can"
	echo "  change while the interface is up. As no zones are specified as dynamic this should never happen."
	echo "Solution:"
	echo "  Make sure the line 'net/ipv4/ip_dynaddr=0' is in /etc/sysctl.conf"
	RES_WARNINGS="${RES_WARNINGS} net/ipv4/ip_dynaddr=0"
	let WARNINGS++
    elif (( IP_DYNADDR == 0 && DYNAMIC_ZONES == 1 )); then
	echo "Warning: Dynamic addresses disabled."
	echo "  The kernel is set not to allow address changes on intercases while they are up."
	echo "  However, as one or more zones are specified as dynamic, this indicates that this may happen."
	echo "Solution:"
        echo "  Make sure the line 'net/ipv4/ip_dynaddr=1' is in /etc/sysctl.conf"
	RES_WARNINGS="${RES_WARNINGS} net/ipv4/ip_dynaddr=1"
	let WARNINGS++
    fi
	
    if (( ERRORS > 0 || WARNINGS > 0 )); then
	echo 
	echo "Problems found: ${ERRORS}"
	echo "Warnings found: ${WARNINGS}"
	echo
        echo "Above is given a description of the problems and warnings found while"
	echo "examining the system, and solutions are given."
	echo "Please read carefully, since it might improve overall security of your system"
	echo "For more information, please see: "
	echo "  http://www.linuxhq.com/kernel/v2.4/doc/networking/ip-sysctl.txt.html" 
	echo "  http://ipsysctl-tutorial.frozentux.net/ipsysctl-tutorial.html"
    else
	echo "Done."
    fi
}

function set_proc ()
{
    local STATE_FILE=$1

    local RES_ERRORS
    local RES_WARNINGS
    check_network_settings > /dev/null

    if [[ -n ${STATE_FILE} ]]; then
	rm -fr ${STATE_FILE}
	touch ${STATE_FILE}
	chown root:root ${STATE_FILE}
	chmod 600 ${STATE_FILE}
    else
	STATE_FILE=/dev/null
    fi

    local FILE
    local VALUE
    if [ "${SET_PROC_ERRORS}" = "1" ]; then
	for CHANGE in ${RES_ERRORS}; do
	    FILE=/proc/sys/${CHANGE%=*}
	    VALUE=${CHANGE#*=}
	    echo "${CHANGE%=*}=$(cat ${FILE})" >> ${PROC_STATE_FILE}
	    echo ${VALUE} > ${FILE}
	done
    fi
      
  
    if [ "${SET_PROC_WARNINGS}" = "1" ]; then
	for CHANGE in ${RES_WARNINGS}; do
	    FILE=/proc/sys/${CHANGE%=*}
	    VALUE=${CHANGE#*=}
	    echo "${CHANGE%=*}=$(cat ${FILE})" >> ${STATE_FILE}
	    echo ${VALUE} > ${FILE}
	done
    fi
}

function restore_proc ()
{
    # This could also be done by sysctl, if we wanted to depend on it.
    local STATE_FILE=$1
    echo -n "Restoring proc settings: "

    if [[ -f ${STATE_FILE} ]]; then
	CHANGES=$(cat ${STATE_FILE}) 
	for CHANGE in ${CHANGES}; do
	    FILE=/proc/sys/${CHANGE%=*}
	    VALUE=${CHANGE#*=}
	    echo ${VALUE} > ${FILE}
	done
    else
	echo -n "File '${STATE_FILE}' not found " 
    fi
    echo "Done."
}
