/* $Id: EndTransfer.hpp 4323 2009-01-27 13:48:12Z potyra $ 
 *
 * EndTransfer: free space for transfer region again.
 *
 * Copyright (C) 2008-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */
#ifndef __END_TRANSFER_HPP_INCLUDED
#define __END_TRANSFER_HPP_INCLUDED

#include "intermediate/opcodes/OpCode.hpp"
#include "intermediate/operands/Reference.hpp"
#include "intermediate/operands/ImmediateOperand.hpp"

namespace intermediate {


//! free space for the transfer region.
/** This opcode will free space for the named transfer region.
 *
 *  Read operands: src, cleanupStack
 *  Write operands: no publicly visible operands.
 *
 *  Operation:
 *      Free space for the transfer section of component src at an 
 *      implementation defined location, in case cleanupStack is set.
 *
 *  Errors:
 *      It is an error if EndTransfer wasn't called after BeginTransfer was
 *      called from within the same text segment.
 *      It is an error if EndTransfer references a different component
 *      than the preceding BeginTransfer call.
 */
class EndTransfer : public OpCode {
public:
	//! c'tor
	/** @param source reference to a component denoting the transfer
	 *         section.
	 *  @param cleanup when executing end transfer, cleanup the 
	 *         stack entries or not? (useful for _create calls,
	 *         which create processes that need the parent stack to
	 *         be present).
	 */
	EndTransfer(
		Reference *source, 
		ImmediateOperand *cleanup
		) : 	src(source), cleanupStack(cleanup) {}

	//! Accept a Visitor.
 	/** All intermediate code nodes need to implement this method.
         *
         *  @param v the Visitor that can visit this node.
         */
	virtual void accept(Visitor& v) {
		v.visit(*this);
	}

	//! references a component (denoting the transfer region of it)
	Reference *src;

	//! cleanup the stack 
	ImmediateOperand *cleanupStack;

protected:
	virtual ~EndTransfer() {
		util::MiscUtil::terminate(this->src);
		util::MiscUtil::terminate(this->cleanupStack);
	}
};

}; /* namespace intermediate */

#endif /* __END_TRANSFER_HPP_INCLUDED */
