;;; its/hangul.el --- Hangul Input in Egg Input Method Architecture

;; Copyright (C) 1999,2000 PFU LIMITED

;; Author: NIIBE Yutaka <gniibe@chroot.org>
;;         KATAYAMA Yoshio <kate@pfu.co.jp>

;; Maintainer: TOMURA Satoru <tomura@etl.go.jp>

;; Keywords: mule, multilingual, input method

;; This file is part of EGG.

;; EGG is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; EGG is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:


;;; Code:


(eval-when-compile
  (require 'its)
  (require 'cl))

(eval-when (compile)
  (defconst its-compaction-enable t))

(defvar its-hangul-enable-jeonkak-alphabet
  (if (boundp 'its-enable-fullwidth-alphabet)
      its-enable-fullwidth-alphabet
    t)
  "*Enable Jeonkak alphabet")

(defvar	its-hangul-period ". " "*$(C#.(B")	; ". " "$(C#.(B"
(defvar	its-hangul-comma  ", " "*$(C#,(B")	; ", " "$(C#,(B"
(defvar its-hangul-open-bracket  "$(C!8(B" "*[")  ; "$(C#[(B"
(defvar its-hangul-close-bracket "$(C!9(B" "*]")  ; "$(C#](B"

(eval-when-compile
  (defconst its-hangul-key-alist nil)
  (defconst its-hangul-successor-list nil)

  (defsubst its-hangul-sequence-to-key (seq)
    (concat (mapcar (lambda (c)
		      (cdr (assq c its-hangul-key-alist)))
		    (string-to-list seq))))

  (defsubst its-hangul-make-successor (init keyseq)
    (concat (if init (substring init 0 (1- (length init))) "[")
	    (list (aref keyseq 1))
	    "]")))

(eval-when-compile
  (defun its-define-hangul-key (key output)
    (setq its-hangul-key-alist
	  (cons (cons (string-to-char output)
		      (string-to-char key))
		its-hangul-key-alist))
    (its-defrule key output))

  (defun its-define-hangul-successor-list (keyseq)
    (let ((pair (assq (aref keyseq 0) its-hangul-successor-list)))
      (if (null pair)
	  (setq its-hangul-successor-list
		(cons (cons (aref keyseq 0)
			    (its-hangul-make-successor nil keyseq))
		      its-hangul-successor-list))
	(if (null (string-match (cdr pair) keyseq 1))
	    (setcdr pair (its-hangul-make-successor (cdr pair) keyseq))))))

  (defun its-defrule-hangul (input output)
    (let* ((state (its-defrule input output nil nil))
	   (prev-out (its-get-output (car its-parent-states)))
	   pair)
     (if (and (= (length (string-to-list prev-out)) 1)
	      (> (string-to-char prev-out) ?\377)
	      (setq pair (assq (string-to-char (substring input -1))
			       its-hangul-successor-list)))
	 (progn
	   (its-set-interim-terminal-state state)
	   (its-defrule-otherwise state nil (cdr pair) -2)))))

  (defmacro its-define-hangul (&rest list)
    (cons 'progn
	  (mapcar (lambda (l)
		    (let ((case-fold-search nil)
			  (keyseq (its-hangul-sequence-to-key (car l))))
		      (if (>= (length keyseq) 2)
			  (its-define-hangul-successor-list keyseq))
		      (list 'its-defrule-hangul keyseq (nth 1 l))))
		  list))))

(define-its-state-machine its-hangul-map
  "hangul" "$(CGQ(B" Korean
  "Map for Hangul input. (Korean)"

  (defconst its-jeonkak-escape "Z")
  (defconst its-pankak-escape  "X")

  (its-defrule-select-mode-temporally "B" downcase)
  (its-defrule-select-mode-temporally "J" jeonkak-downcase)

  (its-define-hangul-key	"r"	"$(C$!(B")
  (its-define-hangul-key	"s"	"$(C$$(B")
  (its-define-hangul-key	"e"	"$(C$'(B")
  (its-define-hangul-key	"f"	"$(C$)(B")
  (its-define-hangul-key	"a"	"$(C$1(B")
  (its-define-hangul-key	"q"	"$(C$2(B")
  (its-define-hangul-key	"t"	"$(C$5(B")
  (its-define-hangul-key	"d"	"$(C$7(B")
  (its-define-hangul-key	"w"	"$(C$8(B")
  (its-define-hangul-key	"c"	"$(C$:(B")
  (its-define-hangul-key	"z"	"$(C$;(B")
  (its-define-hangul-key	"x"	"$(C$<(B")
  (its-define-hangul-key	"v"	"$(C$=(B")
  (its-define-hangul-key	"g"	"$(C$>(B")
  (its-define-hangul-key	"R"	"$(C$"(B")
  (its-define-hangul-key	"E"	"$(C$((B")
  (its-define-hangul-key	"Q"	"$(C$3(B")
  (its-define-hangul-key	"T"	"$(C$6(B")
  (its-define-hangul-key	"W"	"$(C$9(B")

  (its-define-hangul-key	"k"	"$(C$?(B")
  (its-define-hangul-key	"i"	"$(C$A(B")
  (its-define-hangul-key	"j"	"$(C$C(B")
  (its-define-hangul-key	"u"	"$(C$E(B")
  (its-define-hangul-key	"h"	"$(C$G(B")
  (its-define-hangul-key	"y"	"$(C$K(B")
  (its-define-hangul-key	"n"	"$(C$L(B")
  (its-define-hangul-key	"b"	"$(C$P(B")
  (its-define-hangul-key	"m"	"$(C$Q(B")
  (its-define-hangul-key	"l"	"$(C$S(B")
  (its-define-hangul-key	"o"	"$(C$@(B")
  (its-define-hangul-key	"p"	"$(C$D(B")
  (its-define-hangul-key	"O"	"$(C$B(B")
  (its-define-hangul-key	"P"	"$(C$F(B")

  (its-define-hangul
   ("$(C$!$?(B"	"$(C0!(B")
   ("$(C$!$?$!(B"	"$(C0"(B")
   ("$(C$!$?$$(B"	"$(C0#(B")
   ("$(C$!$?$'(B"	"$(C0$(B")
   ("$(C$!$?$)(B"	"$(C0%(B")
   ("$(C$!$?$)$!(B"	"$(C0&(B")
   ("$(C$!$?$)$1(B"	"$(C0'(B")
   ("$(C$!$?$1(B"	"$(C0((B")
   ("$(C$!$?$2(B"	"$(C0)(B")
   ("$(C$!$?$2$5(B"	"$(C0*(B")
   ("$(C$!$?$5(B"	"$(C0+(B")
   ("$(C$!$?$6(B"	"$(C0,(B")
   ("$(C$!$?$7(B"	"$(C0-(B")
   ("$(C$!$?$8(B"	"$(C0.(B")
   ("$(C$!$?$:(B"	"$(C0/(B")
   ("$(C$!$?$<(B"	"$(C00(B")
   ("$(C$!$?$=(B"	"$(C01(B")
   ("$(C$!$?$>(B"	"$(C02(B")

   ("$(C$!$@(B"	"$(C03(B")
   ("$(C$!$@$!(B"	"$(C04(B")
   ("$(C$!$@$$(B"	"$(C05(B")
   ("$(C$!$@$)(B"	"$(C06(B")
   ("$(C$!$@$1(B"	"$(C07(B")
   ("$(C$!$@$2(B"	"$(C08(B")
   ("$(C$!$@$5(B"	"$(C09(B")
   ("$(C$!$@$6(B"	"$(C0:(B")
   ("$(C$!$@$7(B"	"$(C0;(B")

   ("$(C$!$A(B"	"$(C0<(B")
   ("$(C$!$A$!(B"	"$(C0=(B")
   ("$(C$!$A$$(B"	"$(C0>(B")
   ("$(C$!$A$)(B"	"$(C0?(B")
   ("$(C$!$A$5(B"	"$(C0@(B")
   ("$(C$!$A$7(B"	"$(C0A(B")

   ("$(C$!$B(B"	"$(C0B(B")
   ("$(C$!$B$$(B"	"$(C0C(B")
   ("$(C$!$B$)(B"	"$(C0D(B")

   ("$(C$!$C(B"	"$(C0E(B")
   ("$(C$!$C$!(B"	"$(C0F(B")
   ("$(C$!$C$$(B"	"$(C0G(B")
   ("$(C$!$C$'(B"	"$(C0H(B")
   ("$(C$!$C$)(B"	"$(C0I(B")
   ("$(C$!$C$)$1(B"	"$(C0J(B")
   ("$(C$!$C$1(B"	"$(C0K(B")
   ("$(C$!$C$2(B"	"$(C0L(B")
   ("$(C$!$C$5(B"	"$(C0M(B")
   ("$(C$!$C$6(B"	"$(C0N(B")
   ("$(C$!$C$7(B"	"$(C0O(B")
   ("$(C$!$C$8(B"	"$(C0P(B")
   ("$(C$!$C$<(B"	"$(C0Q(B")
   ("$(C$!$C$=(B"	"$(C0R(B")
   ("$(C$!$C$>(B"	"$(C0S(B")

   ("$(C$!$D(B"	"$(C0T(B")
   ("$(C$!$D$$(B"	"$(C0U(B")
   ("$(C$!$D$)(B"	"$(C0V(B")
   ("$(C$!$D$1(B"	"$(C0W(B")
   ("$(C$!$D$2(B"	"$(C0X(B")
   ("$(C$!$D$5(B"	"$(C0Y(B")
   ("$(C$!$D$6(B"	"$(C0Z(B")
   ("$(C$!$D$7(B"	"$(C0[(B")

   ("$(C$!$E(B"	"$(C0\(B")
   ("$(C$!$E$!(B"	"$(C0](B")
   ("$(C$!$E$"(B"	"$(C0^(B")
   ("$(C$!$E$$(B"	"$(C0_(B")
   ("$(C$!$E$'(B"	"$(C0`(B")
   ("$(C$!$E$)(B"	"$(C0a(B")
   ("$(C$!$E$1(B"	"$(C0b(B")
   ("$(C$!$E$2(B"	"$(C0c(B")
   ("$(C$!$E$5(B"	"$(C0d(B")
   ("$(C$!$E$6(B"	"$(C0e(B")
   ("$(C$!$E$7(B"	"$(C0f(B")
   ("$(C$!$E$<(B"	"$(C0g(B")

   ("$(C$!$F(B"	"$(C0h(B")
   ("$(C$!$F$$(B"	"$(C0i(B")
   ("$(C$!$F$)(B"	"$(C0j(B")
   ("$(C$!$F$2(B"	"$(C0k(B")
   ("$(C$!$F$5(B"	"$(C0l(B")

   ("$(C$!$G(B"	"$(C0m(B")
   ("$(C$!$G$!(B"	"$(C0n(B")
   ("$(C$!$G$$(B"	"$(C0o(B")
   ("$(C$!$G$'(B"	"$(C0p(B")
   ("$(C$!$G$)(B"	"$(C0q(B")
   ("$(C$!$G$)$1(B"	"$(C0r(B")
   ("$(C$!$G$)$5(B"	"$(C0s(B")
   ("$(C$!$G$)$>(B"	"$(C0t(B")
   ("$(C$!$G$1(B"	"$(C0u(B")
   ("$(C$!$G$2(B"	"$(C0v(B")
   ("$(C$!$G$5(B"	"$(C0w(B")
   ("$(C$!$G$7(B"	"$(C0x(B")
   ("$(C$!$G$8(B"	"$(C0y(B")

   ("$(C$!$G$?(B"	"$(C0z(B")
   ("$(C$!$G$?$!(B"	"$(C0{(B")
   ("$(C$!$G$?$$(B"	"$(C0|(B")
   ("$(C$!$G$?$)(B"	"$(C0}(B")
   ("$(C$!$G$?$)$1(B"	"$(C0~(B")
   ("$(C$!$G$?$1(B"	"$(C1!(B")
   ("$(C$!$G$?$2(B"	"$(C1"(B")
   ("$(C$!$G$?$5(B"	"$(C1#(B")
   ("$(C$!$G$?$7(B"	"$(C1$(B")

   ("$(C$!$G$@(B"	"$(C1%(B")
   ("$(C$!$G$@$$(B"	"$(C1&(B")
   ("$(C$!$G$@$)(B"	"$(C1'(B")
   ("$(C$!$G$@$2(B"	"$(C1((B")
   ("$(C$!$G$@$6(B"	"$(C1)(B")
   ("$(C$!$G$@$7(B"	"$(C1*(B")

   ("$(C$!$G$S(B"	"$(C1+(B")
   ("$(C$!$G$S$!(B"	"$(C1,(B")
   ("$(C$!$G$S$$(B"	"$(C1-(B")
   ("$(C$!$G$S$)(B"	"$(C1.(B")
   ("$(C$!$G$S$1(B"	"$(C1/(B")
   ("$(C$!$G$S$2(B"	"$(C10(B")
   ("$(C$!$G$S$5(B"	"$(C11(B")
   ("$(C$!$G$S$7(B"	"$(C12(B")

   ("$(C$!$K(B"	"$(C13(B")
   ("$(C$!$K$$(B"	"$(C14(B")
   ("$(C$!$K$)(B"	"$(C15(B")
   ("$(C$!$K$2(B"	"$(C16(B")
   ("$(C$!$K$5(B"	"$(C17(B")

   ("$(C$!$L(B"	"$(C18(B")
   ("$(C$!$L$!(B"	"$(C19(B")
   ("$(C$!$L$$(B"	"$(C1:(B")
   ("$(C$!$L$'(B"	"$(C1;(B")
   ("$(C$!$L$)(B"	"$(C1<(B")
   ("$(C$!$L$)$!(B"	"$(C1=(B")
   ("$(C$!$L$)$1(B"	"$(C1>(B")
   ("$(C$!$L$)$>(B"	"$(C1?(B")
   ("$(C$!$L$1(B"	"$(C1@(B")
   ("$(C$!$L$2(B"	"$(C1A(B")
   ("$(C$!$L$5(B"	"$(C1B(B")
   ("$(C$!$L$7(B"	"$(C1C(B")
   ("$(C$!$L$8(B"	"$(C1D(B")

   ("$(C$!$L$C(B"	"$(C1E(B")
   ("$(C$!$L$C$!(B"	"$(C1F(B")
   ("$(C$!$L$C$$(B"	"$(C1G(B")
   ("$(C$!$L$C$)(B"	"$(C1H(B")
   ("$(C$!$L$C$6(B"	"$(C1I(B")
   ("$(C$!$L$C$7(B"	"$(C1J(B")

   ("$(C$!$L$D(B"	"$(C1K(B")
   ("$(C$!$L$D$5(B"	"$(C1L(B")

   ("$(C$!$L$S(B"	"$(C1M(B")
   ("$(C$!$L$S$!(B"	"$(C1N(B")
   ("$(C$!$L$S$$(B"	"$(C1O(B")
   ("$(C$!$L$S$)(B"	"$(C1P(B")
   ("$(C$!$L$S$1(B"	"$(C1Q(B")
   ("$(C$!$L$S$2(B"	"$(C1R(B")
   ("$(C$!$L$S$5(B"	"$(C1S(B")

   ("$(C$!$P(B"	"$(C1T(B")
   ("$(C$!$P$$(B"	"$(C1U(B")
   ("$(C$!$P$)(B"	"$(C1V(B")

   ("$(C$!$Q(B"	"$(C1W(B")
   ("$(C$!$Q$!(B"	"$(C1X(B")
   ("$(C$!$Q$$(B"	"$(C1Y(B")
   ("$(C$!$Q$'(B"	"$(C1Z(B")
   ("$(C$!$Q$)(B"	"$(C1[(B")
   ("$(C$!$Q$)$!(B"	"$(C1\(B")
   ("$(C$!$Q$1(B"	"$(C1](B")
   ("$(C$!$Q$2(B"	"$(C1^(B")
   ("$(C$!$Q$5(B"	"$(C1_(B")
   ("$(C$!$Q$7(B"	"$(C1`(B")

   ("$(C$!$Q$S(B"	"$(C1a(B")

   ("$(C$!$S(B"	"$(C1b(B")
   ("$(C$!$S$!(B"	"$(C1c(B")
   ("$(C$!$S$$(B"	"$(C1d(B")
   ("$(C$!$S$'(B"	"$(C1e(B")
   ("$(C$!$S$)(B"	"$(C1f(B")
   ("$(C$!$S$)$1(B"	"$(C1g(B")
   ("$(C$!$S$1(B"	"$(C1h(B")
   ("$(C$!$S$2(B"	"$(C1i(B")
   ("$(C$!$S$5(B"	"$(C1j(B")
   ("$(C$!$S$7(B"	"$(C1k(B")
   ("$(C$!$S$8(B"	"$(C1l(B")
   ("$(C$!$S$=(B"	"$(C1m(B")

   ("$(C$"$?(B"	"$(C1n(B")
   ("$(C$"$?$!(B"	"$(C1o(B")
   ("$(C$"$?$"(B"	"$(C1p(B")
   ("$(C$"$?$$(B"	"$(C1q(B")
   ("$(C$"$?$)(B"	"$(C1r(B")
   ("$(C$"$?$)$1(B"	"$(C1s(B")
   ("$(C$"$?$1(B"	"$(C1t(B")
   ("$(C$"$?$2(B"	"$(C1u(B")
   ("$(C$"$?$5(B"	"$(C1v(B")
   ("$(C$"$?$6(B"	"$(C1w(B")
   ("$(C$"$?$7(B"	"$(C1x(B")
   ("$(C$"$?$<(B"	"$(C1y(B")

   ("$(C$"$@(B"	"$(C1z(B")
   ("$(C$"$@$!(B"	"$(C1{(B")
   ("$(C$"$@$$(B"	"$(C1|(B")
   ("$(C$"$@$)(B"	"$(C1}(B")
   ("$(C$"$@$1(B"	"$(C1~(B")
   ("$(C$"$@$2(B"	"$(C2!(B")
   ("$(C$"$@$5(B"	"$(C2"(B")
   ("$(C$"$@$6(B"	"$(C2#(B")
   ("$(C$"$@$7(B"	"$(C2$(B")

   ("$(C$"$A(B"	"$(C2%(B")
   ("$(C$"$A$!(B"	"$(C2&(B")
   ("$(C$"$A$)(B"	"$(C2'(B")

   ("$(C$"$C(B"	"$(C2((B")
   ("$(C$"$C$!(B"	"$(C2)(B")
   ("$(C$"$C$"(B"	"$(C2*(B")
   ("$(C$"$C$$(B"	"$(C2+(B")
   ("$(C$"$C$)(B"	"$(C2,(B")
   ("$(C$"$C$1(B"	"$(C2-(B")
   ("$(C$"$C$2(B"	"$(C2.(B")
   ("$(C$"$C$5(B"	"$(C2/(B")
   ("$(C$"$C$6(B"	"$(C20(B")
   ("$(C$"$C$7(B"	"$(C21(B")

   ("$(C$"$D(B"	"$(C22(B")
   ("$(C$"$D$!(B"	"$(C23(B")
   ("$(C$"$D$$(B"	"$(C24(B")
   ("$(C$"$D$1(B"	"$(C25(B")
   ("$(C$"$D$5(B"	"$(C26(B")
   ("$(C$"$D$7(B"	"$(C27(B")

   ("$(C$"$E(B"	"$(C28(B")
   ("$(C$"$E$$(B"	"$(C29(B")
   ("$(C$"$E$)(B"	"$(C2:(B")
   ("$(C$"$E$5(B"	"$(C2;(B")
   ("$(C$"$E$6(B"	"$(C2<(B")
   ("$(C$"$E$<(B"	"$(C2=(B")

   ("$(C$"$F(B"	"$(C2>(B")

   ("$(C$"$G(B"	"$(C2?(B")
   ("$(C$"$G$!(B"	"$(C2@(B")
   ("$(C$"$G$$(B"	"$(C2A(B")
   ("$(C$"$G$$$>(B"	"$(C2B(B")
   ("$(C$"$G$)(B"	"$(C2C(B")
   ("$(C$"$G$1(B"	"$(C2D(B")
   ("$(C$"$G$2(B"	"$(C2E(B")
   ("$(C$"$G$5(B"	"$(C2F(B")
   ("$(C$"$G$7(B"	"$(C2G(B")
   ("$(C$"$G$8(B"	"$(C2H(B")
   ("$(C$"$G$:(B"	"$(C2I(B")

   ("$(C$"$G$?(B"	"$(C2J(B")
   ("$(C$"$G$?$!(B"	"$(C2K(B")
   ("$(C$"$G$?$)(B"	"$(C2L(B")
   ("$(C$"$G$?$6(B"	"$(C2M(B")
   ("$(C$"$G$?$7(B"	"$(C2N(B")

   ("$(C$"$G$@(B"	"$(C2O(B")
   ("$(C$"$G$@$!(B"	"$(C2P(B")
   ("$(C$"$G$@$7(B"	"$(C2Q(B")

   ("$(C$"$G$S(B"	"$(C2R(B")
   ("$(C$"$G$S$$(B"	"$(C2S(B")
   ("$(C$"$G$S$)(B"	"$(C2T(B")
   ("$(C$"$G$S$1(B"	"$(C2U(B")
   ("$(C$"$G$S$2(B"	"$(C2V(B")
   ("$(C$"$G$S$7(B"	"$(C2W(B")

   ("$(C$"$K(B"	"$(C2X(B")

   ("$(C$"$L(B"	"$(C2Y(B")
   ("$(C$"$L$!(B"	"$(C2Z(B")
   ("$(C$"$L$$(B"	"$(C2[(B")
   ("$(C$"$L$)(B"	"$(C2\(B")
   ("$(C$"$L$)$>(B"	"$(C2](B")
   ("$(C$"$L$1(B"	"$(C2^(B")
   ("$(C$"$L$2(B"	"$(C2_(B")
   ("$(C$"$L$5(B"	"$(C2`(B")
   ("$(C$"$L$7(B"	"$(C2a(B")
   ("$(C$"$L$8(B"	"$(C2b(B")

   ("$(C$"$L$C(B"	"$(C2c(B")
   ("$(C$"$L$C$)(B"	"$(C2d(B")
   ("$(C$"$L$C$6(B"	"$(C2e(B")
   ("$(C$"$L$C$7(B"	"$(C2f(B")

   ("$(C$"$L$D(B"	"$(C2g(B")
   ("$(C$"$L$D$!(B"	"$(C2h(B")
   ("$(C$"$L$D$$(B"	"$(C2i(B")
   ("$(C$"$L$D$)(B"	"$(C2j(B")
   ("$(C$"$L$D$1(B"	"$(C2k(B")
   ("$(C$"$L$D$2(B"	"$(C2l(B")
   ("$(C$"$L$D$6(B"	"$(C2m(B")

   ("$(C$"$L$S(B"	"$(C2n(B")
   ("$(C$"$L$S$$(B"	"$(C2o(B")
   ("$(C$"$L$S$)(B"	"$(C2p(B")
   ("$(C$"$L$S$1(B"	"$(C2q(B")
   ("$(C$"$L$S$2(B"	"$(C2r(B")

   ("$(C$"$P(B"	"$(C2s(B")

   ("$(C$"$Q(B"	"$(C2t(B")
   ("$(C$"$Q$!(B"	"$(C2u(B")
   ("$(C$"$Q$$(B"	"$(C2v(B")
   ("$(C$"$Q$$$>(B"	"$(C2w(B")
   ("$(C$"$Q$)(B"	"$(C2x(B")
   ("$(C$"$Q$)$1(B"	"$(C2y(B")
   ("$(C$"$Q$)$>(B"	"$(C2z(B")
   ("$(C$"$Q$1(B"	"$(C2{(B")
   ("$(C$"$Q$2(B"	"$(C2|(B")
   ("$(C$"$Q$5(B"	"$(C2}(B")
   ("$(C$"$Q$7(B"	"$(C2~(B")
   ("$(C$"$Q$<(B"	"$(C3!(B")

   ("$(C$"$S(B"	"$(C3"(B")
   ("$(C$"$S$!(B"	"$(C3#(B")
   ("$(C$"$S$$(B"	"$(C3$(B")
   ("$(C$"$S$)(B"	"$(C3%(B")
   ("$(C$"$S$1(B"	"$(C3&(B")
   ("$(C$"$S$2(B"	"$(C3'(B")
   ("$(C$"$S$5(B"	"$(C3((B")
   ("$(C$"$S$7(B"	"$(C3)(B")

   ("$(C$$$?(B"	"$(C3*(B")
   ("$(C$$$?$!(B"	"$(C3+(B")
   ("$(C$$$?$"(B"	"$(C3,(B")
   ("$(C$$$?$$(B"	"$(C3-(B")
   ("$(C$$$?$'(B"	"$(C3.(B")
   ("$(C$$$?$)(B"	"$(C3/(B")
   ("$(C$$$?$)$!(B"	"$(C30(B")
   ("$(C$$$?$)$1(B"	"$(C31(B")
   ("$(C$$$?$1(B"	"$(C32(B")
   ("$(C$$$?$2(B"	"$(C33(B")
   ("$(C$$$?$5(B"	"$(C34(B")
   ("$(C$$$?$6(B"	"$(C35(B")
   ("$(C$$$?$7(B"	"$(C36(B")
   ("$(C$$$?$8(B"	"$(C37(B")
   ("$(C$$$?$:(B"	"$(C38(B")
   ("$(C$$$?$<(B"	"$(C39(B")
   ("$(C$$$?$>(B"	"$(C3:(B")

   ("$(C$$$@(B"	"$(C3;(B")
   ("$(C$$$@$!(B"	"$(C3<(B")
   ("$(C$$$@$$(B"	"$(C3=(B")
   ("$(C$$$@$)(B"	"$(C3>(B")
   ("$(C$$$@$1(B"	"$(C3?(B")
   ("$(C$$$@$2(B"	"$(C3@(B")
   ("$(C$$$@$5(B"	"$(C3A(B")
   ("$(C$$$@$6(B"	"$(C3B(B")
   ("$(C$$$@$7(B"	"$(C3C(B")

   ("$(C$$$A(B"	"$(C3D(B")
   ("$(C$$$A$!(B"	"$(C3E(B")
   ("$(C$$$A$$(B"	"$(C3F(B")
   ("$(C$$$A$)(B"	"$(C3G(B")
   ("$(C$$$A$1(B"	"$(C3H(B")
   ("$(C$$$A$7(B"	"$(C3I(B")

   ("$(C$$$C(B"	"$(C3J(B")
   ("$(C$$$C$!(B"	"$(C3K(B")
   ("$(C$$$C$!$5(B"	"$(C3L(B")
   ("$(C$$$C$$(B"	"$(C3M(B")
   ("$(C$$$C$)(B"	"$(C3N(B")
   ("$(C$$$C$)$1(B"	"$(C3O(B")
   ("$(C$$$C$)$2(B"	"$(C3P(B")
   ("$(C$$$C$1(B"	"$(C3Q(B")
   ("$(C$$$C$2(B"	"$(C3R(B")
   ("$(C$$$C$5(B"	"$(C3S(B")
   ("$(C$$$C$6(B"	"$(C3T(B")
   ("$(C$$$C$7(B"	"$(C3U(B")
   ("$(C$$$C$>(B"	"$(C3V(B")

   ("$(C$$$D(B"	"$(C3W(B")
   ("$(C$$$D$!(B"	"$(C3X(B")
   ("$(C$$$D$$(B"	"$(C3Y(B")
   ("$(C$$$D$)(B"	"$(C3Z(B")
   ("$(C$$$D$1(B"	"$(C3[(B")
   ("$(C$$$D$2(B"	"$(C3\(B")
   ("$(C$$$D$5(B"	"$(C3](B")
   ("$(C$$$D$6(B"	"$(C3^(B")
   ("$(C$$$D$7(B"	"$(C3_(B")

   ("$(C$$$E(B"	"$(C3`(B")
   ("$(C$$$E$!(B"	"$(C3a(B")
   ("$(C$$$E$$(B"	"$(C3b(B")
   ("$(C$$$E$)(B"	"$(C3c(B")
   ("$(C$$$E$1(B"	"$(C3d(B")
   ("$(C$$$E$2(B"	"$(C3e(B")
   ("$(C$$$E$6(B"	"$(C3f(B")
   ("$(C$$$E$7(B"	"$(C3g(B")
   ("$(C$$$E$;(B"	"$(C3h(B")

   ("$(C$$$F(B"	"$(C3i(B")
   ("$(C$$$F$$(B"	"$(C3j(B")

   ("$(C$$$G(B"	"$(C3k(B")
   ("$(C$$$G$!(B"	"$(C3l(B")
   ("$(C$$$G$$(B"	"$(C3m(B")
   ("$(C$$$G$)(B"	"$(C3n(B")
   ("$(C$$$G$)$1(B"	"$(C3o(B")
   ("$(C$$$G$1(B"	"$(C3p(B")
   ("$(C$$$G$2(B"	"$(C3q(B")
   ("$(C$$$G$5(B"	"$(C3r(B")
   ("$(C$$$G$7(B"	"$(C3s(B")
   ("$(C$$$G$=(B"	"$(C3t(B")
   ("$(C$$$G$>(B"	"$(C3u(B")

   ("$(C$$$G$?(B"	"$(C3v(B")
   ("$(C$$$G$?$$(B"	"$(C3w(B")
   ("$(C$$$G$?$)(B"	"$(C3x(B")
   ("$(C$$$G$?$6(B"	"$(C3y(B")

   ("$(C$$$G$S(B"	"$(C3z(B")
   ("$(C$$$G$S$$(B"	"$(C3{(B")
   ("$(C$$$G$S$)(B"	"$(C3|(B")
   ("$(C$$$G$S$1(B"	"$(C3}(B")
   ("$(C$$$G$S$2(B"	"$(C3~(B")
   ("$(C$$$G$S$5(B"	"$(C4!(B")

   ("$(C$$$K(B"	"$(C4"(B")
   ("$(C$$$K$!(B"	"$(C4#(B")
   ("$(C$$$K$$(B"	"$(C4$(B")
   ("$(C$$$K$)(B"	"$(C4%(B")
   ("$(C$$$K$2(B"	"$(C4&(B")
   ("$(C$$$K$5(B"	"$(C4'(B")
   ("$(C$$$K$7(B"	"$(C4((B")

   ("$(C$$$L(B"	"$(C4)(B")
   ("$(C$$$L$!(B"	"$(C4*(B")
   ("$(C$$$L$$(B"	"$(C4+(B")
   ("$(C$$$L$'(B"	"$(C4,(B")
   ("$(C$$$L$)(B"	"$(C4-(B")
   ("$(C$$$L$1(B"	"$(C4.(B")
   ("$(C$$$L$2(B"	"$(C4/(B")
   ("$(C$$$L$5(B"	"$(C40(B")
   ("$(C$$$L$7(B"	"$(C41(B")

   ("$(C$$$L$C(B"	"$(C42(B")
   ("$(C$$$L$C$6(B"	"$(C43(B")

   ("$(C$$$L$D(B"	"$(C44(B")

   ("$(C$$$L$S(B"	"$(C45(B")
   ("$(C$$$L$S$$(B"	"$(C46(B")
   ("$(C$$$L$S$)(B"	"$(C47(B")
   ("$(C$$$L$S$1(B"	"$(C48(B")
   ("$(C$$$L$S$2(B"	"$(C49(B")

   ("$(C$$$P(B"	"$(C4:(B")
   ("$(C$$$P$!(B"	"$(C4;(B")
   ("$(C$$$P$)(B"	"$(C4<(B")
   ("$(C$$$P$1(B"	"$(C4=(B")
   ("$(C$$$P$2(B"	"$(C4>(B")
   ("$(C$$$P$7(B"	"$(C4?(B")

   ("$(C$$$Q(B"	"$(C4@(B")
   ("$(C$$$Q$!(B"	"$(C4A(B")
   ("$(C$$$Q$$(B"	"$(C4B(B")
   ("$(C$$$Q$)(B"	"$(C4C(B")
   ("$(C$$$Q$)$!(B"	"$(C4D(B")
   ("$(C$$$Q$)$1(B"	"$(C4E(B")
   ("$(C$$$Q$1(B"	"$(C4F(B")
   ("$(C$$$Q$2(B"	"$(C4G(B")
   ("$(C$$$Q$5(B"	"$(C4H(B")
   ("$(C$$$Q$7(B"	"$(C4I(B")
   ("$(C$$$Q$8(B"	"$(C4J(B")
   ("$(C$$$Q$=(B"	"$(C4K(B")

   ("$(C$$$Q$S(B"	"$(C4L(B")
   ("$(C$$$Q$S$$(B"	"$(C4M(B")
   ("$(C$$$Q$S$)(B"	"$(C4N(B")

   ("$(C$$$S(B"	"$(C4O(B")
   ("$(C$$$S$!(B"	"$(C4P(B")
   ("$(C$$$S$$(B"	"$(C4Q(B")
   ("$(C$$$S$)(B"	"$(C4R(B")
   ("$(C$$$S$)$1(B"	"$(C4S(B")
   ("$(C$$$S$1(B"	"$(C4T(B")
   ("$(C$$$S$2(B"	"$(C4U(B")
   ("$(C$$$S$5(B"	"$(C4V(B")
   ("$(C$$$S$7(B"	"$(C4W(B")
   ("$(C$$$S$=(B"	"$(C4X(B")

   ("$(C$'$?(B"	"$(C4Y(B")
   ("$(C$'$?$!(B"	"$(C4Z(B")
   ("$(C$'$?$"(B"	"$(C4[(B")
   ("$(C$'$?$$(B"	"$(C4\(B")
   ("$(C$'$?$'(B"	"$(C4](B")
   ("$(C$'$?$)(B"	"$(C4^(B")
   ("$(C$'$?$)$!(B"	"$(C4_(B")
   ("$(C$'$?$)$1(B"	"$(C4`(B")
   ("$(C$'$?$)$2(B"	"$(C4a(B")
   ("$(C$'$?$)$>(B"	"$(C4b(B")
   ("$(C$'$?$1(B"	"$(C4c(B")
   ("$(C$'$?$2(B"	"$(C4d(B")
   ("$(C$'$?$5(B"	"$(C4e(B")
   ("$(C$'$?$6(B"	"$(C4f(B")
   ("$(C$'$?$7(B"	"$(C4g(B")
   ("$(C$'$?$8(B"	"$(C4h(B")
   ("$(C$'$?$:(B"	"$(C4i(B")
   ("$(C$'$?$>(B"	"$(C4j(B")

   ("$(C$'$@(B"	"$(C4k(B")
   ("$(C$'$@$!(B"	"$(C4l(B")
   ("$(C$'$@$$(B"	"$(C4m(B")
   ("$(C$'$@$)(B"	"$(C4n(B")
   ("$(C$'$@$1(B"	"$(C4o(B")
   ("$(C$'$@$2(B"	"$(C4p(B")
   ("$(C$'$@$5(B"	"$(C4q(B")
   ("$(C$'$@$6(B"	"$(C4r(B")
   ("$(C$'$@$7(B"	"$(C4s(B")

   ("$(C$'$A(B"	"$(C4t(B")

   ("$(C$'$C(B"	"$(C4u(B")
   ("$(C$'$C$!(B"	"$(C4v(B")
   ("$(C$'$C$"(B"	"$(C4w(B")
   ("$(C$'$C$$(B"	"$(C4x(B")
   ("$(C$'$C$'(B"	"$(C4y(B")
   ("$(C$'$C$)(B"	"$(C4z(B")
   ("$(C$'$C$)$1(B"	"$(C4{(B")
   ("$(C$'$C$)$2(B"	"$(C4|(B")
   ("$(C$'$C$1(B"	"$(C4}(B")
   ("$(C$'$C$2(B"	"$(C4~(B")
   ("$(C$'$C$5(B"	"$(C5!(B")
   ("$(C$'$C$7(B"	"$(C5"(B")
   ("$(C$'$C$:(B"	"$(C5#(B")
   ("$(C$'$C$=(B"	"$(C5$(B")

   ("$(C$'$D(B"	"$(C5%(B")
   ("$(C$'$D$!(B"	"$(C5&(B")
   ("$(C$'$D$$(B"	"$(C5'(B")
   ("$(C$'$D$)(B"	"$(C5((B")
   ("$(C$'$D$1(B"	"$(C5)(B")
   ("$(C$'$D$2(B"	"$(C5*(B")
   ("$(C$'$D$5(B"	"$(C5+(B")
   ("$(C$'$D$6(B"	"$(C5,(B")
   ("$(C$'$D$7(B"	"$(C5-(B")

   ("$(C$'$E(B"	"$(C5.(B")
   ("$(C$'$E$$(B"	"$(C5/(B")
   ("$(C$'$E$)(B"	"$(C50(B")
   ("$(C$'$E$6(B"	"$(C51(B")
   ("$(C$'$E$7(B"	"$(C52(B")

   ("$(C$'$F(B"	"$(C53(B")
   ("$(C$'$F$$(B"	"$(C54(B")

   ("$(C$'$G(B"	"$(C55(B")
   ("$(C$'$G$!(B"	"$(C56(B")
   ("$(C$'$G$$(B"	"$(C57(B")
   ("$(C$'$G$'(B"	"$(C58(B")
   ("$(C$'$G$)(B"	"$(C59(B")
   ("$(C$'$G$)$1(B"	"$(C5:(B")
   ("$(C$'$G$)$5(B"	"$(C5;(B")
   ("$(C$'$G$1(B"	"$(C5<(B")
   ("$(C$'$G$2(B"	"$(C5=(B")
   ("$(C$'$G$5(B"	"$(C5>(B")
   ("$(C$'$G$7(B"	"$(C5?(B")
   ("$(C$'$G$:(B"	"$(C5@(B")
   ("$(C$'$G$<(B"	"$(C5A(B")

   ("$(C$'$G$?(B"	"$(C5B(B")
   ("$(C$'$G$?$$(B"	"$(C5C(B")
   ("$(C$'$G$?$)(B"	"$(C5D(B")

   ("$(C$'$G$@(B"	"$(C5E(B")
   ("$(C$'$G$@$6(B"	"$(C5F(B")

   ("$(C$'$G$S(B"	"$(C5G(B")
   ("$(C$'$G$S$$(B"	"$(C5H(B")
   ("$(C$'$G$S$)(B"	"$(C5I(B")
   ("$(C$'$G$S$1(B"	"$(C5J(B")
   ("$(C$'$G$S$2(B"	"$(C5K(B")
   ("$(C$'$G$S$5(B"	"$(C5L(B")

   ("$(C$'$K(B"	"$(C5M(B")

   ("$(C$'$L(B"	"$(C5N(B")
   ("$(C$'$L$!(B"	"$(C5O(B")
   ("$(C$'$L$$(B"	"$(C5P(B")
   ("$(C$'$L$)(B"	"$(C5Q(B")
   ("$(C$'$L$1(B"	"$(C5R(B")
   ("$(C$'$L$2(B"	"$(C5S(B")
   ("$(C$'$L$5(B"	"$(C5T(B")
   ("$(C$'$L$7(B"	"$(C5U(B")

   ("$(C$'$L$C(B"	"$(C5V(B")
   ("$(C$'$L$C$6(B"	"$(C5W(B")

   ("$(C$'$L$D(B"	"$(C5X(B")
   ("$(C$'$L$D$7(B"	"$(C5Y(B")

   ("$(C$'$L$S(B"	"$(C5Z(B")
   ("$(C$'$L$S$$(B"	"$(C5[(B")
   ("$(C$'$L$S$)(B"	"$(C5\(B")
   ("$(C$'$L$S$2(B"	"$(C5](B")
   ("$(C$'$L$S$5(B"	"$(C5^(B")
   ("$(C$'$L$S$7(B"	"$(C5_(B")

   ("$(C$'$P(B"	"$(C5`(B")
   ("$(C$'$P$$(B"	"$(C5a(B")
   ("$(C$'$P$)(B"	"$(C5b(B")
   ("$(C$'$P$1(B"	"$(C5c(B")
   ("$(C$'$P$7(B"	"$(C5d(B")

   ("$(C$'$Q(B"	"$(C5e(B")
   ("$(C$'$Q$!(B"	"$(C5f(B")
   ("$(C$'$Q$$(B"	"$(C5g(B")
   ("$(C$'$Q$'(B"	"$(C5h(B")
   ("$(C$'$Q$)(B"	"$(C5i(B")
   ("$(C$'$Q$)$1(B"	"$(C5j(B")
   ("$(C$'$Q$1(B"	"$(C5k(B")
   ("$(C$'$Q$2(B"	"$(C5l(B")
   ("$(C$'$Q$5(B"	"$(C5m(B")
   ("$(C$'$Q$7(B"	"$(C5n(B")

   ("$(C$'$Q$S(B"	"$(C5o(B")

   ("$(C$'$S(B"	"$(C5p(B")
   ("$(C$'$S$!(B"	"$(C5q(B")
   ("$(C$'$S$$(B"	"$(C5r(B")
   ("$(C$'$S$'(B"	"$(C5s(B")
   ("$(C$'$S$)(B"	"$(C5t(B")
   ("$(C$'$S$1(B"	"$(C5u(B")
   ("$(C$'$S$2(B"	"$(C5v(B")
   ("$(C$'$S$5(B"	"$(C5w(B")
   ("$(C$'$S$6(B"	"$(C5x(B")
   ("$(C$'$S$7(B"	"$(C5y(B")
   ("$(C$'$S$8(B"	"$(C5z(B")

   ("$(C$($?(B"	"$(C5{(B")
   ("$(C$($?$!(B"	"$(C5|(B")
   ("$(C$($?$$(B"	"$(C5}(B")
   ("$(C$($?$)(B"	"$(C5~(B")
   ("$(C$($?$1(B"	"$(C6!(B")
   ("$(C$($?$2(B"	"$(C6"(B")
   ("$(C$($?$5(B"	"$(C6#(B")
   ("$(C$($?$6(B"	"$(C6$(B")
   ("$(C$($?$7(B"	"$(C6%(B")
   ("$(C$($?$>(B"	"$(C6&(B")

   ("$(C$($@(B"	"$(C6'(B")
   ("$(C$($@$!(B"	"$(C6((B")
   ("$(C$($@$$(B"	"$(C6)(B")
   ("$(C$($@$)(B"	"$(C6*(B")
   ("$(C$($@$1(B"	"$(C6+(B")
   ("$(C$($@$2(B"	"$(C6,(B")
   ("$(C$($@$5(B"	"$(C6-(B")
   ("$(C$($@$6(B"	"$(C6.(B")
   ("$(C$($@$7(B"	"$(C6/(B")

   ("$(C$($C(B"	"$(C60(B")
   ("$(C$($C$!(B"	"$(C61(B")
   ("$(C$($C$$(B"	"$(C62(B")
   ("$(C$($C$)(B"	"$(C63(B")
   ("$(C$($C$)$1(B"	"$(C64(B")
   ("$(C$($C$)$2(B"	"$(C65(B")
   ("$(C$($C$1(B"	"$(C66(B")
   ("$(C$($C$2(B"	"$(C67(B")
   ("$(C$($C$5(B"	"$(C68(B")
   ("$(C$($C$6(B"	"$(C69(B")
   ("$(C$($C$7(B"	"$(C6:(B")
   ("$(C$($C$>(B"	"$(C6;(B")

   ("$(C$($D(B"	"$(C6<(B")
   ("$(C$($D$!(B"	"$(C6=(B")
   ("$(C$($D$$(B"	"$(C6>(B")
   ("$(C$($D$)(B"	"$(C6?(B")
   ("$(C$($D$1(B"	"$(C6@(B")
   ("$(C$($D$2(B"	"$(C6A(B")
   ("$(C$($D$5(B"	"$(C6B(B")
   ("$(C$($D$6(B"	"$(C6C(B")
   ("$(C$($D$7(B"	"$(C6D(B")

   ("$(C$($E(B"	"$(C6E(B")
   ("$(C$($E$6(B"	"$(C6F(B")

   ("$(C$($G(B"	"$(C6G(B")
   ("$(C$($G$!(B"	"$(C6H(B")
   ("$(C$($G$$(B"	"$(C6I(B")
   ("$(C$($G$)(B"	"$(C6J(B")
   ("$(C$($G$7(B"	"$(C6K(B")

   ("$(C$($G$?(B"	"$(C6L(B")
   ("$(C$($G$?$)(B"	"$(C6M(B")

   ("$(C$($G$@(B"	"$(C6N(B")

   ("$(C$($G$S(B"	"$(C6O(B")
   ("$(C$($G$S$$(B"	"$(C6P(B")

   ("$(C$($L(B"	"$(C6Q(B")
   ("$(C$($L$!(B"	"$(C6R(B")
   ("$(C$($L$$(B"	"$(C6S(B")
   ("$(C$($L$)(B"	"$(C6T(B")
   ("$(C$($L$)$>(B"	"$(C6U(B")
   ("$(C$($L$1(B"	"$(C6V(B")
   ("$(C$($L$7(B"	"$(C6W(B")

   ("$(C$($L$D(B"	"$(C6X(B")

   ("$(C$($L$S(B"	"$(C6Y(B")
   ("$(C$($L$S$$(B"	"$(C6Z(B")
   ("$(C$($L$S$)(B"	"$(C6[(B")
   ("$(C$($L$S$1(B"	"$(C6\(B")
   ("$(C$($L$S$2(B"	"$(C6](B")
   ("$(C$($L$S$7(B"	"$(C6^(B")

   ("$(C$($Q(B"	"$(C6_(B")
   ("$(C$($Q$!(B"	"$(C6`(B")
   ("$(C$($Q$$(B"	"$(C6a(B")
   ("$(C$($Q$'(B"	"$(C6b(B")
   ("$(C$($Q$)(B"	"$(C6c(B")
   ("$(C$($Q$1(B"	"$(C6d(B")
   ("$(C$($Q$2(B"	"$(C6e(B")
   ("$(C$($Q$5(B"	"$(C6f(B")

   ("$(C$($Q$S(B"	"$(C6g(B")
   ("$(C$($Q$S$$(B"	"$(C6h(B")
   ("$(C$($Q$S$)(B"	"$(C6i(B")
   ("$(C$($Q$S$1(B"	"$(C6j(B")
   ("$(C$($Q$S$2(B"	"$(C6k(B")

   ("$(C$($S(B"	"$(C6l(B")
   ("$(C$($S$$(B"	"$(C6m(B")
   ("$(C$($S$)(B"	"$(C6n(B")
   ("$(C$($S$1(B"	"$(C6o(B")
   ("$(C$($S$2(B"	"$(C6p(B")
   ("$(C$($S$5(B"	"$(C6q(B")
   ("$(C$($S$7(B"	"$(C6r(B")

   ("$(C$)$?(B"	"$(C6s(B")
   ("$(C$)$?$!(B"	"$(C6t(B")
   ("$(C$)$?$$(B"	"$(C6u(B")
   ("$(C$)$?$)(B"	"$(C6v(B")
   ("$(C$)$?$1(B"	"$(C6w(B")
   ("$(C$)$?$2(B"	"$(C6x(B")
   ("$(C$)$?$5(B"	"$(C6y(B")
   ("$(C$)$?$6(B"	"$(C6z(B")
   ("$(C$)$?$7(B"	"$(C6{(B")
   ("$(C$)$?$8(B"	"$(C6|(B")
   ("$(C$)$?$=(B"	"$(C6}(B")
   ("$(C$)$?$>(B"	"$(C6~(B")

   ("$(C$)$@(B"	"$(C7!(B")
   ("$(C$)$@$!(B"	"$(C7"(B")
   ("$(C$)$@$$(B"	"$(C7#(B")
   ("$(C$)$@$)(B"	"$(C7$(B")
   ("$(C$)$@$1(B"	"$(C7%(B")
   ("$(C$)$@$2(B"	"$(C7&(B")
   ("$(C$)$@$5(B"	"$(C7'(B")
   ("$(C$)$@$6(B"	"$(C7((B")
   ("$(C$)$@$7(B"	"$(C7)(B")

   ("$(C$)$A(B"	"$(C7*(B")
   ("$(C$)$A$!(B"	"$(C7+(B")
   ("$(C$)$A$$(B"	"$(C7,(B")
   ("$(C$)$A$5(B"	"$(C7-(B")
   ("$(C$)$A$7(B"	"$(C7.(B")

   ("$(C$)$C(B"	"$(C7/(B")
   ("$(C$)$C$!(B"	"$(C70(B")
   ("$(C$)$C$$(B"	"$(C71(B")
   ("$(C$)$C$)(B"	"$(C72(B")
   ("$(C$)$C$1(B"	"$(C73(B")
   ("$(C$)$C$2(B"	"$(C74(B")
   ("$(C$)$C$5(B"	"$(C75(B")
   ("$(C$)$C$6(B"	"$(C76(B")
   ("$(C$)$C$7(B"	"$(C77(B")
   ("$(C$)$C$>(B"	"$(C78(B")

   ("$(C$)$D(B"	"$(C79(B")
   ("$(C$)$D$!(B"	"$(C7:(B")
   ("$(C$)$D$$(B"	"$(C7;(B")
   ("$(C$)$D$)(B"	"$(C7<(B")
   ("$(C$)$D$1(B"	"$(C7=(B")
   ("$(C$)$D$2(B"	"$(C7>(B")
   ("$(C$)$D$5(B"	"$(C7?(B")
   ("$(C$)$D$7(B"	"$(C7@(B")

   ("$(C$)$E(B"	"$(C7A(B")
   ("$(C$)$E$!(B"	"$(C7B(B")
   ("$(C$)$E$$(B"	"$(C7C(B")
   ("$(C$)$E$)(B"	"$(C7D(B")
   ("$(C$)$E$1(B"	"$(C7E(B")
   ("$(C$)$E$2(B"	"$(C7F(B")
   ("$(C$)$E$5(B"	"$(C7G(B")
   ("$(C$)$E$6(B"	"$(C7H(B")
   ("$(C$)$E$7(B"	"$(C7I(B")

   ("$(C$)$F(B"	"$(C7J(B")
   ("$(C$)$F$$(B"	"$(C7K(B")
   ("$(C$)$F$2(B"	"$(C7L(B")
   ("$(C$)$F$5(B"	"$(C7M(B")

   ("$(C$)$G(B"	"$(C7N(B")
   ("$(C$)$G$!(B"	"$(C7O(B")
   ("$(C$)$G$$(B"	"$(C7P(B")
   ("$(C$)$G$)(B"	"$(C7Q(B")
   ("$(C$)$G$1(B"	"$(C7R(B")
   ("$(C$)$G$2(B"	"$(C7S(B")
   ("$(C$)$G$5(B"	"$(C7T(B")
   ("$(C$)$G$7(B"	"$(C7U(B")

   ("$(C$)$G$?(B"	"$(C7V(B")
   ("$(C$)$G$?$$(B"	"$(C7W(B")
   ("$(C$)$G$?$7(B"	"$(C7X(B")

   ("$(C$)$G$@$6(B"	"$(C7Y(B")

   ("$(C$)$G$S(B"	"$(C7Z(B")
   ("$(C$)$G$S$$(B"	"$(C7[(B")
   ("$(C$)$G$S$)(B"	"$(C7\(B")
   ("$(C$)$G$S$1(B"	"$(C7](B")
   ("$(C$)$G$S$2(B"	"$(C7^(B")
   ("$(C$)$G$S$5(B"	"$(C7_(B")
   ("$(C$)$G$S$7(B"	"$(C7`(B")

   ("$(C$)$K(B"	"$(C7a(B")
   ("$(C$)$K$$(B"	"$(C7b(B")
   ("$(C$)$K$)(B"	"$(C7c(B")
   ("$(C$)$K$2(B"	"$(C7d(B")
   ("$(C$)$K$5(B"	"$(C7e(B")
   ("$(C$)$K$7(B"	"$(C7f(B")

   ("$(C$)$L(B"	"$(C7g(B")
   ("$(C$)$L$!(B"	"$(C7h(B")
   ("$(C$)$L$$(B"	"$(C7i(B")
   ("$(C$)$L$)(B"	"$(C7j(B")
   ("$(C$)$L$1(B"	"$(C7k(B")
   ("$(C$)$L$2(B"	"$(C7l(B")
   ("$(C$)$L$5(B"	"$(C7m(B")
   ("$(C$)$L$7(B"	"$(C7n(B")

   ("$(C$)$L$C(B"	"$(C7o(B")
   ("$(C$)$L$C$6(B"	"$(C7p(B")

   ("$(C$)$L$D(B"	"$(C7q(B")

   ("$(C$)$L$S(B"	"$(C7r(B")
   ("$(C$)$L$S$!(B"	"$(C7s(B")
   ("$(C$)$L$S$$(B"	"$(C7t(B")
   ("$(C$)$L$S$)(B"	"$(C7u(B")
   ("$(C$)$L$S$1(B"	"$(C7v(B")
   ("$(C$)$L$S$5(B"	"$(C7w(B")
   ("$(C$)$L$S$7(B"	"$(C7x(B")

   ("$(C$)$P(B"	"$(C7y(B")
   ("$(C$)$P$!(B"	"$(C7z(B")
   ("$(C$)$P$$(B"	"$(C7{(B")
   ("$(C$)$P$)(B"	"$(C7|(B")
   ("$(C$)$P$1(B"	"$(C7}(B")
   ("$(C$)$P$2(B"	"$(C7~(B")
   ("$(C$)$P$5(B"	"$(C8!(B")
   ("$(C$)$P$7(B"	"$(C8"(B")

   ("$(C$)$Q(B"	"$(C8#(B")
   ("$(C$)$Q$!(B"	"$(C8$(B")
   ("$(C$)$Q$$(B"	"$(C8%(B")
   ("$(C$)$Q$)(B"	"$(C8&(B")
   ("$(C$)$Q$1(B"	"$(C8'(B")
   ("$(C$)$Q$2(B"	"$(C8((B")
   ("$(C$)$Q$5(B"	"$(C8)(B")
   ("$(C$)$Q$7(B"	"$(C8*(B")
   ("$(C$)$Q$8(B"	"$(C8+(B")
   ("$(C$)$Q$<(B"	"$(C8,(B")
   ("$(C$)$Q$=(B"	"$(C8-(B")

   ("$(C$)$S(B"	"$(C8.(B")
   ("$(C$)$S$!(B"	"$(C8/(B")
   ("$(C$)$S$$(B"	"$(C80(B")
   ("$(C$)$S$)(B"	"$(C81(B")
   ("$(C$)$S$1(B"	"$(C82(B")
   ("$(C$)$S$2(B"	"$(C83(B")
   ("$(C$)$S$5(B"	"$(C84(B")
   ("$(C$)$S$7(B"	"$(C85(B")

   ("$(C$1$?(B"	"$(C86(B")
   ("$(C$1$?$!(B"	"$(C87(B")
   ("$(C$1$?$$(B"	"$(C88(B")
   ("$(C$1$?$$$>(B"	"$(C89(B")
   ("$(C$1$?$'(B"	"$(C8:(B")
   ("$(C$1$?$)(B"	"$(C8;(B")
   ("$(C$1$?$)$!(B"	"$(C8<(B")
   ("$(C$1$?$)$1(B"	"$(C8=(B")
   ("$(C$1$?$1(B"	"$(C8>(B")
   ("$(C$1$?$2(B"	"$(C8?(B")
   ("$(C$1$?$5(B"	"$(C8@(B")
   ("$(C$1$?$7(B"	"$(C8A(B")
   ("$(C$1$?$8(B"	"$(C8B(B")
   ("$(C$1$?$<(B"	"$(C8C(B")
   ("$(C$1$?$>(B"	"$(C8D(B")

   ("$(C$1$@(B"	"$(C8E(B")
   ("$(C$1$@$!(B"	"$(C8F(B")
   ("$(C$1$@$$(B"	"$(C8G(B")
   ("$(C$1$@$)(B"	"$(C8H(B")
   ("$(C$1$@$1(B"	"$(C8I(B")
   ("$(C$1$@$2(B"	"$(C8J(B")
   ("$(C$1$@$5(B"	"$(C8K(B")
   ("$(C$1$@$6(B"	"$(C8L(B")
   ("$(C$1$@$7(B"	"$(C8M(B")
   ("$(C$1$@$8(B"	"$(C8N(B")

   ("$(C$1$A(B"	"$(C8O(B")
   ("$(C$1$A$!(B"	"$(C8P(B")
   ("$(C$1$A$)(B"	"$(C8Q(B")
   ("$(C$1$A$7(B"	"$(C8R(B")

   ("$(C$1$C(B"	"$(C8S(B")
   ("$(C$1$C$!(B"	"$(C8T(B")
   ("$(C$1$C$$(B"	"$(C8U(B")
   ("$(C$1$C$)(B"	"$(C8V(B")
   ("$(C$1$C$)$1(B"	"$(C8W(B")
   ("$(C$1$C$1(B"	"$(C8X(B")
   ("$(C$1$C$2(B"	"$(C8Y(B")
   ("$(C$1$C$5(B"	"$(C8Z(B")
   ("$(C$1$C$7(B"	"$(C8[(B")
   ("$(C$1$C$8(B"	"$(C8\(B")
   ("$(C$1$C$>(B"	"$(C8](B")

   ("$(C$1$D(B"	"$(C8^(B")
   ("$(C$1$D$!(B"	"$(C8_(B")
   ("$(C$1$D$$(B"	"$(C8`(B")
   ("$(C$1$D$)(B"	"$(C8a(B")
   ("$(C$1$D$1(B"	"$(C8b(B")
   ("$(C$1$D$2(B"	"$(C8c(B")
   ("$(C$1$D$5(B"	"$(C8d(B")
   ("$(C$1$D$6(B"	"$(C8e(B")
   ("$(C$1$D$7(B"	"$(C8f(B")

   ("$(C$1$E(B"	"$(C8g(B")
   ("$(C$1$E$!(B"	"$(C8h(B")
   ("$(C$1$E$$(B"	"$(C8i(B")
   ("$(C$1$E$)(B"	"$(C8j(B")
   ("$(C$1$E$5(B"	"$(C8k(B")
   ("$(C$1$E$6(B"	"$(C8l(B")
   ("$(C$1$E$7(B"	"$(C8m(B")
   ("$(C$1$E$:(B"	"$(C8n(B")

   ("$(C$1$F(B"	"$(C8o(B")

   ("$(C$1$G(B"	"$(C8p(B")
   ("$(C$1$G$!(B"	"$(C8q(B")
   ("$(C$1$G$!$5(B"	"$(C8r(B")
   ("$(C$1$G$$(B"	"$(C8s(B")
   ("$(C$1$G$)(B"	"$(C8t(B")
   ("$(C$1$G$)$1(B"	"$(C8u(B")
   ("$(C$1$G$1(B"	"$(C8v(B")
   ("$(C$1$G$2(B"	"$(C8w(B")
   ("$(C$1$G$5(B"	"$(C8x(B")
   ("$(C$1$G$7(B"	"$(C8y(B")

   ("$(C$1$G$?(B"	"$(C8z(B")
   ("$(C$1$G$?$$(B"	"$(C8{(B")
   ("$(C$1$G$?$6(B"	"$(C8|(B")
   ("$(C$1$G$?$7(B"	"$(C8}(B")

   ("$(C$1$G$S(B"	"$(C8~(B")
   ("$(C$1$G$S$$(B"	"$(C9!(B")
   ("$(C$1$G$S$)(B"	"$(C9"(B")
   ("$(C$1$G$S$2(B"	"$(C9#(B")
   ("$(C$1$G$S$5(B"	"$(C9$(B")
   ("$(C$1$G$S$7(B"	"$(C9%(B")

   ("$(C$1$K(B"	"$(C9&(B")
   ("$(C$1$K$$(B"	"$(C9'(B")
   ("$(C$1$K$)(B"	"$(C9((B")
   ("$(C$1$K$2(B"	"$(C9)(B")
   ("$(C$1$K$5(B"	"$(C9*(B")

   ("$(C$1$L(B"	"$(C9+(B")
   ("$(C$1$L$!(B"	"$(C9,(B")
   ("$(C$1$L$"(B"	"$(C9-(B")
   ("$(C$1$L$$(B"	"$(C9.(B")
   ("$(C$1$L$'(B"	"$(C9/(B")
   ("$(C$1$L$)(B"	"$(C90(B")
   ("$(C$1$L$)$!(B"	"$(C91(B")
   ("$(C$1$L$)$1(B"	"$(C92(B")
   ("$(C$1$L$1(B"	"$(C93(B")
   ("$(C$1$L$2(B"	"$(C94(B")
   ("$(C$1$L$5(B"	"$(C95(B")
   ("$(C$1$L$7(B"	"$(C96(B")
   ("$(C$1$L$<(B"	"$(C97(B")
   ("$(C$1$L$>(B"	"$(C98(B")

   ("$(C$1$L$C(B"	"$(C99(B")
   ("$(C$1$L$C$$(B"	"$(C9:(B")
   ("$(C$1$L$C$)(B"	"$(C9;(B")
   ("$(C$1$L$C$2(B"	"$(C9<(B")
   ("$(C$1$L$C$5(B"	"$(C9=(B")

   ("$(C$1$L$D(B"	"$(C9>(B")

   ("$(C$1$L$S(B"	"$(C9?(B")
   ("$(C$1$L$S$$(B"	"$(C9@(B")
   ("$(C$1$L$S$)(B"	"$(C9A(B")

   ("$(C$1$P(B"	"$(C9B(B")
   ("$(C$1$P$$(B"	"$(C9C(B")
   ("$(C$1$P$)(B"	"$(C9D(B")
   ("$(C$1$P$1(B"	"$(C9E(B")
   ("$(C$1$P$5(B"	"$(C9F(B")

   ("$(C$1$Q(B"	"$(C9G(B")
   ("$(C$1$Q$$(B"	"$(C9H(B")
   ("$(C$1$Q$)(B"	"$(C9I(B")
   ("$(C$1$Q$1(B"	"$(C9J(B")
   ("$(C$1$Q$5(B"	"$(C9K(B")

   ("$(C$1$S(B"	"$(C9L(B")
   ("$(C$1$S$!(B"	"$(C9M(B")
   ("$(C$1$S$$(B"	"$(C9N(B")
   ("$(C$1$S$'(B"	"$(C9O(B")
   ("$(C$1$S$)(B"	"$(C9P(B")
   ("$(C$1$S$)$1(B"	"$(C9Q(B")
   ("$(C$1$S$1(B"	"$(C9R(B")
   ("$(C$1$S$2(B"	"$(C9S(B")
   ("$(C$1$S$5(B"	"$(C9T(B")
   ("$(C$1$S$6(B"	"$(C9U(B")
   ("$(C$1$S$7(B"	"$(C9V(B")
   ("$(C$1$S$:(B"	"$(C9W(B")
   ("$(C$1$S$<(B"	"$(C9X(B")

   ("$(C$2$?(B"	"$(C9Y(B")
   ("$(C$2$?$!(B"	"$(C9Z(B")
   ("$(C$2$?$!$5(B"	"$(C9\(B")
   ("$(C$2$?$"(B"	"$(C9[(B")
   ("$(C$2$?$$(B"	"$(C9](B")
   ("$(C$2$?$'(B"	"$(C9^(B")
   ("$(C$2$?$)(B"	"$(C9_(B")
   ("$(C$2$?$)$!(B"	"$(C9`(B")
   ("$(C$2$?$)$1(B"	"$(C9a(B")
   ("$(C$2$?$)$2(B"	"$(C9b(B")
   ("$(C$2$?$1(B"	"$(C9c(B")
   ("$(C$2$?$2(B"	"$(C9d(B")
   ("$(C$2$?$5(B"	"$(C9e(B")
   ("$(C$2$?$7(B"	"$(C9f(B")
   ("$(C$2$?$<(B"	"$(C9g(B")

   ("$(C$2$@(B"	"$(C9h(B")
   ("$(C$2$@$!(B"	"$(C9i(B")
   ("$(C$2$@$$(B"	"$(C9j(B")
   ("$(C$2$@$)(B"	"$(C9k(B")
   ("$(C$2$@$1(B"	"$(C9l(B")
   ("$(C$2$@$2(B"	"$(C9m(B")
   ("$(C$2$@$5(B"	"$(C9n(B")
   ("$(C$2$@$6(B"	"$(C9o(B")
   ("$(C$2$@$7(B"	"$(C9p(B")
   ("$(C$2$@$<(B"	"$(C9q(B")

   ("$(C$2$A(B"	"$(C9r(B")
   ("$(C$2$A$!(B"	"$(C9s(B")
   ("$(C$2$A$$(B"	"$(C9t(B")
   ("$(C$2$A$2(B"	"$(C9u(B")

   ("$(C$2$C(B"	"$(C9v(B")
   ("$(C$2$C$!(B"	"$(C9w(B")
   ("$(C$2$C$$(B"	"$(C9x(B")
   ("$(C$2$C$'(B"	"$(C9y(B")
   ("$(C$2$C$)(B"	"$(C9z(B")
   ("$(C$2$C$)$1(B"	"$(C9{(B")
   ("$(C$2$C$1(B"	"$(C9|(B")
   ("$(C$2$C$2(B"	"$(C9}(B")
   ("$(C$2$C$5(B"	"$(C9~(B")
   ("$(C$2$C$7(B"	"$(C:!(B")
   ("$(C$2$C$8(B"	"$(C:"(B")

   ("$(C$2$D(B"	"$(C:#(B")
   ("$(C$2$D$!(B"	"$(C:$(B")
   ("$(C$2$D$$(B"	"$(C:%(B")
   ("$(C$2$D$'(B"	"$(C:&(B")
   ("$(C$2$D$)(B"	"$(C:'(B")
   ("$(C$2$D$1(B"	"$(C:((B")
   ("$(C$2$D$2(B"	"$(C:)(B")
   ("$(C$2$D$5(B"	"$(C:*(B")
   ("$(C$2$D$6(B"	"$(C:+(B")
   ("$(C$2$D$7(B"	"$(C:,(B")

   ("$(C$2$E(B"	"$(C:-(B")
   ("$(C$2$E$!(B"	"$(C:.(B")
   ("$(C$2$E$$(B"	"$(C:/(B")
   ("$(C$2$E$)(B"	"$(C:0(B")
   ("$(C$2$E$2(B"	"$(C:1(B")
   ("$(C$2$E$5(B"	"$(C:2(B")
   ("$(C$2$E$6(B"	"$(C:3(B")
   ("$(C$2$E$7(B"	"$(C:4(B")
   ("$(C$2$E$<(B"	"$(C:5(B")

   ("$(C$2$F(B"	"$(C:6(B")
   ("$(C$2$F$$(B"	"$(C:7(B")

   ("$(C$2$G(B"	"$(C:8(B")
   ("$(C$2$G$!(B"	"$(C:9(B")
   ("$(C$2$G$"(B"	"$(C::(B")
   ("$(C$2$G$$(B"	"$(C:;(B")
   ("$(C$2$G$)(B"	"$(C:<(B")
   ("$(C$2$G$1(B"	"$(C:=(B")
   ("$(C$2$G$2(B"	"$(C:>(B")
   ("$(C$2$G$5(B"	"$(C:?(B")
   ("$(C$2$G$7(B"	"$(C:@(B")

   ("$(C$2$G$?(B"	"$(C:A(B")
   ("$(C$2$G$?$$(B"	"$(C:B(B")
   ("$(C$2$G$?$6(B"	"$(C:C(B")

   ("$(C$2$G$@(B"	"$(C:D(B")
   ("$(C$2$G$@$6(B"	"$(C:E(B")

   ("$(C$2$G$S(B"	"$(C:F(B")
   ("$(C$2$G$S$!(B"	"$(C:G(B")
   ("$(C$2$G$S$$(B"	"$(C:H(B")
   ("$(C$2$G$S$)(B"	"$(C:I(B")
   ("$(C$2$G$S$1(B"	"$(C:J(B")
   ("$(C$2$G$S$2(B"	"$(C:K(B")

   ("$(C$2$K(B"	"$(C:L(B")
   ("$(C$2$K$$(B"	"$(C:M(B")

   ("$(C$2$L(B"	"$(C:N(B")
   ("$(C$2$L$!(B"	"$(C:O(B")
   ("$(C$2$L$$(B"	"$(C:P(B")
   ("$(C$2$L$'(B"	"$(C:Q(B")
   ("$(C$2$L$)(B"	"$(C:R(B")
   ("$(C$2$L$)$!(B"	"$(C:S(B")
   ("$(C$2$L$)$1(B"	"$(C:T(B")
   ("$(C$2$L$1(B"	"$(C:U(B")
   ("$(C$2$L$2(B"	"$(C:V(B")
   ("$(C$2$L$5(B"	"$(C:W(B")
   ("$(C$2$L$7(B"	"$(C:X(B")
   ("$(C$2$L$<(B"	"$(C:Y(B")
   ("$(C$2$L$=(B"	"$(C:Z(B")

   ("$(C$2$L$C(B"	"$(C:[(B")
   ("$(C$2$L$C$)(B"	"$(C:\(B")
   ("$(C$2$L$C$6(B"	"$(C:](B")

   ("$(C$2$L$D(B"	"$(C:^(B")

   ("$(C$2$L$S(B"	"$(C:_(B")
   ("$(C$2$L$S$!(B"	"$(C:`(B")
   ("$(C$2$L$S$$(B"	"$(C:a(B")
   ("$(C$2$L$S$)(B"	"$(C:b(B")
   ("$(C$2$L$S$7(B"	"$(C:c(B")

   ("$(C$2$P(B"	"$(C:d(B")
   ("$(C$2$P$$(B"	"$(C:e(B")
   ("$(C$2$P$)(B"	"$(C:f(B")
   ("$(C$2$P$1(B"	"$(C:g(B")
   ("$(C$2$P$5(B"	"$(C:h(B")
   ("$(C$2$P$7(B"	"$(C:i(B")

   ("$(C$2$Q(B"	"$(C:j(B")
   ("$(C$2$Q$!(B"	"$(C:k(B")
   ("$(C$2$Q$$(B"	"$(C:l(B")
   ("$(C$2$Q$)(B"	"$(C:m(B")
   ("$(C$2$Q$1(B"	"$(C:n(B")
   ("$(C$2$Q$2(B"	"$(C:o(B")
   ("$(C$2$Q$5(B"	"$(C:p(B")

   ("$(C$2$S(B"	"$(C:q(B")
   ("$(C$2$S$!(B"	"$(C:r(B")
   ("$(C$2$S$$(B"	"$(C:s(B")
   ("$(C$2$S$)(B"	"$(C:t(B")
   ("$(C$2$S$)$1(B"	"$(C:u(B")
   ("$(C$2$S$1(B"	"$(C:v(B")
   ("$(C$2$S$2(B"	"$(C:w(B")
   ("$(C$2$S$5(B"	"$(C:x(B")
   ("$(C$2$S$7(B"	"$(C:y(B")
   ("$(C$2$S$8(B"	"$(C:z(B")
   ("$(C$2$S$:(B"	"$(C:{(B")

   ("$(C$3$?(B"	"$(C:|(B")
   ("$(C$3$?$!(B"	"$(C:}(B")
   ("$(C$3$?$$(B"	"$(C:~(B")
   ("$(C$3$?$)(B"	"$(C;!(B")
   ("$(C$3$?$)$1(B"	"$(C;"(B")
   ("$(C$3$?$1(B"	"$(C;#(B")
   ("$(C$3$?$2(B"	"$(C;$(B")
   ("$(C$3$?$5(B"	"$(C;%(B")
   ("$(C$3$?$6(B"	"$(C;&(B")
   ("$(C$3$?$7(B"	"$(C;'(B")
   ("$(C$3$?$>(B"	"$(C;((B")

   ("$(C$3$@(B"	"$(C;)(B")
   ("$(C$3$@$!(B"	"$(C;*(B")
   ("$(C$3$@$$(B"	"$(C;+(B")
   ("$(C$3$@$)(B"	"$(C;,(B")
   ("$(C$3$@$1(B"	"$(C;-(B")
   ("$(C$3$@$2(B"	"$(C;.(B")
   ("$(C$3$@$5(B"	"$(C;/(B")
   ("$(C$3$@$6(B"	"$(C;0(B")
   ("$(C$3$@$7(B"	"$(C;1(B")

   ("$(C$3$A(B"	"$(C;2(B")
   ("$(C$3$A$!(B"	"$(C;3(B")
   ("$(C$3$A$1(B"	"$(C;4(B")

   ("$(C$3$C(B"	"$(C;5(B")
   ("$(C$3$C$!(B"	"$(C;6(B")
   ("$(C$3$C$$(B"	"$(C;7(B")
   ("$(C$3$C$'(B"	"$(C;8(B")
   ("$(C$3$C$)(B"	"$(C;9(B")
   ("$(C$3$C$1(B"	"$(C;:(B")
   ("$(C$3$C$5(B"	"$(C;;(B")
   ("$(C$3$C$6(B"	"$(C;<(B")
   ("$(C$3$C$7(B"	"$(C;=(B")

   ("$(C$3$D(B"	"$(C;>(B")
   ("$(C$3$D$7(B"	"$(C;?(B")

   ("$(C$3$E(B"	"$(C;@(B")
   ("$(C$3$E$!(B"	"$(C;A(B")
   ("$(C$3$E$1(B"	"$(C;B(B")
   ("$(C$3$E$2(B"	"$(C;C(B")
   ("$(C$3$E$5(B"	"$(C;D(B")
   ("$(C$3$E$6(B"	"$(C;E(B")
   ("$(C$3$E$7(B"	"$(C;F(B")

   ("$(C$3$G(B"	"$(C;G(B")
   ("$(C$3$G$!(B"	"$(C;H(B")
   ("$(C$3$G$$(B"	"$(C;I(B")
   ("$(C$3$G$)(B"	"$(C;J(B")
   ("$(C$3$G$1(B"	"$(C;K(B")
   ("$(C$3$G$2(B"	"$(C;L(B")
   ("$(C$3$G$7(B"	"$(C;M(B")

   ("$(C$3$G$S(B"	"$(C;N(B")

   ("$(C$3$K(B"	"$(C;O(B")
   ("$(C$3$K$7(B"	"$(C;P(B")

   ("$(C$3$L(B"	"$(C;Q(B")
   ("$(C$3$L$!(B"	"$(C;R(B")
   ("$(C$3$L$$(B"	"$(C;S(B")
   ("$(C$3$L$)(B"	"$(C;T(B")
   ("$(C$3$L$1(B"	"$(C;U(B")
   ("$(C$3$L$5(B"	"$(C;V(B")
   ("$(C$3$L$7(B"	"$(C;W(B")

   ("$(C$3$P(B"	"$(C;X(B")
   ("$(C$3$P$7(B"	"$(C;Y(B")

   ("$(C$3$Q(B"	"$(C;Z(B")
   ("$(C$3$Q$$(B"	"$(C;[(B")
   ("$(C$3$Q$)(B"	"$(C;\(B")
   ("$(C$3$Q$1(B"	"$(C;](B")
   ("$(C$3$Q$2(B"	"$(C;^(B")

   ("$(C$3$S(B"	"$(C;_(B")
   ("$(C$3$S$!(B"	"$(C;`(B")
   ("$(C$3$S$$(B"	"$(C;a(B")
   ("$(C$3$S$)(B"	"$(C;b(B")
   ("$(C$3$S$1(B"	"$(C;c(B")
   ("$(C$3$S$2(B"	"$(C;d(B")
   ("$(C$3$S$5(B"	"$(C;e(B")
   ("$(C$3$S$7(B"	"$(C;f(B")

   ("$(C$5$?(B"	"$(C;g(B")
   ("$(C$5$?$!(B"	"$(C;h(B")
   ("$(C$5$?$!$5(B"	"$(C;i(B")
   ("$(C$5$?$$(B"	"$(C;j(B")
   ("$(C$5$?$'(B"	"$(C;k(B")
   ("$(C$5$?$)(B"	"$(C;l(B")
   ("$(C$5$?$)$!(B"	"$(C;m(B")
   ("$(C$5$?$)$1(B"	"$(C;n(B")
   ("$(C$5$?$1(B"	"$(C;o(B")
   ("$(C$5$?$2(B"	"$(C;p(B")
   ("$(C$5$?$5(B"	"$(C;q(B")
   ("$(C$5$?$5$6(B"	"$(C;r(B")
   ("$(C$5$?$7(B"	"$(C;s(B")
   ("$(C$5$?$<(B"	"$(C;t(B")

   ("$(C$5$@(B"	"$(C;u(B")
   ("$(C$5$@$!(B"	"$(C;v(B")
   ("$(C$5$@$$(B"	"$(C;w(B")
   ("$(C$5$@$)(B"	"$(C;x(B")
   ("$(C$5$@$1(B"	"$(C;y(B")
   ("$(C$5$@$2(B"	"$(C;z(B")
   ("$(C$5$@$5(B"	"$(C;{(B")
   ("$(C$5$@$6(B"	"$(C;|(B")
   ("$(C$5$@$7(B"	"$(C;}(B")

   ("$(C$5$A(B"	"$(C;~(B")
   ("$(C$5$A$!(B"	"$(C<!(B")
   ("$(C$5$A$$(B"	"$(C<"(B")
   ("$(C$5$A$)(B"	"$(C<#(B")
   ("$(C$5$A$1(B"	"$(C<$(B")
   ("$(C$5$A$2(B"	"$(C<%(B")
   ("$(C$5$A$5(B"	"$(C<&(B")
   ("$(C$5$A$7(B"	"$(C<'(B")

   ("$(C$5$B(B"	"$(C<((B")
   ("$(C$5$B$$(B"	"$(C<)(B")
   ("$(C$5$B$)(B"	"$(C<*(B")
   ("$(C$5$B$1(B"	"$(C<+(B")
   ("$(C$5$B$7(B"	"$(C<,(B")

   ("$(C$5$C(B"	"$(C<-(B")
   ("$(C$5$C$!(B"	"$(C<.(B")
   ("$(C$5$C$!$5(B"	"$(C<0(B")
   ("$(C$5$C$"(B"	"$(C</(B")
   ("$(C$5$C$$(B"	"$(C<1(B")
   ("$(C$5$C$'(B"	"$(C<2(B")
   ("$(C$5$C$)(B"	"$(C<3(B")
   ("$(C$5$C$)$1(B"	"$(C<4(B")
   ("$(C$5$C$)$2(B"	"$(C<5(B")
   ("$(C$5$C$1(B"	"$(C<6(B")
   ("$(C$5$C$2(B"	"$(C<7(B")
   ("$(C$5$C$5(B"	"$(C<8(B")
   ("$(C$5$C$6(B"	"$(C<9(B")
   ("$(C$5$C$7(B"	"$(C<:(B")
   ("$(C$5$C$=(B"	"$(C<;(B")

   ("$(C$5$D(B"	"$(C<<(B")
   ("$(C$5$D$!(B"	"$(C<=(B")
   ("$(C$5$D$$(B"	"$(C<>(B")
   ("$(C$5$D$)(B"	"$(C<?(B")
   ("$(C$5$D$1(B"	"$(C<@(B")
   ("$(C$5$D$2(B"	"$(C<A(B")
   ("$(C$5$D$5(B"	"$(C<B(B")
   ("$(C$5$D$6(B"	"$(C<C(B")
   ("$(C$5$D$7(B"	"$(C<D(B")

   ("$(C$5$E(B"	"$(C<E(B")
   ("$(C$5$E$!(B"	"$(C<F(B")
   ("$(C$5$E$$(B"	"$(C<G(B")
   ("$(C$5$E$)(B"	"$(C<H(B")
   ("$(C$5$E$1(B"	"$(C<I(B")
   ("$(C$5$E$2(B"	"$(C<J(B")
   ("$(C$5$E$5(B"	"$(C<K(B")
   ("$(C$5$E$6(B"	"$(C<L(B")
   ("$(C$5$E$7(B"	"$(C<M(B")

   ("$(C$5$F(B"	"$(C<N(B")
   ("$(C$5$F$$(B"	"$(C<O(B")
   ("$(C$5$F$)(B"	"$(C<P(B")
   ("$(C$5$F$7(B"	"$(C<Q(B")

   ("$(C$5$G(B"	"$(C<R(B")
   ("$(C$5$G$!(B"	"$(C<S(B")
   ("$(C$5$G$"(B"	"$(C<T(B")
   ("$(C$5$G$$(B"	"$(C<U(B")
   ("$(C$5$G$)(B"	"$(C<V(B")
   ("$(C$5$G$)$1(B"	"$(C<W(B")
   ("$(C$5$G$1(B"	"$(C<X(B")
   ("$(C$5$G$2(B"	"$(C<Y(B")
   ("$(C$5$G$5(B"	"$(C<Z(B")
   ("$(C$5$G$7(B"	"$(C<[(B")
   ("$(C$5$G$<(B"	"$(C<\(B")

   ("$(C$5$G$?(B"	"$(C<](B")
   ("$(C$5$G$?$!(B"	"$(C<^(B")
   ("$(C$5$G$?$$(B"	"$(C<_(B")
   ("$(C$5$G$?$)(B"	"$(C<`(B")
   ("$(C$5$G$?$7(B"	"$(C<a(B")

   ("$(C$5$G$@(B"	"$(C<b(B")
   ("$(C$5$G$@$$(B"	"$(C<c(B")
   ("$(C$5$G$@$)(B"	"$(C<d(B")
   ("$(C$5$G$@$1(B"	"$(C<e(B")
   ("$(C$5$G$@$5(B"	"$(C<f(B")
   ("$(C$5$G$@$6(B"	"$(C<g(B")

   ("$(C$5$G$S(B"	"$(C<h(B")
   ("$(C$5$G$S$$(B"	"$(C<i(B")
   ("$(C$5$G$S$)(B"	"$(C<j(B")
   ("$(C$5$G$S$1(B"	"$(C<k(B")
   ("$(C$5$G$S$2(B"	"$(C<l(B")
   ("$(C$5$G$S$5(B"	"$(C<m(B")

   ("$(C$5$K(B"	"$(C<n(B")
   ("$(C$5$K$!(B"	"$(C<o(B")
   ("$(C$5$K$$(B"	"$(C<p(B")
   ("$(C$5$K$)(B"	"$(C<q(B")
   ("$(C$5$K$1(B"	"$(C<r(B")
   ("$(C$5$K$2(B"	"$(C<s(B")
   ("$(C$5$K$5(B"	"$(C<t(B")
   ("$(C$5$K$7(B"	"$(C<u(B")

   ("$(C$5$L(B"	"$(C<v(B")
   ("$(C$5$L$!(B"	"$(C<w(B")
   ("$(C$5$L$$(B"	"$(C<x(B")
   ("$(C$5$L$'(B"	"$(C<y(B")
   ("$(C$5$L$)(B"	"$(C<z(B")
   ("$(C$5$L$1(B"	"$(C<{(B")
   ("$(C$5$L$2(B"	"$(C<|(B")
   ("$(C$5$L$5(B"	"$(C<}(B")
   ("$(C$5$L$7(B"	"$(C<~(B")
   ("$(C$5$L$:(B"	"$(C=!(B")
   ("$(C$5$L$<(B"	"$(C="(B")
   ("$(C$5$L$=(B"	"$(C=#(B")

   ("$(C$5$L$C(B"	"$(C=$(B")
   ("$(C$5$L$C$6(B"	"$(C=%(B")

   ("$(C$5$L$D(B"	"$(C=&(B")
   ("$(C$5$L$D$!(B"	"$(C='(B")
   ("$(C$5$L$D$$(B"	"$(C=((B")
   ("$(C$5$L$D$)(B"	"$(C=)(B")
   ("$(C$5$L$D$1(B"	"$(C=*(B")
   ("$(C$5$L$D$7(B"	"$(C=+(B")

   ("$(C$5$L$S(B"	"$(C=,(B")
   ("$(C$5$L$S$!(B"	"$(C=-(B")
   ("$(C$5$L$S$$(B"	"$(C=.(B")
   ("$(C$5$L$S$)(B"	"$(C=/(B")
   ("$(C$5$L$S$1(B"	"$(C=0(B")
   ("$(C$5$L$S$2(B"	"$(C=1(B")
   ("$(C$5$L$S$5(B"	"$(C=2(B")
   ("$(C$5$L$S$7(B"	"$(C=3(B")

   ("$(C$5$P(B"	"$(C=4(B")
   ("$(C$5$P$!(B"	"$(C=5(B")
   ("$(C$5$P$)(B"	"$(C=6(B")
   ("$(C$5$P$1(B"	"$(C=7(B")
   ("$(C$5$P$5(B"	"$(C=8(B")
   ("$(C$5$P$7(B"	"$(C=9(B")

   ("$(C$5$Q(B"	"$(C=:(B")
   ("$(C$5$Q$!(B"	"$(C=;(B")
   ("$(C$5$Q$$(B"	"$(C=<(B")
   ("$(C$5$Q$)(B"	"$(C==(B")
   ("$(C$5$Q$)$!(B"	"$(C=>(B")
   ("$(C$5$Q$1(B"	"$(C=?(B")
   ("$(C$5$Q$2(B"	"$(C=@(B")
   ("$(C$5$Q$5(B"	"$(C=A(B")
   ("$(C$5$Q$7(B"	"$(C=B(B")

   ("$(C$5$S(B"	"$(C=C(B")
   ("$(C$5$S$!(B"	"$(C=D(B")
   ("$(C$5$S$$(B"	"$(C=E(B")
   ("$(C$5$S$'(B"	"$(C=F(B")
   ("$(C$5$S$)(B"	"$(C=G(B")
   ("$(C$5$S$)$>(B"	"$(C=H(B")
   ("$(C$5$S$1(B"	"$(C=I(B")
   ("$(C$5$S$2(B"	"$(C=J(B")
   ("$(C$5$S$5(B"	"$(C=K(B")
   ("$(C$5$S$7(B"	"$(C=L(B")
   ("$(C$5$S$=(B"	"$(C=M(B")

   ("$(C$6$?(B"	"$(C=N(B")
   ("$(C$6$?$!(B"	"$(C=O(B")
   ("$(C$6$?$!$5(B"	"$(C=P(B")
   ("$(C$6$?$$(B"	"$(C=Q(B")
   ("$(C$6$?$)(B"	"$(C=R(B")
   ("$(C$6$?$1(B"	"$(C=S(B")
   ("$(C$6$?$2(B"	"$(C=T(B")
   ("$(C$6$?$6(B"	"$(C=U(B")
   ("$(C$6$?$7(B"	"$(C=V(B")
   ("$(C$6$?$>(B"	"$(C=W(B")

   ("$(C$6$@(B"	"$(C=X(B")
   ("$(C$6$@$!(B"	"$(C=Y(B")
   ("$(C$6$@$$(B"	"$(C=Z(B")
   ("$(C$6$@$)(B"	"$(C=[(B")
   ("$(C$6$@$1(B"	"$(C=\(B")
   ("$(C$6$@$2(B"	"$(C=](B")
   ("$(C$6$@$6(B"	"$(C=^(B")
   ("$(C$6$@$7(B"	"$(C=_(B")

   ("$(C$6$A$7(B"	"$(C=`(B")

   ("$(C$6$C(B"	"$(C=a(B")
   ("$(C$6$C$!(B"	"$(C=b(B")
   ("$(C$6$C$$(B"	"$(C=c(B")
   ("$(C$6$C$)(B"	"$(C=d(B")
   ("$(C$6$C$)$2(B"	"$(C=e(B")
   ("$(C$6$C$1(B"	"$(C=f(B")
   ("$(C$6$C$2(B"	"$(C=g(B")
   ("$(C$6$C$6(B"	"$(C=h(B")
   ("$(C$6$C$7(B"	"$(C=i(B")

   ("$(C$6$D(B"	"$(C=j(B")
   ("$(C$6$D$$(B"	"$(C=k(B")
   ("$(C$6$D$)(B"	"$(C=l(B")

   ("$(C$6$F$$(B"	"$(C=m(B")

   ("$(C$6$G(B"	"$(C=n(B")
   ("$(C$6$G$!(B"	"$(C=o(B")
   ("$(C$6$G$$(B"	"$(C=p(B")
   ("$(C$6$G$'(B"	"$(C=q(B")
   ("$(C$6$G$)(B"	"$(C=r(B")
   ("$(C$6$G$)$1(B"	"$(C=s(B")
   ("$(C$6$G$1(B"	"$(C=t(B")
   ("$(C$6$G$2(B"	"$(C=u(B")
   ("$(C$6$G$7(B"	"$(C=v(B")

   ("$(C$6$G$?(B"	"$(C=w(B")
   ("$(C$6$G$?$!(B"	"$(C=x(B")
   ("$(C$6$G$?$$(B"	"$(C=y(B")
   ("$(C$6$G$?$6(B"	"$(C=z(B")

   ("$(C$6$G$@(B"	"$(C={(B")
   ("$(C$6$G$@$6(B"	"$(C=|(B")

   ("$(C$6$G$S(B"	"$(C=}(B")
   ("$(C$6$G$S$$(B"	"$(C=~(B")
   ("$(C$6$G$S$)(B"	"$(C>!(B")
   ("$(C$6$G$S$1(B"	"$(C>"(B")
   ("$(C$6$G$S$2(B"	"$(C>#(B")

   ("$(C$6$K(B"	"$(C>$(B")

   ("$(C$6$L(B"	"$(C>%(B")
   ("$(C$6$L$!(B"	"$(C>&(B")
   ("$(C$6$L$$(B"	"$(C>'(B")
   ("$(C$6$L$)(B"	"$(C>((B")
   ("$(C$6$L$1(B"	"$(C>)(B")
   ("$(C$6$L$2(B"	"$(C>*(B")
   ("$(C$6$L$7(B"	"$(C>+(B")

   ("$(C$6$L$C(B"	"$(C>,(B")
   ("$(C$6$L$C$6(B"	"$(C>-(B")

   ("$(C$6$L$D(B"	"$(C>.(B")

   ("$(C$6$L$S(B"	"$(C>/(B")
   ("$(C$6$L$S$$(B"	"$(C>0(B")

   ("$(C$6$P$7(B"	"$(C>1(B")

   ("$(C$6$Q(B"	"$(C>2(B")
   ("$(C$6$Q$!(B"	"$(C>3(B")
   ("$(C$6$Q$$(B"	"$(C>4(B")
   ("$(C$6$Q$)(B"	"$(C>5(B")
   ("$(C$6$Q$)$1(B"	"$(C>6(B")
   ("$(C$6$Q$)$>(B"	"$(C>7(B")
   ("$(C$6$Q$1(B"	"$(C>8(B")
   ("$(C$6$Q$2(B"	"$(C>9(B")

   ("$(C$6$Q$S(B"	"$(C>:(B")
   ("$(C$6$Q$S$$(B"	"$(C>;(B")
   ("$(C$6$Q$S$)(B"	"$(C><(B")
   ("$(C$6$Q$S$1(B"	"$(C>=(B")

   ("$(C$6$S(B"	"$(C>>(B")
   ("$(C$6$S$!(B"	"$(C>?(B")
   ("$(C$6$S$$(B"	"$(C>@(B")
   ("$(C$6$S$)(B"	"$(C>A(B")
   ("$(C$6$S$1(B"	"$(C>B(B")
   ("$(C$6$S$2(B"	"$(C>C(B")
   ("$(C$6$S$5(B"	"$(C>D(B")
   ("$(C$6$S$7(B"	"$(C>E(B")

   ("$(C$7$?(B"	"$(C>F(B")
   ("$(C$7$?$!(B"	"$(C>G(B")
   ("$(C$7$?$$(B"	"$(C>H(B")
   ("$(C$7$?$$$8(B"	"$(C>I(B")
   ("$(C$7$?$$$>(B"	"$(C>J(B")
   ("$(C$7$?$)(B"	"$(C>K(B")
   ("$(C$7$?$)$!(B"	"$(C>L(B")
   ("$(C$7$?$)$1(B"	"$(C>M(B")
   ("$(C$7$?$)$>(B"	"$(C>N(B")
   ("$(C$7$?$1(B"	"$(C>O(B")
   ("$(C$7$?$2(B"	"$(C>P(B")
   ("$(C$7$?$5(B"	"$(C>Q(B")
   ("$(C$7$?$6(B"	"$(C>R(B")
   ("$(C$7$?$7(B"	"$(C>S(B")
   ("$(C$7$?$<(B"	"$(C>T(B")
   ("$(C$7$?$=(B"	"$(C>U(B")

   ("$(C$7$@(B"	"$(C>V(B")
   ("$(C$7$@$!(B"	"$(C>W(B")
   ("$(C$7$@$$(B"	"$(C>X(B")
   ("$(C$7$@$)(B"	"$(C>Y(B")
   ("$(C$7$@$1(B"	"$(C>Z(B")
   ("$(C$7$@$2(B"	"$(C>[(B")
   ("$(C$7$@$5(B"	"$(C>\(B")
   ("$(C$7$@$6(B"	"$(C>](B")
   ("$(C$7$@$7(B"	"$(C>^(B")

   ("$(C$7$A(B"	"$(C>_(B")
   ("$(C$7$A$!(B"	"$(C>`(B")
   ("$(C$7$A$$(B"	"$(C>a(B")
   ("$(C$7$A$)(B"	"$(C>b(B")
   ("$(C$7$A$)$2(B"	"$(C>c(B")
   ("$(C$7$A$1(B"	"$(C>d(B")
   ("$(C$7$A$2(B"	"$(C>e(B")
   ("$(C$7$A$5(B"	"$(C>f(B")
   ("$(C$7$A$7(B"	"$(C>g(B")
   ("$(C$7$A$<(B"	"$(C>h(B")
   ("$(C$7$A$>(B"	"$(C>i(B")

   ("$(C$7$B(B"	"$(C>j(B")
   ("$(C$7$B$$(B"	"$(C>k(B")
   ("$(C$7$B$)(B"	"$(C>l(B")
   ("$(C$7$B$2(B"	"$(C>m(B")

   ("$(C$7$C(B"	"$(C>n(B")
   ("$(C$7$C$!(B"	"$(C>o(B")
   ("$(C$7$C$$(B"	"$(C>p(B")
   ("$(C$7$C$$$8(B"	"$(C>q(B")
   ("$(C$7$C$'(B"	"$(C>r(B")
   ("$(C$7$C$)(B"	"$(C>s(B")
   ("$(C$7$C$)$!(B"	"$(C>t(B")
   ("$(C$7$C$)$1(B"	"$(C>u(B")
   ("$(C$7$C$1(B"	"$(C>v(B")
   ("$(C$7$C$2(B"	"$(C>w(B")
   ("$(C$7$C$2$5(B"	"$(C>x(B")
   ("$(C$7$C$5(B"	"$(C>y(B")
   ("$(C$7$C$6(B"	"$(C>z(B")
   ("$(C$7$C$7(B"	"$(C>{(B")
   ("$(C$7$C$8(B"	"$(C>|(B")
   ("$(C$7$C$;(B"	"$(C>}(B")
   ("$(C$7$C$=(B"	"$(C>~(B")

   ("$(C$7$D(B"	"$(C?!(B")
   ("$(C$7$D$!(B"	"$(C?"(B")
   ("$(C$7$D$$(B"	"$(C?#(B")
   ("$(C$7$D$)(B"	"$(C?$(B")
   ("$(C$7$D$1(B"	"$(C?%(B")
   ("$(C$7$D$2(B"	"$(C?&(B")
   ("$(C$7$D$5(B"	"$(C?'(B")
   ("$(C$7$D$7(B"	"$(C?((B")

   ("$(C$7$E(B"	"$(C?)(B")
   ("$(C$7$E$!(B"	"$(C?*(B")
   ("$(C$7$E$"(B"	"$(C?+(B")
   ("$(C$7$E$$(B"	"$(C?,(B")
   ("$(C$7$E$)(B"	"$(C?-(B")
   ("$(C$7$E$)$1(B"	"$(C?.(B")
   ("$(C$7$E$)$2(B"	"$(C?/(B")
   ("$(C$7$E$1(B"	"$(C?0(B")
   ("$(C$7$E$2(B"	"$(C?1(B")
   ("$(C$7$E$2$5(B"	"$(C?2(B")
   ("$(C$7$E$5(B"	"$(C?3(B")
   ("$(C$7$E$6(B"	"$(C?4(B")
   ("$(C$7$E$7(B"	"$(C?5(B")
   ("$(C$7$E$<(B"	"$(C?6(B")
   ("$(C$7$E$=(B"	"$(C?7(B")
   ("$(C$7$E$>(B"	"$(C?8(B")

   ("$(C$7$F(B"	"$(C?9(B")
   ("$(C$7$F$$(B"	"$(C?:(B")
   ("$(C$7$F$)(B"	"$(C?;(B")
   ("$(C$7$F$1(B"	"$(C?<(B")
   ("$(C$7$F$2(B"	"$(C?=(B")
   ("$(C$7$F$5(B"	"$(C?>(B")
   ("$(C$7$F$6(B"	"$(C??(B")

   ("$(C$7$G(B"	"$(C?@(B")
   ("$(C$7$G$!(B"	"$(C?A(B")
   ("$(C$7$G$$(B"	"$(C?B(B")
   ("$(C$7$G$)(B"	"$(C?C(B")
   ("$(C$7$G$)$!(B"	"$(C?D(B")
   ("$(C$7$G$)$1(B"	"$(C?E(B")
   ("$(C$7$G$)$5(B"	"$(C?F(B")
   ("$(C$7$G$)$>(B"	"$(C?G(B")
   ("$(C$7$G$1(B"	"$(C?H(B")
   ("$(C$7$G$2(B"	"$(C?I(B")
   ("$(C$7$G$5(B"	"$(C?J(B")
   ("$(C$7$G$7(B"	"$(C?K(B")
   ("$(C$7$G$:(B"	"$(C?L(B")

   ("$(C$7$G$?(B"	"$(C?M(B")
   ("$(C$7$G$?$!(B"	"$(C?N(B")
   ("$(C$7$G$?$$(B"	"$(C?O(B")
   ("$(C$7$G$?$)(B"	"$(C?P(B")
   ("$(C$7$G$?$1(B"	"$(C?Q(B")
   ("$(C$7$G$?$2(B"	"$(C?R(B")
   ("$(C$7$G$?$5(B"	"$(C?S(B")
   ("$(C$7$G$?$6(B"	"$(C?T(B")
   ("$(C$7$G$?$7(B"	"$(C?U(B")

   ("$(C$7$G$@(B"	"$(C?V(B")
   ("$(C$7$G$@$!(B"	"$(C?W(B")
   ("$(C$7$G$@$$(B"	"$(C?X(B")
   ("$(C$7$G$@$1(B"	"$(C?Y(B")
   ("$(C$7$G$@$5(B"	"$(C?Z(B")
   ("$(C$7$G$@$7(B"	"$(C?[(B")

   ("$(C$7$G$S(B"	"$(C?\(B")
   ("$(C$7$G$S$!(B"	"$(C?](B")
   ("$(C$7$G$S$$(B"	"$(C?^(B")
   ("$(C$7$G$S$)(B"	"$(C?_(B")
   ("$(C$7$G$S$1(B"	"$(C?`(B")
   ("$(C$7$G$S$2(B"	"$(C?a(B")
   ("$(C$7$G$S$5(B"	"$(C?b(B")
   ("$(C$7$G$S$7(B"	"$(C?c(B")

   ("$(C$7$K(B"	"$(C?d(B")
   ("$(C$7$K$!(B"	"$(C?e(B")
   ("$(C$7$K$$(B"	"$(C?f(B")
   ("$(C$7$K$)(B"	"$(C?g(B")
   ("$(C$7$K$1(B"	"$(C?h(B")
   ("$(C$7$K$2(B"	"$(C?i(B")
   ("$(C$7$K$5(B"	"$(C?j(B")
   ("$(C$7$K$7(B"	"$(C?k(B")

   ("$(C$7$L(B"	"$(C?l(B")
   ("$(C$7$L$!(B"	"$(C?m(B")
   ("$(C$7$L$$(B"	"$(C?n(B")
   ("$(C$7$L$)(B"	"$(C?o(B")
   ("$(C$7$L$)$!(B"	"$(C?p(B")
   ("$(C$7$L$)$1(B"	"$(C?q(B")
   ("$(C$7$L$1(B"	"$(C?r(B")
   ("$(C$7$L$2(B"	"$(C?s(B")
   ("$(C$7$L$5(B"	"$(C?t(B")
   ("$(C$7$L$7(B"	"$(C?u(B")

   ("$(C$7$L$C(B"	"$(C?v(B")
   ("$(C$7$L$C$!(B"	"$(C?w(B")
   ("$(C$7$L$C$$(B"	"$(C?x(B")
   ("$(C$7$L$C$)(B"	"$(C?y(B")
   ("$(C$7$L$C$1(B"	"$(C?z(B")
   ("$(C$7$L$C$2(B"	"$(C?{(B")
   ("$(C$7$L$C$6(B"	"$(C?|(B")
   ("$(C$7$L$C$7(B"	"$(C?}(B")

   ("$(C$7$L$D(B"	"$(C?~(B")
   ("$(C$7$L$D$!(B"	"$(C@!(B")
   ("$(C$7$L$D$$(B"	"$(C@"(B")
   ("$(C$7$L$D$)(B"	"$(C@#(B")
   ("$(C$7$L$D$1(B"	"$(C@$(B")
   ("$(C$7$L$D$2(B"	"$(C@%(B")
   ("$(C$7$L$D$7(B"	"$(C@&(B")

   ("$(C$7$L$S(B"	"$(C@'(B")
   ("$(C$7$L$S$!(B"	"$(C@((B")
   ("$(C$7$L$S$$(B"	"$(C@)(B")
   ("$(C$7$L$S$)(B"	"$(C@*(B")
   ("$(C$7$L$S$1(B"	"$(C@+(B")
   ("$(C$7$L$S$2(B"	"$(C@,(B")
   ("$(C$7$L$S$5(B"	"$(C@-(B")
   ("$(C$7$L$S$7(B"	"$(C@.(B")

   ("$(C$7$P(B"	"$(C@/(B")
   ("$(C$7$P$!(B"	"$(C@0(B")
   ("$(C$7$P$$(B"	"$(C@1(B")
   ("$(C$7$P$)(B"	"$(C@2(B")
   ("$(C$7$P$1(B"	"$(C@3(B")
   ("$(C$7$P$2(B"	"$(C@4(B")
   ("$(C$7$P$5(B"	"$(C@5(B")
   ("$(C$7$P$7(B"	"$(C@6(B")
   ("$(C$7$P$:(B"	"$(C@7(B")

   ("$(C$7$Q(B"	"$(C@8(B")
   ("$(C$7$Q$!(B"	"$(C@9(B")
   ("$(C$7$Q$$(B"	"$(C@:(B")
   ("$(C$7$Q$)(B"	"$(C@;(B")
   ("$(C$7$Q$)$=(B"	"$(C@<(B")
   ("$(C$7$Q$1(B"	"$(C@=(B")
   ("$(C$7$Q$2(B"	"$(C@>(B")
   ("$(C$7$Q$5(B"	"$(C@?(B")
   ("$(C$7$Q$7(B"	"$(C@@(B")
   ("$(C$7$Q$8(B"	"$(C@A(B")
   ("$(C$7$Q$:(B"	"$(C@B(B")
   ("$(C$7$Q$;(B"	"$(C@C(B")
   ("$(C$7$Q$<(B"	"$(C@D(B")
   ("$(C$7$Q$=(B"	"$(C@E(B")
   ("$(C$7$Q$>(B"	"$(C@F(B")

   ("$(C$7$Q$S(B"	"$(C@G(B")
   ("$(C$7$Q$S$$(B"	"$(C@H(B")
   ("$(C$7$Q$S$)(B"	"$(C@I(B")
   ("$(C$7$Q$S$1(B"	"$(C@J(B")
   ("$(C$7$Q$S$5(B"	"$(C@K(B")

   ("$(C$7$S(B"	"$(C@L(B")
   ("$(C$7$S$!(B"	"$(C@M(B")
   ("$(C$7$S$$(B"	"$(C@N(B")
   ("$(C$7$S$)(B"	"$(C@O(B")
   ("$(C$7$S$)$!(B"	"$(C@P(B")
   ("$(C$7$S$)$1(B"	"$(C@Q(B")
   ("$(C$7$S$)$>(B"	"$(C@R(B")
   ("$(C$7$S$1(B"	"$(C@S(B")
   ("$(C$7$S$2(B"	"$(C@T(B")
   ("$(C$7$S$5(B"	"$(C@U(B")
   ("$(C$7$S$6(B"	"$(C@V(B")
   ("$(C$7$S$7(B"	"$(C@W(B")
   ("$(C$7$S$8(B"	"$(C@X(B")
   ("$(C$7$S$=(B"	"$(C@Y(B")

   ("$(C$8$?(B"	"$(C@Z(B")
   ("$(C$8$?$!(B"	"$(C@[(B")
   ("$(C$8$?$$(B"	"$(C@\(B")
   ("$(C$8$?$$$>(B"	"$(C@](B")
   ("$(C$8$?$'(B"	"$(C@^(B")
   ("$(C$8$?$)(B"	"$(C@_(B")
   ("$(C$8$?$)$1(B"	"$(C@`(B")
   ("$(C$8$?$1(B"	"$(C@a(B")
   ("$(C$8$?$2(B"	"$(C@b(B")
   ("$(C$8$?$5(B"	"$(C@c(B")
   ("$(C$8$?$6(B"	"$(C@d(B")
   ("$(C$8$?$7(B"	"$(C@e(B")
   ("$(C$8$?$8(B"	"$(C@f(B")

   ("$(C$8$@(B"	"$(C@g(B")
   ("$(C$8$@$!(B"	"$(C@h(B")
   ("$(C$8$@$$(B"	"$(C@i(B")
   ("$(C$8$@$)(B"	"$(C@j(B")
   ("$(C$8$@$1(B"	"$(C@k(B")
   ("$(C$8$@$2(B"	"$(C@l(B")
   ("$(C$8$@$5(B"	"$(C@m(B")
   ("$(C$8$@$6(B"	"$(C@n(B")
   ("$(C$8$@$7(B"	"$(C@o(B")

   ("$(C$8$A(B"	"$(C@p(B")
   ("$(C$8$A$!(B"	"$(C@q(B")
   ("$(C$8$A$$(B"	"$(C@r(B")
   ("$(C$8$A$$$>(B"	"$(C@s(B")
   ("$(C$8$A$)(B"	"$(C@t(B")
   ("$(C$8$A$1(B"	"$(C@u(B")
   ("$(C$8$A$7(B"	"$(C@v(B")

   ("$(C$8$B(B"	"$(C@w(B")
   ("$(C$8$B$$(B"	"$(C@x(B")
   ("$(C$8$B$)(B"	"$(C@y(B")

   ("$(C$8$C(B"	"$(C@z(B")
   ("$(C$8$C$!(B"	"$(C@{(B")
   ("$(C$8$C$$(B"	"$(C@|(B")
   ("$(C$8$C$)(B"	"$(C@}(B")
   ("$(C$8$C$)$1(B"	"$(C@~(B")
   ("$(C$8$C$1(B"	"$(CA!(B")
   ("$(C$8$C$2(B"	"$(CA"(B")
   ("$(C$8$C$5(B"	"$(CA#(B")
   ("$(C$8$C$7(B"	"$(CA$(B")
   ("$(C$8$C$8(B"	"$(CA%(B")

   ("$(C$8$D(B"	"$(CA&(B")
   ("$(C$8$D$!(B"	"$(CA'(B")
   ("$(C$8$D$$(B"	"$(CA((B")
   ("$(C$8$D$)(B"	"$(CA)(B")
   ("$(C$8$D$1(B"	"$(CA*(B")
   ("$(C$8$D$2(B"	"$(CA+(B")
   ("$(C$8$D$5(B"	"$(CA,(B")
   ("$(C$8$D$7(B"	"$(CA-(B")

   ("$(C$8$E(B"	"$(CA.(B")
   ("$(C$8$E$$(B"	"$(CA/(B")
   ("$(C$8$E$)(B"	"$(CA0(B")
   ("$(C$8$E$1(B"	"$(CA1(B")
   ("$(C$8$E$2(B"	"$(CA2(B")
   ("$(C$8$E$6(B"	"$(CA3(B")
   ("$(C$8$E$7(B"	"$(CA4(B")

   ("$(C$8$F(B"	"$(CA5(B")

   ("$(C$8$G(B"	"$(CA6(B")
   ("$(C$8$G$!(B"	"$(CA7(B")
   ("$(C$8$G$$(B"	"$(CA8(B")
   ("$(C$8$G$)(B"	"$(CA9(B")
   ("$(C$8$G$)$1(B"	"$(CA:(B")
   ("$(C$8$G$1(B"	"$(CA;(B")
   ("$(C$8$G$2(B"	"$(CA<(B")
   ("$(C$8$G$5(B"	"$(CA=(B")
   ("$(C$8$G$7(B"	"$(CA>(B")
   ("$(C$8$G$8(B"	"$(CA?(B")
   ("$(C$8$G$:(B"	"$(CA@(B")
   ("$(C$8$G$>(B"	"$(CAA(B")

   ("$(C$8$G$?(B"	"$(CAB(B")
   ("$(C$8$G$?$!(B"	"$(CAC(B")
   ("$(C$8$G$?$)(B"	"$(CAD(B")
   ("$(C$8$G$?$2(B"	"$(CAE(B")
   ("$(C$8$G$?$5(B"	"$(CAF(B")
   ("$(C$8$G$?$7(B"	"$(CAG(B")

   ("$(C$8$G$@(B"	"$(CAH(B")
   ("$(C$8$G$@$6(B"	"$(CAI(B")
   ("$(C$8$G$@$7(B"	"$(CAJ(B")

   ("$(C$8$G$S(B"	"$(CAK(B")
   ("$(C$8$G$S$$(B"	"$(CAL(B")
   ("$(C$8$G$S$)(B"	"$(CAM(B")
   ("$(C$8$G$S$1(B"	"$(CAN(B")
   ("$(C$8$G$S$2(B"	"$(CAO(B")
   ("$(C$8$G$S$5(B"	"$(CAP(B")
   ("$(C$8$G$S$7(B"	"$(CAQ(B")

   ("$(C$8$K(B"	"$(CAR(B")
   ("$(C$8$K$!(B"	"$(CAS(B")
   ("$(C$8$K$$(B"	"$(CAT(B")
   ("$(C$8$K$7(B"	"$(CAU(B")

   ("$(C$8$L(B"	"$(CAV(B")
   ("$(C$8$L$!(B"	"$(CAW(B")
   ("$(C$8$L$$(B"	"$(CAX(B")
   ("$(C$8$L$)(B"	"$(CAY(B")
   ("$(C$8$L$)$!(B"	"$(CAZ(B")
   ("$(C$8$L$)$1(B"	"$(CA[(B")
   ("$(C$8$L$1(B"	"$(CA\(B")
   ("$(C$8$L$2(B"	"$(CA](B")
   ("$(C$8$L$5(B"	"$(CA^(B")
   ("$(C$8$L$7(B"	"$(CA_(B")

   ("$(C$8$L$C(B"	"$(CA`(B")
   ("$(C$8$L$C$6(B"	"$(CAa(B")

   ("$(C$8$L$D(B"	"$(CAb(B")

   ("$(C$8$L$S(B"	"$(CAc(B")
   ("$(C$8$L$S$!(B"	"$(CAd(B")
   ("$(C$8$L$S$$(B"	"$(CAe(B")
   ("$(C$8$L$S$)(B"	"$(CAf(B")
   ("$(C$8$L$S$1(B"	"$(CAg(B")
   ("$(C$8$L$S$2(B"	"$(CAh(B")
   ("$(C$8$L$S$5(B"	"$(CAi(B")

   ("$(C$8$P(B"	"$(CAj(B")
   ("$(C$8$P$$(B"	"$(CAk(B")
   ("$(C$8$P$)(B"	"$(CAl(B")
   ("$(C$8$P$1(B"	"$(CAm(B")

   ("$(C$8$Q(B"	"$(CAn(B")
   ("$(C$8$Q$!(B"	"$(CAo(B")
   ("$(C$8$Q$$(B"	"$(CAp(B")
   ("$(C$8$Q$)(B"	"$(CAq(B")
   ("$(C$8$Q$1(B"	"$(CAr(B")
   ("$(C$8$Q$2(B"	"$(CAs(B")
   ("$(C$8$Q$5(B"	"$(CAt(B")
   ("$(C$8$Q$7(B"	"$(CAu(B")

   ("$(C$8$S(B"	"$(CAv(B")
   ("$(C$8$S$!(B"	"$(CAw(B")
   ("$(C$8$S$$(B"	"$(CAx(B")
   ("$(C$8$S$'(B"	"$(CAy(B")
   ("$(C$8$S$)(B"	"$(CAz(B")
   ("$(C$8$S$)$1(B"	"$(CA{(B")
   ("$(C$8$S$1(B"	"$(CA|(B")
   ("$(C$8$S$2(B"	"$(CA}(B")
   ("$(C$8$S$5(B"	"$(CA~(B")
   ("$(C$8$S$7(B"	"$(CB!(B")
   ("$(C$8$S$8(B"	"$(CB"(B")
   ("$(C$8$S$<(B"	"$(CB#(B")
   ("$(C$8$S$=(B"	"$(CB$(B")

   ("$(C$9$?(B"	"$(CB%(B")
   ("$(C$9$?$!(B"	"$(CB&(B")
   ("$(C$9$?$$(B"	"$(CB'(B")
   ("$(C$9$?$$$>(B"	"$(CB((B")
   ("$(C$9$?$)(B"	"$(CB)(B")
   ("$(C$9$?$)$2(B"	"$(CB*(B")
   ("$(C$9$?$1(B"	"$(CB+(B")
   ("$(C$9$?$2(B"	"$(CB,(B")
   ("$(C$9$?$5(B"	"$(CB-(B")
   ("$(C$9$?$6(B"	"$(CB.(B")
   ("$(C$9$?$7(B"	"$(CB/(B")

   ("$(C$9$@(B"	"$(CB0(B")
   ("$(C$9$@$!(B"	"$(CB1(B")
   ("$(C$9$@$$(B"	"$(CB2(B")
   ("$(C$9$@$)(B"	"$(CB3(B")
   ("$(C$9$@$1(B"	"$(CB4(B")
   ("$(C$9$@$2(B"	"$(CB5(B")
   ("$(C$9$@$5(B"	"$(CB6(B")
   ("$(C$9$@$6(B"	"$(CB7(B")
   ("$(C$9$@$7(B"	"$(CB8(B")

   ("$(C$9$A(B"	"$(CB9(B")
   ("$(C$9$A$$(B"	"$(CB:(B")
   ("$(C$9$A$7(B"	"$(CB;(B")

   ("$(C$9$C(B"	"$(CB<(B")
   ("$(C$9$C$!(B"	"$(CB=(B")
   ("$(C$9$C$$(B"	"$(CB>(B")
   ("$(C$9$C$)(B"	"$(CB?(B")
   ("$(C$9$C$1(B"	"$(CB@(B")
   ("$(C$9$C$2(B"	"$(CBA(B")
   ("$(C$9$C$5(B"	"$(CBB(B")
   ("$(C$9$C$6(B"	"$(CBC(B")
   ("$(C$9$C$7(B"	"$(CBD(B")

   ("$(C$9$D(B"	"$(CBE(B")
   ("$(C$9$D$7(B"	"$(CBF(B")

   ("$(C$9$E(B"	"$(CBG(B")
   ("$(C$9$E$6(B"	"$(CBH(B")

   ("$(C$9$G(B"	"$(CBI(B")
   ("$(C$9$G$!(B"	"$(CBJ(B")
   ("$(C$9$G$$(B"	"$(CBK(B")
   ("$(C$9$G$)(B"	"$(CBL(B")
   ("$(C$9$G$1(B"	"$(CBM(B")
   ("$(C$9$G$2(B"	"$(CBN(B")
   ("$(C$9$G$5(B"	"$(CBO(B")
   ("$(C$9$G$7(B"	"$(CBP(B")
   ("$(C$9$G$:(B"	"$(CBQ(B")

   ("$(C$9$G$?(B"	"$(CBR(B")
   ("$(C$9$G$?$!(B"	"$(CBS(B")
   ("$(C$9$G$?$)(B"	"$(CBT(B")
   ("$(C$9$G$?$6(B"	"$(CBU(B")

   ("$(C$9$G$@(B"	"$(CBV(B")
   ("$(C$9$G$@$6(B"	"$(CBW(B")

   ("$(C$9$G$S(B"	"$(CBX(B")
   ("$(C$9$G$S$$(B"	"$(CBY(B")
   ("$(C$9$G$S$)(B"	"$(CBZ(B")
   ("$(C$9$G$S$1(B"	"$(CB[(B")
   ("$(C$9$G$S$2(B"	"$(CB\(B")

   ("$(C$9$K$7(B"	"$(CB](B")

   ("$(C$9$L(B"	"$(CB^(B")
   ("$(C$9$L$!(B"	"$(CB_(B")
   ("$(C$9$L$$(B"	"$(CB`(B")
   ("$(C$9$L$)(B"	"$(CBa(B")
   ("$(C$9$L$1(B"	"$(CBb(B")
   ("$(C$9$L$2(B"	"$(CBc(B")
   ("$(C$9$L$7(B"	"$(CBd(B")

   ("$(C$9$L$C(B"	"$(CBe(B")
   ("$(C$9$L$C$6(B"	"$(CBf(B")
   ("$(C$9$L$C$7(B"	"$(CBg(B")

   ("$(C$9$L$S(B"	"$(CBh(B")

   ("$(C$9$P(B"	"$(CBi(B")

   ("$(C$9$Q(B"	"$(CBj(B")
   ("$(C$9$Q$1(B"	"$(CBk(B")
   ("$(C$9$Q$5(B"	"$(CBl(B")
   ("$(C$9$Q$7(B"	"$(CBm(B")

   ("$(C$9$S(B"	"$(CBn(B")
   ("$(C$9$S$!(B"	"$(CBo(B")
   ("$(C$9$S$$(B"	"$(CBp(B")
   ("$(C$9$S$)(B"	"$(CBq(B")
   ("$(C$9$S$1(B"	"$(CBr(B")
   ("$(C$9$S$2(B"	"$(CBs(B")
   ("$(C$9$S$7(B"	"$(CBt(B")
   ("$(C$9$S$8(B"	"$(CBu(B")
   ("$(C$9$S$>(B"	"$(CBv(B")

   ("$(C$:$?(B"	"$(CBw(B")
   ("$(C$:$?$!(B"	"$(CBx(B")
   ("$(C$:$?$$(B"	"$(CBy(B")
   ("$(C$:$?$$$>(B"	"$(CBz(B")
   ("$(C$:$?$)(B"	"$(CB{(B")
   ("$(C$:$?$1(B"	"$(CB|(B")
   ("$(C$:$?$2(B"	"$(CB}(B")
   ("$(C$:$?$5(B"	"$(CB~(B")
   ("$(C$:$?$6(B"	"$(CC!(B")
   ("$(C$:$?$7(B"	"$(CC"(B")
   ("$(C$:$?$8(B"	"$(CC#(B")

   ("$(C$:$@(B"	"$(CC$(B")
   ("$(C$:$@$!(B"	"$(CC%(B")
   ("$(C$:$@$$(B"	"$(CC&(B")
   ("$(C$:$@$)(B"	"$(CC'(B")
   ("$(C$:$@$1(B"	"$(CC((B")
   ("$(C$:$@$2(B"	"$(CC)(B")
   ("$(C$:$@$5(B"	"$(CC*(B")
   ("$(C$:$@$6(B"	"$(CC+(B")
   ("$(C$:$@$7(B"	"$(CC,(B")

   ("$(C$:$A(B"	"$(CC-(B")
   ("$(C$:$A$$(B"	"$(CC.(B")
   ("$(C$:$A$$$>(B"	"$(CC/(B")
   ("$(C$:$A$)(B"	"$(CC0(B")
   ("$(C$:$A$1(B"	"$(CC1(B")
   ("$(C$:$A$7(B"	"$(CC2(B")

   ("$(C$:$C(B"	"$(CC3(B")
   ("$(C$:$C$!(B"	"$(CC4(B")
   ("$(C$:$C$$(B"	"$(CC5(B")
   ("$(C$:$C$)(B"	"$(CC6(B")
   ("$(C$:$C$1(B"	"$(CC7(B")
   ("$(C$:$C$2(B"	"$(CC8(B")
   ("$(C$:$C$5(B"	"$(CC9(B")
   ("$(C$:$C$6(B"	"$(CC:(B")
   ("$(C$:$C$7(B"	"$(CC;(B")

   ("$(C$:$D(B"	"$(CC<(B")
   ("$(C$:$D$!(B"	"$(CC=(B")
   ("$(C$:$D$$(B"	"$(CC>(B")
   ("$(C$:$D$)(B"	"$(CC?(B")
   ("$(C$:$D$1(B"	"$(CC@(B")
   ("$(C$:$D$2(B"	"$(CCA(B")
   ("$(C$:$D$5(B"	"$(CCB(B")
   ("$(C$:$D$7(B"	"$(CCC(B")

   ("$(C$:$E(B"	"$(CCD(B")
   ("$(C$:$E$$(B"	"$(CCE(B")
   ("$(C$:$E$6(B"	"$(CCF(B")

   ("$(C$:$F(B"	"$(CCG(B")
   ("$(C$:$F$$(B"	"$(CCH(B")
   ("$(C$:$F$7(B"	"$(CCI(B")

   ("$(C$:$G(B"	"$(CCJ(B")
   ("$(C$:$G$!(B"	"$(CCK(B")
   ("$(C$:$G$$(B"	"$(CCL(B")
   ("$(C$:$G$)(B"	"$(CCM(B")
   ("$(C$:$G$1(B"	"$(CCN(B")
   ("$(C$:$G$2(B"	"$(CCO(B")
   ("$(C$:$G$5(B"	"$(CCP(B")
   ("$(C$:$G$7(B"	"$(CCQ(B")

   ("$(C$:$G$?(B"	"$(CCR(B")
   ("$(C$:$G$?$$(B"	"$(CCS(B")
   ("$(C$:$G$?$)(B"	"$(CCT(B")
   ("$(C$:$G$?$7(B"	"$(CCU(B")

   ("$(C$:$G$S(B"	"$(CCV(B")
   ("$(C$:$G$S$$(B"	"$(CCW(B")
   ("$(C$:$G$S$)(B"	"$(CCX(B")
   ("$(C$:$G$S$1(B"	"$(CCY(B")
   ("$(C$:$G$S$2(B"	"$(CCZ(B")
   ("$(C$:$G$S$5(B"	"$(CC[(B")
   ("$(C$:$G$S$7(B"	"$(CC\(B")

   ("$(C$:$K(B"	"$(CC](B")
   ("$(C$:$K$1(B"	"$(CC^(B")

   ("$(C$:$L(B"	"$(CC_(B")
   ("$(C$:$L$!(B"	"$(CC`(B")
   ("$(C$:$L$$(B"	"$(CCa(B")
   ("$(C$:$L$)(B"	"$(CCb(B")
   ("$(C$:$L$1(B"	"$(CCc(B")
   ("$(C$:$L$2(B"	"$(CCd(B")
   ("$(C$:$L$5(B"	"$(CCe(B")
   ("$(C$:$L$7(B"	"$(CCf(B")

   ("$(C$:$L$C(B"	"$(CCg(B")
   ("$(C$:$L$C$6(B"	"$(CCh(B")

   ("$(C$:$L$D(B"	"$(CCi(B")
   ("$(C$:$L$D$$(B"	"$(CCj(B")

   ("$(C$:$L$S(B"	"$(CCk(B")
   ("$(C$:$L$S$$(B"	"$(CCl(B")
   ("$(C$:$L$S$)(B"	"$(CCm(B")
   ("$(C$:$L$S$1(B"	"$(CCn(B")
   ("$(C$:$L$S$2(B"	"$(CCo(B")
   ("$(C$:$L$S$5(B"	"$(CCp(B")
   ("$(C$:$L$S$7(B"	"$(CCq(B")

   ("$(C$:$P(B"	"$(CCr(B")
   ("$(C$:$P$$(B"	"$(CCs(B")
   ("$(C$:$P$)(B"	"$(CCt(B")
   ("$(C$:$P$1(B"	"$(CCu(B")
   ("$(C$:$P$7(B"	"$(CCv(B")

   ("$(C$:$Q(B"	"$(CCw(B")
   ("$(C$:$Q$!(B"	"$(CCx(B")
   ("$(C$:$Q$$(B"	"$(CCy(B")
   ("$(C$:$Q$)(B"	"$(CCz(B")
   ("$(C$:$Q$1(B"	"$(CC{(B")
   ("$(C$:$Q$2(B"	"$(CC|(B")
   ("$(C$:$Q$5(B"	"$(CC}(B")
   ("$(C$:$Q$7(B"	"$(CC~(B")

   ("$(C$:$S(B"	"$(CD!(B")
   ("$(C$:$S$!(B"	"$(CD"(B")
   ("$(C$:$S$$(B"	"$(CD#(B")
   ("$(C$:$S$'(B"	"$(CD$(B")
   ("$(C$:$S$)(B"	"$(CD%(B")
   ("$(C$:$S$)$!(B"	"$(CD&(B")
   ("$(C$:$S$1(B"	"$(CD'(B")
   ("$(C$:$S$2(B"	"$(CD((B")
   ("$(C$:$S$5(B"	"$(CD)(B")
   ("$(C$:$S$7(B"	"$(CD*(B")

   ("$(C$;$?(B"	"$(CD+(B")
   ("$(C$;$?$!(B"	"$(CD,(B")
   ("$(C$;$?$$(B"	"$(CD-(B")
   ("$(C$;$?$)(B"	"$(CD.(B")
   ("$(C$;$?$1(B"	"$(CD/(B")
   ("$(C$;$?$2(B"	"$(CD0(B")
   ("$(C$;$?$5(B"	"$(CD1(B")
   ("$(C$;$?$7(B"	"$(CD2(B")

   ("$(C$;$@(B"	"$(CD3(B")
   ("$(C$;$@$!(B"	"$(CD4(B")
   ("$(C$;$@$$(B"	"$(CD5(B")
   ("$(C$;$@$)(B"	"$(CD6(B")
   ("$(C$;$@$1(B"	"$(CD7(B")
   ("$(C$;$@$2(B"	"$(CD8(B")
   ("$(C$;$@$5(B"	"$(CD9(B")
   ("$(C$;$@$6(B"	"$(CD:(B")
   ("$(C$;$@$7(B"	"$(CD;(B")

   ("$(C$;$A(B"	"$(CD<(B")
   ("$(C$;$A$!(B"	"$(CD=(B")
   ("$(C$;$A$7(B"	"$(CD>(B")

   ("$(C$;$C(B"	"$(CD?(B")
   ("$(C$;$C$!(B"	"$(CD@(B")
   ("$(C$;$C$$(B"	"$(CDA(B")
   ("$(C$;$C$'(B"	"$(CDB(B")
   ("$(C$;$C$)(B"	"$(CDC(B")
   ("$(C$;$C$1(B"	"$(CDD(B")
   ("$(C$;$C$2(B"	"$(CDE(B")
   ("$(C$;$C$5(B"	"$(CDF(B")
   ("$(C$;$C$6(B"	"$(CDG(B")
   ("$(C$;$C$7(B"	"$(CDH(B")

   ("$(C$;$D(B"	"$(CDI(B")
   ("$(C$;$D$!(B"	"$(CDJ(B")
   ("$(C$;$D$$(B"	"$(CDK(B")
   ("$(C$;$D$)(B"	"$(CDL(B")
   ("$(C$;$D$1(B"	"$(CDM(B")
   ("$(C$;$D$2(B"	"$(CDN(B")
   ("$(C$;$D$5(B"	"$(CDO(B")
   ("$(C$;$D$7(B"	"$(CDP(B")

   ("$(C$;$E(B"	"$(CDQ(B")
   ("$(C$;$E$$(B"	"$(CDR(B")
   ("$(C$;$E$)(B"	"$(CDS(B")
   ("$(C$;$E$1(B"	"$(CDT(B")
   ("$(C$;$E$2(B"	"$(CDU(B")
   ("$(C$;$E$5(B"	"$(CDV(B")
   ("$(C$;$E$6(B"	"$(CDW(B")
   ("$(C$;$E$7(B"	"$(CDX(B")

   ("$(C$;$F(B"	"$(CDY(B")

   ("$(C$;$G(B"	"$(CDZ(B")
   ("$(C$;$G$!(B"	"$(CD[(B")
   ("$(C$;$G$$(B"	"$(CD\(B")
   ("$(C$;$G$)(B"	"$(CD](B")
   ("$(C$;$G$1(B"	"$(CD^(B")
   ("$(C$;$G$2(B"	"$(CD_(B")
   ("$(C$;$G$5(B"	"$(CD`(B")
   ("$(C$;$G$7(B"	"$(CDa(B")

   ("$(C$;$G$?(B"	"$(CDb(B")
   ("$(C$;$G$?$!(B"	"$(CDc(B")
   ("$(C$;$G$?$$(B"	"$(CDd(B")
   ("$(C$;$G$?$)(B"	"$(CDe(B")
   ("$(C$;$G$?$1(B"	"$(CDf(B")
   ("$(C$;$G$?$7(B"	"$(CDg(B")

   ("$(C$;$G$@(B"	"$(CDh(B")
   ("$(C$;$G$@$7(B"	"$(CDi(B")

   ("$(C$;$G$S(B"	"$(CDj(B")
   ("$(C$;$G$S$)(B"	"$(CDk(B")

   ("$(C$;$K(B"	"$(CDl(B")

   ("$(C$;$L(B"	"$(CDm(B")
   ("$(C$;$L$!(B"	"$(CDn(B")
   ("$(C$;$L$)(B"	"$(CDp(B")
   ("$(C$;$L$1(B"	"$(CDq(B")
   ("$(C$;$L$2(B"	"$(CDr(B")
   ("$(C$;$L$5(B"	"$(CDs(B")
   ("$(C$;$L$7(B"	"$(CDt(B")

   ("$(C$;$L$C(B"	"$(CDu(B")
   ("$(C$;$L$C$$(B"	"$(CDv(B")
   ("$(C$;$L$C$)(B"	"$(CDw(B")
   ("$(C$;$L$C$7(B"	"$(CDx(B")

   ("$(C$;$L$D(B"	"$(CDy(B")
   ("$(C$;$L$D$7(B"	"$(CDz(B")

   ("$(C$;$L$S(B"	"$(CD{(B")
   ("$(C$;$L$S$!(B"	"$(CD|(B")
   ("$(C$;$L$S$$(B"	"$(CD}(B")
   ("$(C$;$L$S$)(B"	"$(CD~(B")
   ("$(C$;$L$S$1(B"	"$(CE!(B")
   ("$(C$;$L$S$2(B"	"$(CE"(B")
   ("$(C$;$L$S$5(B"	"$(CE#(B")
   ("$(C$;$L$S$7(B"	"$(CE$(B")

   ("$(C$;$P(B"	"$(CE%(B")
   ("$(C$;$P$$(B"	"$(CE&(B")
   ("$(C$;$P$)(B"	"$(CE'(B")
   ("$(C$;$P$1(B"	"$(CE((B")

   ("$(C$;$Q(B"	"$(CE)(B")
   ("$(C$;$Q$!(B"	"$(CE*(B")
   ("$(C$;$Q$$(B"	"$(CE+(B")
   ("$(C$;$Q$)(B"	"$(CE,(B")
   ("$(C$;$Q$1(B"	"$(CE-(B")
   ("$(C$;$Q$2(B"	"$(CE.(B")
   ("$(C$;$Q$7(B"	"$(CE/(B")

   ("$(C$;$S(B"	"$(CE0(B")
   ("$(C$;$S$!(B"	"$(CE1(B")
   ("$(C$;$S$$(B"	"$(CE2(B")
   ("$(C$;$S$)(B"	"$(CE3(B")
   ("$(C$;$S$1(B"	"$(CE4(B")
   ("$(C$;$S$2(B"	"$(CE5(B")
   ("$(C$;$S$5(B"	"$(CE6(B")
   ("$(C$;$S$7(B"	"$(CE7(B")

   ("$(C$<$?(B"	"$(CE8(B")
   ("$(C$<$?$!(B"	"$(CE9(B")
   ("$(C$<$?$$(B"	"$(CE:(B")
   ("$(C$<$?$)(B"	"$(CE;(B")
   ("$(C$<$?$)$!(B"	"$(CE<(B")
   ("$(C$<$?$1(B"	"$(CE=(B")
   ("$(C$<$?$2(B"	"$(CE>(B")
   ("$(C$<$?$5(B"	"$(CE?(B")
   ("$(C$<$?$6(B"	"$(CE@(B")
   ("$(C$<$?$7(B"	"$(CEA(B")

   ("$(C$<$@(B"	"$(CEB(B")
   ("$(C$<$@$!(B"	"$(CEC(B")
   ("$(C$<$@$$(B"	"$(CED(B")
   ("$(C$<$@$)(B"	"$(CEE(B")
   ("$(C$<$@$1(B"	"$(CEF(B")
   ("$(C$<$@$2(B"	"$(CEG(B")
   ("$(C$<$@$5(B"	"$(CEH(B")
   ("$(C$<$@$6(B"	"$(CEI(B")
   ("$(C$<$@$7(B"	"$(CEJ(B")

   ("$(C$<$A(B"	"$(CEK(B")
   ("$(C$<$A$7(B"	"$(CEL(B")

   ("$(C$<$C(B"	"$(CEM(B")
   ("$(C$<$C$!(B"	"$(CEN(B")
   ("$(C$<$C$$(B"	"$(CEO(B")
   ("$(C$<$C$)(B"	"$(CEP(B")
   ("$(C$<$C$)$1(B"	"$(CEQ(B")
   ("$(C$<$C$1(B"	"$(CER(B")
   ("$(C$<$C$2(B"	"$(CES(B")
   ("$(C$<$C$5(B"	"$(CET(B")
   ("$(C$<$C$6(B"	"$(CEU(B")
   ("$(C$<$C$7(B"	"$(CEV(B")

   ("$(C$<$D(B"	"$(CEW(B")
   ("$(C$<$D$!(B"	"$(CEX(B")
   ("$(C$<$D$$(B"	"$(CEY(B")
   ("$(C$<$D$)(B"	"$(CEZ(B")
   ("$(C$<$D$1(B"	"$(CE[(B")
   ("$(C$<$D$2(B"	"$(CE\(B")
   ("$(C$<$D$5(B"	"$(CE](B")
   ("$(C$<$D$7(B"	"$(CE^(B")

   ("$(C$<$E(B"	"$(CE_(B")
   ("$(C$<$E$$(B"	"$(CE`(B")
   ("$(C$<$E$6(B"	"$(CEa(B")

   ("$(C$<$F(B"	"$(CEb(B")
   ("$(C$<$F$$(B"	"$(CEc(B")

   ("$(C$<$G(B"	"$(CEd(B")
   ("$(C$<$G$!(B"	"$(CEe(B")
   ("$(C$<$G$$(B"	"$(CEf(B")
   ("$(C$<$G$)(B"	"$(CEg(B")
   ("$(C$<$G$1(B"	"$(CEh(B")
   ("$(C$<$G$2(B"	"$(CEi(B")
   ("$(C$<$G$5(B"	"$(CEj(B")
   ("$(C$<$G$7(B"	"$(CEk(B")
   ("$(C$<$G$=(B"	"$(CEl(B")

   ("$(C$<$G$?(B"	"$(CEm(B")
   ("$(C$<$G$?$$(B"	"$(CEn(B")

   ("$(C$<$G$@(B"	"$(CEo(B")

   ("$(C$<$G$S(B"	"$(CEp(B")
   ("$(C$<$G$S$$(B"	"$(CEq(B")
   ("$(C$<$G$S$5(B"	"$(CEr(B")
   ("$(C$<$G$S$7(B"	"$(CEs(B")

   ("$(C$<$K(B"	"$(CEt(B")

   ("$(C$<$L(B"	"$(CEu(B")
   ("$(C$<$L$!(B"	"$(CEv(B")
   ("$(C$<$L$$(B"	"$(CEw(B")
   ("$(C$<$L$)(B"	"$(CEx(B")
   ("$(C$<$L$1(B"	"$(CEy(B")
   ("$(C$<$L$2(B"	"$(CEz(B")
   ("$(C$<$L$5(B"	"$(CE{(B")
   ("$(C$<$L$7(B"	"$(CE|(B")

   ("$(C$<$L$C(B"	"$(CE}(B")
   ("$(C$<$L$C$6(B"	"$(CE~(B")

   ("$(C$<$L$D(B"	"$(CF!(B")

   ("$(C$<$L$S(B"	"$(CF"(B")
   ("$(C$<$L$S$!(B"	"$(CF#(B")
   ("$(C$<$L$S$$(B"	"$(CF$(B")
   ("$(C$<$L$S$)(B"	"$(CF%(B")
   ("$(C$<$L$S$1(B"	"$(CF&(B")
   ("$(C$<$L$S$2(B"	"$(CF'(B")
   ("$(C$<$L$S$7(B"	"$(CF((B")

   ("$(C$<$P(B"	"$(CF)(B")
   ("$(C$<$P$$(B"	"$(CF*(B")
   ("$(C$<$P$)(B"	"$(CF+(B")
   ("$(C$<$P$1(B"	"$(CF,(B")
   ("$(C$<$P$7(B"	"$(CF-(B")

   ("$(C$<$Q(B"	"$(CF.(B")
   ("$(C$<$Q$!(B"	"$(CF/(B")
   ("$(C$<$Q$$(B"	"$(CF0(B")
   ("$(C$<$Q$'(B"	"$(CF1(B")
   ("$(C$<$Q$)(B"	"$(CF2(B")
   ("$(C$<$Q$)$1(B"	"$(CF3(B")
   ("$(C$<$Q$1(B"	"$(CF4(B")
   ("$(C$<$Q$2(B"	"$(CF5(B")
   ("$(C$<$Q$5(B"	"$(CF6(B")

   ("$(C$<$Q$S(B"	"$(CF7(B")
   ("$(C$<$Q$S$$(B"	"$(CF8(B")
   ("$(C$<$Q$S$)(B"	"$(CF9(B")
   ("$(C$<$Q$S$1(B"	"$(CF:(B")
   ("$(C$<$Q$S$2(B"	"$(CF;(B")

   ("$(C$<$S(B"	"$(CF<(B")
   ("$(C$<$S$!(B"	"$(CF=(B")
   ("$(C$<$S$$(B"	"$(CF>(B")
   ("$(C$<$S$)(B"	"$(CF?(B")
   ("$(C$<$S$1(B"	"$(CF@(B")
   ("$(C$<$S$2(B"	"$(CFA(B")
   ("$(C$<$S$5(B"	"$(CFB(B")
   ("$(C$<$S$7(B"	"$(CFC(B")

   ("$(C$=$?(B"	"$(CFD(B")
   ("$(C$=$?$!(B"	"$(CFE(B")
   ("$(C$=$?$"(B"	"$(CFF(B")
   ("$(C$=$?$$(B"	"$(CFG(B")
   ("$(C$=$?$)(B"	"$(CFH(B")
   ("$(C$=$?$)$1(B"	"$(CFI(B")
   ("$(C$=$?$1(B"	"$(CFJ(B")
   ("$(C$=$?$2(B"	"$(CFK(B")
   ("$(C$=$?$5(B"	"$(CFL(B")
   ("$(C$=$?$6(B"	"$(CFM(B")
   ("$(C$=$?$7(B"	"$(CFN(B")
   ("$(C$=$?$<(B"	"$(CFO(B")

   ("$(C$=$@(B"	"$(CFP(B")
   ("$(C$=$@$!(B"	"$(CFQ(B")
   ("$(C$=$@$$(B"	"$(CFR(B")
   ("$(C$=$@$)(B"	"$(CFS(B")
   ("$(C$=$@$1(B"	"$(CFT(B")
   ("$(C$=$@$2(B"	"$(CFU(B")
   ("$(C$=$@$5(B"	"$(CFV(B")
   ("$(C$=$@$6(B"	"$(CFW(B")
   ("$(C$=$@$7(B"	"$(CFX(B")

   ("$(C$=$A(B"	"$(CFY(B")
   ("$(C$=$A$!(B"	"$(CFZ(B")

   ("$(C$=$C(B"	"$(CF[(B")
   ("$(C$=$C$!(B"	"$(CF\(B")
   ("$(C$=$C$$(B"	"$(CF](B")
   ("$(C$=$C$1(B"	"$(CF_(B")
   ("$(C$=$C$2(B"	"$(CF`(B")
   ("$(C$=$C$5(B"	"$(CFa(B")
   ("$(C$=$C$6(B"	"$(CFb(B")
   ("$(C$=$C$7(B"	"$(CFc(B")

   ("$(C$=$D(B"	"$(CFd(B")
   ("$(C$=$D$!(B"	"$(CFe(B")
   ("$(C$=$D$$(B"	"$(CFf(B")
   ("$(C$=$D$)(B"	"$(CFg(B")
   ("$(C$=$D$1(B"	"$(CFh(B")
   ("$(C$=$D$2(B"	"$(CFi(B")
   ("$(C$=$D$5(B"	"$(CFj(B")
   ("$(C$=$D$7(B"	"$(CFk(B")

   ("$(C$=$E(B"	"$(CFl(B")
   ("$(C$=$E$$(B"	"$(CFm(B")
   ("$(C$=$E$)(B"	"$(CFn(B")
   ("$(C$=$E$1(B"	"$(CFo(B")
   ("$(C$=$E$2(B"	"$(CFp(B")
   ("$(C$=$E$6(B"	"$(CFq(B")
   ("$(C$=$E$7(B"	"$(CFr(B")

   ("$(C$=$F(B"	"$(CFs(B")
   ("$(C$=$F$)(B"	"$(CFt(B")
   ("$(C$=$F$2(B"	"$(CFu(B")
   ("$(C$=$F$5(B"	"$(CFv(B")

   ("$(C$=$G(B"	"$(CFw(B")
   ("$(C$=$G$!(B"	"$(CFx(B")
   ("$(C$=$G$$(B"	"$(CFy(B")
   ("$(C$=$G$)(B"	"$(CFz(B")
   ("$(C$=$G$1(B"	"$(CF{(B")
   ("$(C$=$G$2(B"	"$(CF|(B")
   ("$(C$=$G$5(B"	"$(CF}(B")
   ("$(C$=$G$7(B"	"$(CF~(B")

   ("$(C$=$G$?(B"	"$(CG!(B")
   ("$(C$=$G$?$7(B"	"$(CG"(B")

   ("$(C$=$G$S(B"	"$(CG#(B")
   ("$(C$=$G$S$$(B"	"$(CG$(B")

   ("$(C$=$K(B"	"$(CG%(B")
   ("$(C$=$K$$(B"	"$(CG&(B")
   ("$(C$=$K$)(B"	"$(CG'(B")
   ("$(C$=$K$2(B"	"$(CG((B")
   ("$(C$=$K$5(B"	"$(CG)(B")

   ("$(C$=$L(B"	"$(CG*(B")
   ("$(C$=$L$!(B"	"$(CG+(B")
   ("$(C$=$L$$(B"	"$(CG,(B")
   ("$(C$=$L$'(B"	"$(CG-(B")
   ("$(C$=$L$)(B"	"$(CG.(B")
   ("$(C$=$L$)$1(B"	"$(CG/(B")
   ("$(C$=$L$1(B"	"$(CG0(B")
   ("$(C$=$L$2(B"	"$(CG1(B")
   ("$(C$=$L$5(B"	"$(CG2(B")
   ("$(C$=$L$7(B"	"$(CG3(B")

   ("$(C$=$L$C(B"	"$(CG4(B")
   ("$(C$=$L$C$7(B"	"$(CG5(B")

   ("$(C$=$L$S(B"	"$(CG6(B")
   ("$(C$=$L$S$$(B"	"$(CG7(B")
   ("$(C$=$L$S$)(B"	"$(CG8(B")
   ("$(C$=$L$S$1(B"	"$(CG9(B")
   ("$(C$=$L$S$5(B"	"$(CG:(B")

   ("$(C$=$P(B"	"$(CG;(B")
   ("$(C$=$P$$(B"	"$(CG<(B")
   ("$(C$=$P$)(B"	"$(CG=(B")
   ("$(C$=$P$1(B"	"$(CG>(B")
   ("$(C$=$P$5(B"	"$(CG?(B")
   ("$(C$=$P$7(B"	"$(CG@(B")

   ("$(C$=$Q(B"	"$(CGA(B")
   ("$(C$=$Q$$(B"	"$(CGB(B")
   ("$(C$=$Q$)(B"	"$(CGC(B")
   ("$(C$=$Q$1(B"	"$(CGD(B")
   ("$(C$=$Q$2(B"	"$(CGE(B")
   ("$(C$=$Q$5(B"	"$(CGF(B")

   ("$(C$=$S(B"	"$(CGG(B")
   ("$(C$=$S$!(B"	"$(CGH(B")
   ("$(C$=$S$$(B"	"$(CGI(B")
   ("$(C$=$S$)(B"	"$(CGJ(B")
   ("$(C$=$S$1(B"	"$(CGK(B")
   ("$(C$=$S$2(B"	"$(CGL(B")
   ("$(C$=$S$5(B"	"$(CGM(B")
   ("$(C$=$S$7(B"	"$(CGN(B")

   ("$(C$>$?(B"	"$(CGO(B")
   ("$(C$>$?$!(B"	"$(CGP(B")
   ("$(C$>$?$$(B"	"$(CGQ(B")
   ("$(C$>$?$)(B"	"$(CGR(B")
   ("$(C$>$?$)$<(B"	"$(CGS(B")
   ("$(C$>$?$1(B"	"$(CGT(B")
   ("$(C$>$?$2(B"	"$(CGU(B")
   ("$(C$>$?$5(B"	"$(CGV(B")
   ("$(C$>$?$7(B"	"$(CGW(B")

   ("$(C$>$@(B"	"$(CGX(B")
   ("$(C$>$@$!(B"	"$(CGY(B")
   ("$(C$>$@$$(B"	"$(CGZ(B")
   ("$(C$>$@$)(B"	"$(CG[(B")
   ("$(C$>$@$1(B"	"$(CG\(B")
   ("$(C$>$@$2(B"	"$(CG](B")
   ("$(C$>$@$5(B"	"$(CG^(B")
   ("$(C$>$@$6(B"	"$(CG_(B")
   ("$(C$>$@$7(B"	"$(CG`(B")

   ("$(C$>$A(B"	"$(CGa(B")
   ("$(C$>$A$7(B"	"$(CGb(B")

   ("$(C$>$C(B"	"$(CGc(B")
   ("$(C$>$C$!(B"	"$(CGd(B")
   ("$(C$>$C$$(B"	"$(CGe(B")
   ("$(C$>$C$)(B"	"$(CGf(B")
   ("$(C$>$C$)$1(B"	"$(CGg(B")
   ("$(C$>$C$1(B"	"$(CGh(B")
   ("$(C$>$C$2(B"	"$(CGi(B")
   ("$(C$>$C$5(B"	"$(CGj(B")
   ("$(C$>$C$7(B"	"$(CGk(B")

   ("$(C$>$D(B"	"$(CGl(B")
   ("$(C$>$D$!(B"	"$(CGm(B")
   ("$(C$>$D$$(B"	"$(CGn(B")
   ("$(C$>$D$)(B"	"$(CGo(B")
   ("$(C$>$D$1(B"	"$(CGp(B")
   ("$(C$>$D$2(B"	"$(CGq(B")
   ("$(C$>$D$5(B"	"$(CGr(B")
   ("$(C$>$D$7(B"	"$(CGs(B")

   ("$(C$>$E(B"	"$(CGt(B")
   ("$(C$>$E$!(B"	"$(CGu(B")
   ("$(C$>$E$$(B"	"$(CGv(B")
   ("$(C$>$E$)(B"	"$(CGw(B")
   ("$(C$>$E$1(B"	"$(CGx(B")
   ("$(C$>$E$2(B"	"$(CGy(B")
   ("$(C$>$E$5(B"	"$(CGz(B")
   ("$(C$>$E$6(B"	"$(CG{(B")
   ("$(C$>$E$7(B"	"$(CG|(B")

   ("$(C$>$F(B"	"$(CG}(B")
   ("$(C$>$F$$(B"	"$(CG~(B")
   ("$(C$>$F$)(B"	"$(CH!(B")
   ("$(C$>$F$2(B"	"$(CH"(B")

   ("$(C$>$G(B"	"$(CH#(B")
   ("$(C$>$G$!(B"	"$(CH$(B")
   ("$(C$>$G$$(B"	"$(CH%(B")
   ("$(C$>$G$)(B"	"$(CH&(B")
   ("$(C$>$G$)$<(B"	"$(CH'(B")
   ("$(C$>$G$1(B"	"$(CH((B")
   ("$(C$>$G$2(B"	"$(CH)(B")
   ("$(C$>$G$5(B"	"$(CH*(B")
   ("$(C$>$G$7(B"	"$(CH+(B")
   ("$(C$>$G$<(B"	"$(CH,(B")

   ("$(C$>$G$?(B"	"$(CH-(B")
   ("$(C$>$G$?$!(B"	"$(CH.(B")
   ("$(C$>$G$?$$(B"	"$(CH/(B")
   ("$(C$>$G$?$)(B"	"$(CH0(B")
   ("$(C$>$G$?$5(B"	"$(CH1(B")
   ("$(C$>$G$?$7(B"	"$(CH2(B")

   ("$(C$>$G$@(B"	"$(CH3(B")
   ("$(C$>$G$@$!(B"	"$(CH4(B")
   ("$(C$>$G$@$$(B"	"$(CH5(B")
   ("$(C$>$G$@$5(B"	"$(CH6(B")
   ("$(C$>$G$@$7(B"	"$(CH7(B")

   ("$(C$>$G$S(B"	"$(CH8(B")
   ("$(C$>$G$S$!(B"	"$(CH9(B")
   ("$(C$>$G$S$$(B"	"$(CH:(B")
   ("$(C$>$G$S$)(B"	"$(CH;(B")
   ("$(C$>$G$S$2(B"	"$(CH<(B")
   ("$(C$>$G$S$5(B"	"$(CH=(B")
   ("$(C$>$G$S$7(B"	"$(CH>(B")

   ("$(C$>$K(B"	"$(CH?(B")
   ("$(C$>$K$$(B"	"$(CH@(B")
   ("$(C$>$K$)(B"	"$(CHA(B")
   ("$(C$>$K$2(B"	"$(CHB(B")
   ("$(C$>$K$5(B"	"$(CHC(B")

   ("$(C$>$L(B"	"$(CHD(B")
   ("$(C$>$L$!(B"	"$(CHE(B")
   ("$(C$>$L$$(B"	"$(CHF(B")
   ("$(C$>$L$)(B"	"$(CHG(B")
   ("$(C$>$L$)$<(B"	"$(CHH(B")
   ("$(C$>$L$1(B"	"$(CHI(B")
   ("$(C$>$L$5(B"	"$(CHJ(B")
   ("$(C$>$L$7(B"	"$(CHK(B")

   ("$(C$>$L$C(B"	"$(CHL(B")
   ("$(C$>$L$C$$(B"	"$(CHM(B")
   ("$(C$>$L$C$)(B"	"$(CHN(B")
   ("$(C$>$L$C$1(B"	"$(CHO(B")
   ("$(C$>$L$C$7(B"	"$(CHP(B")

   ("$(C$>$L$D(B"	"$(CHQ(B")
   ("$(C$>$L$D$!(B"	"$(CHR(B")
   ("$(C$>$L$D$$(B"	"$(CHS(B")
   ("$(C$>$L$D$)(B"	"$(CHT(B")
   ("$(C$>$L$D$7(B"	"$(CHU(B")

   ("$(C$>$L$S(B"	"$(CHV(B")
   ("$(C$>$L$S$!(B"	"$(CHW(B")
   ("$(C$>$L$S$$(B"	"$(CHX(B")
   ("$(C$>$L$S$)(B"	"$(CHY(B")
   ("$(C$>$L$S$1(B"	"$(CHZ(B")
   ("$(C$>$L$S$2(B"	"$(CH[(B")
   ("$(C$>$L$S$5(B"	"$(CH\(B")
   ("$(C$>$L$S$7(B"	"$(CH](B")

   ("$(C$>$P(B"	"$(CH^(B")
   ("$(C$>$P$!(B"	"$(CH_(B")
   ("$(C$>$P$$(B"	"$(CH`(B")
   ("$(C$>$P$)(B"	"$(CHa(B")
   ("$(C$>$P$1(B"	"$(CHb(B")
   ("$(C$>$P$5(B"	"$(CHc(B")
   ("$(C$>$P$7(B"	"$(CHd(B")

   ("$(C$>$Q(B"	"$(CHe(B")
   ("$(C$>$Q$!(B"	"$(CHf(B")
   ("$(C$>$Q$$(B"	"$(CHg(B")
   ("$(C$>$Q$$$>(B"	"$(CHh(B")
   ("$(C$>$Q$'(B"	"$(CHi(B")
   ("$(C$>$Q$)(B"	"$(CHj(B")
   ("$(C$>$Q$)$!(B"	"$(CHk(B")
   ("$(C$>$Q$1(B"	"$(CHl(B")
   ("$(C$>$Q$2(B"	"$(CHm(B")
   ("$(C$>$Q$5(B"	"$(CHn(B")
   ("$(C$>$Q$7(B"	"$(CHo(B")
   ("$(C$>$Q$<(B"	"$(CHp(B")

   ("$(C$>$Q$S(B"	"$(CHq(B")
   ("$(C$>$Q$S$$(B"	"$(CHr(B")
   ("$(C$>$Q$S$)(B"	"$(CHs(B")
   ("$(C$>$Q$S$1(B"	"$(CHt(B")
   ("$(C$>$Q$S$2(B"	"$(CHu(B")
   ("$(C$>$Q$S$7(B"	"$(CHv(B")

   ("$(C$>$S(B"	"$(CHw(B")
   ("$(C$>$S$!(B"	"$(CHx(B")
   ("$(C$>$S$$(B"	"$(CHy(B")
   ("$(C$>$S$)(B"	"$(CHz(B")
   ("$(C$>$S$1(B"	"$(CH{(B")
   ("$(C$>$S$2(B"	"$(CH|(B")
   ("$(C$>$S$5(B"	"$(CH}(B")
   ("$(C$>$S$7(B"	"$(CH~(B"))

  (dolist (ascii '(("1" . "$(C#1(B")  ("2" . "$(C#2(B")  ("3" . "$(C#3(B")  ("4" . "$(C#4(B")
		     ("5" . "$(C#5(B")  ("6" . "$(C#6(B")  ("7" . "$(C#7(B")  ("8" . "$(C#8(B")
		     ("9" . "$(C#9(B")  ("0" . "$(C#0(B")
		     (" " . "$(C!!(B")  ("!" . "$(C#!(B")  ("@" . "$(C#@(B")  ("#" . "$(C##(B")
		     ("$" . "$(C#$(B")  ("%" . "$(C#%(B")  ("^" . "$(C#^(B")  ("&" . "$(C#&(B")
		     ("*" . "$(C#*(B")  ("(" . "$(C#((B")  (")" . "$(C#)(B")
		     ("-" . "$(C#-(B")  ("=" . "$(C#=(B")  ("`" . "$(C#`(B")  ("\\" . "$(C#\(B")
		     ("|" . "$(C#|(B")  ("_" . "$(C#_(B")  ("+" . "$(C#+(B")  ("~" . "$(C#~(B")
		     ("[" . "$(C!8(B")  ("]" . "$(C!9(B")  ("{" . "$(C#{(B")  ("}" . "$(C#}(B")
		     (":" . "$(C#:(B")  (";" . "$(C#;(B")  ("\"" . "$(C#"(B") ("'" . "$(C#'(B")
		     ("<" . "$(C#<(B")  (">" . "$(C#>(B")  ("?" . "$(C#?(B")  ("/" . "$(C#/(B")
		     ("," . "$(C#,(B")  ("." . "$(C#.(B")
		     ("a" . "$(C#a(B")  ("b" . "$(C#b(B")  ("c" . "$(C#c(B")  ("d" . "$(C#d(B")
		     ("e" . "$(C#e(B")  ("f" . "$(C#f(B")  ("g" . "$(C#g(B")  ("h" . "$(C#h(B")
		     ("i" . "$(C#i(B")  ("j" . "$(C#j(B")  ("k" . "$(C#k(B")  ("l" . "$(C#l(B")
		     ("m" . "$(C#m(B")  ("n" . "$(C#n(B")  ("o" . "$(C#o(B")  ("p" . "$(C#p(B")
		     ("q" . "$(C#q(B")  ("r" . "$(C#r(B")  ("s" . "$(C#s(B")  ("t" . "$(C#t(B")
		     ("u" . "$(C#u(B")  ("v" . "$(C#v(B")  ("w" . "$(C#w(B")  ("x" . "$(C#x(B")
		     ("y" . "$(C#y(B")  ("z" . "$(C#z(B")
		     ("A" . "$(C#A(B")  ("B" . "$(C#B(B")  ("C" . "$(C#C(B")  ("D" . "$(C#D(B")
		     ("E" . "$(C#E(B")  ("F" . "$(C#F(B")  ("G" . "$(C#G(B")  ("H" . "$(C#H(B")
		     ("I" . "$(C#I(B")  ("J" . "$(C#J(B")  ("K" . "$(C#K(B")  ("L" . "$(C#L(B")
		     ("M" . "$(C#M(B")  ("N" . "$(C#N(B")  ("O" . "$(C#O(B")  ("P" . "$(C#P(B")
		     ("Q" . "$(C#Q(B")  ("R" . "$(C#R(B")  ("S" . "$(C#S(B")  ("T" . "$(C#T(B")
		     ("U" . "$(C#U(B")  ("V" . "$(C#V(B")  ("W" . "$(C#W(B")  ("X" . "$(C#X(B")
		     ("Y" . "$(C#Y(B")  ("Z" . "$(C#Z(B")))
    (let ((in (car ascii)) (out (cdr ascii)))
	(its-defrule (concat its-pankak-escape in) in)
	(its-defrule (concat its-jeonkak-escape in) out))))

(define-its-state-machine-append its-hangul-map
  (its-defrule "[" its-hangul-open-bracket)
  (its-defrule "]" its-hangul-close-bracket)
  (its-defrule "." its-hangul-period)
  (its-defrule "," its-hangul-comma)

  (if its-hangul-enable-jeonkak-alphabet
	(progn
	  (its-defrule   "1"   "$(C#1(B")  (its-defrule   "2"   "$(C#2(B")
	  (its-defrule   "3"   "$(C#3(B")  (its-defrule   "4"   "$(C#4(B")
	  (its-defrule   "5"   "$(C#5(B")  (its-defrule   "6"   "$(C#6(B")
	  (its-defrule   "7"   "$(C#7(B")  (its-defrule   "8"   "$(C#8(B")
	  (its-defrule   "9"   "$(C#9(B")  (its-defrule   "0"   "$(C#0(B")
	  (its-defrule   "!"   "$(C#!(B")  (its-defrule   "@"   "$(C#@(B")
	  (its-defrule   "#"   "$(C##(B")  (its-defrule   "$"   "$(C#$(B")
	  (its-defrule   "%"   "$(C#%(B")  (its-defrule   "^"   "$(C#^(B")
	  (its-defrule   "&"   "$(C#&(B")  (its-defrule   "*"   "$(C#*(B")
	  (its-defrule   "("   "$(C#((B")  (its-defrule   ")"   "$(C#)(B")
	  (its-defrule   "-"   "$(C#-(B")  (its-defrule   "~"   "$(C#~(B")
	  (its-defrule   "="   "$(C#=(B")  (its-defrule   "`"   "$(C#`(B")
	  (its-defrule   "\\"  "$(C#\(B")  (its-defrule   "|"   "$(C#|(B")
	  (its-defrule   "_"   "$(C#_(B")  (its-defrule   "+"   "$(C#+(B")
	  (its-defrule   "{"   "$(C#{(B")  (its-defrule   "}"   "$(C#}(B")
	  (its-defrule   ":"   "$(C#:(B")  (its-defrule   ";"   "$(C#;(B")
	  (its-defrule   "\""  "$(C#"(B")  (its-defrule   "'"   "$(C#'(B")
	  (its-defrule   "<"   "$(C#<(B")  (its-defrule   ">"   "$(C#>(B")
	  (its-defrule   "?"   "$(C#?(B")  (its-defrule   "/"   "$(C#/(B"))
    (progn
	(its-defrule   "1"   "1")  (its-defrule   "2"   "2")
	(its-defrule   "3"   "3")  (its-defrule   "4"   "4")
	(its-defrule   "5"   "5")  (its-defrule   "6"   "6")
	(its-defrule   "7"   "7")  (its-defrule   "8"   "8")
	(its-defrule   "9"   "9")  (its-defrule   "0"   "0")
	(its-defrule   "!"   "!")  (its-defrule   "@"   "@")
	(its-defrule   "#"   "#")  (its-defrule   "$"   "$")
	(its-defrule   "%"   "%")  (its-defrule   "^"   "^")
	(its-defrule   "&"   "&")  (its-defrule   "*"   "*")
	(its-defrule   "("   "(")  (its-defrule   ")"   ")")
	(its-defrule   "-"   "-")  (its-defrule   "~"   "~")
	(its-defrule   "="   "=")  (its-defrule   "`"   "`")
	(its-defrule   "\\"  "\\") (its-defrule   "|"   "|")
	(its-defrule   "_"   "_")  (its-defrule   "+"   "+")
	(its-defrule   "{"   "{")  (its-defrule   "}"   "}")
	(its-defrule   ":"   ":")  (its-defrule   ";"   ";")
	(its-defrule   "\""  "\"") (its-defrule   "'"   "'")
	(its-defrule   "<"   "<")  (its-defrule   ">"   ">")
	(its-defrule   "?"   "?")  (its-defrule   "/"   "/"))))

(provide 'its/hangul)
