// Copyright 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "cc/output/software_output_device.h"

#include "base/logging.h"
#include "cc/output/software_frame_data.h"
#include "third_party/skia/include/core/SkBitmapDevice.h"
#include "third_party/skia/include/core/SkCanvas.h"
#include "ui/gfx/skia_util.h"
#include "ui/gfx/vsync_provider.h"

namespace cc {

SoftwareOutputDevice::SoftwareOutputDevice() {}

SoftwareOutputDevice::~SoftwareOutputDevice() {}

void SoftwareOutputDevice::Resize(const gfx::Size& viewport_size) {
  if (viewport_size_ == viewport_size)
    return;

  viewport_size_ = viewport_size;
  device_ = skia::AdoptRef(new SkBitmapDevice(SkBitmap::kARGB_8888_Config,
      viewport_size.width(), viewport_size.height(), true));
  canvas_ = skia::AdoptRef(new SkCanvas(device_.get()));
}

SkCanvas* SoftwareOutputDevice::BeginPaint(const gfx::Rect& damage_rect) {
  DCHECK(device_);
  damage_rect_ = damage_rect;
  return canvas_.get();
}

void SoftwareOutputDevice::EndPaint(SoftwareFrameData* frame_data) {
  DCHECK(frame_data);
  frame_data->id = 0;
  frame_data->size = viewport_size_;
  frame_data->damage_rect = damage_rect_;
  frame_data->handle = base::SharedMemory::NULLHandle();
}

void SoftwareOutputDevice::CopyToBitmap(
    const gfx::Rect& rect, SkBitmap* output) {
  DCHECK(device_);
  const SkBitmap& bitmap = device_->accessBitmap(false);
  bitmap.extractSubset(output, gfx::RectToSkIRect(rect));
}

void SoftwareOutputDevice::Scroll(const gfx::Vector2d& delta,
                                  const gfx::Rect& clip_rect) {
  NOTIMPLEMENTED();
}

void SoftwareOutputDevice::ReclaimSoftwareFrame(unsigned id) {
  NOTIMPLEMENTED();
}

gfx::VSyncProvider* SoftwareOutputDevice::GetVSyncProvider() {
  return vsync_provider_.get();
}

}  // namespace cc
