# search, a bzr plugin for searching within bzr branches/repositories.
# Copyright (C) 2011 Jelmer Vernooij
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as published
# by the Free Software Foundation.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
# 

"""Tests for the smart server verbs."""

from bzrlib import tests
from bzrlib.branch import Branch
from bzrlib.smart import (
    request as smart_req,
    )

from bzrlib.plugins.search import (
    errors,
    index,
    )
from bzrlib.plugins.search.remote import (
    RemoteIndex,
    SmartServerBranchRequestOpenIndex,
    )


class TestSmartServerBranchRequestOpenIndex(
    tests.TestCaseWithMemoryTransport):

    def test_missing(self):
        """For an empty branch, the result is ('no', )."""
        backing = self.get_transport()
        request = SmartServerBranchRequestOpenIndex(backing)
        self.make_branch('.')
        self.assertEqual(smart_req.SmartServerResponse(('no', )),
            request.execute(''))

    def test_present(self):
        """For a branch with an index, ('yes', ) is returned."""
        backing = self.get_transport()
        request = SmartServerBranchRequestOpenIndex(backing)
        b = self.make_branch('.')
        index.init_index(b)
        self.assertEqual(smart_req.SmartServerResponse(('yes', )),
            request.execute(''))


class TestRemoteIndex(tests.TestCaseWithTransport):

    def test_no_index(self):
        local_branch = self.make_branch('.')
        remote_transport = self.make_smart_server('.')
        remote_branch = Branch.open_from_transport(remote_transport)
        self.assertRaises(errors.NoSearchIndex, RemoteIndex.open,
            remote_branch)

    def test_open(self):
        local_branch = self.make_branch('.')
        index.init_index(local_branch)
        remote_transport = self.make_smart_server('.')
        remote_branch = Branch.open_from_transport(remote_transport)
        idx = RemoteIndex.open(remote_branch)
        self.assertIsInstance(idx, RemoteIndex)

    def test_init(self):
        local_branch = self.make_branch('.')
        remote_transport = self.make_smart_server('.')
        remote_branch = Branch.open_from_transport(remote_transport)
        idx = index.init_index(remote_branch)
        self.assertIsInstance(idx, RemoteIndex)

    def test_init_exists(self):
        local_branch = self.make_branch('.')
        index.init_index(local_branch)
        remote_transport = self.make_smart_server('.')
        remote_branch = Branch.open_from_transport(remote_transport)
        #self.assertRaises( index.init_index, remote_branch)


class TestWithRemoteIndex(tests.TestCaseWithTransport):

    def make_remote_index(self):
        tree = self.make_branch_and_tree('.')
        local_branch = tree.branch
        index.init_index(local_branch)
        remote_transport = self.make_smart_server('.')
        remote_branch = Branch.open_from_transport(remote_transport)
        return tree, remote_branch, RemoteIndex.open(remote_branch)

    def test_index_revisions(self):
        tree, branch, index = self.make_remote_index()
        tree.commit(message="message", rev_id='revid1')
        index.index_revisions(branch, ['revid1'])
        self.assertEquals([('revid1',)], list(index.indexed_revisions()))

    def test_indexed_revisions(self):
        tree, branch, remote_index = self.make_remote_index()
        tree.commit(message="message", rev_id='revid1')
        self.assertEquals([], list(remote_index.indexed_revisions()))
        local_index = index.open_index_branch(tree.branch)
        local_index.index_revisions(tree.branch, ['revid1'])
        self.assertEquals([('revid1',)], list(remote_index.indexed_revisions()))

    def test_suggest(self):
        tree, branch, remote_index = self.make_remote_index()
        tree.commit(message="first", rev_id='revid1')
        local_index = index.open_index_branch(tree.branch)
        local_index.index_revisions(tree.branch, ['revid1'])
        self.assertEquals([(u'first',)], list(remote_index.suggest([(u'f',)])))

    def test_search(self):
        tree, branch, remote_index = self.make_remote_index()
        # The double-space is a cheap smoke test for the tokeniser.
        revid = tree.commit('first  post')
        remote_index.index_revisions(branch, [revid])
        results = list(remote_index.search([('post',)]))
        self.assertEqual(1, len(results))
        self.assertIsInstance(results[0], index.RevisionHit)
        self.assertEqual((revid,), results[0].revision_key)
