//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/SampleDesigner/ParticleForm.cpp
//! @brief     Implements class ParticleForm
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/SampleDesigner/ParticleForm.h"
#include "GUI/Model/Sample/FormFactorItemCatalog.h"
#include "GUI/Model/Sample/FormFactorItems.h"
#include "GUI/Model/Sample/ParticleItem.h"
#include "GUI/Support/Util/ActionFactory.h"
#include "GUI/View/SampleDesigner/FormLayouter.h"
#include "GUI/View/SampleDesigner/MaterialInplaceForm.h"
#include "GUI/View/Tool/GroupBoxCollapser.h"
#include <QAction>

ParticleForm::ParticleForm(QWidget* parent, ParticleItem* particleItem, bool allowAbundance,
                           SampleEditorController* ec, bool allowRemove)
    : QGroupBox(parent)
{
    const QString formfactor = FormFactorItemCatalog::menuEntry(particleItem->formFactorItem());
    setTitle("Particle (" + formfactor + ")");

    FormLayouter layouter(this, ec);
    layouter.addRow("Material", new MaterialInplaceForm(this, particleItem, ec));
    layouter.addGroupOfValues("Geometry", particleItem->formFactorItem()->geometryProperties());
    layouter.addVector(particleItem->position(), false);
    layouter.addSelection(particleItem->rotationSelection());
    if (allowAbundance)
        layouter.addValue(particleItem->abundance());

    auto* collapser = GroupBoxCollapser::installIntoGroupBox(this);
    collapser->setExpanded(particleItem->isExpandParticle());
    connect(collapser, &GroupBoxCollapser::toggled, this,
            [particleItem](bool b) { particleItem->setExpandParticle(b); });

    // top right corner actions
    // show in real space
    {
        auto* showInRealspaceAction = ActionFactory::createShowInRealspaceAction(
            this, "particle", [ec, particleItem] { ec->requestViewInRealspace(particleItem); });
        collapser->addAction(showInRealspaceAction);
    }
    // duplicate
    {
        m_duplicateAction = ActionFactory::createDuplicateAction(
            this, "particle", [ec, particleItem] { ec->duplicateItemWithParticles(particleItem); });
        collapser->addAction(m_duplicateAction);
    }
    // remove
    {
        m_removeAction = ActionFactory::createRemoveAction(
            this, "particle", [ec, particleItem] { ec->removeParticle(particleItem); });
        if (allowRemove)
            collapser->addAction(m_removeAction);
    }
}

void ParticleForm::enableStructureEditing(bool b)
{
    m_removeAction->setVisible(b);
    m_duplicateAction->setVisible(b);
}
