/* xoreos-tools - Tools to help with xoreos development
 *
 * xoreos-tools is the legal property of its developers, whose names
 * can be found in the AUTHORS file distributed with this source
 * distribution.
 *
 * xoreos-tools is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * xoreos-tools is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with xoreos-tools. If not, see <http://www.gnu.org/licenses/>.
 */

/** @file
 *   Resolve a GDA column header hash back to its string.
 */

#include "src/common/util.h"
#include "src/common/binsearch.h"

#include "src/aurora/gdaheaders.h"

namespace Aurora {

typedef Common::BinSearchValue<uint32, const char *> GDAHeaderHash;

/** All currently known GDA column header strings, together with their CRC32 hashes.
 *
 *  Note: For the binary search to work, this list needs to stay sorted by hash value!
 */
static const GDAHeaderHash kGDAHeaderHashes[] = {
	{   1421660U, "AttackScatter"               },
	{   3607720U, "DIScanCode"                  },
	{   4376397U, "FPS"                         },
	{   6714328U, "SpecialCase"                 },
	{   7038726U, "Bank3"                       },
	{   8828510U, "Sylvan"                      },
	{   8937265U, "Draw_Weapon"                 },
	{   9293919U, "Closed_Unlocked"             },
	{  12124062U, "Test2"                       },
	{  14904291U, "Top0"                        },
	{  15699404U, "A5"                          },
	{  17060417U, "TargetFade"                  },
	{  21861984U, "Used"                        },
	{  22033083U, "Projectile"                  },
	{  23668384U, "Button3"                     },
	{  24195820U, "RuneCount"                   },
	{  25383073U, "Orz200ar_Commons"            },
	{  26291545U, "ChaseTable"                  },
	{  28006648U, "Extreme_Effect"              },
	{  28895966U, "CanRotateIcon"               },
	{  36343957U, "WidgetType"                  },
	{  38134867U, "Facing"                      },
	{  42582947U, "NumberRings"                 },
	{  44583290U, "Prop1Value"                  },
	{  48449665U, "Queue"                       },
	{  49643810U, "Inescapable"                 },
	{  51797261U, "WaspSwarm"                   },
	{  51929172U, "Game_CbtStrFSpeed"           },
	{  55381987U, "DRMod"                       },
	{  55683016U, "LookDownAnimation"           },
	{  59147525U, "Loops9"                      },
	{  60769971U, "FGlowIntensity"              },
	{  61447571U, "Opening"                     },
	{  61946878U, "Open_2"                      },
	{  62325948U, "AbilityScoreModifier"        },
	{  62550588U, "Animation1"                  },
	{  63435298U, "Condition"                   },
	{  63456459U, "SliderMin"                   },
	{  65813985U, "NrrgalWorker"                },
	{  69936867U, "CharGenLabel"                },
	{  70083147U, "Weapon13"                    },
	{  73189044U, "Apr"                         },
	{  73889260U, "Idle2"                       },
	{  74508159U, "Effect1_Resource"            },
	{  75921650U, "Mage1"                       },
	{  78333924U, "PC"                          },
	{  79362247U, "DefaultNameFemale"           },
	{  81184778U, "PrereqAbility"               },
	{  81635738U, "RewardStrRef"                },
	{  82090968U, "Action6"                     },
	{  86764033U, "Appearance"                  },
	{  92240245U, "DamageStrRef1"               },
	{  93174895U, "Extra1"                      },
	{  93648475U, "Warrior8"                    },
	{  95998482U, "Disabled_Weapon_Sounds"      },
	{  97523184U, "BackgroundTable"             },
	{  97640263U, "CranePuzzle"                 },
	{  98250892U, "DisplayStr3"                 },
	{ 100880733U, "MasterTemplate"              },
	{ 103378247U, "SpellScript"                 },
	{ 104956291U, "Auto_Remove"                 },
	{ 105401216U, "HideLaunch"                  },
	{ 105467695U, "AnimSpeed"                   },
	{ 106843480U, "Maximum_Range"               },
	{ 109660124U, "Duration"                    },
	{ 112392041U, "Bhn100cr_Rat"                },
	{ 112745350U, "Type_Use"                    },
	{ 113233188U, "ConditionLabel"              },
	{ 113271235U, "AttackFlWidth"               },
	{ 116043176U, "ArmorPerLevel"               },
	{ 116982239U, "ShakeType"                   },
	{ 120860322U, "Condition_Mode"              },
	{ 121391276U, "ArmyBufferMax"               },
	{ 121969149U, "StrAdjust"                   },
	{ 121999331U, "WeaponScale"                 },
	{ 122537988U, "ModalabilityBlendTree"       },
	{ 123905048U, "FResistanceMaximum"          },
	{ 124259164U, "LowerStrRef"                 },
	{ 126523709U, "NocturnScout"                },
	{ 127328571U, "ImpactTimeout"               },
	{ 132538785U, "Switch2_Class"               },
	{ 133119168U, "DisableLookAt"               },
	{ 133492742U, "Speed"                       },
	{ 137764198U, "String_Description_Technical"},
	{ 137912134U, "ShowRegardless"              },
	{ 141090729U, "Position"                    },
	{ 142817396U, "ArmyTotalMax"                },
	{ 144909505U, "Eyes_Worksheet"              },
	{ 148813584U, "AttributePoints"             },
	{ 151762438U, "PowerPod"                    },
	{ 152745570U, "OpposingAI"                  },
	{ 158145021U, "EventName"                   },
	{ 158150364U, "Icon4"                       },
	{ 158468564U, "Cir340ar_Fade_Templar_Night" },
	{ 159676877U, "Human"                       },
	{ 163068403U, "DiceTest"                    },
	{ 165682124U, "MediumBoots"                 },
	{ 166026344U, "Den211ar_Arl_Eamon_Estate_1" },
	{ 166594787U, "Rogue8"                      },
	{ 168961103U, "TS_HighTable"                },
	{ 170570597U, "UseCommonGestures"           },
	{ 172161555U, "Scale"                       },
	{ 172521407U, "MaxAnimationYaw"             },
	{ 175395745U, "ShowOnSelf"                  },
	{ 177236338U, "Win32LocaleID"               },
	{ 178686292U, "Omega"                       },
	{ 181568409U, "Rumble_Type"                 },
	{ 188312821U, "Movement_BlendTree"          },
	{ 190481838U, "MaxSpeed"                    },
	{ 194200504U, "FlagSold"                    },
	{ 201332038U, "Frequency"                   },
	{ 202000020U, "ModelPath"                   },
	{ 206109278U, "FFogColor0"                  },
	{ 207936954U, "PrecacheWaveData"            },
	{ 212353258U, "RandChance"                  },
	{ 214037785U, "SpecPoints"                  },
	{ 214256908U, "CrustID"                     },
	{ 217159566U, "Major"                       },
	{ 218438067U, "IdleLeft"                    },
	{ 219074858U, "Medium"                      },
	{ 221805479U, "FilterEquipment"             },
	{ 230943666U, "Game_CbtSpeed"               },
	{ 231543438U, "Shield"                      },
	{ 231863489U, "Height"                      },
	{ 234285194U, "Command"                     },
	{ 234932771U, "100061"                      },
	{ 235767865U, "APBase"                      },
	{ 235772159U, "BasePriority"                },
	{ 239140105U, "Icon16"                      },
	{ 239588056U, "WeightWarrior"               },
	{ 240147400U, "AOE_Index"                   },
	{ 240253765U, "Weight8"                     },
	{ 243621393U, "Hostility"                   },
	{ 255100003U, "AttackLeech"                 },
	{ 258221097U, "TargetType"                  },
	{ 268811058U, "Bullet"                      },
	{ 269175082U, "Type_Category"               },
	{ 271665430U, "Damage1"                     },
	{ 273492170U, "Effect4_Float2"              },
	{ 275566037U, "EngineLink"                  },
	{ 275937400U, "Anim_CbtSpeed"               },
	{ 276050140U, "Orz100ar_Mountain_Pass"      },
	{ 276176797U, "Icon5"                       },
	{ 278409112U, "StartingCharges"             },
	{ 282710090U, "DateFormat"                  },
	{ 283459715U, "MaxBlur"                     },
	{ 284038808U, "ExploreScale"                },
	{ 285860889U, "BlendTreeName"               },
	{ 296588017U, "AttackBRRing"                },
	{ 299361257U, "FCloudRangeMultiplier2"      },
	{ 299891537U, "Den974ar_Rogue_K"            },
	{ 300173649U, "FDeathBlow"                  },
	{ 301153996U, "NameStrRef1"                 },
	{ 305348204U, "Show_Progress"               },
	{ 305735670U, "AttackArmorPiercing"         },
	{ 307130823U, "Neutral"                     },
	{ 309174154U, "MeleeRingDisabledEnd"        },
	{ 309693073U, "AttackFR_Reach"              },
	{ 311884604U, "StrIDDesc"                   },
	{ 312484724U, "Type"                        },
	{ 312895368U, "InjuryMax"                   },
	{ 313239950U, "AttackFRRing"                },
	{ 313340222U, "DonationScript"              },
	{ 316961852U, "IsTutorial"                  },
	{ 317131746U, "Mask"                        },
	{ 320748900U, "FBloomVBlurWidth"            },
	{ 327831836U, "TintColor"                   },
	{ 335144827U, "FCloudDepth"                 },
	{ 337025345U, "SoundImpact"                 },
	{ 338373011U, "BaseAttack"                  },
	{ 345355522U, "Major_Effect"                },
	{ 351631277U, "Cli400ar_City_Gates"         },
	{ 353347367U, "Rank"                        },
	{ 353593310U, "RumblePattern"               },
	{ 355201928U, "StaminaUpkeep"               },
	{ 356206790U, "AttackFL_Angle"              },
	{ 356254124U, "AttributeValue"              },
	{ 357813023U, "FFogColor1"                  },
	{ 358332623U, "Respawn"                     },
	{ 362825144U, "WeaponOffset"                },
	{ 365187742U, "AttackBack"                  },
	{ 365472634U, "AcceptStrRef"                },
	{ 370351214U, "Luck"                        },
	{ 374225939U, "HelmModelVariation"          },
	{ 376642509U, "ArmourPenetration"           },
	{ 378792221U, "KronWarrior"                 },
	{ 379833238U, "ActionSet"                   },
	{ 381681950U, "Helmet"                      },
	{ 384414793U, "Set2"                        },
	{ 386629853U, "BaseArmorPenetration"        },
	{ 387808467U, "Template"                    },
	{ 388364897U, "Exit"                        },
	{ 388489002U, "Resist1"                     },
	{ 389180699U, "DestroyVFX"                  },
	{ 389910404U, "CharybEmerald"               },
	{ 390777821U, "AllowInstant"                },
	{ 391845401U, "OrientationZ"                },
	{ 393084110U, "ShakeDuration"               },
	{ 393353036U, "StrengthModifier"            },
	{ 393916426U, "MaxCount"                    },
	{ 395537274U, "E1011"                       },
	{ 399523528U, "Background"                  },
	{ 401839034U, "Cooldown"                    },
	{ 402355206U, "SoundDuration"               },
	{ 409681986U, "AverageGiftValue"            },
	{ 410130401U, "Button2"                     },
	{ 410244073U, "LastName"                    },
	{ 417591947U, "GlovesModelVariation"        },
	{ 418010895U, "MeleeRingsData"              },
	{ 422542238U, "SetTotal"                    },
	{ 424251708U, "HasIcon"                     },
	{ 426792519U, "Bank2"                       },
	{ 427966477U, "RestartCooldown"             },
	{ 430165727U, "Test3"                       },
	{ 434820324U, "Precision"                   },
	{ 435469453U, "A4"                          },
	{ 435706530U, "Top1"                        },
	{ 435939347U, "RulesRace"                   },
	{ 436197576U, "AreaSound"                   },
	{ 436356403U, "AreaTag"                     },
	{ 441910439U, "PlayerChoice"                },
	{ 442252728U, "Cir200ar_Tower_Level_1"      },
	{ 446301406U, "AirSkiffBoarders"            },
	{ 446301495U, "EffectResisted"              },
	{ 446542916U, "Loops8"                      },
	{ 448620133U, "String_ToolTip"              },
	{ 450830773U, "ToolTipOrder"                },
	{ 454939259U, "LargeShieldLevel"            },
	{ 455333445U, "Height_F"                    },
	{ 457565276U, "FastBodyDecay"               },
	{ 458561398U, "Xbox360"                     },
	{ 459129800U, "ArmorGloves"                 },
	{ 465087263U, "Crust_Effect"                },
	{ 466929058U, "Count"                       },
	{ 467566277U, "Power"                       },
	{ 467896477U, "TotalEntries"                },
	{ 469220555U, "ShakeYFreq"                  },
	{ 469627836U, "EquipEvent"                  },
	{ 469879395U, "StartState"                  },
	{ 471570315U, "Cover"                       },
	{ 471793379U, "DamageBase"                  },
	{ 471864314U, "GestureProbability"          },
	{ 473822677U, "E1020"                       },
	{ 474069449U, "PortraitPalette"             },
	{ 475070168U, "Encounters1"                 },
	{ 475640144U, "Playable"                    },
	{ 476496821U, "Effect3_Int1"                },
	{ 477931057U, "AutoTarget"                  },
	{ 480685819U, "StartingClass"               },
	{ 480819987U, "SoundID"                     },
	{ 480850684U, "NrrgalWarrior"               },
	{ 482345421U, "DisplayStr2"                 },
	{ 482801730U, "ModelVariation"              },
	{ 483423542U, "Walkmesh"                    },
	{ 485030011U, "SoundDurationVolume"         },
	{ 485993930U, "FailureVFX"                  },
	{ 490099466U, "Weapon12"                    },
	{ 501497862U, "WeaponWield"                 },
	{ 501916138U, "RemoveWhenDead"              },
	{ 502085276U, "ArmorBonus"                  },
	{ 503294617U, "Action7"                     },
	{ 505940847U, "LightYAdjust"                },
	{ 507292028U, "AttachScale"                 },
	{ 508717349U, "IdleRight"                   },
	{ 510075347U, "Achievement"                 },
	{ 516713787U, "Rumble_BaseIntensity"        },
	{ 519759590U, "AppearVFX"                   },
	{ 523692622U, "GeneratePerFollower"         },
	{ 524419732U, "BConsiderHostile"            },
	{ 524451592U, "Action12"                    },
	{ 527143153U, "Default_Sound_Set"           },
	{ 527232728U, "AttackBLWidth"               },
	{ 528951897U, "Stat"                        },
	{ 532773143U, "HSVContrast"                 },
	{ 534088598U, "InventorySmallIcon"          },
	{ 537308113U, "FocusDistance"               },
	{ 538851679U, "CanFlee"                     },
	{ 542698763U, "Type_Cost"                   },
	{ 545276281U, "SubCommand"                  },
	{ 547060113U, "IsPlotShape"                 },
	{ 551989405U, "LaserSentinel"               },
	{ 552037852U, "MapTitle"                    },
	{ 553309852U, "MiniMapPin"                  },
	{ 556327241U, "WeaponTable"                 },
	{ 556880460U, "Ntb210ar_Brecilian_Forestne" },
	{ 557052645U, "FResistanceModifier"         },
	{ 558482149U, "AttackBR"                    },
	{ 561196209U, "Offset"                      },
	{ 562937546U, "Table5"                      },
	{ 567233250U, "WieldType"                   },
	{ 568406850U, "ArmourValue"                 },
	{ 569414898U, "Reward"                      },
	{ 569761468U, "ShowDonationButton"          },
	{ 570895252U, "Damage3"                     },
	{ 571803576U, "WeaponBowStrRef"             },
	{ 572733205U, "FocalRadius"                 },
	{ 573963242U, "UVAnimSpeed"                 },
	{ 574651679U, "Icon7"                       },
	{ 578153022U, "WeaponBowLevel"              },
	{ 584200531U, "MovementTable"               },
	{ 591372477U, "RaceAbility"                 },
	{ 592308015U, "AttackBlast"                 },
	{ 600136782U, "NameStrRef3"                 },
	{ 613447705U, "SynchAnim"                   },
	{ 613840734U, "LocalizationTag"             },
	{ 616634588U, "AvoidMeleeenEmies"           },
	{ 619652321U, "Resist29"                    },
	{ 619856184U, "ScrollSpeedY"                },
	{ 621919656U, "Resist3"                     },
	{ 627930267U, "OrientationX"                },
	{ 627945162U, "Icon15"                      },
	{ 629825572U, "ArmorProps"                  },
	{ 633431445U, "NextID"                      },
	{ 638368012U, "Ntb200ar_Brecilian_Forestnw" },
	{ 641893199U, "DmgScalingThresh"            },
	{ 642686768U, "Den500ar_Arl_Exterior"       },
	{ 643117804U, "ReqProp2Mod"                 },
	{ 645531618U, "Dmg_Flags"                   },
	{ 648283245U, "Material9"                   },
	{ 649384513U, "ToolTip"                     },
	{ 649989203U, "GoReLevel"                   },
	{ 650030685U, "Boots"                       },
	{ 659615381U, "BaseManaStamina"             },
	{ 660264680U, "ImpactVFX"                   },
	{ 662478783U, "To"                          },
	{ 662593128U, "CostUpkeep"                  },
	{ 664900028U, "PrecollisionUVAnimName"      },
	{ 669506298U, "AttributeID"                 },
	{ 669915811U, "Animation_Override"          },
	{ 672560738U, "BootsModelTint"              },
	{ 673229691U, "IsPlot"                      },
	{ 673256352U, "Georg_Test"                  },
	{ 673436973U, "Variation"                   },
	{ 676703212U, "React"                       },
	{ 678408551U, "PrecollisionTextureAnimName" },
	{ 678931606U, "EffectResistance"            },
	{ 680993279U, "Animation2"                  },
	{ 681261933U, "Left5"                       },
	{ 683738427U, "Closed_Locked"               },
	{ 684336042U, "AttackBLRing"                },
	{ 684797042U, "Modifier"                    },
	{ 685785939U, "Positional_Type"             },
	{ 688000362U, "GunPawn"                     },
	{ 689116208U, "BottomScreenType"            },
	{ 698258082U, "SuperSonicBase"              },
	{ 698879837U, "TreeID"                      },
	{ 699195393U, "EnemyFlankBonus"             },
	{ 702494183U, "ChanceToFail"                },
	{ 703914747U, "ProbabilityModifier"         },
	{ 706409779U, "FollowerAI"                  },
	{ 709193354U, "Title"                       },
	{ 713808508U, "PressurePlate"               },
	{ 714224051U, "Cli700ar_Markets"            },
	{ 728489119U, "ConjureVFX"                  },
	{ 734191119U, "A6"                          },
	{ 734795989U, "AttackFLRing"                },
	{ 734935072U, "Top3"                        },
	{ 735260114U, "Prio"                        },
	{ 739006026U, "TS_ItemNum"                  },
	{ 740392269U, "CancelOnPlotEvent"           },
	{ 740609531U, "FMoonAlpha"                  },
	{ 741315769U, "Table"                       },
	{ 742774051U, "Simple"                      },
	{ 743366305U, "Multiplier"                  },
	{ 745217151U, "DefaultWeapon"               },
	{ 748508185U, "LightGloves"                 },
	{ 748562610U, "TravelWaterSlide"            },
	{ 752030797U, "Stone_Golem"                 },
	{ 755638087U, "DamageInfo"                  },
	{ 757587883U, "SoundSet"                    },
	{ 758267220U, "WeaponSet2LeftStack"         },
	{ 760745214U, "Orz530ar_Ortan_Thaig"        },
	{ 761761592U, "StrID_Effect"                },
	{ 762325979U, "Major_Facing"                },
	{ 762584458U, "Action10"                    },
	{ 777136822U, "DamageStrRef2"               },
	{ 780131386U, "CombatVFXTester"             },
	{ 780600894U, "Fur"                         },
	{ 781319190U, "ParameterWorksheet"          },
	{ 783767132U, "AOE_Type"                    },
	{ 784959010U, "FDistanceMultiplier"         },
	{ 788541832U, "Weapon10"                    },
	{ 789102311U, "FastAngle"                   },
	{ 792836686U, "Resist18"                    },
	{ 793388591U, "Idle1"                       },
	{ 794244628U, "FStealingModifier"           },
	{ 796695736U, "ArmySpawnWP"                 },
	{ 797070517U, "FSunIntensity"               },
	{ 798515487U, "MissileDeflection"           },
	{ 798966067U, "Sound_X"                     },
	{ 799357003U, "Ability9"                    },
	{ 799746865U, "Mage2"                       },
	{ 800606204U, "FrayleighMultiplier"         },
	{ 801696160U, "BaseHealth"                  },
	{ 801753115U, "Action5"                     },
	{ 801777985U, "ThousandSeparator"           },
	{ 804927328U, "Effect4_Resource"            },
	{ 805082355U, "ExploreMicroPortrait"        },
	{ 806781662U, "Amulet"                      },
	{ 809634884U, "AllowedList"                 },
	{ 815716057U, "Gloves"                      },
	{ 817532226U, "IsSpecial"                   },
	{ 818480006U, "VaultID"                     },
	{ 819084727U, "2da"                         },
	{ 823930364U, "BlendOut"                    },
	{ 829691757U, "AddInUID"                    },
	{ 830474796U, "Left4"                       },
	{ 831262910U, "Animation3"                  },
	{ 839215464U, "HeavyArmorStrRef"            },
	{ 842287077U, "FadeoutTime"                 },
	{ 843608276U, "Unlocked"                    },
	{ 844711912U, "Skeleton"                    },
	{ 844876799U, "Captain"                     },
	{ 844957060U, "Bank1"                       },
	{ 846426395U, "Editor"                      },
	{ 846460685U, "NameStrRef"                  },
	{ 852822369U, "Top2"                        },
	{ 853143374U, "A7"                          },
	{ 857689869U, "FireWait"                    },
	{ 859215507U, "MarkupTag"                   },
	{ 859320354U, "Switch1_Level"               },
	{ 861881378U, "Button1"                     },
	{ 864926133U, "Blend"                       },
	{ 875636771U, "HeraldryFile"                },
	{ 879701195U, "Action11"                    },
	{ 882956673U, "Flesh"                       },
	{ 885148109U, "BottomScreenID"              },
	{ 886840987U, "Melee"                       },
	{ 887292623U, "BUsesMana"                   },
	{ 887465112U, "X4"                          },
	{ 887959913U, "Bolt"                        },
	{ 888131383U, "TalentPoints"                },
	{ 889059708U, "Use_Condition_Mode"          },
	{ 889372311U, "SentryDrone"                 },
	{ 890123734U, "ArmyDeathCountVar"           },
	{ 894889380U, "JournalSort"                 },
	{ 897077506U, "CombatEffect_Precisions"     },
	{ 897986624U, "Pil"                         },
	{ 898872317U, "DescStrID"                   },
	{ 900158400U, "Layout"                      },
	{ 901443381U, "WeaponSet2Right"             },
	{ 902126647U, "Scaled_Field"                },
	{ 903554986U, "SoundMatType"                },
	{ 904673712U, "MaxWeight"                   },
	{ 904782749U, "Team4"                       },
	{ 904873550U, "Rumble_Iterations"           },
	{ 905976579U, "ScalingVector"               },
	{ 906178353U, "UseChance"                   },
	{ 907739337U, "Weapon11"                    },
	{ 908238860U, "SlotsLocked"                 },
	{ 911077012U, "RequiresStrRef"              },
	{ 911167618U, "ModelRace"                   },
	{ 911313448U, "SimpleEffect"                },
	{ 911903503U, "Resist19"                    },
	{ 913113991U, "Misc"                        },
	{ 913851120U, "AmbientOverride"             },
	{ 914179406U, "ClosedToOpened"              },
	{ 914576581U, "PinType"                     },
	{ 917510768U, "Mage3"                       },
	{ 918423818U, "Ability8"                    },
	{ 918461186U, "AttackBonus"                 },
	{ 919547228U, "Gold"                        },
	{ 919796058U, "Action4"                     },
	{ 920169643U, "RaptorHawk"                  },
	{ 922863136U, "Ps3"                         },
	{ 923900119U, "EventID"                     },
	{ 927029009U, "TargetResult"                },
	{ 927538167U, "DamageStrRef3"               },
	{ 927699062U, "Effect3_Int2"                },
	{ 928266959U, "StepUp"                      },
	{ 930095302U, "Den964ar_End_Assassin"       },
	{ 930483761U, "BaseClass"                   },
	{ 934087747U, "Strategy"                    },
	{ 938299918U, "DisplayStr1"                 },
	{ 942104125U, "Effect_Object0"              },
	{ 942568561U, "MediumHelm"                  },
	{ 944686625U, "Extreme_Facing"              },
	{ 949391243U, "Table4"                      },
	{ 950825350U, "Staff"                       },
	{ 951820807U, "PreferMelee"                 },
	{ 958419230U, "AllowPermanent"              },
	{ 959218980U, "WeaponMeleeLevel"            },
	{ 961531844U, "ShakeDelay"                  },
	{ 966068516U, "Expression"                  },
	{ 966213922U, "StackSize"                   },
	{ 970998895U, "TargetLevel"                 },
	{ 978058292U, "CameraXAdjust"               },
	{ 979339582U, "PassiveTalent2"              },
	{ 980112970U, "DefenseBonus"                },
	{ 987655439U, "NameStrRef2"                 },
	{ 989507626U, "FCloudRangeMultiplier1"      },
	{ 991534468U, "CombatOnly"                  },
	{ 991697621U, "Damage2"                     },
	{ 995839070U, "Icon6"                       },
	{ 995946045U, "Lot105ar_Lothering_Lc"       },
	{ 996178697U, "Effect4_Float1"              },
	{ 998350641U, "Humanoid"                    },
	{1000372874U, "Effect"                      },
	{1007324393U, "Resist2"                     },
	{1014384090U, "OrientationY"                },
	{1014407051U, "Icon14"                      },
	{1021270669U, "Cream"                       },
	{1029174277U, "WalkWaterSlide"              },
	{1030739818U, "ConvNamePlural"              },
	{1033821394U, "PointsPer"                   },
	{1034033145U, "ShakeYawFreq"                },
	{1034153634U, "Impact_Table"                },
	{1035312742U, "Weapon20"                    },
	{1036314506U, "Set1"                        },
	{1037872334U, "CastAnim_VFX"                },
	{1038708857U, "ScrollSpeedX"                },
	{1039015064U, "Urn200ar_Ruined_Temple"      },
	{1039406496U, "Resist28"                    },
	{1048226227U, "VoxaiElite"                  },
	{1048484060U, "FFogColor2"                  },
	{1049388871U, "Arl100ar_Redcliffe_Village"  },
	{1055179836U, "CameraZoom"                  },
	{1055755202U, "VFX_Deselected"              },
	{1055959567U, "AttackFR_Width"              },
	{1058702582U, "BaseDamage"                  },
	{1069495596U, "Material8"                   },
	{1069628897U, "Value"                       },
	{1072399968U, "ModelSizes"                  },
	{1075288592U, "FCloudColor2"                },
	{1075377373U, "PickLockFailure"             },
	{1077191953U, "SoundActivate1"              },
	{1077647898U, "DefaultValue"                },
	{1080246718U, "StrRef"                      },
	{1080915046U, "FTurbidity"                  },
	{1081407707U, "Attribute0"                  },
	{1085105840U, "Blast"                       },
	{1089137166U, "Secret"                      },
	{1090612174U, "Icon11"                      },
	{1094297119U, "Effect1_Float1"              },
	{1094503249U, "FCloudSharpness"             },
	{1095395740U, "Percentage"                  },
	{1095827677U, "Threat_Impact"               },
	{1096953813U, "Misc2"                       },
	{1098741932U, "Resist7"                     },
	{1098911410U, "LocationVFX"                 },
	{1109671365U, "WeaponSet1Right"             },
	{1110086366U, "Leucosia"                    },
	{1110144829U, "FollowerTacticsTable"        },
	{1111151200U, "BCanBleed"                   },
	{1112043480U, "TalentTable"                 },
	{1117561743U, "Disabled_Armor_Sounds"       },
	{1117873628U, "Package"                     },
	{1123158435U, "ReqPowerScale"               },
	{1124196876U, "ShieldStrRef"                },
	{1125401843U, "ValidForClass2"              },
	{1125547031U, "StartArea"                   },
	{1126193056U, "Reset"                       },
	{1131797190U, "OverlayTree"                 },
	{1135953671U, "Variation_Worksheet"         },
	{1137236117U, "Animation"                   },
	{1137932793U, "Effect4_Int2"                },
	{1138508078U, "Effect2_Float2"              },
	{1146023318U, "NumDice"                     },
	{1151277051U, "Armadillo"                   },
	{1155368596U, "RadialGroup"                 },
	{1156232721U, "DamageRange"                 },
	{1158546615U, "AttackMod"                   },
	{1159430972U, "SuccessGrade"                },
	{1161224508U, "WillAdjust"                  },
	{1162021597U, "es-es"                       },
	{1164443854U, "Interrupt"                   },
	{1167731541U, "SpecRoot"                    },
	{1170047532U, "PrimaryRange"                },
	{1171118642U, "CollisionSound"              },
	{1172405198U, "Table1"                      },
	{1175767833U, "Template3"                   },
	{1177335835U, "Icon3"                       },
	{1183662587U, "DescriptionStrRef2"          },
	{1184785604U, "PlayHumanoidAnimations"      },
	{1189239913U, "HeaderStrID"                 },
	{1189372269U, "LargeIcon2"                  },
	{1192897652U, "Lieutenant"                  },
	{1193030555U, "Hostile"                     },
	{1193423182U, "Amplitude_Y"                 },
	{1193554600U, "SoundMapBank"                },
	{1197427016U, "Locked"                      },
	{1197724016U, "Attribute"                   },
	{1197810504U, "TyxisEmerald"                },
	{1198034238U, "HeavyBoots"                  },
	{1201770834U, "NotificationFlag"            },
	{1201970273U, "ArenaModelName"              },
	{1203668663U, "AmbushValue"                 },
	{1206423396U, "AttackBL_Width"              },
	{1210638799U, "MassiveArmorStrRef"          },
	{1212918477U, "Health"                      },
	{1215524259U, "PowerRequired"               },
	{1216715632U, "WieldR"                      },
	{1217293259U, "TrapDmgScale"                },
	{1217639316U, "MediumArmorStrRef"           },
	{1218058200U, "Team1"                       },
	{1219675370U, "ProgFX"                      },
	{1220338444U, "ChanceParent"                },
	{1222705433U, "Targetable"                  },
	{1223024337U, "FSunColor2"                  },
	{1224063131U, "Extreme_VFX"                 },
	{1226759481U, "DexterityBase"               },
	{1230570688U, "Weapon8"                     },
	{1231519594U, "Rogue"                       },
	{1234325725U, "X1"                          },
	{1236193562U, "FFogZenith"                  },
	{1242790390U, "HealingModifier"             },
	{1245919004U, "PropertyFlags"               },
	{1246567377U, "Prefix"                      },
	{1249159436U, "DepletableProgression"       },
	{1250117775U, "PlotID"                      },
	{1259444388U, "TertiaryMoveLevel"           },
	{1261004623U, "AL_Attributes"               },
	{1265431692U, "Weapon14"                    },
	{1269107999U, "Action1"                     },
	{1269915261U, "VFX_Idle"                    },
	{1271333407U, "AlignToGround"               },
	{1271393845U, "Mage6"                       },
	{1273222663U, "DRModPlayer"                 },
	{1282255802U, "String_Name_Lowercase"       },
	{1282637801U, "Cli600ar_Elven_Alienage"     },
	{1282849131U, "PoseSync1"                   },
	{1286804638U, "Cir320ar_Fade_Darkspawn_Inv" },
	{1288997802U, "TitleGUI"                    },
	{1291320937U, "Left1"                       },
	{1292626707U, "TrackName"                   },
	{1294530358U, "BufferTime"                  },
	{1296017560U, "BaseItem2"                   },
	{1298081981U, "IsAudio"                     },
	{1298588368U, "Target_Type"                 },
	{1298864158U, "ShakeZ"                      },
	{1300308152U, "DwarfMale"                   },
	{1301029803U, "StartingAbility1"            },
	{1301412045U, "Resist30"                    },
	{1302677678U, "EquipSlot"                   },
	{1303495625U, "TargetWPTableID"             },
	{1305140635U, "BlendIn"                     },
	{1306047343U, "SpeedX"                      },
	{1306988130U, "Shape"                       },
	{1308509061U, "SkillLabel"                  },
	{1311289447U, "Button4"                     },
	{1312685500U, "Silver"                      },
	{1313184795U, "AutoID"                      },
	{1315802531U, "BaseDamageReduction"         },
	{1319429861U, "MagAdjust"                   },
	{1323668607U, "FPointVal"                   },
	{1326512164U, "ReqProp1"                    },
	{1327210026U, "EggBot"                      },
	{1327755936U, "SoundImpactVolume"           },
	{1328214465U, "Bank4"                       },
	{1328913139U, "WeaponMeleeStrRef"           },
	{1330059356U, "PortraitFrame"               },
	{1331238393U, "Switch2_Level"               },
	{1332708879U, "CursorID"                    },
	{1334695051U, "Rumble_Duration"             },
	{1335332433U, "Area"                        },
	{1335860944U, "EquipMask"                   },
	{1336810251U, "A2"                          },
	{1340467571U, "IPType"                      },
	{1342797548U, "ModelTree"                   },
	{1343286723U, "BaseType"                    },
	{1343360548U, "Property"                    },
	{1345178165U, "it-it"                       },
	{1346516353U, "Weapon9"                     },
	{1346782725U, "Cir350ar_Fade_Weisshaupt"    },
	{1350079155U, "Ranged"                      },
	{1350909547U, "AddSound"                    },
	{1352269548U, "Idle"                        },
	{1352861765U, "ShowIntactics"               },
	{1353253814U, "DescStrRef"                  },
	{1359017142U, "BasePrereqAbility"           },
	{1360189385U, "OrientWithObject"            },
	{1362523557U, "Hair_Worksheet"              },
	{1363985241U, "Size1"                       },
	{1367397819U, "Suffix"                      },
	{1368607545U, "ClassAbility"                },
	{1373981521U, "Gore_Max"                    },
	{1377800050U, "ShieldLevel"                 },
	{1378565738U, "XOffset"                     },
	{1379018764U, "BaseDR"                      },
	{1381873520U, "ArenaTextureName"            },
	{1382196325U, "ScaleModifier"               },
	{1383581133U, "Weapon15"                    },
	{1384378074U, "CodexResRef"                 },
	{1385328532U, "Ability12"                   },
	{1386509576U, "WeightBlur"                  },
	{1387280917U, "DexAdjust"                   },
	{1388199006U, "Action0"                     },
	{1390206836U, "Mage7"                       },
	{1391079120U, "BDisableGore"                },
	{1391293631U, "CombatCache"                 },
	{1391977607U, "APBonus"                     },
	{1393811363U, "ChaseEnemy"                  },
	{1394303748U, "Screen"                      },
	{1394599588U, "Mage"                        },
	{1397406625U, "ExtendedModule"              },
	{1400660754U, "E1027"                       },
	{1404491153U, "ShowButton"                  },
	{1407676272U, "EventScr"                    },
	{1410704087U, "AOEVFX"                      },
	{1412822900U, "Instant"                     },
	{1413268743U, "WieldModelOverlay"           },
	{1413515507U, "CollisionAnimName"           },
	{1415095166U, "NCombatTraining"             },
	{1418381708U, "Resist31"                    },
	{1422098990U, "SpeedY"                      },
	{1423448814U, "TattooFile"                  },
	{1424101090U, "UVFrame"                     },
	{1430533312U, "Base"                        },
	{1431448252U, "VarName"                     },
	{1433574882U, "FScale"                      },
	{1439686340U, "Default"                     },
	{1440664305U, "Creature_Giant"              },
	{1441360454U, "Ring"                        },
	{1441582888U, "Left0"                       },
	{1446101120U, "Bank5"                       },
	{1448130598U, "GFFStore_Invalid_List"       },
	{1449286761U, "AutoCancel"                  },
	{1454713418U, "A3"                          },
	{1455107783U, "ConditionBase"               },
	{1455219673U, "Material"                    },
	{1455727433U, "Shield-Steel"                },
	{1457063892U, "Description"                 },
	{1458606196U, "ShakePitch"                  },
	{1460772297U, "ProcessWithEvent"            },
	{1462005596U, "Gender"                      },
	{1462298181U, "Shale"                       },
	{1463973815U, "String_Name"                 },
	{1464961929U, "ConstitutionPerLevel"        },
	{1465694033U, "DefaultResource"             },
	{1471505569U, "TexAnimSpeed"                },
	{1471946011U, "EnterSync"                   },
	{1477484083U, "CessationNodes"              },
	{1477978351U, "Hit"                         },
	{1478122127U, "Icon10"                      },
	{1479630702U, "Ligaia"                      },
	{1483098605U, "Resist6"                     },
	{1484332692U, "Misc3"                       },
	{1486473115U, "Damage"                      },
	{1487442009U, "Condition2"                  },
	{1489089433U, "Wood"                        },
	{1490148931U, "UseItem"                     },
	{1490567978U, "Mud"                         },
	{1496290384U, "SoundActivate0"              },
	{1497575150U, "Scorpion"                    },
	{1500498330U, "Attribute1"                  },
	{1501843707U, "UniqueID"                    },
	{1505481080U, "Dmg_Type"                    },
	{1505761334U, "Pre210ar_Flemeths_Hut_Ext"   },
	{1506554349U, "SubType"                     },
	{1510962610U, "ValidForClass3"              },
	{1511494450U, "TargetWPTable"               },
	{1512200535U, "ClientLoad"                  },
	{1515226505U, "SmallIcon"                   },
	{1517217568U, "CrustVFX"                    },
	{1517365817U, "VictoryTip"                  },
	{1518196115U, "RumbleIntensity"             },
	{1520125712U, "HeaderTintIndex"             },
	{1521459413U, "SpawnChance"                 },
	{1521798933U, "FireSpot"                    },
	{1523611661U, "Success0a"                   },
	{1524768162U, "BridgePuzzle"                },
	{1524871084U, "Root"                        },
	{1527730666U, "OffsetZ"                     },
	{1528504341U, "Pose1"                       },
	{1535422151U, "CameraPitch"                 },
	{1541076270U, "ToBlendTree"                 },
	{1543170391U, "DamageType"                  },
	{1545607643U, "LevelsPerAbility"            },
	{1545704015U, "MiniMapString"               },
	{1547028933U, "PauseAnimate"                },
	{1548828037U, "RngdPenalty"                 },
	{1551418531U, "Resource"                    },
	{1558442343U, "InitialSkillLabel"           },
	{1559271881U, "Weapon_Trail"                },
	{1560553710U, "FadeTime"                    },
	{1560879490U, "ModelType"                   },
	{1564346247U, "AOE_Idx"                     },
	{1565249575U, "MemberName"                  },
	{1567942634U, "Prerequisite"                },
	{1569459569U, "Above10_Secondary"           },
	{1570279467U, "DefaultMaterial"             },
	{1573905635U, "CanTriggerForPrc"            },
	{1577555688U, "PlaceablesTable"             },
	{1578724890U, "Crust"                       },
	{1579589584U, "AOE_Param1"                  },
	{1580793871U, "Amplitude_X"                 },
	{1581103990U, "Sand"                        },
	{1582035077U, "EffectStrRef1"               },
	{1585538048U, "Effect1_Duration"            },
	{1586539016U, "Hatchable"                   },
	{1587291808U, "CreatureTag"                 },
	{1588449107U, "PrimaryMove"                 },
	{1591749853U, "ChanceEquipment"             },
	{1591793879U, "Desc"                        },
	{1591881703U, "Limit"                       },
	{1594882648U, "Template2"                   },
	{1597475162U, "Icon2"                       },
	{1601456351U, "ForcedArea"                  },
	{1601544304U, "TurnRate"                    },
	{1603694778U, "DescriptionStrRef3"          },
	{1605055965U, "DamagePerLevel"              },
	{1610081351U, "LoadProjectile"              },
	{1610567724U, "LargeIcon3"                  },
	{1614120359U, "Follower_TrigOutComb"        },
	{1614673823U, "Advancement"                 },
	{1614690773U, "WeaponProps"                 },
	{1614915407U, "Weapon17"                    },
	{1616724840U, "Slots"                       },
	{1617620079U, "BehaviorType"                },
	{1619549916U, "Action2"                     },
	{1619580266U, "Item8"                       },
	{1620316196U, "PlotFile"                    },
	{1620914698U, "AutoSelect"                  },
	{1621363990U, "Ability10"                   },
	{1621741312U, "CritChanceModifier"          },
	{1622494149U, "IdleAnimation"               },
	{1622802925U, "RespectHoldPosition"         },
	{1625996790U, "Mage5"                       },
	{1630761480U, "EffectDuration"              },
	{1633989983U, "Sword"                       },
	{1636926891U, "TrigOutComb"                 },
	{1641340607U, "Max"                         },
	{1647435361U, "SetLabel"                    },
	{1648963252U, "LightZAdjust"                },
	{1650578647U, "HumanFemale"                 },
	{1650905122U, "Points"                      },
	{1655363253U, "MaxItemProps"                },
	{1656685342U, "X2"                          },
	{1659741960U, "WeaponSet2Left"              },
	{1665166966U, "Permanent"                   },
	{1666135536U, "RootBone"                    },
	{1667875291U, "SquadsTable"                 },
	{1668694829U, "TLKRef"                      },
	{1668823674U, "AccessoryTable"              },
	{1668981211U, "Size3"                       },
	{1672213075U, "Extreme"                     },
	{1672954907U, "Team2"                       },
	{1673258945U, "AnimationSpeeds"             },
	{1674087698U, "FSunColor1"                  },
	{1676630809U, "Triggered"                   },
	{1679887387U, "AreaList"                    },
	{1680136510U, "DefaultPower"                },
	{1681763303U, "ReqProp2"                    },
	{1683019108U, "Info"                        },
	{1684820571U, "AttackOnCombatStart"         },
	{1686325448U, "A1"                          },
	{1687138023U, "Top4"                        },
	{1687992504U, "ArmorTable"                  },
	{1689487304U, "ToolTipColor"                },
	{1693816754U, "MaxProperties"               },
	{1694567723U, "Immunity"                    },
	{1695438083U, "Target_Range_Correction"     },
	{1700701943U, "HelmModelTint"               },
	{1703791551U, "DateTimeFormat"              },
	{1705630292U, "FContrast"                   },
	{1712520027U, "BaseItem1"                   },
	{1714304226U, "Shadow"                      },
	{1715877236U, "Unknown"                     },
	{1715882973U, "ShakeY"                      },
	{1718880933U, "Rumble_Intensity"            },
	{1719161700U, "Overlay_BlendTree"           },
	{1721055339U, "AttackBL_Ring"               },
	{1721189173U, "SoundCessationPitch"         },
	{1721856104U, "StartingAbility2"            },
	{1722818671U, "Override"                    },
	{1725364355U, "Closed"                      },
	{1727412578U, "ShowMaterial"                },
	{1727777078U, "ID"                          },
	{1732394383U, "ReqProp2Val"                 },
	{1733181591U, "RangeType"                   },
	{1733592563U, "Orz560ar_Shale_Taig"         },
	{1734077096U, "PoseSync2"                   },
	{1734208093U, "OptimumRange"                },
	{1739877456U, "UVAnimation"                 },
	{1741638726U, "PlotRef"                     },
	{1741740640U, "Cir300ar_Fade"               },
	{1742384554U, "Left2"                       },
	{1743119172U, "FleeTable"                   },
	{1743148734U, "Combo9"                      },
	{1745226635U, "Blood_Type"                  },
	{1748571952U, "ValidForClass1"              },
	{1750350553U, "PassiveTalent"               },
	{1752896097U, "AudioEvent"                  },
	{1753475552U, "StrengthModifierMultiplier"  },
	{1754512176U, "Random"                      },
	{1755549632U, "ShakeYaw"                    },
	{1756483508U, "AssaultDrone"                },
	{1760629485U, "Effect2_Float1"              },
	{1761487418U, "Effect4_Int1"                },
	{1764573847U, "Pose3"                       },
	{1765074872U, "EdgePattern"                 },
	{1765356392U, "OffsetX"                     },
	{1768205143U, "Millipede"                   },
	{1768959983U, "fr-fr"                       },
	{1769598888U, "Filename"                    },
	{1779758556U, "Effect1_Float2"              },
	{1780221216U, "Water"                       },
	{1781282829U, "Icon12"                      },
	{1783135871U, "Female"                      },
	{1783594006U, "Misc1"                       },
	{1783638895U, "Resist4"                     },
	{1784149743U, "ScreenShotID"                },
	{1784688572U, "DeathBlow_Table"             },
	{1786994493U, "ShakePitchFreq"              },
	{1787983853U, "CombatRoundLimit"            },
	{1791773712U, "StartingXP"                  },
	{1791945787U, "EffectChance"                },
	{1793367871U, "E1014"                       },
	{1794107911U, "PackageType"                 },
	{1794890228U, "Selectable"                  },
	{1799014867U, "FCloudColor1"                },
	{1799866115U, "BHasDisappearAnim"           },
	{1801385690U, "Dog"                         },
	{1801882422U, "MeleeRingDisabledStart"      },
	{1802350586U, "AttackBL_Reach"              },
	{1813199525U, "Den360ar_Slaver_Compound"    },
	{1814261165U, "AttackBR_Angle"              },
	{1818473527U, "DMPlayers"                   },
	{1819443952U, "SwatBot"                     },
	{1820168711U, "EffectStrRef3"               },
	{1821156423U, "FAtmAlpha"                   },
	{1822814658U, "ActionTotal"                 },
	{1825557980U, "Orphanage"                   },
	{1825834492U, "Leech"                       },
	{1839206968U, "DescriptionStrRef1"          },
	{1841291328U, "RightHand"                   },
	{1841585889U, "GameString"                  },
	{1841902254U, "LargeIcon1"                  },
	{1843146524U, "Critter"                     },
	{1843227531U, "BUseProbablityModifier"      },
	{1843556014U, "ArmorPiercing"               },
	{1845304238U, "OneShotKills"                },
	{1846379100U, "ArmorBoots"                  },
	{1847400426U, "Resistance"                  },
	{1850402857U, "SoundDurationPitch"          },
	{1850713868U, "Optimal_Distance"            },
	{1852794291U, "SuppressPropertyVFX"         },
	{1853783890U, "Den100ar_Brothel"            },
	{1857206080U, "Dmg_Function"                },
	{1858873357U, "Table2"                      },
	{1867254814U, "Claws"                       },
	{1870263889U, "Container"                   },
	{1870458014U, "Special"                     },
	{1870467844U, "Syrax"                       },
	{1879631513U, "TS_HighChance"               },
	{1881276969U, "OffsetY"                     },
	{1881638363U, "IxChair"                     },
	{1882616790U, "Pose2"                       },
	{1884768031U, "Cli800ar_Palace_District"    },
	{1885680587U, "TertiaryMove"                },
	{1886236156U, "ConditionParameter"          },
	{1887696852U, "WearR"                       },
	{1895064570U, "BlendTree"                   },
	{1900015509U, "LoadGlobal"                  },
	{1900788385U, "TitleIcon"                   },
	{1905885354U, "Combo_Table"                 },
	{1905901863U, "AvoidNearbyEnemies"          },
	{1908993823U, "UnequipEvent"                },
	{1914151739U, "AlphaFade"                   },
	{1914828946U, "FCloudColor0"                },
	{1923731204U, "Head_Worksheet"              },
	{1932106476U, "Heraldry"                    },
	{1932802301U, "HasQuest"                    },
	{1932986700U, "Icon13"                      },
	{1934301742U, "Resist5"                     },
	{1938146202U, "Condition1"                  },
	{1940417278U, "Attack"                      },
	{1943154587U, "NocturnCannon"               },
	{1943461991U, "Boss"                        },
	{1944091781U, "StaticCollisions"            },
	{1947916953U, "Icon1"                       },
	{1948428699U, "Template1"                   },
	{1948872527U, "Beard_Worksheet"             },
	{1950600339U, "MinLevel"                    },
	{1951267696U, "BumpLevel"                   },
	{1951297390U, "EffectImpact"                },
	{1956284458U, "DisplayInGUI"                },
	{1961421461U, "CostMultiplier"              },
	{1963701279U, "Effect4_Duration"            },
	{1963707411U, "AOE_Param2"                  },
	{1966513034U, "Prop2Value"                  },
	{1966645520U, "Additive_Animation"          },
	{1968014238U, "LightArmorStrRef"            },
	{1969660742U, "EffectStrRef2"               },
	{1969837178U, "AddBehaviorSelf"             },
	{1969954443U, "LightBoots"                  },
	{1970818270U, "ConditionScript"             },
	{1975797129U, "IgnoreAppMaxLevel"           },
	{1976707862U, "Controller"                  },
	{1977488544U, "ArmyScript"                  },
	{1978962668U, "CombatMoveTree"              },
	{1980446075U, "Arrow"                       },
	{1980909877U, "PlayerClass"                 },
	{1983393620U, "Snow"                        },
	{1991792155U, "Sound"                       },
	{1993675371U, "LifeTime"                    },
	{1994170980U, "InventoryLargeIconPalette"   },
	{1994561005U, "XPos"                        },
	{1994645110U, "SpecialGraphic"              },
	{1994767307U, "Priority"                    },
	{1994956158U, "Event"                       },
	{2001231285U, "ValidForTarget"              },
	{2002628611U, "Potions"                     },
	{2005736378U, "CollisionTextureName"        },
	{2006584394U, "ArmorPenetration"            },
	{2008171580U, "Anim_Deselected"             },
	{2010117091U, "Fatigue"                     },
	{2010585420U, "Table3"                      },
	{2015074471U, "GroupID"                     },
	{2015275233U, "ItemCreated"                 },
	{2017962320U, "ShieldRating"                },
	{2020978272U, "FEffectDurNormal"            },
	{2021292997U, "Humanoid_Small"              },
	{2023673708U, "UnlockID"                    },
	{2026787092U, "DoWaterSlide"                },
	{2029400109U, "Woodsman"                    },
	{2032598522U, "ProfilePortraitPrefix"       },
	{2035979790U, "Weapon16"                    },
	{2039705501U, "Action3"                     },
	{2042575959U, "Ability11"                   },
	{2043930033U, "Partsbased"                  },
	{2045137524U, "Race"                        },
	{2045873335U, "Mage4"                       },
	{2048951419U, "TargetTypeLabel"             },
	{2049285982U, "ArmyTable"                   },
	{2050591481U, "AnimName"                    },
	{2050879840U, "NonCombatEffect"             },
	{2053214362U, "Size2"                       },
	{2053794413U, "AttackFL"                    },
	{2055637610U, "AttackerAnimation"           },
	{2056890751U, "Blunt"                       },
	{2058122586U, "Team3"                       },
	{2060260334U, "MeleeRingUpdateAlways"       },
	{2060696659U, "FSunColor0"                  },
	{2061104355U, "SpawnSameWhenDead"           },
	{2062678162U, "Humanoid_Large"              },
	{2063119398U, "RuneSlots"                   },
	{2063788826U, "ModelGenderOverride"         },
	{2066025186U, "Rating"                      },
	{2069283410U, "OnTimeout"                   },
	{2071093817U, "Target_Auto"                 },
	{2071107585U, "FaceCamera"                  },
	{2072462458U, "Switch1_Class"               },
	{2074358367U, "X3"                          },
	{2075838542U, "EncounterChance"             },
	{2078237967U, "Set"                         },
	{2080623492U, "Die"                         },
	{2084374599U, "ClassStrRef"                 },
	{2095137753U, "NumActions"                  },
	{2095263023U, "TriggerFlagSet"              },
	{2097055185U, "PrereqAttValue"              },
	{2100896646U, "Stealth"                     },
	{2102202263U, "NReactiveTier"               },
	{2105997658U, "Sound_Unarmed"               },
	{2106892198U, "Top5"                        },
	{2113789027U, "DisabledStrRef"              },
	{2114258337U, "InvSoundType"                },
	{2114387257U, "MaxAllowedYaw"               },
	{2117442992U, "CanAct"                      },
	{2118179817U, "PoseSync3"                   },
	{2119848778U, "NocturnSoldier"              },
	{2120312040U, "BaseDuration"                },
	{2122132398U, "CessationModel"              },
	{2123536581U, "FFogIntensity"               },
	{2125466280U, "Den950ar_Alley_Justice_1"    },
	{2126617835U, "Left3"                       },
	{2127271995U, "ChestModelTint"              },
	{2128017096U, "GlovesModelTint"             },
	{2130536447U, "Combo8"                      },
	{2133029355U, "BlendTrees"                  },
	{2134488944U, "AttackFL_Ring"               },
	{2135629165U, "Breakable"                   },
	{2136439061U, "Looping"                     },
	{2136832668U, "ShakeX"                      },
	{2140056830U, "Show_Effects"                },
	{2140602746U, "IsSpeech"                    },
	{2141690447U, "Resist32"                    },
	{2142815465U, "Anim_CbtStrFSpeed"           },
	{2143385628U, "Exponent"                    },
	{2147880983U, "SecondaryMove"               },
	{2153136386U, "Token"                       },
	{2156696683U, "IX"                          },
	{2160110268U, "DoNotDrop"                   },
	{2161446662U, "MiniMapWidth"                },
	{2163480618U, "Grass"                       },
	{2164268443U, "FatigueMax"                  },
	{2168401164U, "AttComment"                  },
	{2168416685U, "Tactics"                     },
	{2169280470U, "Ingredient2"                 },
	{2169473928U, "Maximum_Distance"            },
	{2171491163U, "StrRef4"                     },
	{2171568990U, "Den975ar_Rogue_D"            },
	{2171645447U, "Abilityscorerequired2"       },
	{2172565285U, "Constant"                    },
	{2173189519U, "FemaleNameStrRef"            },
	{2173998515U, "Bitmask"                     },
	{2177534914U, "AreaHighlight1"              },
	{2177910009U, "PluralStrRef"                },
	{2178355567U, "A14"                         },
	{2178819685U, "Member3"                     },
	{2178938749U, "TopScreenType"               },
	{2180162436U, "Range"                       },
	{2181445742U, "HSVHue"                      },
	{2181884564U, "Warrior7"                    },
	{2186952706U, "Rules"                       },
	{2191224990U, "WeaponSet1LeftStack"         },
	{2192165329U, "GizoidA"                     },
	{2194249535U, "GUITypeAggressive"           },
	{2196920784U, "MoveSpeed"                   },
	{2197112974U, "BuseInCharGen"               },
	{2198466884U, "AnimationID"                 },
	{2198894407U, "Ability5"                    },
	{2200431129U, "LightXAdjust"                },
	{2200889004U, "OrientWithGround"            },
	{2203044030U, "NonEquipped"                 },
	{2205686216U, "Modifier2"                   },
	{2205944993U, "Item3"                       },
	{2205976343U, "Action9"                     },
	{2209917005U, "Huge"                        },
	{2210137737U, "AttackBR_Ring"               },
	{2213155958U, "MagicalDebuf"                },
	{2213868866U, "Resist14"                    },
	{2215808885U, "Combo2"                      },
	{2216583589U, "Male"                        },
	{2216597962U, "Loops6"                      },
	{2220569185U, "Prop2"                       },
	{2221506381U, "LocationY"                   },
	{2223860817U, "AttributeMod"                },
	{2225042335U, "Feats"                       },
	{2227427034U, "CombatEffect_Precisions_VFX" },
	{2227520371U, "Cir100ar_Docks"              },
	{2233085904U, "TableResRef"                 },
	{2238016334U, "ArenaTextureAnimation"       },
	{2246172466U, "Fire"                        },
	{2247733875U, "Metal"                       },
	{2248136550U, "AngleTarg"                   },
	{2248595083U, "DefaultArmor"                },
	{2249693346U, "AcceptedFlag"                },
	{2249819305U, "Float0"                      },
	{2254055456U, "From"                        },
	{2256724988U, "Symbol"                      },
	{2258792754U, "BlockVis"                    },
	{2266578190U, "LargeShieldStrRef"           },
	{2269210010U, "Ntb100ar_Dalish_Camp"        },
	{2269678206U, "ReactiveLimit"               },
	{2271918650U, "StrengthPerLevel"            },
	{2273289380U, "LowQuality"                  },
	{2275279269U, "TitleIconPalette"            },
	{2277740506U, "Effect2_Int1"                },
	{2278400518U, "SRModPlayer"                 },
	{2279366397U, "AvoidAOEChance"              },
	{2285260970U, "InitialSpeed"                },
	{2287436658U, "Tag"                         },
	{2287627245U, "Resist25"                    },
	{2289118126U, "Visible"                     },
	{2291060341U, "ProfilePortraitPalette"      },
	{2292776868U, "AbilityMode"                 },
	{2295505895U, "DefaultEffectFlags"          },
	{2296059749U, "BloodColr"                   },
	{2301771535U, "ArmyMoveToWP"                },
	{2302340088U, "OfferID"                     },
	{2302623020U, "Slot2"                       },
	{2305563983U, "SoundCessation"              },
	{2307023229U, "Anim_Selected"               },
	{2308108966U, "BombPawn"                    },
	{2310598157U, "Loops12"                     },
	{2311710602U, "Weight7"                     },
	{2312772619U, "WeaponSet1Left"              },
	{2313048561U, "IsHeavy"                     },
	{2314008506U, "Cessation_ProgFX"            },
	{2316389217U, "Material5"                   },
	{2318174445U, "MicroPortrait"               },
	{2318295519U, "FilterAOE"                   },
	{2319723262U, "ValidDeathBlows"             },
	{2322629297U, "AutoScale"                   },
	{2325157920U, "Normal"                      },
	{2329486531U, "Armed"                       },
	{2336417210U, "HealthPerLevel"              },
	{2337223243U, "Effect4"                     },
	{2344338982U, "Variance"                    },
	{2344798917U, "pl-pl"                       },
	{2348350584U, "WieldModel"                  },
	{2350250932U, "Material10"                  },
	{2352872143U, "HeavyArmorLevel"             },
	{2352951228U, "Bonus3"                      },
	{2353000701U, "PackageIDForAI"              },
	{2353685530U, "SkillID"                     },
	{2354693117U, "Y4"                          },
	{2363520797U, "GeneratePathInfo"            },
	{2363921839U, "Den952ar_Alley_Justice_3"    },
	{2364686109U, "SpecPoint"                   },
	{2370953018U, "Raxos"                       },
	{2375360280U, "PreferRange"                 },
	{2378961544U, "Count4"                      },
	{2380290300U, "AreaWidth"                   },
	{2382122039U, "Unsheath"                    },
	{2383800609U, "AbilityCostModifier"         },
	{2384071844U, "Abi3"                        },
	{2390103084U, "Rogue7"                      },
	{2391868211U, "Orz520ar_Aeducan_Thaig"      },
	{2393431076U, "Sound_Default_Armor"         },
	{2395963812U, "AchievementFlag"             },
	{2396536324U, "Spell1"                      },
	{2397161808U, "DurationVFX"                 },
	{2398531649U, "Graphic"                     },
	{2399466007U, "Scatter"                     },
	{2399556628U, "RewardItem2"                 },
	{2403565110U, "LevelsPerSkill"              },
	{2405094532U, "SecondaryMoveLevel"          },
	{2407608778U, "DefaultNameMale"             },
	{2410463603U, "E1033"                       },
	{2412432260U, "AllowTemp"                   },
	{2415796435U, "Cooldown_Group"              },
	{2416085141U, "AttachNode"                  },
	{2418037460U, "QBBehaviour"                 },
	{2418281581U, "Slot3"                       },
	{2420510472U, "DamageBonus"                 },
	{2420655810U, "CessationSound"              },
	{2420973991U, "Open"                        },
	{2421151935U, "BulletVFX"                   },
	{2422687321U, "Enter"                       },
	{2424814557U, "ChestModelVariation"         },
	{2424862182U, "AIabilityUseMod"             },
	{2425390554U, "Distance"                    },
	{2425910871U, "StaticTime"                  },
	{2426390554U, "DialogTag"                   },
	{2429736651U, "Weight6"                     },
	{2430422578U, "Boar"                        },
	{2432202502U, "NoPlayNext"                  },
	{2432471995U, "Mass"                        },
	{2434026493U, "StringReference"             },
	{2436771988U, "OcclusionFactor"             },
	{2436979372U, "Resist24"                    },
	{2439770381U, "Generic"                     },
	{2440773219U, "Green"                       },
	{2440883898U, "FriendlyFire"                },
	{2442771046U, "ColorG"                      },
	{2443253221U, "ModelScaleZ"                 },
	{2444002695U, "Archer"                      },
	{2444111690U, "Armor-Chain"                 },
	{2444912200U, "TopScreenID"                 },
	{2446625278U, "Property1"                   },
	{2451756167U, "ResetDelay"                  },
	{2452757268U, "AngleATK"                    },
	{2454305565U, "FCloudDensity"               },
	{2456492446U, "MinAnimationYaw"             },
	{2456596402U, "GunCommander"                },
	{2457077293U, "AppearanceRestrictionGroup"  },
	{2458885129U, "Pre200ar_Korcari_Wilds"      },
	{2459750058U, "Shade"                       },
	{2466937376U, "Material4"                   },
	{2467120275U, "VFX_Selected"                },
	{2468231584U, "SortOrder"                   },
	{2472045661U, "Variations"                  },
	{2473722178U, "FEarthInScatterPower"        },
	{2474179178U, "Spider"                      },
	{2475312025U, "String_Value"                },
	{2475781560U, "Weapon"                      },
	{2477132903U, "TargetAnimation"             },
	{2477493223U, "Event2"                      },
	{2478392023U, "Style"                       },
	{2480522211U, "AttackFL_Width"              },
	{2480670932U, "InitialTalentLabel"          },
	{2484647422U, "MaxFatigue"                  },
	{2484992863U, "Dirt"                        },
	{2487204286U, "PrimaryTransition"           },
	{2488210060U, "Effect3_Duration"            },
	{2488680355U, "Range_Table"                 },
	{2491191812U, "HumanMale"                   },
	{2491389313U, "ArmorBase"                   },
	{2494482129U, "DescriptionStrRef"           },
	{2495763070U, "ArmyBufferVar"               },
	{2498092949U, "Den200ar_Market"             },
	{2498605588U, "Above10_Resistance"          },
	{2500774645U, "Material11"                  },
	{2500990046U, "VFX"                         },
	{2502172413U, "Bonus2"                      },
	{2507543055U, "Belt"                        },
	{2509185992U, "Game_WalkSpeed"              },
	{2511453992U, "AffectPlot"                  },
	{2515865791U, "MoneyMod"                    },
	{2518508885U, "RewardItem3"                 },
	{2518903197U, "VisibleFlag"                 },
	{2521153418U, "FHealthScale"                },
	{2524156886U, "FBloomBrightness"            },
	{2524313792U, "Dead_2"                      },
	{2531942552U, "SpellPower"                  },
	{2533440997U, "Abi2"                        },
	{2540167044U, "MuzzleFlashVFX"              },
	{2540512621U, "Rogue6"                      },
	{2544446215U, "CameraRotation"              },
	{2546620689U, "DynamicCollisions"           },
	{2547985643U, "Inactive"                    },
	{2549123265U, "PiercingType"                },
	{2549727752U, "BShowInUI"                   },
	{2550745177U, "OpenContainerSound"          },
	{2551479869U, "CanDoDeathBlows"             },
	{2553772083U, "Value_Minimum"               },
	{2555881111U, "Ingredient3"                 },
	{2557233691U, "Big"                         },
	{2559392631U, "Model"                       },
	{2560343945U, "Weapon1"                     },
	{2562700393U, "DecayBehaviour"              },
	{2563013526U, "BitMaskVariable"             },
	{2563053348U, "Member2"                     },
	{2563237122U, "Affect_Plot"                 },
	{2563514414U, "A15"                         },
	{2563881603U, "AreaHighlight0"              },
	{2565960534U, "ConditionalAppear"           },
	{2565995565U, "E1051"                       },
	{2580217947U, "HitPoints"                   },
	{2584263628U, "LevelUpTable"                },
	{2584430086U, "Ability4"                    },
	{2584775058U, "AttackFR_Ring"               },
	{2585176924U, "Hollow_Wood"                 },
	{2585691368U, "PlayerBase"                  },
	{2586970705U, "StateController"             },
	{2590189705U, "Modifier3"                   },
	{2590457312U, "Item2"                       },
	{2590488150U, "Action8"                     },
	{2596187889U, "ConjureAnim"                 },
	{2599404547U, "Resist15"                    },
	{2602032085U, "Warrior6"                    },
	{2602624029U, "Attributes"                  },
	{2603438074U, "BonusAbility"                },
	{2608520048U, "StringID"                    },
	{2608669350U, "ArmyTotalVar"                },
	{2609523902U, "TerrainFollowRoll"           },
	{2613518738U, "SoundLooping"                },
	{2616054951U, "Ability"                     },
	{2617323614U, "CombatTraining"              },
	{2618500507U, "ScrollingBackground"         },
	{2624978556U, "ElapsedTimeFormat"           },
	{2626053968U, "CommandLabel"                },
	{2630132911U, "WieldL"                      },
	{2634393739U, "Loops7"                      },
	{2634661428U, "Combo3"                      },
	{2634892015U, "FarFade"                     },
	{2635689629U, "E1003"                       },
	{2636639984U, "DexBonus"                    },
	{2638218016U, "Prop3"                       },
	{2640422293U, "FleeProbability"             },
	{2641513996U, "LocationX"                   },
	{2649242625U, "DerivdNameStrRef"            },
	{2649596467U, "StringRef"                   },
	{2655317125U, "Shield-Wood"                 },
	{2655362055U, "FixedOrientation"            },
	{2656079984U, "NLevelScale"                 },
	{2656230015U, "Plot"                        },
	{2656347466U, "ProjectileCount"             },
	{2660489180U, "S1"                          },
	{2660798910U, "TrapPlaceable"               },
	{2663266087U, "WeaponStyle"                 },
	{2663803510U, "Skill_1"                     },
	{2664210376U, "TopConversation"             },
	{2664359000U, "WandStrRef"                  },
	{2666294291U, "Z4"                          },
	{2666467289U, "Additive_Facing"             },
	{2667754984U, "Float1"                      },
	{2668097303U, "Immunity_Effect"             },
	{2669677149U, "AttackDamage1"               },
	{2671336833U, "RandTable"                   },
	{2677944863U, "AttWeight"                   },
	{2679300967U, "PaperDollTree"               },
	{2680898618U, "ElfFemale"                   },
	{2682975475U, "MediumChest"                 },
	{2684110634U, "BaseSR"                      },
	{2685066021U, "OneHitKillArea"              },
	{2687893746U, "PlotNum"                     },
	{2693010705U, "Appearance_Female"           },
	{2693800414U, "Warrior"                     },
	{2698827660U, "Game_RunSpeed"               },
	{2700095129U, "XP"                          },
	{2703648311U, "AmmoType"                    },
	{2705070242U, "Material6"                   },
	{2711913808U, "AddAnimation"                },
	{2712753833U, "Highlight5"                  },
	{2718726636U, "IsStronghold"                },
	{2719117039U, "Slot1"                       },
	{2719375735U, "Simulation"                  },
	{2721971644U, "StartPosX"                   },
	{2723798731U, "SpecialStaging"              },
	{2727722446U, "Loops11"                     },
	{2729736236U, "Success0"                    },
	{2730774884U, "UIMapping"                   },
	{2731439909U, "CombatRating"                },
	{2732897353U, "Weight4"                     },
	{2734689965U, "Probability"                 },
	{2735849142U, "Destroyed"                   },
	{2738126379U, "Swish"                       },
	{2739719004U, "IDTag"                       },
	{2740780179U, "GUID"                        },
	{2741177323U, "Int0"                        },
	{2741312754U, "Combo10"                     },
	{2742499374U, "Resist26"                    },
	{2744612711U, "ModelScaleX"                 },
	{2745677577U, "NoTemplates"                 },
	{2746626891U, "EffectDurationMod"           },
	{2748167132U, "Turret"                      },
	{2748184609U, "Active"                      },
	{2753173003U, "Variable"                    },
	{2753723398U, "GUILabel"                    },
	{2754283479U, "RewardItem1"                 },
	{2755631801U, "MaxLevel"                    },
	{2764228619U, "WearL"                       },
	{2766547125U, "BasePrereqAbility2"          },
	{2774222831U, "Rogue4"                      },
	{2780222578U, "Flag"                        },
	{2782421456U, "Icon8"                       },
	{2784325063U, "Spell2"                      },
	{2787925106U, "ColorR"                      },
	{2789204225U, "X"                           },
	{2790341212U, "PrecollisionAnimName"        },
	{2792064960U, "Music"                       },
	{2794026961U, "LightArmorLevel"             },
	{2797134934U, "MaulStrRef"                  },
	{2797796569U, "DurationSound"               },
	{2802487552U, "Additive_Effect"             },
	{2803006591U, "Bonus0"                      },
	{2803349474U, "ExitSync"                    },
	{2805508215U, "Material13"                  },
	{2805741549U, "Desc2StrID"                  },
	{2806884310U, "ConditionBaseComment"        },
	{2807860230U, "ShakeTime"                   },
	{2809758494U, "Drop_Stealth"                },
	{2810036184U, "TitleStrRef"                 },
	{2810073181U, "PointCalc"                   },
	{2811545671U, "Claw"                        },
	{2818209673U, "Target_Range"                },
	{2820888284U, "StaticMovement"              },
	{2822579332U, "Ability6"                    },
	{2823594293U, "DMEasy"                      },
	{2824161803U, "Modifier1"                   },
	{2828136209U, "Level"                       },
	{2828225327U, "Leaves"                      },
	{2828749969U, "Project"                     },
	{2832836225U, "Resist17"                    },
	{2832872719U, "ReqProp1Mod"                 },
	{2833964177U, "Codex"                       },
	{2836223159U, "Map"                         },
	{2837559639U, "Warrior4"                    },
	{2841929536U, "RewardItem"                  },
	{2841940231U, "ConstitutionBase"            },
	{2843764094U, "DefaultIcon"                 },
	{2844039698U, "GizoidB"                     },
	{2854863270U, "LightArmor"                  },
	{2855271166U, "TempXP"                      },
	{2858533909U, "Ingredient1"                 },
	{2863521035U, "Weapon3"                     },
	{2863873581U, "TutorialStrRef"              },
	{2865602632U, "Value_Default"               },
	{2866025753U, "BlendTreeOverlay"            },
	{2867320833U, "AreaHighlight2"              },
	{2867793418U, "NumberChaoEggs"              },
	{2868604139U, "AreaSoundType"               },
	{2871184221U, "UpgradeApr"                  },
	{2871358641U, "PrecollisionTextureName"     },
	{2875327739U, "ModelSubType"                },
	{2879618721U, "TintFile"                    },
	{2880224208U, "Elf"                         },
	{2881795191U, "InvSlotWidth"                },
	{2882905229U, "Worksheet"                   },
	{2883168949U, "Effect2_Resource"            },
	{2888444988U, "Dwarf"                       },
	{2890008990U, "FMieMultiplier"              },
	{2891629248U, "A9"                          },
	{2893422876U, "ObjectMask"                  },
	{2894447321U, "FDamageScale"                },
	{2894994821U, "AI_Armor_Type"               },
	{2896541022U, "S3"                          },
	{2899914118U, "BottomScreenString"          },
	{2900769918U, "Usage"                       },
	{2901335065U, "Effect2_Int2"                },
	{2901394626U, "UseType"                     },
	{2901412084U, "Skill_3"                     },
	{2902653451U, "DwarfFemale"                 },
	{2902731002U, "RegenerationMod"             },
	{2902769563U, "Starfield"                   },
	{2909901670U, "Creature_Group"              },
	{2912732670U, "Instances"                   },
	{2915202862U, "AttackReach"                 },
	{2915345277U, "AreaHeight"                  },
	{2915456045U, "LanguageName"                },
	{2921872386U, "Radius"                      },
	{2922075570U, "AttackFR"                    },
	{2922079733U, "LowOcclusionFactor"          },
	{2922863168U, "FEffectDurNmare"             },
	{2928647253U, "DefensePerLevel"             },
	{2935092114U, "OpenedToClosed"              },
	{2938124350U, "Ice"                         },
	{2938633471U, "Parts"                       },
	{2939406857U, "Loops5"                      },
	{2939636821U, "RockCrusher"                 },
	{2939687179U, "APLevel"                     },
	{2940198070U, "Combo1"                      },
	{2942708804U, "Classification"              },
	{2943771042U, "Prop1"                       },
	{2943990364U, "EnableWeaponSets"            },
	{2950969127U, "DetectLevel"                 },
	{2951384827U, "NameLowerStrID"              },
	{2956625942U, "Warrior5"                    },
	{2957882008U, "PerSpace"                    },
	{2961863763U, "Dagger"                      },
	{2962275647U, "StringText"                  },
	{2964603234U, "Chest"                       },
	{2966073867U, "AbilityID"                   },
	{2966386585U, "LevelScale"                  },
	{2968177003U, "HasTurnAttacks"              },
	{2969293677U, "Armor-Metal"                 },
	{2972063173U, "Ability7"                    },
	{2972972793U, "NameStrID"                   },
	{2974445091U, "Item1"                       },
	{2975607633U, "PrereqAttribute"             },
	{2981204198U, "CollisionTextureAnimName"    },
	{2982320064U, "Resist16"                    },
	{2982866461U, "Conditions"                  },
	{2989293859U, "Name"                        },
	{2990398465U, "Brow"                        },
	{2994120574U, "Copper"                      },
	{2995632265U, "BaseCost"                    },
	{2998385290U, "OvershootTime"               },
	{2998421616U, "IgnoreDeath"                 },
	{3002742850U, "LargeIcon"                   },
	{3003494061U, "Time"                        },
	{3006738729U, "TreasureRank"                },
	{3007707847U, "ZOffset"                     },
	{3007930008U, "KronEmerald"                 },
	{3011865943U, "HSVSaturation"               },
	{3013619372U, "BShowInToolset"              },
	{3015109706U, "Weapon2"                     },
	{3016467590U, "ColorCurves"                 },
	{3017896941U, "A16"                         },
	{3017997743U, "FHSVBrightness"              },
	{3018343655U, "Member1"                     },
	{3019696448U, "AreaHighlight3"              },
	{3020773790U, "AttackDamage2"               },
	{3021738506U, "FactionName"                 },
	{3022079364U, "Combat_BlendTree"            },
	{3022536018U, "Rate"                        },
	{3022801928U, "Orz510ar_Caridins_Cross"     },
	{3023042482U, "ResourceName"                },
	{3023150196U, "ScaleGroup"                  },
	{3024717125U, "ScuffingSound"               },
	{3027340285U, "de-de"                       },
	{3029073762U, "Value_Maximum"               },
	{3029088245U, "CustomAI"                    },
	{3030163814U, "Label"                       },
	{3031987525U, "Group"                       },
	{3032084612U, "Den600ar_Landsmeet_Chamber"  },
	{3032269493U, "Follower"                    },
	{3036612968U, "DSpeed"                      },
	{3036707906U, "PlayerID"                    },
	{3040998273U, "A8"                          },
	{3044067048U, "AppearDisappear"             },
	{3046568111U, "FromBlendTree"               },
	{3046828483U, "PrimaryAttribute"            },
	{3049161759U, "S2"                          },
	{3051425172U, "MediumArmorLevel"            },
	{3051884630U, "Den400ar_Fort"               },
	{3052091829U, "Skill_2"                     },
	{3056139080U, "Loops4"                      },
	{3057327638U, "AreaFlags"                   },
	{3057801397U, "RumbleTime"                  },
	{3058191015U, "AprRulesFlags"               },
	{3059261637U, "JaggingLight"                },
	{3061859009U, "SecondaryTransition"         },
	{3063440168U, "MiniMapHeight"               },
	{3065612152U, "LongNameStrRef"              },
	{3065832028U, "PlayerRace"                  },
	{3067226317U, "TerrainFollowPrecision"      },
	{3069363893U, "BouncePad"                   },
	{3076896379U, "SourceLocation"              },
	{3080144698U, "Palette"                     },
	{3083839057U, "ExploreCache"                },
	{3088195882U, "AttackFR_Angle"              },
	{3089574371U, "Material7"                   },
	{3095726730U, "IsSpec"                      },
	{3095844900U, "Event1"                      },
	{3098175464U, "Highlight4"                  },
	{3100259805U, "Repeat"                      },
	{3103277959U, "Min"                         },
	{3109894807U, "Armor-Leather"               },
	{3111816934U, "Additive"                    },
	{3113677189U, "CameraFollow"                },
	{3119931524U, "Easy"                        },
	{3121081316U, "IxFinal"                     },
	{3121902361U, "Major_VFX"                   },
	{3125319027U, "YPos"                        },
	{3127642479U, "Resist27"                    },
	{3128441514U, "Int1"                        },
	{3129755174U, "ModelScaleY"                 },
	{3130447832U, "ChanceTreasure"              },
	{3133937291U, "FFogCap"                     },
	{3134880526U, "IconName"                    },
	{3136641999U, "Type_Data"                   },
	{3136936509U, "Property2"                   },
	{3137953710U, "Slot0"                       },
	{3139842965U, "AchievementID"               },
	{3139882237U, "StartPosY"                   },
	{3141627752U, "FHG"                         },
	{3146436923U, "Sound_MatType"               },
	{3146681487U, "Loops10"                     },
	{3146809506U, "PurgeGroup"                  },
	{3147552667U, "SoundCessationVolume"        },
	{3148826989U, "Success1"                    },
	{3150794451U, "CountMage"                   },
	{3154085128U, "Weight5"                     },
	{3154183300U, "String_Description"          },
	{3155310338U, "VoxaiCitizen"                },
	{3155931548U, "BloodPool"                   },
	{3157028390U, "Abi1"                        },
	{3158341294U, "Rogue5"                      },
	{3160612977U, "StorePanel"                  },
	{3161679197U, "TerrianFollowData"           },
	{3166916753U, "Icon9"                       },
	{3167176455U, "ConditionParameter2"         },
	{3167562026U, "AbilityScoreRequired"        },
	{3168647281U, "Icon"                        },
	{3169737862U, "Spell3"                      },
	{3170147855U, "cs-cz"                       },
	{3170656785U, "Parameter"                   },
	{3174468721U, "Notes"                       },
	{3176680556U, "MaceStrRef"                  },
	{3178054726U, "Anim_RunSpeed"               },
	{3184299446U, "LookupAnimation"             },
	{3186620195U, "GuardianPawn"                },
	{3188126447U, "CrustEffect"                 },
	{3188280638U, "Bonus1"                      },
	{3190004022U, "Material12"                  },
	{3190443628U, "Orz550ar_Dead_Trenches"      },
	{3193117267U, "Arl200ar_Redcliffe_Castle"   },
	{3200736312U, "Product"                     },
	{3202387690U, "Nodes"                       },
	{3205285445U, "MinItemProps"                },
	{3206860864U, "Y"                           },
	{3210535805U, "AttackFL_Reach"              },
	{3212090977U, "AvailableEquipmentSlots"     },
	{3213122795U, "StartTime"                   },
	{3223400124U, "AllowPermDeath"              },
	{3223982083U, "StorePanelSort"              },
	{3226282117U, "MaxAnimationPitch"           },
	{3229176464U, "ConversationTheme"           },
	{3233343942U, "TutorialImage"               },
	{3233563736U, "OrientUniteable"             },
	{3235301362U, "PropColumn"                  },
	{3236779956U, "E1034"                       },
	{3238537073U, "Totems"                      },
	{3238659137U, "AttackBL_Angle"              },
	{3238748222U, "DecimalSeparator"            },
	{3239242053U, "SoundDeactivate0"            },
	{3242529502U, "ImpactSound"                 },
	{3249439924U, "Blood_Decal"                 },
	{3254083515U, "Animations"                  },
	{3254877640U, "HealthMod"                   },
	{3254968026U, "HSVBrightness"               },
	{3255555192U, "PassiveAbi0"                 },
	{3256973618U, "SmallIcon3"                  },
	{3264054351U, "Count3"                      },
	{3265233046U, "SwitchBroken"                },
	{3265813123U, "RotationRange"               },
	{3268311908U, "HowToStrRef"                 },
	{3268637392U, "PartName"                    },
	{3270381765U, "Combat"                      },
	{3273151802U, "Y3"                          },
	{3276895018U, "BlendCurves"                 },
	{3278431855U, "Frame"                       },
	{3279862139U, "Bonus4"                      },
	{3280206525U, "ConversationPortrait"        },
	{3281192727U, "CancelTime"                  },
	{3283562075U, "Inactive_Disabled"           },
	{3287497183U, "Alignment"                   },
	{3289293964U, "Effect3"                     },
	{3292878117U, "TileSizeX"                   },
	{3295374029U, "Axe"                         },
	{3296075233U, "FollowerTag"                 },
	{3296099539U, "Skills"                      },
	{3296158194U, "Replacement"                 },
	{3296218711U, "GardenPositionX"             },
	{3297718715U, "HilightPrefix"               },
	{3302620885U, "Crust_ID"                    },
	{3303481875U, "PickLockSuccess"             },
	{3303968480U, "Passive_ID"                  },
	{3304958445U, "BHostile"                    },
	{3306227511U, "Tag2"                        },
	{3310411174U, "Material2"                   },
	{3315896675U, "IsBoss"                      },
	{3316592266U, "NearFade"                    },
	{3319632813U, "Highlight1"                  },
	{3321005272U, "MetropolisConveyor"          },
	{3326055794U, "SRMod"                       },
	{3326395926U, "AttackBR_Reach"              },
	{3330190315U, "Slot5"                       },
	{3331949436U, "Effect3_Float1"              },
	{3335108195U, "TS_LowTable"                 },
	{3335535795U, "DexterityPerlevel"           },
	{3336922211U, "Resist8"                     },
	{3340500266U, "Resist22"                    },
	{3341678950U, "ModuleRef"                   },
	{3341915582U, "CooldownMod"                 },
	{3348125760U, "Cir330ar_Fade_Mage_Asunder"  },
	{3349094600U, "BDisabled"                   },
	{3351453152U, "ColorA"                      },
	{3351517635U, "SecondaryRange"              },
	{3351995745U, "Sound_Monster_Size"          },
	{3353721636U, "Condition_AppGroup"          },
	{3355043960U, "Result"                      },
	{3357271082U, "Letter"                      },
	{3358341153U, "NoticeDistance"              },
	{3360153171U, "Creature_Type"               },
	{3364450296U, "WeightMage"                  },
	{3367422869U, "Z2"                          },
	{3367757146U, "InactivePlot"                },
	{3368387340U, "LoadScreen"                  },
	{3368927199U, "StrengthBase"                },
	{3373346266U, "ModuleName"                  },
	{3374383112U, "PortraitTint"                },
	{3374499892U, "AnimationPackage"            },
	{3374767446U, "NocturnElite"                },
	{3377900553U, "VFX_Impact0"                 },
	{3378017265U, "ElfMale"                     },
	{3385002812U, "MultiDestination_VFX"        },
	{3387329613U, "ProcChance"                  },
	{3387939818U, "ModelName2"                  },
	{3395271657U, "Conditional"                 },
	{3395304241U, "ToolTipStrRef"               },
	{3395337964U, "Movement1"                   },
	{3396327154U, "GestureBlendTree"            },
	{3401985754U, "Show_Tactics"                },
	{3402570489U, "GameStrRef"                  },
	{3403351359U, "Steel_Golem"                 },
	{3406679328U, "WeightRogue"                 },
	{3411220914U, "Combo5"                      },
	{3411287379U, "Temporary"                   },
	{3411955939U, "HeroBonus3"                  },
	{3412004621U, "Loops1"                      },
	{3412182292U, "Effect_Int2"                 },
	{3426589286U, "Item4"                       },
	{3427918208U, "Ability2"                    },
	{3427932985U, "DefaultAttack"               },
	{3431059519U, "Large"                       },
	{3434182850U, "Blue"                        },
	{3434439557U, "Resist13"                    },
	{3435783169U, "LODDistance"                 },
	{3437351074U, "NocturnFlyer"                },
	{3438387072U, "ProgressBar"                 },
	{3443383026U, "Bone"                        },
	{3443460669U, "Sonic"                       },
	{3448533363U, "Effect_Float2"               },
	{3449602883U, "Anim_WalkSpeed"              },
	{3450871232U, "ToolsetName"                 },
	{3453288805U, "DIHexValue"                  },
	{3453440946U, "SynchronizedCombat"          },
	{3456783208U, "Small"                       },
	{3459237276U, "StrRef3"                     },
	{3459891149U, "CorpseModel"                 },
	{3465964971U, "FEarthReflectance"           },
	{3465971624U, "A13"                         },
	{3466571938U, "Member4"                     },
	{3467247366U, "AIcommand_Invalid"           },
	{3468595699U, "Effect3_Resource"            },
	{3468761103U, "Weapon7"                     },
	{3471581252U, "DeathBlow_Type"              },
	{3472955154U, "MiscEquipped"                },
	{3473853936U, "Animation_Base"              },
	{3475300124U, "YOffset"                     },
	{3479050997U, "Visualizer"                  },
	{3481404569U, "Total"                       },
	{3482668559U, "Thelxe"                      },
	{3482722700U, "PositionX"                   },
	{3484872528U, "AllowEquip"                  },
	{3489467330U, "SoundSetProbability"         },
	{3490630172U, "Emotion"                     },
	{3494514973U, "LowViolence"                 },
	{3494747464U, "VFX_Impact1"                 },
	{3494953481U, "Animation_VFX"               },
	{3497168302U, "Robodillo"                   },
	{3500893005U, "Minor_VFX"                   },
	{3502527829U, "MaxScaleLevel"               },
	{3503386768U, "TopScreen"                   },
	{3503820429U, "String_Ref"                  },
	{3505711787U, "ModelName3"                  },
	{3508522517U, "TerrainFollowPitch"          },
	{3509156654U, "Gravel"                      },
	{3509931886U, "IntAdjust"                   },
	{3510726663U, "TopScreenString"             },
	{3512885761U, "HeavyChest"                  },
	{3514723787U, "Broken"                      },
	{3517840084U, "Z3"                          },
	{3518693192U, "BackgroundImage"             },
	{3520124964U, "EventData"                   },
	{3520217371U, "S6"                          },
	{3521861138U, "Color"                       },
	{3522156947U, "AreaSoundBank"               },
	{3522837001U, "AmbientRed"                  },
	{3523078299U, "FBloomContrast"              },
	{3523281925U, "Riadne"                      },
	{3525603675U, "TS_Money"                    },
	{3527687756U, "Loops0"                      },
	{3527762338U, "HeroBonus2"                  },
	{3527944435U, "Combo4"                      },
	{3531273301U, "Effect_Int3"                 },
	{3538089905U, "hu-hu"                       },
	{3540162403U, "FakeArea"                    },
	{3544436510U, "AllowProjectileVFX"          },
	{3544764912U, "AreaWeight"                  },
	{3546952750U, "FHue"                        },
	{3549032900U, "FollowerPlot"                },
	{3550183989U, "ArmorType"                   },
	{3555622367U, "Scylla"                      },
	{3557088714U, "Inactive_Enabled"            },
	{3562425618U, "Warrior1"                    },
	{3563377283U, "Threat_Use"                  },
	{3574234808U, "NumMoves"                    },
	{3576081191U, "Item5"                       },
	{3576217033U, "DisabledLabel"               },
	{3577730123U, "EarlyFailure"                },
	{3578053563U, "Mage8"                       },
	{3578279452U, "Creature_Large"              },
	{3578450113U, "Ability3"                    },
	{3579510485U, "BaseAimDelay"                },
	{3579540365U, "MaterialGroup"               },
	{3580784137U, "OverrideTree"                },
	{3584306345U, "Anim_Deselected_Loop"        },
	{3584606049U, "Currency"                    },
	{3584971460U, "Resist12"                    },
	{3585125922U, "EquippableSlots"             },
	{3590573002U, "ArtifactBlur"                },
	{3591498606U, "InitialTalentID"             },
	{3593680700U, "Summoned_Template"           },
	{3594654819U, "DerivdTtipStrRef"            },
	{3594826938U, "InitialSkillID"              },
	{3595590305U, "CreatureProp"                },
	{3599569101U, "PositionY"                   },
	{3608555600U, "Ingredient4"                 },
	{3610571997U, "StrRef2"                     },
	{3614706721U, "Quiver"                      },
	{3616388841U, "A12"                         },
	{3616697880U, "Orz540ar_Anvil_Of_The_Void"  },
	{3619386215U, "StrID"                       },
	{3619864304U, "SlowAngle"                   },
	{3619927371U, "Creature_Small"              },
	{3620801830U, "CreatureID"                  },
	{3621398862U, "Weapon6"                     },
	{3624523780U, "SoundDeactivate1"            },
	{3625297033U, "Den951ar_Alley_Justice_2"    },
	{3626729665U, "TriggerPlotFlag"             },
	{3626808234U, "Rogue1"                      },
	{3627024582U, "AppearanceRating"            },
	{3633303845U, "MinWeight"                   },
	{3633761261U, "RankRating"                  },
	{3636133701U, "Den970ar_Franderel_Estate"   },
	{3640343709U, "AlwaysDraw"                  },
	{3640740754U, "Subtitles"                   },
	{3643543327U, "Feffectdurhard"              },
	{3643811693U, "Disabledcursorid"            },
	{3644612993U, "Hardcore"                    },
	{3646815623U, "ConAdjust"                   },
	{3648819397U, "Flags"                       },
	{3653759900U, "ParameterTable"              },
	{3656261301U, "IsNegative"                  },
	{3656804085U, "E1035"                       },
	{3657655419U, "Y2"                          },
	{3659770047U, "TacticsPreset"               },
	{3662209561U, "Sound_Footstep_Type"         },
	{3665278348U, "Wand"                        },
	{3666259233U, "ImageRepeat"                 },
	{3672431091U, "SliderMax"                   },
	{3672807951U, "Tornado"                     },
	{3673647843U, "MinAnimationPitch"           },
	{3673905671U, "TalentLabel"                 },
	{3673918379U, "PowerScale"                  },
	{3674795958U, "Xbox"                        },
	{3675317561U, "PassiveAbi1"                 },
	{3675711535U, "BottomScreen"                },
	{3676649625U, "CameraZAdjust"               },
	{3676754820U, "MinSpeed"                    },
	{3678029939U, "SmallIcon2"                  },
	{3678150485U, "FEffectDurCasual"            },
	{3678486078U, "Viral"                       },
	{3679415459U, "CommandType"                 },
	{3679752029U, "GenderName"                  },
	{3680500760U, "CostModifier"                },
	{3681554153U, "Effect1_Int2"                },
	{3684087054U, "Count2"                      },
	{3684241138U, "ProjectileCrust"             },
	{3687278596U, "Armor-Cloth"                 },
	{3688663088U, "BDropStealth"                },
	{3691656822U, "Tag3"                        },
	{3691997834U, "InvSlotHeight"               },
	{3693739545U, "HeavyGloves"                 },
	{3695963367U, "Material3"                   },
	{3696343373U, "BlendTime"                   },
	{3701248653U, "Neck_Model"                  },
	{3701848269U, "Comment"                     },
	{3702596860U, "BaseDefense"                 },
	{3704005356U, "Highlight0"                  },
	{3704973875U, "MagicalBuff"                 },
	{3709179341U, "Effect2"                     },
	{3711341858U, "Anim_Selected_Loop"          },
	{3712760058U, "CountWarrior"                },
	{3712977050U, "MaxAllowedPitch"             },
	{3713950820U, "TileSizeY"                   },
	{3713953536U, "Texture"                     },
	{3714165007U, "ChanceMoney"                 },
	{3714285334U, "GardenPositionY"             },
	{3715047050U, "Rock"                        },
	{3715622303U, "CodexFlag"                   },
	{3719478560U, "Gentleman"                   },
	{3721023851U, "ru-ru"                       },
	{3722492933U, "Anim"                        },
	{3723278907U, "Layering"                    },
	{3724594283U, "Resist23"                    },
	{3725900464U, "ShakeRoll"                   },
	{3727849837U, "GUIType"                     },
	{3729993661U, "GroupType"                   },
	{3731832628U, "RelativeValue"               },
	{3732920034U, "Click"                       },
	{3732976664U, "CombatModelTree"             },
	{3734176102U, "Target"                      },
	{3736328496U, "ConditionFollower"           },
	{3736339053U, "EliteBoss"                   },
	{3740457577U, "RateMax"                     },
	{3741378457U, "ResRef"                      },
	{3747977898U, "Slot4"                       },
	{3748127644U, "Immunity_Ability"            },
	{3748560500U, "PrimaryMoveLevel"            },
	{3748899975U, "ValidForTargetComment"       },
	{3750969356U, "Weight1"                     },
	{3757987106U, "Resist9"                     },
	{3758389027U, "ControlData"                 },
	{3759587088U, "BHasAppearAnim"              },
	{3761150793U, "Class"                       },
	{3763115735U, "Effect_Int1"                 },
	{3765591246U, "Loops2"                      },
	{3766372977U, "Combo6"                      },
	{3772772313U, "Type_UI"                     },
	{3776248710U, "AmbientGreen"                },
	{3777353541U, "Category"                    },
	{3779149075U, "Puddle"                      },
	{3779980591U, "Movement2"                   },
	{3783193444U, "PrereqLevel"                 },
	{3784301918U, "Auto_Apply"                  },
	{3785332913U, "EnableCollisionEvent"        },
	{3785946346U, "AbilityType"                 },
	{3787786014U, "BRestricted"                 },
	{3788879801U, "Urn100ar_Cultists_Village"   },
	{3790809094U, "Frequency_Y"                 },
	{3793223813U, "DropTable"                   },
	{3796132863U, "BoardID"                     },
	{3796567885U, "CombatEffect"                },
	{3797788353U, "ParentCategory"              },
	{3798736744U, "Success1a"                   },
	{3800491056U, "PauseRate"                   },
	{3802650159U, "Width"                       },
	{3804206285U, "TyxisWarrior"                },
	{3804399657U, "ModelName1"                  },
	{3809681793U, "DefeatTip"                   },
	{3810745643U, "HandleFlag"                  },
	{3810991966U, "CanRunForPrc"                },
	{3812050623U, "Defense"                     },
	{3812325467U, "Minor_Effect"                },
	{3813617881U, "Use_Condition_Group"         },
	{3813741249U, "AutoDraw"                    },
	{3813982793U, "BottomConversation"          },
	{3814846900U, "LightChest"                  },
	{3818625110U, "Z1"                          },
	{3818830412U, "MinimumCost"                 },
	{3821995487U, "MediumGloves"                },
	{3823379353U, "S4"                          },
	{3823722273U, "AttackFRWidth"               },
	{3828799618U, "Skill"                       },
	{3829211547U, "UnlockedAt"                  },
	{3829640438U, "Condition_Group"             },
	{3831144414U, "ShieldPawn"                  },
	{3831640960U, "UVAnimationFilename"         },
	{3833930352U, "GiantWorm"                   },
	{3835973039U, "NodeName"                    },
	{3839249358U, "AttackerResult"              },
	{3843671363U, "Hide"                        },
	{3849507294U, "AttackExtra"                 },
	{3851166947U, "UnlockNotification"          },
	{3851756701U, "PrecollisionModelName"       },
	{3851926318U, "FadeinTime"                  },
	{3851938683U, "RenderBlurRadius"            },
	{3852881606U, "Areahighlight5"              },
	{3853219791U, "PauseTime"                   },
	{3854292075U, "A10"                         },
	{3857468364U, "Weapon4"                     },
	{3860872062U, "Ammo"                        },
	{3862538437U, "DwarfOffset"                 },
	{3865063312U, "Warrior3"                    },
	{3868453778U, "CodexPlot"                   },
	{3869129205U, "Item"                        },
	{3869319856U, "Effect_Float1"               },
	{3874293234U, "Den300ar_Elven_Alienage"     },
	{3876066832U, "Branch"                      },
	{3876352629U, "Additive_VFX"                },
	{3876637093U, "Item7"                       },
	{3878962831U, "Switch"                      },
	{3882354086U, "String"                      },
	{3883375756U, "DefaultModel"                },
	{3883708995U, "Ability1"                    },
	{3885510726U, "Resist10"                    },
	{3887359648U, "MaskFile"                    },
	{3889806208U, "Weapon18"                    },
	{3890253173U, "Skippable"                   },
	{3890503361U, "Block"                       },
	{3890651564U, "LookCenterAnimation"         },
	{3891682720U, "RepeatVar"                   },
	{3891721707U, "ChanceToSpawn"               },
	{3892269670U, "TextureAnim"                 },
	{3900260016U, "Material14"                  },
	{3901905064U, "ShowUpdated"                 },
	{3902822704U, "AttInit"                     },
	{3910137788U, "DamageMod"                   },
	{3910971971U, "ClassTable"                  },
	{3915752458U, "CollisionModelName"          },
	{3918642513U, "State"                       },
	{3922541164U, "ReqProp1Val"                 },
	{3923340551U, "SoundFlying"                 },
	{3926457399U, "TriggersTable"               },
	{3926984101U, "FullFace"                    },
	{3927626024U, "Rogue3"                      },
	{3928022310U, "FBloomHBlurWidth"            },
	{3928253333U, "ReqScale"                    },
	{3928698813U, "Flags2"                      },
	{3931411289U, "TacticPreset"                },
	{3938576008U, "AttackBR_Width"              },
	{3939967029U, "Special2"                    },
	{3947417642U, "MovementTree"                },
	{3951357517U, "IsCodex"                     },
	{3954156598U, "SubCommandLabel"             },
	{3954299140U, "Dead"                        },
	{3958772845U, "SuicideDrone"                },
	{3961817308U, "SoundVariation"              },
	{3963006697U, "Resist21"                    },
	{3963579354U, "Stone"                       },
	{3964333308U, "TriggerCondPlot"             },
	{3973950146U, "AmbientAnim"                 },
	{3973951270U, "MaulLevel"                   },
	{3974246258U, "ShowInCharGen"               },
	{3975040547U, "ColorB"                      },
	{3976007201U, "MassiveArmorLevel"           },
	{3979750556U, "TexFrame"                    },
	{3979765386U, "AddBehavior"                 },
	{3981703208U, "Slot6"                       },
	{3982012617U, "MeleeRingOffset"             },
	{3982262343U, "Cir310ar_Fade_Burning_Tower" },
	{3984976028U, "DonationMsgStrID"            },
	{3987022478U, "Weight3"                     },
	{3988050111U, "Effect3_Float2"              },
	{3991628673U, "NStealthRank"                },
	{3992987874U, "AmbientTrack"                },
	{3996931316U, "Tag1"                        },
	{4001205861U, "Material1"                   },
	{4005536023U, "Eggman"                      },
	{4005769326U, "Dmg_Param0"                  },
	{4008755310U, "Highlight2"                  },
	{4010074733U, "MaceLevel"                   },
	{4014943936U, "EquipRules"                  },
	{4016806506U, "FollowerBehavior"            },
	{4016958088U, "No_Props"                    },
	{4020760148U, "TripsCounterTable"           },
	{4022271340U, "CameraDistance"              },
	{4023238248U, "ElementalDamage"             },
	{4023283012U, "DefaultAmmoVariation"        },
	{4024732903U, "ChanceAllToFail"             },
	{4026874824U, "PlotFlag"                    },
	{4028046256U, "SmallIcon1"                  },
	{4029050402U, "SuperSonicBall"              },
	{4030559806U, "Cost"                        },
	{4032650538U, "Effect1_Int1"                },
	{4033394785U, "SoundImpactPitch"            },
	{4035217115U, "XPReward"                    },
	{4035341531U, "DoTransition"                },
	{4035707095U, "XPValue"                     },
	{4038846157U, "Count1"                      },
	{4039116744U, "ShakeZFreq"                  },
	{4039406263U, "Lot100ar_Lothering"          },
	{4040075570U, "AffectImmortal"              },
	{4040813163U, "Text"                        },
	{4040874411U, "Above10_MainSystem"          },
	{4041832710U, "HP"                          },
	{4041861912U, "AlbinoWasp"                  },
	{4043451597U, "Den971ar_Franderel_Estate_2" },
	{4044641107U, "CountRogue"                  },
	{4045941194U, "Effect2_Duration"            },
	{4046337976U, "Y1"                          },
	{4048885332U, "BCache"                      },
	{4049735665U, "Material15"                  },
	{4050457386U, "Effect_Label"                },
	{4057317829U, "React_VFX"                   },
	{4064094881U, "ShowInPaperDoll"             },
	{4065552832U, "EntryIdx"                    },
	{4066976339U, "Tint"                        },
	{4068830396U, "NBonusPoints"                },
	{4073030320U, "TacticsTable"                },
	{4073552250U, "AvoidAOE"                    },
	{4074452278U, "E1036"                       },
	{4075059640U, "en-us"                       },
	{4076986473U, "Rogue2"                      },
	{4080270857U, "Script"                      },
	{4083149655U, "InactiveFlag"                },
	{4084955326U, "RumbleBalance"               },
	{4086700450U, "HideWhenUndetected"          },
	{4086974314U, "Defend"                      },
	{4091663077U, "ShakeRollFreq"               },
	{4095279793U, "CollisionUVAnimName"         },
	{4095623476U, "AreaListName"                },
	{4095975552U, "TriggerChance"               },
	{4096883099U, "ProjectileVFX"               },
	{4097169397U, "ShakeXFreq"                  },
	{4098410857U, "Slot7"                       },
	{4101564375U, "MovementSlide"               },
	{4103458417U, "Stature"                     },
	{4103855738U, "Grit"                        },
	{4105460571U, "TS_EquipmentChance"          },
	{4106097615U, "Weight2"                     },
	{4110736530U, "BareHands"                   },
	{4111007034U, "AttackBL"                    },
	{4113407912U, "Resist20"                    },
	{4113510532U, "DisplayMask"                 },
	{4116951280U, "LargeCreature"               },
	{4117239826U, "DisplayType"                 },
	{4124773871U, "FMoonScale"                  },
	{4127988269U, "BaseDefenseBonus"            },
	{4130915854U, "Effect1"                     },
	{4131661443U, "Conversation"                },
	{4132318421U, "GardenPositionZ"             },
	{4150543258U, "OnClick"                     },
	{4152309341U, "InventoryLargeIcon"          },
	{4153844500U, "Herbalism_1"                 },
	{4154413663U, "HostileAOE"                  },
	{4158128431U, "Dmg_Param1"                  },
	{4159435055U, "Highlight3"                  },
	{4160861003U, "Enemy1"                      },
	{4161236323U, "ArmorChest"                  },
	{4165303809U, "Image"                       },
	{4166442094U, "Movement3"                   },
	{4170679821U, "AnimImpact"                  },
	{4171920349U, "Length"                      },
	{4174488590U, "CollisionDamage"             },
	{4175283712U, "AttackInescapable"           },
	{4175960391U, "Frequency_X"                 },
	{4176180651U, "Sheath"                      },
	{4183034357U, "CastAnim"                    },
	{4184177456U, "Combo7"                      },
	{4184355425U, "HeroBonus1"                  },
	{4184436111U, "Loops3"                      },
	{4186912019U, "Chance"                      },
	{4190962293U, "StartWP"                     },
	{4191282132U, "SnowPile"                    },
	{4192749345U, "ArenaAnimation"              },
	{4194070896U, "Armor"                       },
	{4198606075U, "Red"                         },
	{4198879290U, "AttackBRWidth"               },
	{4200054996U, "Projectile_Secondary"        },
	{4201034125U, "ProjectileOffset"            },
	{4201640806U, "BootsModelVariation"         },
	{4204816209U, "BaseVolume"                  },
	{4205337383U, "Abbreviation"                },
	{4208310877U, "GDAName"                     },
	{4208995734U, "Animation_Override_Movement" },
	{4211020504U, "S5"                          },
	{4221383533U, "WaterSlide"                  },
	{4224581544U, "ConversationTag"             },
	{4229539614U, "StrRef1"                     },
	{4238402858U, "A11"                         },
	{4238930973U, "RemoveAnimation"             },
	{4240277383U, "AreaHighlight4"              },
	{4243216356U, "Den961ar_Blood_Mage"         },
	{4243456048U, "FSaturation"                 },
	{4244077197U, "Weapon5"                     },
	{4247185583U, "BShowInPartyPicker"          },
	{4249357382U, "Acceleration"                },
	{4251534318U, "Rouge"                       },
	{4252275623U, "RemoveSound"                 },
	{4252583751U, "Den800ar_Aeonar"             },
	{4252948175U, "Charyb"                      },
	{4253287381U, "ModelName"                   },
	{4253838853U, "AmbientBlue"                 },
	{4254176912U, "TopSpeed"                    },
	{4255138574U, "PositionZ"                   },
	{4260096281U, "KobeDragon"                  },
	{4262197476U, "Item6"                       },
	{4262380966U, "SpeedRotation"               },
	{4263868114U, "Comments"                    },
	{4267385544U, "StrIDName"                   },
	{4268196610U, "Ability0"                    },
	{4269998343U, "Resist11"                    },
	{4274162369U, "Weapon19"                    },
	{4274380637U, "TalentID"                    },
	{4276598643U, "Portrait"                    },
	{4281463324U, "Cli900ar_Fort_Exterior"      },
	{4281518266U, "DerivdDescStrRef"            },
	{4282681231U, "List"                        },
	{4286258897U, "Warrior2"                    },
	{4286400462U, "WandLevel"                   },
	{4289759885U, "ReactiveChance"              },
	{4290384881U, "Effect_Float0"               },
	{4290999015U, "TransitionTo"                },
	{4291291336U, "ActionCode"                  },
	{4294307347U, "SoundMap"                    },
	{4294575364U, "Orz570ar_Deeproad_Outskirts" },
	{4294639615U, "CameraOffset"                }
};

const char *findGDAHeader(uint32 hash) {
	const GDAHeaderHash *header = Common::binarySearch(kGDAHeaderHashes, ARRAYSIZE(kGDAHeaderHashes), hash);
	if (!header)
		return 0;

	return header->value;
}

} // End of namespace Aurora
