/*
 * test-unity-webapps-permissions.c
 * Copyright (C) Canonical LTD 2013
 * 
 * Author: Alexandre Abreu <alexandre.abreu@canonical.com>
 * 
 * unity-webapps is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * unity-webapps is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.";
 */

#include <glib.h>
#include <glib/gprintf.h>
#include <gio/gio.h>

#define G_SETTINGS_ENABLE_BACKEND
#include <gio/gsettingsbackend.h>

#include "unity-webapps-permissions.h"
#include "unity-webapps-debug.h"

static void
test_permissions_integration_allowed (void)
{
  g_assert (unity_webapps_permissions_is_integration_allowed());
}
static void
test_permissions_integration_allowed_set_get (void)
{
  unity_webapps_permissions_set_integration_allowed(TRUE);
  g_assert (unity_webapps_permissions_is_integration_allowed());

  unity_webapps_permissions_set_integration_allowed(FALSE);
  g_assert (!unity_webapps_permissions_is_integration_allowed());
}

static void
test_permissions_integration_allow_disallow_domain (void)
{
  const char *DOMAIN_NAME = "mydomain.com";

  g_assert (!unity_webapps_permissions_get_domain_allowed(DOMAIN_NAME));
  unity_webapps_permissions_allow_domain(DOMAIN_NAME);
  g_assert (unity_webapps_permissions_get_domain_allowed(DOMAIN_NAME));
  g_assert (!unity_webapps_permissions_get_domain_dontask(DOMAIN_NAME));

  unity_webapps_permissions_remove_domain_from_permissions(DOMAIN_NAME);
  g_assert (!unity_webapps_permissions_get_domain_allowed(DOMAIN_NAME));
  g_assert (!unity_webapps_permissions_get_domain_dontask(DOMAIN_NAME));
}

static void
test_permissions_integration_allow_domain_multiple_times (void)
{
  // make sure not added multiple times
  unity_webapps_permissions_allow_domain("mydomain.com");
  unity_webapps_permissions_allow_domain("mydomain.com");
  unity_webapps_permissions_allow_domain("mydomain.com");
  unity_webapps_permissions_allow_domain("mydomain.com");

  g_assert(0 == g_strcmp0(unity_webapps_permissions_get_all_domains(),
                          "{\"allowed\":[\"mydomain.com\"],\"dontask\":[]}")
           );
}

static void
test_permissions_integration_dontask_domain (void)
{
  const char *DOMAIN_NAME = "mydomain.com";

  g_assert (!unity_webapps_permissions_get_domain_dontask(DOMAIN_NAME));
  unity_webapps_permissions_dontask_domain(DOMAIN_NAME);
  g_assert (!unity_webapps_permissions_get_domain_allowed(DOMAIN_NAME));
  g_assert (unity_webapps_permissions_get_domain_dontask(DOMAIN_NAME));

  unity_webapps_permissions_remove_domain_from_permissions(DOMAIN_NAME);
  g_assert (!unity_webapps_permissions_get_domain_allowed(DOMAIN_NAME));
}

static void
test_permissions_integration_allowed_domain_added_to_dontask (void)
{
  const char *DOMAIN_NAME = "mydomain.com";

  g_assert (!unity_webapps_permissions_get_domain_dontask(DOMAIN_NAME));
  g_assert (!unity_webapps_permissions_get_domain_allowed(DOMAIN_NAME));
  unity_webapps_permissions_allow_domain("mydomain.com");
  g_assert (!unity_webapps_permissions_get_domain_dontask(DOMAIN_NAME));
  g_assert (unity_webapps_permissions_get_domain_allowed(DOMAIN_NAME));

  unity_webapps_permissions_dontask_domain(DOMAIN_NAME);
  g_assert (!unity_webapps_permissions_get_domain_allowed(DOMAIN_NAME));
  g_assert (unity_webapps_permissions_get_domain_dontask(DOMAIN_NAME));

  unity_webapps_permissions_remove_domain_from_permissions(DOMAIN_NAME);
  g_assert (!unity_webapps_permissions_get_domain_allowed(DOMAIN_NAME));
  g_assert (!unity_webapps_permissions_get_domain_dontask(DOMAIN_NAME));
}

int 
main (int argc, char **argv)
{
  g_type_init ();

  // remove giomodule warning
  g_io_extension_point_register(G_SETTINGS_BACKEND_EXTENSION_POINT_NAME);

  g_test_init (&argc, &argv, NULL);

  unity_webapps_permissions_tests_use_memory_backend(
      g_build_path("/",
                   g_get_current_dir(),
                   SCHEMADATADIR,
                   NULL));

  g_test_add_func("/Permissions/IntegrationAllowed/Basic",
                  test_permissions_integration_allowed);
  g_test_add_func("/Permissions/IntegrationAllowed/SetGet",
                  test_permissions_integration_allowed_set_get);
  g_test_add_func("/Permissions/AllowDomain/GetSet",
                  test_permissions_integration_allow_disallow_domain);
  g_test_add_func("/Permissions/AllowDomain/MultipleTimes",
                  test_permissions_integration_allow_domain_multiple_times);
  g_test_add_func("/Permissions/AllowDomain/DontAskDomain",
                  test_permissions_integration_dontask_domain);
  g_test_add_func("/Permissions/AllowDomain/AllowedDomainAddedToDontask",
                  test_permissions_integration_allowed_domain_added_to_dontask);

  return g_test_run ();
}
