/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 *  Copyright (C) 2006 Juernjakob Harder
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#   include <config.h>
#endif

#include <glib/gprintf.h>
#include <glib/gi18n-lib.h>
#include <gtk/gtk.h>

#include "tomoe-reading-search.h"
#include "tomoe-char-table.h"

enum {
    SELECTED_SIGNAL,
    LAST_SIGNAL
};

enum {
    PROP_0,
    PROP_TOMOE_CONTEXT
};

enum {
    TERMINATOR = -1,
    UTF8_COLUMN,
    CODE_POINT_COLUMN,
    STROKECOUNT_COLUMN,
    STROKECOUNT_TEXT_COLUMN,
    READING_COLUMN,
    CHAR_COLUMN,
    COLUMN_COUNT
};

typedef struct _TomoeReadingSearchPrivate	TomoeReadingSearchPrivate;
struct _TomoeReadingSearchPrivate
{
    TomoeContext *context;
    GtkListStore *result_store;
    GtkWidget    *input;
    GtkWidget    *min_strokes_label;
    GtkWidget    *max_strokes_label;
    GtkWidget    *min_strokes_spin;
    GtkWidget    *max_strokes_spin;
    GtkWidget    *range_check_button;
    GtkWidget    *expander;
    GtkWidget    *treeview;
};

#define TOMOE_READING_SEARCH_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), TOMOE_TYPE_READING_SEARCH, TomoeReadingSearchPrivate))

G_DEFINE_TYPE (TomoeReadingSearch, tomoe_reading_search, GTK_TYPE_TABLE)

static void dispose                             (GObject        *object);
static void set_property                        (GObject        *object,
                                                 guint           prop_id,
                                                 const GValue   *value,
                                                 GParamSpec     *pspec);
static void get_property                        (GObject        *object,
                                                 guint           prop_id,
                                                 GValue         *value,
                                                 GParamSpec     *pspec);

static void on_find_button_clicked              (GtkButton      *button,
                                                 gpointer        user_data);
static void on_clear_button_clicked             (GtkButton      *button,
                                                 gpointer        user_data);
static void on_input_entry_activate             (GtkEntry       *entry,
                                                 gpointer       *user_data);
static void on_min_strokes_spin_button_changed  (GtkSpinButton *spinbutton,
                                                 gpointer       user_data);
static void on_max_strokes_spin_button_changed  (GtkSpinButton *spinbutton,
                                                 gpointer       user_data);
static void on_specify_range_button_toggled     (GtkToggleButton *togglebutton,
                                                 gpointer       user_data);
static void on_result_view_button_release_event (GtkWidget      *widget,
                                                 GdkEventButton *event,
                                                 gpointer        user_data);

static guint reading_search_signals[LAST_SIGNAL] = { 0 };

GtkWidget *
tomoe_reading_search_new (TomoeContext *context)
{
    return GTK_WIDGET(g_object_new (TOMOE_TYPE_READING_SEARCH,
                                    "tomoe-context", context,
                                    NULL));
}

static void
tomoe_reading_search_class_init (TomoeReadingSearchClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
    GParamSpec *spec;

    gobject_class->dispose      = dispose;
    gobject_class->set_property = set_property;
    gobject_class->get_property = get_property;
    klass->selected             = NULL;

    reading_search_signals[SELECTED_SIGNAL] =
      g_signal_new ("selected",
  		  G_TYPE_FROM_CLASS (klass),
  		  G_SIGNAL_RUN_LAST,
  		  G_STRUCT_OFFSET (TomoeReadingSearchClass, selected),
  		  NULL, NULL,
  		  g_cclosure_marshal_VOID__VOID,
  		  G_TYPE_NONE, 0);

    spec = g_param_spec_object (
        "tomoe-context",
        N_("Tomoe context"),
        N_("A TomoeContext which stores handwriting dictionaries."),
        TOMOE_TYPE_CONTEXT, G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY);
    g_object_class_install_property (gobject_class, PROP_TOMOE_CONTEXT, spec);

    g_type_class_add_private (gobject_class, sizeof (TomoeReadingSearchPrivate));
}

static void
tomoe_reading_search_init (TomoeReadingSearch *page)
{
    TomoeReadingSearchPrivate *priv = TOMOE_READING_SEARCH_GET_PRIVATE (page);
    GtkWidget *hbox, *vbox, *expander;
    GtkWidget *input, *button, *list, *label, *spin, *check;
    GtkCellRenderer   *renderer;
    GtkTreeViewColumn *column;
    GtkTreeSelection *result_sel;
    GtkWidget *scrolled_window;
    GtkObject *adj;

    gtk_table_resize (GTK_TABLE (page), 1, 1);
    gtk_table_set_homogeneous (GTK_TABLE (page), FALSE);

    vbox = gtk_vbox_new (FALSE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (vbox), 8);
    gtk_table_attach_defaults (GTK_TABLE (page), vbox, 0, 1, 0, 1);

    /* input area */
    hbox = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
    gtk_widget_show (hbox);

    label = gtk_label_new (_("Reading:"));
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 4);
    gtk_widget_show (label);

    input = gtk_entry_new ();
    priv->input = input;
    gtk_box_pack_start (GTK_BOX (hbox), input, FALSE, FALSE, 4);
    g_signal_connect (G_OBJECT (input), "activate",
                      G_CALLBACK (on_input_entry_activate),
                      (gpointer) page);
    gtk_widget_show (input);

    button = gtk_button_new_from_stock (GTK_STOCK_FIND);
    gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 4);
    g_signal_connect (G_OBJECT (button), "clicked",
                      G_CALLBACK (on_find_button_clicked),
                      (gpointer) page);
    gtk_widget_show (button);

    button = gtk_button_new_from_stock (GTK_STOCK_CLEAR);
    gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 4);
    g_signal_connect (G_OBJECT (button), "clicked",
                      G_CALLBACK (on_clear_button_clicked),
                      (gpointer) page);
    gtk_widget_show (button);

    /* stroke count area */
    expander = gtk_expander_new_with_mnemonic (_("More options"));
    priv->expander = expander;
    gtk_box_pack_start (GTK_BOX (vbox), expander, FALSE, FALSE, 0);
    gtk_widget_show (expander);

    hbox = gtk_hbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (expander), hbox);
    gtk_widget_show (hbox);

    label = gtk_label_new (_("Stroke count:"));
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 4);
    gtk_widget_show (label);

    label = gtk_label_new (_("Min"));
    priv->min_strokes_label = label;
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 4);
    /*gtk_widget_show (label);*/

    adj = gtk_adjustment_new (1.0, 1.0, 200.0, 1.0, 5.0, 5.0);
    spin = gtk_spin_button_new (GTK_ADJUSTMENT (adj), 0.0, 0);
    priv->min_strokes_spin = spin;
    g_signal_connect (G_OBJECT (spin), "value-changed",
                      G_CALLBACK (on_min_strokes_spin_button_changed),
                      page);
    gtk_box_pack_start (GTK_BOX (hbox), spin, FALSE, FALSE, 4);
    gtk_widget_show (spin);

    label = gtk_label_new (_("Max"));
    priv->max_strokes_label = label;
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 4);
    /*gtk_widget_show (label);*/

    adj = gtk_adjustment_new (20.0, 1.0, 200.0, 1.0, 5.0, 5.0);
    spin = gtk_spin_button_new (GTK_ADJUSTMENT (adj), 0.0, 0);
    priv->max_strokes_spin = spin;
    g_signal_connect (G_OBJECT (spin), "value-changed",
                      G_CALLBACK (on_max_strokes_spin_button_changed),
                      page);
    gtk_box_pack_start (GTK_BOX (hbox), spin, FALSE, FALSE, 4);
    /*gtk_widget_show (spin);*/

    check = gtk_check_button_new_with_label (_("Specify range"));
    priv->range_check_button = check;
    g_signal_connect (G_OBJECT (check), "toggled",
                      G_CALLBACK (on_specify_range_button_toggled),
                      page);
    gtk_box_pack_start (GTK_BOX (hbox), check, FALSE, FALSE, 4);
    gtk_widget_show (check);

    /* result view */
    priv->result_store = gtk_list_store_new (COLUMN_COUNT,
                                             G_TYPE_STRING,
                                             G_TYPE_STRING,
                                             G_TYPE_INT,
                                             G_TYPE_STRING,
                                             G_TYPE_STRING,
                                             G_TYPE_OBJECT);

    list = gtk_tree_view_new_with_model (GTK_TREE_MODEL (priv->result_store));
    priv->treeview = list;
    result_sel = gtk_tree_view_get_selection (GTK_TREE_VIEW (list));
    gtk_tree_selection_set_mode (result_sel, GTK_SELECTION_SINGLE);
    
    scrolled_window = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window), 
                                    GTK_POLICY_AUTOMATIC,
                                    GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolled_window),
					 GTK_SHADOW_ETCHED_IN);
    gtk_container_add (GTK_CONTAINER (scrolled_window), list);
    gtk_box_pack_start (GTK_BOX (vbox), scrolled_window, TRUE, TRUE, 4);
    gtk_widget_show (scrolled_window);

    /* character column */
    renderer = gtk_cell_renderer_text_new ();
    column = gtk_tree_view_column_new_with_attributes (
        _("Character"), renderer,
        "text", UTF8_COLUMN,
        NULL);
    gtk_tree_view_column_set_sort_column_id (column, UTF8_COLUMN);
    gtk_tree_view_append_column (GTK_TREE_VIEW (list), column);

    renderer = gtk_cell_renderer_text_new ();
    column = gtk_tree_view_column_new_with_attributes (
        _("Code point"), renderer,
        "text", CODE_POINT_COLUMN,
        NULL);
    gtk_tree_view_column_set_sort_column_id (column, CODE_POINT_COLUMN);
    gtk_tree_view_append_column (GTK_TREE_VIEW (list), column);

    /* stroke count column */
    renderer = gtk_cell_renderer_text_new ();
    column = gtk_tree_view_column_new_with_attributes (
        _("Stroke count"), renderer,
        "text", STROKECOUNT_TEXT_COLUMN,
        NULL);
    gtk_tree_view_column_set_sort_column_id (column, STROKECOUNT_COLUMN);
    gtk_tree_view_append_column (GTK_TREE_VIEW (list), column);

    /* reading column */
    renderer = gtk_cell_renderer_text_new ();
    column = gtk_tree_view_column_new_with_attributes (
        _("Reading"), renderer,
        "text", READING_COLUMN,
        NULL);
    gtk_tree_view_column_set_sort_column_id (column, READING_COLUMN);
    gtk_tree_view_append_column (GTK_TREE_VIEW (list), column);
    gtk_widget_set_size_request (list, 300, 250);
    g_signal_connect (G_OBJECT (list), "button-release-event",
                      G_CALLBACK (on_result_view_button_release_event),
                      (gpointer) page);
    gtk_widget_show (list);

    gtk_widget_show (vbox);
}

static void
dispose (GObject *object)
{
    TomoeReadingSearchPrivate *priv = TOMOE_READING_SEARCH_GET_PRIVATE (object);

    if (priv->result_store) {
        g_object_unref (priv->result_store);
        priv->result_store = NULL;
    }
    if (priv->context) {
        g_object_unref (priv->context);
        priv->context = NULL;
    }

    if (G_OBJECT_CLASS(tomoe_reading_search_parent_class)->dispose)
        G_OBJECT_CLASS(tomoe_reading_search_parent_class)->dispose(object);
}

static void
set_property (GObject *object,
              guint prop_id,
              const GValue *value,
              GParamSpec *pspec)
{
    TomoeReadingSearch *page = TOMOE_READING_SEARCH (object);
    TomoeReadingSearchPrivate *priv = TOMOE_READING_SEARCH_GET_PRIVATE (page);

    switch (prop_id) {
    case PROP_TOMOE_CONTEXT:
    {
        TomoeContext *ctx = TOMOE_CONTEXT (g_value_get_object (value));
        if (priv->context)
            g_object_unref (priv->context);
        if (ctx)
            g_object_ref (ctx);
        priv->context = ctx;
        break;
    }
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }
}

static void
get_property (GObject *object,
              guint prop_id,
              GValue *value,
              GParamSpec *pspec)
{
    TomoeReadingSearch *page = TOMOE_READING_SEARCH (object);
    TomoeReadingSearchPrivate *priv = TOMOE_READING_SEARCH_GET_PRIVATE (page);

    switch (prop_id) {
    case PROP_TOMOE_CONTEXT:
        g_value_set_object (value, G_OBJECT (priv->context));
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }
}

static void
tomoe_reading_search_start_search (TomoeReadingSearch *page)
{
    GList *result, *list;
    const gchar *reading_text;
    gint min = 0, max = 0;
    GtkSpinButton *min_spin, *max_spin;
    TomoeQuery *query;
    TomoeReading *reading;
    TomoeReadingSearchPrivate *priv = TOMOE_READING_SEARCH_GET_PRIVATE (page);

    g_return_if_fail (GTK_IS_LIST_STORE (priv->result_store));
    g_return_if_fail (GTK_IS_ENTRY (priv->input));

    gtk_tree_view_set_model (GTK_TREE_VIEW (priv->treeview), NULL);
    gtk_list_store_clear (priv->result_store);

    query = tomoe_query_new ();

    reading_text = gtk_entry_get_text (GTK_ENTRY (priv->input));
    if (reading_text && *reading_text) {
        reading = tomoe_reading_new (TOMOE_READING_UNKNOWN, reading_text);
        tomoe_query_add_reading (query, reading);
        g_object_unref (reading);
    }

    min_spin = GTK_SPIN_BUTTON (priv->min_strokes_spin);
    max_spin = GTK_SPIN_BUTTON (priv->max_strokes_spin);

    if (gtk_expander_get_expanded (GTK_EXPANDER (priv->expander))) {
        min = gtk_spin_button_get_value (min_spin);
        tomoe_query_set_min_n_strokes (query, min);
    }

    if (gtk_expander_get_expanded (GTK_EXPANDER (priv->expander))) {
        GtkToggleButton *toggle = GTK_TOGGLE_BUTTON (priv->range_check_button);
        if (gtk_toggle_button_get_active (toggle))
            max = gtk_spin_button_get_value (max_spin);
        else
            max = gtk_spin_button_get_value (min_spin);
        tomoe_query_set_max_n_strokes (query, max);
    }

    if ((!reading_text || !*reading_text) && !min && !max) {
        g_object_unref (G_OBJECT (query));
        return;
    }

    result = tomoe_context_search (priv->context, query);
    g_object_unref (query);

    for (list = result; list; list = g_list_next (list)) {
        TomoeCandidate *cand = TOMOE_CANDIDATE (list->data);
        TomoeChar *c = tomoe_candidate_get_char (cand);
        const GList *readings = tomoe_char_get_readings (c);
        gchar *strokes_text, *readings_text;
        gchar *utf8;
        gint i, utf8_len;
        GString *code_point_text;
        gint strokes = 0;
        GtkTreeIter iter;

        code_point_text = g_string_new ("");
        utf8 = (gchar *)tomoe_char_get_utf8 (c);
        utf8_len = g_utf8_strlen (utf8, -1);
        for (i = 0; i < utf8_len; i++) {
            char *format;
            gunichar c = g_utf8_get_char (utf8);
            if (c > 0xFFFFF)
                format = "U+%X6";
            else if (c > 0xFFFF)
                format = "U+%X5";
            else
                format = "U+%X4";
            g_string_append_printf (code_point_text, format, c);
            utf8 = g_utf8_next_char (utf8);
        }
        if (code_point_text->len > 0)
            g_string_erase (code_point_text, code_point_text->len - 1, 1);

        strokes = tomoe_char_get_n_strokes (c);
        if (strokes == 0 && tomoe_char_get_writing (c))
            strokes = tomoe_writing_get_n_strokes (tomoe_char_get_writing (c));

        if (strokes > 0)
            strokes_text = g_strdup_printf ("%d", strokes);
        else
            strokes_text = g_strdup ("?");

        if (readings) {
            gchar **str_array;
            guint reading_num, i;
            reading_num = g_list_length ((GList *) readings);
            str_array = g_new0 (gchar *, reading_num + 1);
            str_array[reading_num] = NULL;
            for (i = 0; i < reading_num; i++) {
    	        TomoeReading *reading
                    = TOMOE_READING (g_list_nth_data ((GList *) readings, i));
                str_array[i]  = (gchar *) tomoe_reading_get_reading (reading);
            }
            readings_text = g_strjoinv (" ", str_array);
            g_free (str_array);
        } else {
            readings_text = g_strdup ("?");
        }
        gtk_list_store_append (priv->result_store, &iter);
        gtk_list_store_set (priv->result_store, &iter,
                            UTF8_COLUMN,             tomoe_char_get_utf8 (c),
                            CODE_POINT_COLUMN,       code_point_text->str,
                            STROKECOUNT_COLUMN,      strokes,
                            STROKECOUNT_TEXT_COLUMN, strokes_text,
                            READING_COLUMN,          readings_text,
                            CHAR_COLUMN,             c,
                            TERMINATOR);

        g_string_free (code_point_text, TRUE);
        g_free (readings_text);
        g_free (strokes_text);
    }

    g_list_foreach (result, (GFunc) g_object_unref, NULL);
    g_list_free (result);

    gtk_tree_view_set_model (GTK_TREE_VIEW (priv->treeview),
                             GTK_TREE_MODEL (priv->result_store));
}

static void
tomoe_reading_search_clear (TomoeReadingSearch *page)
{

    TomoeReadingSearchPrivate *priv = TOMOE_READING_SEARCH_GET_PRIVATE (page);

    g_return_if_fail (GTK_IS_LIST_STORE (priv->result_store));
    g_return_if_fail (GTK_IS_ENTRY (priv->input));

    gtk_entry_set_text (GTK_ENTRY (priv->input), "");
    gtk_tree_view_set_model (GTK_TREE_VIEW (priv->treeview), NULL);
    gtk_list_store_clear (priv->result_store);
    gtk_tree_view_set_model (GTK_TREE_VIEW (priv->treeview),
                             GTK_TREE_MODEL (priv->result_store));
}

const gchar *
tomoe_reading_search_get_selected_char (TomoeReadingSearch *page)
{
    TomoeReadingSearchPrivate *priv;
    TomoeChar *c;

    g_return_val_if_fail (TOMOE_IS_READING_SEARCH (page), NULL);
    priv = TOMOE_READING_SEARCH_GET_PRIVATE (page);

    c = tomoe_reading_search_get_selected_tomoe_char (page);
    if (!c) return NULL;

    return tomoe_char_get_utf8 (c);
}

TomoeChar *
tomoe_reading_search_get_selected_tomoe_char (TomoeReadingSearch *page)
{
    TomoeReadingSearchPrivate *priv;
    GtkTreeModel *treemodel;
    GtkTreePath *treepath = NULL;
    GtkTreeIter iter;
    TomoeChar *c = NULL;

    g_return_val_if_fail (TOMOE_IS_READING_SEARCH (page), NULL);
    priv = TOMOE_READING_SEARCH_GET_PRIVATE (page);

    treemodel = GTK_TREE_MODEL (priv->result_store);

    gtk_tree_view_get_cursor (GTK_TREE_VIEW (priv->treeview), &treepath, NULL);
    if (!treepath)
        return NULL;
    if (!gtk_tree_model_get_iter (treemodel, &iter, treepath))
        return NULL;

    gtk_tree_model_get (treemodel, &iter,
                        CHAR_COLUMN, &c,
                        TERMINATOR);

    /* reference count should be increased by user */
    if (c) g_object_unref (c);

    gtk_tree_path_free (treepath);

    return c;
}

static void
on_input_entry_activate (GtkEntry *entry, gpointer *user_data)
{
    TomoeReadingSearch *page = TOMOE_READING_SEARCH (user_data);

    g_return_if_fail (TOMOE_IS_READING_SEARCH (page));

    tomoe_reading_search_start_search (page);
}

static void
on_min_strokes_spin_button_changed (GtkSpinButton *spinbutton,
                                    gpointer       user_data)
{
    gint min, max;

    TomoeReadingSearch *page = TOMOE_READING_SEARCH (user_data);
    TomoeReadingSearchPrivate *priv = TOMOE_READING_SEARCH_GET_PRIVATE (page);

    min = gtk_spin_button_get_value (GTK_SPIN_BUTTON (priv->min_strokes_spin));
    max = gtk_spin_button_get_value (GTK_SPIN_BUTTON (priv->max_strokes_spin));

    if (min > max)
        gtk_spin_button_set_value (GTK_SPIN_BUTTON (priv->max_strokes_spin),
                                   min);
}

static void
on_max_strokes_spin_button_changed (GtkSpinButton *spinbutton,
                                    gpointer       user_data)
{
    gint min, max;

    TomoeReadingSearch *page = TOMOE_READING_SEARCH (user_data);
    TomoeReadingSearchPrivate *priv = TOMOE_READING_SEARCH_GET_PRIVATE (page);

    min = gtk_spin_button_get_value (GTK_SPIN_BUTTON (priv->min_strokes_spin));
    max = gtk_spin_button_get_value (GTK_SPIN_BUTTON (priv->max_strokes_spin));

    if (max < min)
        gtk_spin_button_set_value (GTK_SPIN_BUTTON (priv->min_strokes_spin),
                                   max);
}

static void
on_specify_range_button_toggled (GtkToggleButton *togglebutton,
                                 gpointer user_data)
{
    TomoeReadingSearch *page = TOMOE_READING_SEARCH (user_data);
    TomoeReadingSearchPrivate *priv = TOMOE_READING_SEARCH_GET_PRIVATE (page);

    if (gtk_toggle_button_get_active (togglebutton)) {
        gtk_widget_show (priv->min_strokes_label);
        gtk_widget_show (priv->max_strokes_label);
        gtk_widget_show (priv->max_strokes_spin);
    } else {
        gtk_widget_hide (priv->min_strokes_label);
        gtk_widget_hide (priv->max_strokes_label);
        gtk_widget_hide (priv->max_strokes_spin);
    }
}

static void
on_find_button_clicked (GtkButton *button, gpointer user_data)
{
    TomoeReadingSearch *page = TOMOE_READING_SEARCH (user_data);

    g_return_if_fail (TOMOE_IS_READING_SEARCH (page));

    tomoe_reading_search_start_search (page);
}

static void
on_clear_button_clicked (GtkButton *button, gpointer user_data)
{
    TomoeReadingSearch *page = TOMOE_READING_SEARCH (user_data);

    g_return_if_fail (TOMOE_IS_READING_SEARCH (page));

    tomoe_reading_search_clear (page);
}

static void
on_result_view_button_release_event (GtkWidget *widget,
                                     GdkEventButton *event,
                                     gpointer user_data)
{
    TomoeReadingSearch *page = TOMOE_READING_SEARCH (user_data);
    TomoeReadingSearchPrivate *priv = TOMOE_READING_SEARCH_GET_PRIVATE (page);
    gboolean exist;

    exist = gtk_tree_view_get_path_at_pos (GTK_TREE_VIEW (priv->treeview),
                                           event->x, event->y,
                                           NULL, NULL, NULL, NULL);
    if (!exist) return;

    if (tomoe_reading_search_get_selected_tomoe_char (page))
        g_signal_emit (G_OBJECT (page),
                       reading_search_signals[SELECTED_SIGNAL], 0);
}
/*
 * vi:ts=4:nowrap:ai:expandtab
 */
