/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
 /*
 *  Copyright (C) 2006 Juernjakob Harder
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#   include <config.h>
#endif

#include <glib/gi18n-lib.h>
#include <gtk/gtk.h>
#include <tomoe.h>

#include "tomoe-edit-strokes.h"
#include "tomoe-canvas.h"

enum {
    LAST_SIGNAL
};

enum
{
    PROP_0,
    PROP_TOMOE_CHAR
};

typedef struct _TomoeEditStrokesPrivate	TomoeEditStrokesPrivate;
struct _TomoeEditStrokesPrivate
{
    GtkWidget    *canvas;
    GtkWidget    *go_back_button;
    GtkWidget    *clear_button;
    GtkWidget    *normalize_button;
    TomoeChar    *character;
};

#define TOMOE_EDIT_STROKES_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), TOMOE_TYPE_EDIT_STROKES, TomoeEditStrokesPrivate))

G_DEFINE_TYPE (TomoeEditStrokes, tomoe_edit_strokes, GTK_TYPE_DIALOG)

static void dispose      (GObject       *object);
static void set_property (GObject       *object,
                          guint          prop_id,
                          const GValue  *value,
                          GParamSpec    *pspec);
static void get_property (GObject       *object,
                          guint          prop_id,
                          GValue        *value,
                          GParamSpec    *pspec);
static void tomoe_edit_strokes_set_sensitive (TomoeEditStrokes         *page);

static void on_ok_button_clicked          (GtkButton           *button,
                                           gpointer             user_data);
static void on_cancel_button_clicked      (GtkButton           *button,
                                           gpointer             user_data);
static void on_go_back_button_clicked     (GtkButton           *button,
                                           gpointer             user_data);
static void on_clear_button_clicked       (GtkButton           *button,
                                           gpointer             user_data);
static void on_normalize_button_clicked   (GtkButton           *button,
                                           gpointer             user_data);
static void on_canvas_stroke_added        (TomoeCanvas          *canvas,
                                           gpointer             user_data);

GtkWidget *
tomoe_edit_strokes_new (TomoeChar *chr)
{
    GtkWidget *w = GTK_WIDGET(g_object_new (TOMOE_TYPE_EDIT_STROKES,
                                            "tomoe-char", chr,
                                            NULL));
    return w;
}


static void
tomoe_edit_strokes_class_init (TomoeEditStrokesClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

    gobject_class->dispose      = dispose;
    gobject_class->set_property = set_property;
    gobject_class->get_property = get_property;
    g_object_class_install_property (gobject_class,
                                     PROP_TOMOE_CHAR,
                                     g_param_spec_object ("tomoe-char",
                                         N_("TomoeChar object"),
                                         N_("The Object of TomoeChar"),
                                         TOMOE_TYPE_CHAR,
                                         G_PARAM_READWRITE |
                                         G_PARAM_CONSTRUCT_ONLY));
    g_type_class_add_private (gobject_class, sizeof (TomoeEditStrokesPrivate));
}

static void
set_property (GObject      *object,
              guint         prop_id,
              const GValue *value,
              GParamSpec   *pspec)
{
    TomoeEditStrokesPrivate *priv = TOMOE_EDIT_STROKES_GET_PRIVATE (object);

    switch (prop_id) {
        case PROP_TOMOE_CHAR:
            priv->character = g_object_ref (g_value_get_object (value));
            break;

        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
            break;
    }
}

static void
get_property (GObject    *object,
              guint       prop_id,
              GValue     *value,
              GParamSpec *pspec)
{
    TomoeEditStrokesPrivate *priv = TOMOE_EDIT_STROKES_GET_PRIVATE (object);

    switch (prop_id) {
        case PROP_TOMOE_CHAR:
            g_value_set_object (value, priv->character);
            break;

        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
            break;
    }
}

static void
dispose (GObject *object)
{
    TomoeEditStrokesPrivate *priv = TOMOE_EDIT_STROKES_GET_PRIVATE (object);

    if (priv->character)
        g_object_unref (priv->character);
    priv->character = NULL;

    if (G_OBJECT_CLASS(tomoe_edit_strokes_parent_class)->dispose)
        G_OBJECT_CLASS(tomoe_edit_strokes_parent_class)->dispose(object);
}

static void
tomoe_edit_strokes_init (TomoeEditStrokes *dialog)
{
    GtkWidget *vbox, *hbox, *button, *alignment, *canvas, *frame;
    TomoeEditStrokesPrivate *priv = TOMOE_EDIT_STROKES_GET_PRIVATE (dialog);

    gtk_window_set_title (GTK_WINDOW (dialog),
                          _("Edit Strokes"));

    gtk_window_set_resizable (GTK_WINDOW (dialog), FALSE);
    gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
    gtk_container_set_border_width (GTK_CONTAINER (dialog), 5);
    gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_DELETE_EVENT);
    gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);

    hbox = gtk_hbox_new (FALSE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (hbox), 8);
    gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), hbox);
    gtk_widget_show (hbox);

    frame = gtk_frame_new (NULL);
    gtk_box_pack_start (GTK_BOX (hbox), frame, TRUE, TRUE, 4);
    gtk_widget_show (frame);

    canvas = tomoe_canvas_new ();
    priv->canvas = canvas;
    gtk_container_add (GTK_CONTAINER (frame), canvas);
    gtk_widget_set_size_request (canvas, 300, 300);
    g_signal_connect (G_OBJECT (canvas), "stroke-added",
                      G_CALLBACK (on_canvas_stroke_added), (gpointer) dialog);
    gtk_widget_show (canvas);

    /* button area */
    alignment = gtk_alignment_new (0.5, 0.5, 1.0, 1.0);
    gtk_alignment_set_padding (GTK_ALIGNMENT (alignment), 0, 0, 8, 0);
    gtk_box_pack_start (GTK_BOX (hbox), alignment, FALSE, FALSE, 0);
    gtk_widget_show (alignment);

    vbox = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (alignment), vbox);
    gtk_widget_show (vbox);

    button = gtk_button_new_from_stock (GTK_STOCK_OK);
    gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 4);
    g_signal_connect (G_OBJECT (button), "clicked",
                      G_CALLBACK (on_ok_button_clicked),
                      (gpointer) dialog);
    gtk_widget_show (button);

    button = gtk_button_new_from_stock (GTK_STOCK_CANCEL);
    gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 4);
    g_signal_connect (G_OBJECT (button), "clicked",
                      G_CALLBACK (on_cancel_button_clicked),
                      (gpointer) dialog);
    gtk_widget_show (button);

    button = gtk_button_new_from_stock (GTK_STOCK_GO_BACK);
    priv->go_back_button = button;
    gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 4);
    g_signal_connect (G_OBJECT (button), "clicked",
                      G_CALLBACK (on_go_back_button_clicked),
                      (gpointer) dialog);
    gtk_widget_show (button);

    button = gtk_button_new_from_stock (GTK_STOCK_CLEAR);
    priv->clear_button = button;
    gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 4);
    g_signal_connect (G_OBJECT (button), "clicked",
                      G_CALLBACK (on_clear_button_clicked),
                      (gpointer) dialog);
    gtk_widget_show (button);

    button = gtk_button_new_from_stock (GTK_STOCK_ZOOM_FIT);
    priv->normalize_button = button;
    gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 4);
    g_signal_connect (G_OBJECT (button), "clicked",
                      G_CALLBACK (on_normalize_button_clicked),
                      (gpointer) dialog);
    gtk_widget_show (button);

    tomoe_edit_strokes_set_sensitive (dialog);
}

static void
tomoe_edit_strokes_set_sensitive (TomoeEditStrokes *dialog)
{
    TomoeEditStrokesPrivate *priv = TOMOE_EDIT_STROKES_GET_PRIVATE (dialog);
    gboolean editable = tomoe_canvas_has_stroke (TOMOE_CANVAS (priv->canvas));

    gtk_widget_set_sensitive (priv->go_back_button,   editable);
    gtk_widget_set_sensitive (priv->clear_button,     editable);
    gtk_widget_set_sensitive (priv->normalize_button, editable);
}

static void
on_ok_button_clicked (GtkButton *button, gpointer user_data)
{
    TomoeEditStrokes *dialog = TOMOE_EDIT_STROKES (user_data);
    TomoeEditStrokesPrivate *priv = TOMOE_EDIT_STROKES_GET_PRIVATE (dialog);
    TomoeWriting *writing;

    g_return_if_fail (TOMOE_IS_EDIT_STROKES (dialog));
    g_return_if_fail (TOMOE_IS_CANVAS (priv->canvas));

    writing = tomoe_canvas_get_writing (TOMOE_CANVAS (priv->canvas));
    tomoe_char_set_writing (priv->character, writing);

    gtk_dialog_response (GTK_DIALOG (dialog), GTK_RESPONSE_OK);
}

static void
on_cancel_button_clicked (GtkButton *button, gpointer user_data)
{
    GtkDialog *dlg = GTK_DIALOG (user_data);

    g_return_if_fail (GTK_IS_DIALOG (dlg));

    gtk_dialog_response (dlg, GTK_RESPONSE_CANCEL);
}

static void
on_go_back_button_clicked (GtkButton *button, gpointer user_data)
{
    TomoeEditStrokes *dialog = TOMOE_EDIT_STROKES (user_data);
    TomoeEditStrokesPrivate *priv = TOMOE_EDIT_STROKES_GET_PRIVATE (dialog);

    g_return_if_fail (TOMOE_IS_EDIT_STROKES (dialog));
    g_return_if_fail (TOMOE_IS_CANVAS (priv->canvas));

    tomoe_canvas_revert_stroke (TOMOE_CANVAS (priv->canvas));
    tomoe_canvas_find (TOMOE_CANVAS (priv->canvas));
    tomoe_edit_strokes_set_sensitive (dialog);
}

static void
on_clear_button_clicked (GtkButton *button, gpointer user_data)
{
    TomoeEditStrokes *dialog = TOMOE_EDIT_STROKES (user_data);
    TomoeEditStrokesPrivate *priv = TOMOE_EDIT_STROKES_GET_PRIVATE (dialog);

    g_return_if_fail (TOMOE_IS_EDIT_STROKES (dialog));
    g_return_if_fail (TOMOE_IS_CANVAS (priv->canvas));

    tomoe_canvas_clear (TOMOE_CANVAS (priv->canvas));
    tomoe_edit_strokes_set_sensitive (dialog);
}

static void
on_canvas_stroke_added (TomoeCanvas *canvas, gpointer user_data)
{
    TomoeEditStrokes *page = TOMOE_EDIT_STROKES (user_data);
    tomoe_edit_strokes_set_sensitive (page);
}

static void
on_normalize_button_clicked (GtkButton *button, gpointer user_data)
{
    TomoeEditStrokes *dialog = TOMOE_EDIT_STROKES (user_data);
    TomoeEditStrokesPrivate *priv = TOMOE_EDIT_STROKES_GET_PRIVATE (dialog);

    g_return_if_fail (TOMOE_EDIT_STROKES (dialog));
    g_return_if_fail (TOMOE_IS_CANVAS (priv->canvas));

    tomoe_canvas_normalize (TOMOE_CANVAS (priv->canvas));
}

/*
 * vi:ts=4:nowrap:ai:expandtab
 */
