/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <gdk/gdkx.h>
#include <X11/Xlib.h>

#include <libxfce4util/debug.h>

#include "gtktoxevent.h"

static GtkToXEventFilterStack *filterstack = NULL;
static GtkWidget *gtk_win = NULL;
static GdkWindow *event_win = NULL;

static GtkToXEventFilterStatus
default_event_filter (XEvent * xevent, gpointer data)
{
    switch (xevent->type)
    {
	case KeyPress:
	    TRACE ("Unhandled KeyPress event");
	    break;
	case KeyRelease:
	    TRACE ("Unhandled KeyRelease event");
	    break;
	case ButtonPress:
	    TRACE ("Unhandled ButtonPress event");
	    break;
	case ButtonRelease:
	    TRACE ("Unhandled ButtonRelease event");
	    break;
	case MotionNotify:
	    TRACE ("Unhandled MotionNotify event");
	    break;
	case EnterNotify:
	    TRACE ("Unhandled EnterNotify event");
	    break;
	case LeaveNotify:
	    TRACE ("Unhandled LeaveNotify event");
	    break;
	case FocusIn:
	    TRACE ("Unhandled FocusIn event");
	    break;
	case FocusOut:
	    TRACE ("Unhandled FocusOut event");
	    break;
	case KeymapNotify:
	    TRACE ("Unhandled KeymapNotify event");
	    break;
	case Expose:
	    TRACE ("Unhandled Expose event");
	    break;
	case GraphicsExpose:
	    TRACE ("Unhandled GraphicsExpose event");
	    break;
	case NoExpose:
	    TRACE ("Unhandled NoExpose event");
	    break;
	case VisibilityNotify:
	    TRACE ("Unhandled VisibilityNotify event");
	    break;
	case DestroyNotify:
	    TRACE ("Unhandled DestroyNotify event");
	    break;
	case UnmapNotify:
	    TRACE ("Unhandled UnmapNotify event");
	    break;
	case MapNotify:
	    TRACE ("Unhandled MapNotify event");
	    break;
	case MapRequest:
	    TRACE ("Unhandled MapRequest event");
	    break;
	case ReparentNotify:
	    TRACE ("Unhandled ReparentNotify event");
	    break;
	case ConfigureNotify:
	    TRACE ("Unhandled ConfigureNotify event");
	    break;
	case ConfigureRequest:
	    TRACE ("Unhandled ConfigureRequest event");
	    break;
	case GravityNotify:
	    TRACE ("Unhandled GravityNotify event");
	    break;
	case ResizeRequest:
	    TRACE ("Unhandled ResizeRequest event");
	    break;
	case CirculateNotify:
	    TRACE ("Unhandled CirculateNotify event");
	    break;
	case CirculateRequest:
	    TRACE ("Unhandled CirculateRequest event");
	    break;
	case PropertyNotify:
	    TRACE ("Unhandled PropertyNotify event");
	    break;
	case SelectionClear:
	    TRACE ("Unhandled SelectionClear event");
	    break;
	case SelectionRequest:
	    TRACE ("Unhandled SelectionRequest event");
	    break;
	case SelectionNotify:
	    TRACE ("Unhandled SelectionNotify event");
	    break;
	case ColormapNotify:
	    TRACE ("Unhandled ColormapNotify event");
	    break;
	case MappingNotify:
	    TRACE ("Unhandled MappingNotify event");
	    break;
	default:
	    TRACE ("Unhandled Unknown event");
	    break;
    }
    /* This is supposed to be the default fallback event handler, so we return XEV_FILTER_STOP since we have "treated" the event */
    return XEV_FILTER_STOP;
}

static GdkFilterReturn
gdkXEventFilter (GdkXEvent * gdk_xevent, GdkEvent * event, gpointer data)
{
    XEvent *xevent = (XEvent *) gdk_xevent;
    GtkToXEventFilterStatus loop = XEV_FILTER_CONTINUE;
    GtkToXEventFilterStack *filterelt = filterstack;

    g_return_val_if_fail (filterelt != NULL, GDK_FILTER_CONTINUE);

    while ((filterelt) && (loop == XEV_FILTER_CONTINUE))
    {
	GtkToXEventFilterStack *filterelt_next = filterelt->next;
	loop = (*filterelt->filter) (xevent, filterelt->data);
	filterelt = filterelt_next;
    }
    return GDK_FILTER_CONTINUE;
}

GtkToXEventFilterStack *
pushEventFilter (GtkToXEventFilter filter, gpointer data)
{
    g_assert (filter != NULL);
    if (filterstack)
    {
	GtkToXEventFilterStack *newfilterstack =
	    (GtkToXEventFilterStack *) g_new (GtkToXEventFilterStack, 1);
	newfilterstack->filter = filter;
	newfilterstack->data = data;
	newfilterstack->next = filterstack;
	filterstack = newfilterstack;
    }
    else
    {
	filterstack =
	    (GtkToXEventFilterStack *) g_new (GtkToXEventFilterStack, 1);
	filterstack->filter = filter;
	filterstack->data = data;
	filterstack->next = NULL;
    }
    return (filterstack);
}

GtkToXEventFilterStack *
popEventFilter (void)
{
    GtkToXEventFilterStack *oldfilterstack = filterstack;
    g_return_val_if_fail (filterstack != NULL, NULL);
    filterstack = oldfilterstack->next;
    g_free (oldfilterstack);
    return (filterstack);
}

GtkToXEventFilterStack *
initEventFilter (long event_mask, gpointer data, const gchar * widget_name)
{
    XWindowAttributes attribs;

    gdk_error_trap_push ();
    gdk_x11_grab_server ();

    pushEventFilter (default_event_filter, data);
    event_win = gdk_window_foreign_new (GDK_ROOT_WINDOW ());
    gdk_window_add_filter (NULL, gdkXEventFilter, NULL);

    XGetWindowAttributes (GDK_DISPLAY (), GDK_ROOT_WINDOW (), &attribs);
    XSelectInput (GDK_DISPLAY (), GDK_ROOT_WINDOW (),
		  attribs.your_event_mask | event_mask);
    gdk_x11_ungrab_server ();

    if (gdk_error_trap_pop ())
    {
	/* Something wrong, probably another window manager running */
	return (NULL);
    }
    /* Create a GTK window so that we are just like any other GTK application */
    gtk_win = gtk_window_new (GTK_WINDOW_POPUP);
    gtk_window_resize (GTK_WINDOW (gtk_win), 5, 5);
    gtk_window_move (GTK_WINDOW (gtk_win), -1000, -1000);
    if (widget_name)
    {
	gtk_widget_set_name (gtk_win, widget_name);
    }
    gtk_widget_show_now (gtk_win);
    gdk_window_set_user_data (event_win, gtk_win);
    gdk_flush ();

    return (filterstack);
}

void
closeEventFilter (void)
{
    GtkToXEventFilterStack *filterelt = filterstack;
    while ((filterelt = popEventFilter ()));
    gdk_window_remove_filter (NULL, gdkXEventFilter, NULL);
    gdk_window_set_user_data (event_win, NULL);
    gtk_widget_destroy (gtk_win);
    gdk_flush ();
}

GtkWidget *
getDefaultGtkWidget (void)
{
    return (GTK_WIDGET (gtk_win));
}

Window
getDefaultXWindow (void)
{
    g_return_val_if_fail (gtk_win != NULL, None);
    return (GDK_WINDOW_XWINDOW (gtk_win->window));
}

GdkWindow *
getGdkEventWindow (void)
{
    return (event_win);
}

GdkWindow *
getDefaultGdkWindow (void)
{
    g_return_val_if_fail (gtk_win != NULL, NULL);
    return ((GdkWindow *) gtk_win->window);
}
