/*****************************************************************************\
    hpijs97.cpp : HP Inkjet Server (0.97 backward compatable only)

    Copyright (c) 1996 - 2002, Hewlett-Packard Co.
    All rights reserved.

    Redistribution and use in source and binary forms, with or without
    modification, are permitted provided that the following conditions
    are met:
    1. Redistributions of source code must retain the above copyright
       notice, this list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright
       notice, this list of conditions and the following disclaimer in the
       documentation and/or other materials provided with the distribution.
    3. Neither the name of the Hewlett-Packard nor the names of its
       contributors may be used to endorse or promote products derived
       from this software without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
    IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
    OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
    IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
    INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
    NOT LIMITED TO, PATENT INFRINGEMENT; PROCUREMENT OF SUBSTITUTE GOODS OR
    SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
    HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
    STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
    IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
    POSSIBILITY OF SUCH DAMAGE.
\*****************************************************************************/

#include <sys/stat.h>
#include <sys/shm.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <syslog.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <limits.h>
#include "header.h"
#include "ijs.h"
#include "ijs_server.h"
#include "services.h"
#include "hpijs97.h"

/* Get packet from the client. */
int hpijs_get_pk(PK * pk, int fd)
{
   return read(fd, pk, PIPE_BUF - 1);
}

/* Write packet to the client. */
int hpijs_put_pk(PK * pk, int fd)
{
   return write(fd, pk, sizeof(PK));
}

/* Init server discriptor. */
int hpijs_init_sd(SRVD * sd)
{
   sd->fds2c = -1;
   sd->fdc2s = -1;
   sd->shmid = -1;
   sd->shmbuf = NULL;
   sd->srv_to_client = NULL;
   sd->client_to_srv = NULL;
   return 0;
}

/* Find out what printer models are supported. */
void GetSupport(DEV *models, int *maxModel)
{
   unsigned int index = 0;
   int i = 0;

   UXServices *pSS = new UXServices();
   if (pSS->constructor_error != NO_ERROR)
      return;

   PrintContext *pPC = new PrintContext(pSS);
   if (pPC->constructor_error != NO_ERROR)
      return;

   PRINTER_TYPE p = pPC->EnumDevices(index);
   while (p != UNSUPPORTED)
   {
      models[i].model = p;
      models[i].name = pPC->PrintertypeToString(p);
      i++;
      p = pPC->EnumDevices(index);
   }

   *maxModel = i;

   delete pPC;
   delete pSS;
}

int MapModel(DEV *models, int maxModel, const char *nam)
{
   int i;
   int model = -1;

   /* Find model number from our enumerated device table. */
   for (i = 0; i < maxModel; i++)
   {
      if (strcmp(nam, models[i].name) == 0)
      {
         model = models[i].model;
         break;
      }
   }
   return model;
}

int hpijs97()
{
   UXServices *pSS = NULL;
   PrintContext *pPC = NULL;
   Job *pJob = NULL;
   SRVD sd;
   PK pk;
   DEV models[MAX_PRINTER_TYPE];
   PAPER_SIZE paper;
   int max, width, outwidth, firstRaster = 1;
   int status = EXIT_FAILURE;
   int res = 0;
   int model, maxModel = 0;

   hpijs_init_sd(&sd);

   sd.srv_to_client = getenv("SRV_TO_CLIENT");
   sd.client_to_srv = getenv("CLIENT_TO_SRV");

   if (sd.srv_to_client == NULL)
   {
      fprintf(stderr, "\nHewlett-Packard Co. Inkjet Server %s\n", VERSION);
      fprintf(stderr, "Copyright (c) 2001-2002, Hewlett-Packard Co.\n");
      fprintf(stderr, "invalid usage\n");
      return(status);
   }

   /* Open fifos. */
   if ((sd.fds2c = open(sd.srv_to_client, O_WRONLY)) < 0)
   {
      bug("unable to open fifo %d: %m\n", sd.srv_to_client);
      goto BUGOUT;
   }
   if ((sd.fdc2s = open(sd.client_to_srv, O_RDONLY)) < 0)
   {
      bug("unable to open fifo %d: %m\n", sd.client_to_srv);
      goto BUGOUT;
   }

   pSS = new UXServices();
   if (pSS->constructor_error != NO_ERROR)
   {
      bug("unable to open Services object err=%d\n", pSS->constructor_error);
      goto BUGOUT;
   }

#ifdef CAPTURE
   if ((pSS->InitScript("/tmp/capout", TRUE)) != NO_ERROR)
      bug("unable to init capture");
#endif

   pPC = new PrintContext(pSS, 0, 0);
   if (pPC->constructor_error != NO_ERROR)
   {
      bug("unable to open PrintContext object err=%d\n", pPC->constructor_error);
      goto BUGOUT;
   }

   GetSupport(models, &maxModel);
   pSS->OutputPath = STDOUT_FILENO;   /* set prn_stream as output of SS::ToDevice */

   while (1)
   {
      if (hpijs_get_pk(&pk, sd.fdc2s) < 0)
      {
         bug("unable to read fifo: %m\n");
         goto BUGOUT;
      }

      switch (pk.cmd)
      {
      case SND_RASTER:
         if (firstRaster)
         {
            firstRaster = 0;
            pJob = new Job(pPC);
            if (pJob->constructor_error != NO_ERROR)
            {
               bug("unable to create Job object err=%d\n",
               pJob->constructor_error);
               goto BUGOUT;
            }
         }
         pJob->SendRasters((unsigned char *) sd.shmbuf);
         break;
      case SND_NULL_RASTER:
         if (firstRaster)
         {
            firstRaster = 0;
            pJob = new Job(pPC);
            if (pJob->constructor_error != NO_ERROR)
            {
               bug("unable to create Job object err=%d\n",
               pJob->constructor_error);
               goto BUGOUT;
            }
         }
         pJob->SendRasters((unsigned char *) 0);
         break;
      case NEW_PAGE:
         pJob->NewPage();
         break;
      case SET_RESOLUTION:
         res = pk.res.x;
         break;
      case SET_PIXELS_PER_ROW:
         width = pk.ppr.width;
         outwidth = pk.ppr.outwidth;
    
         /* Do sanity check on width. */
         max = res * 8;
         if (width > max)
         {
            bug("unable to set width=%d, default to page width=%d\n", width, max);   
            width = max;
         }
         if (outwidth > max)
         {
            bug("unable to set outwidth=%d, default to page width=%d\n", outwidth, max);   
            outwidth = max;
         }

    //    bug("job width=%d outwidth=%d effRes=%d\n", width, outwidth, max);

         if (pPC->SetPixelsPerRow(width, outwidth) != NO_ERROR)
            bug("unable to SetPixelsPerRow width=%d outwidth=%d\n", width, outwidth);
         break;
      case SET_DEVICE_NAME:

         /* Map device name to model number. */
         model = MapModel(models, maxModel, pk.dev.name);

         if (model == -1)
         {
            bug("unable to select DeviceName=%s\n", pk.dev.name);
            goto BUGOUT;
         }

         if (pPC->SelectDevice((PRINTER_TYPE)model) != NO_ERROR)
         {
            bug("unable to select device=%d\n", model);
            goto BUGOUT;
         }
         break;
      case SET_PRINTMODE:
         if (pPC->SelectPrintMode(pk.mode.val) != NO_ERROR)
            bug("unable to select print mode=%d\n", pk.mode.val);
         break;
      case SET_PAPER:
         /* Map ghostscript sizes to APDK sizes. */
         switch (pk.paper.size)
         {
            case PAPER_SIZE_LEGAL:
               paper = LEGAL;
               break;
            case PAPER_SIZE_A4:
               paper = A4;
               break;
            case PAPER_SIZE_LETTER:
            default:
               paper = LETTER;
               break;
         }

         if (pPC->SetPaperSize(paper) != NO_ERROR)
            bug("unable to set paper size=%d\n", pk.paper.size);
         break;
      case SET_SHMID:
         sd.shmid = pk.shm.id;
         if ((sd.shmbuf = (char *) shmat(sd.shmid, 0, 0)) < 0)
         {
            bug("unable to attach shared memory: %m\n");
            goto BUGOUT;
         }
         break;
      case GET_MODEL:
         strcpy(pk.dev.name, pPC->PrinterModel());
         break;
      case GET_PRINTMODE:
         pk.mode.val = pPC->CurrentPrintMode();
         break;
      case GET_PRINTMODE_CNT:
         pk.mode.val = pPC->GetModeCount();
         break;
      case GET_PAPER:
         pk.paper.size = pPC->GetPaperSize();
         break;
      case GET_EFFECTIVE_RESOLUTION:
         pk.res.x = pPC->EffectiveResolutionX();
         pk.res.y = pPC->EffectiveResolutionY();
         break;
      case GET_PIXELS_PER_ROW:
         pk.ppr.width = pPC->InputPixelsPerRow();
         pk.ppr.outwidth = pPC->OutputPixelsPerRow();
         break;      
      case GET_SRV_VERSION:
         strcpy(pk.ver.str, VERSION); 
         break;
      case GET_PRINTABLE_AREA:
         pk.parea.width = pPC->PrintableWidth();
         pk.parea.height = pPC->PrintableHeight();
         break;
      case GET_PHYSICAL_PAGE_SIZE:
         pk.psize.x = pPC->PhysicalPageSizeX();
         pk.psize.y = pPC->PhysicalPageSizeY();
         break;
      case GET_PRINTABLE_START:
         pk.pstart.x = pPC->PrintableStartX();
         pk.pstart.y = pPC->PrintableStartY();
         break;
      case KILL:
         status = 0;
         goto BUGOUT;     /* exit server, no ack */
         break;
      default:
         bug("invalid packet cmd %d\n", pk.cmd);
         break;
      }
      pk.cmd = ACK;
      hpijs_put_pk(&pk, sd.fds2c);
   }

 BUGOUT:
   if (pJob != NULL)
      delete pJob;
   if (pPC != NULL)
      delete pPC;
   if (pSS != NULL)
   {
#ifdef CAPTURE
      pSS->EndScript();
#endif
      delete pSS;
   }
   if (sd.shmbuf != NULL)
      shmdt(sd.shmbuf);
   close(sd.fds2c);
   close(sd.fdc2s);

   return(status);
}
