/*
 *  Copyright (C) 2001-2002, Richard J. Moore <rich@kde.org>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this library; see the file COPYING.LIB.  If not, write to
 *  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 *  Boston, MA 02111-1307, USA.
 */

#include <klocale.h>
#include <kprocess.h>

#include "messagelogwidget.h"
#include "messagelogwidget.moc"

namespace KJSEmbed {

MessageLogWidget::MessageLogWidget( QWidget *parent, const char *name )
    : QMultiLineEdit( parent, (name ? name : "message_log") ),
      proc(0)
{
    setReadOnly( true );
    setUndoRedoEnabled( false );
    setTextFormat( Qt::RichText );
    setText( "<qt><pre>" );
}

MessageLogWidget::~MessageLogWidget()
{
}

void MessageLogWidget::raw( const QString &msg )
{
    this->msg= msg;
    insertLine( msg );
    setCursorPosition( numLines() - 1, 0 );
}

void MessageLogWidget::message( const QString &msg )
{
    this->msg= msg;
    QString m( "%1<br>" );
    insertLine( m.arg(msg) );
    setCursorPosition( numLines() - 1, 0 );
}

void MessageLogWidget::error( const QString &msg )
{
    this->msg= msg;
    QString err( "<font color=\"red\"><b>%1</b></font><br>" );
    insertLine( err.arg(msg) );
    setCursorPosition( numLines() - 1, 0 );
}

//
// Process Handling
//
bool MessageLogWidget::run( const QString &cmd )
{
    if ( proc )
	return false;

    message( cmd );
    proc = new KShellProcess("/bin/sh");
    *proc << cmd;

    connect( proc, SIGNAL( processExited(KProcess *) ),
	     this, SLOT(childExited()) );
    connect( proc, SIGNAL( receivedStdout(KProcess *, char *, int) ),
	     this, SLOT( receivedStdOutput(KProcess *, char *, int) ) );
    connect( proc, SIGNAL( receivedStderr(KProcess *, char *, int) ),
	     this, SLOT( receivedStdError(KProcess *, char *, int) ) );

    return proc->start( KProcess::NotifyOnExit,
			KProcess::Communication( KProcess::Stdout|KProcess::Stderr ));
}


void MessageLogWidget::childExited()
{
    QString s;
    if ( proc->normalExit() ) {
	if ( proc->exitStatus() )
	    s = i18n( "[Exited with status %1]\n" ).arg( proc->exitStatus() );
	else
	    s = i18n( "[Finished]\n" );
	message( s );
    }
    else {
        s = i18n("[Aborted]\n");
	error( s );
    }

    delete proc;
    proc = 0;
}

void MessageLogWidget::receivedStdOutput( KProcess *, char *buffer, int buflen )
{
    QCString buf = QCString( buffer, buflen+1 );
    message( QString(buf) );
}

void MessageLogWidget::receivedStdError( KProcess *, char *buffer, int buflen )
{
    QCString buf = QCString( buffer, buflen+1 );
    error( QString(buf) );
}

}; // namespace KJSEmbed

// Local Variables:
// c-basic-offset: 4
// End:
