/*
   Copyright (C) 2000, 2001, 2002 Dawit Alemayehu <adawit@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License version 2 as published by the Free Software Foundation.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include <unistd.h>
#include <sys/types.h>

#include <kdebug.h>
#include <klocale.h>
#include <kinstance.h>

#include <kaction.h>
#include <kpopupmenu.h>
#include <kmessagebox.h>
#include <kiconloader.h>

#include <kdirlister.h>
#include <konq_dirpart.h>
#include <konq_propsview.h>
#include <kstaticdeleter.h>
#include <kgenericfactory.h>
#include <kparts/browserextension.h>

#include "dirfilterplugin.h"

typedef KGenericFactory<DirFilterPlugin> DirFilterFactory;
K_EXPORT_COMPONENT_FACTORY(libdirfilterplugin,
                           DirFilterFactory("dirfilterplugin"));

SessionManager* SessionManager::m_self = 0;
KStaticDeleter<SessionManager> dirfiltersd;


SessionManager *SessionManager::self ()
{
  if (!m_self)
    m_self = dirfiltersd.setObject(new SessionManager);

  return m_self;
}

SessionManager::SessionManager()
{
  m_bSettingsLoaded = false;
  loadSettings ();
}

SessionManager::~SessionManager()
{
  saveSettings();
  m_self = 0;
}

QString SessionManager::generateKey (const KURL& url)
{
  QString key;

  key = url.protocol ();
  key += ':';

  if (!url.host ().isEmpty ())
  {
    key += url.host ();
    key += ':';
  }

  key += url.path ();
  key += ':';
  key += QString::number (m_pid);

  return key;
}

QStringList SessionManager::restore (const KURL& url)
{
  QString key = generateKey (url);

  if (m_filters.contains(key))
    return m_filters[key];
  else
    return QStringList ();
}

void SessionManager::save (const KURL& url, const QStringList& filters)
{
  QString key = generateKey(url);
  m_filters[key] = filters;
}

void SessionManager::saveSettings()
{
  KConfig cfg ("dirfilterrc", false, false);
  cfg.setGroup ("General");

  cfg.writeEntry ("ShowCount", showCount);
  cfg.writeEntry ("UseMultipleFilters", useMultipleFilters);
  cfg.sync();
}

void SessionManager::loadSettings()
{
  if (m_bSettingsLoaded)
    return;

  KConfig cfg ("dirfilterrc", false, false);
  cfg.setGroup ("General");

  showCount = cfg.readBoolEntry ("ShowCount", false);
  useMultipleFilters = cfg.readBoolEntry ("UseMultipleFilters", true);
  m_pid = getpid ();
  m_bSettingsLoaded = true;
}



DirFilterPlugin::DirFilterPlugin (QObject* parent, const char* name,
                                  const QStringList &)
                :KParts::Plugin (parent, name)
{
  m_part = 0L;

  m_pFilterMenu = new KActionMenu (i18n("View &Filter"), "filter",
                                   actionCollection(), "filterdir");
  m_pFilterMenu->setDelayed (false);

  connect (m_pFilterMenu->popupMenu(), SIGNAL (aboutToShow()),
           SLOT (slotShowPopup()));

  if (parent && parent->inherits ("KonqDirPart"))
  {
    m_part = dynamic_cast<KonqDirPart*>(parent);

    connect (m_part, SIGNAL (itemRemoved(const KFileItem*)),
             SLOT( slotItemRemoved (const KFileItem*)));
    connect (m_part, SIGNAL (itemsAdded(const KFileItemList&)),
             SLOT (slotItemsAdded(const KFileItemList&)));
    connect (m_part, SIGNAL (itemsFilteredByMime(const KFileItemList&)),
             SLOT (slotItemsAdded(const KFileItemList&)));
    connect (m_part, SIGNAL(aboutToOpenURL()), SLOT(slotOpenURL()));
  }
}

DirFilterPlugin::~DirFilterPlugin()
{
  delete m_pFilterMenu;
}

void DirFilterPlugin::slotOpenURL ()
{
  KURL url = m_part->url();

  kdDebug() << "DirFilterPlugin::slotOpenURL: " << url.url() << endl;

  if (m_pURL != url)
  {
    m_pURL = url;
    m_part->setMimeFilter (SessionManager::self()->restore(url));
    kdDebug() << "Checking MIME FILTERS for: " << url.url() << endl;
  }
}

void DirFilterPlugin::slotShowPopup()
{
  int id;
  uint enableReset;

  QString label;
  QStringList inodes;

  if (!m_part)
  {
    m_pFilterMenu->setEnabled (false);
    return;
  }

  // Only update the list if something changed!
  id = 0;
  enableReset = 0;

  m_pFilterMenu->popupMenu()->clear();
  m_pFilterMenu->popupMenu()->insertTitle (i18n("Only show items of type:"));

  MimeInfoIterator it = m_pMimeInfo.begin();
  for (; it != m_pMimeInfo.end() ; ++it)
  {
    if (it.key().startsWith("inode"))
    {
      inodes << it.key();
      continue;
    }

    if (!SessionManager::self()->showCount)
    {
      label = it.data().mimeComment;
    }
    else
    {
      label = it.data().mimeComment;
      label += "  (";
      label += QString::number (it.data().count);
      label += ")";
    }

    m_pMimeInfo[it.key()].id = m_pFilterMenu->popupMenu()->insertItem (
                               SmallIconSet(it.data().iconName), label,
                               this, SLOT(slotItemSelected(int)), 0, ++id);

    if (it.data().useAsFilter)
    {
      m_pFilterMenu->popupMenu()->setItemChecked (id, true);
      enableReset++;
    }
  }

  // Add all the items with "inode/*" here...
  if (!inodes.isEmpty())
  {
    m_pFilterMenu->popupMenu()->insertSeparator ();

    QStringList::Iterator it = inodes.begin();
    for (; it != inodes.end(); ++it)
    {
      if (!SessionManager::self()->showCount)
      {
        label = m_pMimeInfo[(*it)].mimeComment;
      }
      else
      {
        label = m_pMimeInfo[(*it)].mimeComment;
        label += "  (";
        label += QString::number (m_pMimeInfo[(*it)].count);
        label += ")";
      }

      m_pMimeInfo[(*it)].id = m_pFilterMenu->popupMenu()->insertItem (
                              SmallIconSet(m_pMimeInfo[(*it)].iconName), label,
                              this, SLOT(slotItemSelected(int)), 0, ++id);

      if (m_pMimeInfo[(*it)].useAsFilter)
      {
        m_pFilterMenu->popupMenu()->setItemChecked (id, true);
        enableReset ++;
      }
    }
  }

  m_pFilterMenu->popupMenu()->insertSeparator ();
  id = m_pFilterMenu->popupMenu()->insertItem (i18n("Use Multiple Filters"),
                                               this, SLOT(slotMultipleFilters()));
  m_pFilterMenu->popupMenu()->setItemEnabled (id, !enableReset);
  m_pFilterMenu->popupMenu()->setItemChecked (id, !enableReset &&
                                              SessionManager::self()->useMultipleFilters);

  label = SessionManager::self()->showCount ? i18n("Hide Count"):i18n("Show Count");
  m_pFilterMenu->popupMenu()->insertItem (label, this, SLOT(slotShowCount()));

  id = m_pFilterMenu->popupMenu()->insertItem (i18n("Reset"), this,
                                               SLOT(slotReset()));

  m_pFilterMenu->popupMenu()->setItemEnabled (id, enableReset);
}

void DirFilterPlugin::slotItemSelected (int id)
{
  if (!m_part)
    return;

  MimeInfoIterator it = m_pMimeInfo.begin();
  while (it != m_pMimeInfo.end () && id != it.data().id)  it++;

  if (it != m_pMimeInfo.end())
  {
    QStringList filters;

    if (it.data().useAsFilter)
    {
      it.data().useAsFilter = false;
      filters = m_part->mimeFilter ();
      if (filters.remove (it.key()))
        m_part->setMimeFilter (filters);
    }
    else
    {
      m_pMimeInfo[it.key()].useAsFilter = true;

      if (SessionManager::self()->useMultipleFilters)
      {
        filters = m_part->mimeFilter ();
        filters << it.key();
      }
      else
      {
        filters << it.key();
      }

      m_part->setMimeFilter (filters);
    }

    KURL url = m_part->url();
    m_part->openURL (url);
    SessionManager::self()->save (url, filters);
  }
}

void DirFilterPlugin::slotItemsAdded (const KFileItemList& list)
{
  KURL url;

  QString name;
  QString mimeType;
  QStringList filters;

  url = m_part->url();

  if (list.count() == 0 || !m_part || !m_part->nameFilter().isEmpty())
  {
    m_pFilterMenu->setEnabled (m_part->nameFilter().isEmpty());
    return;
  }

  if (m_pURL != url)
  {
    m_pURL = url;
    m_pMimeInfo.clear();
  }

  KFileItemListIterator it (list);
  for (; it.current () ; ++it)
  {
    name = it.current()->name();
    mimeType = it.current()->mimetype();

    if (mimeType.isEmpty ())
      continue;

    if (!m_pMimeInfo.contains (mimeType))
    {
      filters = m_part->mimeFilter ();

      m_pMimeInfo[mimeType].useAsFilter = (!filters.isEmpty () &&
                                           filters.contains (mimeType));
      m_pMimeInfo[mimeType].mimeComment = it.current ()->mimeComment();
      m_pMimeInfo[mimeType].iconName = it.current ()->iconName();
      m_pMimeInfo[mimeType].filenames << name;
      m_pMimeInfo[mimeType].count++;
    }
    else
    {
      if (!m_pMimeInfo[mimeType].filenames.contains (name))
      {
        m_pMimeInfo[mimeType].count++;
        m_pMimeInfo[mimeType].filenames << name;
      }
    }
  }
}

void DirFilterPlugin::slotItemRemoved (const KFileItem* item)
{
  MimeInfo info;
  QString mimeType;
  QStringList filters;

  if (!item || !m_part)
    return;

  mimeType = item->mimetype().stripWhiteSpace();

  if (m_pMimeInfo.contains (mimeType))
  {
    info = m_pMimeInfo [mimeType];

    if (info.count == 1)
    {
      if (info.useAsFilter)
      {
        filters = m_part->mimeFilter ();
        filters.remove (mimeType);
        m_part->setMimeFilter (filters);
        KURL url = m_part->url();
        m_part->openURL (url);
        SessionManager::self()->save (url, filters);
      }
      m_pMimeInfo.remove (mimeType);
    }
    else
    {
      m_pMimeInfo [mimeType].count--;
      m_pMimeInfo [mimeType].filenames.remove (item->name ());
    }
  }
}

void DirFilterPlugin::slotReset()
{
  if (!m_part)
    return;

  MimeInfoIterator it = m_pMimeInfo.begin();
  for (; it != m_pMimeInfo.end(); ++it)
    it.data().useAsFilter = false;

  QStringList filters;
  KURL url = m_part->url();

  m_part->setMimeFilter (filters);
  m_part->openURL (url);
  SessionManager::self()->save (m_part->url(), filters);
}

void DirFilterPlugin::slotShowCount()
{
  if (SessionManager::self()->showCount)
    SessionManager::self()->showCount = false;
  else
    SessionManager::self()->showCount = true;
}

void DirFilterPlugin::slotMultipleFilters()
{
  if (SessionManager::self()->useMultipleFilters)
    SessionManager::self()->useMultipleFilters = false;
  else
    SessionManager::self()->useMultipleFilters = true;
}


#include "dirfilterplugin.moc"
