    /*

    Copyright (C) 2000 Stefan Westerfeld
                       stefan@space.twc.de

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.
  
    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.
   
    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.

    */

#include "config.h"

#ifdef HAVE_ARTS_GSL
#include "virtualports.h"
#include "startupmanager.h"
#include "gslschedule.h"
#include "debug.h"
#include "asyncschedule.h"
#include "audiosubsys.h"
#include <gsl/gslcommon.h>
#include <gsl/gslengine.h>
#include <algorithm>
#include <stdio.h>
#include <iostream>
#include <stack>

/* HACK */
class GslMainLoop {
protected:
	std::list<GslClass *> freeClassList;

public:
	GslEngineLoop loop;

	static bool waitOnTransNeedData;
	static bool gslDataCalculated;

	/* static check function */
	static gboolean gslCheck(gpointer data, guint n_values, glong *timeout_p,
                         	 guint n_fds, const GslPollFD *fds,
						 	 gboolean revents_filled)
	{
		return waitOnTransNeedData;
	}
	/* mainloop integration: initialize (called to get initial loop setup) */
	void initialize()
	{
		gsl_transact(gsl_job_add_poll (gslCheck, 0, 0, 0, 0), 0);
		gsl_engine_prepare(&loop);

		for(unsigned int i = 0; i != loop.n_fds; i++)
		{
			printf("TODO: engine fd %d\n",i);
		}
	}
	/* mainloop integration: process (TODO - should be called by IOManager) */
	void process()
	{
		printf("TODO: mainloop wrapper for fd watches\n");
		if(gsl_engine_check(&loop))
			gsl_engine_dispatch();
	}
	/* wait for a transaction */
	void waitOnTrans()
	{
		arts_return_if_fail(waitOnTransNeedData == false);
		gsl_engine_wait_on_trans();
	}
	/* make the engine calculate something */
	void run()
	{
		waitOnTransNeedData = true;
		gslDataCalculated = false;

		while(!gslDataCalculated && gsl_engine_check(&loop))
			gsl_engine_dispatch();
	
		gslDataCalculated = false;
		waitOnTransNeedData = false;

		if(!freeClassList.empty())
		{
			/*
			 * make sure that all transactions that are still pending
			 * get finished (especially important in threaded case,
			 * since an entry in the free list doesn't necessarily
			 * mean that the module has entierly been freed)
			 */
			waitOnTrans();

			std::list<GslClass *>::iterator fi;
			for(fi = freeClassList.begin(); fi != freeClassList.end(); fi++)
				free(*fi);

			freeClassList.clear();
		}
	}
	void freeGslClass(GslClass *klass)
	{
		freeClassList.push_back(klass);
	}
} gslMainLoop;

bool GslMainLoop::waitOnTransNeedData = false;
bool GslMainLoop::gslDataCalculated = false;
namespace Arts { extern void *gslGlobalMutexTable; }


using namespace std;
using namespace Arts;

// well, this was tuneable once...

static unsigned long requestSize()
{
	unsigned long reqSize = 0;
	if(!reqSize)
		reqSize = AudioSubSystem::the()->fragmentSize()/4;

	return reqSize;
}

// ----------- SynthBuffer -------------

SynthBuffer::SynthBuffer(float initialvalue, unsigned long size)
{
	this->size = size;
	data = new float[size];

	setValue(initialvalue);

	position = 0;
	needread = 0;
}

void SynthBuffer::setValue(float value)
{
	unsigned long i;
	for(i=0;i<size;i++) data[i] = value;
}

SynthBuffer::~SynthBuffer()
{
	delete[] data;
}

// ----------- Port -----------

Port::Port(string name, void *ptr, long flags, StdScheduleNode* parent)
	: _name(name), _ptr(ptr), _flags((AttributeType)flags),
	  parent(parent), _dynamicPort(false)
{
	_vport = new VPort(this);
}

Port::~Port()
{
	if(_vport)
		delete _vport;
}

AttributeType Port::flags()
{
	return _flags;
}

string Port::name()
{
	return _name;
}

ASyncPort *Port::asyncPort()
{
	return 0;
}

AudioPort *Port::audioPort()
{
	return 0;
}

void Port::addAutoDisconnect(Port *source)
{
	autoDisconnect.push_back(source);
	source->autoDisconnect.push_back(this);
}

void Port::removeAutoDisconnect(Port *source)
{
	std::list<Port *>::iterator adi;

	// remove our autodisconnection entry for source port
	adi = find(autoDisconnect.begin(),autoDisconnect.end(),source);
	assert(adi != autoDisconnect.end());
	autoDisconnect.erase(adi);

	// remove the source port autodisconnection entry to us
	adi=find(source->autoDisconnect.begin(),source->autoDisconnect.end(),this);
	assert(adi != source->autoDisconnect.end());
	source->autoDisconnect.erase(adi);
}

void Port::disconnectAll()
{
	if(_vport)
		delete _vport;
	_vport = 0;
	assert(autoDisconnect.empty());
	while(!autoDisconnect.empty())
	{
		Port *other = *autoDisconnect.begin();

		// syntax is disconnect(source)
		if(_flags & streamIn)
			// if we're incoming, other port is source
			vport()->disconnect(other->vport());
		else
			// if we're outgoing, we're the source
			other->vport()->disconnect(this->vport());
	}
}

void Port::setPtr(void *ptr)
{
	_ptr = ptr;
}

// ------- AudioPort ---------

AudioPort::AudioPort(string name, void *ptr, long flags,StdScheduleNode *parent)
		: Port(name,ptr,flags,parent)
{
	position = 0;
	destcount = 0;
	sourcemodule = 0;
	source = 0;
	gslIsConstant = false;
	lbuffer = buffer = new SynthBuffer(0.0, requestSize());
}

AudioPort::~AudioPort()
{
	delete lbuffer;
}

AudioPort *AudioPort::audioPort()
{
	return this;
}

void AudioPort::setFloatValue(float f)
{
	gslIsConstant = true;
	gslConstantValue = f;
	buffer->setValue(f);
}

void AudioPort::connect(Port *psource)
{
	source = psource->audioPort();
	assert(source);
	addAutoDisconnect(psource);

	buffer = source->buffer;
	position = buffer->position;
	source->destcount++;
	sourcemodule = source->parent;

	GslTrans *trans = gsl_trans_open();
	gsl_trans_add(trans, gsl_job_connect(source->parent->gslModule,
										 source->gslEngineChannel,
										 parent->gslModule,
										 gslEngineChannel));
	gsl_trans_commit(trans);
}

void AudioPort::disconnect(Port *psource)
{
	assert(source);
	assert(source == psource->audioPort());
	removeAutoDisconnect(psource);

	assert(sourcemodule == source->parent);
	sourcemodule = 0;

	// skip the remaining stuff in the buffer
	read(buffer->position - position);
	source->destcount--;
	source = 0;

	position = lbuffer->position;
	buffer = lbuffer;

	// GSL disconnect
	GslTrans *trans = gsl_trans_open();
	gsl_trans_add(trans, gsl_job_disconnect(parent->gslModule,
										    gslEngineChannel));
	gsl_trans_commit(trans);
}

// --------- MultiPort ----------

MultiPort::MultiPort(string name, void *ptr, long flags,StdScheduleNode *parent)
	: Port(name,ptr,flags,parent)
{
	conns = 0;
	nextID = 0;
	initConns();
}

MultiPort::~MultiPort()
{
	if(conns)
	{
		delete[] conns;
		conns = 0;
	}
}

void MultiPort::initConns()
{
	if(conns != 0) delete[] conns;
	conns = new float_ptr[parts.size() + 1];
	conns[parts.size()] = (float *)0;

	*(float ***)_ptr = conns;

	long n = 0;
	std::list<AudioPort *>::iterator i;
	for(i = parts.begin();i != parts.end(); i++)
	{
		AudioPort *p = *i;
		p->setPtr((void *)&conns[n++]);
	}
}

void MultiPort::connect(Port *port)
{
	AudioPort *dport;
	char sid[20];
	sprintf(sid,"%ld",nextID++);

	addAutoDisconnect(port);

	dport = new AudioPort("_"+_name+string(sid),0,streamIn,parent);
	parts.push_back(dport);
	initConns();

	parent->addDynamicPort(dport);
	dport->vport()->connect(port->vport());
}

void MultiPort::disconnect(Port *sport)
{
	AudioPort *port = (AudioPort *)sport;
	removeAutoDisconnect(sport);

	std::list<AudioPort *>::iterator i;
	for(i = parts.begin(); i != parts.end(); i++)
	{
		AudioPort *dport = *i;

		if(dport->buffer == port->buffer)
		{
			parts.erase(i);
			initConns();

			dport->vport()->disconnect(port->vport());
			parent->removeDynamicPort(dport);

			delete dport;
			return;
		}
	}
}

// -------- StdScheduleNode ---------

void StdScheduleNode::freeConn()
{
	if(inConn)
	{
		delete[] inConn;
		inConn = 0;
	}
	if(outConn)
	{
		delete[] outConn;
		outConn = 0;
	}
	inConnCount = outConnCount = 0;

	if(gslModule)
	{
		gsl_transact(gsl_job_discard(gslModule),0);

		gslModule = 0;
		gslRunning = false;
	}
}

void StdScheduleNode::gslProcess(GslModule *module, guint n_values)
{
	StdScheduleNode *node = (StdScheduleNode *)module->user_data;
	if(!node->running)		/* FIXME: need reasonable suspend in the engine */
		return;

	arts_return_if_fail(node->module != 0);

	GslMainLoop::gslDataCalculated = true;
	//fprintf(stderr,"TODO: write process wrapper\n");
	unsigned long j, donecycles = 0, cando = 0;

	for(j=0;j<node->inConnCount;j++)
	{
		if(node->inConn[j]->gslIsConstant)
			*((float **)node->inConn[j]->_ptr) =
				gsl_engine_const_values(node->inConn[j]->gslConstantValue);
		else
			*((float **)node->inConn[j]->_ptr) = const_cast<float *>(module->istreams[j].values);
	}

	for(j=0;j<node->outConnCount;j++)
		*((float **)node->outConn[j]->_ptr) = module->ostreams[j].values;

	node->module->calculateBlock(n_values);
}

static void gslModuleFree(gpointer data, const GslClass *klass)
{
	gslMainLoop.freeGslClass(const_cast<GslClass *>(klass));
}

void StdScheduleNode::rebuildConn()
{
	std::list<Port *>::iterator i;

	freeConn();

	inConnCount = outConnCount = 0;
	inConn = new AudioPort_ptr[ports.size()];
	outConn = new AudioPort_ptr[ports.size()];

	for(i=ports.begin();i != ports.end();i++)
	{
		AudioPort *p = (*i)->audioPort();
		if(p)
		{
			if(p->flags() & streamIn)
			{
				p->gslEngineChannel = inConnCount;
				inConn[inConnCount++] = p;
			}
			if(p->flags() & streamOut)
			{
				p->gslEngineChannel = outConnCount;
				outConn[outConnCount++] = p;
			}
		}
	}

	/* create GSL node */
	GslClass *gslClass = (GslClass *)calloc(sizeof(GslClass),1);
	gslClass->n_istreams = inConnCount;
	gslClass->n_ostreams = outConnCount;
	gslClass->process = gslProcess;
	gslClass->free = gslModuleFree;

	gslModule = gsl_module_new (gslClass, (StdScheduleNode *)this);

	GslTrans *trans = gsl_trans_open();
	gsl_trans_add(trans,gsl_job_integrate(gslModule));
	gsl_trans_add(trans,gsl_job_set_consumer(gslModule, running));
	gslRunning = running;

	/* since destroying the old module and creating a new one will destroy
	 * all the connections, we need to restore them here
	 */
	int c;
	for(c = 0; c < inConnCount; c++)
	{
		if(inConn[c]->source)
		{
			gsl_trans_add(trans,
				gsl_job_connect(inConn[c]->source->parent->gslModule,
						 	 	inConn[c]->source->gslEngineChannel,
								inConn[c]->parent->gslModule,
								inConn[c]->gslEngineChannel));
		}
	}
	for(c = 0; c < outConnCount; c++)
	{
		std::list<Port *>::iterator ci;
		
		for(ci = outConn[c]->autoDisconnect.begin();
			ci != outConn[c]->autoDisconnect.end(); ci++)
		{
			AudioPort *dest = (*ci)->audioPort();
			assert(dest);

			gsl_trans_add(trans,
				gsl_job_connect(outConn[c]->parent->gslModule,
								outConn[c]->gslEngineChannel,
								dest->parent->gslModule,
								dest->gslEngineChannel));
		}
	}
	gsl_trans_commit(trans);
}

Object_skel *StdScheduleNode::object()
{
	return _object;
}

void *StdScheduleNode::cast(const string &target)
{
	if(target == "StdScheduleNode") return (StdScheduleNode *)this;
	return 0;
}


void StdScheduleNode::accessModule()
{
	if(module) return;

	module = (SynthModule_base *)_object->_cast(Arts::SynthModule_base::_IID);
	if(!module)
		arts_warning("Error using interface %s in the flowsystem: only objects"
					 " implementing Arts::SynthModule should carry streams.",
					 _object->_interfaceName().c_str());
}

StdScheduleNode::StdScheduleNode(Object_skel *object, StdFlowSystem *flowSystem) : ScheduleNode(object)
{
	_object = object;
	this->flowSystem = flowSystem;
	running = false;
	suspended = false;
	module = 0;
	gslModule = 0;
	gslRunning = false;
	queryInitStreamFunc = 0;
	inConn = outConn = 0;
	inConnCount = outConnCount = 0;
	Busy = BusyHit = NeedCycles = CanPerform = 0;
}

StdScheduleNode::~StdScheduleNode()
{
	/* stop module if still running */
	if(running) stop();
	/* disconnect all ports */
	stack<Port *> disconnect_stack;

	/*
	 * we must be a bit careful here, as dynamic ports (which are created
	 * for connections by MultiPorts) will suddenly start disappearing, so
	 * we better make a copy of those ports that will stay, and disconnect
	 * them then
	 */
	std::list<Port *>::iterator i;
	for(i=ports.begin();i != ports.end();i++)
	{
		if(!(*i)->dynamicPort()) disconnect_stack.push(*i);
	}

	while(!disconnect_stack.empty())
	{
		disconnect_stack.top()->disconnectAll();
		disconnect_stack.pop();
	}
	/* free them */
	for(i=ports.begin();i != ports.end();i++)
		delete (*i);
	ports.clear();

	freeConn();
}

void StdScheduleNode::initStream(const string& name, void *ptr, long flags)
{
	if(flags == -1)
	{
		queryInitStreamFunc = (QueryInitStreamFunc)ptr;
	}
	else if(flags & streamAsync)
	{
		ports.push_back(new ASyncPort(name,ptr,flags,this));
	}
	else if(flags & streamMulti)
	{
		ports.push_back(new MultiPort(name,ptr,flags,this));
	}
	else
	{
		ports.push_back(new AudioPort(name,ptr,flags,this));
	}

	// TODO: maybe initialize a bit later
	rebuildConn();
}

void StdScheduleNode::addDynamicPort(Port *port)
{
	port->setDynamicPort();
	ports.push_back(port);
	rebuildConn();
}

void StdScheduleNode::removeDynamicPort(Port *port)
{
	std::list<Port *>::iterator i;
	for(i=ports.begin();i!=ports.end();i++)
	{
		Port *p = *i;
		if(p->name() == port->name())
		{
			ports.erase(i);
			rebuildConn();
			return;	
		}
	}
}

void StdScheduleNode::start()
{
	assert(!running);
	running = true;

	//cout << "start" << endl;
	accessModule();
	module->streamInit();
	module->streamStart();
	flowSystem->startedChanged();
}

void StdScheduleNode::stop()
{
	assert(running);
	running = false;

	accessModule();
	module->streamEnd();
	flowSystem->startedChanged();
}

void StdScheduleNode::requireFlow()
{
	// cout << "rf" << module->_interfaceName() << endl;
	//request(requestSize());
	/* GSL! flowSystem->schedule(requestSize()); */
	flowSystem->updateStarted();
	gslMainLoop.run();
}

bool StdScheduleNode::suspendable()
{
	if(running) {
		accessModule();
		return (module->autoSuspend() != asNoSuspend);
	}
	// if its not running, who cares?
	return true;
}

void StdScheduleNode::suspend()
{
	if(running) {
		accessModule();
		suspended = true;
		if(module->autoSuspend() == asSuspendStop) stop();
	}
}

void StdScheduleNode::restart()
{
	if(running == false && suspended == true) {
		accessModule();
		suspended = false;
		if(module->autoSuspend() == asSuspendStop) start();
	}
}

Port *StdScheduleNode::findPort(string name)
{
	std::list<Port *>::iterator i;
	for(i=ports.begin();i!=ports.end();i++)
	{
		Port *p = *i;
		if(p->name() == name) return p;
	}
	if(queryInitStreamFunc)
	{
		if(queryInitStreamFunc(_object,name))
		{
			for(i=ports.begin();i!=ports.end();i++)
			{
				Port *p = *i;
				if(p->name() == name) return p;
			}
		}
	}
	return 0;
}

void StdScheduleNode::virtualize(const std::string& port,
		                         ScheduleNode *implNode,
								 const std::string& implPort)
{
	StdScheduleNode *impl=(StdScheduleNode *)implNode->cast("StdScheduleNode");
	if(impl)
	{
		Port *p1 = findPort(port);
		Port *p2 = impl->findPort(implPort);

		assert(p1);
		assert(p2);
		p1->vport()->virtualize(p2->vport());
	}
}

void StdScheduleNode::devirtualize(const std::string& port, 
		                           ScheduleNode *implNode,
								   const std::string& implPort)
{
	StdScheduleNode *impl=(StdScheduleNode *)implNode->cast("StdScheduleNode");
	if(impl)
	{
		Port *p1 = findPort(port);
		Port *p2 = impl->findPort(implPort);

		p1->vport()->devirtualize(p2->vport());
	}
}

void StdScheduleNode::connect(const string& port, ScheduleNode *dest,
		                                          const string& destport)
{
	RemoteScheduleNode *rsn = dest->remoteScheduleNode();
	if(rsn)
	{
		// RemoteScheduleNodes know better how to connect remotely
		rsn->connect(destport,this,port);
		return;
	}

	Port *p1 = findPort(port);
	Port *p2 = ((StdScheduleNode *)dest)->findPort(destport);

	if(p1 && p2)
	{
		if((p1->flags() & streamIn) && (p2->flags() & streamOut))
		{
			p1->vport()->connect(p2->vport());
		}
		else if((p2->flags() & streamIn) && (p1->flags() & streamOut))
		{
			p2->vport()->connect(p1->vport());
		}
	}
}

void StdScheduleNode::disconnect(const string& port, ScheduleNode *dest,
		                                             const string& destport)
{
	RemoteScheduleNode *rsn = dest->remoteScheduleNode();
	if(rsn)
	{
		// RemoteScheduleNodes know better how to disconnect remotely
		rsn->disconnect(destport,this,port);
		return;
	}

	Port *p1 = findPort(port);
	Port *p2 = ((StdScheduleNode *)dest)->findPort(destport);

	if(p1 && p2)
	{
		if((p1->flags() & streamIn) && (p2->flags() & streamOut))
		{
			p1->vport()->disconnect(p2->vport());
		}
		else if((p2->flags() & streamIn) && (p1->flags() & streamOut))
		{
			p2->vport()->disconnect(p1->vport());
		}
	}
}

AttributeType StdScheduleNode::queryFlags(const std::string& port)
{
	arts_debug("findPort(%s)", port.c_str());
	arts_debug("have %ld ports", ports.size());
	Port *p1 = findPort(port);
	arts_debug("done");

	if(p1)
	{
		arts_debug("result %d",(long)p1->flags());
		return p1->flags();
	}
	arts_debug("failed");
	return (AttributeType)0;
}

void StdScheduleNode::setFloatValue(const string& port, float value)           
{
	AudioPort *p = findPort(port)->audioPort();

	if(p) {
		p->vport()->setFloatValue(value);
	} else {
		assert(false);
	}
}

/* request a module to calculate a number of turns
   will return -1 if busy, count that has been done otherwise */

long StdScheduleNode::request(long amount)
{
	unsigned long in;
	int have_in,need_more,have_done,have_done_total = 0;

	if(!running)
	{
		arts_fatal("Calculating data on a module which was not started!\n"
					"Start modules by calling module.start() "
					"before connecting them to avoid this.");
		// not reached
	}

	if(Busy) { BusyHit++; return(-1); }

	Busy = 1;
	if(NeedCycles < amount)
	{
		NeedCycles = amount;
	}

	// cout << "re" << module->_interfaceName() << endl;
	//artsdebug("DSP %s: request of %d cycles\n",getClassName(),amount);
	do
	{
		/* assume we can satisfy the request */
		CanPerform = NeedCycles;

		/* then, check wether the input channels supply enough data to do so. */

		for(in=0;in<inConnCount;in++)
		{
			have_in = inConn[in]->haveIn();

			if(have_in < NeedCycles)
			{
				//artsdebug("connection %d indicates to have %d, "
                //       "thats not enough\n", in,have_in);
				/* if we can't calculate enough stuff due to a certain
				   ingoing connection, go to the associated module and
				   tell it that we need more data
				*/
				need_more = NeedCycles - have_in;
				//artsdebug("requesting %d\n", need_more);
				/* when there is no source (constant input value), then
					we don't need to request something, it's just that
					it can't supply more because the buffer isn't big enough
				*/
				if(inConn[in]->sourcemodule)
					inConn[in]->sourcemodule->request(need_more);

				have_in = inConn[in]->haveIn();
	
				//artsdebug("now connection %d indicates to have %d\n",in,have_in);
				if(CanPerform > have_in) CanPerform = have_in;
			}
		}
		have_done = calc(CanPerform);
		have_done_total += have_done;

		/*
		if(dsp->m_BusyHit != 0) artsdebug("got busyhit: %s\n",dsp->m_Name);
		*/

	} while(BusyHit && NeedCycles != CanPerform && have_done);
	/* makes only sense to repeat if
		 - there was a busyhit which indicates we are in a feedback loop
		 - we should have done more than we have done
         - actually something could be calculated
     */
	Busy = 0;
	return(have_done);
}

/* This routine would now actually let the plugin calculate some data
   that means generate sinus waves, mix audio signals etc.

   The number of cycles is guaranteed to work without input underrun
   by the flow system. But the routine still needs to check output
   stall situations (ring buffer full).
*/

unsigned long StdScheduleNode::calc(unsigned long cycles)
{
	unsigned long i,room;

	//cout << "ca" << module->_interfaceName() << endl;
	/* output sanity check:
	   when there is not enough room in one of the buffers, we
	   can't calculate that much cycles
	*/
	for(i=0;i<outConnCount;i++)
	{
		room = outConn[i]->outRoom();
		if(room < cycles)
		{
			cycles = room;
			/*
			artsdebug("- reduced calculation to %d due to lack of space\n",cycles);
			*/
		}
	}

	if(cycles == 0) return(0);

	//artsdebug("DSP %s: calculation of %d cycles\n",getClassName(),cycles);
	/* input sanity check:
		it's guaranteed that we have enough input, but do the check
		anyway... - you never know
	*/
	for(i=0;i<inConnCount;i++)
	{
		/* otherwise input is "overconsumed" */
		assert(inConn[i]->haveIn() >= cycles);

		/* check sanity of the needread setting:
			either data comes from fixed value (sourcemodule not assigned)
			then we can't expect needread to contain sensible setting,
			but otherwise: needread should have expected that we'll want
			to read the data and thus be more than the cycles! */

		assert((!inConn[i]->sourcemodule)
				|| (inConn[i]->buffer->needread >= cycles));
	}

	unsigned long j, donecycles = 0, cando = 0;
	
	while(donecycles != cycles)
	{	
		cando = cycles-donecycles;

		for(j=0;j<inConnCount;j++)
			cando = inConn[j]->readSegment(donecycles,cando);

		for(j=0;j<outConnCount;j++)
			cando = outConn[j]->writeSegment(donecycles,cando);

		module->calculateBlock(cando);
		donecycles += cando;
	}
	assert(donecycles == cycles);

	// actually update buffer by subtracting consumed input
	for(i=0;i<inConnCount;i++)
		inConn[i]->read(cycles);

	// and adding fresh output
	for(i=0;i<outConnCount;i++)
		outConn[i]->write(cycles);

	NeedCycles -= cycles;
	CanPerform -= cycles;
	return(cycles);
}

StdFlowSystem::StdFlowSystem()
{
	_suspended = false;
	needUpdateStarted = false;

	/* TODO: correct parameters */
	static bool gsl_is_initialized = false;
	if(!gsl_is_initialized)
	{
		gsl_is_initialized = true;

		g_thread_init(0);
		gsl_init(0, (GslMutexTable *)gslGlobalMutexTable);
		gsl_engine_init(false, 512, 44100);
		if(gslGlobalMutexTable)
			arts_debug("gsl: using Unix98 pthreads directly for mutexes and conditions");
		/*gsl_engine_debug_enable(GslEngineDebugLevel(GSL_ENGINE_DEBUG_JOBS | GSL_ENGINE_DEBUG_SCHED));*/
	}
	gslMainLoop.initialize();
}

ScheduleNode *StdFlowSystem::addObject(Object_skel *object)
{
	// do not add new modules when being in suspended state
	restart();

	StdScheduleNode *node = new StdScheduleNode(object,this);
	nodes.push_back(node);
	return node;
}

void StdFlowSystem::removeObject(ScheduleNode *node)
{
	StdScheduleNode *xnode = (StdScheduleNode *)node->cast("StdScheduleNode");
	assert(xnode);
	nodes.remove(xnode);
	delete xnode;
}

bool StdFlowSystem::suspended()
{
	return _suspended;
}

bool StdFlowSystem::suspendable()
{
	std::list<StdScheduleNode *>::iterator i;
	for(i = nodes.begin();i != nodes.end();i++)
	{
		StdScheduleNode *node = *i;
		if(!node->suspendable()) return false;
	}
	return true;
}

void StdFlowSystem::suspend()
{
	if(!_suspended)
	{
		std::list<StdScheduleNode *>::iterator i;
		for(i = nodes.begin();i != nodes.end();i++)
		{
			StdScheduleNode *node = *i;
			node->suspend();
		}
		_suspended = true;
	}
}

void StdFlowSystem::restart()
{
	if(_suspended)
	{
		std::list<StdScheduleNode *>::iterator i;
		for(i = nodes.begin();i != nodes.end();i++)
		{
			StdScheduleNode *node = *i;
			node->restart();
		}
		_suspended = false;
	}
}

/* remote accessibility */

void StdFlowSystem::startObject(Object node)
{
	StdScheduleNode *sn =
		(StdScheduleNode *)node._node()->cast("StdScheduleNode");
	sn->start();
}

void StdFlowSystem::stopObject(Object node)
{
	StdScheduleNode *sn =
		(StdScheduleNode *)node._node()->cast("StdScheduleNode");
	sn->stop();
}

void StdFlowSystem::connectObject(Object sourceObject,const string& sourcePort,
					Object destObject, const std::string& destPort)
{
	arts_debug("connect port %s to %s", sourcePort.c_str(), destPort.c_str());
	StdScheduleNode *sn =
		(StdScheduleNode *)sourceObject._node()->cast("StdScheduleNode");
	assert(sn);

	Port *port = sn->findPort(sourcePort);
	assert(port);

	StdScheduleNode *destsn =
		(StdScheduleNode *)destObject._node()->cast("StdScheduleNode");
	if(destsn)
	{
		sn->connect(sourcePort,destsn,destPort);
		return;
	}
	
	ASyncPort *ap = port->asyncPort();

	if(ap)
	{
		FlowSystemSender sender;
		FlowSystemReceiver receiver;
		FlowSystem remoteFs;

		string dest = destObject.toString() + ":" + destPort;
		ASyncNetSend *netsend = new ASyncNetSend(ap, dest);

		sender = FlowSystemSender::_from_base(netsend); // don't release netsend
		remoteFs = destObject._flowSystem();
		receiver = remoteFs.createReceiver(destObject, destPort, sender);
		netsend->setReceiver(receiver);
		arts_debug("connected an asyncnetsend");
	}
}

void StdFlowSystem::disconnectObject(Object sourceObject,
  	const string& sourcePort, Object destObject, const std::string& destPort)
{
	arts_debug("disconnect port %s and %s",sourcePort.c_str(),destPort.c_str());
	StdScheduleNode *sn =
		(StdScheduleNode *)sourceObject._node()->cast("StdScheduleNode");
	assert(sn);

	Port *port = sn->findPort(sourcePort);
	assert(port);

	StdScheduleNode *destsn =
		(StdScheduleNode *)destObject._node()->cast("StdScheduleNode");
	if(destsn)
	{
		sn->disconnect(sourcePort,destsn,destPort);
		return;
	}

	ASyncPort *ap = port->asyncPort();
	if(ap)
	{
		string dest = destObject.toString() + ":" + destPort;
		ap->disconnectRemote(dest);
		arts_debug("disconnected an asyncnetsend");
	}
}

AttributeType StdFlowSystem::queryFlags(Object node, const std::string& port)
{
	StdScheduleNode *sn =
		(StdScheduleNode *)node._node()->cast("StdScheduleNode");
	assert(sn);
	return sn->queryFlags(port);
}

void StdFlowSystem::setFloatValue(Object node, const std::string& port,
							float value)
{
	StdScheduleNode *sn =
		(StdScheduleNode *)node._node()->cast("StdScheduleNode");
	assert(sn);
	sn->setFloatValue(port,value);
}

FlowSystemReceiver StdFlowSystem::createReceiver(Object object,
							const string &port, FlowSystemSender sender)
{
	StdScheduleNode *sn =
		(StdScheduleNode *)object._node()->cast("StdScheduleNode");

	Port *p = sn->findPort(port);
	assert(p);

	ASyncPort *ap = p->asyncPort();

	if(ap)
	{
		arts_debug("creating packet receiver");
		return FlowSystemReceiver::_from_base(new ASyncNetReceive(ap, sender));
	}
	return FlowSystemReceiver::null();
}

void StdFlowSystem::updateStarted()
{
	if(!needUpdateStarted)
		return;

	needUpdateStarted = false;

	std::list<StdScheduleNode*>::iterator ni;
	GslTrans *trans = 0;

	for(ni = nodes.begin(); ni != nodes.end(); ni++)
	{
		StdScheduleNode *node = *ni;

		if(node->running != node->gslRunning)
		{
			if(!trans)
				trans = gsl_trans_open();
			gsl_trans_add(trans, gsl_job_set_consumer(node->gslModule, node->running));
			node->gslRunning = node->running;
		}
	}
	if(trans)
		gsl_trans_commit(trans);
}

void StdFlowSystem::startedChanged()
{
	needUpdateStarted = true;
}

void StdFlowSystem::schedule(unsigned long samples)
{
/** new style (dynamic size) scheduling **/

	unsigned long MCount = nodes.size();
	unsigned long *done = (unsigned long *)calloc(MCount,sizeof(unsigned long));
	unsigned long i;
	std::list<StdScheduleNode*>::iterator ni;
	long incomplete, died = 0;

	//printf("entering; samples = %d\n",samples);
	do {
		incomplete = 0;		/* assume we have calculated all samples for all
								consumers, and later increment if some are
								still missing */
		for(ni = nodes.begin(), i = 0; ni != nodes.end(); ni++, i++)
		{
			StdScheduleNode *node = *ni;

            unsigned int outConnections = 0,c;
            for(c=0;c<node->outConnCount && !outConnections;c++)
                outConnections += node->outConn[c]->destcount;                  

			if(outConnections == 0 && node->running)
			{
				//printf("consumber = %s,done = %d, samples = %d\n",SynthModules[i]->getClassName(),done[i],samples);
				/* a module whose input is not comsumed from other modules
					is a "push delivery" style module, such as speakers,
					or writing audio to log file, etc. and has to get
					external requests from the scheduling system */

				if(done[i] != samples)
					done[i] += node->request(samples-done[i]);
				assert(done[i] <= samples);
				if(done[i] != samples) incomplete++;
				//printf("*scheduler*\n");
				died ++;
				if(died > 10000)
				{
					free(done);
					arts_warning("scheduler confusion: circle?");
					return;
				}
			}
		}
	} while(incomplete);

	//printf("<=> done!!\n");
	free(done);
	//return true;
}

// hacked initialization of Dispatcher::the()->flowSystem ;)

namespace Arts {

static class SetFlowSystem : public StartupClass {
	FlowSystem_impl *fs;
public:
	void startup()
	{
		fs = new StdFlowSystem;
		Dispatcher::the()->setFlowSystem(fs);
	}
	void shutdown()
	{
		fs->_release();
	}
} sfs;

};
#endif
