/*
 * GQview
 * (C) 2001 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */


#include "gqview.h"
#include "menu.h"

#include "cache_maint.h"
#include "collect.h"
#include "collect-dlg.h"
#include "dupe.h"
#include "filelist.h"
#include "img-main.h"
#include "img-view.h"
#include "preferences.h"
#include "slideshow.h"
#include "utilops.h"
#include "window.h"
#include "ui_fileops.h"
#include "ui_tabcomp.h"
#include "ui_menu.h"

#include "icons/icon_thumb.xpm"
#include "icons/icon_home.xpm"
#include "icons/icon_reload.xpm"
#include "icons/icon_zoom_in.xpm"
#include "icons/icon_zoom_out.xpm"
#include "icons/icon_zoom_fit.xpm"
#include "icons/icon_zoom_norm.xpm"
#include "icons/icon_config.xpm"
#include "icons/icon_float.xpm"


#define MENU_EDIT_ACTION_OFFSET 16

static GtkWidget *real_submenu_add_alter(GtkWidget *menu, GtkSignalFunc func, gpointer data,
					 GtkAccelGroup *accel_group);


static void set_thumbnails(gint mode)
{
	GtkWidget *thumb_item;

	if (thumbnails_enabled == mode) return;

	thumbnails_enabled = mode;

	thumb_item = gtk_item_factory_get_item(gqview_fact, "/View/Thumbnails");
	gtk_check_menu_item_set_state(GTK_CHECK_MENU_ITEM(thumb_item), thumbnails_enabled);

	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(thumb_button), thumbnails_enabled);
	filelist_populate_clist();
}

/*
 *-----------------------------------------------------------------------------
 * menu and button callbacks (private)
 *-----------------------------------------------------------------------------
 */ 

static void exit_cb(void)
{
	exit_gqview();
}

static void help_cb(void)
{
	help_window_show("documentation");
}

static void notes_cb(void)
{
	help_window_show("release_notes");
}

static void about_cb(void)
{
	show_about_window();
}

static void new_collect_cb(void)
{
	collection_window_new(NULL);
}

static void open_collect_cb(void)
{
	collection_dialog_load(NULL);
}

static void find_dupes_cb(void)
{
	dupe_window_new(DUPE_MATCH_NAME);
}

static void create_dir_cb(void)
{
	file_util_create_dir(current_path);
}

static void zoom_in_cb(void)
{
	main_image_adjust_zoom(get_zoom_increment());
}

static void zoom_out_cb(void)
{
	main_image_adjust_zoom(-get_zoom_increment());
}

static void zoom_1_1_cb(void)
{
	main_image_set_zoom(1.0);
}

static void zoom_fit_cb(void)
{
	main_image_set_zoom(0.0);
}

static void thumb_menu_cb(gpointer data, guint action, GtkWidget *widget)
{
	set_thumbnails(GTK_CHECK_MENU_ITEM(widget)->active);
}

static void thumb_button_cb(GtkWidget *widget, gpointer data)
{
	set_thumbnails(GTK_TOGGLE_BUTTON(widget)->active);
}

static void refresh_cb(void)
{
	filelist_refresh();
	main_image_reload();
}

static void refresh_filelist_cb(void)
{
	file_clist_highlight_unset();
	refresh_cb();
}

static void float_cb(void)
{
	toolwindow_float();
}

static void hide_cb(void)
{
	toolwindow_hide();
}

static void slideshow_cb(void)
{
	main_image_slideshow_toggle();
}

static void slideshow_start_cb(void)
{
	main_image_slideshow_start();
}

static void slideshow_stop_cb(void)
{
	main_image_slideshow_stop();
}

static void slideshow_pause_cb(void)
{
	main_image_slideshow_pause_toggle();
}

static void home_dir_cb(void)
{
	const gchar *path = homedir();
	if (path) filelist_change_to(path);
}

static void select_all_cb(void)
{
	file_select_all();
}

static void unselect_all_cb(void)
{
	file_unselect_all();
}

static void config_cb(void)
{
	show_config_window();
}

static void remove_thumb_cb(void)
{
	cache_maintain_home(FALSE);
}

static void full_screen_cb(void)
{
	main_image_full_screen_toggle();
}

static void wallpaper_image_cb(void)
{
	main_image_to_root();
}

static void alter_button_image_cb(GtkWidget *widget, gpointer data)
{
	AlterType type;

	type = GPOINTER_TO_INT(data);
	main_image_alter(type);
}

static void alter_menu_image_cb(gpointer data, guint action, GtkWidget *widget)
{
	main_image_alter((AlterType)action);
}

/*
 *-----------------------------------------------------------------------------
 * image menu callbacks (private)
 *-----------------------------------------------------------------------------
 */ 

static void copy_image_cb(void)
{
	main_image_full_screen_stop();
	file_util_copy(main_image_get_path(), NULL, NULL);
}

static void move_image_cb(void)
{
	main_image_full_screen_stop();
	file_util_move(main_image_get_path(), NULL, NULL);
}

static void rename_image_cb(void)
{
	main_image_full_screen_stop();
	file_util_rename(main_image_get_path(), NULL);
}

static void delete_image_cb(void)
{
	main_image_full_screen_stop();
	file_util_delete(main_image_get_path(), NULL);
}

/*
 *-----------------------------------------------------------------------------
 * file menu callbacks (private)
 *-----------------------------------------------------------------------------
 */ 

static void copy_file_cb(void)
{
	file_util_copy(NULL, file_get_selected_list(), NULL);
}

static void move_file_cb(void)
{
	file_util_move(NULL, file_get_selected_list(), NULL);
}

static void rename_file_cb(void)
{
	file_util_rename(NULL, file_get_selected_list());
}

static void delete_file_cb(void)
{
	file_util_delete(NULL, file_get_selected_list());
}

/*
 *-----------------------------------------------------------------------------
 * filelist popup menu callbacks (private)
 *-----------------------------------------------------------------------------
 */ 

static void popup_copy_file_cb(GtkWidget *widget, gpointer data)
{
	if (file_clicked_is_selected())
		{
		file_util_copy(NULL, file_get_selected_list(), NULL);
		}
	else
		{
		gchar *path = file_clicked_get_path();
		file_util_copy(path, NULL, NULL);
		g_free(path);
		}
}

static void popup_move_file_cb(GtkWidget *widget, gpointer data)
{
	if (file_clicked_is_selected())
		{
		file_util_move(NULL, file_get_selected_list(), NULL);
		}
	else
		{
		gchar *path = file_clicked_get_path();
		file_util_move(path, NULL, NULL);
		g_free(path);
		}
}

static void popup_rename_file_cb(GtkWidget *widget, gpointer data)
{
	if (file_clicked_is_selected())
		{
		file_util_rename(NULL, file_get_selected_list());
		}
	else
		{
		gchar *path = file_clicked_get_path();
		file_util_rename(path, NULL);
		g_free(path);
		}
}

static void popup_delete_file_cb(GtkWidget *widget, gpointer data)
{
	if (file_clicked_is_selected())
		{
		file_util_delete(NULL, file_get_selected_list());
		}
	else
		{
		gchar *path = file_clicked_get_path();
		file_util_delete(path, NULL);
		g_free(path);
		}
}

static void edit_image_cb(GtkWidget *widget, gpointer data)
{
	gint n = GPOINTER_TO_INT(data);
	start_editor_from_image(n);
}

static void edit_menu_list_cb(gpointer data, guint action, GtkWidget *widget)
{
	start_editor_from_list(action - MENU_EDIT_ACTION_OFFSET);
}

static void edit_full_cb(GtkWidget *widget, gpointer data)
{
	gint n = GPOINTER_TO_INT(data);
	main_image_full_screen_stop();
	start_editor_from_image(n);
}

static void popup_edit_list_cb(GtkWidget *widget, gpointer data)
{
	gint n = GPOINTER_TO_INT(data);

	if (file_clicked_is_selected())
		{
		start_editor_from_list(n);
		}
	else
		{
		gchar *path;
		path = file_clicked_get_path();
		start_editor_from_file(n, path);
		g_free(path);
		}
}

static void new_window_file_cb(GtkWidget *widget, gpointer data)
{
	gchar *path = file_clicked_get_path();
	view_window_new(path);
	g_free(path);
}

static void new_window_image_cb(GtkWidget *widget, gpointer data)
{
	view_window_new(main_image_get_path());
}

/*
 *-----------------------------------------------------------------------------
 * menu addition utilities (private)
 *-----------------------------------------------------------------------------
 */ 

static GtkWidget *add_menu_item(GtkWidget *menu, gchar *label, GtkAccelGroup *accel_group,
				guint accel_key, guint accel_mods, GtkSignalFunc func, gpointer data)
{
	GtkWidget *item;

	item = gtk_menu_item_new_with_label(label);
	gtk_widget_add_accelerator (item, "activate", accel_group, accel_key, accel_mods, GTK_ACCEL_VISIBLE);
	gtk_signal_connect (GTK_OBJECT (item), "activate",(GtkSignalFunc) func, data);
	gtk_menu_append(GTK_MENU(menu), item);
	gtk_widget_show(item);

	return item;
}

/*
 *-----------------------------------------------------------------------------
 * edit menu routines
 *-----------------------------------------------------------------------------
 */ 

static void add_edit_items(GtkWidget *menu, GtkSignalFunc func, GtkAccelGroup *accel_grp)
{
	gint i;
	for (i = 0; i < 8; i++)
		{
		if (editor_command[i] && strlen(editor_command[i]) > 0)
			{
			gchar *text;
			if (editor_name[i] && strlen(editor_name[i]) > 0)
				text = g_strdup_printf(_("in %s..."), editor_name[i]);
			else
				text = g_strdup(_("in (unknown)..."));
			if (accel_grp)
				add_menu_item(menu, text, accel_grp, i + 49, GDK_CONTROL_MASK, func, GINT_TO_POINTER(i));
			else
				menu_item_add(menu, text, func, GINT_TO_POINTER(i));
			g_free(text);
			}
		}
}

void update_edit_menus(GtkAccelGroup *accel_grp)
{
	gint i;

	/* main edit menu */

	for(i = 0; i < 8; i++)
		{
		GtkWidget *menu_item;
		menu_item = gtk_item_factory_get_item_by_action(gqview_fact, MENU_EDIT_ACTION_OFFSET + i);
		if (editor_command[i] && strlen(editor_command[i]) > 0)
			{
			gchar *text;
			if (editor_name[i] && strlen(editor_name[i]) > 0)
				{
				text = g_strdup_printf(_("in %s..."), editor_name[i]);
				}
			else
				{
				text = g_strdup(_("in (unknown)..."));
				}
			gtk_label_set_text(GTK_LABEL(GTK_BIN(menu_item)->child), text);
			g_free(text);
			gtk_widget_show(menu_item);
			}
		else
			{
			gtk_widget_hide(menu_item);
			}
		}
}

/*
 *-----------------------------------------------------------------------------
 * generic popup menu assembly routines
 *-----------------------------------------------------------------------------
 */

gpointer submenu_item_get_data(GtkWidget *menu)
{
	if (!menu->parent || !GTK_IS_MENU(menu->parent)) return NULL;

	return gtk_object_get_data(GTK_OBJECT(menu->parent), "submenu_data");
}

GtkWidget *submenu_add_edit(GtkWidget *menu, GtkWidget **menu_item, GtkSignalFunc func, gpointer data)
{
	GtkWidget *item;
	GtkWidget *submenu;

	item = menu_item_add(menu, _("Edit"), NULL, NULL);

	submenu = gtk_menu_new();
	gtk_object_set_data(GTK_OBJECT(submenu), "submenu_data", data);
	add_edit_items(submenu, func, NULL);

	gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), submenu);

	if (menu_item) *menu_item = item;

	return submenu;
}

gchar *sort_type_get_text(SortType method)
{
	switch (method)
		{
		case SORT_SIZE:
			return _("Sorted by size");
			break;
		case SORT_TIME:
			return _("Sorted by date");
			break;
		case SORT_NONE:
			return _("Unsorted");
			break;
		case SORT_PATH:
			return _("Sorted by path");
			break;
		case SORT_NUMBER:
			return _("Sorted by number");
			break;
		case SORT_NAME:
		default:
			return _("Sorted by name");
			break;
		}

	return "";
}

static void submenu_add_sort_item(GtkWidget *menu, GtkSignalFunc func, SortType type)
{
	menu_item_add(menu, sort_type_get_text(type), func, GINT_TO_POINTER((gint)type));
}

GtkWidget *submenu_add_sort(GtkWidget *menu, GtkSignalFunc func, gpointer data,
			    gint include_none, gint include_path)
{
	GtkWidget *submenu;

	submenu = gtk_menu_new();
	gtk_object_set_data(GTK_OBJECT(submenu), "submenu_data", data);

	submenu_add_sort_item(submenu, func, SORT_NAME);
#ifdef HAVE_STRVERSCMP
	submenu_add_sort_item(submenu, func, SORT_NUMBER);
#endif
	submenu_add_sort_item(submenu, func, SORT_TIME);
	submenu_add_sort_item(submenu, func, SORT_SIZE);
	if (include_path) submenu_add_sort_item(submenu, func, SORT_PATH);
	if (include_none) submenu_add_sort_item(submenu, func, SORT_NONE);

	if (menu)
		{
		GtkWidget *item;

		item = menu_item_add(menu, _("Sort"), NULL, NULL);
		gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), submenu);
		return item;
		}

	return submenu;
}

gchar *alter_type_get_text(AlterType type)
{
	switch (type)
		{
		case ALTER_ROTATE_90:
			return _("Rotate clockwise");
			break;
		case ALTER_ROTATE_90_CC:
			return _("Rotate counterclockwise");
			break;
		case ALTER_ROTATE_180:
			return _("Rotate 180");
			break;
		case ALTER_MIRROR:
			return _("Mirror");
			break;
		case ALTER_FLIP:
			return _("Flip");
			break;
		default:
			break;
		}

	return "";
}

static void submenu_add_alter_item(GtkWidget *menu, GtkSignalFunc func, AlterType type,
				   GtkAccelGroup *accel_group, guint accel_key, guint accel_mods)
{
	if (accel_group)
		{
		add_menu_item(menu, alter_type_get_text(type), accel_group,
			      accel_key, accel_mods, func, GINT_TO_POINTER((gint)type));
		
		}
	else
		{
		menu_item_add(menu, alter_type_get_text(type), func, GINT_TO_POINTER((gint)type));
		}
}

static GtkWidget *real_submenu_add_alter(GtkWidget *menu, GtkSignalFunc func, gpointer data,
					 GtkAccelGroup *accel_group)
{
	GtkWidget *submenu;

	submenu = gtk_menu_new();
	gtk_object_set_data(GTK_OBJECT(submenu), "submenu_data", data);

	submenu_add_alter_item(submenu, func, ALTER_ROTATE_90, accel_group, ']', 0);
	submenu_add_alter_item(submenu, func, ALTER_ROTATE_90_CC, accel_group, '[', 0);
	submenu_add_alter_item(submenu, func, ALTER_ROTATE_180, accel_group, 'R', GDK_SHIFT_MASK);
	submenu_add_alter_item(submenu, func, ALTER_MIRROR, accel_group, 'M', GDK_SHIFT_MASK);
	submenu_add_alter_item(submenu, func, ALTER_FLIP, accel_group, 'F', GDK_SHIFT_MASK);

	if (menu)
		{
		GtkWidget *item;

		item = menu_item_add(menu, _("Adjust"), NULL, NULL);
		gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), submenu);
		return item;
		}

	return submenu;
}

GtkWidget *submenu_add_alter(GtkWidget *menu, GtkSignalFunc func, gpointer data)
{
	return real_submenu_add_alter(menu, func, data, NULL);
}

/*
 *-----------------------------------------------------------------------------
 * recent open stuff
 *-----------------------------------------------------------------------------
 */

static void recent_menu_cb(GtkWidget *widget, gpointer data)
{
	gint n;
	gchar *path;

	n = GPOINTER_TO_INT(data);

	path = g_list_nth_data(history_list_get_by_key("recent"), n);

	if (!path) return;

	/* make a copy of it */
	path = g_strdup(path);
	collection_window_new(path);
	g_free(path);
}

static void recent_menu_update(void)
{
	GtkWidget *menu;
	GtkWidget *recent;
	GList *list;
	gint n;

	list = history_list_get_by_key("recent");
	n = 0;

	menu = gtk_menu_new();

	while(list)
		{
		menu_item_add(menu, filename_from_path((gchar *)list->data),
				    recent_menu_cb, GINT_TO_POINTER(n));
		list = list->next;
		n++;
		}

	if (n == 0)
		{
		menu_item_add(menu, _("Empty"), NULL, NULL);
		}

	recent = gtk_item_factory_get_item(gqview_fact, "/File/Open recent");
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(recent), menu);
	gtk_widget_set_sensitive(recent, (n != 0));
}

void recent_menu_add_path(gchar *path)
{
	if (!path) return;

	history_list_add_to_key("recent", path, recent_list_max);

	recent_menu_update();
}

/*
 *-----------------------------------------------------------------------------
 * menu bar setup routines
 *-----------------------------------------------------------------------------
 */ 

static GtkItemFactoryEntry gqview_menu_items[] =
{
	{ N_("/_File"),				NULL,		NULL,			0, "<Branch>" },
	{ N_("/File/tear1"),			NULL,		NULL,			0, "<Tearoff>" },
	{ N_("/File/_New collection"),		"C",		new_collect_cb,		0, NULL },
	{ N_("/File/_Open collection..."),	"O",		open_collect_cb,	0, NULL },
	{ N_("/File/Open _recent"),		NULL,		NULL,			0, NULL },
	{ N_("/File/sep1"),			NULL,		NULL,			0, "<Separator>" },
	{ N_("/File/_Find duplicates..."),	"D",		find_dupes_cb,		0, NULL },
	{ N_("/File/sep2"),			NULL,		NULL,			0, "<Separator>" },
	{ N_("/File/Create _Dir..."),		"<control>N",	create_dir_cb,		0, NULL },
	{ N_("/File/sep3"),			NULL,		NULL,			0, "<Separator>" },
	{ N_("/File/_Copy..."),			"<control>C",	copy_file_cb,		0, NULL },
	{ N_("/File/_Move..."),			"<control>M",	move_file_cb,		0, NULL },
	{ N_("/File/_Rename..."),		"<control>R",	rename_file_cb,		0, NULL },
	{ N_("/File/_Delete..."),		"<control>D",	delete_file_cb,		0, NULL },
	{ N_("/File/sep4"),			NULL,		NULL,			0, "<Separator>" },
	{ N_("/File/E_xit"),			"<control>Q",	exit_cb,		0, NULL },

	{ N_("/_Edit"),				NULL,		NULL,			0, "<Branch>" },
	{ N_("/Edit/tear1"),			NULL,		NULL,			0, "<Tearoff>" },
	{ N_("/Edit/editor1"),			"<control>1",	edit_menu_list_cb,	MENU_EDIT_ACTION_OFFSET + 0, NULL },
	{ N_("/Edit/editor2"),			"<control>2",	edit_menu_list_cb,	MENU_EDIT_ACTION_OFFSET + 1, NULL },
	{ N_("/Edit/editor3"),			"<control>3",	edit_menu_list_cb,	MENU_EDIT_ACTION_OFFSET + 2, NULL },
	{ N_("/Edit/editor4"),			"<control>4",	edit_menu_list_cb,	MENU_EDIT_ACTION_OFFSET + 3, NULL },
	{ N_("/Edit/editor5"),			"<control>5",	edit_menu_list_cb,	MENU_EDIT_ACTION_OFFSET + 4, NULL },
	{ N_("/Edit/editor6"),			"<control>6",	edit_menu_list_cb,	MENU_EDIT_ACTION_OFFSET + 5, NULL },
	{ N_("/Edit/editor7"),			"<control>7",	edit_menu_list_cb,	MENU_EDIT_ACTION_OFFSET + 6, NULL },
	{ N_("/Edit/editor8"),			"<control>8",	edit_menu_list_cb,	MENU_EDIT_ACTION_OFFSET + 7, NULL },
	{ N_("/Edit/sep1"),			NULL,		NULL,			0, "<Separator>" },
	{ N_("/Edit/_Adjust"),			NULL,		NULL,			0, "<Branch>" },
	{ N_("/Edit/Adjust/tear1"),		NULL,		NULL,			0, "<Tearoff>" },
	{ N_("/Edit/Adjust/_Rotate clockwise"),		"bracketright",	alter_menu_image_cb,	ALTER_ROTATE_90, NULL },
	{ N_("/Edit/Adjust/Rotate _counterclockwise"),	"bracketleft",	alter_menu_image_cb,	ALTER_ROTATE_90_CC, NULL },
	{ N_("/Edit/Adjust/Rotate 1_80"),	"<shift>R",	alter_menu_image_cb,	ALTER_ROTATE_180, NULL },
	{ N_("/Edit/Adjust/_Mirror"),		"<shift>M",	alter_menu_image_cb,	ALTER_MIRROR, NULL },
	{ N_("/Edit/Adjust/_Flip"),		"<shift>F",	alter_menu_image_cb,	ALTER_FLIP, NULL },
	{ N_("/Edit/sep2"),			NULL,		NULL,			0, "<Separator>" },
	{ N_("/Edit/Select _all"),		"<control>A",	select_all_cb,		0, NULL },
	{ N_("/Edit/Select _none"),	"<control><shift>A",	unselect_all_cb,	0, NULL },
	{ N_("/Edit/sep3"),			NULL,		NULL,			0, "<Separator>" },
	{ N_("/Edit/_Options..."),		"<control>O",	config_cb,		0, NULL },
	{ N_("/Edit/_Remove old thumbnails"),	"<control>T",	remove_thumb_cb,	0, NULL },
	{ N_("/Edit/sep4"),			NULL,		NULL,			0, "<Separator>" },
	{ N_("/Edit/Set as _wallpaper"),	"<control>W",	wallpaper_image_cb,	0, NULL },

	{ N_("/_View"),				NULL,		NULL,			0, "<Branch>" },
	{ N_("/View/tear1"),			NULL,		NULL,			0, "<Tearoff>" },
	{ N_("/View/Zoom _in"),			"equal",	zoom_in_cb,		0, NULL },
	{ N_("/View/Zoom _out"),		"minus",	zoom_out_cb,		0, NULL },
	{ N_("/View/Zoom _1:1"),		"Z",		zoom_1_1_cb,		0, NULL },
	{ N_("/View/_Zoom to fit"),		"X",		zoom_fit_cb,		0, NULL },
	{ N_("/View/sep1"),			NULL,		NULL,			0, "<Separator>" },

	{ N_("/View/F_ull screen"),		"V",		full_screen_cb,		0, NULL },
	{ N_("/View/_Thumbnails"),		"T",		thumb_menu_cb,		0, "<CheckItem>" },

	{ N_("/View/sep2"),			NULL,		NULL,			0, "<Separator>" },
	{ N_("/View/_Refresh Lists"),		"R",		refresh_cb,		0, NULL },
	{ N_("/View/(Un)_Float file list"),	"F",		float_cb,		0, NULL },
	{ N_("/View/(Un)_Hide file list"),	"H",		hide_cb,		0, NULL },
	{ N_("/View/sep3"),			NULL,		NULL,			0, "<Separator>" },
	{ N_("/View/Toggle _slideshow"),	"S",		slideshow_cb,		0, NULL },

	{ N_("/_Help"),				NULL,		NULL,			0, "<Branch>" },
	{ N_("/Help/tear1"),			NULL,		NULL,			0, "<Tearoff>" },
	{ N_("/Help/_Keyboard shortcuts"),	"F1",		help_cb,		0, NULL},
	{ N_("/Help/_Release notes"),		NULL,		notes_cb,		0, NULL},
	{ N_("/Help/sep1"),			NULL,		NULL,			0, "<Separator>" },
	{ N_("/Help/_About"),			NULL,		about_cb,		0, NULL },
};

static int gqview_nmenu_items = sizeof (gqview_menu_items) / sizeof (gqview_menu_items[0]);

static gchar *menu_translate(const gchar *path, gpointer data)
{
	return _(path);
}

GtkWidget *create_menu_bar(GtkAccelGroup *accel_grp)
{
	gqview_fact = gtk_item_factory_new(GTK_TYPE_MENU_BAR, "<main>", accel_grp);
	gtk_item_factory_set_translate_func(gqview_fact, menu_translate, NULL, NULL);

	gtk_item_factory_create_items(gqview_fact, gqview_nmenu_items, gqview_menu_items, NULL);

	recent_menu_update();

	gtk_check_menu_item_set_state(GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item(gqview_fact, "/View/Thumbnails")), thumbnails_enabled);

	return gtk_item_factory_get_widget (gqview_fact, "<main>");
}

/*
 *-----------------------------------------------------------------------------
 * pop-up menu file / dir list
 *-----------------------------------------------------------------------------
 */ 

static void popup_dir_slide_cb(GtkWidget *widget, gpointer data)
{
	gchar *path;

	path = dir_get_path(GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(dir_clist), "click_row")));
	if (!path) return;

	filelist_change_to(path);
	main_image_slideshow_stop();
	main_image_slideshow_start();

	g_free(path);
}

static void popup_dir_slide_rec_cb(GtkWidget *widget, gpointer data)
{
	gchar *path;
	GList *list;

	path = dir_get_path(GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(dir_clist), "click_row")));
	if (!path) return;

	list = path_list_recursive(path);

	main_image_slideshow_stop();
	main_image_slideshow_start_from_list(list);
	
	g_free(path);
}

static void popup_dir_find_dupe(gint recursive)
{
	DupeWindow *dw;
	gchar *path;
	GList *list = NULL;

	path = dir_get_path(GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(dir_clist), "click_row")));
	if (!path) return;

	if (recursive)
		{
		list = g_list_append(list, g_strdup(path));
		}
	else
		{
		path_list(path, &list, NULL);
		list = path_list_filter(list, FALSE);
		}

	dw = dupe_window_new(DUPE_MATCH_NAME);
	dupe_window_add_files(dw, list, recursive);

	path_list_free(list);
	g_free(path);
}

static void popup_dir_find_dupe_cb(GtkWidget *widget, gpointer data)
{
	popup_dir_find_dupe(FALSE);
}

static void popup_dir_find_dupe_rec_cb(GtkWidget *widget, gpointer data)
{
	popup_dir_find_dupe(TRUE);
}

GtkWidget *menu_popup_dir_list(gint active)
{
	GtkWidget *menu;
	GtkWidget *item;

	menu = popup_menu_short_lived();

	item = menu_item_add(menu, _("Slideshow"), popup_dir_slide_cb, NULL);
	gtk_widget_set_sensitive(item, active);
	item = menu_item_add(menu, _("Slideshow recursive"), popup_dir_slide_rec_cb, NULL);
	gtk_widget_set_sensitive(item, active);

	menu_item_add_divider(menu);
	item = menu_item_add(menu, _("Find duplicates..."), popup_dir_find_dupe_cb, NULL);
	gtk_widget_set_sensitive(item, active);
	item = menu_item_add(menu, _("Find duplicates recursive..."), popup_dir_find_dupe_rec_cb, NULL);
	gtk_widget_set_sensitive(item, active);

	menu_item_add_divider(menu);
	menu_item_add(menu, _("Refresh"), refresh_cb, NULL);

	return menu;
}

GtkWidget *menu_popup_file_list(gint active)
{
	GtkWidget *menu;
	GtkWidget *item;

	menu = popup_menu_short_lived();

	submenu_add_edit(menu, &item, popup_edit_list_cb, NULL);
	gtk_widget_set_sensitive(item, active);

	item = menu_item_add(menu, _("View in new window"), new_window_file_cb, NULL);
	gtk_widget_set_sensitive(item, active);

	menu_item_add_divider(menu);
	item = menu_item_add(menu, _("Copy..."), popup_copy_file_cb, NULL);
	gtk_widget_set_sensitive(item, active);
	item = menu_item_add(menu, _("Move..."), popup_move_file_cb, NULL);
	gtk_widget_set_sensitive(item, active);
	item = menu_item_add(menu, _("Rename..."), popup_rename_file_cb, NULL);
	gtk_widget_set_sensitive(item, active);
	item = menu_item_add(menu, _("Delete..."), popup_delete_file_cb, NULL);
	gtk_widget_set_sensitive(item, active);

	menu_item_add_divider(menu);
	menu_item_add(menu, _("Refresh"), refresh_filelist_cb, NULL);

	return menu;
}

/*
 *-----------------------------------------------------------------------------
 * pop-up menu image routines
 *-----------------------------------------------------------------------------
 */ 

GtkWidget *menu_popup_image(void)
{
	GtkWidget *menu;
	GtkWidget *item;
	GtkWidget *submenu;
	const gchar *path;
	gint fullscreen;

	path = main_image_get_path();
	fullscreen = main_image_full_screen_active();

	menu = popup_menu_short_lived();

	menu_item_add(menu, _("Zoom in"), zoom_in_cb, NULL);
	menu_item_add(menu, _("Zoom out"), zoom_out_cb, NULL);
	menu_item_add(menu, _("Zoom 1:1"), zoom_1_1_cb, NULL);
	menu_item_add(menu, _("Fit image to window"), zoom_fit_cb, NULL);
	menu_item_add_divider(menu);

	if (!fullscreen)
		{
		submenu = submenu_add_edit(menu, &item, edit_image_cb, NULL);
		}
	else
		{
		submenu = submenu_add_edit(menu, &item, edit_full_cb, NULL);
		}
	if (!path) gtk_widget_set_sensitive(item, FALSE);
	menu_item_add_divider(submenu);
	menu_item_add(submenu, _("Set as wallpaper"), wallpaper_image_cb, NULL);

	item = submenu_add_alter(menu, alter_button_image_cb, NULL);

	item = menu_item_add(menu, _("View in new window"), new_window_image_cb, NULL);
	if (!path || fullscreen) gtk_widget_set_sensitive(item, FALSE);

	menu_item_add_divider(menu);

	item = menu_item_add(menu, _("Copy..."), copy_image_cb, NULL);
	if (!path) gtk_widget_set_sensitive(item, FALSE);
	item = menu_item_add(menu, _("Move..."), move_image_cb, NULL);
	if (!path) gtk_widget_set_sensitive(item, FALSE);
	item = menu_item_add(menu, _("Rename..."), rename_image_cb, NULL);
	if (!path) gtk_widget_set_sensitive(item, FALSE);
	item = menu_item_add(menu, _("Delete..."), delete_image_cb, NULL);
	if (!path) gtk_widget_set_sensitive(item, FALSE);

	menu_item_add_divider(menu);

	if (main_image_slideshow_active())
		{
		menu_item_add(menu, _("Stop slideshow"), slideshow_stop_cb, NULL);
		if (main_image_slideshow_paused())
			{
			item = menu_item_add(menu, _("Continue slideshow"), slideshow_pause_cb, NULL);
			}
		else
			{
			item = menu_item_add(menu, _("Pause slideshow"), slideshow_pause_cb, NULL);
			}
		}
	else
                {
		menu_item_add(menu, _("Start slideshow"), slideshow_start_cb, NULL);
		item = menu_item_add(menu, _("Pause slideshow"), slideshow_pause_cb, NULL);
		gtk_widget_set_sensitive(item, FALSE);
		}

	if (!fullscreen)
		{
		menu_item_add(menu, _("Full screen"), full_screen_cb, NULL);
		}
	else
		{
		menu_item_add(menu, _("Exit full screen"), full_screen_cb, NULL);
		}

	menu_item_add_divider(menu);

	item = menu_item_add(menu, _("(Un)Hide file list"), hide_cb, NULL);
	if (fullscreen) gtk_widget_set_sensitive(item, FALSE);

	return menu;
}

/*
 *-----------------------------------------------------------------------------
 * toolbar routines
 *-----------------------------------------------------------------------------
 */ 

static void add_button_to_bar(GtkWidget *hbox, gchar **pixmap_data,
			      GtkTooltips *tooltips, gchar *tip_text,
			      GtkSignalFunc func, gpointer data)
{
	GtkWidget *button;
	GtkWidget *pixmapwid;
	GdkPixbuf *pixbuf;
	GdkPixmap *pixmap;
	GdkBitmap *mask;

	button = gtk_button_new ();
	gtk_signal_connect (GTK_OBJECT (button), "clicked",(GtkSignalFunc) func, thumb_button);
	gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
	gtk_widget_show (button);
	gtk_tooltips_set_tip (tooltips, button, tip_text, NULL);

	pixbuf = gdk_pixbuf_new_from_xpm_data((const char **)pixmap_data);
	gdk_pixbuf_render_pixmap_and_mask(pixbuf, &pixmap, &mask, 128);
	gdk_pixbuf_unref(pixbuf);

	pixmapwid = gtk_pixmap_new(pixmap, mask);
	gtk_widget_show(pixmapwid);
	gtk_container_add(GTK_CONTAINER(button), pixmapwid);

	gdk_pixmap_unref(pixmap);
	if (mask) gdk_bitmap_unref(mask);
}

GtkWidget *create_button_bar(GtkTooltips *tooltips)
{
	GtkWidget *hbox;
	GtkStyle *style;
	GtkWidget *pixmapwid;
	GdkPixmap *pixmap;
	GdkBitmap *mask;

	style = gtk_widget_get_style(mainwindow);
	hbox = gtk_hbox_new(FALSE, 0);

	thumb_button = gtk_toggle_button_new ();
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(thumb_button), thumbnails_enabled);
	gtk_signal_connect (GTK_OBJECT (thumb_button), "clicked",(GtkSignalFunc) thumb_button_cb, thumb_button);
	gtk_box_pack_start (GTK_BOX (hbox), thumb_button, FALSE, FALSE, 0);
	gtk_widget_show (thumb_button);
	gtk_tooltips_set_tip (tooltips, thumb_button, _("Create thumbnails"), NULL);

	pixmap = gdk_pixmap_create_from_xpm_d(mainwindow->window,  &mask,
                 &style->bg[GTK_STATE_NORMAL], (gchar **)icon_thumb_xpm );
	pixmapwid = gtk_pixmap_new(pixmap, mask);
	gtk_widget_show(pixmapwid);
	gtk_container_add(GTK_CONTAINER(thumb_button), pixmapwid);

	add_button_to_bar(hbox, (gchar **)icon_home_xpm,
			 tooltips, _("Change to home directory"), home_dir_cb, NULL);
	add_button_to_bar(hbox, (gchar **)icon_reload_xpm,
			 tooltips, _("Refresh file list"), refresh_cb, NULL);
	add_button_to_bar(hbox, (gchar **)icon_zoom_in_xpm,
			 tooltips, _("Zoom in"), zoom_in_cb, NULL);
	add_button_to_bar(hbox, (gchar **)icon_zoom_out_xpm,
			 tooltips, _("Zoom out"), zoom_out_cb, NULL);
	add_button_to_bar(hbox, (gchar **)icon_zoom_fit_xpm,
			 tooltips, _("Fit image to window"), zoom_fit_cb, NULL);
	add_button_to_bar(hbox, (gchar **)icon_zoom_norm_xpm,
			 tooltips, _("Set zoom 1:1"), zoom_1_1_cb, NULL);
	add_button_to_bar(hbox, (gchar **)icon_config_xpm,
			 tooltips, _("Configure options"), config_cb, NULL);
	add_button_to_bar(hbox, (gchar **)icon_float_xpm,
			 tooltips, _("Float Controls"), float_cb, NULL);

	return hbox;
}

