"""HTTP server base class.

Note: the class in this module doesn't implement any HTTP request; see
SimpleHTTPServer for simple implementations of GET, HEAD and POST
(including CGI scripts).

Contents:

- BaseHTTPRequestHandler: HTTP request handler base class
- test: test function

XXX To do:

- send server version
- log requests even later (to capture byte count)
- log user-agent header and other interesting goodies
- send error log to separate file
- are request names really case sensitive?

"""


# See also:
#
# HTTP Working Group                                        T. Berners-Lee
# INTERNET-DRAFT                                            R. T. Fielding
# <draft-ietf-http-v10-spec-00.txt>                     H. Frystyk Nielsen
# Expires September 8, 1995                                  March 8, 1995
#
# URL: http://www.ics.uci.edu/pub/ietf/http/draft-ietf-http-v10-spec-00.txt


# Log files
# ---------
# 
# Here's a quote from the NCSA httpd docs about log file format.
# 
# | The logfile format is as follows. Each line consists of: 
# | 
# | host rfc931 authuser [DD/Mon/YYYY:hh:mm:ss] "request" ddd bbbb 
# | 
# |        host: Either the DNS name or the IP number of the remote client 
# |        rfc931: Any information returned by identd for this person,
# |                - otherwise. 
# |        authuser: If user sent a userid for authentication, the user name,
# |                  - otherwise. 
# |        DD: Day 
# |        Mon: Month (calendar name) 
# |        YYYY: Year 
# |        hh: hour (24-hour format, the machine's timezone) 
# |        mm: minutes 
# |        ss: seconds 
# |        request: The first line of the HTTP request as sent by the client. 
# |        ddd: the status code returned by the server, - if not available. 
# |        bbbb: the total number of bytes sent,
# |              *not including the HTTP/1.0 header*, - if not available 
# | 
# | You can determine the name of the file accessed through request.
# 
# (Actually, the latter is only true if you know the server configuration
# at the time the request was made!)


__version__ = "0.2"


import sys
import time
import socket # For gethostbyaddr()
import string
import rfc822
import mimetools
import select
from LimitedTCPServer import LimitedTCPServer, ClientHandlerStub
from Selector import Selector
from SockStream import SockStream

# Default error message
DEFAULT_ERROR_MESSAGE = """\
<head>
<title>Error response</title>
</head>
<body>
<h1>Error response</h1>
<p>Error code %(code)d.
<p>Message: %(message)s.
<p>Error code explanation: %(code)s = %(explain)s.
</body>
"""

PAGE = """\
<head>
<title>A page</title>
</head>
<body>
Hello, how are you ?<br>
You requested this:
<dl>
<dt>Command:</dt><dd>%(command)s</dd>
<dt>Path:</dt><dd>%(path)s</dd>
<dt>Address:</dt><dd>%(client)s</dd>
</dl>
<a href="plus/">+</a> | <a href="/">up</a>
</body>
"""



class HTTPServer(LimitedTCPServer):
		
	def createClientHandler(self, sock, peer):
		return HTTPClientHandler(self, sock, peer)


class HTTPClientHandler(ClientHandlerStub):
	# The Python system version, truncated to its first component.
	sys_version = "Python/" + string.split(sys.version)[0]

	# The server software version.  You may want to override this.
	# The format is multiple whitespace-separated strings,
	# where each string is of the form name[/version].
	server_version = "BaseHTTP/" + __version__

	def __init__(self, sock, addr, timeout = 30):
		self.Sock = sock
		self.rfile = self.Sock.makefile('r',0)
		self.wfile = self.Sock.makefile('w',0)
		self.client_address = addr
		self.Headers = {}
		self.Request = None
		self.request_version = 'HTTP/0.9'
		self.requestline = ''
		self.command = '(unknown command)'
		self.path = ''
		self.Str = SockStream(self.Sock, eom='\n', 
					zing='\0', zong='\0', probe='\0')
		self.TimeOut = timeout
		
	def rejectConnection(self):
		self.send_error(502)

	def disconnect(self):
		self.Sock.close()
		self.Str = None
		#print 'disconnected'

	def run(self):
		#print 'run...'
		tstart = time.time()
		need_more = 1
		while (self.TimeOut == None or time.time() < tstart + self.TimeOut) \
					and need_more:
			need_more = self.read_more()
		if self.RequestReceived:
			mname = 'do_' + self.command
			if not hasattr(self, mname):
				self.send_error(501, "Unsupported method (%s)" % `self.command`)
			else:
				method = getattr(self, mname)
				method()
		self.disconnect()

	def read_more(self):
		#print 'read_more...'
		r,w,e = select.select([self.Sock], [], [], 10)
		if not self.Sock in r:	return 1
		self.Str.readMore()
		while self.Str.msgReady():
			msg = self.Str.getMsg()
			#print 'msg=<%s>' % msg
			if msg:
				msg = string.strip(msg)

				if not self.Request:
					# first line is request
					requestline = msg
					self.requestline = msg
					version = 'HTTP/0.9'
					command = None
					path = None
					words = string.split(requestline)
					if len(words) == 3:
						[command, path, version] = words
						if version[:5] != 'HTTP/':
							self.send_error(400, "Bad request version (%s)" % `version`)
							return 0
					elif len(words) == 2:
						[command, path] = words
						if command != 'GET':
							self.send_error(400,
									"Bad HTTP/0.9 request type (%s)" % `command`)
							return 0
					else:
						self.send_error(400, "Bad request syntax (%s)" % `requestline`)
						return 0
					self.command, self.path, self.request_version = command, path, version
					self.Request = (self.command, self.path, self.request_version)
				else:
					# headers
					if not msg:
						# end of request
						self.RequestReceived = 1
						return 0
					words = string.split(msg, ' ', 1)
					if len(words) != 2:
						self.send_error(400, "Bad mime header syntax (%s)" % (msg,))
						return 0
					hdr = words[0]
					if hdr[-1] != ':' or hdr == ':':
						self.send_error(400, "Bad mime header syntax (%s)" % (msg,))
						return 0
					hdr = hdr[:-1]
					self.Headers[hdr] = words[1]
		return 1

	def doReadOld(self, fd, sel):
		"""Handle a single HTTP request.

		You normally don't need to override this method; see the class
		__doc__ string for information on how to handle specific HTTP
		commands such as GET and POST.

		"""

		if fd != self.Sock.fileno():    return
		#print 'doRead...'
		self.raw_requestline = self.rfile.readline()
		requestline = self.raw_requestline
		#print 'Rcvd: <%s>' % requestline
		if not requestline:
			#print 'disconnect'
			self.disconnect()
			return

		self.request_version = version = "HTTP/0.9" # Default

		if requestline[-2:] == '\r\n':
			requestline = requestline[:-2]
		elif requestline[-1:] == '\n':
			requestline = requestline[:-1]
		self.requestline = requestline
		words = string.split(requestline)
		if len(words) == 3:
			[command, path, version] = words
			if version[:5] != 'HTTP/':
				self.send_error(400, "Bad request version (%s)" % `version`)
				return
		elif len(words) == 2:
			[command, path] = words
			if command != 'GET':
				self.send_error(400,
						"Bad HTTP/0.9 request type (%s)" % `command`)
				return
		else:
			self.send_error(400, "Bad request syntax (%s)" % `requestline`)
			return
		self.command, self.path, self.request_version = command, path, version
		self.headers = self.MessageClass(self.rfile, 0)
		mname = 'do_' + command
		if not hasattr(self, mname):
			self.send_error(501, "Unsupported method (%s)" % `command`)
			return
		method = getattr(self, mname)
		method()

	def send_error(self, code, message=None):
		"""Send and log an error reply.

		Arguments are the error code, and a detailed message.
		The detailed message defaults to the short entry matching the
		response code.

		This sends an error response (so it must be called before any
		output has been generated), logs the error, and finally sends
		a piece of HTML explaining the error to the user.

		"""

		try:
			short, long = self.responses[code]
		except KeyError:
			short, long = '???', '???'
		if not message:
			message = short
		explain = long
		self.log_error("code %d, message %s", code, message)
		self.send_response(code, message)
		self.end_headers()
		self.wfile.write(self.error_message_format %
				 {'code': code,
				  'message': message,
				  'explain': explain})

	error_message_format = DEFAULT_ERROR_MESSAGE

	def send_response(self, code, message=None):
		"""Send the response header and log the response code.

		Also send two standard headers with the server software
		version and the current date.

		"""
		self.log_request(code)
		if message is None:
			if self.responses.has_key(code):
				message = self.responses[code][0]
			else:
				message = ''
		if self.request_version != 'HTTP/0.9':
			self.wfile.write("%s %s %s\r\n" %
					 (self.protocol_version, str(code), message))
		self.send_header('Server', self.version_string())
		self.send_header('Date', self.date_time_string())

	def send_header(self, keyword, value):
		"""Send a MIME header."""
		if self.request_version != 'HTTP/0.9':
			self.wfile.write("%s: %s\r\n" % (keyword, value))

	def end_headers(self):
		"""Send the blank line ending the MIME headers."""
		if self.request_version != 'HTTP/0.9':
			self.wfile.write("\r\n")

	def log_request(self, code='-', size='-'):
		"""Log an accepted request.

		This is called by send_reponse().

		"""

		self.log_message('"%s" %s %s',
				 self.requestline, str(code), str(size))

	def log_error(self, *args):
		"""Log an error.

		This is called when a request cannot be fulfilled.  By
		default it passes the message on to log_message().

		Arguments are the same as for log_message().

		XXX This should go to the separate error log.

		"""

		apply(self.log_message, args)

	def log_message(self, format, *args):
		"""Log an arbitrary message.

		This is used by all other logging functions.  Override
		it if you have specific logging wishes.

		The first argument, FORMAT, is a format string for the
		message to be logged.  If the format string contains
		any % escapes requiring parameters, they should be
		specified as subsequent arguments (it's just like
		printf!).

		The client host and current date/time are prefixed to
		every message.

		"""

		sys.stderr.write("%s - - [%s] %s\n" %
				 (self.address_string(),
				  self.log_date_time_string(),
				  format%args))

	def version_string(self):
		"""Return the server software version string."""
		return self.server_version + ' ' + self.sys_version

	def date_time_string(self):
			"""Return the current date and time formatted for a message header."""
			now = time.time()
			year, month, day, hh, mm, ss, wd, y, z = time.gmtime(now)
			s = "%s, %02d %3s %4d %02d:%02d:%02d GMT" % (
				self.weekdayname[wd],
				day, self.monthname[month], year,
				hh, mm, ss)
			return s

	def log_date_time_string(self):
			"""Return the current time formatted for logging."""
			now = time.time()
			year, month, day, hh, mm, ss, x, y, z = time.localtime(now)
			s = "%02d/%3s/%04d %02d:%02d:%02d" % (
				day, self.monthname[month], year, hh, mm, ss)
			return s

	weekdayname = ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun']

	monthname = [None,
		 'Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun',
		 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec']

	def address_string(self):
			"""Return the client address formatted for logging.

			This version looks up the full hostname using gethostbyaddr(),
			and tries to find a name that contains at least one dot.

			"""

			(host, port) = self.client_address
			try:
				name, names, addresses = socket.gethostbyaddr(host)
			except socket.error, msg:
				return host
			names.insert(0, name)
			for name in names:
				if '.' in name: return name
			return names[0]


	# Essentially static class variables

	# The version of the HTTP protocol we support.
	# Don't override unless you know what you're doing (hint: incoming
	# requests are required to have exactly this version string).
	protocol_version = "HTTP/1.0"

	# The Message-like class used to parse headers
	MessageClass = mimetools.Message

	# Table mapping response codes to messages; entries have the
	# form {code: (shortmessage, longmessage)}.
	# See http://www.w3.org/hypertext/WWW/Protocols/HTTP/HTRESP.html
	responses = {
		200: ('OK', 'Request fulfilled, document follows'),
		201: ('Created', 'Document created, URL follows'),
		202: ('Accepted',
			  'Request accepted, processing continues off-line'),
		203: ('Partial information', 'Request fulfilled from cache'),
		204: ('No response', 'Request fulfilled, nothing follows'),

		301: ('Moved', 'Object moved permanently -- see URI list'),
		302: ('Found', 'Object moved temporarily -- see URI list'),
		303: ('Method', 'Object moved -- see Method and URL list'),
		304: ('Not modified',
			  'Document has not changed singe given time'),

		400: ('Bad request',
			  'Bad request syntax or unsupported method'),
		401: ('Unauthorized',
			  'No permission -- see authorization schemes'),
		402: ('Payment required',
			  'No payment -- see charging schemes'),
		403: ('Forbidden',
			  'Request forbidden -- authorization will not help'),
		404: ('Not found', 'Nothing matches the given URI'),

		500: ('Internal error', 'Server got itself in trouble'),
		501: ('Not implemented',
			  'Server does not support this operation'),
		502: ('Service temporarily overloaded',
			  'The server cannot process the request due to a high load'),
		503: ('Gateway timeout',
			  'The gateway server did not receive a timely response'),

	}

	def do_GET(self):
		self.send_response(200)
		self.send_header("Content-type", 'text/html')
		self.end_headers()
		self.wfile.write(PAGE % 
			{	'path':self.path,
				'client':self.client_address,
				'command':self.command
			})
		self.disconnect()


def test():
    """Test the HTTP request handler class.

    This runs an HTTP server on port 8000 (or the first command line
    argument).

    """

    if sys.argv[1:]:
	port = string.atoi(sys.argv[1])
    else:
	port = 8000

    sel = Selector()
    httpd = HTTPServer(port, sel)

    print "Serving HTTP on port", port, "..."
    while 1:
		sel.select(100)


if __name__ == '__main__':
    test()
