#
# @(#) $Id: exec.py,v 1.12 2003/08/20 18:58:57 ivm Exp $
#
# Name: exec.py
#
# This is a script which can be used to emulate LSF-style
# job submission with interactive capabilities.
#
# Usage: python exec.py 
#	-q <queue> [-I] [-v] [-n <numproc>] [-p <proctype>] 
#	[-w <seconds or -1>] [-h <hold-date-time>]
#	[-r <resources>] [-i <priority increment>]
#	[-s <section-name>] [-o <stdout>] [-e <stderr>]
#	[-a (pack|spread)] [-N "<node> ..."]
#	<command> [<argument> ...]
#
# exec.py submits single-section FBSNG job that
# will execute the specified command. If requested,
# exec.py will wait for the section to complete and exit with
# section exit code. User can specify additional parameters
# such as process type, hold time, extra resources and
# FBSNG queue priority increment.
# If requested, the script will print section parameters as
# JDF and report where and when individual processes start.
#
# $Log: exec.py,v $
# Revision 1.12  2003/08/20 18:58:57  ivm
# Implemented CPU power, round-robin-over-users scheduling inside queuei,
# other minor things.
#
# Revision 1.11  2001/11/05 19:22:03  ivm
# Fixed some bugs
# Made inetractive spawner independent of UPS
#
# Revision 1.10  2001/10/17 17:31:16  ivm
# Implemented -m (mail-to) option
#
# Revision 1.9  2001/06/12 19:55:02  ivm
# Added RCS records
#
#

from FBS_API import *
from Selector import *
from TCPServer import *
from SockStream import *
import socket
import string
import getopt
import sys
import Parser
import time
import os
import tty

usage = """
exec.py -q <queue> [-I] [-v] [-n <numproc>] [-p <proctype>] 
	[-w <seconds or -1>] [-h <hold-date-time>]
	[-r <resources>] [-i <priority increment>] 
	[-s <section-name>] [-o <stdout>] [-e <stderr>]
	[-a (pack|spread)] [-N "<node> ..."] [-m <e-mail address>]
	[-C]
	<command> [<argument> ...]
"""

SpawnerCmd = '#interactive'

class	ProcessConnection:
	def __init__(self, srv, procno, peer, sock):
		self.ProcNo = procno
		self.Sock = sock
		self.Connected = 1
		self.Srv = srv

	def doRead(self, fd, sel):
		#print 'Proc#%d: doRead()' % self.ProcNo
		if self.Sock.fileno() != fd:
			return
		eof = 0
		try:	data = self.Sock.recv(1000)
		except:
			eof = 1
		else:
			if not data:	eof = 1
		if eof:
			self.Srv.procDisconnected(self.ProcNo)
			self.Sock.close()
			return
		self.Srv.dataFromProc(self.ProcNo, data)

	def send(self, data):
		self.Sock.send(data)

	def register(self, sel):	
		sel.register(self, rd=self.Sock.fileno())

	def unregister(self, sel):	
		sel.unregister(rd=self.Sock.fileno())

	def disconnect(self):
		self.Sock.close()

class	CmdServer(TCPServer):
	def __init__(self, cmd_args):
		self.Sel = Selector()
		self.Node = socket.gethostname()
		TCPServer.__init__(self, 0, self.Sel)
		self.Closed = 0
		self.Cmd = tuple(cmd_args)
		#print self.Sock.fileno()
		#print self.Sock.getsockname()
		self.Processes = {}
		self.CurProc = None
		self.TTYMode = None
		self.Terminate = 0
		self.Kill = 0
		
	def run(self):
		while not self.Terminate:
			t = time.time()
			self.Sel.select(5)
			if time.time() > t + 2:
				break
		if self.Terminate:
			self.restoreTTY()
			if self.Kill:
				return 'kill'
			else:
				return 'terminate'
		else:
			return 'continue'

	def disconnectAll(self):
		for p in self.Processes.values():
			p.disconnect()
				

	def createClientInterface(self, sock, peer, sel):
		# print 'Client connected from ', peer
		# self.disableServer()
		str = SockStream(sock)
		hello = str.recv()
		if not hello:
			sock.close()
			self.Closed = 1
			return
		words = string.split(hello)
		if len(words) < 2 or words[0] != 'HELLO':
			sock.close()
			self.Closed = 1
			return
		try:	procno = string.atoi(words[1])	
		except:
			sock.close()
			self.Closed = 1
			return
		if self.Processes.has_key(procno):
			sock.close()
			self.Closed = 1
			return
		ans = str.sendAndRecv(repr(self.Cmd))
		#print 'Sent cmd, received <%s>' % ans
		if ans != 'OK':
			sock.close()
			self.Closed = 1
			return
		self.Processes[procno] = ProcessConnection(self, procno, 
			peer, sock)
		if not self.CurProc:
			self.Sel.register(self, rd=0)
			self.switchToProc(procno)

	def printMsg(self, msg):
		os.write(1, '\r\n[%s]\r\n' % msg)

	def switchToProc(self, procno):
		self.setRawTTY()
		oldProc = '(none)'
		if self.CurProc:
			oldProc = self.CurProc.ProcNo
		if not self.Processes.has_key(procno):
			self.printMsg('Process #%s not connected' % procno)
			procno = oldProc
		else:
			if self.CurProc:
				self.CurProc.unregister(self.Sel)
			self.CurProc = self.Processes[procno]
			self.CurProc.register(self.Sel)
		self.printMsg( 'Connected to process #%s' % procno )

	def setRawTTY(self):
		#print 'setRawTTY'
		if not self.TTYMode:
			self.TTYMode = tty.tcgetattr(0)
		tty.setraw(0)
		#print 'setRawTTY done'

	def restoreTTY(self):
		if self.TTYMode:
			tty.tcsetattr(0, tty.TCSAFLUSH, self.TTYMode)

	def doRead(self, fd, sel):
		#print 'CmdServer.doRead(%d)' % fd
		if fd == 0:
			#print 'input>', 
			line = os.read(0, 1000)
			if not line:	return
			if line[0] == chr(29):
				self.prompt()
			else:
				self.CurProc.send(line)
		else:
			TCPServer.doRead(self, fd, sel)

	def prompt(self):
		self.restoreTTY()
		print '\n# to switch to proc #, ? to list, q to quit, k to kill> ',
		cmd = sys.stdin.readline()
		cmd = string.strip(cmd)
		if cmd == 'q' or cmd == 'Q':
			self.disconnectAll()
			self.Terminate = 1
		elif cmd == 'k' or cmd == 'K':
			self.disconnectAll()
			self.Terminate = 1
			self.Kill = 1
		elif cmd == '?':
			print 'Connected processes: ', 
			lst = self.Processes.keys()
			lst.sort()
			for p in lst:
				print p,
			print ''
		elif cmd:
			procno = None
			try:	procno = string.atoi(cmd)
			except: pass
			if procno and procno >= 1:
				self.switchToProc(procno)
			else:
				self.printMsg('canceled')
		self.setRawTTY()

	def procDisconnected(self, procno):
		if not self.Processes.has_key(procno):
			return
		self.printMsg( 'Process #%d disconnected' % procno)
		proc = self.Processes[procno]
		proc.unregister(self.Sel)
		del self.Processes[procno]
		if not self.Processes:
			self.finish()
			return
		l = self.Processes.keys()
		l.sort()
		procno = l[0]
		self.switchToProc(procno)

	def finish(self):
		self.Sel.unregister(rd=0)
		self.restoreTTY()

	def dataFromProc(self, procno, data):
		os.write(0, data)

class	MyEventListener(FBSEventListener):
	def __init__(self, sid, verbose = 0):
		FBSEventListener.__init__(self)
		self.Sid = sid
		self.Exited = 0
		self.Deleted = 0
		self.ExitStatus = 0
		self.Started = 0
		self.Verbose = verbose
		try:	self.subscribe(sid)
		except KeyError:
			self.sectionDeleted(sid)
		else:
			state = self.sectionState(sid)
			self.sectionStateChanged(sid, 'ready', state)
				
	def sectionDeleted(self, sid):
		if self.Verbose:
			print 'deleted'
		self.Deleted = 1
		
	def sectionStateChanged(self, sid, old, new):
		if self.Verbose:
			print new
		if not new in ['waiting','running','ready']:
			self.Exited = 1
			if new != 'done':
				self.ExitStatus = 1
		elif new == 'running':
			self.Started = 1
		

def main():
	over_dict = {}
	sn = "Exec"		# default section name
	wait = 0		# by default, do not wait
	verbose = 0
	queue = None
	interactive = 0	
	kill_on_timeout = 0

	try:	opts, args = getopt.getopt(sys.argv[1:], 
		'Cvq:p:n:w:s:r:a:i:Io:e:h:a:N:m:')
	except getopt.error, msg:
		print msg
		return
	for opt, val in opts:
		if opt == '-q':		queue = val
		elif opt == '-m':	over_dict['MailTo'] = val
		elif opt == '-N':	over_dict['OnNodes'] = string.split(val)
		elif opt == '-I':	interactive = 1
		elif opt == '-v':	verbose = 1
		elif opt == '-h':	over_dict["HoldTime"] = val
		elif opt == '-p':	over_dict["ProcType"] = val
		elif opt == '-s':	sn = val
		elif opt == '-e':	over_dict['Stderr'] = val
		elif opt == '-o':	over_dict['Stdout'] = val
		elif opt == '-a':	over_dict['Placement'] = val
		elif opt == '-r': 	over_dict["PerProcRsrc"] = \
				Parser.wordsToDict(val, defValue = 0)
		elif opt == '-n':		
			over_dict["NProc"] = string.atoi(val)
		elif opt == '-i':
			over_dict["PrioInc"] = string.atoi(val)
		elif opt == '-w':
			if val == 'forever':	wait = -1
			else:		wait = string.atoi(val)
		elif opt == '-C':
			kill_on_timeout = 1

	if not queue:
		# queue is required
		print usage
		return 1

	cmd = string.join(args)
	if not cmd:
		print usage
		return 1
		
	cmd_args = args

	srv = None
	if interactive:
		srv = CmdServer(cmd_args)
		addr = srv.Sock.getsockname()
		proc_exec = [SpawnerCmd, socket.gethostname(), '%s' % addr[1]]
	else:
		proc_exec = cmd_args
	fc=FBSClient()		# establish connection to FBSNG
	j=FBSJobDesc()		# create job description

	# create section description, set section parameters
	s=FBSSectionDesc(sn, override = over_dict,
		Queue = queue, Exec = proc_exec)

	if verbose:
		print s	# print as JDF
	
	# add the section to the job and submit the job
	j.addSection(s)
	sts, jid = fc.submitJob(j)
	if not sts:
		# error
		print 'Submit failed: %s' % jid
		return 1
	else:
		# job submitted, print its id
		print 'Job %s' % jid

	if not interactive and wait == 0:
		return 0

	sid = fbs_misc.encodeSectionID(jid, sn)
	evl = MyEventListener(sid, verbose)
	s = fc.getSection(sid)
	tstart = time.time()

	# wait for the job to start
	while not evl.Deleted and not evl.Exited and not evl.Started and \
			(wait < 0 or time.time() < tstart + wait):
		evl.wait(1, max(1, tstart + wait - time.time()))

	if not evl.Started and not evl.Deleted and kill_on_timeout and wait > 0:
		if verbose:
			print 'Job start time-out'
		s.kill()	
		sys.exit(137)

	# wait for the job to end
	while not evl.Deleted and not evl.Exited and \
			(interactive or wait < 0 or time.time() < tstart + wait):

		if interactive:
				cmd = srv.run()
				if cmd == 'terminate':	break
				elif cmd == 'kill':
					s.kill()
				evl.wait(tmo=0)
		else:
			if wait >= 0:
				evl.wait(1, max(1, tstart + wait - time.time()))
			else:
				evl.wait(1)

	if evl.Exited:
		sys.exit(evl.ExitStatus)
	else:
		sys.exit(137)

if __name__ == '__main__':
	sys.exit(main())
