#
# @(#) $Id: PType.py,v 1.14 2003/08/22 17:25:09 ivm Exp $
#
# $Log: PType.py,v $
# Revision 1.14  2003/08/22 17:25:09  ivm
# Tested min. nice for proctype
#
# Revision 1.13  2003/08/20 18:58:57  ivm
# Implemented CPU power, round-robin-over-users scheduling inside queuei,
# other minor things.
#
# Revision 1.12  2001/11/20 19:42:14  ivm
# Implemented CPU and real time limits for proc. type
# Fixed launcher reconfiguration bug
#
# Revision 1.11  2001/10/27 18:00:14  ivm
# Implemented non-blocking process start
# Fixed some bugs
#
# Revision 1.10  2001/10/24 18:34:37  ivm
# Implemented ProcType.max_nodes
# Fixed section priority incrementing
#
# Revision 1.9  2001/09/10 20:44:31  ivm
# Removed debug print-outs
#
# Revision 1.8  2001/08/28 21:23:55  ivm
# Removed debug output
#
# Revision 1.7  2001/08/27 18:28:41  ivm
# Fixed status.py
# Use PType.wouldExceedQuota() in RM
#
# Revision 1.6  2001/08/23 19:05:53  ivm
# Implemented search with feedback RM algorithm
# Added -v option to status.py
# Fixed handling of time limits in FBSSectionInfo
#
# Revision 1.5  2001/06/12 19:47:52  ivm
# Updated for Python v2.1
#
# Revision 1.4  2001/05/11 15:09:00  ivm
# Fixed many bugs
#
# Revision 1.3  2001/03/19 15:14:51  ivm
# Return only non-zero usage resources for ProcTypeInfo
# Added users, nodes, ptypes to UI commands
#
# Revision 1.2  2001/03/13 15:59:01  ivm
# Fixed bugs in object ownership
#
# Revision 1.1  2001/03/12 19:28:37  ivm
# Added "object ownersip" feature
#
#

from MiscClasses import *	# import RV

class	ProcType:
	def __init__(self, name, cfg = None):
		self.Name = name
		self.Quota = {}
		self.SDefs = {}
		self.PDefs = {}
		self.MaxPrioInc = None
		self.Users = ['*']
		self.NodesAllow = ['*']
		self.NodesDisallow = []
		self.ProcsOnNode = RV()		# node -> no. of procs running
		self.MaxNodes = None
		self.RealTimeLimit = -1
		self.CPUTimeLimit = -1
		self.MinNice = 0
		if cfg:
			#print 'configuring...'
			self.configure(cfg)
		# fields for Resource Manager
		self.Usage = RV()

	
	def configure(self, cfg):
		pt = self.Name
		#print 'calling getValueDict()'
		#print cfg.__dict__
		qta = cfg.getValueDict('proc_type',pt,'resource_quota', 
			defValue=None)
		#print 'qta = ', qta
		if qta != None:
			for k, v in qta.items():
				if v != None:
					self.Quota[k] = v
		#print 'quota set'

		prdef = cfg.getValueDict('proc_type',pt,'proc_rsrc_defaults', 
			defValue=None)
		if prdef != None:
			self.PDefs = prdef
		#print 'prdef set'

		srdef = cfg.getValueDict('proc_type',pt,'sect_rsrc_defaults', 
			defValue=0)
		if srdef != None:
			self.SDefs = srdef
		#print 'srdef set'

		self.MaxPrioInc = cfg.getValue('proc_type',pt,'max_prio_inc',None)
		self.MaxNodes = cfg.getValue('proc_type',pt,'max_nodes',None)
		self.RealTimeLimit = cfg.getValue('proc_type',pt,'realtime',-1)
		self.CPUTimeLimit = cfg.getValue('proc_type',pt,'cputime',-1)
		lst = cfg.getValueList('proc_type', pt, 'nodes_allow')
		if lst:
			lst.sort()
			self.NodesAllow = lst[:]
		#print 'nodes allow set'
		
		lst = cfg.getValueList('proc_type', pt, 'nodes_disallow')
		if lst:
			lst.sort()
			self.NodesDisallow = lst[:]
		
		lst = cfg.getValueList('proc_type', pt, 'users')
		if lst:
			lst.sort()
			self.Users = lst[:]
		#print 'users set'
		
		self.MinNice = cfg.getValue('proc_type', pt, 'nice', 0)

	def configDict(self):
		ptdict = {}

		# quotas
		dict = {}
		for k, v in self.Quota.items():
			if v != None:
				dict[k] = v
		if dict:
			ptdict['resource_quota'] = dict

		# proc defaults
		dict = {}
		for k, v in self.PDefs.items():
			if v == None:	v = ''
			dict[k] = v
		if dict:
			ptdict['proc_rsrc_defaults'] = dict

		# sect defaults
		dict = {}
		for k, v in self.SDefs.items():
			if v == None:	v = ''
			dict[k] = v
		if dict:
			ptdict['sect_rsrc_defaults'] = dict

		if self.MaxPrioInc != None:
			ptdict['max_prio_inc'] = self.MaxPrioInc
		if self.MaxNodes != None:
			ptdict['max_nodes'] = self.MaxNodes
		if self.RealTimeLimit > 0:
			ptdict['realtime'] = self.RealTimeLimit
		if self.CPUTimeLimit > 0:
			ptdict['cputime'] = self.CPUTimeLimit
		if self.MinNice != 0:
			ptdict['nice'] = self.MinNice

		if not '*' in self.Users:
			if self.Users:
				ptdict['users'] = self.Users[:]
			else:
				ptdict['users'] = '-'
		
			
		if not '*' in self.NodesAllow:
			ptdict['nodes_allow'] = self.NodesAllow[:]
			
		if self.NodesDisallow:
			ptdict['nodes_disallow'] = self.NodesDisallow[:]
			
		return ptdict

	UserAttrs = ['NODESA','NODESD','MAXPI']
		
	def setParams(self, admin, username, pname, pval):
		# to be called from NetIF
		# validate permissions
		if not admin:
			if not pname in self.UserAttrs or \
					not '*' in self.Users and not username in self.Users:
				return 'PERM','User not allowed to modify %s' % pname
			
		try:		
			if pname == 'PRDEF':
				self.setPDefs(pval)
			elif pname == 'SRDEF':
				self.setSDefs(pval)
			elif pname == 'QUOTA':
				self.setQuota(pval)
			elif pname == 'NODESA':
				self.setNodesAllow(pval)
			elif pname == 'NODESD':
				self.setNodesDisallow(pval)
			elif pname == 'USERS':
				self.setUsers(pval)
			elif pname == 'MAXNODES':
				if pval != None and type(pval) != type(1):
					return 'ERR','Invalid value type for MaxNodeCount'
				self.MaxNodes = pval
			elif pname == 'CPUTIME':
				if pval != None and type(pval) != type(1):
					return 'ERR','Invalid value type for CPUTimeLimit'
				if pval == None: pval = -1	
				self.CPUTimeLimit = pval
			elif pname == 'REALTIME':
				if pval != None and type(pval) != type(1):
					return 'ERR','Invalid value type for RealTimeLimit'
				if pval == None: pval = -1	
				self.RealTimeLimit = pval
				
			elif pname == 'MAXPI':
				if pval != None and type(pval) != type(1):
					return 'ERR','Invalid value type for MaxPrioInc'
				self.MaxPrioInc = pval
			elif pname == 'NICE':
				if type(pval) != type(1):
					return 'ERR', 'Invalid value for Nice'
				self.MinNice = pval
			else:
				return 'ERR','Unknown parameter name <%s>' % pname
		except ValueError, txt:
			return 'ERR', txt
		return 'OK',''

	# methods for Resource Manager
	def addResource(self, rn):
		pass	# RV takes care of this

	def removeResource(self, rn):
		if self.Usage[rn]:
			raise ValueError, 'Resource <%s> is in use' % rn
		del self.Usage[rn]
		if self.Quota.has_key(rn):	del self.Quota[rn]
		if self.SDefs.has_key(rn):	del self.SDefs[rn]
		if self.PDefs.has_key(rn):	del self.PDefs[rn]

	def wouldExceedQuota(self, usg, mult = 1, node = None):
		# usg must be RV
		if node and not self.MaxNodes is None and \
				self.ProcsOnNode[node] == 0 and \
				self.CurNodes >= self.MaxNodes:
			return 1
		for k, v in self.Quota.items():
			if usg[k]*mult + self.Usage[k] > v:
				return 1
		return 0

	def procsOnNodes(self):
		# returns RV() {node:nprocs} with only nodes where nprocs > 0
		return self.ProcsOnNode.filter(lambda nn, np: np > 0).dict()

	def procsOnNode(self, nname):
		return self.ProcsOnNode[nname]

	def currentNodeCount(self):
		return reduce(lambda total, on_node:
			total + max(0, min(1, on_node)),
			self.ProcsOnNode.values(), 0)
			
	def nodeLimitExceeded(self):
		return not self.MaxNodes is None and \
			self.currentNodeCount() > self.MaxNodes
				
	def allocateResources(self, rsrc, mult = 1, node = None):
		self.Usage.add(rsrc, mult)
		if not node is None:
			self.ProcsOnNode[node] = max(0, self.ProcsOnNode[node] + mult)

	#def resourceUsage(self, rpdefs):
	#	usg = RV(self.Usage)
	#	for rpn, urlst in rpdefs.items():
	#		for urn in urlst:
	#			usg[rpn] = usg[rpn] + usg[urn]

	# other methods
	def setQuota(self, dct):
		if type(dct) != type({}):
			raise ValueError, 'Invalid type for Quota. Must be dict.'
		tmp = {}
		for k, v in dct.items():
			if v == None:	continue
			if type(v) != type(1):
				raise ValueError, 'Invalid type for quota value'
			tmp[k] = v
		self.Quota = tmp
	
	def setPDefs(self, dct):	
		if type(dct) != type({}):
			return ValueError,'Invalid value type for PDefs. Must be dict.'
		kv = {}
		for kk, vv in dct.items():
			if vv == None or type(vv) == type(1):
				kv[kk] = vv
			else:
				raise ValueError,'Invalid value for PDefs[%s]' % kk
		self.PDefs = kv

	def setSDefs(self, dct):	
		if type(dct) != type({}):
			return ValueError,'Invalid value type for SDefs. Must be dict.'
		kv = {}
		for kk, vv in dct.items():
			if type(vv) == type(1):
				kv[kk] = vv
			else:
				raise ValueError,'Invalid value for SDefs[%s]. Must be int.' % kk
		self.PDefs = kv

	def setNodesAllow(self, par):
		if type(par) != type([]):
			raise ValueError,'Invalid value type for NodesAllow. Must be List'
		par = par[:]
		if '*' in par:
			par = ['*']
		par.sort()
		self.NodesAllow = par

	def setNodesDisallow(self, par):
		if type(par) != type([]):
			raise ValueError,'Invalid value type for NodesDisallow. Must be List'
		par = par[:]
		if '*' in par:
			par = ['*']
		par.sort()
		self.NodesDisallow = par

	def setUsers(self, par):
		if type(par) != type([]):
			raise ValueError,'Invalid value type for Users. Must be List'
		par = par[:]
		if '*' in par:
			par = ['*']
		par.sort()
		self.Users = par

	def nodeAllowed(self, nn):
		if '*' in self.NodesDisallow:
			return nn in self.NodesAllow
		elif '*' in self.NodesAllow:
			return not nn in self.NodesDisallow
		else:
			return nn in self.NodesAllow and not nn in self.NodesDisallow

	def submitAllowed(self, admin, user):
		if admin or\
			user in self.Users or\
			'*' in self.Users:	return 1, ''
		return 0, 'User <%s> is not permitted to use process type <%s>' % (
			user, self.Name)

	def getUsageDict(self):
		# return Usage with zeros removed
		dict = self.Usage.dict()
		for k, v in dict.items():
			if not v:
				del dict[k]
		return dict
				
