#
# @(#) $Id: FBS_API.py,v 1.135 2003/01/14 22:05:02 ivm Exp $
#
# $Log: FBS_API.py,v $
# Revision 1.135  2003/01/14 22:05:02  ivm
# Implemented node class power
#
# Revision 1.134  2002/07/15 20:26:55  ivm
# Fixed memory leak in _FBSEventListener
#
# Revision 1.133  2002/06/19 16:52:52  ivm
# Fixed some bugs
#
# Revision 1.132  2001/12/13 18:13:27  ivm
# Implemented positional arguments in "submit" command
# Added "blocked" status to "queues"
# Updates Users Guide with new "exec" args and positional args of "submit"
# Implemented process start time comparison in launcher
#
# Revision 1.131  2001/06/01 14:45:38  ivm
# Fixed some bugs
#
# Revision 1.130  2001/05/25 21:39:36  ivm
# Implemented hold node and disconnect
#
# Revision 1.129  2001/04/20 17:41:25  ivm
# Fixed numerous bugs
#
# Revision 1.127  2001/04/11 16:41:37  ivm
# Added -N to exec -- OnNodes
#
# Revision 1.125  2001/03/26 19:43:05  ivm
# Added EventManager
#
# Revision 1.124  2001/03/19 16:22:12  ivm
# Added new fields to "show ptype" and "show queue" in farm_config
#
# Revision 1.123  2001/03/15 21:47:54  ivm
# Implemented "on nodes"
# Fixed protocol version handling in lch, lchif
#
# Revision 1.122  2001/03/13 15:59:00  ivm
# Fixed bugs in object ownership
#
# Revision 1.121  2001/02/05 20:17:28  ivm
# Allow queues without default process type
#
# Revision 1.119  2001/01/24 22:25:50  ivm
# Added SENDAUTH command to k5 authentication protocol
#
# Revision 1.118  2000/12/05 21:42:11  ivm
# Attempt to re-connect in FBSClient
# Fixed username -> uid mapping in status.py
# v1_2a_1
#
# Revision 1.117  2000/11/30 20:23:15  ivm
# Fixed bugs
# Made Scheduler more conservative about unknown queues/ptypes
# Use /tmp/launcher.pid for launcher inter-locking
#
# Revision 1.116  2000/11/08 20:27:00  ivm
# Implemented FBSClient.getServerOptions()
# Fixed bugs
#
# Revision 1.115  2000/11/06 15:40:14  ivm
# Modified Scheduler and Queue to use new normalization algorithm
#
# Revision 1.114  2000/11/03 21:16:42  ivm
# Tested RM pack, spread features
# Print detailed error message on ValueError in submit.py
# Fixed getJob w.r.t. Username
#
# Revision 1.113  2000/11/03 19:24:19  ivm
# Implemented Placement in RM and JDF
# Renamed tar files in clean-up scripts
# Added RPC library for Stats.so
#
# Revision 1.111  2000/10/31 16:29:18  ivm
# Added Username
#
# Revision 1.110  2000/10/24 20:16:08  ivm
# Added "remove" functionality
# Made building of Kerberos module optional
# Included kerberos as optional dependency to UPS table
# v1_2_1
# Fixed exec.py to send command as list
#
# Revision 1.108  2000/10/11 21:51:54  ivm
# Fixed bugs in launcher
# removed debug print-outs from API
#
# Revision 1.107  2000/10/11 19:36:57  ivm
# Added FBSNodeInfo.holdNode()
# Implemented credentials creation by Launcher
#
# Revision 1.105  2000/09/15 18:10:17  ivm
# Sort node names and job ids
#
# Revision 1.104  2000/09/12 14:59:59  ivm
# Implemented createLocalResource method and all related changes
#
# Revision 1.103  2000/09/06 14:43:32  ivm
# Fixed "return = ..." bug
#
# Revision 1.102  2000/09/05 21:06:31  ivm
# Always generate SyntaxError when readin from JDF
#
# Revision 1.100  2000/09/05 19:58:44  ivm
# Check for empty response from BMGR, generate FBSError
#
#
# Revision 1.97  2000/09/05 16:12:13  ivm
# Fixed bug in farm_config
# Use serialize module to send node info
#
# Revision 1.96  2000/08/29 17:06:44  ivm
# Fixed some bugs
#
# Revision 1.94  2000/08/25 14:37:27  ivm
# Check for quota specification format
#
# Revision 1.91  2000/08/24 14:23:18  ivm
# Fixed bugs
# Shift queue priorities back to minimum
#
# Revision 1.90  2000/08/22 17:28:15  ivm
# Fixed bug in RM.getLocalRsrc with attributes on down nodes
# Added "show" functionality to config
#
# Revision 1.89  2000/08/21 21:28:21  ivm
# Fixed some bugs
#
# Revision 1.86  2000/08/18 21:15:39  ivm
# Implemented dynamic re-configuration
#
# Revision 1.85  2000/08/03 15:39:33  ivm
# Fixed some bugs
#
# Revision 1.84  2000/08/02 16:36:04  ivm
# Fixed numerous bugs
#
# Revision 1.83  2000/08/02 15:03:21  ivm
# Be ready to receive None for local resource usage
#
# Revision 1.82  2000/07/25 16:56:17  ivm
# Made directories listed in cfg file example consistent with
# template directory structure (log->logs, jdb)
#
# Revision 1.81  2000/07/17 17:25:09  ivm
# Added queue locking/unlocking
# Generate KeyError if queue not found in hold/releaseQueue
#
# Revision 1.80  2000/07/05 15:57:24  ivm
# Do not allow empty jobs
#
# Revision 1.79  2000/06/28 21:31:12  ivm
# Cache BMGR's IP address to avoid DNS communication in FBSClient
# Re-arranged debug messages in LCHIF
#
# Revision 1.77  2000/06/28 18:09:47  ivm
# Print and send hold time as text in FBSSectionDesc
#
# Revision 1.74  2000/06/21 14:43:26  ivm
# Added killJob()
# Optimized EXITED/DEL part of launcher protocol
#
# Revision 1.73  2000/06/19 21:27:26  ivm
# Generate KeyErrors in API get...resource() methods
#
# Revision 1.72  2000/06/15 18:30:55  ivm
# Generate KeyError if requested things are not found
#
# Revision 1.71  2000/06/14 20:43:18  ivm
# Remove 'NF' from reason returned from some methods
#
# Revision 1.70  2000/06/12 18:32:16  ivm
# Removed hold/release/kill job methods
#
# Revision 1.69  2000/06/08 18:56:32  ivm
# Sort process types alphabetically
#
# Revision 1.66  2000/06/05 20:45:23  ivm
# Mase sure numeric section names come out as strings
# Restore signals in JobDB
# Fixed pid file name in Launcher
# Fixed memory leak in NetIF
#
# Revision 1.65  2000/05/23 22:17:18  ivm
# SECT_OUTPUT -> SECT_STDOUT
#
# Revision 1.64  2000/05/23 22:15:19  ivm
# Implemented SectOutput
#
# Revision 1.63  2000/05/23 18:04:43  ivm
# Fixed FBSJobDesc.getSection()
#
# Revision 1.62  2000/05/19 21:26:46  ivm
# Fixed getSectionOutput()
#
# Revision 1.61  2000/05/15 15:17:19  ivm
# Moved open-send-receive-close sequences into separate functions
# Make sure connection is open and closed after each use
#
# Revision 1.59  2000/05/11 17:22:00  ivm
# Added r2shm and related code.
# Added SectOutput related code
#
# Revision 1.58  2000/05/08 20:49:55  ivm
# Fixed some bugs,
# Implemented totals for local resource utilization
#
# Revision 1.56  2000/05/08 18:36:44  ivm
# Generate SyntaxError in fromJDF() and provide (line-number, line-text, message)
# as the exception argument
#
# Revision 1.55  2000/05/02 18:43:52  ivm
# cuz*3
#
# Revision 1.54  2000/05/02 18:42:44  ivm
# cuz cuz
#
# Revision 1.53  2000/05/02 18:38:34  ivm
# cuz
#
# Revision 1.52  2000/04/26 17:03:27  ivm
# Added JDFSeq field to SectParam
#
# Revision 1.51  2000/04/21 21:27:42  ivm
# Fixed FBSProcTypeInfo
#
# Revision 1.50  2000/04/20 21:18:41  ivm
# Fixed inter-queue scheduling
#
# Revision 1.49  2000/04/19 14:58:43  ivm
# Use -1, not -1.0 for 'hold forever'
#
# Revision 1.48  2000/04/11 15:19:31  ivm
# Made sure "hold forever" goes through
#
# Revision 1.47  2000/04/10 15:26:55  ivm
# Added FBSClient.getProcess()
#
# Revision 1.46  2000/04/10 14:58:10  ivm
# Removed duplicate incSectPrio
#
# Revision 1.45  2000/04/07 21:14:22  ivm
# Implemented incSectPrio()
#
# Revision 1.44  2000/04/06 20:37:00  ivm
# Fixed LeaderOnly <-> MailTo mix-up
#
# Revision 1.42  2000/04/06 15:18:39  ivm
# Removed history access from FBSClient
#
# Revision 1.41  2000/04/03 22:12:58  ivm
# Implemented hard kill
#
# Revision 1.39  2000/03/31 15:14:11  ivm
# Ignore procExit() in Section for non-running processes
#
# Revision 1.38  2000/03/30 21:06:55  ivm
# Removed SectionDesc.get/setDependencies()
#
# Revision 1.36  2000/03/30 20:44:46  ivm
# Removed and added some printouts
#
# Revision 1.33  2000/03/28 20:02:18  ivm
# Modified History and implemented some history functionality in FBSClient
#
# Revision 1.30  2000/03/24 21:02:46  ivm
# Fixed naming error
#
# Revision 1.25  2000/03/22 19:45:34  ivm
# Do not look into history for now
#
# Revision 1.22  2000/03/22 19:30:27  ivm
# Implemented hold/release job/section
#
# Revision 1.19  2000/03/21 20:11:01  ivm
# Implemented get*Resources, NodeClassInfo
#
# Revision 1.18  2000/03/17 16:29:11  ivm
# Use job[sname] instead of job.getSection(sname) in DepParser
#
# Revision 1.16  2000/03/14 16:15:10  ivm
# Added sections() and getSection()
#
# Revision 1.14  2000/03/13 19:48:03  ivm
# Added section.ExitCode
#
# Revision 1.11  2000/03/09 19:44:12  ivm
# Fixed bug with maxWords in job info
#
# Revision 1.9  2000/03/07 15:04:59  ivm
# Fixed syntax error
#
# Revision 1.8  2000/03/06 21:46:46  ivm
# Added import sys to FBS_API.py
#
# Revision 1.6  2000/03/06 21:16:50  ivm
# Queue can be held/released
#
# Revision 1.1  2000/03/02 22:17:38  ivm
# Initial version, submit, get job/section, kill
#
#

import os
import Parser
from config import *
from SockStream import SockStream
from FBSJobInfo import *
from FBSQueueInfo import *
from FBSSectionInfo import *
from FBSProcessInfo import *
from FBSNodeInfo import *
from FBSNodeClassInfo import *
from FBSProcTypeInfo import *
from lchstatif import *
from socket import *

#from Tracer import *			# debug !! comment out !!

from DepParser import *
import fbs_misc
import sys
import cvtime
import pwd
import time
import serialize
import re

FBS_k5imported = 1

try:	import	krb5
except: FBS_k5imported = 0

FBSError = 'FBS Error'

class	FBSJobDesc:
	def __init__(self, jdf = None, params=[]):
		self.Sections = {}
		self.NextSeq = 1
		if jdf != None:
			self.fromJDF(jdf, params)
	
	def _nextSeq(self):
		ret = self.NextSeq
		self.NextSeq = self.NextSeq + 1
		return ret
		
	def addSection(self, sect):
		if self.hasSection(sect.Name):
			raise ValueError, 'Section <%s> already exists' % sect.Name
		sect._JDFSeq = self._nextSeq()
		self.Sections[sect.Name] = sect

	def hasSection(self, sname):
		return self.Sections.has_key(sname)

	def sections(self):
		return self.Sections.keys()
		
	def __getitem__(self, sname):
		return self.Sections[sname]

	def getSection(self, sname):
		return self[sname]
		
	def validateDependencies(self):
		# 1. Check for non-existent sections
		# and make 2 dependency map dictionaries:
		# map[sname] = [sname1, sname2, ...] -- section sname depends on
		#					sname1, sname2, ...
		# rmap[sname] = [sname1, sname2, ...] -- sections sname1, sname2, ...
		#					depend on sname
		map = {}
		rmap = {}
		for sn, s in self.Sections.items():
			depstr = s.Depend
			try:	depexp = DepExpression(depstr)
			except SyntaxError, val:
				return 0, 'Syntax error: %s' % val
			lst = depexp.depList()
			#print 'Deps for sect %s:' % sn, lst
			map[sn] = lst + []

			# make sure all sections have entries in rmap too
			if not rmap.has_key(sn):
				rmap[sn] = []

			for dn in lst:
				#print dn
				if not self.Sections.has_key(dn):
					return 0, \
						'Section %s depends on non-existent section %s' % \
						(sn, dn)
				#map[sn].append(dn)
				if not rmap.has_key(dn):
					rmap[dn] = []
				rmap[dn].append(sn)

		while map:
			# print map
			# find a section which does not depend on any other, independent
			# section
			indep = ''
			for sn, deps in map.items():
				if len(deps) == 0:
					indep = sn
					break
			if indep == '':
				return 0, 'Circular dependencies detected'
			# remove indep section from map
			del map[indep]
			# remove all dependencies on this section (assume they are
			# satisfied)
			for dn in rmap[indep]:
				map[dn].remove(indep)
		return 1, 'OK'

	def fromJDF(self, jdf, params=[]):
		if type(jdf) == type(''):
			jdf = open(jdf,'r')
		self.Sections = {}
		lno = 1
		l = jdf.readline()
		while l:
			l = string.strip(l)
			words = string.split(l)
			if words and words[0] == 'SECTION':
				# new section
				if len(words) < 2:
					raise SyntaxError, (lno, l, 'Section without name')
				s = FBSSectionDesc(words[1])
				l, lno = s._fromJDF(jdf, lno, params)
				self.addSection(s)
			else:
				# ignore line ???
				l = jdf.readline()
				lno = lno + 1
		jdf.close()

	def _dump(self, toPrint=1):
		lst = []
		for sn, s in self.Sections.items():
			lst = lst + s._dump(toPrint)
		return lst

	def __repr__(self):
		lst = self._dump()
		str = ''
		for line in lst:
			str = str + ('%s\n' % line)
		return str

	def validate(self):
		if not self.Sections:
			return 0, 'Empty job'
		sts, reason = self.validateDependencies()
		if not sts:
			return sts, reason
		for s in self.Sections.values():
			sts, reason = s.validate()
			if not sts:
				return sts, reason
		return 1, 'OK'		

class	FBSSectionDesc:
	_JDFFieldMap = {
		'MAILTO':		('MailTo','S'),
		'SECT_STDOUT':	('SectOutput','S'),
		'QUEUE':		('Queue','S'),
		'STDOUT':		('Stdout','S'),
		'STDERR':		('Stderr','S'),
		'EXEC':			('Exec','S'),
		'PROC_TYPE':	('ProcType','S'),
		'DEPEND':		('Depend','S'),
		'PROC_RESOURCES':	('PerProcRsrc','D'),
		'SECT_RESOURCES':	('PerSectRsrc','D'),
		'NUMPROC':		('NProc','I'),
		'NEED':			('Need','I'),
		'LEADER_ONLY':	('LeaderOnly','I'),
		'NICE':			('Nice','I'),
		'PRIO_INC':		('PrioInc','I'),
		'HOLD_TIME':	('HoldTime','S'),
		'AFTER':		('HoldTime','S'),
		'PLACEMENT':	('Placement','S'),
		'ON_NODES': 	('OnNodes','L')
		}	

	_ParRE = re.compile('\$\((?P<number>\d)\)')

	def __init__(self, name, override={}, **pars):
		if '.' in name or ' ' in name:
			raise ValueError, 'Invalid section name <%s>' % name
		self.Name = name
		self.ProcType = ''
		self.HoldTime = None
		self.PerProcRsrc = {}
		self.PerSectRsrc = {}
		self.Queue = ''
		self.NProc = 1
		self.Nice = 0
		self.PrioInc = 0
		self.Stderr = '.'
		self.Stdout = '.'
		self.Depend = ''
		self.Need = 0
		self.LeaderOnly = 0
		self.Exec = []
		self.OnNodes = None
		self.MailTo = ''
		self._JDFSeq = None
		self.SectOutput = None
		self.Placement = 'round-robin'
		self.fillFromDict(override)
		self.fillFromDict(pars)
	
	def __repr__(self):
		lst = self._dump()
		str = ''
		for line in lst:
			str = str + ('%s\n' % line)
		return str
			
	def cvtJDFField(self, x, t):
		if type(x) != type(''):
			return x
		str = string.strip(x)
		if t == 'S':
			return str
		elif t == 'D':
			return Parser.wordsToDict(str, defValue = None)
		elif t == 'L':
			return Parser.parseWords(str, cvtInts=0)
		elif t == 'I':
			try:	return string.atoi(str)
			except:
				raise ValueError, 'Integer value expected: <%s>' % str
				
	def fillFromDict(self, dict):
		for pn, pv in dict.items():
			if pn == 'Name':		continue
			if self.__dict__.has_key(pn):
				self.__dict__[pn] = pv
			else:
				# does it come from JDF ?
				try:	fn, ft = self._JDFFieldMap[pn]
				except:
					raise ValueError, 'Unknown argument "%s"' % pn
				val = self.cvtJDFField(pv, ft)
				#print 'Converted %s = %s to %s = %s' % (pn, pv, fn, val)
				self.__dict__[fn] = val

		if type(self.HoldTime) == type(''):
			if self.HoldTime == 'forever':
				self.HoldTime = -1
			elif self.HoldTime == 'None':
				self.HoldTime = None
			else:
				try:	self.HoldTime = cvtime.parseDateTime(self.HoldTime)
				except:
					raise ValueError, 'Invalid hold time <%s>' % self.HoldTime
				
	def clone(self, name, override={}, **pars):
		s = FBSSectionDesc(name)
		for pn, pv in self.__dict__.items():
			if pn != 'Name' and pn != 'Depend' and pn[0] != '_':
				s.__dict__[pn] = pv
		s.fillFromDict(override)		
		s.fillFromDict(pars)
		return s

	def _applyParams(self, str, params):
		newstr = ''
		i = 0
		x = self._ParRE.search(str, i)
		while x != None:
			newstr = newstr + str[i:x.start()]
			ipar = int(x.group('number'))
			try:	param = params[ipar-1]
			except:
				param=''
			newstr = newstr + param
			i = x.end()
			x = self._ParRE.search(str, i)
		newstr = newstr + str[i:]
		return newstr

	def _fromJDF(self, jdf, lno, params=[]):
		dict = {}
		l = jdf.readline()
		while l:
			lno = lno + 1
			l = string.strip(l)
			words = string.split(l)
			if words and words[0] == 'SECTION':
				break
			try:	fn, fv = self._parseLine(l)
			except SyntaxError, msg:
				raise SyntaxError, (lno, l, msg.msg)
			#print fn, fv
			if fn:
				if fv:	fv = self._applyParams(fv, params)
				if not self._JDFFieldMap.has_key(fn):
					raise SyntaxError, (lno, l, 'Invalid keyword <%s>' % fn)
				name, typ = self._JDFFieldMap[fn]
				if name == 'HoldTime' and type(fv) == type(''):
					if fv == 'forever': 	fv = -1
					elif fv == 'None':		fv = None
					else:
						try:	fv = cvtime.parseDateTime(fv)
						except:
							raise SyntaxError, (lno, l, 
								'Invalid hold time specification <%s>' % fv)
				else:
					try:	fv = self.cvtJDFField(fv, typ)
					except ValueError, txt:
						raise SyntaxError, (lno, l, txt)
					except SyntaxError, txt:
						raise SyntaxError, (lno, l, txt)
				if name == 'PerProcRsrc':
					if type(fv) != type({}):
						raise SyntaxError, (lno, l, 'Invalid field value')
					for k, w in fv.items():
						if w != None and (type(w) != type(1) or w < 0):
							raise SyntaxError, (lno, l, 'Invalid resource requirement specification')
				elif name == 'PerSectRsrc':
					if type(fv) != type({}):
						raise SyntaxError, (lno, l, 'Invalid field value')
					for k, w in fv.items():
						if type(w) != type(1) or w < 0:
							raise SyntaxError, (lno, l, 'Invalid resource requirement specification')
				dict[fn] = fv
			l = jdf.readline()
		self.fillFromDict(dict)
		return l, lno
			
	def _parseLine(self, line):
		# this comes from JDF
		inx = string.find(line, '#')
		if inx >= 0:
			line = string.strip(line[:inx])
		if not line:	return	None, None # comment or blank
		kw, args = Parser.parseLine(line)
		if kw == None:
			raise SyntaxError, 'Invalid line syntax'
		inx = string.find(line, '=')
		rest = string.strip(line[inx+1:])
		return kw, rest

	def _dump(self, toPrint = 1):
		# this is sent to BMGR and received by SectParam object
		# retruns list of messages
		lst = []
		lst.append('SECTION %s' % self.Name)
		lst.append('NUMPROC = %d' % self.NProc)
		execStr = ''
		if toPrint:
			if type(self.Exec) == type([]):
				for w in self.Exec:
					if string.find(w,' ') >= 0:
						w = "'%s'" % w
					execStr = '%s%s ' % (execStr, w)
			else:
				execStr = self.Exec
		else:
			if type(self.Exec) == type([]):
				execStr = serialize.serialize(self.Exec)
			else:
				exec_lst = Parser.parseWords(self.Exec, cvtInts=0)
				#print self.Exec, exec_lst
				execStr = serialize.serialize(exec_lst)
		lst.append('EXEC = %s' % execStr)
		if self.Queue:		lst.append('QUEUE = %s' % self.Queue)
		if self.ProcType:	lst.append('PROC_TYPE = %s' % self.ProcType)
		if self.Depend:		lst.append('DEPEND = %s' % self.Depend)
		if self.Need:			lst.append('NEED = %d' % self.Need)
		if self.Stdout != '.':	lst.append('STDOUT = %s' % self.Stdout)
		if self.Stderr != '.':	lst.append('STDERR = %s' % self.Stderr)
		if self.LeaderOnly:		lst.append('LEADER_ONLY = %d' % self.LeaderOnly)
		if self.MailTo:			lst.append('MAILTO = %s' % self.MailTo)
		if self.Nice:			lst.append('NICE = %d' % self.Nice)
		if self.SectOutput: 	lst.append('SECT_OUT = %s' % self.SectOutput)
		if self.Placement != 'round-robin':
				lst.append('PLACEMENT = %s' % self.Placement)
		if self.HoldTime:		lst.append('HOLD_TIME = %s' % 
				fbs_misc.encodeTime(self.HoldTime))
		if self.PerProcRsrc:
			lst.append('PROC_RESOURCES = %s' % 
				self.dumpDict(self.PerProcRsrc))
		if self.PerSectRsrc:	
			lst.append('SECT_RESOURCES = %s' % self.dumpDict(self.PerSectRsrc))
		if self.PrioInc:		lst.append('PRIO_INC = %d' % self.PrioInc)
		if not toPrint and self._JDFSeq != None:
			lst.append('_JDF_SEQ = %s' % self._JDFSeq)
		if self.OnNodes:
			lst.append('ON_NODES = %s' % string.join(self.OnNodes))
		return lst

	def dumpDict(self, dict):
		str = ''
		for k, v in dict.items():
			if type(v) == type(()) or type(v) == type([]):
				v = v[0]
			if v == None:
				str = str + (' %s' % k)
			else:
				str = str + ' %s:%s'	% (k, v)
		return str

	def validate(self):
		dct = self.PerProcRsrc
		if type(dct) != type({}):
			return 0, 'Invalid type of PerProcRsrc in section %s' % self.Name
		for k, v in dct.items():
			if type(v) != type(1) and v != None:
				return 0, 'Invalid type of PerProcRsrc[%s] in section %s' %\
						(k, self.Name)
		dct = self.PerSectRsrc
		if type(dct) != type({}):
			return 0, 'Invalid type of PerSectRsrc in section %s' % self.Name
		for k, v in dct.items():
			if type(v) != type(1):
				return 0, 'Invalid type of PerSectRsrc[%s] in section %s' %\
						(k, self.Name)
		return 1, 'OK'
		

class	FBSClient:
	def __init__(self, whoAmI = 'API', cfg = None):
		if cfg == None:
			cfg = os.environ['FBS_CONFIG']
		if type(cfg) == type(''):
			cfg = ConfigFile(cfg)
		self.Cfg = cfg
		self.WhoAmI = whoAmI
		self.Str = None
		self.Sock = None
		bmgr_ip = self.Cfg.getValue('bmgr','*', 'host')
		try:	bmgr_ip = gethostbyname(bmgr_ip)
		except:
			raise FBSError, 'BMGR host <%s> not found' % bmgr_ip
		bmgr_port = self.Cfg.getValue('bmgr','*', 'api_port')
		self.BmgrAddr = (bmgr_ip, bmgr_port)
		self.LchIF = LauncherStatIF(cfg)
		self.ConnectTimeOut = -1
		self.ServerOptions = {}
		
	def getServerOptions(self):
		try:	ans = self.openSocketSayHello()
		except:
			t, v = sys.exc_type, sys.exc_value
			self.closeSocket()
			raise t, v
		#print 'openSocketSayHello() -> <%s>' % ans
		self.closeSocket()
		return self.ServerOptions

	def kAuth(self, svc, host):
		# assumes self.Sock is open
		if not FBS_k5imported:
			return 0, 'KRB5 error: kerberos not available'
		try:
			cp = krb5.get_login_principal()
			#print 'Client principal: <%s>' % cp.name()
			sp = krb5.get_svc_principal(svc, host)
			#print 'Service principal: <%s>' % sp.name()
			ac = krb5.send_auth(self.Sock, cp, sp)
		except krb5.Krb5Error, val:
			return 0, 'KRB5 error: %s' % val.message
		return 1, 'OK'

	def setTimeOut(self, tmo = -1):
		old = self.ConnectTimeOut
		self.ConnectTimeOut = tmo
		return old
		
	def getSectionOutput(self, sid):
		dir = self.Cfg.getValue('bmgr','*','section_log_dir')
		if dir == None:
			raise KeyError, 'Section log directory is not configured'
	  	try:  f = open('%s/%s.log' % (dir, sid), 'r')
		except: raise KeyError, 'Section log file not found'
		lst = []
		line = ''
		t = 0
		l = f.readline()
		while l:
			if l and l[0] in [' ','\t']:
				line = line  + '\n' + string.strip(l)
			elif l:
				l = string.strip(l)
				if line:
					lst.append((t,string.strip(line)))
					line = ''
				words = string.split(l, '|')
				t = string.atof(words[0])
				line = words[3]
			l = f.readline()
		if line:
			lst.append((t,string.strip(line)))
		f.close()
		return lst
		
	def closeSocket(self):
		#print 'API: closing connection'
		self.Str = None
		if self.Sock != None:
			self.Sock.close()
		self.Sock = None
					
	def openSocketSayHello(self):
		ans = 'OK'
		#print 'API: opening connection...'
		while self.Sock == None:
			uid = os.getuid()
			gid = os.getgid()
			user = pwd.getpwuid(uid)[0]
			connected = 0
			t = time.time()
			ext, exv = FBSError, 'Can not connect to BMGR'
			while not connected and (self.ConnectTimeOut == -1 or
					time.time() < t + self.ConnectTimeOut):
				self.Sock = socket(AF_INET, SOCK_STREAM)
				try:	self.Sock.connect(self.BmgrAddr)
				except:
					ext = sys.exc_type
					exv = sys.exc_value
					self.Sock.close()
					time.sleep(1)
					continue
				self.Str = SockStream(self.Sock, '\n')
				ans = self.Str.sendAndRecv('HELLO %s %d %d %s' % 
					(self.WhoAmI, uid, gid, user))
				if not ans:
					self.closeSocket()
					time.sleep(1)
					continue
				lst = string.split(ans)
				ans, opts = lst[0], lst[1:]
				self.ServerOptions = Parser.wordsToDict(opts)
				connected = 1
			if not connected:
				raise ext, exv
		#else:
		#	print 'API: connection already open'
		return ans

	def sendAndRecv(self, msg):
		# send request and receive response. Run through
		# Kerberos authentication if required
		answered = 0
		ans = None
		while not answered:
			try:	ans = self.openSocketSayHello()
			except:
				t, v = sys.exc_type, sys.exc_value
				self.closeSocket()
				raise t, v
			#print 'openSocketSayHello() -> <%s>' % ans
			if ans != 'OK':
				self.closeSocket()
				raise FBSError, 'Error connecting to BMGR: %s' % ans
			ans = self.Str.sendAndRecv(msg)
			#print 'sent <%s>, recvd <%s>' % (msg, ans)
			if ans:
				answered = 1
			else:
				self.closeSocket()
				time.sleep(1)
		words = string.split(ans, None, 1)
		if len(words) < 1:
			self.closeSocket()
			raise FBSError, 'Communication error: %s' % ans
		if words[0] == 'K5AUTH':	
			words = string.split(ans)
			if len(words) < 3:
				self.closeSocket()
				raise FBSError, 'Communication error: %s' % ans
			svc = words[1]
			host = words[2]
			ans = self.Str.sendAndRecv('OK')
			if ans != 'SENDAUTH':
				self.closeSocket()
				raise FBSError, 'Communication error: %s' % ans				
			#print 'sent <OK>, initiating kAuth'
			sts, reason = self.kAuth(svc, host)
			#print 'kAuth -> %s %s' % (sts, reason)
			if not sts:
				self.closeSocket()
				raise FBSError, 'User authentication failed: %s' % reason
			#print 'waiting for answer for <%s>...' % msg
			ans = self.Str.recv()
		#print 'answer for <%s>: <%s>' % (msg, ans)
		if not ans:
			self.closeSocket()
			raise FBSError, 'BMGR closed connection'
		# self.closeSocket()
		return ans

	def sendAndRecvList(self, msg):
		try:	ans = self.openSocketSayHello()
		except:
			t, v = sys.exc_type, sys.exc_value
			self.closeSocket()
			raise t, v
		if ans != 'OK':
			self.closeSocket()
			raise FBSError, 'Communication error: %s' % ans
		self.Str.send(msg)
		try:	
			lst = self._recvList()
		except:
			t, v = sys.exc_type, sys.exc_value
			self.closeSocket()
			raise t, v
		self.closeSocket()
		return lst

	#def _getProcTypeUsage(self, pt):
	#	return self.sendAndRecvList('GETPTUSG %s' % pt)

	def _getPIFromBMGR(self, sid, procno):
		return self.sendAndRecvList('GETPROC %s %d' % (sid, procno))

	def _getPIFromLch(self, node, bpid):
		pi, txt = self.LchIF.getProcInfo(bpid, node)
		if pi == None:
			raise FBSError, txt
		return pi

	def getProcess(self, bpid, local_details = 1):
		pi = FBSProcessInfo(bpid, self)
		pi.refresh(local_details)
		return pi

	def getProcessTypeList(self):
		lst = self._getAList('GETPTL')
		lst.sort()
		return lst

	def getProcessType(self, pt):
		pt = FBSProcTypeInfo(self, self.Cfg, pt)
		pt.refresh()
		return pt

	def getNodeList(self, cname = None):
		if not cname:	
			lst = self._getAList('GETNODEL')
		else:
			lst = self._getAList('GETNODEL %s' % cname)
		lst.sort()
		return lst

	def killJob(self, jids, now = 0):
		if type(jids) != type([]):
			jids = [jids]
		if not jids:
			return 1, 'Empty job list'
		ans = self.sendAndRecv('KILLJOB %s %s' % (now, string.join(jids)))
		words, rest = Parser.parseWords(ans, maxWords = 1)
		if len(words) < 1:
			raise FBSError, 'Communication error: <%s>' % ans
		if words[0] == 'OK':
			return 1, 'OK'
		elif words[0] == 'W':
			return 1, rest
		elif words[0] == 'PERM':
			return 0, rest
		else:
			raise FBSError, 'Communication error: <%s>' % ans
		
		
	def getNodeClassList(self):
		lst = self._getAList('GETNCL')
		lst.sort()
		return lst

	def getNodeClass(self, cname):
		ci = FBSNodeClassInfo(self, cname)
		ci.refresh()
		return ci

	def _getAList(self, msg, cvtInts = 1):
		ans = self.simpleGet(msg)
		#print '_getAList: ans=<%s>' % ans
		return Parser.parseWords(ans, cvtInts = cvtInts)

	def getGlobalRsrcList(self):
		return self._getAList('GETGRL')
	
	def getGlobalPoolList(self):
		return self._getAList('GETGPL')
	
	def getLocalRsrcList(self):
		return self._getAList('GETLRL')
	
	def getLocalPoolList(self):
		return self._getAList('GETLPL')
	
	def getResourcePool(self, rp):
		return self._getAList('GETRP %s' % rp)

	def getLocalResource(self, rn, proctype = None, node = None):
		if proctype != None and node != None:
			raise ValueError, 'Specify either process type, or node name, not both'
		pt = proctype
		nn = node
		if pt == None:	pt = '*'
		if nn == None:	nn = ''
		ans = self.sendAndRecv('GETLRSRC %s %s %s' % (rn, pt, nn))
		words = string.split(ans)
		rest = string.join(words[1:])
		if len(words) < 1:
			raise FBSError, 'Communication error: <%s>' % ans
		if words[0] == 'OK':
			if words[1] == 'None':
				usg = None
			else:
				usg = string.atoi(words[1])
			if words[2] == 'None':
				cap = None
			else:
				cap = string.atoi(words[2])
			return usg, cap
		elif words[0] == 'NF':
			raise KeyError, rest
		else:
			raise FBSError, ans

	def getLclRsrcUsage(self, rn, node=None):
		return self.getLocalResource(rn, node = node)

	def getLclRsrcQuota(self, rn, proctype):
		return self.getLocalResource(rn, proctype = proctype)

	def getGlobalResource(self, rn, proctype = None):
		pt = proctype
		if pt == None:	pt = ''
		ans = self.sendAndRecv('GETGRSRC %s %s' % (rn, pt))
		words = string.split(ans)
		rest = string.join(words[1:])
		if len(words) < 1:
			raise FBSError, 'Communication error: <%s>' % ans
		if words[0] == 'OK':
			usg = string.atoi(words[1])
			if words[2] == 'None':
				cap = None
			else:
				cap = string.atoi(words[2])
			return usg, cap
		elif words[0] == 'NF':
			raise KeyError, rest
		else:
			raise FBSError, ans

	def getGblRsrcQuota(self, rn, proctype):
		return self.getGlobalResource(rn, proctype = proctype)
		
	def getGblRsrcUsage(self, rn):
		return self.getGlobalResource(rn)
		
	def holdSection(self, sid, t = None):
		if type(t) == type(''):
			try:	t = cvtime.parseDateTime(t)
			except:
				raise ValueError, 'Invalid hold time specification <%s>' % t
		if t == None:
			t = ''
		else:
			t = fbs_misc.encodeTime(t)
		ans = self.sendAndRecv('HOLDSECT %s %s' % (sid, t))
		words, rest = Parser.parseWords(ans, maxWords = 1)
		if len(words) < 1:
			raise FBSError, 'Communication error: <%s>' % ans
		if words[0] == 'OK':
			return 1, 'OK'
		elif words[0] == 'NF':
			raise KeyError, rest
		elif words[0] == 'W':
			return 1, rest
		else:
			raise FBSError, 'Communication error: <%s>' % ans

	def releaseSection(self, sid):
		ans = self.sendAndRecv('RELSECT %s' % sid)
		words, rest = Parser.parseWords(ans, maxWords = 1)
		if len(words) < 1:
			raise FBSError, 'Communication error: <%s>' % ans
		if words[0] == 'OK':
			return 1, 'OK'
		elif words[0] == 'NF':
			raise KeyError, rest
		elif words[0] == 'W':
			return 1, rest
		else:
			raise FBSError, 'Communication error: <%s>' % ans

	def holdNode(self, nname, reason = 'reason unknown', disconnect = 0):
		if disconnect:	d = '-d'
		else:			d = ''
		ans = self.sendAndRecv('HOLDNODE %s %s %s' % (d, nname, reason))
		words, rest = Parser.parseWords(ans, maxWords = 1)
		if len(words) < 1:
			raise FBSError, 'Communication error: <%s>' % ans
		if words[0] == 'OK':
			return 1, 'OK'
		elif words[0] == 'NF':
			raise KeyError, rest
		elif words[0] == 'W':
			return 1, rest
		else:
			raise FBSError, 'Communication error: <%s>' % ans
				
	def releaseNode(self, nname):
		ans = self.sendAndRecv('RELNODE %s' % nname)
		words, rest = Parser.parseWords(ans, maxWords = 1)
		if len(words) < 1:
			raise FBSError, 'Communication error: <%s>' % ans
		if words[0] == 'OK':
			return 1, 'OK'
		elif words[0] == 'NF':
			raise KeyError, rest
		elif words[0] == 'W':
			return 1, rest
		else:
			raise FBSError, 'Communication error: <%s>' % ans
				

	def submitJob(self, job):
		sts, reason = job.validate()
		if not sts:
			return 0, reason
		ans = self.openSocketSayHello()
		if ans != 'OK':
			return 0, 'Error connecting to BMGR: %s' % ans
		lst = job._dump(toPrint=0)
		#print 'submitJob(): dumpForSubmit -> %s' % lst
		ans = self.sendAndRecv('SUBMIT')
		if ans != 'OK':
			self.closeSocket()
			return 0, '%s' % ans
		self.Str.send(lst)
		ans = self.Str.sendAndRecv('EOJ')
		if not ans:
			self.closeSocket()
			return 0, 'BMGR closed connection'
		words, rest = Parser.parseWords(ans, maxWords = 2, cvtInts = 0)
		if len(words) < 2:
			self.closeSocket()
			return 0, 'Wrong answer from BMGR: <%s>' % ans
		if words[0] == 'OK':
			self.closeSocket()
			return 1, words[1]
		else:
			self.closeSocket()
			return 0, rest
			
	def getJob(self, jid):
		j = FBSJobInfo(self, jid)
		j.refresh()
		return j

	def _getJobInfo(self, jid):
		ans = self.sendAndRecv('GETJOB %s' % jid)
		#print 'ans = %s' % ans
		# reply: OK state UID GID section ...
		words = string.split(ans, None, 1)
		if len(words) < 2:
			raise FBSError, 'Communication error: %s' % ans
		if words[0] == 'OK':
			pars, rest = Parser.parseWords(words[1], maxWords = 4)
			if len(pars) < 4:
				raise FBSError, 'Protocol error'
			sects = Parser.wordsToDict(rest, defValue = 0)
			return pars, sects
		elif words[0] == 'NF':
			raise KeyError, 'Job %s not found' % jid
		else:
			raise FBSError, 'Communication error: %s' % ans

	def getSection(self, sid):
		s = FBSSectionInfo(self, sid)
		s.refresh()
		return s

	def killSection(self, sid, now = 0):
		nowstr = ''
		if now: nowstr = 'now'
		ans = self.sendAndRecv('KILLSECT %s %s' % (sid, nowstr))
		words, rest = Parser.parseWords(ans, maxWords = 1)
		if len(words) < 1:
			raise FBSError, 'Communication error: %s' % ans
		if words[0] == 'OK':
			return 1, 'OK'
		elif words[0] == 'NF':
			return 0, rest
		else:
			self.closeSocket()
			return 0, ans

	def _getSectionInfo(self, sid):
		try:	return self.sendAndRecvList('GETSECT %s' % sid)
		#T.trace('getSectionInfo: request sent')
		except KeyError, v:
			raise KeyError, 'Section %s not found' % sid
		
	def getJobList(self, username = None, uid = -1, gid = -1):
		if not username:	username = None
		lst = self._getAList('GETJOBS %s %d %d' % (username, uid, gid), cvtInts = 0)
		return lst

	def _recvList(self):
		#T.trace('_recvList: begin')
		ans = self.Str.recv(1000)
		#T.trace('_recvList: answer received: <%s>' % ans)
		#print '_recvList: %s' % ans
		if ans == None:
			raise FBSError, 'BMGR closed connection'
		words, rest = Parser.parseWords(ans, maxWords = 2, cvtInts = 0)

		#print '_recvList: words = ', words
		if len(words) < 2:
				raise FBSError, 'Communication error: %s' % ans
		if words[0] != 'DATA':
			if words[0] == 'ERR' and words[1] == 'NF':
				raise KeyError, rest
			elif words[0] == 'NF':
				raise KeyError, words[1] + ' ' + rest
			else:
				raise FBSError, ans
		eod = words[1]
		lst = []
		gotEod = 0
		i = 0
		#T.trace('_recvList: before line loop')
		while not gotEod and not self.Str.eof():
			i = i + 1
			l = self.Str.recv(1000)
			#T.trace('_recvList: line #%d' % i)
			#print '_recvList: %s' % ans
			if l == eod:
				gotEod = 1
				break
			elif l:
				lst.append(l)
		if gotEod:
			return lst
		else:
			# unexpected EOF, list is incomplete
			raise FBSError, 'BMGR closed connection'

	def getQueueList(self):
		return self._getAList('GETQS')

	def getQueue(self, qname):
		q = FBSQueueInfo(self, qname)
		q.refresh()
		return q

	def lockQueue(self, qname):
		return self.simpleRequest('LCKQ %s' % qname)

	def unlockQueue(self, qname):
		return self.simpleRequest('UNLCKQ %s' % qname)

	def simpleRequest(self, req):
		ans = self.sendAndRecv(req)
		if not ans:
			raise FBSError, 'BMGR disconnected'
		words, rest = Parser.parseWords(ans, maxWords = 1)
		if not words:
			raise FBSError, 'Communication error <%s>' % ans
		if words[0] == 'OK':
			return 1, 'OK'
		elif words[0] == 'NF':
			return 0, rest
		else:
			return 0, ans

	def simpleGet(self, getstr):
		ans = self.sendAndRecv(getstr)
		if not ans:
			raise FBSError, 'BMGR disconnected'
		words, rest = Parser.parseWords(ans, maxWords=1)
		if not words:
			raise FBSError, 'Communication error: <%s>' % ans
		if words[0] == 'OK':
			return rest
		elif words[0] == 'NF':
			raise KeyError, rest
		else:
			raise FBSError, rest

	def holdQueue(self, qname):
		return self.simpleRequest('HOLDQ %s' % qname)

	def releaseQueue(self, qname):
		return self.simpleRequest('RELQ %s' % qname)

	def _getQueueInfo(self, qname):
		return self.simpleGet('GETQ %s' % qname)

	def getNode(self, nname):
		n = FBSNodeInfo(self, nname)
		n.refresh()
		return n

	def _addNode(self, ncn, nn):
		return self.simpleRequest('ADDNODE %s %s' % (ncn, nn))
		
	def _removeNode(self, nn):
		return self.simpleRequest('REMNODE %s' % nn)
		
	def _getNodeInfo(self, nname):
		return self.simpleGet('GETNODE %s' % nname)
		
	def _getNodeClassInfo(self, ncn):
		return self.simpleGet('GETNC %s' % ncn)
		
	def incSectPrio(self, sid, incr):
		return self.simpleRequest('INCSPRIO %s %d' % (sid, incr))
	
	def _getProcTypeInfo(self, pt):
		# called by FBSProcTypeInfo.refresh()
		return self.simpleGet('GETPT %s' % pt)
	
	# dynamic re-configuration methods
	def _updateProcTypeInfo(self, pt, what, val):
		dstr = serialize.serialize(val)
		return self.simpleRequest('SETPT %s %s %s' % (pt, what, dstr))
		
	def _updateNodeClassInfo(self, ncn, what, val):
		# dump the dictionary
		dstr = serialize.serialize(val)
		return self.simpleRequest('SETNC %s %s %s' % (ncn, what, dstr))
		
	def createNodeClass(self, name):
		sts, reason = self.simpleRequest('ADDNC %s' % name)
		if not sts:
			raise ValueError, reason
		return self.getNodeClass(name)
		
	def createProcessType(self, name):
		sts, reason = self.simpleRequest('ADDPT %s' % name)
		if not sts:
			raise ValueError, reason
		return self.getProcessType(name)

	def createGlobalResource(self, name, cap):
		return self.simpleRequest('ADDGR %s %d' % (name, cap))

	def createLocalResource(self, name):
		return self.simpleRequest('ADDLR %s' % name)

	def setGlobalResource(self, name, cap):
		return self.simpleRequest('SETGR %s %d' % (name, cap))

	def _updateQueueParams(self, qn, dict):
		str = serialize.serialize(dict)
		return self.simpleRequest('SETQ %s %s' % (qn, str))
		
	def createQueue(self, qn, defpt=None):
		if defpt == None:	defpt = ''
		sts, reason = self.simpleRequest('ADDQ %s %s' % (qn, defpt))
		if not sts:
			raise ValueError, reason
		return self.getQueue(qn)

	def createRsrcPool(self, name, urlst):
		return self.simpleRequest('ADDRP %s %s' % (name, string.join(urlst)))

	def setRsrcPool(self, name, urlst):
		return self.simpleRequest('SETRP %s %s' % (name, string.join(urlst)))
		
	def removeRsrcPool(self, name):
		return self.simpleRequest('REMRP %s' % name)

	def removeProcessType(self, name):
		return self.simpleRequest('REMPT %s' % name)

	def removeNodeClass(self, name):
		return self.simpleRequest('REMNC %s' % name)

	def removeQueue(self, name):
		return self.simpleRequest('REMQ %s' % name)

	def removeLocalResource(self, name):
		return self.simpleRequest('REMLR %s' % name)

	def removeGlobalResource(self, name):
		return self.simpleRequest('REMGR %s' % name)
		
class	_FBSEventListener:
	def __init__(self, erecvr = None, whoAmI = 'EVT', cfg = None):
		self.EventReceiver = erecvr
		if cfg == None:
			cfg = os.environ['FBS_CONFIG']
		if type(cfg) == type(''):
			cfg = ConfigFile(cfg)
		self.Cfg = cfg
		self.WhoAmI = whoAmI
		self.Str = None
		self.Sock = None
		bmgr_ip = self.Cfg.getValue('bmgr','*', 'host')
		try:	bmgr_ip = gethostbyname(bmgr_ip)
		except:
			raise FBSError, 'BMGR host <%s> not found' % bmgr_ip
		evt_port = self.Cfg.getValue('bmgr','*', 'event_mgr_port')
		if not evt_port:
			evt_port = self.Cfg.getValue('bmgr','*', 'api_port') + 1
		self.BmgrAddr = (bmgr_ip, evt_port)
		self.LastState = {} 	# sid -> (sstate, [procstates])
		self.ConnectTimeOut = -1
		self.EventQueue = []
		
	def openSocketSayHello(self, tmo = -1):
		ans = 'OK'
		#print 'API: opening connection...'
		while self.Sock == None:
			connected = 0
			t = time.time()
			ext, exv = FBSError, 'Can not connect to BMGR'
			while not connected and (tmo == -1 or
					time.time() < t + tmo):
				self.Sock = socket(AF_INET, SOCK_STREAM)
				try:	self.Sock.connect(self.BmgrAddr)
				except:
					ext = sys.exc_type
					exv = sys.exc_value
					self.Sock.close()
					time.sleep(1)
					continue
				self.Str = SockStream(self.Sock, '\n')
				connected = 1
			if not connected:
				raise ext, exv
		#else:
		#	print 'API: connection already open'
		return ans == 'OK'

	def disconnect(self):
		self.Sock.close()
		self.Str = None
		self.Sock = None

	def reconnect(self, tmo = -1):
		t = time.time()
		doSleep = 0
		while not self.Str and (tmo < 0 or time.time() < t + tmo):
			if doSleep: time.sleep(1)
			doSleep = 1
			self.openSocketSayHello(tmo)
			# re-subscribe to all sections
			for sid in self.LastState.keys():
				sts, sstat, pstats = self.sendSubscribe(sid, tmo)
				if sts == 'EOF':
					self.disconnect()
					break
				if sts == 'NF':
					# fake DEL event
					self.EventQueue.append(['SECT',sid,'DEL',''])
					continue
				elif sts != 'OK':
					raise FBSError, sts
				self.reconcile(sid, sstat, pstats)
			
	def reconcile(self, sid, sstat, pstats):
		if not self.LastState.has_key(sid):
			return
		oldsstat, oldpstats = self.LastState[sid]
		for i in range(len(pstats)):
			if oldpstats[i] != pstats[i]:
				self.EventQueue.append(['PROC', sid, i+1, 'STATE', pstats[i]])
		if oldsstat != sstat:
			self.EventQueue.append(['SECT',sid,'STATE',sstat])
		self.LastState[sid] = (sstat, pstats)
		
			
	def sendSubscribe(self, sid, tmo = -1):
		ans = self.sendAndRecv('SUBS %s' % sid)
		if not ans:
			return 'EOF', None, None
		sts, rest = tuple(string.split(ans, None, 1))
		if sts == 'NF': return 'NF', None, None
		if sts != 'OK': return ans, None, None
		stats = string.split(rest)
		return sts, stats[0], stats[1:]

	def subscribe(self, sid):
		# do not re-subscribe
		done = self.LastState.has_key(sid)
		while not done:
			self.reconnect()
			sts, sstat, pstats = self.sendSubscribe(sid)
			if sts == 'NF':
				raise KeyError, 'Section <%s> not found' % sid
			if sts == 'EOF':
				time.sleep(1)
				continue
			if sts != 'OK':
				raise FBSError, sts
			self.LastState[sid] = (sstat, pstats)
			done = 1				

	def unsubscribe(self, sid = None):
		if sid:
			if self.LastState.has_key(sid):
				del self.LastState[sid]
			self.sendAndRecv('UNSUB %s' % sid)
		else:
			self.LastState = {}
			self.sendAndRecv('UNSUB')
		
	def sendAndRecv(self, msg):
		# send msg, read answer(s), put events into event buffer
		ans = self.Str.sendAndRecv(msg)
		done = 0
		while not done and ans:
			words = string.split(ans)
			if words[0] == 'EVENT':
				self.EventQueue.append(words[1:])
				ans = self.Str.recv()
			else:
				done = 1
		return ans

	def processEvent(self, elst):
		#print 'processEvent(%s)' % elst
		if len(elst) < 2:	return
		et = elst[0]
		sid = elst[1]
		if not self.LastState.has_key(sid):
			self.unsubscribe(sid)
			return
		oldstate, oldpstat = self.LastState[sid]
		if et == 'SECT':
			if elst[2] == 'STATE':
				if len(elst) != 4:	return
				newstate = elst[3]
				if oldstate == newstate:	return
				self.LastState[sid] = (newstate, oldpstat)
				self.sectionStateChanged(sid, oldstate, newstate)
			elif elst[2] == 'DEL':
				del self.LastState[sid]
				self.sectionDeleted(sid)
		elif et == 'PROC':
			if len(elst) != 5:	return
			pno = string.atoi(elst[2])
			pinx = pno - 1
			newstate = elst[4]
			if newstate == oldpstat[pinx]:	return
			tmp = oldpstat[pinx]
			oldpstat[pinx] = newstate
			self.LastState[sid] = (oldstate, oldpstat)
			self.processStateChanged(sid, pno, tmp, newstate)
				
	def flushEvents(self, nmax=None):
		n = 0
		while self.EventQueue and (nmax == None or n < nmax):
			evt = self.EventQueue[0]
			self.EventQueue = self.EventQueue[1:]
			self.processEvent(evt)
			n = n + 1
		return n

	def wait(self, nmax=None, tmo = -1):
		if not self.LastState and tmo == -1:
			raise KeyError, 'Not subscribed'
		t = time.time()
		while not self.Sock:
			self.reconnect()
			if not self.Sock:
				time.sleep(5)
		done = 0
		n = self.flushEvents()
		while not done and (nmax == None or n < nmax):
			tmo1 = -1
			if tmo >= 0:
				tmo1 = int(t + tmo - time.time())
				if tmo1 < 0:	tmo1 = 0
			try:	msg = self.Str.recv(tmo=tmo1)
			except IOError:
				# timeout
				done = 1
				break
			else:
				if not msg:
					self.disconnect()
					return n
				words = string.split(msg)
				if not words or words[0] != 'EVENT':	continue
				self.processEvent(words[1:])
				n = n + 1
		return n

	def sectionState(self, sid):
		try:	sst, pst = self.LastState[sid]
		except KeyError:
			raise KeyError, 'Not subscribed'
		return sst
		
	def processStates(self, sid, pno):
		try:	sst, pst = self.LastState[sid]
		except KeyError:
			raise KeyError, 'Not subscribed'
		return pst[pno-1]
		
							
	# overridables
	def sectionStateChanged(self, sid, oldstate, newstate):
		if self.EventReceiver != None:
			self.EventReceiver.sectionStateChanged(sid, oldstate, newstate)
		
	def processStateChanged(self, sid, pno, oldstate, newstate):
		if self.EventReceiver != None:
			self.EventReceiver.processStateChanged(sid, pno, oldstate, newstate)
		
	def sectionDeleted(self, sid):
		if self.EventReceiver != None:
			self.EventReceiver.sectionDeleted(sid)

# FBSEventListener is a virtual base class given to users
# to derive from. It has "real" event listener object hidden
# from users so that they accidentally do not override
# data members. This is done because Python does not have
# private or protected data members.

class	FBSEventListener:
	def __init__(self, cfg = None):
		self.EL = _FBSEventListener(erecvr = self, cfg = cfg)

	def subscribe(self, sid):
		self.EL.subscribe(sid)		

	def unsubscribe(self, sid = None):
		self.EL.unsubscribe(sid)

	def wait(self, nmax = None, tmo = -1):
		return self.EL.wait(nmax, tmo)

	def sectionState(self, sid):
		return self.EL.sectionState(sid)
		
	def processStates(self, sid):
		return self.EL.processStates(sid)
		
	def sectionStateChanged(self, sid, oldstate, newstate):
		pass
				
	def processStateChanged(self, sid, pno, oldstate, newstate):
		pass
				
	def sectionDeleted(self, sid):
		pass

	def destroy(self):
		self.EL = None		# need to break this link so that memory
							# can be reclaimed
		
if __name__ == '__main__':
	import sys
	fc = FBSClient(cfg = sys.argv[1])
	print fc.getQueueList()
	print fc.getQueue('LongQ').__dict__
