#
# @(#) $Id: DepParser.py,v 1.15 2003/08/20 18:58:57 ivm Exp $
#
# $Log: DepParser.py,v $
# Revision 1.15  2003/08/20 18:58:57  ivm
# Implemented CPU power, round-robin-over-users scheduling inside queuei,
# other minor things.
#
# Revision 1.14  2001/06/12 19:47:53  ivm
# Updated for Python v2.1
#
# Revision 1.13  2001/02/09 17:29:09  ivm
# Allow section name to start with '_'
#
# Revision 1.12  2001/02/06 18:25:25  ivm
# Fixed yet another memory mishadnling in farm_history, simplified syntax
# Added canceled() dependency type
# Added job ranges in kill.py
# Show processes for just ended sections
# Fixed decoding of status/signal/core in Section
#
# Revision 1.11  2000/06/01 14:06:42  ivm
# Catch SIGINT in JobDB
# Return 'false' if section not found in DepExpression
#
# Revision 1.9  2000/03/23 16:27:18  ivm
# Fixed bugs
#
# Revision 1.8  2000/03/17 16:29:12  ivm
# Use job[sname] instead of job.getSection(sname) in DepParser
#
# Revision 1.6  2000/03/17 16:03:16  ivm
# Added zombie, make failed and exit synonym
#
# Revision 1.4  2000/03/17 15:38:52  ivm
# Added evaluate() and depList()
#
#

import string
import re

"""
Syntax is:

expression  
		= or_expression
		| <empty>
			
or_expression
		= and_expression '|' or_expression
		| and_expression
		
and_expression
		= simple_expression '&' and_expression
		| simple_expression
		
simple_expression
		= '(' or_expression ')'
		| fname '(' sname ')'
		| '~' simple_expression
"""

class	DepTreeNode:
	def __init__(self, t, a1 = None, a2 = None):
		self.Type = t
		self.A1 = a1
		self.A2 = a2
		self.Value = None

	def __repr__(self):
		return '<%s(%s) %s %s>' % (self.Type, self.Value,
			repr(self.A1), repr(self.A2))

	def listOfSections(self):
		lst = []
		self.addToSectList(lst)
		return lst
		
	def addToSectList(self, lst):
		if self.Type == 's':
			if not self.A2 in lst:
				lst.append(self.A2)
		else:
			if self.A1 != None: 	self.A1.addToSectList(lst)
			if self.A2 != None: 	self.A2.addToSectList(lst)
		
	def evaluate(self, job):
		if self.Value == None:
			if self.Type == '&':
				a1 = self.A1.evaluate(job)
				if a1 == 0 or a1 == None:	return a1
				a2 = self.A2.evaluate(job)
				self.Value = self.evalAnd(a1, a2)
			elif self.Type == '|':
				a1 = self.A1.evaluate(job)
				if a1 == 1: return 1
				a2 = self.A2.evaluate(job)
				self.Value = self.evalOr(a1, a2)
			elif self.Type == '~':
				a1 = self.A1.evaluate(job)
				self.Value = self.evalNot(a1)
			elif self.Type == 's':
				self.Value = self.evalSection(job, self.A1, self.A2)
		return self.Value
				
	def evalAnd(self, a1, a2):
		if a1 == 1 and a2 == 1: 	return 1
		if a1 == 0 or a2 == 0:		return 0
		return None
		

	def evalOr(self, a1, a2):
		if a1 == 1 or a2 == 1:		return 1
		if a1 == 0 and a2 == 0: 	return 0
		return None
		
	def evalNot(self, a):
		if a == None:		return None
		return 1-a

	EvalTable = {	# {dtype:{state:answer}}
		'done': {
			'waiting':		None,
			'ready':		None,
			'running':		None,
			'done': 		1,
			'failed':		0
		},
		'failed': {
			'waiting':		None,
			'ready':		None,
			'running':		None,
			'done': 		0,
			'failed':		1
		},
		'ended': {
			'waiting':		None,
			'ready':		None,
			'running':		None,
			'done': 		1,
			'failed':		1
		},
		'started': {
			'waiting':		None,
			'ready':		None,
			'running':		1,
			'done': 		1,
			'failed':		1
		},
		'canceled': {
			'waiting':		None,
			'ready':		None,
			'canceled':		1
		},
		'zombie': {
			'waiting':		None,
			'ready':		None,
			'zombie':		1
		},
	}

					
	def evalSection(self, job, dtype, sname):
		s = job[sname]
		st = s.state()
		try:
			return self.EvalTable[dtype][st]
		except:
			return 0

class	DepExpression:

	FcnRe = re.compile('done|exit|failed|ended|started|zombie|canceled')
	SNRe = re.compile('[A-Za-z_][A-Za-z0-9_]*')

	def __init__(self, str):
		self.Tree = self.parseString(str)
	
	def evaluate(self, job):
		if self.Tree != None:
			# if some sections not found (DB corrupted)
			# evaluate to zombie
			try:	result = self.Tree.evaluate(job)
			except KeyError:
				return 0
		else:
			result = 1	# no dependencies
		#print 'DepExp: evaluate: tree = ', repr(self.Tree),
		#print '--> ', result
		return result

	def depList(self):
		# return list of sections this section depends on
		if self.Tree == None:	return []
		return self.Tree.listOfSections()
			
	def getToken(self, str):
		# returns (type, value, rest-of-string)
		# removes spaces
		# generates Syntax error
		str = string.strip(str)
		if str[0] == '(':
			return '(',None, str[1:]
		if str[0] == ')':
			return ')',None, str[1:]
		elif str[:2] == '&&':
			return '&',None,str[2:]
		elif str[:2] == '||':
			return '|',None,str[2:]
		elif str[0] == '~' or str[0] == '!':
			return '~',None,str[1:]
		# function name ?
		match = self.FcnRe.match(str)
		if match != None:
			dtype = match.group() 
			if dtype == 'exit': 	dtype = 'failed'	# synonyms
			return 'f', dtype, str[match.end():]
		match = self.SNRe.match(str)
		if match != None:
			return 's', match.group(), str[match.end():]
		raise SyntaxError, str
			
	def tokenize(self, str):
		tokens = []
		while str:
			tt, tv, rest = self.getToken(str)
			tokens.append((tt, tv, str))
			str = rest
		return tokens
		
	def parseString(self, str):
		tokens = self.tokenize(str)
		exp, tokens = self.parseExp(tokens)
		if tokens:
			raise SyntaxError, tokens[0][2]
		return exp

	def parseExp(self, tokens):
		if len(tokens):
			return self.parseOrExp(tokens)
		else:
			return None,[]
	
	def parseOrExp(self, tokens):
		exp, rest = self.parseAndExp(tokens)
		if rest:
			tt, tv, str = rest[0]
			if tt == '|':
				exp1, rest = self.parseOrExp(rest[1:])
				return DepTreeNode('|', exp, exp1), rest
			else:
				return exp, rest
		else:
			return exp, rest

	def parseAndExp(self, tokens):
		exp, rest = self.parseSimpleExp(tokens)
		if rest:
			tt, tv, str = rest[0]
			if tt == '&':
				exp1, rest = self.parseAndExp(rest[1:])
				return DepTreeNode('&', exp, exp1), rest
			else:
				return exp, rest
		else:
			return exp, rest

	def parseSimpleExp(self, tokens):
		tt, tv, str = tokens[0]
		if tt == '(':
			exp, rest = self.parseOrExp(tokens[1:])
			if not rest or rest[0][0] != ')':
				raise SyntaxError, 'Parenthesis missmatch: %s' % str
			return exp, rest[1:]
		elif tt == 'f':
			if len(tokens) < 4 or tokens[1][0] != '(' or \
				(not tokens[2][0] in 'sf') or tokens[3][0] != ')':
					raise SyntaxError, str
			return DepTreeNode('s', tokens[0][1], tokens[2][1]), tokens[4:]
		elif tt == '~':
			exp, rest = self.parseSimpleExp(tokens[1:])
			return DepTreeNode('~',exp, None), rest
		else:
			raise SyntaxError, str

	def tree2pn(self, tree):
		op = tree[0]
		lst = []
		if op in ['&','|']:
			for x in tree[1:]:
				lst = lst + self.tree2pn(x)
		elif op == '~':
			lst = lst + self.tree2pn(tree[1])
		else:
			# fn(sn):
			lst.append(tree[1])
		lst.append(op)
		return lst
					
if __name__ == '__main__':
	dp = DepParser()	
