# -*- coding: ascii -*-
#
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2002-2005 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#  Copyright (C) 2003 by Shun-ichi TAHARA <jado@flowernet.gr.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import os
import re
import sys
import urlparse
import random
import types

if os.environ.has_key("DISPLAY"):
    import gtk

import ninix.seriko
import ninix.menu

import pix

range_scale = [(" 100%",  100),
               ("  90%",   90),
               ("  80%",   80),
               ("  70%",   70),
               ("  60%",   60),
               ("  50%",   50),
               ("  40%",   40),
               (" 200%",  200),
               (" 300%",  300),
               ("1000%", 1000)]

class Surface:
    # keyval/name mapping
    if os.environ.has_key("DISPLAY"):
        from ninix.keymap import keymap_old, keymap_new
    else:
        keymap_old = {}
        keymap_new = {}

    def __init__(self, sakura, debug=0):
        self.sakura = sakura
        self.debug = debug
        self.window = []
        self.__mikire = 0
        self.__kasanari = 0
        self.__menu = None
        self.__scale = 100 # %
        self.desc = None
        self.__use_pna = 0

    def set_debug(self, debug):
        self.debug = debug
        for surface_window in self.window:
            surface_window.set_debug(debug)
            surface_window.reset_surface()

    def set_use_pna(self, flag):
        if flag:
            self.__use_pna = 1
        else:
            self.__use_pna = 0
        for surface_window in self.window:
            surface_window.set_use_pna(flag)
            surface_window.reset_surface()

    def get_scale(self):
        return self.__scale

    def set_scale(self, scale):
        ##if self.__scale == scale:
        ##    return
        self.__scale = scale # %
        for surface_window in self.window:
            surface_window.set_scale(scale)
            surface_window.reset_surface()

    def finalize(self):
        for surface_window in self.window:
            surface_window.destroy()
        self.window = []
        ##self.__menu.finalize()
        self.__menu = None

    def create_gtk_window(self, title):
        window = pix.TranslucentWindow()
        window.set_title(title)
        window.set_decorated(False)
        window.set_resizable(False)
        window.connect("delete_event", self.delete)
        window.connect("key_press_event", self.key_press)
        window.connect("window_state_event", self.window_state)
        window.set_events(gtk.gdk.KEY_PRESS_MASK)
        window.realize()
        return window

    def window_state(self, window, event):
        if not self.sakura.running: ## FIXME
            return
        if event.new_window_state & gtk.gdk.WINDOW_STATE_ICONIFIED:
            if window == self.window[0].window:
                self.sakura.ghost.notify_iconified() ## FIXME
            for surface_window in self.window:
                gtk_window = surface_window.window
                if gtk_window != window and not gtk_window.window.get_state() & gtk.gdk.WINDOW_STATE_ICONIFIED:
                    gtk_window.iconify()
        else:
            for surface_window in self.window:
                gtk_window = surface_window.window
                if gtk_window != window and gtk_window.window.get_state() & gtk.gdk.WINDOW_STATE_ICONIFIED:
                    gtk_window.deiconify()
            if window == self.window[0].window:
                self.sakura.ghost.notify_deiconified() ## FIXME
        return

    def delete(self, window, event):
        self.sakura.ghost.quit() ## FIXME
        return False

    def key_press(self, window, event):
        name = self.keymap_old.get(event.keyval, event.string)
        keycode = self.keymap_new.get(event.keyval, event.string)
        if name or keycode:
            self.sakura.notify_event("OnKeyPress", name, keycode)
        return True

    def window_stick(self, action):
        stick = self.__menu.get_stick()
        for window in self.window:
            if stick:
                window.window.stick()
            else:
                window.window.unstick()

    def open_popup_menu(self, button, side):
        self.__menu.popup(button, side)

    re_surface_id = re.compile("^surface([0-9]+)$")

    def new(self, desc, alias, surface, name, prefix):
        self.desc = desc
        self.name = name
        self.prefix = prefix
        if alias is None:
            alias0 = alias1 = None
        else:
            alias0 = alias.get("sakura.surface.alias")
            alias1 = alias.get("kero.surface.alias")
        # load pixmap
        pixbufs = {}
        elements = {}
        for basename in surface.keys():
            path, config = surface[basename]
            if path is None:
                continue
            if not os.path.exists(path):
                name, suffix = os.path.splitext(path)
                dgp_path = name + '.dgp'
                if not os.path.exists(dgp_path):
                    print "%s: file not found (ignored)" % path
                    continue
                else:
                    path = dgp_path
            elements[basename] = [[path], None, None]
            match = self.re_surface_id.match(basename)
            if not match:
                continue
            key = match.group(1)
            pixbufs[key] = elements[basename]
        # compose surface elements
        composite_pixbuf = {}
        for basename in surface.keys():
            match = self.re_surface_id.match(basename)
            if not match:
                continue
            key = match.group(1)
            path, config = surface[basename]
            if config.has_key("element0"):
                if self.debug & 8192:
                    print "surface", key
                composite_pixbuf[key] = self.compose_elements(elements, config)
        pixbufs.update(composite_pixbuf)
        # check if necessary surfaces have been loaded
        for key in ["0", "10"]:
            if not pixbufs.has_key(key):
                sys.stderr.write("cannot load surface #%s (abort)\n" % key)
                sys.exit(1)
        self.__pixbufs = pixbufs
        # arrange surface configurations
        seriko = {}
        region = {}
        for basename in surface.keys():
            match = self.re_surface_id.match(basename)
            if not match:
                continue
            key = match.group(1)
            path, config = surface[basename]
            # define animation patterns
            seriko[key] = ninix.seriko.get_actors(config)
            # define collision areas
            buffer = []
            for n in range(256):
                rect = config.get("collision" + str(n)) # "redo" syntax
                if rect is None:
                    continue
                values = rect.split(",")
                if len(values) != 5:
                    continue
                try:
                    x1, y1, x2, y2 = [int(value) for value in values[:4]]
                except ValueError:
                    continue
                buffer.append((values[4].strip(), x1, y1, x2, y2))
            for part in ["head", "face", "bust"]:
                rect = config.get("collision." + part) # "inverse" syntax
                if not rect:
                    continue
                try:
                    x1, y1, x2, y2 = [int(value) for value in rect.split(",")]
                except ValueError:
                    pass
                buffer.append((part.capitalize(), x1, y1, x2, y2))
            region[key] = buffer
        self.__seriko = seriko
        self.__region = region
        # MAYUNA
        mayuna = {}
        for basename in surface.keys():
            match = self.re_surface_id.match(basename)
            if not match:
                continue
            key = match.group(1)
            path, config = surface[basename]
            # define animation patterns
            mayuna[key] = ninix.seriko.get_mayuna(config)
        bind = {}
        for side in ["sakura", "kero"]:
            bind[side] = {}
            for index in range(128):
                name = self.desc.get('%s.bindgroup%d.name' % (side, index), None)
                default = self.desc.get('%s.bindgroup%d.default' % (side, index), 0)
                if name != None:
                    bind[side][index] = [name, default]
        self.mayuna = {}
        for side in ["sakura", "kero"]:
            self.mayuna[side] = []
            for index in range(128):
                key = self.desc.get('%s.menuitem%d' % (side, index), None)
                if key == '-':
                    self.mayuna[side].append([key, None, 0])
                else:
                    try:
                        key = int(key)
                    except:
                        pass
                    else:
                        if bind[side].has_key(key):
                            name = bind[side][key][0].split(',')
                            self.mayuna[side].append([key, name[1], bind[side][key][1]])
        # create surface windows
        for surface_window in self.window:
            surface_window.destroy()
        self.window = []
        for name, side, default, alias in [('sakura', 0, "0", alias0), ('kero', 1, "10", alias1)]:
            if name == "sakura":
                title = self.sakura.get_selfname() or 'surface.' + name
            else:
                title = self.sakura.get_keroname() or 'surface.' + name
            gtk_window = self.create_gtk_window(title)
            surface_window = SurfaceWindow(
                gtk_window, side, self.sakura, desc, alias, surface,
                pixbufs, seriko, region, mayuna, bind[name],
                default, self.__use_pna, self.debug)
            self.window.append(surface_window)
        self.__surface = surface
        self.__menu = ninix.menu.Menu(self.sakura.ghost, self) ## FIXME
        dir = self.prefix
        name = self.desc.get("menu.background.bitmap.filename", "menu_background.png")
        name = name.replace('\\', '/')
        path = os.path.join(dir, name)
        if not os.path.exists(path):
            dir = os.path.join(self.sakura.ghost.get_prefix(), "ghost", "master") ## FIXME
            path = os.path.join(dir, "menu_background.png")
            if os.path.exists(path):
                path_background = path
            else:
                path_background = None
            path = os.path.join(dir, "menu_sidebar.png")
            if os.path.exists(path):
                path_sidebar = path
            else:
                path_sidebar = None
        else:
            path_background = path
            name = self.desc.get("menu.sidebar.bitmap.filename", "menu_sidebar.png")
            name = name.replace('\\', '/')
            path = os.path.join(dir, name)
            if os.path.exists(path):
                path_sidebar = path
            else:
                path_sidebar = None
        if path_background:
            self.__menu.set_pixmap(path_background, path_sidebar)
        fontcolor_r = self.desc.getint('menu.background.font.color.r', 0)
        fontcolor_g = self.desc.getint('menu.background.font.color.g', 0)
        fontcolor_b = self.desc.getint('menu.background.font.color.b', 0)
        self.__menu.set_fontcolor(fontcolor_r, fontcolor_g, fontcolor_b)
        self.__menu.create_mayuna_menu(self.get_mayuna_menu())

    def add_window(self, side, default):
        assert side >= 2 and len(self.window) == side ## FIXME
        name = "char%d" % side
        title = 'surface.' + name
        gtk_window = self.create_gtk_window(title)
        surface_window = SurfaceWindow(
            gtk_window, side, self.sakura, self.desc, None, self.__surface,
            self.__pixbufs, self.__seriko, self.__region, {}, {},
            default, self.__use_pna, self.debug)
        self.window.append(surface_window)
        surface_window.set_scale(self.__scale)
        ##surface_window.set_surface(default)

    def get_mayuna_menu(self):
        for side, index in [('sakura', 0), ('kero', 1)]:
            for menu in self.mayuna[side]:
                if menu[0] != '-':
                    menu[2] = self.window[index].bind[menu[0]][1]
        return self.mayuna

    def compose_elements(self, elements, config):
        error = None
        for n in range(256):
            key = "element" + str(n)
            if not config.has_key(key):
                break
            spec = [value.strip() for value in config[key].split(",")]
            try:
                method, filename, x, y = spec
                x = int(x)
                y = int(y)
            except ValueError:
                error = "invalid element spec for %s: %s" % (key, config[key])
                break
            basename, suffix = os.path.splitext(filename)
            if suffix.lower() != ".png":
                error = "unsupported file format for %s: %s" % (key, filename)
                break
            basename = basename.lower()
            if not elements.has_key(basename):
                error = "%s file not found: %s" % (key, filename)
                break
            pixbuf = elements[basename][0][0]
            if n == 0: # base surface
                pixbuf_list = [pixbuf]
            elif method == "overlay":
                pixbuf_list.append((pixbuf, x, y))
            elif method == "base":
                pixbuf_list.append((pixbuf, x, y))
            else:
                error = "unknown method for %s: %s" % (key, method)
                break
            if self.debug & 8192:
                print "%s: %s %s, x=%d, y=%d, w=%d, h=%d" % (
                    key, method, filename, x, y, w, h)
        if error is not None:
            print error
            pixbuf_list = []
        return [pixbuf_list, None, None]

    def reset_surface(self, side):
        if len(self.window) > side:
            self.window[side].reset_surface()

    def set_surface_default(self):
        for side in range(len(self.window)):
            self.window[side].set_surface_default()

    def set_surface(self, side, id):
        if len(self.window) > side:
            self.window[side].set_surface(id)

    def get_surface(self, side):
        if len(self.window) > side:
            return self.window[side].get_surface()

    def get_surface_size(self, side):
        if len(self.window) > side:
            return self.window[side].get_surface_size()

    def get_surface_offset(self, side):
        if len(self.window) > side:
            return self.window[side].get_surface_offset()

    def get_touched_region(self, side, x, y):
        if len(self.window) > side:
            return self.window[side].get_touched_region(x, y)


    def get_direction(self, side):
        if len(self.window) > side:
            return self.window[side].get_direction()

    def set_direction(self, side, dir):
        if len(self.window) > side:
            self.window[side].set_direction(dir)

    def reset_balloon_position(self):
        top_margin = self.sakura.ghost.get_top_margin() ## FIXME
        bottom_margin = self.sakura.ghost.get_bottom_margin() ## FIMXE
        scrn_w = gtk.gdk.screen_width()
        scrn_h = gtk.gdk.screen_height() - bottom_margin
        for side in range(len(self.window)):
            x, y = self.get_position(side)
            sox, soy = self.window[side].get_surface_offset()
            w, h = self.get_surface_size(side)
            bw, bh = self.sakura.ghost.get_balloon_size(side) ## FIXME
            ox, oy = self.get_balloon_offset(side)
            direction = self.get_direction(side)
            align = self.get_alignment(side)
            if direction == 0:
                bx = max(x + sox - bw + ox, 0)
            else:
                bx =  min(x + sox + w - ox, scrn_w - bw)
            if align == 0:
                by = min(y + soy + oy, scrn_h - bh)
            elif align == 1:
                by = max(y + soy + oy, 0 + top_margin)
            else:
                if y + soy < scrn_h/2:
                    by = max(y + soy + oy, 0 + top_margin)
                else:
                    by = min(y + soy + oy, scrn_h - bh)
            self.set_position(side, x, y)
            self.set_direction(side, direction)
            self.sakura.ghost.set_balloon_position(side, bx, by) ## FIXME

    def reset_position(self):
        top_margin = self.sakura.ghost.get_top_margin() ## FIXME
        bottom_margin = self.sakura.ghost.get_bottom_margin() ## FIMXE
        scrn_w = gtk.gdk.screen_width()
        scrn_h = gtk.gdk.screen_height() - bottom_margin
        for side in range(len(self.window)):
            align = self.get_alignment(side)
            if side == 0: # sakura
                w, h = self.get_surface_size(side)
                x = scrn_w - w
                if align == 1: # top
                    y = 0 + top_margin
                else:
                    y = scrn_h - h
                bw, bh = self.sakura.ghost.get_balloon_size(side) ## FIXME
                ox, oy = self.get_balloon_offset(side)
                direction = 0 # left
                bx = max(x - bw + ox, 0)
            else:
                b0w, b0h = self.sakura.ghost.get_balloon_size(side - 1) ## FIMXE
                b1w, b1h = self.sakura.ghost.get_balloon_size(side) ## FIMXE
                o0x, o0y = self.get_balloon_offset(side - 1)
                o1x, o1y = self.get_balloon_offset(side)
                w, h = self.get_surface_size(side)
                offset = max(0, b1w - (b0w - o0x))
                if (s0x + o0x - b0w) - offset - w + o1x < 0:
                    x = 0
                else:
                    x = (s0x + o0x - b0w) - offset - w + o1x
                if align == 1: # top
                    y = 0 + top_margin
                else:
                    y = scrn_h - h
                direction = 1 # right
                bx =  min(x + w - o1x, scrn_w - b1w)
            if align == 0:
                by = min(y + oy, scrn_h - bh)
            elif align == 1:
                by = max(y + oy, 0 + top_margin)
            else:
                if y < scrn_h/2:
                    by = max(y + oy, 0 + top_margin)
                else:
                    by = min(y + oy, scrn_h - bh)
            self.set_position(side, x, y)
            self.set_direction(side, direction)
            self.sakura.ghost.set_balloon_position(side, bx, by) ## FIXME
            s0x, s0y, s0w, s0h = x, y, w, h

    def set_position(self, side, x, y):
        if len(self.window) > side:
            self.window[side].set_position(x, y)

    def get_position(self, side):
        if len(self.window) > side:
            return self.window[side].get_position()

    def set_alignment_current(self):
        for side in range(len(self.window)):
            self.window[side].set_alignment_current()

    def set_alignment(self, side, align):
        if len(self.window) > side:
            self.window[side].set_alignment(align)

    def get_alignment(self, side):
        if len(self.window) > side:
            return self.window[side].get_alignment()

    def reset_alignment(self):
        if self.desc.get('seriko.alignmenttodesktop') == 'free':
            align = 2
        else:
            align = 0
        for side in range(len(self.window)):
            self.set_alignment(side, align)

    def is_shown(self, side):
        if len(self.window) > side:
            if self.window[side].is_shown():
                return 1
            else:
                return 0
        else:
            return 0

    def show(self, side):
        if len(self.window) > side:
            self.window[side].show()

    def hide_all(self):
        for side in range(len(self.window)):
            self.window[side].hide()

    def hide(self, side):
        if len(self.window) > side:
            self.window[side].hide()

    def raise_all(self):
        for side in range(len(self.window)):
            self.window[side].raise_()

    def raise_(self, side):
        if len(self.window) > side:
            self.window[side].raise_()

    def lower_all(self):
        for side in range(len(self.window)):
            self.window[side].lower()

    def lower(self, side):
        if len(self.window) > side:
            self.window[side].lower()

    def invoke(self, side, id):
        if len(self.window) > side:
            self.window[side].invoke(id)

    def invoke_yen_e(self, side, id):
        if len(self.window) > side:
            self.window[side].invoke_yen_e(id)

    def invoke_talk(self, side, id, count):
        if len(self.window) > side:
            return self.window[side].invoke_talk(id, count)

    def update(self):
        for surface_window in self.window:
            surface_window.update()

    def set_icon(self, path):
        pixbuf = None
        if path != None:
            try:
                pixbuf = pix.create_pixbuf_from_file(path, 'ico')
            except:
                pixbuf = None
        for window in self.window:
            window.window.set_icon(pixbuf)

    def __check_mikire_kasanari(self):
        scrn_w = gtk.gdk.screen_width()
        scrn_h = gtk.gdk.screen_height()
        x0, y0 = self.get_position(0)
        x1, y1 = self.get_position(1)
        s0w, s0h = self.get_surface_size(0)
        s1w, s1h = self.get_surface_size(1)
        if x0 + s0w/4 < 0 or x0 + s0w*3/4 > scrn_w or \
           y0 + s0h/4 < 0 or y0 + s0h*3/4 > scrn_h:
            self.__mikire = 1
        else:
            self.__mikire = 0
        if (x1 + s1w/2 > x0 and x1 + s1w/2 < x0 + s0w and \
            y1 + s1h/2 > y0 and y1 + s1h/2 < y0 + s0h) or \
           (x0 + s0w/2 > x1 and x0 + s0w/2 < x1 + s1w and \
            y0 + s0h/2 > y1 and y0 + s0h/2 < y1 + s1h):
            self.__kasanari = 1
        else:
            self.__kasanari = 0

    def get_mikire_kasanari(self):
        self.__check_mikire_kasanari()
        return self.__mikire, self.__kasanari

    def get_name(self):
        return self.name

    def get_username(self):
        if self.desc is None:
            return
        else:
            return self.desc.get("user.defaultname")

    def get_selfname(self):
        if self.desc is None:
            return
        else:
            return self.desc.get("sakura.name")

    def get_selfname2(self):
        if self.desc is None:
            return
        else:
            return self.desc.get("sakura.name2")

    def get_keroname(self):
        if self.desc is None:
            return
        else:
            return self.desc.get("kero.name")

    def get_friendname(self):
        if self.desc is None:
            return
        else:
            return self.desc.get("sakura.friend.name")

    def get_balloon_offset(self, side):
        if side == 0:
            x, y = self.window[side].get_balloon_offset()
            if x == None:
                x = self.desc.getint("sakura.balloon.offsetx", 0)
            if y == None:
                y = self.desc.getint("sakura.balloon.offsety", 0)
        elif side == 1:
            x, y = self.window[side].get_balloon_offset()
            if x == None:
                x = self.desc.getint("kero.balloon.offsetx", 0)
            if y == None:
                y = self.desc.getint("kero.balloon.offsety", 0)
        else:
            x, y = None, None
            if len(self.window) > side:
                x, y = self.window[side].get_balloon_offset()
            if x == None:
                x = self.desc.getint("char%d.balloon.offsetx" % side, 0)
            if y == None:
                y = self.desc.getint("char%d.balloon.offsety" % side, 0)
        if self.__scale != 100:
            x = x * self.__scale / 100
            y = y * self.__scale / 100
        return x, y

    def toggle_bind(self, event, args):
        side, id = args
        self.window[side].toggle_bind(id)

    def get_collision_area(self, side, part):
        if len(self.window) > side:
            for p, x1, y1, x2, y2 in self.window[side].collisions:
                if p == part:
                    if self.__scale != 100:
                        x1 = x1 * self.__scale / 100
                        x2 = x2 * self.__scale / 100
                        y1 = y1 * self.__scale / 100
                        y2 = y2 * self.__scale / 100
                    return x1, y1, x2, y2
        return None

    def get_config_int(self, side, name):
        if len(self.window) > side:
            basename = 'surface' +  self.window[side].surface_id
            path, config = self.window[side].surface[basename]
            return config.getint(name)

class SurfaceWindow:
    # DnD data types
    dnd_targets = [
        ("text/plain", 0, 0),
        ]

    def __init__(self, window, side, sakura, desc, alias, surface,
                 pixbuf, seriko, region, mayuna, bind,
                 default_id, use_pna, debug):
        self.window = window
        self.side = side
        self.sakura = sakura
        self.desc = desc
        self.alias = alias
        self.align = 0
        self.__scale = 100 # %
        self.__use_pna = use_pna
        if self.alias is not None:
            default_id = self.alias.get(default_id, [default_id])[0]
        self.surface = surface
        self.surface_id = default_id
        self.base_id = default_id
        self.exclusive_actor = None
        self.pixbuf = pixbuf
        self.seriko = seriko
        self.region = region
        self.mayuna = mayuna
        self.bind = bind
        self.surface_cache = {}
        self.default_id = default_id
        self.debug = debug
        self.__shown = 0
        self.dragged = 0
        self.window_offset = (0, 0)
        self.set_position(0, 0)
        self.set_direction(0)
        self.x_root = None
        self.y_root = None
        self.reset_overlays()
        # create drawing area
        self.darea = gtk.DrawingArea()
        self.darea.show()
        self.darea.set_events(gtk.gdk.EXPOSURE_MASK|
                              gtk.gdk.BUTTON_PRESS_MASK|
                              gtk.gdk.BUTTON_RELEASE_MASK|
                              gtk.gdk.POINTER_MOTION_MASK|
                              gtk.gdk.POINTER_MOTION_HINT_MASK|
                              gtk.gdk.SCROLL_MASK)
        self.callbacks = []
        for signal, func in [("expose_event",         self.redraw),
                             ("button_press_event",   self.button_press),
                             ("button_release_event", self.button_release),
                             ("motion_notify_event",  self.motion_notify),
                             ("drag_data_received",   self.drag_data_received),
                             ("scroll_event",         self.scroll),
                             ]:
            self.callbacks.append(self.darea.connect(signal, func))
        self.darea.drag_dest_set(gtk.DEST_DEFAULT_ALL, self.dnd_targets,
                                 gtk.gdk.ACTION_COPY)
        self.window.add(self.darea)
        self.darea.realize()
        self.darea.window.set_back_pixmap(None, False)
        self.set_surface(None)

    def set_debug(self, debug):
        self.debug = debug

    def set_use_pna(self, flag):
        if flag:
            self.__use_pna = 1
        else:
            self.__use_pna = 0
        self.reset_pixbuf_cache()

    def get_scale(self):
        return self.__scale

    def set_scale(self, scale):
        ##if self.__scale == scale:
        ##    return
        self.__scale = scale # %

    def drag_data_received(self, widget, context, x, y, data, info, time):
        if self.debug & 8192:
            print "Content-type:", data.type
            print "Content-length:", data.length
            print repr(data.data)
        if str(data.type) == "text/plain":
            list = []
            for line in data.data.split("\r\n"):
                scheme, host, path, params, query, fragment = \
                        urlparse.urlparse(line)
                if scheme == "file" and os.path.exists(path):
                    list.append(path)
            if list:
                self.sakura.notify_file_drop(list, self.side)
        return True

    def invoke(self, id, update=0):
        for actor in self.seriko[self.base_id]:
            if id == actor.get_id():
                actor.invoke()
                if update:
                    actor.update(self)
                if actor.exclusive:
                    self.exclusive_actor = actor
                    break

    def invoke_yen_e(self, id):
        for actor in self.seriko[id]:
            if actor.get_interval() == "yen-e":
                actor_id = actor.get_id()
                self.invoke(actor_id)
                break

    def invoke_talk(self, id, count):
        interval_count = None
        for actor in self.seriko[id]:
            interval = actor.get_interval()
            if interval[:4] == "talk":
                interval_count = int(interval[5])
                actor_id = actor.get_id()
                break
        if interval_count != None and count >= interval_count:
            self.invoke(actor_id)
            return 1
        else:
            return 0

    def update(self):
        if self.exclusive_actor:
            self.reset_overlays()
            self.exclusive_actor.update(self)
            self.draw_overlays()
            if self.exclusive_actor == None or \
               self.exclusive_actor.wait == -1:
                self.exclusive_actor = None
        else:
            last_overlays = self.overlays.copy()
            for actor in self.seriko[self.base_id]:
                if actor.exclusive and actor.wait == 1:
                    self.reset_overlays()
                    actor.update(self)
                    self.draw_overlays()
                    if self.exclusive_actor != None and \
                       self.exclusive_actor.wait != -1:
                        self.exclusive_actor = actor
                    else:
                        self.exclusive_actor = None
                    return
                else:
                    actor.update(self)
            if self.overlays != last_overlays:
                self.draw_overlays()

    def terminate(self):
        for actor in self.seriko[self.base_id]:
            actor.terminate()
        self.reset_overlays()

    def reset_surface(self):
        id = self.get_surface()
        self.set_surface(id, restart=0)

    def set_surface_default(self):
        self.set_surface(self.default_id)

    def set_surface(self, id, restart=1):
        if self.alias is not None and self.alias.has_key(id):
            aliases = self.alias.get(id)
            if len(aliases) > 0:
                id = random.choice(aliases)
        if id == "-2" or restart:
            self.terminate()
        not_update = 0
        if id in ["-1", "-2"]:
            pass
        elif not self.pixbuf.has_key(id):
            self.surface_id = self.default_id
            not_update = 1
        else:
            self.surface_id = id
        if restart:
            if self.seriko.has_key(id):
                self.base_id = id
            else:
                self.base_id = self.surface_id
            self.exclusive_actor = None
            for actor in self.seriko[self.base_id]:
                if actor.get_interval() == "runonce":
                    actor.invoke()
                    if actor.exclusive:
                        self.exclusive_actor = actor
                        break
        # define collision areas
        self.collisions = self.region[self.surface_id]
        # update window offset
        x, y = self.get_position()
        w, h = self.get_surface_size(self.surface_id)
        dw, dh = self.get_surface_size(self.default_id) # default surface size
        xoffset = (dw - w)/2
        if self.get_alignment() == 0:
            yoffset = dh - h
            scrn_h = gtk.gdk.screen_height() - self.sakura.ghost.get_bottom_margin() ## FIXME
            y = scrn_h - dh
        elif self.get_alignment() == 1:
            yoffset = 0
        else:
            yoffset = (dh - h)/2
        self.window_offset = (xoffset, yoffset)
        if not_update:
            return
        self.draw_surface()
        # resize window
        self.darea.set_size_request(w, h)
        self.show_surface()
        # relocate window
        self.set_position(x, y)
        self.sakura.ghost.notify_surface_change(self.side) ## FIXME

    def compose_surface(self, surface_pixbuf, mayuna, done):
        for pattern in mayuna.patterns:
            surface, interval, method, args = pattern
            if method in ['bind', 'add']:
                if self.pixbuf.has_key(surface):
                    x, y = args
                    pixbuf = self.get_pixbuf(surface)
                    w = pixbuf.get_width()
                    h = pixbuf.get_height()
                    # overlay surface pixbuf
                    sw = surface_pixbuf.get_width()
                    sh = surface_pixbuf.get_height()
                    if x + w > sw:
                        w = sw - x
                    if y + h > sh:
                        h = sh - y
                    if x < 0:
                        dest_x = 0
                        w += x
                    else:
                        dest_x = x
                    if y < 0:
                        dest_y = 0
                        h += y
                    else:
                        dest_y = y
                    pixbuf.composite(surface_pixbuf, dest_x, dest_y, w, h, x, y, 1.0, 1.0, gtk.gdk.INTERP_BILINEAR, 255)
            elif method == 'reduce':
                if self.pixbuf.has_key(surface):
                    x, y = args
                    pixbuf = self.get_pixbuf(surface)
                    w = pixbuf.get_width()
                    h = pixbuf.get_height()
                    surface_pixbuf = pix.reduce_pixbuf(surface_pixbuf, pixbuf, x, y)
            elif method == 'insert':
                index = args[0]
                for actor in self.mayuna[self.surface_id]:
                    id = actor.get_id()
                    if id == index:
                        if self.bind.has_key(id) and self.bind[id][1] and \
                           id not in done:
                            done.append(id)
                            sueface_pixbuf = self.compose_surface(surface_pixbuf, actor, done)
                        else:
                            break
            else:
                raise RuntimeError, "should not reach here"
        return surface_pixbuf

    def reset_pixbuf_cache(self):
        self.surface_cache = {}
        for id in self.pixbuf.keys():
            self.pixbuf[id][1] = None

    def get_pixbuf(self, id):
        if not self.pixbuf.has_key(id):
            return pix.create_blank_pixbuf(8, 8)
        if self.pixbuf[id][1] != None:
            pixbuf = self.pixbuf[id][1]
        else:
            if len(self.pixbuf[id][0]) == 0:
                pixbuf = pix.create_blank_pixbuf(8, 8)
            else:
                try:
                    pixbuf = pix.create_pixbuf_from_file(self.pixbuf[id][0][0], use_pna=self.__use_pna)
                except:
                    if self.debug & 4:
                        print 'cannot load surface #%d' % id
                    return pix.create_blank_pixbuf(8, 8)
                for element, x, y in self.pixbuf[id][0][1:]:
                    try:
                        overlay = pix.create_pixbuf_from_file(element, use_pna=self.__use_pna)
                    except:
                        continue
                    w = overlay.get_width()
                    h = overlay.get_height()
                    sw = pixbuf.get_width()
                    sh = pixbuf.get_height()
                    if x + w > sw:
                        w = sw - x
                    if y + h > sh:
                        h = sh - y
                    if x < 0:
                        dest_x = 0
                        w += x
                    else:
                        dest_x = x
                    if y < 0:
                        dest_y = 0
                        h += y
                    else:
                        dest_y = y
                    overlay.composite(pixbuf, dest_x, dest_y, w, h, x, y, 1.0, 1.0, gtk.gdk.INTERP_BILINEAR, 255)
            self.pixbuf[id][1] = pixbuf
            for key in self.pixbuf.keys():
                if key not in [id, '0', '10'] and self.pixbuf[key][2] != None:
                    self.pixbuf[key][2] = self.pixbuf[key][2] - 1
                    if self.pixbuf[key][2] < 0:
                        self.pixbuf[key][1] = None
                        self.pixbuf[id][2] = None 
        self.pixbuf[id][2] = 15
        return pixbuf

    def draw_surface(self):
        pixbuf = self.get_pixbuf(self.surface_id)
        w = pixbuf.get_width()
        h = pixbuf.get_height()
        if self.surface_cache.has_key(self.surface_id):
            surface_pixbuf =  self.surface_cache[self.surface_id].copy()
        else:
            surface_pixbuf = pixbuf.copy()
            if self.mayuna.has_key(self.surface_id):
                done = []
                for actor in self.mayuna[self.surface_id]:
                    id = actor.get_id()
                    if self.bind.has_key(id) and self.bind[id][1] and \
                       id not in done:
                        done.append(id)
                        sueface_pixbuf = self.compose_surface(surface_pixbuf, actor, done)
                self.surface_cache[self.surface_id] = surface_pixbuf
        if self.__scale != 100:
            w = w * self.__scale / 100
            h = h * self.__scale / 100
            surface_pixbuf = surface_pixbuf.scale_simple(w, h, gtk.gdk.INTERP_BILINEAR)
        self.surface_pixbuf = surface_pixbuf
        surface_pixmap, self.mask_pixmap = surface_pixbuf.render_pixmap_and_mask(1)

    def draw_region(self):
        surface_gc = self.darea.window.new_gc()
        surface_gc.function = gtk.gdk.INVERT
        for part, x1, y1, x2, y2 in self.collisions:
            if self.__scale != 100:
                x1 = x1 * self.__scale / 100
                x2 = x2 * self.__scale / 100
                y1 = y1 * self.__scale / 100
                y2 = y2 * self.__scale / 100
            self.darea.window.draw_rectangle(surface_gc, 0, x1, y1, x2 - x1, y2 - y1)

    def show_surface(self):
        self.darea.queue_draw()
        self.window.shape_combine_mask(self.mask_pixmap, 0, 0)

    def redraw(self, darea, event):
        self.darea.window.draw_pixbuf(None, self.surface_pixbuf, 0, 0, 0, 0, -1, -1)
        if self.debug & 4096:
            self.draw_region()

    def reset_overlays(self):
        self.overlays = {}

    def remove_overlay(self, actor):
        try:
            del self.overlays[actor]
        except KeyError:
            pass

    def add_overlay(self, actor, id, x, y):
        if id == "-2":
            self.terminate()
        if id in ["-1", "-2"]:
            self.remove_overlay(actor)
            return
        self.overlays[actor] = (id, x, y)

    def draw_overlays(self):
        ##print "draw_overlays()"
        pixbuf = self.get_pixbuf(self.surface_id)
        w = pixbuf.get_width()
        h = pixbuf.get_height()
        if self.surface_cache.has_key(self.surface_id):
            surface_pixbuf =  self.surface_cache[self.surface_id].copy()
        else:
            surface_pixbuf = pixbuf.copy()
            if self.mayuna.has_key(self.surface_id):
                done = []
                for actor in self.mayuna[self.surface_id]:
                    id = actor.get_id()
                    if self.bind.has_key(id) and self.bind[id][1] and \
                       id not in done:
                        done.append(id)
                        sueface_pixbuf = self.compose_surface(surface_pixbuf, actor, done)
                self.surface_cache[self.surface_id] = surface_pixbuf.copy()
        actors = self.overlays.keys()
        actors.sort(lambda a1, a2: cmp(a1.get_id(), a2.get_id()))
        for actor in actors:
            id, x, y = self.overlays[actor]
            ##print "actor=%d, id=%s, x=%d, y=%d" % (actor.get_id(), id, x, y)
            try:
                pixbuf = self.get_pixbuf(id)
                w = pixbuf.get_width()
                h = pixbuf.get_height()
            except:
                continue
            # overlay surface pixbuf
            sw = surface_pixbuf.get_width()
            sh = surface_pixbuf.get_height()
            if x + w > sw:
                w = sw - x
            if y + h > sh:
                h = sh - y
            if x < 0:
                dest_x = 0
                w += x
            else:
                dest_x = x
            if y < 0:
                dest_y = 0
                h += y
            else:
                dest_y = y
            pixbuf.composite(surface_pixbuf, dest_x, dest_y, w, h, x, y, 1.0, 1.0, gtk.gdk.INTERP_BILINEAR, 255)
        if self.__scale != 100:
            w = surface_pixbuf.get_width() * self.__scale / 100
            h = surface_pixbuf.get_height() * self.__scale / 100
            surface_pixbuf = surface_pixbuf.scale_simple(w, h, gtk.gdk.INTERP_BILINEAR)
        self.surface_pixbuf = surface_pixbuf
        surface_pixmap, self.mask_pixmap = surface_pixbuf.render_pixmap_and_mask(1)
        self.show_surface()

    def __move(self, xoffset=0, yoffset=0):
        x, y = self.get_position()
        p, q = self.window_offset
        self.window.move(x + p + xoffset, y + q + yoffset)        

    def move_surface(self, xoffset, yoffset):
        self.__move(xoffset, yoffset)
        self.sakura.ghost.notify_surface_move(self.side, xoffset, yoffset) ## FIXME

    def get_balloon_offset(self):
        path, config = self.surface['surface' + self.surface_id]
        if self.side == 0:
            x = config.getint("sakura.balloon.offsetx")
            y = config.getint("sakura.balloon.offsety")
        else:
            x = config.getint("kero.balloon.offsetx")
            y = config.getint("kero.balloon.offsety")
        return x, y

    def get_surface(self):
        return self.surface_id

    def get_surface_size(self, id=None):
        if id == None:
            pixbuf = self.get_pixbuf(self.surface_id)
        else:
            pixbuf = self.get_pixbuf(id)
        w = pixbuf.get_width()
        h = pixbuf.get_height()
        if self.__scale != 100:
            w = int(w * self.__scale / 100)
            h = int(h * self.__scale / 100)
        return w, h

    def get_surface_offset(self):
        return self.window_offset

    def get_touched_region(self, x, y):
        for part, x1, y1, x2, y2 in self.collisions:
            if x1 <= x <= x2 and y1 <= y <= y2:
                ##print part, "touched"
                return part
        return ""

    def get_direction(self):
        return self.direction

    def set_direction(self, dir):
        self.direction = dir # 0: left, 1: right
        self.sakura.ghost.set_balloon_direction(self.side, dir) ## FIXME

    def set_position(self, x, y):
        self.position = (x, y)
        if self.__shown:
            self.__move()
        if x > gtk.gdk.screen_width() / 2:
            dir = 0
        else:
            dir = 1
        self.set_direction(dir)

    def get_position(self):
        return self.position

    def set_alignment_current(self):
        self.set_alignment(self.get_alignment())

    def set_alignment(self, align):
        if align == 0:
            scrn_h = gtk.gdk.screen_height() - self.sakura.ghost.get_bottom_margin() ## FIXME
            sw, sh = self.get_surface_size()
            sx, sy = self.get_position()
            sy = scrn_h - sh
            self.set_position(sx, sy)
        elif align == 1:
            sx, sy = self.get_position()
            sy = 0 + self.sakura.ghost.get_top_margin() ## FIXME
            self.set_position(sx, sy)
        else: # free
            pass
        if align in [0, 1, 2]:
            self.align = align

    def get_alignment(self):
        return self.align

    def destroy(self):
        self.surface_cache = {}
        for tag in self.callbacks:
            self.darea.disconnect(tag)
        self.window.remove(self.darea)
        self.darea.destroy()
        self.window.destroy()

    def is_shown(self):
        if self.__shown:
            return 1
        else:
            return 0

    def show(self):
        if not self.__shown:
            self.darea.show()
            self.window.show()
            self.show_surface()
            self.__move()
            self.__shown = 1

    def hide(self):
        if self.__shown:
            ##self.window.hide()
            self.darea.hide()
            self.__shown = 0

    def raise_(self):
        self.window.window.raise_()

    def lower(self):
        self.window.window.lower()

    def button_press(self, window, event):
        self.sakura.reset_idle_time()
        x = int(event.x)
        y = int(event.y)
        if self.__scale != 100:
            x = int(x * 100 / self.__scale)
            y = int(y * 100 / self.__scale)
        self.x_root = event.x_root
        self.y_root = event.y_root
        if event.type == gtk.gdk.BUTTON_PRESS:
            click = 1
        else:
            click = 2
        self.sakura.ghost.notify_surface_click(event.button, click, self.side, x, y) ## FIXME
        return True

    def button_release(self, window, event):
        if self.dragged:
            self.dragged = 0
        self.x_root = None
        self.y_root = None
        return True

    def motion_notify(self, darea, event):
        if event.is_hint:
            x, y, state = self.darea.window.get_pointer()
        else:
            x, y, state = event.x, event.y, event.state
        if not self.sakura.busy():
            if state & gtk.gdk.BUTTON2_MASK:
                if self.x_root is not None and \
                   self.y_root is not None:
                    x_delta = int(event.x_root - self.x_root)
                    y_delta = int(event.y_root - self.y_root)
                    self.dragged = 1
                    self.sakura.ghost.notify_surface_drag(self.side, x_delta, y_delta) ## FIXME
                    self.x_root = event.x_root
                    self.y_root = event.y_root
            elif state & gtk.gdk.BUTTON1_MASK or \
                 state & gtk.gdk.BUTTON3_MASK:
                pass
            else:
                if self.__scale != 100:
                    x = int(x * 100 / self.__scale)
                    y = int(y * 100 / self.__scale)
                part = self.get_touched_region(x, y)
                if part:
                    cursor = gtk.gdk.Cursor(gtk.gdk.HAND1)
                    self.darea.window.set_cursor(cursor)
                else:
                    self.darea.window.set_cursor(None)
                self.sakura.notify_surface_mouse_motion(self.side, x, y)
        return True

    def scroll(self, darea, event):
        x = int(event.x)
        y = int(event.y)
        if self.__scale != 100:
            x = int(x * 100 / self.__scale)
            y = int(y * 100 / self.__scale)
        if event.direction == gtk.gdk.SCROLL_UP:
            count = 1
        elif event.direction == gtk.gdk.SCROLL_DOWN:
            count = -1
        else:
            count = 0
        if count != 0:
            part = self.get_touched_region(x, y)
            self.sakura.notify_event("OnMouseWheel",
                                     x, y, count, self.side, part)
        return True

    def toggle_bind(self, id):
        if self.bind.has_key(id):
            current = self.bind[id][1]
            self.bind[id][1] = not current
            self.surface_cache = {}
            self.reset_surface()

def test():
    pass

if __name__ == "__main__":
    test()
