# -*- coding: UTF-8 -*-
#
#  Copyright (C) 2004, 2005 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

# TODO:
# - きのことサーフェスの間に他のウインドウが入ることができてしまうのを直す.
# - NAYUKI/2.0
# - 透明度の設定

import os
import sys

if os.environ.has_key("DISPLAY"):
    import gtk
    import gobject
    import pango
    import ninix.pix

import ninix.config
import ninix.seriko

class Menu:

    def __init__(self, kinoko, accelgroup):
        self.__kinoko = kinoko
        ui_info = '''
        <ui>
          <popup name='popup'>
            <menuitem action='Settings'/>
            <menu action='Skin'>
            </menu>
            <separator/>
            <menuitem action='Exit'/>
          </popup>
        </ui>
        '''
        self.__menu_list = {
            'settings': [('Settings', None, ('Settings...(_O)'), None,
                          '', self.__kinoko.edit_preferences),
                         '/ui/popup/Settings'],
            'skin':     [('Skin', None, _('Skin(_K)'), None),
                         None, '/ui/popup/Skin'],
            'exit':     [('Exit', None, _('Exit(_Q)'), None,
                          '', self.__kinoko.close),
                         '/ui/popup/Exit'],
            }
        self.__skin_list = None
        actions = gtk.ActionGroup("Actions")
        entry = []
        for key in self.__menu_list.keys():
            entry.append(self.__menu_list[key][0])
        actions.add_actions(tuple(entry))
        ui_manager = gtk.UIManager()
        ui_manager.insert_action_group(actions, 0)
        ui_manager.add_ui_from_string(ui_info)
        self.__popup_menu = ui_manager.get_widget("/ui/popup")
        for key in self.__menu_list.keys():
            path = self.__menu_list[key][-1]
            self.__menu_list[key][1] = ui_manager.get_widget(path)

    def popup(self, button):
        skin_list = self.__kinoko.get_skin_list()
        self.__set_skin_menu(skin_list)
        self.__popup_menu.popup(None, None, None, button, gtk.get_current_event_time())

    def __set_skin_menu(self, list): ## FIXME
        key = 'skin'
        if list:
            menu = gtk.Menu()
            for skin in list:
                item = gtk.MenuItem(skin['title'])
                item.connect("activate", self.__kinoko.select_skin, (skin))
                menu.add(item)
                item.show()
            self.__menu_list[key][1].set_submenu(menu)
            menu.show()
            self.__menu_list[key][1].show()
        else:
            self.__menu_list[key][1].hide()

class Nayuki:

    def __init__(self, kinoko):
        self.kinoko = kinoko

class Kinoko:

    def __init__(self, skin_list):
        self.skin_list = skin_list

    def edit_preferences(self, action):
        pass

    def finalize(self):
        self.__running = 0
        self.target.delete_observer(self)
        self.skin.destroy()

    def notify_observer(self, event, args): ## FIXME
        if event in ['set position', 'set surface']:
            self.skin.set_position()
            self.skin.show()
        elif event == 'set scale':
            scale = self.target.get_surface_scale()
            self.skin.set_scale(scale)
        elif event == 'hide':
            if not self.target.surface_is_shown(0):
                self.skin.hide()
        elif event == 'iconified':
            self.skin.hide()
        elif event == 'deiconified':
            self.skin.show()
        elif event == 'finalize':
            self.finalize()
        elif event == 'move surface':
            side, xoffset, yoffset = args
            if side == 0:
                self.skin.set_position(xoffset, yoffset)
        elif event == 'raise':
            side = args
            if side == 0:
                self.skin.set_position() ## FIXME
        else:
            ##print 'OBSERVER(kinoko): ignore -', event
            pass

    def load(self, data, target):
        self.data = data
        self.target = target
        self.target.set_observer(self)
        self.accelgroup = gtk.AccelGroup()
        scale = self.target.get_surface_scale()
        self.skin = Skin(self.data, self.accelgroup, self, scale)
        if self.skin == None:
            return 0
        else:
            self.send_event('OnKinokoObjectCreate')
        self.__running = 1
        gobject.timeout_add(10, self.do_idle_tasks) # 10ms

    def do_idle_tasks(self):
        if not self.__running:
            return False
        self.skin.update()
        return True

    def close(self, action):
        self.finalize()
        self.send_event('OnKinokoObjectDestroy')

    def send_event(self, event):
        if event not in ['OnKinokoObjectCreate', 'OnKinokoObjectDestroy',
                         'OnKinokoObjectChanging', 'OnKinokoObjectChanged',
                         'OnKinokoObjectInstalled']:
                         ## 'OnBatteryLow', 'OnBatteryCritical',
                         ## 'OnSysResourceLow', 'OnSysResourceCritical'
            return
        args = (self.data['title'],
                self.data['ghost'],
                self.data['category'])
        sakura = self.target.get_sakura()
        sakura.notify_event(event, *args)

    def get_skin_list(self):
        return self.skin_list

    def select_skin(self, widget, args):
        self.send_event('OnKinokoObjectChanging')
        self.skin.destroy()
        self.data = args
        scale = self.target.get_surface_scale()
        self.skin = Skin(self.data, self.accelgroup, self, scale)
        if self.skin == None:
            return 0
        else:
            self.send_event('OnKinokoObjectChanged')

class Skin:

    def __init__(self, data, accelgroup, kinoko, scale):
        self.data = data
        self.accelgroup = accelgroup
        self.kinoko = kinoko
        if self.data['ontop']:
            self.__ontop = 1
        else:
            self.__ontop = 0
        self.__menu = Menu(self.kinoko, self.accelgroup)
        self.__scale = scale
        self.__shown = 0
        self.window = gtk.Window()
        ##self.window.set_title('surface.' + name)
        self.window.set_decorated(False)
        self.window.set_resizable(False)
        self.window.connect("delete_event", self.delete)
        self.window.connect("button_press_event", self.button_press)
        self.window.connect("button_release_event", self.button_release)
        self.window.connect("motion_notify_event", self.motion_notify)
        self.window.connect("leave_notify_event", self.leave_notify)
        self.window.set_events(gtk.gdk.BUTTON_PRESS_MASK|
                               gtk.gdk.BUTTON_RELEASE_MASK|
                               gtk.gdk.POINTER_MOTION_MASK|
                               gtk.gdk.LEAVE_NOTIFY_MASK)
        self.window.realize()
        self.window.add_accel_group(self.accelgroup)
        scrn_w = gtk.gdk.screen_width()
        scrn_h = gtk.gdk.screen_height()
        if self.data['animation'] is not None:
            path = os.path.join(self.data['dir'], self.data['animation'])
            self.seriko = ninix.seriko.get_actors(ninix.config.open(path))
        else:
            self.seriko = []
        path = os.path.join(self.data['dir'], self.data['base'])
        try:
            self.pixbuf = ninix.pix.create_pixbuf_from_file(path)
            surface_pixbuf = self.pixbuf.copy()
            if self.__scale != 100:
                w = surface_pixbuf.get_width() * self.__scale / 100
                h = surface_pixbuf.get_height() * self.__scale / 100
                surface_pixbuf = surface_pixbuf.scale_simple(w, h, gtk.gdk.INTERP_BILINEAR)
            image, mask = surface_pixbuf.render_pixmap_and_mask(255)
        except: ## FIXME
            self.kinoko.close(None)
            return None
        ## FIXME
        self.w, self.h = image.get_size()
        self.set_position()
        self.surface_pixmap = image
        self.mask_pixmap = mask
        self.window.shape_combine_mask(self.mask_pixmap, 0, 0)
        self.darea = gtk.DrawingArea()
        self.darea.set_events(gtk.gdk.EXPOSURE_MASK)
        self.darea.connect("expose_event", self.redraw)
        self.darea.set_size_request(self.w, self.h)
        self.darea.show()
        self.window.add(self.darea)
        self.darea.realize()
        self.darea.window.set_back_pixmap(image, False)
        self.show()
        self.reset_overlays()
        for actor in self.seriko:
            if actor.get_interval() == "runonce":
                actor.invoke()
        if self.data['ontop']:
            self.window.set_keep_above(True)
        else:
            self.window.set_keep_below(True)

    def show(self):
        if not self.__shown:
            self.window.show()
            self.__shown = 1

    def hide(self):
        if self.__shown:
            self.window.hide()
            self.__shown = 0

    def set_position(self, xoffset=0, yoffset=0): ## FIXME
        base_x, base_y = self.kinoko.target.get_kinoko_position(self.data['baseposition'])
        a, b = [(0.5, 1), (0.5, 0), (0, 0.5), (1, 0.5), (0, 1),
                (1, 1), (0, 0), (1, 0), (0.5, 0.5)][self.data['baseadjust']]
        offsetx = self.data['offsetx']
        offsety = self.data['offsety']
        if self.__scale != 100:
            offsetx = offsetx * self.__scale / 100
            offsety = offsety * self.__scale / 100
        self.x = base_x - int(self.w * a) + offsetx + xoffset
        self.y = base_y - int(self.h * b) + offsety + yoffset
        self.window.move(self.x, self.y)

    def set_scale(self, scale):
        self.__scale = scale
        self.draw_overlays() ## FIXME
        self.set_position()

    def get_surface(self): ## FIXME
        return None

    def redraw(self, widget, event): ## FIXME
        pass

    def show_surface(self):
        self.darea.window.set_back_pixmap(self.surface_pixmap, False)
        self.window.shape_combine_mask(self.mask_pixmap, 0, 0)
        #print 'DRAW:', dir(self.darea)
        #self.darea.queue_draw_area(0, 0, w, h)
        self.darea.queue_draw()

    def get_pixbuf(self, id):
        path = os.path.join(self.data['dir'], 'surface' + id + '.png')
        pixbuf = ninix.pix.create_pixbuf_from_file(path)
        return pixbuf

    def draw_overlays(self): ## FIXME
        surface_pixbuf = self.pixbuf.copy()
        actors = self.overlays.keys()
        actors.sort(lambda a1, a2: cmp(a1.get_id(), a2.get_id()))
        for actor in actors:
            id, x, y = self.overlays[actor]
            ##print "actor=%d, id=%s, x=%d, y=%d" % (actor.get_id(), id, x, y)
            try:
                pixbuf = self.get_pixbuf(id)
                w = pixbuf.get_width()
                h = pixbuf.get_height()
            except:
                print 'cannot load surface #%d' % id
                continue
            # overlay surface pixbuf
            sw = surface_pixbuf.get_width()
            sh = surface_pixbuf.get_height()
            if x + w > sw:
                w = sw - x
            if y + h > sh:
                h = sh - y
            if x < 0:
                dest_x = 0
                w += x
            else:
                dest_x = x
            if y < 0:
                dest_y = 0
                h += y
            else:
                dest_y = y
            pixbuf.composite(surface_pixbuf, dest_x, dest_y, w, h, x, y, 1.0, 1.0, gtk.gdk.INTERP_BILINEAR, 255)
        if self.__scale != 100:
            w = surface_pixbuf.get_width() * self.__scale / 100
            h = surface_pixbuf.get_height() * self.__scale / 100
            surface_pixbuf = surface_pixbuf.scale_simple(w, h, gtk.gdk.INTERP_BILINEAR)
        self.surface_pixmap, self.mask_pixmap = surface_pixbuf.render_pixmap_and_mask(255)
        self.w, self.h = self.surface_pixmap.get_size()
        self.show_surface()

    def reset_overlays(self):
        self.overlays = {}

    def terminate(self):
        for actor in self.seriko:
            actor.terminate()
        self.reset_overlays()

    def add_overlay(self, actor, id, x, y):
        if id == "-2":
            self.terminate()
        if id in ["-1", "-2"]:
            self.remove_overlay(actor)
            return
        self.overlays[actor] = (id, x, y)

    def remove_overlay(self, actor):
        try:
            del self.overlays[actor]
        except KeyError:
            pass

    def move_surface(self, xoffset, yoffset): ## FIXME
        self.window.move(self.x + xoffset, self.y + yoffset)

    def set_surface(self, id, restart=1): ## FIXME
        self.pixbuf = self.get_pixbuf(id)
        self.update()

    def invoke(self, id, update=0):
        for actor in self.seriko:
            if id == actor.get_id():
                actor.invoke()
                if update:
                    actor.update(self)

    def update(self):
        last_overlays = self.overlays.copy()
        for actor in self.seriko:
            if actor.wait == 1:
                self.reset_overlays()
                actor.update(self)
                self.draw_overlays()
                return
            else:
                actor.update(self)
        if self.overlays != last_overlays:
            self.draw_overlays()

    def delete(self, widget, event):
        self.kinoko.close(None)

    def destroy(self): ## FIXME
        self.window.destroy()

    def button_press(self, widget, event): ## FIXME
        x = int(event.x)
        y = int(event.y)
        self.x_root = event.x_root
        self.y_root = event.y_root
        if event.type == gtk.gdk.BUTTON_PRESS:
            click = 1
        else:
            click = 2
        button = event.button
        if button == 3 and click == 1:
            self.__menu.popup(button)
        return True

    def button_release(self,  widget, event): ## FIXME
        pass

    def motion_notify(self,  widget, event): ## FIXME
        pass

    def leave_notify(self,  widget, event): ## FIXME
        pass
