# -*- coding: ascii -*-
#
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2002-2004 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
# $Id: home.py,v 1.24.4.1 2005/05/27 07:38:05 shy Exp $
#

import os
import re
import string
import sys

import ninix.config
import ninix.alias
import ninix.dll

NINIX_HOME = "~/.ninix"
NINIX_USER = None

def print_error(message):
    sys.stderr.write(message + "\n")

def get_ninix_home():
    return os.path.expanduser(NINIX_HOME)

def get_ninix_user():
    if NINIX_USER is None:
        return get_ninix_home()
    return os.path.expanduser(NINIX_USER)

def get_gtkrc():
    path = os.path.join(get_ninix_user(), "gtkrc")
    if os.path.exists(path):
        return path
    return os.path.join(get_ninix_home(), "gtkrc")

def get_pango_fontrc():
    return os.path.join(get_ninix_user(), "pango_fontrc")

def get_preferences():
    return os.path.join(get_ninix_user(), "preferences")

def load_config():
    home_dir = get_ninix_home()
    if not os.path.exists(home_dir):
        return None
    ghosts, shells = search_ghosts(home_dir)
    balloons = search_balloons(home_dir)
    plugins = search_plugins(home_dir)
    nekoninni = search_nekoninni(home_dir)
    katochan = search_katochan(home_dir)
    kinoko = search_kinoko(home_dir)
    return ghosts, shells, balloons, plugins, nekoninni, katochan, kinoko

def get_shiori(path):
    table = {}
    shiori_lib = ninix.dll.Library('shiori', path, saori_lib=None)
    for file in os.listdir(path):
        if is_readable(os.path.join(path, file)):
            name = None
            if file[-3:] == '.py':
                name =  file[:-3]
            elif file[-4:] == '.pyc':
                name =  file[:-4]
            if name and not table.has_key(name):
                shiori = shiori_lib.request(('', name))
                if shiori:
                    table[name] = shiori
    return table

def search_ghosts(home_dir, target=None):
    ghosts = []
    shells = []
    if target:
        dirlist = []
        dirlist.extend(target)
    else:
        try:
            dirlist = os.listdir(os.path.join(home_dir, "ghost"))
        except OSError:
            dirlist = []
    shiori_table = get_shiori(os.path.join(os.environ['PYTHONPATH'], 'ninix/dll'))
    for subdir in dirlist:
        prefix = os.path.join(home_dir, "ghost", subdir)
        ghost_dir = os.path.join(prefix, "ghost", "master")
        desc = read_descript_txt(ghost_dir)
        if desc is None:
            desc = ninix.config.null_config()
        shiori_dll = desc.get('shiori')
        # find a pseudo AI, shells, and a built-in balloon
        candidate = {'name': '', 'score': 0}
        # SHIORI compatible modules
        for name in shiori_table.keys():
            score = int(shiori_table[name].find(ghost_dir, shiori_dll))
            if score > candidate['score']:
                candidate['name'] = name
                candidate['score'] = score
        shell_name, surface_set = find_surface_set(prefix)
        balloon = find_balloon(prefix)
        if candidate['score'] == 0:
            if surface_set:
                # use as a shell
                shells.append((shell_name, surface_set, balloon))
            continue
        shiori_name = candidate['name']
        if desc.get("name") == "default":
            pos = 0
        else:
            pos = len(ghosts)
        use_makoto = find_makoto_dll(ghost_dir)
        ghosts.insert(pos, (
            desc, ghost_dir, use_makoto, surface_set, balloon, prefix,
            shiori_dll, shiori_name))
    return ghosts, shells

def search_balloons(home_dir):
    balloon_dir = os.path.join(home_dir, "balloon")
    try:
        dirlist = os.listdir(balloon_dir)
    except OSError:
        dirlist = []
    buffer = []
    for subdir in dirlist:
        path = os.path.join(balloon_dir, subdir)
        if not os.path.isdir(path):
            continue
        desc = read_descript_txt(path) # REQUIRED
        if not desc:
            continue
        balloon_info = read_balloon_info(path) # REQUIRED
        if not balloon_info:
            continue
        if balloon_info.has_key('balloon_dir'): # XXX
            print 'Oops: key confliction'
            continue
        else:
            balloon_info['balloon_dir'] = subdir
        if desc.get("name") == "default":
            pos = 0
        else:
            pos = len(buffer)
        buffer.insert(pos, (desc, balloon_info))
    return buffer

def search_plugins(home_dir):
    buffer = []
    dir = os.path.join(home_dir, "plugin")
    try:
        dirlist = os.listdir(dir)
    except OSError:
        dirlist = []
    for subdir in dirlist:
        plugin = read_plugin_txt(os.path.join(dir, subdir))
        if plugin is None:
            continue
        buffer.append(plugin)
    return buffer

def search_nekoninni(home_dir): ## FIXME
    buffer = []
    dir = os.path.join(home_dir, "nekodorif/skin")
    try:
        dirlist = os.listdir(dir)
    except OSError:
        dirlist = []
    for subdir in dirlist:
        nekoninni = read_profile_txt(os.path.join(dir, subdir))
        if nekoninni is None:
            continue
        buffer.append(nekoninni)
    return buffer

def search_katochan(home_dir): ## FIXME
    buffer = []
    dir = os.path.join(home_dir, "nekodorif/katochan")
    try:
        dirlist = os.listdir(dir)
    except OSError:
        dirlist = []
    for subdir in dirlist:
        katochan = read_katochan_txt(os.path.join(dir, subdir))
        if katochan is None:
            continue
        buffer.append(katochan)
    return buffer

def search_kinoko(home_dir):
    buffer = []
    dir = os.path.join(home_dir, "kinoko")
    try:
        dirlist = os.listdir(dir)
    except OSError:
        dirlist = []
    for subdir in dirlist:
        kinoko = read_kinoko_ini(os.path.join(dir, subdir))
        if kinoko is None:
            continue
        buffer.append(kinoko)
    return buffer

def read_kinoko_ini(dir):
    path = os.path.join(dir, 'kinoko.ini')
    kinoko = {}
    kinoko['base'] = 'surface0.png'
    kinoko['animation'] = None
    kinoko['category'] = None
    kinoko['title'] = None
    kinoko['ghost'] = None
    kinoko['dir'] = dir
    kinoko['offsetx'] = 0
    kinoko['offsety'] = 0
    kinoko['ontop'] = 0
    kinoko['baseposition'] = 0
    kinoko['baseadjust'] = 0
    kinoko['extractpath'] = None
    kinoko['nayuki'] = None
    if is_readable(path):
        file = open(path)
        line = file.readline()
        if not line.strip() or line.strip() != '[KINOKO]':
            file.close()
            return None
        lineno = 0
        for line in file:
            lineno += 1
            if not line.strip():
                continue
            pos = line.find("=")
            if pos < 0:
                error = "line %d: syntax error" % lineno
                break
            name, value = line[:pos].strip(), line[pos+1:].strip()
            if name in ["title", "ghost", "category"]:
                kinoko[name] = unicode(value, 'Shift_JIS', 'ignore')
            elif name in ["offsetx", "offsety"]:
                kinoko[name] = int(value)
            elif name in ["base", "animation", "extractpath"]:
                kinoko[name] = value
            elif name in ["ontop", "baseposition", "baseadjust"]:
                kinoko[name] = int(value)
        file.close()
    if kinoko['title']:
        return kinoko
    else:
        return None

def read_profile_txt(dir): ## FIXME
    path = os.path.join(dir, 'profile.txt')
    name = None
    if is_readable(path):
        file = open(path)
        line = file.readline()
        if line:
            name = unicode(line.strip(), 'Shift_JIS', 'ignore')
        file.close()
    if name:
        return (name, dir) ## FIXME
    else:
        return None

def read_katochan_txt(dir): ## FIXME
    path = os.path.join(dir, 'katochan.txt')
    katochan = {}
    katochan['dir'] = dir
    if is_readable(path):
        file = open(path)
        name = None
        lineno = 0
        for line in file:
            lineno += 1
            if not line.strip():
                continue
            if line[0] == '#':
                name = line[1:].strip()
                continue
            elif not name:
                error = "line %d: syntax error" % lineno
                break
            else:
                value = line.strip()
                if name in ["name", "category", "before.script", "hit.script",
                            "after.script", "end.script", "dodge.script"]:
                    katochan[name] = unicode(value, 'Shift_JIS', 'ignore')
                elif name in ["before.fall.speed", "before.slide.magnitude",
                              "before.slide.sinwave.degspeed",
                              "before.appear.ofset.x",
                              "before.appear.ofset.y",
                              "hit.waittime", "hit.ofset.x", "hit.ofset.y",
                              "after.fall.speed", "after.slide.magnitude",
                              "after.slide.sinwave.degspeed"]:
                    katochan[name] = int(value)
                elif name in ["target",
                              "before.fall.type", "before.slide.type",
                              "before.wave", "before.wave.loop",
                              "before.appear.direction",
                              "hit.wave", "hit.wave.loop",
                              "after.fall.type", "after.slide.type",
                              "after.wave", "after.wave.loop",
                              "end.wave", "end.wave.loop",
                              "end.leave.direction",
                              "dodge.wave", "dodge.wave.loop"]:
                    katochan[name] = value
                name = None
        file.close()
    if katochan['name']:
        return katochan
    else:
        return None

def read_descript_txt(dir):
    path = os.path.join(dir, "descript.txt")
    if is_readable(path):
        return ninix.config.open(path)
    return None

def read_install_txt(dir):
    path = os.path.join(dir, "install.txt")
    if is_readable(path):
        return ninix.config.open(path)
    return None

def read_alias_txt(dir):
    path = os.path.join(dir, "alias.txt")
    if is_readable(path):
        return ninix.alias.open(path)
    return None

def find_makoto_dll(dir):
    if is_readable(os.path.join(dir, "makoto.dll")):
        return 1
    return 0

def find_surface_set(dir):
    desc = read_descript_txt(os.path.join(dir, "ghost", "master"))
    if desc:
        shell_name = desc.get("name")
    else:
        shell_name = None
    if not shell_name:
        inst = read_install_txt(dir)
        if inst:
            shell_name = inst.get("name")
    surface_set = []
    shell_dir = os.path.join(dir, "shell")
    for name, desc, subdir in find_surface_dir(shell_dir):
        surface_dir = os.path.join(shell_dir, subdir)
        surface_info, alias = read_surface_info(surface_dir)
        if surface_info and \
           surface_info.has_key("surface0") and \
           surface_info.has_key("surface10"):
            if alias is None:
                alias = read_alias_txt(surface_dir)
            surface_set.append((name, surface_dir, desc, alias, surface_info))
    return shell_name, surface_set

def find_surface_dir(dir):
    buffer = []
    path = os.path.join(dir, "surface.txt")
    if os.path.exists(path):
        config = ninix.config.open(path)
        for name, subdir in config.itemlist:
            desc = read_descript_txt(os.path.join(dir, subdir.lower()))
            if desc is None:
                desc = ninix.config.null_config()
            buffer.append((name, desc, subdir.lower()))
    else:
        try:
            dirlist = os.listdir(dir)
        except OSError:
            dirlist = []
        for subdir in dirlist:
            desc = read_descript_txt(os.path.join(dir, subdir))
            if desc is None:
                desc = ninix.config.null_config()
            name = desc.get("name", subdir)
            buffer.append((name, desc, subdir))
    return buffer

re_surface = re.compile("surface([0-9]+)\.(png|dgp)")

def read_surface_info(surface_dir):
    surface = {}
    try:
        filelist = os.listdir(surface_dir)
    except OSError:
        filelist = []
    filename_alias = {}
    path = os.path.join(surface_dir, "alias.txt")
    if os.path.exists(path):
        dict = ninix.alias.open(path)
        for basename, alias in dict.items():
            if basename[:7] == "surface":
                filename_alias[alias] = basename
    # find png image and associated configuration file
    for filename in filelist:
        basename, suffix = os.path.splitext(filename)
        if filename_alias.has_key(basename):
            match = re_surface.match(filename_alias[basename] + suffix)
        else:
            match = re_surface.match(filename)
        if not match:
            continue
        img = os.path.join(surface_dir, filename)
        if not is_readable(img):
            continue
        key = "surface" + str(int(match.group(1)))
        txt = os.path.join(surface_dir, basename + "s.txt")
        if is_readable(txt):
            config = ninix.config.open(txt)
        else:
            config = ninix.config.null_config()
        txt = os.path.join(surface_dir, basename + "a.txt")
        if is_readable(txt):
            config.update(ninix.config.open(txt))
        surface[key] = (img, config)
    # find surfaces.txt
    alias = None
    for key, config in read_surfaces_txt(surface_dir):
        if key == "__alias__":
            alias = config
        elif key[:7] == "surface":
            try:
                img, null_config = surface[key]
            except KeyError:
                img = None
            surface[key] = (img, config)
    # find surface elements
    for key in surface.keys():
        img, config = surface[key]
        for key, method, filename, x, y in list_surface_elements(config):
            filename = filename.lower()
            basename, suffix = os.path.splitext(filename)
            if not surface.has_key(basename):
                surface[basename] = (os.path.join(surface_dir, filename),
                                     ninix.config.null_config())
    return surface, alias

def read_surfaces_txt(surface_dir):
    list = []
    path = os.path.join(surface_dir, "surfaces.txt")
    try:
        file = open(path)
    except IOError:
        return list
    alias_buffer = []
    while 1:
        line = file.readline()
        if not line:
            break
        if line[0] == "#" or line[:2] == "//":
            continue
        key = line.strip()
        if not key:
            continue
        try:
            while 1:
                line = file.readline()
                if not line:
                    raise ValueError, "unexpected end of file"
                line = line.replace("\x81\x40", "").strip()
                if not line:
                    continue
                elif line == "{":
                    break
                key = line # ignore the preceding key
            buffer = []
            while 1:
                line = file.readline()
                if not line:
                    raise ValueError, "unexpected end of file"
                line = line.replace("\x81\x40", "").strip()
                if not line:
                    continue
                elif line == "}":
                    break
                buffer.append(line)
            if key in ["sakura.surface.alias", "kero.surface.alias"]:
                alias_buffer.append(key)
                alias_buffer.append("{")
                alias_buffer.extend(buffer)
                alias_buffer.append("}")
            elif key[:7] == "surface":
                try:
                    key = key[:7] + str(int(key[7:]))
                except ValueError:
                    pass
                list.append((key, ninix.config.new_config(buffer)))
        except ValueError, error:
            print_error("%s: %s (parsing not completed)" % (path, error))
            break
    if alias_buffer:
        list.append(("__alias__", ninix.alias.new_alias(alias_buffer)))
    return list

def list_surface_elements(config):
    buffer = []
    for n in range(256):
        key = "element" + str(n)
        if not config.has_key(key):
            break
        spec = [value.strip() for value in config[key].split(",")]
        try:
            method, filename, x, y = spec
            x = int(x)
            y = int(y)
        except ValueError:
            print_error("invalid element spec for %s: %s" % (key, config[key]))
            continue
        buffer.append((key, method, filename, x, y))
    return buffer

def find_balloon(dir):
    inst = read_install_txt(dir)
    if inst:
        balloon_dir = inst.get("balloon.directory")
        if balloon_dir:
            path = os.path.join(dir, "ghost", "master", balloon_dir.lower())
            desc = read_descript_txt(path)
            info = read_balloon_info(path)
            if desc and info:
                return (desc, info)
    return None

re_balloon = re.compile("balloon([skc][0-9]+)\.(png)")
re_annex   = re.compile("(arrow[01]|sstp)\.(png)")

def read_balloon_info(balloon_dir):
    balloon = {}
    try:
        filelist = os.listdir(balloon_dir)
    except OSError:
        filelist = []
    for filename in filelist:
        match = re_balloon.match(filename)
        if not match:
            continue
        img = os.path.join(balloon_dir, filename)
        if match.group(2) != "png" and \
           is_readable(img[-3:] + "png"):
                continue
        if not is_readable(img):
            continue
        key = match.group(1)
        txt = os.path.join(balloon_dir, "balloon%ss.txt" % key)
        if is_readable(txt):
            config = ninix.config.open(txt)
        else:
            config = ninix.config.null_config()
        balloon[key] = (img, config)
    for filename in filelist:
        match = re_annex.match(filename)
        if not match:
            continue
        img = os.path.join(balloon_dir, filename)
        if not is_readable(img):
            continue
        key = match.group(1)
        config = ninix.config.null_config()
        balloon[key] = (img, config)
    return balloon

def read_plugin_txt(plugin_dir):
    path = os.path.join(plugin_dir, "plugin.txt")
    try:
        file = open(path)
    except IOError:
        return None
    charset = 'EUC-JP' # default
    plugin_name = startup = None
    menu_items = []
    error = None
    lineno = 0
    for line in file:
        lineno += 1
        if not line.strip() or line[0] == "#":
            continue
        pos = line.find(":")
        if pos < 0:
            error = "line %d: syntax error" % lineno
            break
        name, value = line[:pos].strip(), line[pos+1:].strip()
        if name == "charset":
            charset = value
        elif name == "name":
            plugin_name = unicode(value, charset, 'ignore')
        elif name == "startup":
            list = value.split(",")
            list[0] = os.path.join(plugin_dir, list[0])
            if not os.path.exists(list[0]):
                error = "line %d: invalid program name" % lineno
                break
            startup = list
        elif name == "menuitem":
            list = unicode(value, charset, 'ignore').split(",")
            if len(list) < 2:
                error = "line %d: syntax error" % lineno
                break
            list[1] = os.path.join(plugin_dir, list[1])
            if not os.path.exists(list[1]):
                error = "line %d: invalid program name" % lineno
                break
            menu_items.append((list[0], list[1:]))
        else:
            error = "line %d: syntax error" % lineno
            break
    else:
        if plugin_name is None:
            error = 'the "name" header field is required'
        elif not startup and not menu_items:
            error = 'either "startup" or "menuitem" header field is required'
    file.close()
    if error:
        sys.stderr.write("Error: %s\n%s (skipped)\n" % (error, path))
        return None
    return plugin_name, plugin_dir, startup, menu_items

def is_readable(path):
    try:
        open(path).read(64)
    except IOError:
        return 0
    return 1
    
###   TEST   ###

def test():
    import locale
    locale.setlocale(locale.LC_ALL, '')
    global NINIX_HOME
    try:
        NINIX_HOME = os.environ["NINIX_HOME"]
    except KeyError:
        pass
    config = load_config()
    if config is None:
        sys.stderr.write("Home directory not found.\n")
        sys.exit(1)
    ghosts, shells, balloons, plugins, nekoninni, katochan, kinoko = config ## FIXME
    # ghosts
    for desc, shiori_dir, use_makoto, surface_set, balloon, prefix, shiori_dll, shiori_name in ghosts:
        print "GHOST", "=" * 50
        print prefix
        print str(desc).encode('utf-8', 'ignore')
        print shiori_dir
        print shiori_dll
        print shiori_name
        print "use_makoto =", use_makoto
        if surface_set:
            for name, dir, desc, alias, surface in surface_set:
                print "-" * 50
                print "surface:", name.encode('utf-8', 'ignore')
                print str(desc).encode('utf-8', 'ignore')
                for k, v in surface.items():
                    print k, "=", v[0]
                    print str(v[1]).encode('utf-8', 'ignore')
                if alias:
                    buffer = []
                    for k, v in alias.items():
                        if k in ["sakura.surface.alias", "kero.surface.alias"]:
                            print k + ":"
                            for id, list in v.items():
                                print id, "= [" + string.join(list, ", ") + "]"
                            print
                        else:
                            buffer.append((k, v))
                    if buffer:
                        print "filename alias:"
                        for k, v in buffer:
                            print k, "=", v
                        print
        if balloon:
            print "-" * 50
            desc, balloon = balloon
            print str(desc).encode('utf-8', 'ignore')
            for k, v in balloon.items():
                print k, "=", v[0]
                print str(v[1]).encode('utf-8', 'ignore')
    # shells
    for shell_name, surface_set, balloon in shells:
        print "SHELL", "=" * 50
        print shell_name.encode('utf-8', 'ignore')
        for name, dir, desc, alias, surface in surface_set:
            print "-" * 50
            print "surface:", name.encode('utf-8', 'ignore')
            print str(desc).encode('utf-8', 'ignore')
            for k, v in surface.items():
                print k, "=", v[0]
                print str(v[1]).encode('utf-8', 'ignore')
            if alias:
                buffer = []
                for k, v in alias.items():
                    if k in ["sakura.surface.alias", "kero.surface.alias"]:
                        print k + ":"
                        for id, list in v.items():
                            print id, "= [" + string.join(list, ", ") + "]"
                        print
                    else:
                        buffer.append((k, v))
                if buffer:
                    print "filename alias:"
                    for k, v in buffer:
                        print k, "=", v
                    print
        if balloon:
            print "-" * 50
            desc, balloon = balloon
            print str(desc).encode('utf-8', 'ignore')
            for k, v in balloon.items():
                print k, "=", v[0]
                print str(v[1]).encode('utf-8', 'ignore')
    # balloons
    for desc, balloon in balloons:
        print "BALLOON", "=" * 50
        print str(desc).encode('utf-8', 'ignore')
        for k, v in balloon.items():
            print k, "=", v[0]
            print str(v[1]).encode('utf-8', 'ignore')
    # plugins
    for plugin_name, plugin_dir, startup, menu_items in plugins:
        print "PLUGIN", "=" * 50
        print "name =", plugin_name.encode('utf-8', 'ignore')
        if startup:
            print "startup =", '["' + string.join(startup, '", "') + '"]'
        for label, argv in menu_items:
            print 'menuitem "%s" =' % label.encode('utf-8', 'ignore'),
            print '["' + string.join(argv, '", "') + '"]'

if __name__ == "__main__":
    test()
