/*
 * Copyright 2008 Sony Corporation
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *   * Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *   * Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *   * Neither the names of the copyright holders nor the names of their
 *     contributors may be used to endorse or promote products derived from this
 *     software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "e_cell.h"
#include "e_cell_err.h"

#include "mars_runtime.h"

#include <string.h>

spe_task_t *spe_task_create(spe_program_handle_t *prog)
	{
	spe_task_t *task;
	int ret;

	DPRINTF("\n");

	task = OPENSSL_malloc(sizeof(*task));
	if (!task)
		{
		CELLerr(CELL_F_SPE_TASK_RUN, CELL_R_MEMORY_ALLOCATION_FAILED);
		return NULL;
		}

	memset(task, 0, sizeof(*task));

	ret = mars_context_create(&task->mars, 0, 1 /* shared */);
	if (ret)
		{
		CELLerr(CELL_F_SPE_TASK_RUN, CELL_R_RUNTIME_SYSTEM_INITIALIZATION_FAILED);
		OPENSSL_free(task);
		return NULL;
		}

	ret = mars_task_create(task->mars, &task->task_id,
		"OpenSSL Task", prog->elf_image,
		MARS_TASK_CONTEXT_SAVE_SIZE_MAX);
	if (ret != MARS_SUCCESS)
		{
		CELLerr(CELL_F_SPE_TASK_RUN, CELL_R_TASK_INITIALIZATION_FAILED);
		mars_context_destroy(task->mars);
		OPENSSL_free(task);
		return NULL;
		}

	return task;
	}

int spe_task_destroy(spe_task_t *task)
	{
	DPRINTF("\n");

	mars_task_destroy(&task->task_id);
	mars_context_destroy(task->mars);
	OPENSSL_free(task);

	return 1;
	}

int spe_task_run(spe_task_t *task, uint64_t in_q, uint64_t out_q)
	{
	int ret;
	struct mars_task_args args;
	uint8_t priority;

	DPRINTF("\n");

	CRYPTO_r_lock(CRYPTO_LOCK_ENGINE);
	priority = cell_task_priority < 0 ? 0 : cell_task_priority;
	CRYPTO_r_unlock(CRYPTO_LOCK_ENGINE);

	args.type.u64[0] = in_q;
	args.type.u64[1] = out_q;

	ret = mars_task_schedule(&task->task_id, &args, priority);
	if (ret != MARS_SUCCESS)
		{
		CELLerr(CELL_F_SPE_TASK_RUN, CELL_R_TASK_SCHEDULE_FAILED);
		return 0;
		}

	return 1;
	}

int spe_task_wait(spe_task_t *task)
	{
	DPRINTF("\n");

	mars_task_wait(&task->task_id, NULL);

	return 1;
	}
