/*
 * Copyright (C) 2025 wolfSSL Inc.
 *
 * This file is part of wolfSSL.
 *
 * wolfSSL is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * wolfSSL is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1335, USA
 */

/*!
This module provides a Rust wrapper for the wolfCrypt library's Diffie-Hellman
(DH) functionality.

The primary component is the `DH` struct, which manages the lifecycle of a
wolfSSL `DhKey` object. It ensures proper initialization and deallocation.
*/

#![cfg(dh)]

use crate::sys;
use crate::wolfcrypt::random::RNG;
use std::mem::{MaybeUninit};

pub struct DH {
    wc_dhkey: sys::DhKey,
}

impl DH {
    /// ffdhe2048 named parameter group.
    pub const FFDHE_2048: i32 = sys::WC_FFDHE_2048 as i32;
    /// ffdhe3072 named parameter group.
    pub const FFDHE_3072: i32 = sys::WC_FFDHE_3072 as i32;
    /// ffdhe4096 named parameter group.
    pub const FFDHE_4096: i32 = sys::WC_FFDHE_4096 as i32;
    /// ffdhe6144 named parameter group.
    pub const FFDHE_6144: i32 = sys::WC_FFDHE_6144 as i32;
    /// ffdhe8192 named parameter group.
    pub const FFDHE_8192: i32 = sys::WC_FFDHE_8192 as i32;

    /// Perform quick validity check of public key value against prime.
    ///
    /// This checks that:
    /// * Public key is not 0 or 1.
    /// * Public key is not equal to prime or prime - 1.
    /// * Public key is not bigger than prime.
    ///
    /// # Parameters
    ///
    /// * `prime`: Buffer containing prime value.
    /// * `public`: Buffer containing public key.
    ///
    /// # Returns
    ///
    /// Returns either Ok(()) on success or Err(e) containing the wolfSSL
    /// library error code value.
    ///
    /// # Example
    ///
    /// ```rust
    /// use wolfssl::wolfcrypt::random::RNG;
    /// use wolfssl::wolfcrypt::dh::DH;
    /// let mut rng = RNG::new().expect("Failed to create RNG");
    /// let mut dh = DH::new_named(DH::FFDHE_2048).expect("Error with new_named()");
    /// let mut private = [0u8; 256];
    /// let mut private_size = 0u32;
    /// let mut public = [0u8; 256];
    /// let mut public_size = 0u32;
    /// dh.generate_key_pair(&mut rng, &mut private, &mut private_size, &mut public, &mut public_size).expect("Error with generate_key_pair()");
    /// let public = &public[0..(public_size as usize)];
    /// let mut p = [0u8; 256];
    /// let mut q = [0u8; 256];
    /// let mut g = [0u8; 256];
    /// let mut p_size = 0u32;
    /// let mut g_size = 0u32;
    /// let mut q_size = 0u32;
    /// dh.export_params_raw(&mut p, &mut p_size, &mut q, &mut q_size, &mut g, &mut g_size).expect("Error with export_params_raw()");
    /// let p = &p[0..(p_size as usize)];
    /// DH::check_pub_value(p, public).expect("Error with check_pub_value()");
    /// ```
    pub fn check_pub_value(prime: &[u8], public: &[u8]) -> Result<(), i32> {
        let prime_size = prime.len() as u32;
        let public_size = public.len() as u32;
        let rc = unsafe {
            sys::wc_DhCheckPubValue(prime.as_ptr(), prime_size,
                public.as_ptr(), public_size)
        };
        if rc != 0 {
            return Err(rc);
        }
        Ok(())
    }

    /// Compare given DH parameters to named parameter set.
    ///
    /// # Parameters
    ///
    /// * `name`: DH parameters name, one of DH::FFDHE_*.
    /// * `p`: DH `p` parameter value.
    /// * `g`: DH `g` parameter value.
    /// * `q`: DH `q` parameter value (optional).
    ///
    /// # Returns
    ///
    /// Returns whether the parameters match the named parameters.
    ///
    /// # Example
    ///
    /// ```rust
    /// use wolfssl::wolfcrypt::random::RNG;
    /// use wolfssl::wolfcrypt::dh::DH;
    /// let mut dh = DH::new_named(DH::FFDHE_2048).expect("Error with new_named()");
    /// let mut p = [0u8; 256];
    /// let mut q = [0u8; 256];
    /// let mut g = [0u8; 256];
    /// let mut p_size = 0u32;
    /// let mut q_size = 0u32;
    /// let mut g_size = 0u32;
    /// dh.export_params_raw(&mut p, &mut p_size, &mut q, &mut q_size, &mut g, &mut g_size).expect("Error with export_params_raw()");
    /// let p = &p[0..(p_size as usize)];
    /// let g = &g[0..(g_size as usize)];
    /// assert!(DH::compare_named_key(DH::FFDHE_2048, p, g, None));
    /// ```
    pub fn compare_named_key(name: i32, p: &[u8], g: &[u8], q: Option<&[u8]>) -> bool {
        let p_size = p.len() as u32;
        let g_size = g.len() as u32;
        let mut no_q = 1i32;
        let mut q_ptr: *const u8 = core::ptr::null();
        let mut q_size = 0u32;
        if let Some(q) = q {
            no_q = 0;
            q_ptr = q.as_ptr();
            q_size = q.len() as u32;
        }
        let rc = unsafe {
            sys::wc_DhCmpNamedKey(name, no_q,
                p.as_ptr(), p_size,
                g.as_ptr(), g_size,
                q_ptr, q_size)
        };
        rc != 0
    }

    /// Create a new DH context by generating parameters.
    ///
    /// # Parameters
    ///
    /// * `rng`: `RNG` struct instance to use for random number generation.
    /// * `modulus_size`: Modulus size in bits.
    ///
    /// # Returns
    ///
    /// Returns either Ok(dh) containing the DH struct instance or Err(e)
    /// containing the wolfSSL library error code value.
    ///
    /// # Example
    ///
    /// ```rust
    /// #[cfg(dh_keygen)]
    /// {
    /// use wolfssl::wolfcrypt::random::RNG;
    /// use wolfssl::wolfcrypt::dh::DH;
    /// let mut rng = RNG::new().expect("Error with RNG::new()");
    /// let mut dh = DH::generate(&mut rng, 2048).expect("Error with generate()");
    /// }
    /// ```
    #[cfg(dh_keygen)]
    pub fn generate(rng: &mut RNG, modulus_size: i32) -> Result<Self, i32> {
        Self::generate_ex(rng, modulus_size, None, None)
    }

    /// Create a new DH context by generating parameters with optional heap and
    /// device ID.
    ///
    /// # Parameters
    ///
    /// * `rng`: `RNG` struct instance to use for random number generation.
    /// * `modulus_size`: Modulus size in bits.
    /// * `heap`: Optional heap hint.
    /// * `dev_id` Optional device ID to use with crypto callbacks or async hardware.
    ///
    /// # Returns
    ///
    /// Returns either Ok(dh) containing the DH struct instance or Err(e)
    /// containing the wolfSSL library error code value.
    ///
    /// # Example
    ///
    /// ```rust
    /// #[cfg(dh_keygen)]
    /// {
    /// use wolfssl::wolfcrypt::random::RNG;
    /// use wolfssl::wolfcrypt::dh::DH;
    /// let mut rng = RNG::new().expect("Error with RNG::new()");
    /// let mut dh = DH::generate_ex(&mut rng, 2048, None, None).expect("Error with generate_ex()");
    /// }
    /// ```
    #[cfg(dh_keygen)]
    pub fn generate_ex(rng: &mut RNG, modulus_size: i32, heap: Option<*mut std::os::raw::c_void>, dev_id: Option<i32>) -> Result<Self, i32> {
        let mut wc_dhkey: MaybeUninit<sys::DhKey> = MaybeUninit::uninit();
        let heap = match heap {
            Some(heap) => heap,
            None => core::ptr::null_mut(),
        };
        let dev_id = match dev_id {
            Some(dev_id) => dev_id,
            None => sys::INVALID_DEVID,
        };
        let rc = unsafe { sys::wc_InitDhKey_ex(wc_dhkey.as_mut_ptr(), heap, dev_id) };
        if rc != 0 {
            return Err(rc);
        }
        let wc_dhkey = unsafe { wc_dhkey.assume_init() };
        let mut dh = DH { wc_dhkey };
        let rc = unsafe {
            sys::wc_DhGenerateParams(&mut rng.wc_rng, modulus_size, &mut dh.wc_dhkey)
        };
        if rc != 0 {
            return Err(rc);
        }
        Ok(dh)
    }

    /// Get minimum key size for DH named parameter set.
    ///
    /// # Parameters
    ///
    /// * `name`: DH parameters name, one of DH::FFDHE_*.
    ///
    /// # Returns
    ///
    /// Minimum key size for the DH named parameter set.
    ///
    /// # Example
    ///
    /// ```rust
    /// use wolfssl::wolfcrypt::random::RNG;
    /// use wolfssl::wolfcrypt::dh::DH;
    /// let min_key_size = DH::get_min_key_size_for_named_parameters(DH::FFDHE_2048);
    /// assert_eq!(min_key_size, 29);
    /// ```
    pub fn get_min_key_size_for_named_parameters(name: i32) -> u32 {
        unsafe { sys::wc_DhGetNamedKeyMinSize(name) }
    }

    /// Get parameter sizes for a named parameter set.
    ///
    /// # Parameters
    ///
    /// * `name`: DH parameters name, one of DH::FFDHE_*.
    /// * `p_size`: Output parameter containing size of DH `p` parameter.
    /// * `g_size`: Output parameter containing size of DH `g` parameter.
    /// * `q_size`: Output parameter containing size of DH `q` parameter.
    ///
    /// # Example
    ///
    /// ```rust
    /// use wolfssl::wolfcrypt::random::RNG;
    /// use wolfssl::wolfcrypt::dh::DH;
    /// let mut p_size = 0u32;
    /// let mut g_size = 0u32;
    /// let mut q_size = 0u32;
    /// DH::get_named_parameter_sizes(DH::FFDHE_2048, &mut p_size, &mut g_size, &mut q_size);
    /// ```
    pub fn get_named_parameter_sizes(name: i32, p_size: &mut u32, g_size: &mut u32, q_size: &mut u32) {
        unsafe {
            sys::wc_DhGetNamedKeyParamSize(name, p_size, g_size, q_size)
        };
    }

    /// Create a new DH context using the named parameter set.
    ///
    /// # Parameters
    ///
    /// * `name`: DH parameters name, one of DH::FFDHE_*.
    ///
    /// # Returns
    ///
    /// Returns either Ok(dh) containing the DH struct instance or Err(e)
    /// containing the wolfSSL library error code value.
    ///
    /// # Example
    ///
    /// ```rust
    /// use wolfssl::wolfcrypt::random::RNG;
    /// use wolfssl::wolfcrypt::dh::DH;
    /// let mut dh = DH::new_named(DH::FFDHE_2048).expect("Error with new_named()");
    /// ```
    pub fn new_named(name: i32) -> Result<Self, i32> {
        Self::new_named_ex(name, None, None)
    }

    /// Create a new DH context using the named parameter set with optional
    /// heap and device ID.
    ///
    /// # Parameters
    ///
    /// * `name`: DH parameters name, one of DH::FFDHE_*.
    /// * `heap`: Optional heap hint.
    /// * `dev_id` Optional device ID to use with crypto callbacks or async hardware.
    ///
    /// # Returns
    ///
    /// Returns either Ok(dh) containing the DH struct instance or Err(e)
    /// containing the wolfSSL library error code value.
    ///
    /// # Example
    ///
    /// ```rust
    /// use wolfssl::wolfcrypt::random::RNG;
    /// use wolfssl::wolfcrypt::dh::DH;
    /// let mut dh = DH::new_named_ex(DH::FFDHE_2048, None, None).expect("Error with new_named_ex()");
    /// ```
    pub fn new_named_ex(name: i32, heap: Option<*mut std::os::raw::c_void>, dev_id: Option<i32>) -> Result<Self, i32> {
        let mut wc_dhkey: MaybeUninit<sys::DhKey> = MaybeUninit::uninit();
        let heap = match heap {
            Some(heap) => heap,
            None => core::ptr::null_mut(),
        };
        let dev_id = match dev_id {
            Some(dev_id) => dev_id,
            None => sys::INVALID_DEVID,
        };
        let rc = unsafe { sys::wc_InitDhKey_ex(wc_dhkey.as_mut_ptr(), heap, dev_id) };
        if rc != 0 {
            return Err(rc);
        }
        let wc_dhkey = unsafe { wc_dhkey.assume_init() };
        let mut dh = DH { wc_dhkey };
        let rc = unsafe { sys::wc_DhSetNamedKey(&mut dh.wc_dhkey, name) };
        if rc != 0 {
            return Err(rc);
        }
        Ok(dh)
    }

    /// Create a new DH context using the given p and g parameters.
    ///
    /// # Parameters
    ///
    /// * `p`: DH 'p' parameter value.
    /// * `g`: DH 'g' parameter value.
    ///
    /// # Returns
    ///
    /// Returns either Ok(dh) containing the DH struct instance or Err(e)
    /// containing the wolfSSL library error code value.
    ///
    /// # Example
    ///
    /// ```rust
    /// use wolfssl::wolfcrypt::random::RNG;
    /// use wolfssl::wolfcrypt::dh::DH;
    /// let p = [
    ///     0xc5u8, 0x7c, 0xa2, 0x4f, 0x4b, 0xd6, 0x8c, 0x3c,
    ///     0xda, 0xc7, 0xba, 0xaa, 0xea, 0x2e, 0x5c, 0x1e,
    ///     0x18, 0xb2, 0x7b, 0x8c, 0x55, 0x65, 0x9f, 0xea,
    ///     0xe0, 0xa1, 0x36, 0x53, 0x2b, 0x36, 0xe0, 0x4e,
    ///     0x3e, 0x64, 0xa9, 0xe4, 0xfc, 0x8f, 0x32, 0x62,
    ///     0x97, 0xe4, 0xbe, 0xf7, 0xc1, 0xde, 0x07, 0x5a,
    ///     0x89, 0x28, 0xf3, 0xfe, 0x4f, 0xfe, 0x68, 0xbc,
    ///     0xfb, 0x0a, 0x7c, 0xa4, 0xb3, 0x14, 0x48, 0x89,
    ///     0x9f, 0xaf, 0xb8, 0x43, 0xe2, 0xa0, 0x62, 0x5c,
    ///     0xb4, 0x88, 0x3f, 0x06, 0x50, 0x11, 0xfe, 0x65,
    ///     0x8d, 0x49, 0xd2, 0xf5, 0x4b, 0x74, 0x79, 0xdb,
    ///     0x06, 0x62, 0x92, 0x89, 0xed, 0xda, 0xcb, 0x87,
    ///     0x37, 0x16, 0xd2, 0xa1, 0x7a, 0xe8, 0xde, 0x92,
    ///     0xee, 0x3e, 0x41, 0x4a, 0x91, 0x5e, 0xed, 0xf3,
    ///     0x6c, 0x6b, 0x7e, 0xfd, 0x15, 0x92, 0x18, 0xfc,
    ///     0xa7, 0xac, 0x42, 0x85, 0x57, 0xe9, 0xdc, 0xda,
    ///     0x55, 0xc9, 0x8b, 0x28, 0x9e, 0xc1, 0xc4, 0x46,
    ///     0x4d, 0x88, 0xed, 0x62, 0x8e, 0xdb, 0x3f, 0xb9,
    ///     0xd7, 0xc8, 0xe3, 0xcf, 0xb8, 0x34, 0x2c, 0xd2,
    ///     0x6f, 0x28, 0x06, 0x41, 0xe3, 0x66, 0x8c, 0xfc,
    ///     0x72, 0xff, 0x26, 0x3b, 0x6b, 0x6c, 0x6f, 0x73,
    ///     0xde, 0xf2, 0x90, 0x29, 0xe0, 0x61, 0x32, 0xc4,
    ///     0x12, 0x74, 0x09, 0x52, 0xec, 0xf3, 0x1b, 0xa6,
    ///     0x45, 0x98, 0xac, 0xf9, 0x1c, 0x65, 0x8e, 0x3a,
    ///     0x91, 0x84, 0x4b, 0x23, 0x8a, 0xb2, 0x3c, 0xc9,
    ///     0xfa, 0xea, 0xf1, 0x38, 0xce, 0xd8, 0x05, 0xe0,
    ///     0xfa, 0x44, 0x68, 0x1f, 0xeb, 0xd9, 0x57, 0xb8,
    ///     0x4a, 0x97, 0x5b, 0x88, 0xc5, 0xf1, 0xbb, 0xb0,
    ///     0x49, 0xc3, 0x91, 0x7c, 0xd3, 0x13, 0xb9, 0x47,
    ///     0xbb, 0x91, 0x8f, 0xe5, 0x26, 0x07, 0xab, 0xa9,
    ///     0xc5, 0xd0, 0x3d, 0x95, 0x41, 0x26, 0x92, 0x9d,
    ///     0x13, 0x67, 0xf2, 0x7e, 0x11, 0x88, 0xdc, 0x2d
    /// ];
    /// let g = [
    ///     0x4au8, 0x1a, 0xf3, 0xa4, 0x92, 0xe9, 0xee, 0x74,
    ///     0x6e, 0x57, 0xd5, 0x8c, 0x2c, 0x5b, 0x41, 0x41,
    ///     0x5e, 0xd4, 0x55, 0x19, 0xdc, 0xd9, 0x32, 0x91,
    ///     0xf7, 0xfd, 0xc2, 0x57, 0xff, 0x03, 0x14, 0xdb,
    ///     0xf1, 0xb7, 0x60, 0x0c, 0x43, 0x59, 0x3f, 0xff,
    ///     0xac, 0xf1, 0x80, 0x9a, 0x15, 0x6f, 0xd8, 0x6e,
    ///     0xb7, 0x85, 0x18, 0xc8, 0xec, 0x4e, 0x59, 0x4a,
    ///     0xe2, 0x91, 0x43, 0x4c, 0xeb, 0x95, 0xb6, 0x2e,
    ///     0x9a, 0xea, 0x53, 0x68, 0x80, 0x64, 0x69, 0x40,
    ///     0xf9, 0xec, 0xbd, 0x85, 0x89, 0x26, 0x97, 0x67,
    ///     0xaf, 0xb0, 0xad, 0x00, 0x1b, 0xd4, 0xfd, 0x94,
    ///     0xd3, 0xe9, 0x92, 0xb1, 0xb4, 0xbc, 0x5a, 0xaa,
    ///     0x92, 0x80, 0x89, 0x3b, 0x39, 0x05, 0x6c, 0x22,
    ///     0x26, 0xfe, 0x5a, 0x28, 0x6c, 0x37, 0x50, 0x5a,
    ///     0x38, 0x99, 0xcf, 0xf3, 0xc1, 0x96, 0x45, 0xdc,
    ///     0x01, 0xcb, 0x20, 0x87, 0xa5, 0x00, 0x8c, 0xf5,
    ///     0x4d, 0xc2, 0xef, 0xb8, 0x9b, 0xd1, 0x87, 0xbe,
    ///     0xed, 0xd5, 0x0a, 0x29, 0x15, 0x34, 0x59, 0x4c,
    ///     0x3a, 0x05, 0x22, 0x05, 0x44, 0x4f, 0x9f, 0xc8,
    ///     0x47, 0x12, 0x24, 0x8e, 0xa8, 0x79, 0xe4, 0x67,
    ///     0xba, 0x4d, 0x5b, 0x75, 0x56, 0x95, 0xeb, 0xe8,
    ///     0x8a, 0xfa, 0x8e, 0x01, 0x8c, 0x1b, 0x74, 0x63,
    ///     0xd9, 0x2f, 0xf7, 0xd3, 0x44, 0x8f, 0xa8, 0xf5,
    ///     0xaf, 0x6c, 0x4f, 0xdb, 0xe7, 0xc9, 0x6c, 0x71,
    ///     0x22, 0xa3, 0x1d, 0xf1, 0x40, 0xb2, 0xe0, 0x9a,
    ///     0xb6, 0x72, 0xc9, 0xc0, 0x13, 0x16, 0xa2, 0x4a,
    ///     0xe1, 0x92, 0xc7, 0x54, 0x23, 0xab, 0x9d, 0xa1,
    ///     0xa1, 0xe5, 0x0b, 0xed, 0xba, 0xe8, 0x84, 0x37,
    ///     0xb2, 0xe7, 0xfe, 0x32, 0x8d, 0xfa, 0x1c, 0x53,
    ///     0x77, 0x97, 0xc7, 0xf3, 0x48, 0xc9, 0xdb, 0x2d,
    ///     0x75, 0x52, 0x9d, 0x42, 0x51, 0x78, 0x62, 0x68,
    ///     0x05, 0x45, 0x15, 0xf8, 0xa2, 0x4e, 0xf3, 0x0b
    /// ];
    /// let dh = DH::new_from_pg(&p, &g).expect("Error with new_from_pg()");
    /// ```
    pub fn new_from_pg(p: &[u8], g: &[u8]) -> Result<Self, i32> {
        Self::new_from_pg_ex(p, g, None, None)
    }

    /// Create a new DH context using the given p and g parameters with
    /// optional heap and device ID.
    ///
    /// # Parameters
    ///
    /// * `p`: DH 'p' parameter value.
    /// * `g`: DH 'g' parameter value.
    /// * `heap`: Optional heap hint.
    /// * `dev_id` Optional device ID to use with crypto callbacks or async hardware.
    ///
    /// # Returns
    ///
    /// Returns either Ok(dh) containing the DH struct instance or Err(e)
    /// containing the wolfSSL library error code value.
    ///
    /// # Example
    ///
    /// ```rust
    /// use wolfssl::wolfcrypt::random::RNG;
    /// use wolfssl::wolfcrypt::dh::DH;
    /// let p = [
    ///     0xc5u8, 0x7c, 0xa2, 0x4f, 0x4b, 0xd6, 0x8c, 0x3c,
    ///     0xda, 0xc7, 0xba, 0xaa, 0xea, 0x2e, 0x5c, 0x1e,
    ///     0x18, 0xb2, 0x7b, 0x8c, 0x55, 0x65, 0x9f, 0xea,
    ///     0xe0, 0xa1, 0x36, 0x53, 0x2b, 0x36, 0xe0, 0x4e,
    ///     0x3e, 0x64, 0xa9, 0xe4, 0xfc, 0x8f, 0x32, 0x62,
    ///     0x97, 0xe4, 0xbe, 0xf7, 0xc1, 0xde, 0x07, 0x5a,
    ///     0x89, 0x28, 0xf3, 0xfe, 0x4f, 0xfe, 0x68, 0xbc,
    ///     0xfb, 0x0a, 0x7c, 0xa4, 0xb3, 0x14, 0x48, 0x89,
    ///     0x9f, 0xaf, 0xb8, 0x43, 0xe2, 0xa0, 0x62, 0x5c,
    ///     0xb4, 0x88, 0x3f, 0x06, 0x50, 0x11, 0xfe, 0x65,
    ///     0x8d, 0x49, 0xd2, 0xf5, 0x4b, 0x74, 0x79, 0xdb,
    ///     0x06, 0x62, 0x92, 0x89, 0xed, 0xda, 0xcb, 0x87,
    ///     0x37, 0x16, 0xd2, 0xa1, 0x7a, 0xe8, 0xde, 0x92,
    ///     0xee, 0x3e, 0x41, 0x4a, 0x91, 0x5e, 0xed, 0xf3,
    ///     0x6c, 0x6b, 0x7e, 0xfd, 0x15, 0x92, 0x18, 0xfc,
    ///     0xa7, 0xac, 0x42, 0x85, 0x57, 0xe9, 0xdc, 0xda,
    ///     0x55, 0xc9, 0x8b, 0x28, 0x9e, 0xc1, 0xc4, 0x46,
    ///     0x4d, 0x88, 0xed, 0x62, 0x8e, 0xdb, 0x3f, 0xb9,
    ///     0xd7, 0xc8, 0xe3, 0xcf, 0xb8, 0x34, 0x2c, 0xd2,
    ///     0x6f, 0x28, 0x06, 0x41, 0xe3, 0x66, 0x8c, 0xfc,
    ///     0x72, 0xff, 0x26, 0x3b, 0x6b, 0x6c, 0x6f, 0x73,
    ///     0xde, 0xf2, 0x90, 0x29, 0xe0, 0x61, 0x32, 0xc4,
    ///     0x12, 0x74, 0x09, 0x52, 0xec, 0xf3, 0x1b, 0xa6,
    ///     0x45, 0x98, 0xac, 0xf9, 0x1c, 0x65, 0x8e, 0x3a,
    ///     0x91, 0x84, 0x4b, 0x23, 0x8a, 0xb2, 0x3c, 0xc9,
    ///     0xfa, 0xea, 0xf1, 0x38, 0xce, 0xd8, 0x05, 0xe0,
    ///     0xfa, 0x44, 0x68, 0x1f, 0xeb, 0xd9, 0x57, 0xb8,
    ///     0x4a, 0x97, 0x5b, 0x88, 0xc5, 0xf1, 0xbb, 0xb0,
    ///     0x49, 0xc3, 0x91, 0x7c, 0xd3, 0x13, 0xb9, 0x47,
    ///     0xbb, 0x91, 0x8f, 0xe5, 0x26, 0x07, 0xab, 0xa9,
    ///     0xc5, 0xd0, 0x3d, 0x95, 0x41, 0x26, 0x92, 0x9d,
    ///     0x13, 0x67, 0xf2, 0x7e, 0x11, 0x88, 0xdc, 0x2d
    /// ];
    /// let g = [
    ///     0x4au8, 0x1a, 0xf3, 0xa4, 0x92, 0xe9, 0xee, 0x74,
    ///     0x6e, 0x57, 0xd5, 0x8c, 0x2c, 0x5b, 0x41, 0x41,
    ///     0x5e, 0xd4, 0x55, 0x19, 0xdc, 0xd9, 0x32, 0x91,
    ///     0xf7, 0xfd, 0xc2, 0x57, 0xff, 0x03, 0x14, 0xdb,
    ///     0xf1, 0xb7, 0x60, 0x0c, 0x43, 0x59, 0x3f, 0xff,
    ///     0xac, 0xf1, 0x80, 0x9a, 0x15, 0x6f, 0xd8, 0x6e,
    ///     0xb7, 0x85, 0x18, 0xc8, 0xec, 0x4e, 0x59, 0x4a,
    ///     0xe2, 0x91, 0x43, 0x4c, 0xeb, 0x95, 0xb6, 0x2e,
    ///     0x9a, 0xea, 0x53, 0x68, 0x80, 0x64, 0x69, 0x40,
    ///     0xf9, 0xec, 0xbd, 0x85, 0x89, 0x26, 0x97, 0x67,
    ///     0xaf, 0xb0, 0xad, 0x00, 0x1b, 0xd4, 0xfd, 0x94,
    ///     0xd3, 0xe9, 0x92, 0xb1, 0xb4, 0xbc, 0x5a, 0xaa,
    ///     0x92, 0x80, 0x89, 0x3b, 0x39, 0x05, 0x6c, 0x22,
    ///     0x26, 0xfe, 0x5a, 0x28, 0x6c, 0x37, 0x50, 0x5a,
    ///     0x38, 0x99, 0xcf, 0xf3, 0xc1, 0x96, 0x45, 0xdc,
    ///     0x01, 0xcb, 0x20, 0x87, 0xa5, 0x00, 0x8c, 0xf5,
    ///     0x4d, 0xc2, 0xef, 0xb8, 0x9b, 0xd1, 0x87, 0xbe,
    ///     0xed, 0xd5, 0x0a, 0x29, 0x15, 0x34, 0x59, 0x4c,
    ///     0x3a, 0x05, 0x22, 0x05, 0x44, 0x4f, 0x9f, 0xc8,
    ///     0x47, 0x12, 0x24, 0x8e, 0xa8, 0x79, 0xe4, 0x67,
    ///     0xba, 0x4d, 0x5b, 0x75, 0x56, 0x95, 0xeb, 0xe8,
    ///     0x8a, 0xfa, 0x8e, 0x01, 0x8c, 0x1b, 0x74, 0x63,
    ///     0xd9, 0x2f, 0xf7, 0xd3, 0x44, 0x8f, 0xa8, 0xf5,
    ///     0xaf, 0x6c, 0x4f, 0xdb, 0xe7, 0xc9, 0x6c, 0x71,
    ///     0x22, 0xa3, 0x1d, 0xf1, 0x40, 0xb2, 0xe0, 0x9a,
    ///     0xb6, 0x72, 0xc9, 0xc0, 0x13, 0x16, 0xa2, 0x4a,
    ///     0xe1, 0x92, 0xc7, 0x54, 0x23, 0xab, 0x9d, 0xa1,
    ///     0xa1, 0xe5, 0x0b, 0xed, 0xba, 0xe8, 0x84, 0x37,
    ///     0xb2, 0xe7, 0xfe, 0x32, 0x8d, 0xfa, 0x1c, 0x53,
    ///     0x77, 0x97, 0xc7, 0xf3, 0x48, 0xc9, 0xdb, 0x2d,
    ///     0x75, 0x52, 0x9d, 0x42, 0x51, 0x78, 0x62, 0x68,
    ///     0x05, 0x45, 0x15, 0xf8, 0xa2, 0x4e, 0xf3, 0x0b
    /// ];
    /// let dh = DH::new_from_pg_ex(&p, &g, None, None).expect("Error with new_from_pg_ex()");
    /// ```
    pub fn new_from_pg_ex(p: &[u8], g: &[u8], heap: Option<*mut std::os::raw::c_void>, dev_id: Option<i32>) -> Result<Self, i32> {
        let p_size = p.len() as u32;
        let g_size = g.len() as u32;
        let mut wc_dhkey: MaybeUninit<sys::DhKey> = MaybeUninit::uninit();
        let heap = match heap {
            Some(heap) => heap,
            None => core::ptr::null_mut(),
        };
        let dev_id = match dev_id {
            Some(dev_id) => dev_id,
            None => sys::INVALID_DEVID,
        };
        let rc = unsafe { sys::wc_InitDhKey_ex(wc_dhkey.as_mut_ptr(), heap, dev_id) };
        if rc != 0 {
            return Err(rc);
        }
        let wc_dhkey = unsafe { wc_dhkey.assume_init() };
        let mut dh = DH { wc_dhkey };
        let rc = unsafe {
            sys::wc_DhSetKey(&mut dh.wc_dhkey, p.as_ptr(), p_size, g.as_ptr(), g_size)
        };
        if rc != 0 {
            return Err(rc);
        }
        Ok(dh)
    }

    /// Create a new DH context using the given p, g, and q parameters.
    ///
    /// # Parameters
    ///
    /// * `p`: DH 'p' parameter value.
    /// * `g`: DH 'g' parameter value.
    /// * `q`: DH 'q' parameter value.
    ///
    /// # Returns
    ///
    /// Returns either Ok(dh) containing the DH struct instance or Err(e)
    /// containing the wolfSSL library error code value.
    ///
    /// # Example
    ///
    /// ```rust
    /// use wolfssl::wolfcrypt::random::RNG;
    /// use wolfssl::wolfcrypt::dh::DH;
    /// let p = [
    ///     0xc5u8, 0x7c, 0xa2, 0x4f, 0x4b, 0xd6, 0x8c, 0x3c,
    ///     0xda, 0xc7, 0xba, 0xaa, 0xea, 0x2e, 0x5c, 0x1e,
    ///     0x18, 0xb2, 0x7b, 0x8c, 0x55, 0x65, 0x9f, 0xea,
    ///     0xe0, 0xa1, 0x36, 0x53, 0x2b, 0x36, 0xe0, 0x4e,
    ///     0x3e, 0x64, 0xa9, 0xe4, 0xfc, 0x8f, 0x32, 0x62,
    ///     0x97, 0xe4, 0xbe, 0xf7, 0xc1, 0xde, 0x07, 0x5a,
    ///     0x89, 0x28, 0xf3, 0xfe, 0x4f, 0xfe, 0x68, 0xbc,
    ///     0xfb, 0x0a, 0x7c, 0xa4, 0xb3, 0x14, 0x48, 0x89,
    ///     0x9f, 0xaf, 0xb8, 0x43, 0xe2, 0xa0, 0x62, 0x5c,
    ///     0xb4, 0x88, 0x3f, 0x06, 0x50, 0x11, 0xfe, 0x65,
    ///     0x8d, 0x49, 0xd2, 0xf5, 0x4b, 0x74, 0x79, 0xdb,
    ///     0x06, 0x62, 0x92, 0x89, 0xed, 0xda, 0xcb, 0x87,
    ///     0x37, 0x16, 0xd2, 0xa1, 0x7a, 0xe8, 0xde, 0x92,
    ///     0xee, 0x3e, 0x41, 0x4a, 0x91, 0x5e, 0xed, 0xf3,
    ///     0x6c, 0x6b, 0x7e, 0xfd, 0x15, 0x92, 0x18, 0xfc,
    ///     0xa7, 0xac, 0x42, 0x85, 0x57, 0xe9, 0xdc, 0xda,
    ///     0x55, 0xc9, 0x8b, 0x28, 0x9e, 0xc1, 0xc4, 0x46,
    ///     0x4d, 0x88, 0xed, 0x62, 0x8e, 0xdb, 0x3f, 0xb9,
    ///     0xd7, 0xc8, 0xe3, 0xcf, 0xb8, 0x34, 0x2c, 0xd2,
    ///     0x6f, 0x28, 0x06, 0x41, 0xe3, 0x66, 0x8c, 0xfc,
    ///     0x72, 0xff, 0x26, 0x3b, 0x6b, 0x6c, 0x6f, 0x73,
    ///     0xde, 0xf2, 0x90, 0x29, 0xe0, 0x61, 0x32, 0xc4,
    ///     0x12, 0x74, 0x09, 0x52, 0xec, 0xf3, 0x1b, 0xa6,
    ///     0x45, 0x98, 0xac, 0xf9, 0x1c, 0x65, 0x8e, 0x3a,
    ///     0x91, 0x84, 0x4b, 0x23, 0x8a, 0xb2, 0x3c, 0xc9,
    ///     0xfa, 0xea, 0xf1, 0x38, 0xce, 0xd8, 0x05, 0xe0,
    ///     0xfa, 0x44, 0x68, 0x1f, 0xeb, 0xd9, 0x57, 0xb8,
    ///     0x4a, 0x97, 0x5b, 0x88, 0xc5, 0xf1, 0xbb, 0xb0,
    ///     0x49, 0xc3, 0x91, 0x7c, 0xd3, 0x13, 0xb9, 0x47,
    ///     0xbb, 0x91, 0x8f, 0xe5, 0x26, 0x07, 0xab, 0xa9,
    ///     0xc5, 0xd0, 0x3d, 0x95, 0x41, 0x26, 0x92, 0x9d,
    ///     0x13, 0x67, 0xf2, 0x7e, 0x11, 0x88, 0xdc, 0x2d
    /// ];
    /// let g = [
    ///     0x4au8, 0x1a, 0xf3, 0xa4, 0x92, 0xe9, 0xee, 0x74,
    ///     0x6e, 0x57, 0xd5, 0x8c, 0x2c, 0x5b, 0x41, 0x41,
    ///     0x5e, 0xd4, 0x55, 0x19, 0xdc, 0xd9, 0x32, 0x91,
    ///     0xf7, 0xfd, 0xc2, 0x57, 0xff, 0x03, 0x14, 0xdb,
    ///     0xf1, 0xb7, 0x60, 0x0c, 0x43, 0x59, 0x3f, 0xff,
    ///     0xac, 0xf1, 0x80, 0x9a, 0x15, 0x6f, 0xd8, 0x6e,
    ///     0xb7, 0x85, 0x18, 0xc8, 0xec, 0x4e, 0x59, 0x4a,
    ///     0xe2, 0x91, 0x43, 0x4c, 0xeb, 0x95, 0xb6, 0x2e,
    ///     0x9a, 0xea, 0x53, 0x68, 0x80, 0x64, 0x69, 0x40,
    ///     0xf9, 0xec, 0xbd, 0x85, 0x89, 0x26, 0x97, 0x67,
    ///     0xaf, 0xb0, 0xad, 0x00, 0x1b, 0xd4, 0xfd, 0x94,
    ///     0xd3, 0xe9, 0x92, 0xb1, 0xb4, 0xbc, 0x5a, 0xaa,
    ///     0x92, 0x80, 0x89, 0x3b, 0x39, 0x05, 0x6c, 0x22,
    ///     0x26, 0xfe, 0x5a, 0x28, 0x6c, 0x37, 0x50, 0x5a,
    ///     0x38, 0x99, 0xcf, 0xf3, 0xc1, 0x96, 0x45, 0xdc,
    ///     0x01, 0xcb, 0x20, 0x87, 0xa5, 0x00, 0x8c, 0xf5,
    ///     0x4d, 0xc2, 0xef, 0xb8, 0x9b, 0xd1, 0x87, 0xbe,
    ///     0xed, 0xd5, 0x0a, 0x29, 0x15, 0x34, 0x59, 0x4c,
    ///     0x3a, 0x05, 0x22, 0x05, 0x44, 0x4f, 0x9f, 0xc8,
    ///     0x47, 0x12, 0x24, 0x8e, 0xa8, 0x79, 0xe4, 0x67,
    ///     0xba, 0x4d, 0x5b, 0x75, 0x56, 0x95, 0xeb, 0xe8,
    ///     0x8a, 0xfa, 0x8e, 0x01, 0x8c, 0x1b, 0x74, 0x63,
    ///     0xd9, 0x2f, 0xf7, 0xd3, 0x44, 0x8f, 0xa8, 0xf5,
    ///     0xaf, 0x6c, 0x4f, 0xdb, 0xe7, 0xc9, 0x6c, 0x71,
    ///     0x22, 0xa3, 0x1d, 0xf1, 0x40, 0xb2, 0xe0, 0x9a,
    ///     0xb6, 0x72, 0xc9, 0xc0, 0x13, 0x16, 0xa2, 0x4a,
    ///     0xe1, 0x92, 0xc7, 0x54, 0x23, 0xab, 0x9d, 0xa1,
    ///     0xa1, 0xe5, 0x0b, 0xed, 0xba, 0xe8, 0x84, 0x37,
    ///     0xb2, 0xe7, 0xfe, 0x32, 0x8d, 0xfa, 0x1c, 0x53,
    ///     0x77, 0x97, 0xc7, 0xf3, 0x48, 0xc9, 0xdb, 0x2d,
    ///     0x75, 0x52, 0x9d, 0x42, 0x51, 0x78, 0x62, 0x68,
    ///     0x05, 0x45, 0x15, 0xf8, 0xa2, 0x4e, 0xf3, 0x0b
    /// ];
    /// let q = [
    ///     0xe0u8, 0x35, 0x37, 0xaf, 0xb2, 0x50, 0x91, 0x8e,
    ///     0xf2, 0x62, 0x2b, 0xd9, 0x9f, 0x6c, 0x11, 0x75,
    ///     0xec, 0x24, 0x5d, 0x78, 0x59, 0xe7, 0x8d, 0xb5,
    ///     0x40, 0x52, 0xed, 0x41
    /// ];
    /// let dh = DH::new_from_pgq(&p, &g, &q).expect("Error with new_from_pgq()");
    /// ```
    pub fn new_from_pgq(p: &[u8], g: &[u8], q: &[u8]) -> Result<Self, i32> {
        Self::new_from_pgq_ex(p, g, q, None, None)
    }

    /// Create a new DH context using the given p, g, and q parameters with
    /// optional heap and device ID.
    ///
    /// # Parameters
    ///
    /// * `p`: DH 'p' parameter value.
    /// * `g`: DH 'g' parameter value.
    /// * `q`: DH 'q' parameter value.
    /// * `heap`: Optional heap hint.
    /// * `dev_id` Optional device ID to use with crypto callbacks or async hardware.
    ///
    /// # Returns
    ///
    /// Returns either Ok(dh) containing the DH struct instance or Err(e)
    /// containing the wolfSSL library error code value.
    ///
    /// # Example
    ///
    /// ```rust
    /// use wolfssl::wolfcrypt::random::RNG;
    /// use wolfssl::wolfcrypt::dh::DH;
    /// let p = [
    ///     0xc5u8, 0x7c, 0xa2, 0x4f, 0x4b, 0xd6, 0x8c, 0x3c,
    ///     0xda, 0xc7, 0xba, 0xaa, 0xea, 0x2e, 0x5c, 0x1e,
    ///     0x18, 0xb2, 0x7b, 0x8c, 0x55, 0x65, 0x9f, 0xea,
    ///     0xe0, 0xa1, 0x36, 0x53, 0x2b, 0x36, 0xe0, 0x4e,
    ///     0x3e, 0x64, 0xa9, 0xe4, 0xfc, 0x8f, 0x32, 0x62,
    ///     0x97, 0xe4, 0xbe, 0xf7, 0xc1, 0xde, 0x07, 0x5a,
    ///     0x89, 0x28, 0xf3, 0xfe, 0x4f, 0xfe, 0x68, 0xbc,
    ///     0xfb, 0x0a, 0x7c, 0xa4, 0xb3, 0x14, 0x48, 0x89,
    ///     0x9f, 0xaf, 0xb8, 0x43, 0xe2, 0xa0, 0x62, 0x5c,
    ///     0xb4, 0x88, 0x3f, 0x06, 0x50, 0x11, 0xfe, 0x65,
    ///     0x8d, 0x49, 0xd2, 0xf5, 0x4b, 0x74, 0x79, 0xdb,
    ///     0x06, 0x62, 0x92, 0x89, 0xed, 0xda, 0xcb, 0x87,
    ///     0x37, 0x16, 0xd2, 0xa1, 0x7a, 0xe8, 0xde, 0x92,
    ///     0xee, 0x3e, 0x41, 0x4a, 0x91, 0x5e, 0xed, 0xf3,
    ///     0x6c, 0x6b, 0x7e, 0xfd, 0x15, 0x92, 0x18, 0xfc,
    ///     0xa7, 0xac, 0x42, 0x85, 0x57, 0xe9, 0xdc, 0xda,
    ///     0x55, 0xc9, 0x8b, 0x28, 0x9e, 0xc1, 0xc4, 0x46,
    ///     0x4d, 0x88, 0xed, 0x62, 0x8e, 0xdb, 0x3f, 0xb9,
    ///     0xd7, 0xc8, 0xe3, 0xcf, 0xb8, 0x34, 0x2c, 0xd2,
    ///     0x6f, 0x28, 0x06, 0x41, 0xe3, 0x66, 0x8c, 0xfc,
    ///     0x72, 0xff, 0x26, 0x3b, 0x6b, 0x6c, 0x6f, 0x73,
    ///     0xde, 0xf2, 0x90, 0x29, 0xe0, 0x61, 0x32, 0xc4,
    ///     0x12, 0x74, 0x09, 0x52, 0xec, 0xf3, 0x1b, 0xa6,
    ///     0x45, 0x98, 0xac, 0xf9, 0x1c, 0x65, 0x8e, 0x3a,
    ///     0x91, 0x84, 0x4b, 0x23, 0x8a, 0xb2, 0x3c, 0xc9,
    ///     0xfa, 0xea, 0xf1, 0x38, 0xce, 0xd8, 0x05, 0xe0,
    ///     0xfa, 0x44, 0x68, 0x1f, 0xeb, 0xd9, 0x57, 0xb8,
    ///     0x4a, 0x97, 0x5b, 0x88, 0xc5, 0xf1, 0xbb, 0xb0,
    ///     0x49, 0xc3, 0x91, 0x7c, 0xd3, 0x13, 0xb9, 0x47,
    ///     0xbb, 0x91, 0x8f, 0xe5, 0x26, 0x07, 0xab, 0xa9,
    ///     0xc5, 0xd0, 0x3d, 0x95, 0x41, 0x26, 0x92, 0x9d,
    ///     0x13, 0x67, 0xf2, 0x7e, 0x11, 0x88, 0xdc, 0x2d
    /// ];
    /// let g = [
    ///     0x4au8, 0x1a, 0xf3, 0xa4, 0x92, 0xe9, 0xee, 0x74,
    ///     0x6e, 0x57, 0xd5, 0x8c, 0x2c, 0x5b, 0x41, 0x41,
    ///     0x5e, 0xd4, 0x55, 0x19, 0xdc, 0xd9, 0x32, 0x91,
    ///     0xf7, 0xfd, 0xc2, 0x57, 0xff, 0x03, 0x14, 0xdb,
    ///     0xf1, 0xb7, 0x60, 0x0c, 0x43, 0x59, 0x3f, 0xff,
    ///     0xac, 0xf1, 0x80, 0x9a, 0x15, 0x6f, 0xd8, 0x6e,
    ///     0xb7, 0x85, 0x18, 0xc8, 0xec, 0x4e, 0x59, 0x4a,
    ///     0xe2, 0x91, 0x43, 0x4c, 0xeb, 0x95, 0xb6, 0x2e,
    ///     0x9a, 0xea, 0x53, 0x68, 0x80, 0x64, 0x69, 0x40,
    ///     0xf9, 0xec, 0xbd, 0x85, 0x89, 0x26, 0x97, 0x67,
    ///     0xaf, 0xb0, 0xad, 0x00, 0x1b, 0xd4, 0xfd, 0x94,
    ///     0xd3, 0xe9, 0x92, 0xb1, 0xb4, 0xbc, 0x5a, 0xaa,
    ///     0x92, 0x80, 0x89, 0x3b, 0x39, 0x05, 0x6c, 0x22,
    ///     0x26, 0xfe, 0x5a, 0x28, 0x6c, 0x37, 0x50, 0x5a,
    ///     0x38, 0x99, 0xcf, 0xf3, 0xc1, 0x96, 0x45, 0xdc,
    ///     0x01, 0xcb, 0x20, 0x87, 0xa5, 0x00, 0x8c, 0xf5,
    ///     0x4d, 0xc2, 0xef, 0xb8, 0x9b, 0xd1, 0x87, 0xbe,
    ///     0xed, 0xd5, 0x0a, 0x29, 0x15, 0x34, 0x59, 0x4c,
    ///     0x3a, 0x05, 0x22, 0x05, 0x44, 0x4f, 0x9f, 0xc8,
    ///     0x47, 0x12, 0x24, 0x8e, 0xa8, 0x79, 0xe4, 0x67,
    ///     0xba, 0x4d, 0x5b, 0x75, 0x56, 0x95, 0xeb, 0xe8,
    ///     0x8a, 0xfa, 0x8e, 0x01, 0x8c, 0x1b, 0x74, 0x63,
    ///     0xd9, 0x2f, 0xf7, 0xd3, 0x44, 0x8f, 0xa8, 0xf5,
    ///     0xaf, 0x6c, 0x4f, 0xdb, 0xe7, 0xc9, 0x6c, 0x71,
    ///     0x22, 0xa3, 0x1d, 0xf1, 0x40, 0xb2, 0xe0, 0x9a,
    ///     0xb6, 0x72, 0xc9, 0xc0, 0x13, 0x16, 0xa2, 0x4a,
    ///     0xe1, 0x92, 0xc7, 0x54, 0x23, 0xab, 0x9d, 0xa1,
    ///     0xa1, 0xe5, 0x0b, 0xed, 0xba, 0xe8, 0x84, 0x37,
    ///     0xb2, 0xe7, 0xfe, 0x32, 0x8d, 0xfa, 0x1c, 0x53,
    ///     0x77, 0x97, 0xc7, 0xf3, 0x48, 0xc9, 0xdb, 0x2d,
    ///     0x75, 0x52, 0x9d, 0x42, 0x51, 0x78, 0x62, 0x68,
    ///     0x05, 0x45, 0x15, 0xf8, 0xa2, 0x4e, 0xf3, 0x0b
    /// ];
    /// let q = [
    ///     0xe0u8, 0x35, 0x37, 0xaf, 0xb2, 0x50, 0x91, 0x8e,
    ///     0xf2, 0x62, 0x2b, 0xd9, 0x9f, 0x6c, 0x11, 0x75,
    ///     0xec, 0x24, 0x5d, 0x78, 0x59, 0xe7, 0x8d, 0xb5,
    ///     0x40, 0x52, 0xed, 0x41
    /// ];
    /// let dh = DH::new_from_pgq_ex(&p, &g, &q, None, None).expect("Error with new_from_pgq_ex()");
    /// ```
    pub fn new_from_pgq_ex(p: &[u8], g: &[u8], q: &[u8], heap: Option<*mut std::os::raw::c_void>, dev_id: Option<i32>) -> Result<Self, i32> {
        let p_size = p.len() as u32;
        let g_size = g.len() as u32;
        let q_size = q.len() as u32;
        let mut wc_dhkey: MaybeUninit<sys::DhKey> = MaybeUninit::uninit();
        let heap = match heap {
            Some(heap) => heap,
            None => core::ptr::null_mut(),
        };
        let dev_id = match dev_id {
            Some(dev_id) => dev_id,
            None => sys::INVALID_DEVID,
        };
        let rc = unsafe { sys::wc_InitDhKey_ex(wc_dhkey.as_mut_ptr(), heap, dev_id) };
        if rc != 0 {
            return Err(rc);
        }
        let wc_dhkey = unsafe { wc_dhkey.assume_init() };
        let mut dh = DH { wc_dhkey };
        let rc = unsafe {
            sys::wc_DhSetKey_ex(&mut dh.wc_dhkey, p.as_ptr(), p_size, g.as_ptr(), g_size, q.as_ptr(), q_size)
        };
        if rc != 0 {
            return Err(rc);
        }
        Ok(dh)
    }

    /// Create a new DH context using the given p, g, and q parameters with
    /// check.
    ///
    /// # Parameters
    ///
    /// * `p`: DH 'p' parameter value.
    /// * `g`: DH 'g' parameter value.
    /// * `q`: DH 'q' parameter value.
    /// * `trusted`: Whether to skip the prime check for `p` parameter and mark
    ///   the DH context as trusted.
    /// * `rng`: `RNG` instance to use for random number generation.
    ///
    /// # Returns
    ///
    /// Returns either Ok(dh) containing the DH struct instance or Err(e)
    /// containing the wolfSSL library error code value.
    ///
    /// # Example
    ///
    /// ```rust
    /// use wolfssl::wolfcrypt::random::RNG;
    /// use wolfssl::wolfcrypt::dh::DH;
    /// let p = [
    ///     0xc5u8, 0x7c, 0xa2, 0x4f, 0x4b, 0xd6, 0x8c, 0x3c,
    ///     0xda, 0xc7, 0xba, 0xaa, 0xea, 0x2e, 0x5c, 0x1e,
    ///     0x18, 0xb2, 0x7b, 0x8c, 0x55, 0x65, 0x9f, 0xea,
    ///     0xe0, 0xa1, 0x36, 0x53, 0x2b, 0x36, 0xe0, 0x4e,
    ///     0x3e, 0x64, 0xa9, 0xe4, 0xfc, 0x8f, 0x32, 0x62,
    ///     0x97, 0xe4, 0xbe, 0xf7, 0xc1, 0xde, 0x07, 0x5a,
    ///     0x89, 0x28, 0xf3, 0xfe, 0x4f, 0xfe, 0x68, 0xbc,
    ///     0xfb, 0x0a, 0x7c, 0xa4, 0xb3, 0x14, 0x48, 0x89,
    ///     0x9f, 0xaf, 0xb8, 0x43, 0xe2, 0xa0, 0x62, 0x5c,
    ///     0xb4, 0x88, 0x3f, 0x06, 0x50, 0x11, 0xfe, 0x65,
    ///     0x8d, 0x49, 0xd2, 0xf5, 0x4b, 0x74, 0x79, 0xdb,
    ///     0x06, 0x62, 0x92, 0x89, 0xed, 0xda, 0xcb, 0x87,
    ///     0x37, 0x16, 0xd2, 0xa1, 0x7a, 0xe8, 0xde, 0x92,
    ///     0xee, 0x3e, 0x41, 0x4a, 0x91, 0x5e, 0xed, 0xf3,
    ///     0x6c, 0x6b, 0x7e, 0xfd, 0x15, 0x92, 0x18, 0xfc,
    ///     0xa7, 0xac, 0x42, 0x85, 0x57, 0xe9, 0xdc, 0xda,
    ///     0x55, 0xc9, 0x8b, 0x28, 0x9e, 0xc1, 0xc4, 0x46,
    ///     0x4d, 0x88, 0xed, 0x62, 0x8e, 0xdb, 0x3f, 0xb9,
    ///     0xd7, 0xc8, 0xe3, 0xcf, 0xb8, 0x34, 0x2c, 0xd2,
    ///     0x6f, 0x28, 0x06, 0x41, 0xe3, 0x66, 0x8c, 0xfc,
    ///     0x72, 0xff, 0x26, 0x3b, 0x6b, 0x6c, 0x6f, 0x73,
    ///     0xde, 0xf2, 0x90, 0x29, 0xe0, 0x61, 0x32, 0xc4,
    ///     0x12, 0x74, 0x09, 0x52, 0xec, 0xf3, 0x1b, 0xa6,
    ///     0x45, 0x98, 0xac, 0xf9, 0x1c, 0x65, 0x8e, 0x3a,
    ///     0x91, 0x84, 0x4b, 0x23, 0x8a, 0xb2, 0x3c, 0xc9,
    ///     0xfa, 0xea, 0xf1, 0x38, 0xce, 0xd8, 0x05, 0xe0,
    ///     0xfa, 0x44, 0x68, 0x1f, 0xeb, 0xd9, 0x57, 0xb8,
    ///     0x4a, 0x97, 0x5b, 0x88, 0xc5, 0xf1, 0xbb, 0xb0,
    ///     0x49, 0xc3, 0x91, 0x7c, 0xd3, 0x13, 0xb9, 0x47,
    ///     0xbb, 0x91, 0x8f, 0xe5, 0x26, 0x07, 0xab, 0xa9,
    ///     0xc5, 0xd0, 0x3d, 0x95, 0x41, 0x26, 0x92, 0x9d,
    ///     0x13, 0x67, 0xf2, 0x7e, 0x11, 0x88, 0xdc, 0x2d
    /// ];
    /// let g = [
    ///     0x4au8, 0x1a, 0xf3, 0xa4, 0x92, 0xe9, 0xee, 0x74,
    ///     0x6e, 0x57, 0xd5, 0x8c, 0x2c, 0x5b, 0x41, 0x41,
    ///     0x5e, 0xd4, 0x55, 0x19, 0xdc, 0xd9, 0x32, 0x91,
    ///     0xf7, 0xfd, 0xc2, 0x57, 0xff, 0x03, 0x14, 0xdb,
    ///     0xf1, 0xb7, 0x60, 0x0c, 0x43, 0x59, 0x3f, 0xff,
    ///     0xac, 0xf1, 0x80, 0x9a, 0x15, 0x6f, 0xd8, 0x6e,
    ///     0xb7, 0x85, 0x18, 0xc8, 0xec, 0x4e, 0x59, 0x4a,
    ///     0xe2, 0x91, 0x43, 0x4c, 0xeb, 0x95, 0xb6, 0x2e,
    ///     0x9a, 0xea, 0x53, 0x68, 0x80, 0x64, 0x69, 0x40,
    ///     0xf9, 0xec, 0xbd, 0x85, 0x89, 0x26, 0x97, 0x67,
    ///     0xaf, 0xb0, 0xad, 0x00, 0x1b, 0xd4, 0xfd, 0x94,
    ///     0xd3, 0xe9, 0x92, 0xb1, 0xb4, 0xbc, 0x5a, 0xaa,
    ///     0x92, 0x80, 0x89, 0x3b, 0x39, 0x05, 0x6c, 0x22,
    ///     0x26, 0xfe, 0x5a, 0x28, 0x6c, 0x37, 0x50, 0x5a,
    ///     0x38, 0x99, 0xcf, 0xf3, 0xc1, 0x96, 0x45, 0xdc,
    ///     0x01, 0xcb, 0x20, 0x87, 0xa5, 0x00, 0x8c, 0xf5,
    ///     0x4d, 0xc2, 0xef, 0xb8, 0x9b, 0xd1, 0x87, 0xbe,
    ///     0xed, 0xd5, 0x0a, 0x29, 0x15, 0x34, 0x59, 0x4c,
    ///     0x3a, 0x05, 0x22, 0x05, 0x44, 0x4f, 0x9f, 0xc8,
    ///     0x47, 0x12, 0x24, 0x8e, 0xa8, 0x79, 0xe4, 0x67,
    ///     0xba, 0x4d, 0x5b, 0x75, 0x56, 0x95, 0xeb, 0xe8,
    ///     0x8a, 0xfa, 0x8e, 0x01, 0x8c, 0x1b, 0x74, 0x63,
    ///     0xd9, 0x2f, 0xf7, 0xd3, 0x44, 0x8f, 0xa8, 0xf5,
    ///     0xaf, 0x6c, 0x4f, 0xdb, 0xe7, 0xc9, 0x6c, 0x71,
    ///     0x22, 0xa3, 0x1d, 0xf1, 0x40, 0xb2, 0xe0, 0x9a,
    ///     0xb6, 0x72, 0xc9, 0xc0, 0x13, 0x16, 0xa2, 0x4a,
    ///     0xe1, 0x92, 0xc7, 0x54, 0x23, 0xab, 0x9d, 0xa1,
    ///     0xa1, 0xe5, 0x0b, 0xed, 0xba, 0xe8, 0x84, 0x37,
    ///     0xb2, 0xe7, 0xfe, 0x32, 0x8d, 0xfa, 0x1c, 0x53,
    ///     0x77, 0x97, 0xc7, 0xf3, 0x48, 0xc9, 0xdb, 0x2d,
    ///     0x75, 0x52, 0x9d, 0x42, 0x51, 0x78, 0x62, 0x68,
    ///     0x05, 0x45, 0x15, 0xf8, 0xa2, 0x4e, 0xf3, 0x0b
    /// ];
    /// let q = [
    ///     0xe0u8, 0x35, 0x37, 0xaf, 0xb2, 0x50, 0x91, 0x8e,
    ///     0xf2, 0x62, 0x2b, 0xd9, 0x9f, 0x6c, 0x11, 0x75,
    ///     0xec, 0x24, 0x5d, 0x78, 0x59, 0xe7, 0x8d, 0xb5,
    ///     0x40, 0x52, 0xed, 0x41
    /// ];
    /// let mut rng = RNG::new().expect("Failed to create RNG");
    /// let dh = DH::new_from_pgq_with_check(&p, &g, &q, 0, &mut rng).expect("Error with new_from_pgq_with_check()");
    /// ```
    pub fn new_from_pgq_with_check(p: &[u8], g: &[u8], q: &[u8], trusted: i32, rng: &mut RNG) -> Result<Self, i32> {
        Self::new_from_pgq_with_check_ex(p, g, q, trusted, rng, None, None)
    }

    /// Create a new DH context using the given p, g, and q parameters with
    /// check and optional heap and device ID.
    ///
    /// # Parameters
    ///
    /// * `p`: DH 'p' parameter value.
    /// * `g`: DH 'g' parameter value.
    /// * `q`: DH 'q' parameter value.
    /// * `trusted`: Whether to skip the prime check for `p` parameter and mark
    ///   the DH context as trusted.
    /// * `rng`: `RNG` instance to use for random number generation.
    /// * `heap`: Optional heap hint.
    /// * `dev_id` Optional device ID to use with crypto callbacks or async hardware.
    ///
    /// # Returns
    ///
    /// Returns either Ok(dh) containing the DH struct instance or Err(e)
    /// containing the wolfSSL library error code value.
    ///
    /// # Example
    ///
    /// ```rust
    /// use wolfssl::wolfcrypt::random::RNG;
    /// use wolfssl::wolfcrypt::dh::DH;
    /// let p = [
    ///     0xc5u8, 0x7c, 0xa2, 0x4f, 0x4b, 0xd6, 0x8c, 0x3c,
    ///     0xda, 0xc7, 0xba, 0xaa, 0xea, 0x2e, 0x5c, 0x1e,
    ///     0x18, 0xb2, 0x7b, 0x8c, 0x55, 0x65, 0x9f, 0xea,
    ///     0xe0, 0xa1, 0x36, 0x53, 0x2b, 0x36, 0xe0, 0x4e,
    ///     0x3e, 0x64, 0xa9, 0xe4, 0xfc, 0x8f, 0x32, 0x62,
    ///     0x97, 0xe4, 0xbe, 0xf7, 0xc1, 0xde, 0x07, 0x5a,
    ///     0x89, 0x28, 0xf3, 0xfe, 0x4f, 0xfe, 0x68, 0xbc,
    ///     0xfb, 0x0a, 0x7c, 0xa4, 0xb3, 0x14, 0x48, 0x89,
    ///     0x9f, 0xaf, 0xb8, 0x43, 0xe2, 0xa0, 0x62, 0x5c,
    ///     0xb4, 0x88, 0x3f, 0x06, 0x50, 0x11, 0xfe, 0x65,
    ///     0x8d, 0x49, 0xd2, 0xf5, 0x4b, 0x74, 0x79, 0xdb,
    ///     0x06, 0x62, 0x92, 0x89, 0xed, 0xda, 0xcb, 0x87,
    ///     0x37, 0x16, 0xd2, 0xa1, 0x7a, 0xe8, 0xde, 0x92,
    ///     0xee, 0x3e, 0x41, 0x4a, 0x91, 0x5e, 0xed, 0xf3,
    ///     0x6c, 0x6b, 0x7e, 0xfd, 0x15, 0x92, 0x18, 0xfc,
    ///     0xa7, 0xac, 0x42, 0x85, 0x57, 0xe9, 0xdc, 0xda,
    ///     0x55, 0xc9, 0x8b, 0x28, 0x9e, 0xc1, 0xc4, 0x46,
    ///     0x4d, 0x88, 0xed, 0x62, 0x8e, 0xdb, 0x3f, 0xb9,
    ///     0xd7, 0xc8, 0xe3, 0xcf, 0xb8, 0x34, 0x2c, 0xd2,
    ///     0x6f, 0x28, 0x06, 0x41, 0xe3, 0x66, 0x8c, 0xfc,
    ///     0x72, 0xff, 0x26, 0x3b, 0x6b, 0x6c, 0x6f, 0x73,
    ///     0xde, 0xf2, 0x90, 0x29, 0xe0, 0x61, 0x32, 0xc4,
    ///     0x12, 0x74, 0x09, 0x52, 0xec, 0xf3, 0x1b, 0xa6,
    ///     0x45, 0x98, 0xac, 0xf9, 0x1c, 0x65, 0x8e, 0x3a,
    ///     0x91, 0x84, 0x4b, 0x23, 0x8a, 0xb2, 0x3c, 0xc9,
    ///     0xfa, 0xea, 0xf1, 0x38, 0xce, 0xd8, 0x05, 0xe0,
    ///     0xfa, 0x44, 0x68, 0x1f, 0xeb, 0xd9, 0x57, 0xb8,
    ///     0x4a, 0x97, 0x5b, 0x88, 0xc5, 0xf1, 0xbb, 0xb0,
    ///     0x49, 0xc3, 0x91, 0x7c, 0xd3, 0x13, 0xb9, 0x47,
    ///     0xbb, 0x91, 0x8f, 0xe5, 0x26, 0x07, 0xab, 0xa9,
    ///     0xc5, 0xd0, 0x3d, 0x95, 0x41, 0x26, 0x92, 0x9d,
    ///     0x13, 0x67, 0xf2, 0x7e, 0x11, 0x88, 0xdc, 0x2d
    /// ];
    /// let g = [
    ///     0x4au8, 0x1a, 0xf3, 0xa4, 0x92, 0xe9, 0xee, 0x74,
    ///     0x6e, 0x57, 0xd5, 0x8c, 0x2c, 0x5b, 0x41, 0x41,
    ///     0x5e, 0xd4, 0x55, 0x19, 0xdc, 0xd9, 0x32, 0x91,
    ///     0xf7, 0xfd, 0xc2, 0x57, 0xff, 0x03, 0x14, 0xdb,
    ///     0xf1, 0xb7, 0x60, 0x0c, 0x43, 0x59, 0x3f, 0xff,
    ///     0xac, 0xf1, 0x80, 0x9a, 0x15, 0x6f, 0xd8, 0x6e,
    ///     0xb7, 0x85, 0x18, 0xc8, 0xec, 0x4e, 0x59, 0x4a,
    ///     0xe2, 0x91, 0x43, 0x4c, 0xeb, 0x95, 0xb6, 0x2e,
    ///     0x9a, 0xea, 0x53, 0x68, 0x80, 0x64, 0x69, 0x40,
    ///     0xf9, 0xec, 0xbd, 0x85, 0x89, 0x26, 0x97, 0x67,
    ///     0xaf, 0xb0, 0xad, 0x00, 0x1b, 0xd4, 0xfd, 0x94,
    ///     0xd3, 0xe9, 0x92, 0xb1, 0xb4, 0xbc, 0x5a, 0xaa,
    ///     0x92, 0x80, 0x89, 0x3b, 0x39, 0x05, 0x6c, 0x22,
    ///     0x26, 0xfe, 0x5a, 0x28, 0x6c, 0x37, 0x50, 0x5a,
    ///     0x38, 0x99, 0xcf, 0xf3, 0xc1, 0x96, 0x45, 0xdc,
    ///     0x01, 0xcb, 0x20, 0x87, 0xa5, 0x00, 0x8c, 0xf5,
    ///     0x4d, 0xc2, 0xef, 0xb8, 0x9b, 0xd1, 0x87, 0xbe,
    ///     0xed, 0xd5, 0x0a, 0x29, 0x15, 0x34, 0x59, 0x4c,
    ///     0x3a, 0x05, 0x22, 0x05, 0x44, 0x4f, 0x9f, 0xc8,
    ///     0x47, 0x12, 0x24, 0x8e, 0xa8, 0x79, 0xe4, 0x67,
    ///     0xba, 0x4d, 0x5b, 0x75, 0x56, 0x95, 0xeb, 0xe8,
    ///     0x8a, 0xfa, 0x8e, 0x01, 0x8c, 0x1b, 0x74, 0x63,
    ///     0xd9, 0x2f, 0xf7, 0xd3, 0x44, 0x8f, 0xa8, 0xf5,
    ///     0xaf, 0x6c, 0x4f, 0xdb, 0xe7, 0xc9, 0x6c, 0x71,
    ///     0x22, 0xa3, 0x1d, 0xf1, 0x40, 0xb2, 0xe0, 0x9a,
    ///     0xb6, 0x72, 0xc9, 0xc0, 0x13, 0x16, 0xa2, 0x4a,
    ///     0xe1, 0x92, 0xc7, 0x54, 0x23, 0xab, 0x9d, 0xa1,
    ///     0xa1, 0xe5, 0x0b, 0xed, 0xba, 0xe8, 0x84, 0x37,
    ///     0xb2, 0xe7, 0xfe, 0x32, 0x8d, 0xfa, 0x1c, 0x53,
    ///     0x77, 0x97, 0xc7, 0xf3, 0x48, 0xc9, 0xdb, 0x2d,
    ///     0x75, 0x52, 0x9d, 0x42, 0x51, 0x78, 0x62, 0x68,
    ///     0x05, 0x45, 0x15, 0xf8, 0xa2, 0x4e, 0xf3, 0x0b
    /// ];
    /// let q = [
    ///     0xe0u8, 0x35, 0x37, 0xaf, 0xb2, 0x50, 0x91, 0x8e,
    ///     0xf2, 0x62, 0x2b, 0xd9, 0x9f, 0x6c, 0x11, 0x75,
    ///     0xec, 0x24, 0x5d, 0x78, 0x59, 0xe7, 0x8d, 0xb5,
    ///     0x40, 0x52, 0xed, 0x41
    /// ];
    /// let mut rng = RNG::new().expect("Failed to create RNG");
    /// let dh = DH::new_from_pgq_with_check_ex(&p, &g, &q, 0, &mut rng, None, None).expect("Error with new_from_pgq_with_check_ex()");
    /// ```
    pub fn new_from_pgq_with_check_ex(p: &[u8], g: &[u8], q: &[u8], trusted: i32, rng: &mut RNG, heap: Option<*mut std::os::raw::c_void>, dev_id: Option<i32>) -> Result<Self, i32> {
        let p_size = p.len() as u32;
        let g_size = g.len() as u32;
        let q_size = q.len() as u32;
        let mut wc_dhkey: MaybeUninit<sys::DhKey> = MaybeUninit::uninit();
        let heap = match heap {
            Some(heap) => heap,
            None => core::ptr::null_mut(),
        };
        let dev_id = match dev_id {
            Some(dev_id) => dev_id,
            None => sys::INVALID_DEVID,
        };
        let rc = unsafe { sys::wc_InitDhKey_ex(wc_dhkey.as_mut_ptr(), heap, dev_id) };
        if rc != 0 {
            return Err(rc);
        }
        let wc_dhkey = unsafe { wc_dhkey.assume_init() };
        let mut dh = DH { wc_dhkey };
        let rc = unsafe {
            sys::wc_DhSetCheckKey(&mut dh.wc_dhkey, p.as_ptr(), p_size, g.as_ptr(), g_size, q.as_ptr(), q_size, trusted, &mut rng.wc_rng)
        };
        if rc != 0 {
            return Err(rc);
        }
        Ok(dh)
    }

    /// Check public/private key pair for pair-wise consistency per process in
    /// SP 800-56Ar3, section 5.6.2.1.4, method (b) for FFC.
    ///
    /// # Parameters
    ///
    /// * `public`: Buffer containing public key.
    /// * `private`: Buffer containing private key.
    ///
    /// # Returns
    ///
    /// Returns either Ok(()) on success or Err(e) containing the wolfSSL
    /// library error code value.
    ///
    /// # Example
    ///
    /// ```rust
    /// use wolfssl::wolfcrypt::random::RNG;
    /// use wolfssl::wolfcrypt::dh::DH;
    /// let mut rng = RNG::new().expect("Error with RNG::new()");
    /// let mut dh = DH::new_named(DH::FFDHE_2048).expect("Error with new_named()");
    /// let mut private = [0u8; 256];
    /// let mut private_size = 0u32;
    /// let mut public = [0u8; 256];
    /// let mut public_size = 0u32;
    /// dh.generate_key_pair(&mut rng, &mut private, &mut private_size, &mut public, &mut public_size).expect("Error with generate_key_pair()");
    /// let private = &private[0..(private_size as usize)];
    /// let public = &public[0..(public_size as usize)];
    /// dh.check_key_pair(public, private).expect("Error with check_key_pair()");
    /// ```
    pub fn check_key_pair(&mut self, public: &[u8], private: &[u8]) -> Result<(), i32> {
        let public_size = public.len() as u32;
        let private_size = private.len() as u32;
        let rc = unsafe {
            sys::wc_DhCheckKeyPair(&mut self.wc_dhkey,
                public.as_ptr(), public_size,
                private.as_ptr(), private_size)
        };
        if rc != 0 {
            return Err(rc);
        }
        Ok(())
    }

    /// Check private key for invalid numbers.
    ///
    /// The check is per process in SP 800-56Ar3, section 5.6.2.1.2.
    ///
    /// # Parameters
    ///
    /// * `private`: Buffer containing private key.
    ///
    /// # Returns
    ///
    /// Returns either Ok(()) on success or Err(e) containing the wolfSSL
    /// library error code value.
    ///
    /// # Example
    ///
    /// ```rust
    /// use wolfssl::wolfcrypt::random::RNG;
    /// use wolfssl::wolfcrypt::dh::DH;
    /// let mut rng = RNG::new().expect("Error with RNG::new()");
    /// let mut dh = DH::new_named(DH::FFDHE_2048).expect("Error with new_named()");
    /// let mut private = [0u8; 256];
    /// let mut private_size = 0u32;
    /// let mut public = [0u8; 256];
    /// let mut public_size = 0u32;
    /// dh.generate_key_pair(&mut rng, &mut private, &mut private_size, &mut public, &mut public_size).expect("Error with generate_key_pair()");
    /// let private = &private[0..(private_size as usize)];
    /// dh.check_priv_key(private).expect("Error with check_priv_key()");
    /// ```
    pub fn check_priv_key(&mut self, private: &[u8]) -> Result<(), i32> {
        let private_size = private.len() as u32;
        let rc = unsafe {
            sys::wc_DhCheckPrivKey(&mut self.wc_dhkey,
                private.as_ptr(), private_size)
        };
        if rc != 0 {
            return Err(rc);
        }
        Ok(())
    }

    /// Check private key for invalid numbers with optional prime value.
    ///
    /// This optionally allows the private key to be checked against the large
    /// prime (q).
    /// The check is per process in SP 800-56Ar3, section 5.6.2.1.2.
    ///
    /// # Parameters
    ///
    /// * `private`: Buffer containing private key.
    /// * `prime`: Buffer containing prime value (optional).
    ///
    /// # Returns
    ///
    /// Returns either Ok(()) on success or Err(e) containing the wolfSSL
    /// library error code value.
    ///
    /// # Example
    ///
    /// ```rust
    /// use wolfssl::wolfcrypt::random::RNG;
    /// use wolfssl::wolfcrypt::dh::DH;
    /// let p = [
    ///     0xc5u8, 0x7c, 0xa2, 0x4f, 0x4b, 0xd6, 0x8c, 0x3c,
    ///     0xda, 0xc7, 0xba, 0xaa, 0xea, 0x2e, 0x5c, 0x1e,
    ///     0x18, 0xb2, 0x7b, 0x8c, 0x55, 0x65, 0x9f, 0xea,
    ///     0xe0, 0xa1, 0x36, 0x53, 0x2b, 0x36, 0xe0, 0x4e,
    ///     0x3e, 0x64, 0xa9, 0xe4, 0xfc, 0x8f, 0x32, 0x62,
    ///     0x97, 0xe4, 0xbe, 0xf7, 0xc1, 0xde, 0x07, 0x5a,
    ///     0x89, 0x28, 0xf3, 0xfe, 0x4f, 0xfe, 0x68, 0xbc,
    ///     0xfb, 0x0a, 0x7c, 0xa4, 0xb3, 0x14, 0x48, 0x89,
    ///     0x9f, 0xaf, 0xb8, 0x43, 0xe2, 0xa0, 0x62, 0x5c,
    ///     0xb4, 0x88, 0x3f, 0x06, 0x50, 0x11, 0xfe, 0x65,
    ///     0x8d, 0x49, 0xd2, 0xf5, 0x4b, 0x74, 0x79, 0xdb,
    ///     0x06, 0x62, 0x92, 0x89, 0xed, 0xda, 0xcb, 0x87,
    ///     0x37, 0x16, 0xd2, 0xa1, 0x7a, 0xe8, 0xde, 0x92,
    ///     0xee, 0x3e, 0x41, 0x4a, 0x91, 0x5e, 0xed, 0xf3,
    ///     0x6c, 0x6b, 0x7e, 0xfd, 0x15, 0x92, 0x18, 0xfc,
    ///     0xa7, 0xac, 0x42, 0x85, 0x57, 0xe9, 0xdc, 0xda,
    ///     0x55, 0xc9, 0x8b, 0x28, 0x9e, 0xc1, 0xc4, 0x46,
    ///     0x4d, 0x88, 0xed, 0x62, 0x8e, 0xdb, 0x3f, 0xb9,
    ///     0xd7, 0xc8, 0xe3, 0xcf, 0xb8, 0x34, 0x2c, 0xd2,
    ///     0x6f, 0x28, 0x06, 0x41, 0xe3, 0x66, 0x8c, 0xfc,
    ///     0x72, 0xff, 0x26, 0x3b, 0x6b, 0x6c, 0x6f, 0x73,
    ///     0xde, 0xf2, 0x90, 0x29, 0xe0, 0x61, 0x32, 0xc4,
    ///     0x12, 0x74, 0x09, 0x52, 0xec, 0xf3, 0x1b, 0xa6,
    ///     0x45, 0x98, 0xac, 0xf9, 0x1c, 0x65, 0x8e, 0x3a,
    ///     0x91, 0x84, 0x4b, 0x23, 0x8a, 0xb2, 0x3c, 0xc9,
    ///     0xfa, 0xea, 0xf1, 0x38, 0xce, 0xd8, 0x05, 0xe0,
    ///     0xfa, 0x44, 0x68, 0x1f, 0xeb, 0xd9, 0x57, 0xb8,
    ///     0x4a, 0x97, 0x5b, 0x88, 0xc5, 0xf1, 0xbb, 0xb0,
    ///     0x49, 0xc3, 0x91, 0x7c, 0xd3, 0x13, 0xb9, 0x47,
    ///     0xbb, 0x91, 0x8f, 0xe5, 0x26, 0x07, 0xab, 0xa9,
    ///     0xc5, 0xd0, 0x3d, 0x95, 0x41, 0x26, 0x92, 0x9d,
    ///     0x13, 0x67, 0xf2, 0x7e, 0x11, 0x88, 0xdc, 0x2d
    /// ];
    /// let g = [
    ///     0x4au8, 0x1a, 0xf3, 0xa4, 0x92, 0xe9, 0xee, 0x74,
    ///     0x6e, 0x57, 0xd5, 0x8c, 0x2c, 0x5b, 0x41, 0x41,
    ///     0x5e, 0xd4, 0x55, 0x19, 0xdc, 0xd9, 0x32, 0x91,
    ///     0xf7, 0xfd, 0xc2, 0x57, 0xff, 0x03, 0x14, 0xdb,
    ///     0xf1, 0xb7, 0x60, 0x0c, 0x43, 0x59, 0x3f, 0xff,
    ///     0xac, 0xf1, 0x80, 0x9a, 0x15, 0x6f, 0xd8, 0x6e,
    ///     0xb7, 0x85, 0x18, 0xc8, 0xec, 0x4e, 0x59, 0x4a,
    ///     0xe2, 0x91, 0x43, 0x4c, 0xeb, 0x95, 0xb6, 0x2e,
    ///     0x9a, 0xea, 0x53, 0x68, 0x80, 0x64, 0x69, 0x40,
    ///     0xf9, 0xec, 0xbd, 0x85, 0x89, 0x26, 0x97, 0x67,
    ///     0xaf, 0xb0, 0xad, 0x00, 0x1b, 0xd4, 0xfd, 0x94,
    ///     0xd3, 0xe9, 0x92, 0xb1, 0xb4, 0xbc, 0x5a, 0xaa,
    ///     0x92, 0x80, 0x89, 0x3b, 0x39, 0x05, 0x6c, 0x22,
    ///     0x26, 0xfe, 0x5a, 0x28, 0x6c, 0x37, 0x50, 0x5a,
    ///     0x38, 0x99, 0xcf, 0xf3, 0xc1, 0x96, 0x45, 0xdc,
    ///     0x01, 0xcb, 0x20, 0x87, 0xa5, 0x00, 0x8c, 0xf5,
    ///     0x4d, 0xc2, 0xef, 0xb8, 0x9b, 0xd1, 0x87, 0xbe,
    ///     0xed, 0xd5, 0x0a, 0x29, 0x15, 0x34, 0x59, 0x4c,
    ///     0x3a, 0x05, 0x22, 0x05, 0x44, 0x4f, 0x9f, 0xc8,
    ///     0x47, 0x12, 0x24, 0x8e, 0xa8, 0x79, 0xe4, 0x67,
    ///     0xba, 0x4d, 0x5b, 0x75, 0x56, 0x95, 0xeb, 0xe8,
    ///     0x8a, 0xfa, 0x8e, 0x01, 0x8c, 0x1b, 0x74, 0x63,
    ///     0xd9, 0x2f, 0xf7, 0xd3, 0x44, 0x8f, 0xa8, 0xf5,
    ///     0xaf, 0x6c, 0x4f, 0xdb, 0xe7, 0xc9, 0x6c, 0x71,
    ///     0x22, 0xa3, 0x1d, 0xf1, 0x40, 0xb2, 0xe0, 0x9a,
    ///     0xb6, 0x72, 0xc9, 0xc0, 0x13, 0x16, 0xa2, 0x4a,
    ///     0xe1, 0x92, 0xc7, 0x54, 0x23, 0xab, 0x9d, 0xa1,
    ///     0xa1, 0xe5, 0x0b, 0xed, 0xba, 0xe8, 0x84, 0x37,
    ///     0xb2, 0xe7, 0xfe, 0x32, 0x8d, 0xfa, 0x1c, 0x53,
    ///     0x77, 0x97, 0xc7, 0xf3, 0x48, 0xc9, 0xdb, 0x2d,
    ///     0x75, 0x52, 0x9d, 0x42, 0x51, 0x78, 0x62, 0x68,
    ///     0x05, 0x45, 0x15, 0xf8, 0xa2, 0x4e, 0xf3, 0x0b
    /// ];
    /// let q = [
    ///     0xe0u8, 0x35, 0x37, 0xaf, 0xb2, 0x50, 0x91, 0x8e,
    ///     0xf2, 0x62, 0x2b, 0xd9, 0x9f, 0x6c, 0x11, 0x75,
    ///     0xec, 0x24, 0x5d, 0x78, 0x59, 0xe7, 0x8d, 0xb5,
    ///     0x40, 0x52, 0xed, 0x41
    /// ];
    /// let mut rng = RNG::new().expect("Error with RNG::new()");
    /// let mut dh = DH::new_from_pgq_with_check(&p, &g, &q, 0, &mut rng).expect("Error with new_from_pgq()");
    /// let mut private = [0u8; 256];
    /// let mut private_size = 0u32;
    /// let mut public = [0u8; 256];
    /// let mut public_size = 0u32;
    /// dh.generate_key_pair(&mut rng, &mut private, &mut private_size, &mut public, &mut public_size).expect("Error with generate_key_pair()");
    /// let private = &private[0..(private_size as usize)];
    /// dh.check_priv_key_ex(private, Some(&q)).expect("Error with check_priv_key_ex()");
    /// ```
    pub fn check_priv_key_ex(&mut self, private: &[u8], prime: Option<&[u8]>) -> Result<(), i32> {
        let private_size = private.len() as u32;
        let mut prime_ptr: *const u8 = core::ptr::null();
        let mut prime_size = 0u32;
        if let Some(prime) = prime {
            prime_ptr = prime.as_ptr();
            prime_size = prime.len() as u32;
        }
        let rc = unsafe {
            sys::wc_DhCheckPrivKey_ex(&mut self.wc_dhkey,
                private.as_ptr(), private_size,
                prime_ptr, prime_size)
        };
        if rc != 0 {
            return Err(rc);
        }
        Ok(())
    }

    /// Check public key for invalid numbers (partial check).
    ///
    /// This performs a partial public-key validation routine.
    ///
    /// # Parameters
    ///
    /// * `public`: Buffer containing public key.
    ///
    /// # Returns
    ///
    /// Returns either Ok(()) on success or Err(e) containing the wolfSSL
    /// library error code value.
    ///
    /// # Example
    ///
    /// ```rust
    /// use wolfssl::wolfcrypt::random::RNG;
    /// use wolfssl::wolfcrypt::dh::DH;
    /// let mut rng = RNG::new().expect("Error with RNG::new()");
    /// let mut dh = DH::new_named(DH::FFDHE_2048).expect("Error with new_named()");
    /// let mut private = [0u8; 256];
    /// let mut private_size = 0u32;
    /// let mut public = [0u8; 256];
    /// let mut public_size = 0u32;
    /// dh.generate_key_pair(&mut rng, &mut private, &mut private_size, &mut public, &mut public_size).expect("Error with generate_key_pair()");
    /// let public = &public[0..(public_size as usize)];
    /// dh.check_pub_key(public).expect("Error with check_pub_key()");
    /// ```
    pub fn check_pub_key(&mut self, public: &[u8]) -> Result<(), i32> {
        let public_size = public.len() as u32;
        let rc = unsafe {
            sys::wc_DhCheckPubKey(&mut self.wc_dhkey, public.as_ptr(), public_size)
        };
        if rc != 0 {
            return Err(rc);
        }
        Ok(())
    }

    /// Check public key for invalid numbers (full check).
    ///
    /// This performs a full public-key validation routine.
    ///
    /// # Parameters
    ///
    /// * `public`: Buffer containing public key.
    /// * `prime`: Buffer containing prime value.
    ///
    /// # Returns
    ///
    /// Returns either Ok(()) on success or Err(e) containing the wolfSSL
    /// library error code value.
    ///
    /// # Example
    ///
    /// ```rust
    /// use wolfssl::wolfcrypt::random::RNG;
    /// use wolfssl::wolfcrypt::dh::DH;
    /// let p = [
    ///     0xc5u8, 0x7c, 0xa2, 0x4f, 0x4b, 0xd6, 0x8c, 0x3c,
    ///     0xda, 0xc7, 0xba, 0xaa, 0xea, 0x2e, 0x5c, 0x1e,
    ///     0x18, 0xb2, 0x7b, 0x8c, 0x55, 0x65, 0x9f, 0xea,
    ///     0xe0, 0xa1, 0x36, 0x53, 0x2b, 0x36, 0xe0, 0x4e,
    ///     0x3e, 0x64, 0xa9, 0xe4, 0xfc, 0x8f, 0x32, 0x62,
    ///     0x97, 0xe4, 0xbe, 0xf7, 0xc1, 0xde, 0x07, 0x5a,
    ///     0x89, 0x28, 0xf3, 0xfe, 0x4f, 0xfe, 0x68, 0xbc,
    ///     0xfb, 0x0a, 0x7c, 0xa4, 0xb3, 0x14, 0x48, 0x89,
    ///     0x9f, 0xaf, 0xb8, 0x43, 0xe2, 0xa0, 0x62, 0x5c,
    ///     0xb4, 0x88, 0x3f, 0x06, 0x50, 0x11, 0xfe, 0x65,
    ///     0x8d, 0x49, 0xd2, 0xf5, 0x4b, 0x74, 0x79, 0xdb,
    ///     0x06, 0x62, 0x92, 0x89, 0xed, 0xda, 0xcb, 0x87,
    ///     0x37, 0x16, 0xd2, 0xa1, 0x7a, 0xe8, 0xde, 0x92,
    ///     0xee, 0x3e, 0x41, 0x4a, 0x91, 0x5e, 0xed, 0xf3,
    ///     0x6c, 0x6b, 0x7e, 0xfd, 0x15, 0x92, 0x18, 0xfc,
    ///     0xa7, 0xac, 0x42, 0x85, 0x57, 0xe9, 0xdc, 0xda,
    ///     0x55, 0xc9, 0x8b, 0x28, 0x9e, 0xc1, 0xc4, 0x46,
    ///     0x4d, 0x88, 0xed, 0x62, 0x8e, 0xdb, 0x3f, 0xb9,
    ///     0xd7, 0xc8, 0xe3, 0xcf, 0xb8, 0x34, 0x2c, 0xd2,
    ///     0x6f, 0x28, 0x06, 0x41, 0xe3, 0x66, 0x8c, 0xfc,
    ///     0x72, 0xff, 0x26, 0x3b, 0x6b, 0x6c, 0x6f, 0x73,
    ///     0xde, 0xf2, 0x90, 0x29, 0xe0, 0x61, 0x32, 0xc4,
    ///     0x12, 0x74, 0x09, 0x52, 0xec, 0xf3, 0x1b, 0xa6,
    ///     0x45, 0x98, 0xac, 0xf9, 0x1c, 0x65, 0x8e, 0x3a,
    ///     0x91, 0x84, 0x4b, 0x23, 0x8a, 0xb2, 0x3c, 0xc9,
    ///     0xfa, 0xea, 0xf1, 0x38, 0xce, 0xd8, 0x05, 0xe0,
    ///     0xfa, 0x44, 0x68, 0x1f, 0xeb, 0xd9, 0x57, 0xb8,
    ///     0x4a, 0x97, 0x5b, 0x88, 0xc5, 0xf1, 0xbb, 0xb0,
    ///     0x49, 0xc3, 0x91, 0x7c, 0xd3, 0x13, 0xb9, 0x47,
    ///     0xbb, 0x91, 0x8f, 0xe5, 0x26, 0x07, 0xab, 0xa9,
    ///     0xc5, 0xd0, 0x3d, 0x95, 0x41, 0x26, 0x92, 0x9d,
    ///     0x13, 0x67, 0xf2, 0x7e, 0x11, 0x88, 0xdc, 0x2d
    /// ];
    /// let g = [
    ///     0x4au8, 0x1a, 0xf3, 0xa4, 0x92, 0xe9, 0xee, 0x74,
    ///     0x6e, 0x57, 0xd5, 0x8c, 0x2c, 0x5b, 0x41, 0x41,
    ///     0x5e, 0xd4, 0x55, 0x19, 0xdc, 0xd9, 0x32, 0x91,
    ///     0xf7, 0xfd, 0xc2, 0x57, 0xff, 0x03, 0x14, 0xdb,
    ///     0xf1, 0xb7, 0x60, 0x0c, 0x43, 0x59, 0x3f, 0xff,
    ///     0xac, 0xf1, 0x80, 0x9a, 0x15, 0x6f, 0xd8, 0x6e,
    ///     0xb7, 0x85, 0x18, 0xc8, 0xec, 0x4e, 0x59, 0x4a,
    ///     0xe2, 0x91, 0x43, 0x4c, 0xeb, 0x95, 0xb6, 0x2e,
    ///     0x9a, 0xea, 0x53, 0x68, 0x80, 0x64, 0x69, 0x40,
    ///     0xf9, 0xec, 0xbd, 0x85, 0x89, 0x26, 0x97, 0x67,
    ///     0xaf, 0xb0, 0xad, 0x00, 0x1b, 0xd4, 0xfd, 0x94,
    ///     0xd3, 0xe9, 0x92, 0xb1, 0xb4, 0xbc, 0x5a, 0xaa,
    ///     0x92, 0x80, 0x89, 0x3b, 0x39, 0x05, 0x6c, 0x22,
    ///     0x26, 0xfe, 0x5a, 0x28, 0x6c, 0x37, 0x50, 0x5a,
    ///     0x38, 0x99, 0xcf, 0xf3, 0xc1, 0x96, 0x45, 0xdc,
    ///     0x01, 0xcb, 0x20, 0x87, 0xa5, 0x00, 0x8c, 0xf5,
    ///     0x4d, 0xc2, 0xef, 0xb8, 0x9b, 0xd1, 0x87, 0xbe,
    ///     0xed, 0xd5, 0x0a, 0x29, 0x15, 0x34, 0x59, 0x4c,
    ///     0x3a, 0x05, 0x22, 0x05, 0x44, 0x4f, 0x9f, 0xc8,
    ///     0x47, 0x12, 0x24, 0x8e, 0xa8, 0x79, 0xe4, 0x67,
    ///     0xba, 0x4d, 0x5b, 0x75, 0x56, 0x95, 0xeb, 0xe8,
    ///     0x8a, 0xfa, 0x8e, 0x01, 0x8c, 0x1b, 0x74, 0x63,
    ///     0xd9, 0x2f, 0xf7, 0xd3, 0x44, 0x8f, 0xa8, 0xf5,
    ///     0xaf, 0x6c, 0x4f, 0xdb, 0xe7, 0xc9, 0x6c, 0x71,
    ///     0x22, 0xa3, 0x1d, 0xf1, 0x40, 0xb2, 0xe0, 0x9a,
    ///     0xb6, 0x72, 0xc9, 0xc0, 0x13, 0x16, 0xa2, 0x4a,
    ///     0xe1, 0x92, 0xc7, 0x54, 0x23, 0xab, 0x9d, 0xa1,
    ///     0xa1, 0xe5, 0x0b, 0xed, 0xba, 0xe8, 0x84, 0x37,
    ///     0xb2, 0xe7, 0xfe, 0x32, 0x8d, 0xfa, 0x1c, 0x53,
    ///     0x77, 0x97, 0xc7, 0xf3, 0x48, 0xc9, 0xdb, 0x2d,
    ///     0x75, 0x52, 0x9d, 0x42, 0x51, 0x78, 0x62, 0x68,
    ///     0x05, 0x45, 0x15, 0xf8, 0xa2, 0x4e, 0xf3, 0x0b
    /// ];
    /// let q = [
    ///     0xe0u8, 0x35, 0x37, 0xaf, 0xb2, 0x50, 0x91, 0x8e,
    ///     0xf2, 0x62, 0x2b, 0xd9, 0x9f, 0x6c, 0x11, 0x75,
    ///     0xec, 0x24, 0x5d, 0x78, 0x59, 0xe7, 0x8d, 0xb5,
    ///     0x40, 0x52, 0xed, 0x41
    /// ];
    /// let q0 = [
    ///                                               0x00u8,
    ///     0xe0, 0x35, 0x37, 0xaf, 0xb2, 0x50, 0x91, 0x8e,
    ///     0xf2, 0x62, 0x2b, 0xd9, 0x9f, 0x6c, 0x11, 0x75,
    ///     0xec, 0x24, 0x5d, 0x78, 0x59, 0xe7, 0x8d, 0xb5,
    ///     0x40, 0x52, 0xed, 0x41
    /// ];
    /// let mut rng = RNG::new().expect("Error with RNG::new()");
    /// let mut dh = DH::new_from_pgq_with_check(&p, &g, &q, 0, &mut rng).expect("Error with new_from_pgq()");
    /// let mut private = [0u8; 256];
    /// let mut private_size = 0u32;
    /// let mut public = [0u8; 256];
    /// let mut public_size = 0u32;
    /// dh.generate_key_pair(&mut rng, &mut private, &mut private_size, &mut public, &mut public_size).expect("Error with generate_key_pair()");
    /// let public = &public[0..(public_size as usize)];
    /// dh.check_pub_key_ex(public, &q0).expect("Error with check_pub_key_ex()");
    /// ```
    pub fn check_pub_key_ex(&mut self, public: &[u8], prime: &[u8]) -> Result<(), i32> {
        let public_size = public.len() as u32;
        let prime_size = prime.len() as u32;
        let rc = unsafe {
            sys::wc_DhCheckPubKey_ex(&mut self.wc_dhkey,
                public.as_ptr(), public_size,
                prime.as_ptr(), prime_size)
        };
        if rc != 0 {
            return Err(rc);
        }
        Ok(())
    }

    /// Export Diffie-Hellman context parameters.
    ///
    /// # Parameters
    ///
    /// * `p`: Buffer in which to store the DH `p` parameter value.
    /// * `p_size`: Output parameter holding number of bytes written to `p`.
    /// * `q`: Buffer in which to store the DH `q` parameter value.
    /// * `q_size`: Output parameter holding number of bytes written to `q`.
    /// * `g`: Buffer in which to store the DH `g` parameter value.
    /// * `g_size`: Output parameter holding number of bytes written to `g`.
    ///
    /// # Returns
    ///
    /// Returns either Ok(()) or Err(e) containing the wolfSSL library error
    /// code value.
    pub fn export_params_raw(&mut self,
            p: &mut [u8], p_size: &mut u32,
            q: &mut [u8], q_size: &mut u32,
            g: &mut [u8], g_size: &mut u32) -> Result<(), i32> {
        *p_size = p.len() as u32;
        *q_size = q.len() as u32;
        *g_size = g.len() as u32;
        let rc = unsafe {
            sys::wc_DhExportParamsRaw(&mut self.wc_dhkey,
                p.as_mut_ptr(), p_size,
                q.as_mut_ptr(), q_size,
                g.as_mut_ptr(), g_size)
        };
        if rc != 0 {
            return Err(rc);
        }
        Ok(())
    }

    /// Generate a public/private key pair for the given DH parameters.
    ///
    /// # Parameters
    ///
    /// * `rng`: `RNG` instance used for random number generation.
    /// * `private`: Buffer in which to store the generated private key.
    /// * `private_size`: Output parameter storing the private key size in bytes.
    /// * `public`: Buffer in which to store the generated public key.
    /// * `public_size`: Output parameter storing the public key size in bytes.
    ///
    /// # Returns
    ///
    /// Returns either Ok(()) or Err(e) containing the wolfSSL library error
    /// code value.
    ///
    /// # Example
    ///
    /// ```rust
    /// use wolfssl::wolfcrypt::random::RNG;
    /// use wolfssl::wolfcrypt::dh::DH;
    /// let mut rng = RNG::new().expect("Failed to create RNG");
    /// let mut dh = DH::new_named(DH::FFDHE_2048).expect("Error with new_named()");
    /// let mut private = [0u8; 256];
    /// let mut private_size = 0u32;
    /// let mut public = [0u8; 256];
    /// let mut public_size = 0u32;
    /// dh.generate_key_pair(&mut rng, &mut private, &mut private_size, &mut public, &mut public_size).expect("Error with generate_key_pair()");
    /// ```
    pub fn generate_key_pair(&mut self, rng: &mut RNG,
            private: &mut [u8], private_size: &mut u32,
            public: &mut [u8], public_size: &mut u32) -> Result<(), i32> {
        *private_size = private.len() as u32;
        *public_size = public.len() as u32;
        let rc = unsafe {
            sys::wc_DhGenerateKeyPair(&mut self.wc_dhkey, &mut rng.wc_rng,
                private.as_mut_ptr(), private_size,
                public.as_mut_ptr(), public_size)
        };
        if rc != 0 {
            return Err(rc);
        }
        Ok(())
    }

    /// Generate a shared secret agreed upon by both parties.
    ///
    /// This function generates an agreed upon secret key based on a local
    /// private key and a received public key. If completed on both sides of an
    /// exchange, this function generates an agreed upon secret key for
    /// symmetric communication. On successfully generating a shared secret
    /// key, the size of the secret key written to `dout` will be returned.
    ///
    /// # Returns
    ///
    /// Returns either Ok(size) containing the size of the key written to
    /// `dout` or Err(e) containing the wolfSSL library error code value.
    ///
    /// # Example
    ///
    /// ```rust
    /// use wolfssl::wolfcrypt::random::RNG;
    /// use wolfssl::wolfcrypt::dh::DH;
    /// let mut rng = RNG::new().expect("Error with RNG::new()");
    /// let mut dh = DH::new_named(DH::FFDHE_2048).expect("Error with new_named()");
    /// let mut private0 = [0u8; 256];
    /// let mut private0_size = 0u32;
    /// let mut public0 = [0u8; 256];
    /// let mut public0_size = 0u32;
    /// dh.generate_key_pair(&mut rng, &mut private0, &mut private0_size, &mut public0, &mut public0_size).expect("Error with generate_key_pair()");
    /// let mut private1 = [0u8; 256];
    /// let mut private1_size = 0u32;
    /// let mut public1 = [0u8; 256];
    /// let mut public1_size = 0u32;
    /// dh.generate_key_pair(&mut rng, &mut private1, &mut private1_size, &mut public1, &mut public1_size).expect("Error with generate_key_pair()");
    /// let mut ss0 = [0u8; 256];
    /// let ss0_size = dh.shared_secret(&mut ss0, &private0, &public1).expect("Error with shared_secret()");
    /// let ss0 = &ss0[0..ss0_size];
    /// ```
    pub fn shared_secret(&mut self, dout: &mut [u8], private: &[u8], other_pub: &[u8]) -> Result<usize, i32> {
        let mut dout_size = dout.len() as u32;
        let private_size = private.len() as u32;
        let other_pub_size = other_pub.len() as u32;
        let rc = unsafe {
            sys::wc_DhAgree(&mut self.wc_dhkey,
                dout.as_mut_ptr(), &mut dout_size,
                private.as_ptr(), private_size,
                other_pub.as_ptr(), other_pub_size)
        };
        if rc != 0 {
            return Err(rc);
        }
        Ok(dout_size as usize)
    }
}

impl Drop for DH {
    /// Safely free the underlying wolfSSL DhKey context.
    ///
    /// This calls the `wc_FreeDhKey()` wolfssl library function.
    ///
    /// The Rust Drop trait guarantees that this method is called when the
    /// DH struct instance goes out of scope, automatically cleaning up
    /// resources and preventing memory leaks.
    fn drop(&mut self) {
        unsafe { sys::wc_FreeDhKey(&mut self.wc_dhkey); }
    }
}
