/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import cdf = require( './../../../../../base/dists/discrete-uniform/cdf' );
import DiscreteUniform = require( './../../../../../base/dists/discrete-uniform/ctor' );
import entropy = require( './../../../../../base/dists/discrete-uniform/entropy' );
import kurtosis = require( './../../../../../base/dists/discrete-uniform/kurtosis' );
import logcdf = require( './../../../../../base/dists/discrete-uniform/logcdf' );
import logpmf = require( './../../../../../base/dists/discrete-uniform/logpmf' );
import mean = require( './../../../../../base/dists/discrete-uniform/mean' );
import median = require( './../../../../../base/dists/discrete-uniform/median' );
import mgf = require( './../../../../../base/dists/discrete-uniform/mgf' );
import pmf = require( './../../../../../base/dists/discrete-uniform/pmf' );
import quantile = require( './../../../../../base/dists/discrete-uniform/quantile' );
import skewness = require( './../../../../../base/dists/discrete-uniform/skewness' );
import stdev = require( './../../../../../base/dists/discrete-uniform/stdev' );
import variance = require( './../../../../../base/dists/discrete-uniform/variance' );

/**
* Interface describing the `discrete-uniform` namespace.
*/
interface Namespace {
	/**
	* Discrete uniform distribution cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param a - minimum support
	* @param b - maximum support
	* @returns evaluated CDF
	*
	* @example
	* var y = ns.cdf( 3.0, 0, 4 );
	* // returns ~0.8
	*
	* var mycdf = ns.cdf.factory( 0, 10 );
	* y = mycdf( 0.5 );
	* // returns ~0.091
	*
	* y = mycdf( 8.0 );
	* // returns ~0.818
	*/
	cdf: typeof cdf;

	/**
	* Discrete uniform Distribution.
	*/
	DiscreteUniform: typeof DiscreteUniform;

	/**
	* Returns the entropy of a discrete uniform distribution.
	*
	* ## Notes
	*
	* -   If `a` or `b` is not an integer value, the function returns `NaN`.
	* -   If provided `a >= b`, the function returns `NaN`.
	*
	* @param a - minimum support
	* @param b - maximum support
	* @returns entropy
	*
	* @example
	* var v = ns.entropy( 0, 1 );
	* // returns ~0.693
	*
	* @example
	* var v = ns.entropy( 4, 12 );
	* // returns ~2.197
	*
	* @example
	* var v = ns.entropy( -4, 4 );
	* // returns ~2.197
	*
	* @example
	* var v = ns.entropy( 1, 2.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( -0.1, 1 );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( 2, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( NaN, 2 );
	* // returns NaN
	*/
	entropy: typeof entropy;

	/**
	* Returns the excess kurtosis of a discrete uniform distribution.
	*
	* ## Notes
	*
	* -   If `a` or `b` is not an integer value, the function returns `NaN`.
	* -   If provided `a >= b`, the function returns `NaN`.
	*
	* @param a - minimum support
	* @param b - maximum support
	* @returns excess kurtosis
	*
	* @example
	* var v = ns.kurtosis( 0, 1 );
	* // returns -2.0
	*
	* @example
	* var v = ns.kurtosis( 4, 12 );
	* // returns ~-1.23
	*
	* @example
	* var v = ns.kurtosis( -4, 4 );
	* // returns ~-1.23
	*
	* @example
	* var v = ns.kurtosis( 1, 0.4 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( -0.1, 1 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( 2, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( NaN, 2 );
	* // returns NaN
	*/
	kurtosis: typeof kurtosis;

	/**
	* Discrete uniform distribution logarithm of cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param a - minimum support
	* @param b - maximum support
	* @returns evaluated logCDF
	*
	* @example
	* var y = ns.logcdf( 3.0, 0, 4 );
	* // returns ~-0.223
	*
	* var mylogcdf = ns.logcdf.factory( 0, 10 );
	* y = mylogcdf( 0.5 );
	* // returns ~-2.398
	*
	* y = mylogcdf( 8.0 );
	* // returns ~-0.201
	*/
	logcdf: typeof logcdf;

	/**
	* Discrete uniform distribution natural logarithm of the probability mass function (logPMF).
	*
	* @param x - input value
	* @param a - minimum support
	* @param b - maximum support
	* @returns evaluated logPMF
	*
	* @example
	* var y = ns.logpmf( 3.0, 2, 6 );
	* // returns ~-1.609
	*
	* var myLogPMF = ns.logpmf.factory( 6, 7 );
	* y = myLogPMF( 7.0 );
	* // returns ~-0.693
	*
	* y = myLogPMF( 5.0 );
	* // returns -Infinity
	*/
	logpmf: typeof logpmf;

	/**
	* Returns the expected value of a discrete uniform distribution.
	*
	* ## Notes
	*
	* -   If `a` or `b` is not an integer value, the function returns `NaN`.
	* -   If provided `a >= b`, the function returns `NaN`.
	*
	* @param a - minimum support
	* @param b - maximum support
	* @returns expected value
	*
	* @example
	* var v = ns.mean( 0, 1 );
	* // returns 0.5
	*
	* @example
	* var v = ns.mean( -4, 4 );
	* // returns 0.0
	*
	* @example
	* var v = ns.mean( 0, 10 );
	* // returns 5.0
	*
	* @example
	* var v = ns.mean( 1, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( -0.1, 1 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( 2, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( NaN, 2 );
	* // returns NaN
	*/
	mean: typeof mean;

	/**
	* Returns the median of a discrete uniform distribution.
	*
	* ## Notes
	*
	* -   If `a` or `b` is not an integer value, the function returns `NaN`.
	* -   If provided `a >= b`, the function returns `NaN`.
	*
	* @param a - minimum support
	* @param b - maximum support
	* @returns median
	*
	* @example
	* var v = ns.median( 0, 1 );
	* // returns 0.5
	*
	* @example
	* var v = ns.median( -4, 4 );
	* // returns 0.0
	*
	* @example
	* var v = ns.median( 0, 10 );
	* // returns 5.0
	*
	* @example
	* var v = ns.median( 1, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.median( -0.1, 1 );
	* // returns NaN
	*
	* @example
	* var v = ns.median( 2, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.median( NaN, 2 );
	* // returns NaN
	*/
	median: typeof median;

	/**
	* Discrete uniform distribution moment-generating function (MGF).
	*
	* @param t - input value
	* @param a - minimum support
	* @param b - maximum support
	* @returns evaluated MGF
	*
	* @example
	* var y = ns.mgf( 2.0, 0, 4 );
	* // returns ~689.475
	*
	* y = ns.mgf( -0.2, 0, 4 );
	* // returns ~0.697
	*
	* y = ns.mgf( 2.0, 0, 1 );
	* // returns ~4.195
	*
	* var mymgf = ns.mgf.factory( 6, 7 );
	* y = mymgf( 0.1 );
	* // returns ~1.918
	*
	* y = mymgf( 1.1 );
	* // returns ~1471.722
	*/
	mgf: typeof mgf;

	/**
	* Discrete uniform distribution probability mass function (PMF).
	*
	* @param x - input value
	* @param a - minimum support
	* @param b - maximum support
	* @returns evaluated PMF
	*
	* @example
	* var y = ns.pmf( 3.0, 2, 6 );
	* // returns ~0.2
	*
	* var myPMF = ns.pmf.factory( 6, 7 );
	* y = myPMF( 7.0 );
	* // returns 0.5
	*
	* y = myPMF( 5.0 );
	* // returns 0.0
	*/
	pmf: typeof pmf;

	/**
	* Discrete uniform distribution quantile function.
	*
	* @param p - input value
	* @param a - minimum support
	* @param b - maximum support
	* @returns evaluated quantile function
	*
	* @example
	* var y = ns.quantile( 0.8, 0, 1 );
	* // returns 1
	*
	* y = ns.quantile( 0.5, 0, 10 );
	* // returns 5
	*
	* var myQuantile = ns.quantile.factory( 0, 4 );
	* y = myQuantile( 0.8 );
	* // returns 4
	*/
	quantile: typeof quantile;

	/**
	* Returns the skewness of a discrete uniform distribution.
	*
	* ## Notes
	*
	* -   If `a` or `b` is not an integer value, the function returns `NaN`.
	* -   If provided `a >= b`, the function returns `NaN`.
	*
	* @param a - minimum support
	* @param b - maximum support
	* @returns skewness
	*
	* @example
	* var v = ns.skewness( 0, 1 );
	* // returns 0.0
	*
	* @example
	* var v = ns.skewness( -4, 4 );
	* // returns 0.0
	*
	* @example
	* var v = ns.skewness( 0, 10 );
	* // returns 0.0
	*
	* @example
	* var v = ns.skewness( 1, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( -0.1, 1 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( 2, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( NaN, 2 );
	* // returns NaN
	*/
	skewness: typeof skewness;

	/**
	* Returns the standard deviation of a discrete uniform distribution.
	*
	* ## Notes
	*
	* -   If `a` or `b` is not an integer value, the function returns `NaN`.
	* -   If provided `a >= b`, the function returns `NaN`.
	*
	* @param a - minimum support
	* @param b - maximum support
	* @returns standard deviation
	*
	* @example
	* var v = ns.stdev( 0, 1 );
	* // returns ~0.5
	*
	* @example
	* var v = ns.stdev( 4, 12 );
	* // returns ~2.582
	*
	* @example
	* var v = ns.stdev( -4, 4 );
	* // returns ~2.582
	*
	* @example
	* var v = ns.stdev( 1, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( 0.5, 1 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( 2, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( NaN, 2 );
	* // returns NaN
	*/
	stdev: typeof stdev;

	/**
	* Returns the variance of a discrete uniform distribution.
	*
	* ## Notes
	*
	* -   If `a` or `b` is not an integer value, the function returns `NaN`.
	* -   If provided `a >= b`, the function returns `NaN`.
	*
	* @param a - minimum support
	* @param b - maximum support
	* @returns variance
	*
	* @example
	* var v = ns.variance( 0, 1 );
	* // returns ~0.25
	*
	* @example
	* var v = ns.variance( 4, 12 );
	* // returns ~6.667
	*
	* @example
	* var v = ns.variance( -4, 4 );
	* // returns ~6.667
	*
	* @example
	* var v = ns.variance( 1, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( 0.5, 1 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( 2, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( NaN, 2 );
	* // returns NaN
	*/
	variance: typeof variance;
}

/**
* Discrete uniform distribution.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
