/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2007-2015 Broad Institute
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package org.broad.igv.util;

import org.broad.igv.AbstractHeadlessTest;
import org.junit.Ignore;
import org.junit.Test;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;

import static org.junit.Assert.*;
import static spark.Spark.get;


/**
 * Test of class HttpUtils (TODO -- need more tests!!)
 */
public class HttpUtilsTest extends AbstractHeadlessTest {


    static String broadURLString = "https://igv-genepattern-org.s3.amazonaws.com/test/fasta/chr22.fa";

    @Test
    public void testSignedURLMatch() throws Exception {
        String aws = "https://amazonaws.com?X-Amz-Signature=foo";  //X-Amz-Signature"
        assertTrue(HttpUtils.isSignedURL(aws));
        String google = "https://google.com?X-Goog-Signature=bar";
        assertTrue(HttpUtils.isSignedURL(google));
        String noMatch = "https://www.google.com";
        assertFalse(HttpUtils.isSignedURL(noMatch));
    }


    @Test
    public void testGetContentLength() throws IOException {
        // Open an input stream just to check permissions
        HttpURLConnection conn = null;
        try {
            conn = (HttpURLConnection) (HttpUtils.createURL(broadURLString)).openConnection();
            conn.setRequestProperty("User-Agent", "IGV");
            String contentLength = conn.getHeaderField("Content-length");
            assertEquals("52330665", contentLength);

        } finally {

            if (conn != null) {
                conn.disconnect();  // <- this really doesn't do anything (see Sun documentation)
            }
        }
    }

    @Ignore
    @Test
    public void testReadByBlob() throws Exception {
        String serverURL = "serverURLhere";

//        String bamURL =  serverURL + "?file=CDNLtx0vjHKW19PP9E7A1zpmzSg1mrg/rw7ZkE+b8FPuoNLNNLNKyL/94CsspYExn2SAJ5Y0wgL9L2mwf/HsP4IwqsQbtAqVBLxtd2CVclTOkute7VHhzqFPJ2KCzWxUjM+Ecb5XfdGTpKAzya1dq/fAtJuIw8+NHQCPmMVbJWreQx1+6z3VKDT17Fy3RbXxL6X/CfQ/HlTcWFpQVe1p+5LgkojOVagWCImYNk/ErPzi8J2oEYPSm6ilOwDM6rGwHcO47qW8DncaPdf8ohpm/XZwAd+aAJwsqkBR689R+X175QCzmpOI07dHxuvzJ4HPlMwu2h2290QxVAJ8Ix5fVA==.bam";
//
//        InputStream is = HttpUtils.getInstance().openConnectionStream(HttpUtils.createURL(bamURL));

        String bedblob = StringUtils.decodeURL("c44z7H5E1gDMSm49T7NyGix051qDS7AbgCqicZ%2FpFkLobpmCim95byvYICc5VT%2Bv8Z%2FzE2gHWZkboBuME9eLxjEsfiO4bwnqZGP9fwoWXooK1LC8e3R5%2F6B9KyP9X3PR102PIApQASPfQGnYHqpBLifFPUbeRMqN%2BSxYi3h7udQJ8pli2QPEappIiOVWQ77cjJ6c0h12me6dT81fVrYT1E5CGHpnfUarIWCADRySVUfxqrwADKpnpaozMiWebh4OaWr5QfWHuG%2F%2F%2FhwVs7YxJaAR9S6pMqqfk213JofydHJjUimkv2V8tM3RJk3Q2y7CZ3Dk8X8wLiAJLfTIaXyreQ%3D%3D");

        String bedURL = serverURL + "?file=" + bedblob + ".bed";
        InputStream is = HttpUtils.getInstance().openConnectionStream(HttpUtils.createURL(bedURL));

        BufferedReader br = new BufferedReader(new InputStreamReader(is));

        String line = null;
        int count = 0;
        while ((line = br.readLine()) != null) {
            System.out.println(line);
            count++;
        }
        System.out.println(count + " lines read");

    }

    @Test
    public void testCacheControl() throws Exception {
        String headerValue = "no-cache, max-age=100";
        HttpUtils.CacheControl cc = HttpUtils.CacheControl.valueOf(headerValue);
        assertTrue(cc.isNoCache());
        assertEquals(100, cc.getMaxAge());
    }

    @Test
    public void testAccessTokenCache() throws MalformedURLException {

        try {
            // Exact match
            HttpUtils.getInstance().setAccessToken("foo", "bar.foo.com");
            String token = HttpUtils.getInstance().getCachedTokenFor(new URL("https://bar.foo.com/path"));
            assertEquals("foo", token);
            HttpUtils.getInstance().clearAccessTokens();

            // Wildcard match
            HttpUtils.getInstance().setAccessToken("foo", "*.foo.com");
            token = HttpUtils.getInstance().getCachedTokenFor(new URL("https://bar.foo.com/path"));
            assertEquals("foo", token);

            // Superceding match
            HttpUtils.getInstance().setAccessToken("foo2", "*.foo.com");
            token = HttpUtils.getInstance().getCachedTokenFor(new URL("https://bar.foo.com/path"));
            assertEquals("foo2", token);


            // Clear token
            HttpUtils.getInstance().clearAccessTokens();
            token = HttpUtils.getInstance().getCachedTokenFor(new URL("https://bar.foo.com/path"));
            assertNull(token);
            HttpUtils.getInstance().clearAccessTokens();

            // Match all hosts
            HttpUtils.getInstance().setAccessToken("foo", "");
            token = HttpUtils.getInstance().getCachedTokenFor(new URL("https://igv.org/path"));
            assertEquals("foo", token);
        } finally {
            HttpUtils.getInstance().clearAccessTokens();
        }
    }

    public class RunnableSparkHttp implements Runnable {

        /// counters incremented on each request:

        // permanent redirect src
        public int permSrcCt = 0;
        // permanent redirect dest
        public int permDestCt = 0;
        // temporary redirect src
        public int tempSrcCt = 0;
        // temporary redirect dest
        public int tempDestCt = 0;

        public void run() {
            System.out.println("run thing");

            get("/perm_redir_src", (req, res) ->
            {
                // increment a counter for tests to inspect
                this.permSrcCt += 1;
                res.status(301); // permanent redirect
                res.header("Location", "http://localhost:4567/perm_redir_dest");
                return "redirecting";
            });

            get("/perm_redir_dest", (req, res) -> {
                this.permDestCt += 1;
                return "done";
            });

            get("/temp_redir_src", (req, res) ->
            {
                this.tempSrcCt += 1;
                res.status(302); // temporary redirect
                res.header("Location", "http://localhost:4567/temp_redir_dest");
                res.header("Cache-Control", "max-age=1"); // expire in 1 second
                return "redirecting";
            });

            get("/temp_redir_dest", (req, res) -> {
                this.tempDestCt += 1;
                return "done";
            });

        }

    }
}
